/********************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2009.
-------------------------------------------------------------------------
File name:   AILog.cpp
$Id$
Description: 

-------------------------------------------------------------------------
History:
- ?
- 2 Mar 2009	: Evgeny Adamenkov: Replaced IRenderer with CDebugDrawContext

*********************************************************************/
#include "StdAfx.h"

#include "AILog.h"
#include "CAISystem.h"
#include "CryAISystem.h"

#include "ISystem.h"
#include "ITimer.h"
#include "IValidator.h"
#include "IConsole.h"
#include "DebugDrawContext.h"
#include "ITestSystem.h"

// these should all be in sync - so testing one for 0 should be the same for all
ISystem *pSystem = 0;

static const char outputPrefix[] = "AI: ";
static const unsigned outputPrefixLen = sizeof(outputPrefix) - 1;

#define DECL_OUTPUT_BUF	char outputBufferLog[MAX_WARNING_LENGTH + outputPrefixLen];	unsigned outputBufferSize = sizeof(outputBufferLog)

static const int maxSavedMsgs = 5;
static const int maxSavedMsgLength = MAX_WARNING_LENGTH + outputPrefixLen + 1;
enum ESavedMsgType {SMT_WARNING, SMT_ERROR};
struct SSavedMsg
{
	ESavedMsgType savedMsgType;
	char savedMsg[maxSavedMsgLength];
	CTimeValue time;
};
static SSavedMsg savedMsgs[maxSavedMsgs];
static int savedMsgIndex = 0;

//====================================================================
// DebugDrawLabel
//====================================================================
static void DebugDrawLabel(ESavedMsgType type, float timeFrac, int col, int row, const char* szText)
{
	float ColumnSize = 11;
	float RowSize = 11;
	float baseY = 10;
	ColorB colorWarning(0, 255, 255);
	ColorB colorError(255, 255, 0);
	ColorB& color = (type == SMT_ERROR) ? colorError : colorWarning;
	CDebugDrawContext	dc;

	float alpha = 1.0f;
	static float fadeFrac = 0.5f;
	if (timeFrac < fadeFrac && fadeFrac > 0.0f)
		alpha = timeFrac / fadeFrac;
	color.a = static_cast<uint8>(255 * alpha);

	float actualCol = ColumnSize*static_cast<float>(col);
	float actualRow;
	if (row >= 0)
		actualRow = baseY+RowSize*static_cast<float>(row);
	else
		actualRow = dc->GetHeight() - (baseY + RowSize * static_cast<float>(-row));

	dc->Draw2dLabel(actualCol, actualRow, 1.2f, color, false, "%s", szText);
}

//====================================================================
// DisplaySavedMsgs
//====================================================================
void AILogDisplaySavedMsgs()
{
  float savedMsgDuration = gAIEnv.CVars.OverlayMessageDuration;
  if (savedMsgDuration < 0.01f)
    return;
  static int col = 1;

  int row = -1;
  CTimeValue currentTime = gEnv->pTimer->GetFrameStartTime();
  CTimeValue time = currentTime - CTimeValue(savedMsgDuration);
  for (int i = 0 ; i < maxSavedMsgs ; ++i)
  {
    int index = (maxSavedMsgs + savedMsgIndex - i) % maxSavedMsgs;
    if (savedMsgs[index].time < time)
      return;
    // get rid of msgs from the future - can happen during load/save
    if (savedMsgs[index].time > currentTime)
			savedMsgs[index].time = time;
//      savedMsgIndex = (maxSavedMsgs + savedMsgIndex - 1) % maxSavedMsgs;

    float timeFrac = (savedMsgs[index].time - time).GetSeconds() / savedMsgDuration;
    DebugDrawLabel(savedMsgs[index].savedMsgType, timeFrac, col, row, savedMsgs[index].savedMsg);
    --row;
  }
}

//====================================================================
// AIInitLog
//====================================================================
void AIInitLog(ISystem * system)
{
	if (pSystem)
		AIWarning("Re-registering AI Logging");

	AIAssert(system);
	if (!system)
		return;
	IConsole *console = gEnv->pConsole;
#ifdef _DEBUG
  int isDebug = 1;
#else
  int isDebug = 0;
#endif

	if(console)
		pSystem = system;

	for (int i = 0 ; i < maxSavedMsgs ; ++i)
	{
		savedMsgs[i].savedMsg[0] = '\0';
		savedMsgs[i].savedMsgType = SMT_WARNING;
		savedMsgs[i].time = CTimeValue(0.0f);
		savedMsgIndex = 0;
	}
}

//====================================================================
// AIGetLogConsoleVerbosity
//====================================================================
int AIGetLogConsoleVerbosity()
{
	return gAIEnv.CVars.LogConsoleVerbosity;
}

//====================================================================
// AIGetLogFileVerbosity
//====================================================================
int AIGetLogFileVerbosity()
{
	return gAIEnv.CVars.LogFileVerbosity;
}

//====================================================================
// AICheckLogVerbosity
//====================================================================
bool AICheckLogVerbosity(const AI_LOG_VERBOSITY CheckVerbosity)
{
	bool bResult = false;

	const int iAILogVerbosity = AIGetLogFileVerbosity();
	const int iAIConsoleVerbosity = AIGetLogConsoleVerbosity();

	if (iAILogVerbosity >= CheckVerbosity || iAIConsoleVerbosity >= CheckVerbosity)
	{
		// Check against actual log system
		const int nVerbosity = gEnv->pLog->GetVerbosityLevel();
		bResult = (nVerbosity >= CheckVerbosity);
	}

	return bResult;
}

//===================================================================
// AIGetWarningErrorsEnabled
//===================================================================
bool AIGetWarningErrorsEnabled()
{
	return gAIEnv.CVars.EnableWarningsErrors != 0;
}

//====================================================================
// AIWarning
//====================================================================
void AIWarning( const char *format,... )
{
	if (!pSystem || !AIGetWarningErrorsEnabled() || !AICheckLogVerbosity(AI_LOG_WARNING)) 
		return;

	DECL_OUTPUT_BUF;

	va_list args;
	va_start(args, format);
	vsnprintf(outputBufferLog, outputBufferSize, format, args);
	outputBufferLog[outputBufferSize-1] = '\0';
	va_end(args);
	pSystem->Warning( VALIDATOR_MODULE_AI, VALIDATOR_WARNING, VALIDATOR_FLAG_AI, 0, "AI: %s", outputBufferLog );

	savedMsgIndex = (savedMsgIndex + 1) % maxSavedMsgs;
	savedMsgs[savedMsgIndex].savedMsgType = SMT_WARNING;
	strncpy(savedMsgs[savedMsgIndex].savedMsg, outputBufferLog, maxSavedMsgLength);
	savedMsgs[savedMsgIndex].savedMsg[maxSavedMsgLength-1] = '\0';
	savedMsgs[savedMsgIndex].time = gEnv->pTimer->GetFrameStartTime();
}

//====================================================================
// AILogAlways
//====================================================================
void AILogAlways(const char *format, ...)
{
	if (!pSystem) 
		return;

	DECL_OUTPUT_BUF;

	strncpy(outputBufferLog, outputPrefix, outputPrefixLen);
	va_list args;
	va_start(args, format);
	vsnprintf(outputBufferLog + outputPrefixLen, outputBufferSize - outputPrefixLen, format, args);
  outputBufferLog[outputBufferSize-1] = '\0';
	va_end(args);

	gEnv->pLog->Log(outputBufferLog);
}

void AILogLoading(const char *format, ...)
{
  if (!pSystem) 
    return;

	DECL_OUTPUT_BUF;

  strncpy(outputBufferLog, outputPrefix, outputPrefixLen);
  va_list args;
  va_start(args, format);
  vsnprintf(outputBufferLog + outputPrefixLen, outputBufferSize - outputPrefixLen, format, args);
  outputBufferLog[outputBufferSize-1] = '\0';
  va_end(args);

  gEnv->pLog->UpdateLoadingScreen(outputBufferLog);
}

//====================================================================
// AIHandleLogMessage
//====================================================================
void AIHandleLogMessage(const char* outputBufferLog)
{
	const int cV = AIGetLogConsoleVerbosity();
	const int fV = AIGetLogFileVerbosity();

	if ( (cV >= AI_LOG_PROGRESS) && (fV >= AI_LOG_PROGRESS) )
		gEnv->pLog->Log(outputBufferLog);
	else if (cV >= AI_LOG_PROGRESS)
		gEnv->pLog->LogToConsole(outputBufferLog);
	else if (fV >= AI_LOG_PROGRESS)
		gEnv->pLog->LogToFile(outputBufferLog);
}

//====================================================================
// AILogProgress
//====================================================================
void AILogProgress(const char *format, ...)
{
	if (!pSystem || !AICheckLogVerbosity(AI_LOG_PROGRESS)) 
		return;

	DECL_OUTPUT_BUF;

	strncpy(outputBufferLog, outputPrefix, outputPrefixLen);
	va_list args;
	va_start(args, format);
	vsnprintf(outputBufferLog + outputPrefixLen, outputBufferSize - outputPrefixLen, format, args);
	outputBufferLog[outputBufferSize-1] = '\0';
	va_end(args);

	AIHandleLogMessage(outputBufferLog);
}

//====================================================================
// AILogEvent
//====================================================================
void AILogEvent(const char *format, ...)
{
	if (!pSystem || !AICheckLogVerbosity(AI_LOG_EVENT)) 
		return;

	DECL_OUTPUT_BUF;

	strncpy(outputBufferLog, outputPrefix, outputPrefixLen);
	va_list args;
	va_start(args, format);
	vsnprintf(outputBufferLog + outputPrefixLen, outputBufferSize - outputPrefixLen, format, args);
	outputBufferLog[outputBufferSize-1] = '\0';
	va_end(args);

	AIHandleLogMessage(outputBufferLog);
}

//====================================================================
// AILogComment
//====================================================================
void AILogComment(const char *format, ...)
{
	if (!pSystem || !AICheckLogVerbosity(AI_LOG_COMMENT)) 
		return;

	DECL_OUTPUT_BUF;

	strncpy(outputBufferLog, outputPrefix, outputPrefixLen);
	va_list args;
	va_start(args, format);
	vsnprintf(outputBufferLog + outputPrefixLen, outputBufferSize - outputPrefixLen, format, args);
	outputBufferLog[outputBufferSize-1] = '\0';
	va_end(args);

	AIHandleLogMessage(outputBufferLog);
}

// for error - we want a message box
// [AlexMcC|17.02.10] Do we really, really want a message box?
#ifdef WIN32
#include <windows.h>

//====================================================================
// AIError
//====================================================================
void AIError( const char *format,... )
{
	if (!pSystem || !AIGetWarningErrorsEnabled() || !AICheckLogVerbosity(AI_LOG_ERROR)) 
		return;

	DECL_OUTPUT_BUF;

	va_list args;
	va_start(args, format);
	vsnprintf(outputBufferLog, outputBufferSize, format, args);
	outputBufferLog[outputBufferSize-1] = '\0';
	va_end(args);
	pSystem->Warning( VALIDATOR_MODULE_AI, VALIDATOR_ERROR, VALIDATOR_FLAG_AI, 0, "AI: Error: %s", outputBufferLog );

	savedMsgIndex = (savedMsgIndex + 1) % maxSavedMsgs;
	savedMsgs[savedMsgIndex].savedMsgType = SMT_ERROR;
	strncpy(savedMsgs[savedMsgIndex].savedMsg, outputBufferLog, maxSavedMsgLength);
	savedMsgs[savedMsgIndex].savedMsg[maxSavedMsgLength-1] = '\0';
	savedMsgs[savedMsgIndex].time = gEnv->pTimer->GetFrameStartTime();

	if(GetISystem()->GetITestSystem())
	{
		// supress user interaction for automated test
		if (gEnv->pLog)
			gEnv->pLog->LogError(outputBufferLog);
	}
	else
	{
		static bool sDoMsgBox = true;
		if (sDoMsgBox && !gEnv->IsEditor())
		{
			static char msg[4096];
			sprintf(msg, 
				"AI: %s \n"
				"ABORT to abort execution\n"
				"RETRY to continue (Don't expect AI to work properly)\n"
				"IGNORE to continue without any more of these AI error msg boxes", outputBufferLog);
			// Write out via MessageBox
			int nCode = MessageBox(
				0,
				msg,
				"AI Error",
				MB_ABORTRETRYIGNORE|MB_ICONHAND|MB_SETFOREGROUND|MB_SYSTEMMODAL);

			// Abort: abort the program
			if (nCode == IDABORT)
				CryFatalError(" AI: %s", outputBufferLog);
			else if (nCode == IDIGNORE)
				sDoMsgBox = false;
		}
	}
}

#else // WIN32

//====================================================================
// AIError
//====================================================================
void AIError( const char *format,... )
{
	if (!pSystem || !AIGetWarningErrorsEnabled() || !AICheckLogVerbosity(AI_LOG_ERROR)) 
		return;

	DECL_OUTPUT_BUF;

	va_list args;
	va_start(args, format);
	vsnprintf(outputBufferLog, outputBufferSize, format, args);
	outputBufferLog[outputBufferSize-1] = '\0';
	va_end(args);
	pSystem->Warning( VALIDATOR_MODULE_AI, VALIDATOR_ERROR, VALIDATOR_FLAG_AI, 0, "AI: Error: %s", outputBufferLog );

	savedMsgIndex = (savedMsgIndex + 1) % maxSavedMsgs;
	savedMsgs[savedMsgIndex].savedMsgType = SMT_ERROR;
	strncpy(savedMsgs[savedMsgIndex].savedMsg, outputBufferLog, maxSavedMsgLength);
	savedMsgs[savedMsgIndex].savedMsg[maxSavedMsgLength-1] = '\0';
	savedMsgs[savedMsgIndex].time = gEnv->pTimer->GetFrameStartTime();
}



#endif  // WIN32
