/********************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2010.
-------------------------------------------------------------------------
File name:   MissLocationSensor.h
$Id$
Description: 

-------------------------------------------------------------------------
History:
- 29 Mar 2010			 : Alex McCarthy: extracted from AIPlayer.h/cpp

*********************************************************************/
#include "StdAfx.h"
#include "MissLocationSensor.h"

CMissLocationSensor::CMissLocationSensor(const CAIActor* owner)
: m_state(Starting)
, m_owner(owner)
{
	AddDestroyableClass("DestroyableObject");
	AddDestroyableClass("BreakableObject");
	AddDestroyableClass("PressurizedObject");
}

void CMissLocationSensor::Update(float timeLimit)
{
	while(true)
	{
		switch (m_state)
		{
		case Starting:
			{
				m_updateCount = 0;
				m_state = Collecting;
			}
			break;
		case Collecting:
			{
				Collect(ent_static | ent_rigid | ent_sleeping_rigid | ent_independent);
				m_state = Filtering;
			}
			return;
		case Filtering:
			{
				if (Filter(timeLimit))
				{
					m_state = Finishing;
					break;
				}
			}
			return;
		case Finishing:
			{
				m_working.swap(m_locations);
				m_working.resize(0);
				m_state = Starting;
			}
			return;
		default:
			{
				assert(0);
				return;
			}
		}
	}

	++m_updateCount;
}

void CMissLocationSensor::Collect(int types)
{
	FUNCTION_PROFILER(gEnv->pSystem, PROFILE_AI);

	const float boxHalfSize = gAIEnv.CVars.CoolMissesBoxSize * 0.5f;
	const float boxHeight = gAIEnv.CVars.CoolMissesBoxHeight;

	const Vec3& feet = m_owner->GetPhysicsPos();
	const Vec3& dir = m_owner->GetViewDir();

	Vec3 pos = feet + Vec3(0.0f, 0.0f, -0.25f) + dir * boxHalfSize * 0.25f;
	Vec3 min(pos.x - boxHalfSize, pos.y - boxHalfSize, pos.z);
	Vec3 max(pos.x + boxHalfSize, pos.y + boxHalfSize, pos.z + boxHeight);

	types |= ent_allocate_list;
	m_entities.resize(MaxCollectedCount);

	IPhysicalEntity** entities = &m_entities.front();
	m_entities.resize(gEnv->pPhysicalWorld->GetEntitiesInBox(min, max, entities, types, MaxCollectedCount));
	assert(m_entities.size() <= MaxCollectedCount);

	// Add refs to prevent physics from deleting the entity
	// TODO(mrcio): Fix this once physics gets a ent_add_refs flag
	{
		MissEntities::iterator it = m_entities.begin();
		MissEntities::iterator end = m_entities.end();

		for ( ; it != end; ++it)
			(*it)->AddRef();
	}
}

bool CMissLocationSensor::Filter(float timeLimit)
{
	if (m_entities.empty())
		return true;

	CTimeValue now = gEnv->pTimer->GetAsyncTime();
	CTimeValue start = now;
	CTimeValue endTime = now + CTimeValue(timeLimit);

	do
	{
		IPhysicalEntity& entity = *m_entities.front();
		std::swap(m_entities.front(), m_entities.back());
		m_entities.pop_back();

		bool destroyable = false;
		if (IEntity* ientity = gEnv->pEntitySystem->GetEntityFromPhysics(&entity))
		{
			for (uint c = 0; c < m_destroyableEntityClasses.size(); ++c)
			{
				if (ientity->GetClass() == m_destroyableEntityClasses[c])
				{
					destroyable = true;
					break;
				}
			}	
		}

		Matrix34 worldTM;

		// Check for idmatBreakable or geom_manually_breakable
		// If the entity was previous known to be destroyable add all it's physical parts
		{
			pe_params_part pp;
			pp.ipart = 0;
			pp.pMtx3x4 = &worldTM;

			uint32 partCount = 0;
			while (entity.GetParams(&pp))
			{
				++partCount;

				if (!pp.pPhysGeom || !pp.pPhysGeom->pGeom)
					continue;

				primitives::box box;
				pp.pPhysGeom->pGeom->GetBBox(&box);

				Vec3 pos = worldTM.TransformPoint(box.center);

				if (!destroyable)
				{
					if (pp.flagsOR & geom_manually_breakable)
					{
						m_working.push_back(
							MissLocation(pos, MissLocation::ManuallyBreakable));
					}

					if (pp.idmatBreakable >= 0)
					{
						m_working.push_back(
							MissLocation(pos, MissLocation::MatBreakable));
					}

					if (pp.idSkeleton >= 0)
					{
						m_working.push_back(
							MissLocation(pos, MissLocation::Deformable));
					}
					else
					{
						m_working.push_back(
							MissLocation(pos, MissLocation::Destroyable));
					}
				}

				pp.ipart = partCount;
				MARK_UNUSED pp.partid;
			}
		}

		// Check if entity contains structural joints
		// Track which parts are connected by joints
		// Add those parts as good miss locations
		{
			uint64 jointConnectedParts = 0;
			pe_params_structural_joint sjp;
			sjp.idx = 0;

			uint32 jointCount = 0;
			while (entity.GetParams(&sjp))
			{
				++jointCount;
				if (sjp.bBreakable && !sjp.bBroken)
				{
					if ((sjp.partid[0] > -1) && (sjp.partid[0] < 64))
						jointConnectedParts |= 1ll << sjp.partid[0];

					if ((sjp.partid[1] > -1) && (sjp.partid[1] < 64))
						jointConnectedParts |= 1ll << sjp.partid[1];
				}

				sjp.idx = jointCount;
				MARK_UNUSED sjp.id;
			}

			if (jointConnectedParts)
			{
				pe_status_pos spos;
				spos.pMtx3x4 = &worldTM;

				for (uint32 p = 0; p < 64; ++p)
				{
					if (jointConnectedParts & (1ll << p))
					{
						spos.partid = p;
						MARK_UNUSED spos.ipart;

						if (!entity.GetStatus(&spos))
							continue;

						if (!spos.pGeom)
							continue;

						primitives::box box;
						spos.pGeom->GetBBox(&box);

						m_working.push_back(
							MissLocation(worldTM.TransformPoint(box.center), MissLocation::JointStructure));
					}
				}
			}
		}

		// Add rope vertices
		pe_status_rope srope;
		pe_params_rope prope;
		if (entity.GetStatus(&srope) && entity.GetParams(&prope))
		{
			if (prope.pEntTiedTo[0] && prope.pEntTiedTo[1])
			{
				uint32 pointCount = std::max(srope.nVtx, (srope.nSegments + 1)); // use the version with the most detail
				uint32 step = 1;

				if (pointCount > MaxRopeVertexCount)
					step = pointCount / MaxRopeVertexCount;

				m_vertices.resize(pointCount);

				if (srope.nVtx < srope.nSegments)
					srope.pPoints = &m_vertices.front();
				else
					srope.pVtx = &m_vertices.front();

				entity.GetStatus(&srope);

				for (uint i = 0; i < m_vertices.size(); i += step)
					m_working.push_back(MissLocation(m_vertices[i], MissLocation::Rope));
			}
		}

		now = gEnv->pTimer->GetAsyncTime();

		entity.Release();

		if (m_entities.empty())
			return true;

	} while (now < endTime);
	
	return false;
}

bool CMissLocationSensor::GetLocation(const Vec3& shootPos, const Vec3& shootDir, float maxAngle, Vec3& pos)
{
	FUNCTION_PROFILER(gEnv->pSystem,PROFILE_AI);

	if (m_locations.empty())
		return false;

	m_goodies.resize(0);

	float maxAngleCos = cry_cosf(maxAngle);
	float angleIntervalInv = 1.0f / (1.0f - maxAngleCos);

	uint32 maxConsidered = min(static_cast<uint32>(MaxConsiderCount), m_locations.size());

	MissLocations::iterator it = m_locations.begin();
	MissLocations::iterator end = m_locations.end();

	for ( ; maxConsidered && (it != end); ++it)
	{
		MissLocation& location = *it;

		Vec3 dir(location.position - shootPos);
		dir.Normalize();

		float angleCos = dir.dot(shootDir);
		if (angleCos <= maxAngleCos)
			continue;

		float typeScore = 0.0f;
		switch(location.type)
		{
		case MissLocation::Destroyable:
			typeScore = 1.0f;
			break;
		case MissLocation::Rope:
			typeScore = 0.95f;
			break;
		case MissLocation::ManuallyBreakable:
			typeScore = 0.85f;
			break;
		case MissLocation::JointStructure:
			typeScore = 0.75f;
			break;
		case MissLocation::Deformable:
			typeScore = 0.65f;
			break;
		case MissLocation::MatBreakable:
			typeScore = 0.5f;
			break;
		case MissLocation::Unbreakable:
		default:
			break;
		}

		float angleScore = (angleCos - maxAngleCos) * angleIntervalInv;

		m_goodies.push_back(location);
		m_goodies.back().score = (angleScore * 0.4f) + (typeScore * 0.6f);
		--maxConsidered;
	}

	std::sort(m_goodies.begin(), m_goodies.end());

	// Ignore anything that would hit the player
	if (IPhysicalEntity* ownerPhysics = m_owner->GetPhysics())
	{
		pe_status_pos ppos;
		if (ownerPhysics->GetStatus(&ppos))
		{
			AABB player(ppos.BBox[0] - ppos.pos, ppos.BBox[1] + ppos.pos);
			player.Expand(Vec3(0.35f));
			
			Lineseg lineOfFire;
			lineOfFire.start = shootPos;

			uint32 locationCount = m_goodies.size();
			for (uint32 i = 0; (i < locationCount) && (i < MaxRandomPool); ++i)
			{
				lineOfFire.end = m_goodies[i].position;
				
				if (Overlap::Lineseg_AABB(lineOfFire, player))
				{
					std::swap(m_goodies[i], m_goodies.back());
					m_goodies.pop_back();
					--i;
					--locationCount;
				}
			}

			//GetAISystem()->AddDebugBox(ZERO, OBB::CreateOBBfromAABB(Quat(IDENTITY), player), 255, 255, 255, 0.33f);
		}
	}

	if (m_goodies.empty())
		return false;

	pos = m_goodies[Random(min(static_cast<uint32>(MaxRandomPool), m_goodies.size()))].position;

	return true;
}

void CMissLocationSensor::AddDestroyableClass(const char* className)
{
	if (IEntityClass* entityClass = gEnv->pEntitySystem->GetClassRegistry()->FindClass(className))
		stl::push_back_unique(m_destroyableEntityClasses, entityClass);
}

void CMissLocationSensor::ResetDestroyableClasses()
{
	m_destroyableEntityClasses.clear();
}
