
#include "StdAfx.h"
#include "AnimationRandomizer.h"



CAnimationRandomizer::CAnimationRandomizer() 
: m_lastAnimId(-1)
, m_probabilitySum(0.0f)
, m_elapsedTimeFromEnterState(0.0f)
, m_initialDelayApplied(false)
{
	m_animations.clear();
}

CAnimationRandomizer::CAnimationRandomizer( CAnimationRandomizer& copyFrom )
: m_lastAnimId(-1)
, m_probabilitySum(0.0f)
, m_elapsedTimeFromEnterState(0.0f)
, m_initialDelayApplied(false)
{
	m_animations.clear();

	// copy everything
	m_lastAnimId = copyFrom.m_lastAnimId;
	m_probabilitySum = copyFrom.m_probabilitySum;
	m_elapsedTimeFromEnterState = copyFrom.m_elapsedTimeFromEnterState;
	m_initialDelayApplied = copyFrom.m_initialDelayApplied;
	m_generalParams = copyFrom.m_generalParams;
	int animCount = copyFrom.m_animations.size();
	for (int i = 0; i < animCount; ++i)
	{
		m_animations.push_back(copyFrom.m_animations[i]);
	}
}

CCryName CAnimationRandomizer::GetNextAssetName()
{
	int animCount = m_animations.size();
	CRY_ASSERT(animCount);

	if (animCount == 1)
	{
		return m_animations[0].animName;
	}

	int nextAssetId = -1;
	bool valid;
	do 
	{
		valid = true;
		float randomNumber = (cry_rand() / (float)RAND_MAX) * m_probabilitySum;

		for (int i = 0; i < animCount; ++i)	
		{
			float assetProb = static_cast<float>(m_animations[i].probability);
			if (randomNumber <= assetProb)
			{
				// maybe this animation was already selected as the last one and should not be started twice in a row
				if (m_animations[i].canBeStartedTwiceInARow == false && m_lastAnimId == i)
				{
					valid = false;
					break;
				}

				nextAssetId = i;
				break;
			}

			randomNumber -= assetProb;
		}
	} while (!valid);

	CRY_ASSERT(nextAssetId >= 0);
	if (nextAssetId >= 0)
	{
		m_lastAnimId = nextAssetId;
		return m_animations[nextAssetId].animName;
	}
	else
	{
		CryLog("CAnimationRandomizer::GetNextAssetName invalid nextAssetId {%d}", nextAssetId);
		return "";
	}
}

bool CAnimationRandomizer::IsItTimeForAnimationSwitch(float animTime /*= -1.0f*/)
{
	if (!m_generalParams.switchAssetsOverTime && !HasInitialDelay())
		return false;

	if (HasInitialDelay() && !m_initialDelayApplied)
	{
		m_initialDelayApplied = m_generalParams.initialDelay < m_elapsedTimeFromEnterState;
		return m_initialDelayApplied;
	}

	// Explanation:
	// Switch randomly once every x seconds (x = m_switchProbabilitySeconds)
	// Probability this frame = (frameTime / x)

	// Is it time for an asset switch?
	if (m_generalParams.switchAssetsOverTime)
	{
		// can the asset currently running be interrupted
		// and if not, is it already done so we can start the next one?
		// irrelevant for looping animations of course
		if (!m_animations[m_lastAnimId].looping && !m_animations[m_lastAnimId].allowInterrupt)
		{
			if (animTime >= 0.0f && animTime < 0.98f)
				return false;
		}

		float randomNumber = (cry_rand() / (float)RAND_MAX);
		float frameTime = gEnv->pTimer->GetFrameTime();
		if (randomNumber < (frameTime / m_generalParams.switchProbabilitySeconds))
		{
			return true;
		}
	}

	return false;
}

void CAnimationRandomizer::SetAnimationParams( CryCharAnimationParams& params )
{
	if (m_lastAnimId < 0)
		return;

	// time warping
	if (m_generalParams.timeAlignBetweenAssets)
		params.m_nFlags |= CA_TRANSITION_TIMEWARPING;
	else
		params.m_nFlags &= ~CA_TRANSITION_TIMEWARPING;

	CRandomAnim anim = m_animations[m_lastAnimId];

	// looping
	if (anim.looping)
		params.m_nFlags |= CA_LOOP_ANIMATION;
	else
		params.m_nFlags &= ~CA_LOOP_ANIMATION;

	// transition time
	params.m_fTransTime = anim.transitionTime;
}

void CAnimationRandomizer::EnterState()
{
	m_elapsedTimeFromEnterState = 0.0f;
	m_initialDelayApplied = false;
}

void CAnimationRandomizer::Update()
{
	m_elapsedTimeFromEnterState += gEnv->pTimer->GetFrameTime();
}

bool CAnimationRandomizer::HasInitialDelay() const
{
	return m_generalParams.initialDelay > 0.0f;
}

bool CAnimationRandomizer::Init( XmlNodeRef node )
{
	// Safety checking
	if (strcmp(node->getTag(), "RandomizerSet") != 0)
	{
		CryWarning(VALIDATOR_MODULE_ANIMATION, VALIDATOR_ERROR, "AnimationGraph: XML node is not an AnimationRadnomizer node. Cannot load data. Aborting.");
		return false;
	}

	if (!m_animations.empty())
	{
		m_animations.clear();
	}

	// go through all children
	int childCount = node->getChildCount();
	for (int i = 0; i < childCount; ++i)
	{
		XmlNodeRef child = node->getChild(i);

		// Could be a parameter's node
		if (m_generalParams.LoadFromXml(child))
			continue;

		// Try to load a random Animation Asset
		CRandomAnim* newAnim = new CRandomAnim();
		if (newAnim->LoadFromXml(child))
			m_animations.push_back(*newAnim);

		SAFE_DELETE(newAnim);
	}

	// final step after loading all animations
	CalculateProbabilitySum();

	return true;
}


void CAnimationRandomizer::SerializeAsFile( bool reading, AG_FILE *file )
{
	FileSerializationHelper h(reading, file);

	m_generalParams.SerializeAsFile(reading, file);

	int animCount = m_animations.size();
	h.Value(&animCount);
	m_animations.resize(animCount);

	for (int i = 0; i < animCount; ++i)
	{
		m_animations[i].SerializeAsFile(reading, file);
	}

	CalculateProbabilitySum();
}

void CAnimationRandomizer::CalculateProbabilitySum()
{
	m_probabilitySum = 0;
	int animCount = m_animations.size();
	for (int i = 0; i < animCount; ++i)
		m_probabilitySum += m_animations[i].probability;
}

//////////////////////////////////////////////////////////////////////////


void CRandomGeneralParams::SerializeAsFile( bool reading, AG_FILE *file )
{
	FileSerializationHelper h(reading, file);

	h.Value(&switchAssetsOverTime);
	h.Value(&timeAlignBetweenAssets);
	h.Value(&switchProbabilitySeconds);
	h.Value(&initialDelay);
}

void CRandomAnim::SerializeAsFile( bool reading, AG_FILE *file )
{
	FileSerializationHelper h(reading, file);

	h.Value(&canBeStartedTwiceInARow);
	h.Value(&looping);
	h.Value(&allowInterrupt);
	h.Value(&probability);
	h.Value(&transitionTime);
	h.Value(&animName);
}
