////////////////////////////////////////////////////////////////////////////////////////////////////
//
//	Crytek Character Parts Manager source code
//	
//	History:
//	10/12/2007 - Created by Maxim Dyachenko
//
//  Contains:
//    Character parts manager interface
/////////////////////////////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"

#ifdef INCLUDE_FACEGEN
#include <CryHeaders.h>

#include <CryCharMorphParams.h>
#include "CharacterPartsManager.h"
#include <../CrySystem/md5.h>
#if !defined(PS3) && !(defined(XENON) && defined(_LIB))
	#include <../CrySystem/md5.c>
#endif
#include <ILevelSystem.h>
#include <IFaceGen.h>

static const char* CC_PHYS_PART_ATTACHMENT_NAME = "$CCPhys$";
static const char* CC_PHYS_MODELNAME = "CCPhys";
static const char* CC_VISFLAG_PART_ATTACHMENT_PREFIX = "$CCVisFlag";
static const char* CC_VISFLAG_PART_ATTACHMENT_FORMAT = "$CCVisFlag%04X$";
static const char* CACHE_VERSION = "0012";

static const int CA_PARTS_PHYSPARTS_START_INDEX = 2000;
static float breast_UpperDamper_Default = 0.05f;
static float breast_LowerDamper_Default = 1.0f;
static float breast_UpperRestRatio_Default = 1.0f;
static float breast_LowerRestRatio_Default = 1.0f;
static float neck_UpperDamper_Default = 0.1f;
static float neck_LowerDamper_Default = 0.3f;

#define CRY_ARRAYSIZE(A) (sizeof(A)/sizeof((A)[0]))

//////////////////////////////////////////////////////////////////////////
static bool isCustomHead(const char* partName,const ICharacterPart* part,EGender gender)
{
  const IPartModel* pModel = part->GetModel(gender);
  bool bcustomHead=(stricmp(partName,"custom_head")==0) && (stricmp(part->GetSlot(),"head")==0);
  if( bcustomHead && pModel && pModel->GetFGTFilename()!=NULL && strlen(pModel->GetFGTFilename())!=0 )
  {
    return true;
  }
  return false;
}
//-------------------------------------------------------------------------
CCompoundCharacter::SAttachedPart::SAttachedPart(const char* name,const char* material,const char* materialName,const char* surface,const char* slot)
{
  m_name = name;
  m_material_id = material;
  m_material_name = materialName;
  m_surface = surface;
  m_slot = slot;
}
//-------------------------------------------------------------------------
CCompoundCharacter::SAttachedPart* CCompoundCharacter::GetAttachedPart(const char* name)
{
  for(int n=0;n<m_attachedParts.size();n++)
  {
    if( m_attachedParts[n].m_name==name )
    {
      return &m_attachedParts[n];
    }
  }
  return NULL;
}
const CCompoundCharacter::SAttachedPart* CCompoundCharacter::GetAttachedPart(const char* name) const
{
  for(int n=0;n<m_attachedParts.size();n++)
  {
    if( m_attachedParts[n].m_name==name )
    {
      return &m_attachedParts[n];
    }
  }
  return NULL;
}
//-------------------------------------------------------------------------
bool SCharacterPartTemplate::ReadTemplate(const char* filename)
{
  XmlNodeRef xeRoot = gEnv->pSystem->LoadXmlFile(filename);
  if (!xeRoot)
    return false;

  m_filename = filename;
  m_name = xeRoot->getAttr("Name");
  if (m_name == "")
    return false;

  m_visibilityMask = 0;

  bool femaleMorphsReaded = false;
  bool femaleSwitchesReaded = false;

  for (int node = 0; node < xeRoot->getChildCount(); ++node)
  {
    XmlNodeRef xeNode = xeRoot->getChild(node);
    const char* nodeTag = xeNode->getTag();

    if (strcmp(nodeTag, "Morphs") == 0)
    {
      // Morphs will be loaded to corresponding array
      std::vector<SPartMorph>* pTargetMorphs;
      if (strcmp(xeNode->getAttr("Gender"), "Female") == 0)
      {
        femaleMorphsReaded = true;
        pTargetMorphs = &m_morphsFemale;
      }
      else
      {
        pTargetMorphs = &m_morphsMale;
      }

      // Reading
      for (int morph = 0; morph < xeNode->getChildCount(); ++morph)
      {
        XmlNodeRef xeMorph = xeNode->getChild(morph);
        pTargetMorphs->push_back(SPartMorph());
        pTargetMorphs->back().SetSwitch(xeMorph->getAttr("Switch"));
        pTargetMorphs->back().SetMorphTarget(xeMorph->getAttr("Target"));
      }
    }
    else if (strcmp(nodeTag, "Switches") == 0)
    {
      // Switches will be loaded to corresponding array
      std::vector<string>* pTargetSwitches;
      if (strcmp(xeNode->getAttr("Gender"), "Female") == 0)
      {
        femaleSwitchesReaded = true;
        pTargetSwitches = &m_switchesFemale;
      }
      else
      {
        pTargetSwitches = &m_switchesMale;
      }

      // Reading
      for (int switchNode = 0; switchNode < xeNode->getChildCount(); ++switchNode)
      {
        XmlNodeRef xeSwitch = xeNode->getChild(switchNode);
        pTargetSwitches->push_back(string(xeSwitch->getAttr("Value")));
      }
    }
    else if (strcmp(nodeTag, "VisibilityMasks") == 0)
    {
      for (int mask = 0; mask < xeNode->getChildCount(); ++mask)
      {
        XmlNodeRef xeMask = xeNode->getChild(mask);
        const char* maskTag = xeMask->getTag();
        if (strcmp(maskTag, "Mask") == 0)
        {
          const char* maskName = xeMask->getAttr("Name");
          if (strcmp(maskName, "VISIBLE_THIRD_PERSON") == 0)
            m_visibilityMask |= VISIBLE_THIRD_PERSON;
          else if (strcmp(maskName, "VISIBLE_FIRST_PERSON") == 0)
            m_visibilityMask |= VISIBLE_FIRST_PERSON;
          else if (strcmp(maskName, "VISIBLE_HIGHQUALITY_THIRD_PERSON") == 0)
            m_visibilityMask |= VISIBLE_HIGHQUALITY_THIRD_PERSON;
          else if (strcmp(maskName, "VISIBLE_SHADOW_GENERATION") == 0)
            m_visibilityMask |= VISIBLE_SHADOW_GENERATION;
        }
        else
        {
          CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Unexpected node type CharacterPart/VisibilityMasks/%s in file %s.", maskTag, filename);
        }
      }
    }
    else
    {
      CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Unexpected node type CharacterPart/%s in file %s.", nodeTag, filename);
    }

  }

  if (femaleMorphsReaded == false)
  {
    m_morphsFemale = m_morphsMale;
  }
  if (femaleSwitchesReaded == false)
  {
    m_switchesFemale = m_switchesMale;
  }
  return true;
}
//-------------------------------------------------------------------------
CCharacterPartsManager::CCharacterPartsManager(CCryAction* pAction)
{
  pAction->RegisterListener(this, "compound_character", FRAMEWORKLISTENERPRIORITY_GAME);
}
//-------------------------------------------------------------------------
void CCharacterPartsManager::RegisterCVars()
{
  // register 
  IConsole *pConsole = gEnv->pConsole;
  pConsole->Register( "cc_mergerDisableAtlas", &m_cc_mergerDisableAtlas, 0, 0, "Enable/Disable atlas generation." );
  pConsole->Register( "cc_mergerUseDiffuseAtlas", &m_cc_mergerUseDiffuseAtlas, 1, 0, "Enable/Disable diffuse atlas only." );
  pConsole->Register( "cc_GroinMaxWeightAngle", &m_cc_groinMaxWeightAngle, 15.0f, 0, "Max angle when groin groin rotation is max." );
  pConsole->Register( "cc_GroinMinWeightAngle", &m_cc_groinMinWeightAngle, 35.0f, 0, "Min angle when groin groin rotation is max." );
  pConsole->Register( "cc_groinSimulation", &m_cc_groinSimulation, 1, 0, "Enables groin simuation." );
  pConsole->Register( "cc_profile", &m_cc_profile, 0, 0, "Profiles compound character: 1-textures profile." );
  pConsole->Register( "cc_breast_simulation", &m_cc_breast_simulation, 1, 0, "Enabled/Disables breast simulation." );
  pConsole->Register( "cc_breast_gravity", &m_cc_breast_gravity, -9.0f, 0, "Gravity for breast simulation." );
  pConsole->Register( "cc_breast_minDist", &m_cc_breast_minDist, 0.05f, 0, "Min distance when breast is not corrected after simulation." );
  pConsole->Register( "cc_breast_maxDist", &m_cc_breast_maxDist, 0.20f, 0, "Max distance when breast is fully corrected to base." );
  pConsole->Register( "cc_breast_upperDamper", &m_cc_breast_upperDamper, breast_UpperDamper_Default, 0, "Upper damper for breast simulation." );
  pConsole->Register( "cc_breast_lowerDamper", &m_cc_breast_lowerDamper, breast_LowerDamper_Default, 0, "Lower damper for breast simulation." );
  pConsole->Register( "cc_breast_upperRestRatio", &m_cc_breast_upperRestRatio, breast_UpperRestRatio_Default, 0, "Upper damper rest len ratio for breast simulation." );
  pConsole->Register( "cc_breast_lowerRestRatio", &m_cc_breast_lowerRestRatio, breast_LowerRestRatio_Default, 0, "Lower damper rest len ratio for breast simulation." );
  pConsole->Register( "cc_neckSimulation", &m_cc_neckSimulation, 1, 0, "Enables neck protection simulation." );
  pConsole->Register( "cc_neckAngleMultiplier", &m_cc_neckAngleMultilier, 2.0f, 0, "Neck angle multiplier." );
  pConsole->Register( "cc_neckAngleBias", &m_cc_neckAngleBias, -5.0f, 0, "Neck angle bias." );
  pConsole->Register( "cc_neck_upperDamper", &m_cc_neck_upperDamper, neck_UpperDamper_Default, 0, "Upper damper for neck simulation." );
  pConsole->Register( "cc_neck_lowerDamper", &m_cc_neck_lowerDamper, neck_LowerDamper_Default, 0, "Lower damper for neck simulation." );
  pConsole->Register( "cc_neck_minDist", &m_cc_neck_minDist, 0.05f, 0, "Min distance when neck is not corrected after simulation." );
  pConsole->Register( "cc_neck_maxDist", &m_cc_neck_maxDist, 0.20f, 0, "Max distance when neck is fully corrected to base." );
  pConsole->Register( "cc_neckAngleMultiplier_LR", &m_cc_neckAngleMultilierLR, 1.0f, 0, "Neck LR angle multiplier." );
  pConsole->Register( "cc_scale_head_factor", &cc_scale_head_factor, 0.1f, 0, "Scale factor for head scaling." );
  pConsole->RegisterString( "fg_default_preset", "defaultMale", 0, "FaceGen default head preset." );

}
//-------------------------------------------------------------------------
void CCharacterPartsManager::UnRegisterCVars()
{
  // register 
  IConsole *pConsole = gEnv->pConsole;
  pConsole->UnregisterVariable( "cc_mergerUseDiffuseAtlas", true );
  pConsole->UnregisterVariable( "cc_mergerDisableAtlas", true );
  pConsole->UnregisterVariable( "cc_GroinMaxWeightAngle", true );
  pConsole->UnregisterVariable( "cc_GroinMinWeightAngle", true );
  pConsole->UnregisterVariable( "cc_groinSimulation", true );
  pConsole->UnregisterVariable( "cc_profile", true );
  pConsole->UnregisterVariable( "cc_breast_simulation", true );
  pConsole->UnregisterVariable( "cc_breast_minDist", true );
  pConsole->UnregisterVariable( "cc_breast_maxDist", true );
  pConsole->UnregisterVariable( "cc_breast_upperDamper", true );
  pConsole->UnregisterVariable( "cc_breast_lowerDamper", true );
  pConsole->UnregisterVariable( "cc_breast_upperRestRatio", true );
  pConsole->UnregisterVariable( "cc_breast_lowerRestRatio", true );
  pConsole->UnregisterVariable( "cc_breast_gravity", true );
  pConsole->UnregisterVariable( "cc_neckAngleMultiplier", true );
  pConsole->UnregisterVariable( "cc_neckAngleBias", true );
  pConsole->UnregisterVariable( "cc_neck_upperDamper", true );
  pConsole->UnregisterVariable( "cc_neck_lowerDamper", true );
  pConsole->UnregisterVariable( "cc_neck_minDist", true );
  pConsole->UnregisterVariable( "cc_neck_maxDist", true );
  pConsole->UnregisterVariable( "cc_neckAngleMultiplier_LR", true );
  pConsole->UnregisterVariable( "cc_scale_head_factor", true );
  //
  pConsole->UnregisterVariable( "fg_default_preset", true );
  gEnv->pGame->GetIGameFramework()->UnregisterListener(this);
}
//-------------------------------------------------------------------------
void CCharacterPartsManager::ScanForParts(const char* folderName)
{
  m_templates.resize(0);
  m_parts.resize(0);

  ScanForParts(folderName, true);
  ScanForParts(folderName, false);
}

//-------------------------------------------------------------------------
void CCharacterPartsManager::AddNewPart(const char* filename)
{
  LoadPart(filename);
}

//-------------------------------------------------------------------------
void CCharacterPartsManager::ScanForParts(const char* folderName, bool scanForTemplates)
{
  string folder = folderName;
  string search = folderName;
  search += "/*.*";

  ICryPak *pPak = gEnv->pSystem->GetIPak();

  _finddata_t fd;
  intptr_t handle = pPak->FindFirst(search.c_str(), &fd);

  if (handle > -1)
  {
    do
    {
      if (!strcmp(fd.name, ".") || !strcmp(fd.name, ".."))
        continue;

      if (fd.attrib & _A_SUBDIR)
      {
        string subFolder = folder + "/" + fd.name;
        ScanForParts(subFolder.c_str(), scanForTemplates);
      }
      else
      {
        if (scanForTemplates && stricmp(PathUtil::GetExt(fd.name), "cpt") == 0)
        {
          string xmlFile = folder + string("/") + string(fd.name);

          SCharacterPartTemplate templateData;
          if (templateData.ReadTemplate(xmlFile.c_str()))
            m_templates.push_back(templateData);
        }
        else if (scanForTemplates == false && stricmp(PathUtil::GetExt(fd.name), "xml") == 0)
        {
          string xmlFile = folder + string("/") + string(fd.name);
          LoadPart(xmlFile);
        }
      }
    }
    while (pPak->FindNext(handle, &fd) >= 0);
  }
}

//-------------------------------------------------------------------------
void CCharacterPartsManager::LoadPart(const char* filename)
{
  XmlNodeRef xeRoot	= gEnv->pSystem->LoadXmlFile(filename);
  if (!xeRoot)
  {
    CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "CryAction: Character part loading failed for file %s.", filename);
    return;
  }

  SCharacterPart newPart;

  newPart.SetFilename(filename);
  newPart.SetName(xeRoot->getAttr("Name"));
  newPart.SetSlot(xeRoot->getAttr("Slot"));
  newPart.SetVisibilityMask(0);

  if (*newPart.GetName() == 0 || *newPart.GetSlot() == 0)
  {
    CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "CryAction: Skipping character part file %s because of wrong format.", filename);
    return; // wrong format
  }

  // Check if we already has part with similar name
  for (int part = 0; part < m_parts.size(); ++part)
  {
    if (strcmp(m_parts[part].GetName(), newPart.GetName()) == 0)
    {
      CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "CryAction: Skipping character part with already defined name %s from file %s. Previous definition is in file %s.", newPart.GetName(), filename, m_parts[part].GetFilename());
      return;
    }
  }

  // Load template data
  newPart.SetTemplateName(xeRoot->getAttr("Template"));
  const ICharacterPartTemplate* pTemplate = GetTemplateByName(newPart.GetTemplateName());
  if (pTemplate)
    newPart.Set(pTemplate);

  for (int node = 0; node < xeRoot->getChildCount(); ++node)
  {   
    XmlNodeRef xeNode = xeRoot->getChild(node);
    const char* nodeTag = xeNode->getTag();

    if (strcmp(nodeTag, "Model") == 0) 
    {
      SPartModel* pTargetModel;
      if (strcmp(xeNode->getAttr("Gender"), "Female") == 0)
        pTargetModel = (SPartModel*)newPart.GetModel(GENDER_FEMALE);
      else
        pTargetModel = (SPartModel*)newPart.GetModel(GENDER_MALE);

      pTargetModel->SetFilename(xeNode->getAttr("File"));
      pTargetModel->SetFGTFilename(xeNode->getAttr("FGTFile"));
      pTargetModel->SetPocketsMaskFilename(xeNode->getAttr("PMask"));

      // Read materials
      if (XmlNodeRef xeMaterials = xeNode->findChild("Materials"))
      {
        pTargetModel->ReserveMaterials(xeMaterials->getChildCount());
        for (int material = 0; material < xeMaterials->getChildCount(); ++material)
        {
          XmlNodeRef xeMaterial = xeMaterials->getChild(material);
          if (strcmp(xeMaterial->getTag(), "Material") == 0)
          {
            string name = xeMaterial->getAttr("Name");
            string file = xeMaterial->getAttr("File");
            pTargetModel->AddMaterial(name, file);
          }
          else
          {
            CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Unexpected node type CharacterPart/Model/Materials/%s in file %s.", xeMaterial->getTag(), filename);
          }
        }
      }

      // Read morphs
      if (XmlNodeRef xeMorphs = xeNode->findChild("Morphs"))
      {
        pTargetModel->ReserveMorphs(xeMorphs->getChildCount());
        for (int morph = 0; morph < xeMorphs->getChildCount(); ++morph)
        {
          XmlNodeRef xeMorph = xeMorphs->getChild(morph);
          string switchName = xeMorph->getAttr("Switch");
          string morphTarget = xeMorph->getAttr("Target");

          // Try to find already defined switch
          bool found = false;
          for (int oldMorph = 0; oldMorph < pTargetModel->GetNumMorphs(); ++oldMorph)
          {
            if (switchName == pTargetModel->GetMorphByIndex(oldMorph)->GetSwitch())
            {
              //CryLog("CharacterPart: Not unique morphs switch %s in file %s.", newMorph.m_switch.c_str(), filename);
              // Update old target with new one
              pTargetModel->GetMorphByIndex(oldMorph)->SetMorphTarget(morphTarget.c_str());
              found = true;
              break;
            }
          }
          if (found == false)
            pTargetModel->AddMorph(switchName.c_str(), morphTarget.c_str());
        }
      }

      // Read switches
      if (XmlNodeRef xeSwitches = xeNode->findChild("Switches"))
      {
        pTargetModel->ReserveSwitches(xeSwitches->getChildCount());
        for (int switchNode = 0; switchNode < xeSwitches->getChildCount(); ++switchNode)
        {
          XmlNodeRef xeSwitch = xeSwitches->getChild(switchNode);
          string newSwitch = xeSwitch->getAttr("Value");

          bool found = false;
          for (int oldSwitch = 0; oldSwitch < pTargetModel->GetNumSwitches(); ++oldSwitch)
            if (newSwitch == pTargetModel->GetSwitchByIndex(oldSwitch))
            {
              found = true;
              break;
            }

            if (found == false)
              pTargetModel->AddSwitch(newSwitch.c_str());
        }
      }

      // Read sockets
      if (XmlNodeRef xeSockets = xeNode->findChild("Sockets"))
      {
        pTargetModel->ReserveSockets(xeSockets->getChildCount());
        for (int socketNode = 0; socketNode < xeSockets->getChildCount(); ++socketNode)
        {
          XmlNodeRef xeSocket = xeSockets->getChild(socketNode);

          int edge = 0;
          float angle = 0.0f;
          float damping = 0.0f;
          xeSocket->getAttr("edge",edge);
          xeSocket->getAttr("angle",angle);
          xeSocket->getAttr("damping",damping);
          //
          IPartSocket* pNewSocket = pTargetModel->AddSocket(xeSocket->getAttr("name"), edge, angle, damping);

          for (int node1 = 0; node1 < xeSocket->getChildCount(); ++node1)
          {
            XmlNodeRef xeNode1 = xeSocket->getChild(node1);
            if (xeNode1->isTag("AllowedTypes"))
            {
              pNewSocket->ReserveAllowedTypes(xeSocket->getChildCount());
              for (int node2 = 0; node2 < xeNode1->getChildCount(); ++node2)
              {
                XmlNodeRef xeNodeType = xeNode1->getChild(node2);
                if (xeNodeType->isTag("type"))
                {
                  pNewSocket->AddAllowedType(xeNodeType->getAttr("name"));
                }
              }

              break;
            }
          }
        }
      }
    }
    else if (strcmp(nodeTag, "VisibilityMasks") == 0)
    {
      uint32 mask = 0;
      for (int i = 0; i < xeNode->getChildCount(); ++i)
      {
        XmlNodeRef xeMask = xeNode->getChild(i);
        const char* maskTag = xeMask->getTag();
        if (strcmp(maskTag, "Mask") == 0)
        {
          const char* maskName = xeMask->getAttr("Name");
          if (strcmp(maskName, "VISIBLE_THIRD_PERSON") == 0)
            mask |= VISIBLE_THIRD_PERSON;
          else if (strcmp(maskName, "VISIBLE_FIRST_PERSON") == 0)
            mask |= VISIBLE_FIRST_PERSON;
          else if (strcmp(maskName, "VISIBLE_HIGHQUALITY_THIRD_PERSON") == 0)
            mask |= VISIBLE_HIGHQUALITY_THIRD_PERSON;
          else if (strcmp(maskName, "VISIBLE_SHADOW_GENERATION") == 0)
            mask |= VISIBLE_SHADOW_GENERATION;
        }
        else
        {
          CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Unexpected node type CharacterPart/VisibilityMasks/%s in file %s.", maskTag, filename);
        }
      }

      newPart.SetVisibilityMask(mask);
    }
    else
    {
      CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Unexpected node type CharacterPart/%s in file %s.", nodeTag, filename);
    }
  }

  m_parts.push_back(newPart);
}

//-------------------------------------------------------------------------
const ICharacterPart* CCharacterPartsManager::GetPartByName(const char* name) const
{
  for (int part = 0; part < m_parts.size(); ++part)
    if (strcmp(m_parts[part].GetName(), name) == 0)
      return &m_parts[part];

  return NULL;
}

//-------------------------------------------------------------------------
const ICharacterPart* CCharacterPartsManager::GetPartByIndex(int index) const
{
  if (m_parts.size() <= index)
    return NULL;
  return &m_parts[index];
}

//-------------------------------------------------------------------------
int CCharacterPartsManager::GetNumParts() const
{
  return m_parts.size();
}

//-------------------------------------------------------------------------
void CCharacterPartsManager::ReloadPart(const char* name)
{
  int partIndex = GetPartIndexByName(name);
  if (partIndex == -1)
    return;

  string partFilename = m_parts[partIndex].GetFilename();
  m_parts.erase(m_parts.begin() + partIndex);

  LoadPart(partFilename);
}

//-------------------------------------------------------------------------
void CCharacterPartsManager::RedefinePart(const char* name, const ICharacterPart* pNewPartData)
{
  int partIndex = GetPartIndexByName(name);
  if (partIndex == -1)
    return;

  m_parts[partIndex].Set(pNewPartData);
}

//-------------------------------------------------------------------------
int CCharacterPartsManager::GetPartIndexByName(const char* name) const
{
  for (int part = 0; part < m_parts.size(); ++part)
    if (strcmp(m_parts[part].GetName(), name) == 0)
      return part;
  return -1;
}

//-------------------------------------------------------------------------
int CCharacterPartsManager::GetNumTemplates() const
{
  return m_templates.size();
}

//-------------------------------------------------------------------------
const ICharacterPartTemplate* CCharacterPartsManager::GetTemplateByIndex(int index) const
{
  if (m_templates.size() <= index)
    return NULL;
  return &m_templates[index];
}

//-------------------------------------------------------------------------
const ICharacterPartTemplate* CCharacterPartsManager::GetTemplateByName(const char* name) const
{
  for (int partTemplate = 0; partTemplate < m_templates.size(); ++partTemplate)
  {
    if (strcmp(m_templates[partTemplate].GetName(), name) == 0)
      return &m_templates[partTemplate];
  }

  return NULL;
}
//-------------------------------------------------------------------------
CCompoundCharacter::~CCompoundCharacter()
{
  //
  CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->UnRegisterCompoundCharacter(this);
  //
  DetachAllParts();
}
//-------------------------------------------------------------------------
void CCompoundCharacter::SetGender(const EGender& gender)
{ 
  //
  WarnCached("SetGender");
  //
  if( m_gender!=gender )
  {
    m_gender = gender; 
    if( !(m_flags & ICompoundCharacter::eFlag_CommitOnCache) )
    {
      ApplyGenderToAttachedParts();
      UpdateMorphs();
    }
    //
    InitSimulation();
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacter::SetFatness(const float& fFatness) 
{ 
  //
  WarnCached("SetFatness");
  //
  m_fFatness = fFatness; 
  if( !(m_flags & ICompoundCharacter::eFlag_CommitOnCache) )
  {
    UpdateMorphs();
  }
  //
  InitSimulation();
}
//-------------------------------------------------------------------------
void CCompoundCharacter::SetNeckFatness(const float& fFatness) 
{ 
  if( m_fNeckFatness!=fFatness )
    //
    WarnCached("SetNeckFatness");
  //
  m_fNeckFatness = fFatness; 
  if( !(m_flags & ICompoundCharacter::eFlag_CommitOnCache) )
  {
    UpdateMorphs();
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacter::SetScale(const float& fScale)
{
  CCharacterPartsManager* pIPartsManager = static_cast<CCharacterPartsManager*>(gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager());
  //
  m_pInstance->SetUniformScale(fScale);
  // head
  int16 jointid = m_pInstance->GetISkeletonPose()->GetJointIDByName("Bip01 Head");
  if( jointid!=-1 )
  {
    float fHeadScale = 1.0f;
    if( fScale>=1.0f )
    {
      float t = (fScale-1.0f)/0.2f;
      fHeadScale  = +CLAMP(t,0.0f,1.0f)*pIPartsManager->cc_scale_head_factor;
    }
    else
    {
      float t = 1.0f-(fScale-0.8f)/0.2f;
      fHeadScale  = -CLAMP(t,0.0f,1.0f)*pIPartsManager->cc_scale_head_factor;
    }
    m_pInstance->SetCustomScaleForJoint(jointid,true,fHeadScale+1.0f);
  }
  //
  IAttachmentManager* pAttachmentManager = m_pInstance->GetIAttachmentManager();
  int32 count = pAttachmentManager->GetAttachmentCount();
  for (uint32 i=0; i<count; ++i)
  {
    if (IAttachment* pAttachment = pAttachmentManager->GetInterfaceByIndex(i))
    {
      if(IAttachmentObject *pAO = pAttachment->GetIAttachmentObject())
      {
        if(pAO->GetAttachmentType()==IAttachmentObject::eAttachment_Entity)
        {
          CEntityAttachment* pEA = static_cast<CEntityAttachment*>(pAO);
          pEA->SetScale(Vec3(fScale));
        }
      }
    }
  }
}
//-------------------------------------------------------------------------
float CCompoundCharacter::GetScale() const
{
  return m_pInstance->GetUniformScale();
}
//-------------------------------------------------------------------------
bool CCompoundCharacter::AttachPart(const char* partName, const char* material,const char* surface_type)
{
  //
  WarnCached("AttachPart");
  //
  return AttachPartInternal(partName,material,surface_type,false);
}
//-------------------------------------------------------------------------
bool CCompoundCharacter::AttachPartInternal(const char* partName, const char* material,const char* surface_type, bool bUseCache,bool bForceAttach,bool bUpdateMorphs)
{
  //
  ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
  //
  const ICharacterPart* pTargetPart = pIPartsManager->GetPartByName(partName);
  if (pTargetPart == NULL)
    return false;
  //
  const IPartModel* pModel = pTargetPart->GetModel(GetGender());
  if (pModel == NULL)
    return false;
  //
  // Set default material
  string material_filename;
  if (pModel->GetNumMaterials() > 0)
  {
    if (material == NULL || material[0] == 0)
      material = "Default";

    int matIndex = 0;
    for (int mat = 0; mat < pModel->GetNumMaterials(); ++mat)
    {
      if (strcmp(pModel->GetMaterialByIndex(mat)->GetName(), material) == 0)
      {
        matIndex = mat;
        break;
      }
    }
    //
    material_filename = pModel->GetMaterialByIndex(matIndex)->GetFilename();
    ExtractMaterialName(material_filename);
  }
  //
  bool bAttachmentCreated = false;
  //
  IAttachment* pIAttachment = NULL;
  CCharacterPartAttachment* pAttachmentObject = NULL;
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  if( !IsCommitOnCache() || bForceAttach )
  {
    pIAttachment = GetInterfaceByName(pTargetPart->GetName());
    if (pIAttachment)
      return true;
    bAttachmentCreated = true;
    pIAttachment = pManager->CreateAttachment(pTargetPart->GetName(), CA_PART, 0,true,true);
    QuatT ident; ident.SetIdentity();
    pIAttachment->SetAttAbsoluteDefault(ident);

    ICharacterInstance* pIChildCharacter = NULL;

    //
    IMaterial* pChrMtl=NULL;
    string filename = GetPartCHRFileName(partName,&pChrMtl);

    if( bUseCache )
    {
      string cachedFilename = CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->GetCHRFileName(this,partName);
      if (!cachedFilename.empty() && gEnv->pCryPak->IsFileExist(cachedFilename))
      {
        filename = cachedFilename;
      }
    }
    //
    string fileExt = PathUtil::GetExt(filename);
    bool IsCHR = (0 == stricmp(fileExt, "chr"));

    if (IsCHR)
    {
      pIChildCharacter = gEnv->pCharacterManager->CreateInstance(filename);
      if (pIChildCharacter)
      {
        pAttachmentObject = new CCharacterPartAttachment(this,pTargetPart->GetVisibilityMask(),GetSurfaceTypeInternal(surface_type));
        pAttachmentObject->m_pCharInstance = pIChildCharacter;
      }
    }
    else if( filename.size()==0 )
    {
      CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Character Part:%s No model found for gender:%s", filename.c_str(),m_gender==GENDER_MALE ? "Male" : "Female");
      pAttachmentObject = new CCharacterPartAttachment(this,pTargetPart->GetVisibilityMask(), -1);
    }
    if (pAttachmentObject == NULL)
      return false;
    //
    // Set material
    if( pChrMtl )
    {
      pAttachmentObject->SetMaterial(pChrMtl);
    }
    else if (pModel->GetNumMaterials() > 0)
    {
      _smart_ptr<IMaterial> pIMaterial;
      pIMaterial = gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(material_filename.c_str(), false);
      pAttachmentObject->SetMaterial(pIMaterial);
    }
  }
  //
  SAttachedPart* pPart = GetAttachedPart(partName);
  if( !pPart )
  {
    m_attachedParts.push_back(SAttachedPart(partName,material,material_filename.c_str(),surface_type,pTargetPart->GetSlot()));
  }
  else
  {
    pPart->m_material_id = material;
    pPart->m_material_name = material_filename.c_str();
    pPart->m_surface = surface_type;
  }
  //
  if( !IsCommitOnCache() || bForceAttach )
  {
    pIAttachment->AddBinding(pAttachmentObject);
    pManager->ProjectAllAttachment();
    if( bUpdateMorphs )
    {
      UpdateMorphs();
    }
  }
  // reatach all FGT parts
  if( bAttachmentCreated && isCustomHead(partName,pTargetPart,GetGender()) )
  {
    //ReloadPart()
    int numParts = GetNumAttached();
    std::vector<string> partToReload;
    for(int n=0;n<numParts;n++)
    {
      const char* a_partName = GetAttachedPartName(n);
      const ICharacterPart* a_part =  pIPartsManager->GetPartByName(a_partName);
      // should never happens
      if( a_part==NULL )
        continue;
      // part
      const IPartModel* a_pModel = a_part->GetModel(GetGender());
      if( !a_pModel )
        continue;
      if( a_pModel->GetFGTFilename()!=NULL && strlen(a_pModel->GetFGTFilename())!=0 )
      {
        if( strcmp(a_partName,partName)!=0 )
        {
          partToReload.push_back(a_partName);
        }
      }
    }
    for(int n=0;n<partToReload.size();n++)
    {
      ReloadPart(partToReload[n],false);
    }
  }
  //
  return true;
}
//-------------------------------------------------------------------------
int CCompoundCharacter::GetSurfaceTypeInternal(const char* surface_type)
{
  int surfaceType = -1;
  if( surface_type )
  {
    ISurfaceTypeManager* pSurfaceMan = gEnv->p3DEngine->GetMaterialManager()->GetSurfaceTypeManager();
    ISurfaceType* pSurfType = pSurfaceMan->GetSurfaceTypeByName(surface_type);
    if( !pSurfType )
    {
      CryWarning(VALIDATOR_MODULE_PHYSICS, VALIDATOR_WARNING, "Unknown surface type:%s",surface_type);
    }
    else
    {
      surfaceType = pSurfType->GetId();      
    }
  }
  return surfaceType;
}
//-------------------------------------------------------------------------
int CCompoundCharacter::GetNumSocketAttachments(const char* partName) const
{
  const SAttachedPart* pPart = GetAttachedPart(partName);
  if( pPart )
  {
    return pPart->m_sockets.size();
  }
  return 0;
}
//-------------------------------------------------------------------------
const char* CCompoundCharacter::GetSocketName(const char* partName,int socketIndex) const
{
  const SAttachedPart* pPart = GetAttachedPart(partName);
  if( pPart )
  {
    return pPart->m_sockets[socketIndex].m_name;
  }
  return NULL;
}
//-------------------------------------------------------------------------
const char* CCompoundCharacter::GetSocketFilename(const char* partName,int socketIndex) const
{
  const SAttachedPart* pPart = GetAttachedPart(partName);
  if( pPart )
  {
    return pPart->m_sockets[socketIndex].m_filename;
  }
  return NULL;
}
//-------------------------------------------------------------------------
bool CCompoundCharacter::AddSocketAttachment(const char* partName,const char* socketName,const char* BindingPath)
{
  //
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();

  IAttachment* pPartAttachment = GetInterfaceByName(partName);
  CCharacterPartAttachment* charAtt = static_cast<CCharacterPartAttachment*>(pPartAttachment->GetIAttachmentObject());
  if( !charAtt )
    return true;

  CCompoundCharacter::SAttachedPart* pPart = GetAttachedPart(partName);
  if( !pPart )
    return true;
  //
  CCompoundCharacter::SAttachedPart::SFaceAttachmentData* faData = NULL;
  for(size_t s=0;s<pPart->m_faceAttachments.size();s++)
  {
    if( pPart->m_faceAttachments[s].m_name==socketName )
    {
      faData = &pPart->m_faceAttachmentsData[s];
      break;
    }
  }
  if( faData )
  {
    IAttachmentManager* partAttachmentMgr = charAtt->GetICharacterInstance()->GetIAttachmentManager();
    if( partAttachmentMgr )
    {
      // remove old
      partAttachmentMgr->RemoveAttachmentByName(socketName);
      //
      if( strlen(BindingPath)==0 )
        return true;
      //
      IAttachment* socketAtt = partAttachmentMgr->CreateAttachment(socketName,CA_FACE,NULL,false);
      socketAtt->SetAttAbsoluteDefault( faData->m_AttAbsoluteDefault );
      socketAtt->SetFaceNr( faData->m_FaceNr );
      //socketAtt->ProjectAttachment();
      socketAtt->SetAttRelativeDefault( faData->m_AttRelativeDefault );
      //
      //static_cast<CAttachment*>(socketAtt)->m_FaceNr = faData->m_FaceNr;
      //static_cast<CAttachment*>(socketAtt)->m_AttAbsoluteDefault = faData->m_AttAbsoluteDefault;
      //static_cast<CAttachment*>(socketAtt)->m_AttRelativeDefault = faData->m_AttRelativeDefault;
      //
      string fileExt = PathUtil::GetExt( BindingPath );
      //
      bool IsCGF = (0 == stricmp(fileExt,"cgf"));
      if (IsCGF) 
      {
        IStatObj* pIStatObj = gEnv->p3DEngine->LoadStatObj( BindingPath );
        if (pIStatObj)
        {
          CCGFAttachment* pStatAttachment = new CCGFAttachment();
          pStatAttachment->pObj  = pIStatObj;
          IAttachmentObject* pIAttachmentObject = (IAttachmentObject*)pStatAttachment;
          socketAtt->AddBinding(pIAttachmentObject);
          //
          socketAtt->SetAttAbsoluteDefault( faData->m_AttAbsoluteDefault );
          socketAtt->SetFaceNr( faData->m_FaceNr );
          socketAtt->SetAttRelativeDefault( faData->m_AttRelativeDefault );
          //((CAttachment*)socketAtt)->m_AttRelativeDefault	= faData->m_AttRelativeDefault;
          //((CAttachment*)socketAtt)->m_FaceNr			=	faData->m_FaceNr;
          ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
          //
          const ICharacterPart* part = pIPartsManager->GetPartByName(partName);
          const IPartModel* model = part->GetModel(m_gender);
          for(size_t n=0;n<model->GetNumSockets();n++)
          {
            const IPartSocket* socket = model->GetSocketByIndex(n);
            if( strcmp(socket->GetName(), socketName) == 0 )
            {
              socketAtt->SetHingeParams(socket->GetEdge(), socket->GetAngle(), socket->GetDamping());
            }
          }
        }
      }
    }
    //
    pPart = GetAttachedPart(partName);
    if( pPart )
    {
      int foundIndex = -1;
      for(int n=0;n<pPart->m_sockets.size();n++)
      {
        if( pPart->m_sockets[n].m_name==socketName )
        {
          foundIndex = n;
          break;
        }
      }
      if( foundIndex==-1 )
      {
        SAttachedPart::SSocketAttachment att;
        att.m_name = socketName;
        att.m_filename = BindingPath;
        pPart->m_sockets.push_back(att);
      }
      else
      {
        pPart->m_sockets[foundIndex].m_name = socketName;
        pPart->m_sockets[foundIndex].m_filename = BindingPath;
      }
    }
    //
    return true;
  }
  //
  return false;
}

static int hingeAxis = -1;
//-------------------------------------------------------------------------
bool CCompoundCharacter::AddFaceAttachment(const char* partName,const char* faceAttName,const char* BindingPath,const QuatT& pt,bool bUseCenterPointDist)
{
  //
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();

  IAttachment* pAtt = GetInterfaceByName(partName);
  CCharacterPartAttachment* charAtt = static_cast<CCharacterPartAttachment*>(pAtt->GetIAttachmentObject());
  if( !charAtt )
    return false;

  if( !charAtt->GetICharacterInstance() )
    return false;

  CCompoundCharacter::SAttachedPart* pPart = GetAttachedPart(partName);
  if( !pPart )
    return false;
  //
  IAttachmentManager* partAttachmentMgr = charAtt->GetICharacterInstance()->GetIAttachmentManager();
  if( partAttachmentMgr )
  {
    // remove old
    partAttachmentMgr->RemoveAttachmentByName(faceAttName);
    //
    if( strlen(BindingPath)==0 )
      return false;
    //
    ICharacterModel* pObjectModel = charAtt->GetICharacterInstance()->GetICharacterModel();
    CloseInfo cf = pObjectModel->FindClosestPointOnMesh( pt.t,bUseCenterPointDist );
    Vec3 x = (cf.tv1-cf.tv0).GetNormalized();
    Vec3 z = ((cf.tv1-cf.tv0)%(cf.tv0-cf.tv2)).GetNormalized();
    Vec3 y = z%x;
    QuatT TriMat=QuatT::CreateFromVectors(x,y,z,cf.middle);
    //
    IAttachment* socketAtt = partAttachmentMgr->CreateAttachment(faceAttName,CA_FACE,NULL,false);
    socketAtt->SetAttAbsoluteDefault( pt );
    socketAtt->SetFaceNr( cf.FaceNr );
    //socketAtt->ProjectAttachment();
    socketAtt->SetAttRelativeDefault( TriMat.GetInverted() *  pt );
    //
    //static_cast<CAttachment*>(socketAtt)->m_FaceNr = faData->m_FaceNr;
    //static_cast<CAttachment*>(socketAtt)->m_AttAbsoluteDefault = faData->m_AttAbsoluteDefault;
    //static_cast<CAttachment*>(socketAtt)->m_AttRelativeDefault = faData->m_AttRelativeDefault;
    //
    string fileExt = PathUtil::GetExt( BindingPath );
    //
    bool IsCGF = (0 == stricmp(fileExt,"cgf"));
    if (IsCGF) 
    {
      IStatObj* pIStatObj = gEnv->p3DEngine->LoadStatObj( BindingPath );
      if (pIStatObj)
      {
        CCGFAttachment* pStatAttachment = new CCGFAttachment();
        pStatAttachment->pObj  = pIStatObj;
        IAttachmentObject* pIAttachmentObject = (IAttachmentObject*)pStatAttachment;
        socketAtt->AddBinding(pIAttachmentObject);
        //
        socketAtt->SetAttAbsoluteDefault( pt );
        socketAtt->SetFaceNr( cf.FaceNr );
        socketAtt->SetAttRelativeDefault( TriMat.GetInverted() *  pt);
        socketAtt->SetHingeParams(hingeAxis,0,0);
      }
    }
    //
    int foundIndex = -1;
    for(int n=0;n<pPart->m_faceAttachments.size();n++)
    {
      if( pPart->m_faceAttachments[n].m_name==faceAttName )
      {
        foundIndex = n;
        break;
      }
    }
    if( foundIndex==-1 )
    {
      SAttachedPart::SSocketAttachment att;
      att.m_name = faceAttName;
      att.m_filename = BindingPath;
      att.m_pos = pt.t;
      pPart->m_faceAttachments.push_back(att);
    }
    else
    {
      pPart->m_faceAttachments[foundIndex].m_name = faceAttName;
      pPart->m_faceAttachments[foundIndex].m_filename = BindingPath;
      pPart->m_faceAttachments[foundIndex].m_qt = pt;
    }
    //
    return true;
  }
  //
  return false;
}

bool CCompoundCharacter::HasFaceAttachment(const char* partName,const char* pocketName)
{
  //
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();

  IAttachment* att = GetInterfaceByName(partName);
  CCharacterPartAttachment* charAtt = static_cast<CCharacterPartAttachment*>(att->GetIAttachmentObject());
  if( !charAtt )
    return false;

  if( !charAtt->GetICharacterInstance() )
    return false;

  CCompoundCharacter::SAttachedPart* pPart = GetAttachedPart(partName);
  if( !pPart )
    return false;
  //
  IAttachmentManager* partAttachmentMgr = charAtt->GetICharacterInstance()->GetIAttachmentManager();
  if( partAttachmentMgr )
  {
    if( partAttachmentMgr->GetInterfaceByName(pocketName) )
      return true;
  }
  //
  return false;
}

//-------------------------------------------------------------------------
bool CCompoundCharacter::MoveFaceAttachment(const char* partName,const char* faceAttName,const QuatT& pt,bool bUseCenterPointDist)
{
  //
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();

  IAttachment* att = GetInterfaceByName(partName);
  CCharacterPartAttachment* charAtt = static_cast<CCharacterPartAttachment*>(att->GetIAttachmentObject());
  if( !charAtt )
    return false;

  if( !charAtt->GetICharacterInstance() )
    return false;

  CCompoundCharacter::SAttachedPart* pPart = GetAttachedPart(partName);
  if( !pPart )
    return false;
  //
  IAttachmentManager* partAttachmentMgr = charAtt->GetICharacterInstance()->GetIAttachmentManager();
  if( partAttachmentMgr )
  {
    IAttachment* socketAtt = partAttachmentMgr->GetInterfaceByName(faceAttName);
    //
    ICharacterModel* pObjectModel = charAtt->GetICharacterInstance()->GetICharacterModel();
    CloseInfo cf = pObjectModel->FindClosestPointOnMesh( pt.t,bUseCenterPointDist );
    Vec3 x = (cf.tv1-cf.tv0).GetNormalized();
    Vec3 z = ((cf.tv1-cf.tv0)%(cf.tv0-cf.tv2)).GetNormalized();
    Vec3 y = z%x;
    QuatT TriMat=QuatT::CreateFromVectors(x,y,z,cf.middle);
    //
    socketAtt->SetFaceNr( cf.FaceNr );
    socketAtt->SetAttAbsoluteDefault( pt );
    socketAtt->SetAttRelativeDefault( TriMat.GetInverted() *  pt );
    //
    partAttachmentMgr->ProjectAllAttachment();
    //
    socketAtt->SetFaceNr( cf.FaceNr );
    socketAtt->SetAttAbsoluteDefault( pt );
    socketAtt->SetAttRelativeDefault( TriMat.GetInverted() *  pt );
    socketAtt->SetHingeParams(hingeAxis,0,0);
    //
    int foundIndex = -1;
    for(int n=0;n<pPart->m_faceAttachments.size();n++)
    {
      if( pPart->m_faceAttachments[n].m_name==faceAttName )
      {
        foundIndex = n;
        break;
      }
    }
    if( foundIndex!=-1 )
    {
      pPart->m_faceAttachments[foundIndex].m_qt = pt;
    }
    //
    return true;
  }
  //
  return false;
}

//-------------------------------------------------------------------------
ICompoundCharacter* CCharacterPartsManager::LoadCPF(const char* cpfFilename, ICharacterInstance* pInInstance,bool bUseCache) const
{
  ICharacterInstance* pInstance = NULL;
  if( pInInstance==NULL )
  {
    ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
    //
    XmlNodeRef xeRoot = gEnv->pSystem->LoadXmlFile(cpfFilename);
    if (!xeRoot)
      return NULL;

    string baseFile = xeRoot->getAttr("base");
    if (baseFile.length() == 0)
      return NULL;

    string fileExt = PathUtil::GetExt(baseFile);

    if (fileExt == "chr")
      pInstance = gEnv->pCharacterManager->CreateCHRInstance( baseFile ); //Loading CHR file.
    else
    {
      return NULL;
    }
    if (pInstance == NULL)
      return NULL;
  }
  else
  {
    pInstance = pInInstance;
  }
  //
  //
  //
  CCompoundCharacter* pCompoundCharacter = static_cast<CCompoundCharacter*>(CreateCompoundCharacter(pInstance,pInInstance ? "InCHR" : "InChrFile"));
  //
  XmlNodeRef xeRoot = gEnv->pSystem->LoadXmlFile(cpfFilename);
  if (!xeRoot)
    return false;
  //
//  bool result = true;

  EGender gender = GENDER_MALE;
  const char* genderStr = xeRoot->getAttr("gender");
  if (strcmp(genderStr, "Female") == 0)
    gender = GENDER_FEMALE;
  //
  pCompoundCharacter->SetGender(gender);
  //
  //
  for (int node = 0; node < xeRoot->getChildCount(); ++node)
  {
    XmlNodeRef xeNode = xeRoot->getChild(node);
    const char* nodeTag = xeNode->getTag();

    if (strcmp(nodeTag, "Part") == 0)
    {
      string name = xeNode->getAttr("name");
      string material = xeNode->getAttr("material");
      string surface_type = xeNode->getAttr("surface_type");
      //
      const char* surf_type_str = surface_type.empty() ? NULL : surface_type;
      if (pCompoundCharacter->AttachPartInternal(name, material,surf_type_str,bUseCache) == false)
      {
//        result = false;
      }
      else
      {
        for (int node2 = 0; node2 < xeNode->getChildCount(); ++node2)
        {
          XmlNodeRef xeAtta = xeNode->getChild(node2);
          if (xeAtta->isTag("SocketAttachment"))
          {
            string socketName = xeAtta->getAttr("socket");
            string BindingPath = xeAtta->getAttr("model");
            pCompoundCharacter->AddSocketAttachment(name,socketName,BindingPath);
          }
        }
      }
    }
    else
    {
      CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Unexpected node type CharacterDefinition/%s in file %s.", nodeTag, cpfFilename);
    }
  }
  // fatness and scale
  float fatness = 0; xeRoot->getAttr("fatness", fatness);
  float neckfatness = 0; xeRoot->getAttr("neck_fatness", neckfatness);
  float scale = 1; xeRoot->getAttr("scale", scale);
  //
  pInstance->SetUniformScale(scale);
  //
  pCompoundCharacter->SetFatness(fatness);
  pCompoundCharacter->SetNeckFatness(neckfatness);
  //
  pCompoundCharacter->ApplyVisibilityMask(VISIBLE_THIRD_PERSON | VISIBLE_SHADOW_GENERATION);
  //
  //
  CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->RegisterCompoundCharacter(pCompoundCharacter);
  //
  return pCompoundCharacter;
}



ICompoundCharacter* CCharacterPartsManager::CreateCompoundCharacter(ICharacterInstance* pInstance,const char* name,ICompoundCharacter::EFlags flags) const
{
	CCompoundCharacter* pCompoundCharacter = new CCompoundCharacter(pInstance,name,flags);
	//
	CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->RegisterCompoundCharacter(pCompoundCharacter);
	//
	if (pInstance==0)
		return 0;

	pInstance->SetFlags(pInstance->GetFlags() | CS_FLAG_COMPOUND_BASE);
	//
	return pCompoundCharacter;
}
//
bool CCompoundCharacter::isMyAttachment(IAttachment* pAtt) const
{
  if( pAtt==NULL )
    return false;
  if( pAtt->GetType()!=CA_PART )
    return false;
  if( !pAtt->GetIAttachmentObject() )
    return false;
  CCharacterPartAttachment* part = static_cast<CCharacterPartAttachment*>(pAtt->GetIAttachmentObject());
  if( !part )
    return false;
  if( part->m_cc != this )
    return false;
  return true;
}
//
void CCompoundCharacter::RemoveAttachmentByName( const char* szName )
{
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  IAttachment* pAttachment = GetInterfaceByName(szName);
  if( pAttachment )
    pManager->RemoveAttachmentByInterface(pAttachment);
}
//
//
IAttachment* CCompoundCharacter::GetInterfaceByName(  const char* szName ) const
{ 
  int32 idx = GetIndexByName( szName ); 
  if (idx==-1) return 0;
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  return pManager->GetInterfaceByIndex( idx );
};

int32 CCompoundCharacter::GetIndexByName( const char* szName ) const
{ 
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  int num = pManager->GetAttachmentCount();
  for (int i = 0; i < num; i++)	
  {
    IAttachment* pA = pManager->GetInterfaceByIndex(i);
    if(stricmp(pA->GetName(),szName)==0 && isMyAttachment(pA)) 
    {
      return i;
    }
  }
  return -1;
}
//-------------------------------------------------------------------------
bool CCompoundCharacter::IsCommitOnCache()
{
  if( m_flags & ICompoundCharacter::eFlag_CommitOnCache )
  {
    return true;
  }
  return false;
}
bool CCompoundCharacter::WarnCached(const char* msg)
{
  if( m_bCached )
  {
    CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "CC: Character is cached. Msg:%s",msg ? msg : "");
    return true;
  }
  return false;
}
bool CCompoundCharacter::WarnCommitOnCache(const char* msg)
{
  if( IsCommitOnCache() )
  {
    CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "CC: Character is CommitOnCache. Msg:%s",msg);
    return true;
  }
  return false;
}

//-------------------------------------------------------------------------
void CCompoundCharacter::DetachPart(const char* name)
{
  //
  WarnCached("DetachPart");
  //
  DetachPartInternal(name,true);
}

//-------------------------------------------------------------------------
void CCompoundCharacter::DetachPartInternal(const char* name,bool bDeletePhysics,bool bIgnoreCommitOnCache,bool bUpdateMorphs)
{
  for(int n=0;n<m_attachedParts.size();n++)
  {
    if( m_attachedParts[n].m_name==name )
    {
      m_attachedParts.erase(m_attachedParts.begin()+n);
      break;
    }
  }
  //
  if( bIgnoreCommitOnCache || !IsCommitOnCache() )
  {
    IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
    if( bDeletePhysics )
    {
      IAttachment* pIAttachment = GetInterfaceByName(name);
      if ( isMyAttachment(pIAttachment) )
      {
        IPhysicalWorld *pPhysWorld = gEnv->pPhysicalWorld;
        if (pPhysWorld)
        {
          CCharacterPartAttachment* part = static_cast<CCharacterPartAttachment*>(pIAttachment->GetIAttachmentObject());
          if(part && part->m_pRootPhysicalEntity)
          {
            for(size_t n=0;n<part->m_addedPartsList.size();n++)
            {
              part->m_pRootPhysicalEntity->RemoveGeometry(part->m_addedPartsList[n].m_partId);
            }
            part->m_addedPartsList.resize(0);
          }
        }
      }
    }
    RemoveAttachmentByName(name);
    pManager->ProjectAllAttachment();
    if( bUpdateMorphs )
    {
      UpdateMorphs();
    }
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacter::DetachAllParts()
{
  //
  m_bCached = false;
  //
  CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->UnScheduleCharacterJobs(this);
  //
  std::vector<string> partsToRemove;
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  int numAttached = pManager->GetAttachmentCount();
  for (int attach = 0; attach < numAttached; attach++)
  {
    IAttachment* att = pManager->GetInterfaceByIndex(attach);
    if ( isMyAttachment(att) )
    {
      partsToRemove.push_back(att->GetName());
    }
  }
  //
  for(int n=0;n<partsToRemove.size();n++)
  {
    DetachPartInternal(partsToRemove[n],true,true);
  }
  //
  m_attachedParts.resize(0);
  //
//  pManager->ProjectAllAttachment();
//  UpdateMorphs();
}
//-------------------------------------------------------------------------
void CCompoundCharacter::ReAttachPart(const char* name)
{
  //
  WarnCached("ReAttachPart");
  //
  if( WarnCommitOnCache("ReAttachPart is not valid for CommitOnCache CC") )
    return;
  //
  ReloadPart(name,false,false,true);
}
//-------------------------------------------------------------------------
void CCompoundCharacter::ReloadPart(const char* name,bool bUseCache,bool bIgnoreCommitOnCache,bool bUpdateMorphs)
{
  //
  if( !bUseCache )
  {
    WarnCached("ReloadPart");
  }
  //
  if( !bIgnoreCommitOnCache )
  {
    if( WarnCommitOnCache("ReloadPart is not valid for CommitOnCache CC") )
      return;
  }
  //
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();

  std::vector<CCharacterPartAttachment::AddedPhysicsParts> oldAddadPartList;
  IAttachment* pIAttachment = GetInterfaceByName(name);
  if( pIAttachment && pIAttachment->GetType() != CA_PART )
  {
    CryLog("Should never happens");
    return;
  }
  CCharacterPartAttachment* part = NULL;
  if (pIAttachment != NULL)
  {
    // 
    // create physics
//    IPhysicalEntity* pEnt = NULL;
    //
    part = static_cast<CCharacterPartAttachment*>(pIAttachment->GetIAttachmentObject());
    if(part && part->m_pRootPhysicalEntity)
    {
//      pEnt = part->m_pRootPhysicalEntity;
      oldAddadPartList = part->m_addedPartsList;
    }
  }
  //
  SAttachedPart partBefore;
  SAttachedPart* pPartBefore = GetAttachedPart(name);
  if( pPartBefore )
  {
    partBefore = *pPartBefore;
  }
  // remember currect chr (because in almost 90% new chr will be the same)
  _smart_ptr<ICharacterInstance> pOldCharacter;
  if( pIAttachment )
  {
    if( pIAttachment->GetIAttachmentObject() )
    {
      pOldCharacter = pIAttachment->GetIAttachmentObject()->GetICharacterInstance();
    }
    DetachPartInternal(name,false,bIgnoreCommitOnCache,false);
  }
  AttachPartInternal(name, partBefore.m_material_id, partBefore.m_surface,bUseCache,bIgnoreCommitOnCache,bUpdateMorphs);
  //
  pOldCharacter = NULL;
  //
  for(int s=0;s<partBefore.m_sockets.size();s++)
  {
    AddSocketAttachment(name,partBefore.m_sockets[s].m_name,partBefore.m_sockets[s].m_filename);
  }
  for(int s=0;s<partBefore.m_faceAttachments.size();s++)
  {
    AddFaceAttachment(name,partBefore.m_faceAttachments[s].m_name,partBefore.m_faceAttachments[s].m_filename,partBefore.m_faceAttachments[s].m_qt,partBefore.m_faceAttachments[s].m_bUseCenterPoint);
  }
  //
  pIAttachment = GetInterfaceByName(name);
  if (pIAttachment == NULL || pIAttachment->GetType() != CA_PART)
    return;
  //
  part = static_cast<CCharacterPartAttachment*>(pIAttachment->GetIAttachmentObject());
  if( part )
  {
    part->m_addedPartsList = oldAddadPartList;
  }
  //
}
bool CCompoundCharacter::SetMergedPhysics(const char* name)
{
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  // remove existing cc phys
  if( IAttachment* pAttachment = GetInterfaceByName(CC_PHYS_PART_ATTACHMENT_NAME) )
  {
    CCharacterPartAttachment* pObject = static_cast<CCharacterPartAttachment*>(pAttachment->GetIAttachmentObject());
    if( pObject )
    {
      if( pObject->m_pCharInstance )
      {
        if( strcmp(pObject->m_pCharInstance->GetFilePath(),name)==0 )
        {
          return true;
        }
      }
    }
    pManager->RemoveAttachmentByInterface(pAttachment);
  }
  // move physics from parts to merged physics attachment
  // create special physics attachment
  IAttachment* pIAttachment = pManager->CreateAttachment(CC_PHYS_PART_ATTACHMENT_NAME, CA_PART, 0);
  pIAttachment->HideAttachment(true);// hide physics attachment
  QuatT ident; ident.SetIdentity();
  pIAttachment->SetAttAbsoluteDefault(ident);
  //
  CCharacterPartAttachment* pAttachmentObject = NULL;
  ICharacterInstance* pIChildCharacter = gEnv->pCharacterManager->CreateCHRInstance(name);
  if (pIChildCharacter)
  {
    std::map<string,int> surfTypes;
    for(int p=0;p<m_attachedParts.size();p++)
    {
      surfTypes[m_attachedParts[p].m_name] = GetSurfaceTypeInternal(m_attachedParts[p].m_surface);
    }
    pAttachmentObject = new CCharacterPartAttachment(this,surfTypes);
    pAttachmentObject->m_pCharInstance = pIChildCharacter;
    //
    // copy physics from parts to phys att
    pAttachmentObject->m_addedPartsList.clear();
    int numAttached = GetNumAttachedInternal();
    for (int attach = 0; attach < numAttached; attach++)
    {
      CCharacterPartAttachment* pPartAttachmentObject = static_cast<CCharacterPartAttachment*>(GetAttachedPartInfoInternal(attach));
      if( pPartAttachmentObject )
      {
        for( int i=0;i<pPartAttachmentObject->m_addedPartsList.size();i++)
        {
          pAttachmentObject->m_pRootPhysicalEntity = pPartAttachmentObject->m_pRootPhysicalEntity;
          pAttachmentObject->m_addedPartsList.push_back(pPartAttachmentObject->m_addedPartsList[i]);
        }
        pPartAttachmentObject->m_addedPartsList.clear();
      }
    }
    //
    pIAttachment->AddBinding(pAttachmentObject);
  }
  else
  {
    RemoveAttachmentByName(CC_PHYS_PART_ATTACHMENT_NAME);
    CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Character Physics:%s not found", name);
    return false;
  }
  return true;
}
//-------------------------------------------------------------------------
bool CCompoundCharacter::SetMergedVisibilityFlag(const char* filename,uint32 VisFlagsAnd,uint32 VisFlag,IMaterial* pMtl)
{
  _smart_ptr<IMaterial> refHolder = pMtl;
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  // remove existing cc phys
  string visFlagAttName;
  visFlagAttName.Format(CC_VISFLAG_PART_ATTACHMENT_FORMAT,VisFlag);
  if( IAttachment* pAttachment = GetInterfaceByName(visFlagAttName) )
  {
    pManager->RemoveAttachmentByInterface(pAttachment);
  }
  // move physics from parts to merged physics attachment
  // create special physics attachment
  IAttachment* pIAttachment = pManager->CreateAttachment(visFlagAttName, CA_PART, 0);
  QuatT ident; ident.SetIdentity();
  pIAttachment->SetAttAbsoluteDefault(ident);
  //
  CCharacterPartAttachment* pAttachmentObject = NULL;
  ICharacterInstance* pIChildCharacter = gEnv->pCharacterManager->CreateCHRInstance(filename);
  if (pIChildCharacter)
  {
    pAttachmentObject = new CCharacterPartAttachment(this,VisFlag,-1);
    pAttachmentObject->m_pCharInstance = pIChildCharacter;
    pAttachmentObject->SetMaterial(pMtl);
    pIAttachment->AddBinding(pAttachmentObject);
    // remove all flags from all parts
    std::vector<string> attToRemove;
    int numParts = GetNumAttachedInternal();
    for(int n=0;n<numParts;n++)
    {
      //
      ICharacterPartAttachment* pCharAttachment = const_cast<ICharacterPartAttachment*>(GetAttachedPartInfoInternal(n));
      if( pCharAttachment==NULL )
        continue;
      if( (pCharAttachment->GetVisibilityMask() & VisFlagsAnd)==VisFlag )
      {
        pCharAttachment->SetVisibilityMask(pCharAttachment->GetVisibilityMask() & (~VisFlag));
        if( pCharAttachment->GetVisibilityMask()==0 )
        {
          attToRemove.push_back(GetAttachedPartName(n));
        }
      }
    }
    for(int n=0;n<attToRemove.size();n++)
    {
      RemoveAttachmentByName(attToRemove[n]);
    }
  }
  else
  {
    RemoveAttachmentByName(visFlagAttName);
    CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Character Vis Flag:%s not found", filename);
    return false;
  }
  return true;
}
//-------------------------------------------------------------------------
void CCompoundCharacter::ApplyVisibilityMask(int mask)
{
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  pManager->ApplyVisibilityMask(mask);
}
//-------------------------------------------------------------------------
int CCompoundCharacter::GetNumAttached() const
{
  return m_attachedParts.size();
}

//-------------------------------------------------------------------------
void CCompoundCharacter::ChangeAttachedPartMaterial(const char* name, const char* material)
{
  //
  if( WarnCommitOnCache("ChangeAttachedPartMaterial is not valid for CommitOnCache CC") )
    return;
  //
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
  //
  IAttachment* pIAttachment = GetInterfaceByName(name);
  if (pIAttachment == NULL || pIAttachment->GetType() != CA_PART)
    return;

  CCharacterPartAttachment* pCharAttachment = (CCharacterPartAttachment*)pIAttachment->GetIAttachmentObject();

  const ICharacterPart* pCharacterPart = pIPartsManager->GetPartByName(pIAttachment->GetName());
  if( pCharacterPart )
  {
    const IPartModel* pModel = pCharacterPart->GetModel(m_gender);
    if (pModel->GetNumMaterials() == 0)
      return;

    if (material == NULL || material[0] == 0)
      material = "Default";

    int matIndex = 0;
    for (int mat = 0; mat < pModel->GetNumMaterials(); ++mat)
    {
      if (strcmp(pModel->GetMaterialByIndex(mat)->GetName(), material) == 0)
      {
        matIndex = mat;
        break;
      }
    }

    string filename = pModel->GetMaterialByIndex(matIndex)->GetFilename();
    ExtractMaterialName(filename);

    _smart_ptr<IMaterial> pINewMaterial = gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(filename.c_str(), false);
    if( pINewMaterial )
    {
      pCharAttachment->SetMaterial(pINewMaterial);
      //
      if( SAttachedPart* part = GetAttachedPart(name) )
      {
        part->m_material_id = pModel->GetMaterialByIndex(matIndex)->GetName();
        part->m_material_name = pINewMaterial->GetName();
      }
    }
  }
}

const char* CCompoundCharacter::GetAssignedMaterial(const char* partName)
{
  if( SAttachedPart* part = GetAttachedPart(partName) )
  {
    return part->m_material_id;
  }
  return NULL;
}
/*
//-------------------------------------------------------------------------
const ICharacterPartAttachment* CCompoundCharacter::GetAttachedPartInfo(int index) const
{
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();

  int numChecked = 0;

  int numAttached = pManager->GetAttachmentCount();
  for (int attach = 0; attach < numAttached; ++attach)
  {
    IAttachment* att = pManager->GetInterfaceByIndex(attach);
    if ( isMyAttachment(att) )
    {
      if( !isServicePart(att->GetName()) )
      {
        if (numChecked == index)
        {
          return static_cast<ICharacterPartAttachment*>(att->GetIAttachmentObject());
        }
        ++numChecked;
      }
    }
  }
  return NULL;
}
*/
//-------------------------------------------------------------------------
int CCompoundCharacter::GetNumAttachedInternal() const
{
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  int result = 0;

  int numAttached = pManager->GetAttachmentCount();
  for (int attach = 0; attach < numAttached; ++attach)
  {
    IAttachment* att = pManager->GetInterfaceByIndex(attach);
    if ( isMyAttachment(att) )
    {
      if( !isServicePart(att->GetName()) )
      {
        ++result;
      }
    }
  }
  return result;
}
//-------------------------------------------------------------------------
ICharacterPartAttachment* CCompoundCharacter::GetAttachedPartInfoInternal(int index) const
{
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();

  int numChecked = 0;

  int numAttached = pManager->GetAttachmentCount();
  for (int attach = 0; attach < numAttached; ++attach)
  {
    IAttachment* att = pManager->GetInterfaceByIndex(attach);
    if ( isMyAttachment(att) )
    {
      if( !isServicePart(att->GetName()) )
      {
        if (numChecked == index)
        {
          return static_cast<ICharacterPartAttachment*>(att->GetIAttachmentObject());
        }
        ++numChecked;
      }
    }
  }
  return NULL;
}

ICharacterPartAttachment* CCompoundCharacter::GetAttachedPartInfoInternal(const char* name) const
{
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();

  int numAttached = pManager->GetAttachmentCount();
  for (int attach = 0; attach < numAttached; ++attach)
  {
    IAttachment* att = pManager->GetInterfaceByIndex(attach);
    if ( isMyAttachment(att) )
    {
      if( strcmp(att->GetName(),name)==0 )
      {
        return static_cast<ICharacterPartAttachment*>(att->GetIAttachmentObject());
      }
    }
  }
  return NULL;
}


bool CCompoundCharacter::isServicePart(const char* partName) const
{
  if( strcmp(partName,CC_PHYS_PART_ATTACHMENT_NAME)==0 )
    return true;
  if( strncmp(partName,CC_VISFLAG_PART_ATTACHMENT_PREFIX,strlen(CC_VISFLAG_PART_ATTACHMENT_PREFIX))==0 )
    return true;
  return false;
}
//-------------------------------------------------------------------------
const char* CCompoundCharacter::GetAttachedPartName(int index) const
{
  return m_attachedParts[index].m_name;
}
//-------------------------------------------------------------------------
void CCompoundCharacter::ApplyGenderToAttachedParts()
{
  //
  WarnCached("ApplyGenderToAttachedParts");
  //
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  //
  std::vector<string> reattachPartsList;
  //
  int numAttached = pManager->GetAttachmentCount();
  for (int attach = 0; attach < numAttached; ++attach)
  {
    IAttachment* att = pManager->GetInterfaceByIndex(attach);
    if ( isMyAttachment(att) )
    {
      CCharacterPartAttachment* part = static_cast<CCharacterPartAttachment*>(att->GetIAttachmentObject());
      if( part )
      {
        //int currMaterial = part->m_currentMaterial;

        // Load corresponding model
        ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
        const ICharacterPart* pCharacterPart = pIPartsManager->GetPartByName(att->GetName());
        if( pCharacterPart )
        {
          const IPartModel* pModel = pCharacterPart->GetModel(m_gender);
          IMaterial* pChrMtl=NULL;
          string partCHRFileName=GetPartCHRFileName(att->GetName(),&pChrMtl);
          string fileExt = PathUtil::GetExt(partCHRFileName);
          bool IsCHR = (0 == stricmp(fileExt, "chr"));

          CCharacterPartAttachment* pAttachmentObject = NULL;

          if (IsCHR)
          {
            ICharacterInstance* pIChildCharacter = gEnv->pCharacterManager->CreateInstance(partCHRFileName);
            if (pIChildCharacter)
            {
              int surfaceType = GetSurfaceTypeInternal(GetSurfaceType(att->GetName()));
              pAttachmentObject = new CCharacterPartAttachment(this,pCharacterPart->GetVisibilityMask(),surfaceType);
              pAttachmentObject->m_pCharInstance = pIChildCharacter;
            }
            else
            {
              pAttachmentObject = new CCharacterPartAttachment(this,pCharacterPart->GetVisibilityMask(),-1);
              pAttachmentObject->m_pCharInstance = NULL;
            }
          }
          else if( partCHRFileName == "" )
          {
            pAttachmentObject = new CCharacterPartAttachment(this,pCharacterPart->GetVisibilityMask(),-1);
            pAttachmentObject->m_pCharInstance = NULL;
          }

          if (pAttachmentObject == NULL)
            continue;

          // Set material
          if( pChrMtl )
          {
            pAttachmentObject->SetMaterial(pChrMtl);
          }
          else if (pModel->GetNumMaterials() > 0)
          {
            int currMaterial = 0;//-1;
            /*
            for(size_t n=0;n<pModel->m_materials.size();n++)
            {
            if( att->GetIAttachmentObject()->GetMaterial() && 
            att->GetIAttachmentObject()->GetMaterial()->GetName()==)
            }
            */
            if (currMaterial >= pModel->GetNumMaterials())
              currMaterial = 0;

            string filename = pModel->GetMaterialByIndex(currMaterial)->GetFilename();
            ExtractMaterialName(filename);

            _smart_ptr<IMaterial> pIMaterial = gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(filename.c_str(), false);
            if (pIMaterial)
            {
              pAttachmentObject->SetMaterial(pIMaterial);
              //            pAttachmentObject->m_currentMaterial = currMaterial;
            }
          }
          // bind
          att->AddBinding(pAttachmentObject);
        }
      }
      else
      {
        reattachPartsList.push_back(att->GetName());
      }
    }
  }
  //
  for(size_t n=0;n<reattachPartsList.size();n++)
  {
    RemoveAttachmentByName(reattachPartsList[n]);
    this->AttachPart(reattachPartsList[n],0);
  }
  //
  pManager->ProjectAllAttachment();
  UpdateMorphs();
}
//-------------------------------------------------------------------------
void CCompoundCharacter::CollectActiveSwitches(std::vector<string>& activeSwitches)
{
  activeSwitches.resize(0);
  for (size_t n = 0; n < m_attachedParts.size(); ++n)
  {
    const ICharacterPart* pCharacterPart = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager()->GetPartByName(m_attachedParts[n].m_name);
    if( pCharacterPart )
    {
      const IPartModel* pModel = pCharacterPart->GetModel(m_gender);
      for (size_t s = 0; s < pModel->GetNumSwitches(); ++s)
      {
        // add only if not exist
        std::vector<string>::iterator it = std::find(activeSwitches.begin(),activeSwitches.end(),pModel->GetSwitchByIndex(s));
        if( it==activeSwitches.end() )
        {
          activeSwitches.push_back(pModel->GetSwitchByIndex(s));
        }
      }
    }
  }
}
//
static void GetFatThinRatios(float fatness,float& fat,float& thin)
{
  fat = CLAMP(fatness, 0, 1);
  thin = fabs(CLAMP(fatness, -1, 0));
}
//-------------------------------------------------------------------------
void CCompoundCharacter::GetMorphsDataToBake(const char* name,CCompoundCharacter::MorphsData& morphData)
{
  morphData.m_partName = name;
  morphData.m_resmorphs.clear();
  //
  float FatValue,ThinValue;
  GetFatThinRatios(GetFatness(),FatValue,ThinValue);
  //
  float NeckFatValue = CLAMP(GetNeckFatness(), 0, 1);
  float NeckThinValue = fabs(CLAMP(GetNeckFatness(), -1, 0));
  //
  // find all switches
  std::vector<string> activeSwitches;
  CollectActiveSwitches(activeSwitches);
  // collect all posible morphs
  const ICharacterPart* pCharacterPart = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager()->GetPartByName(name);
  if (pCharacterPart==NULL) 
    return;
  //
  const IPartModel* pModel = pCharacterPart->GetModel(GetGender());
  for(int m=0;m<pModel->GetNumMorphs();m++)
  {
    const char* pMorphName = pModel->GetMorphByIndex(m)->GetMorphTarget();
    if( strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "fat") == 0 )
    {
      morphData.m_resmorphs.push_back(std::pair<string,float>(pMorphName,FatValue));
    }
    else if( strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "thin") == 0 )
    {
      morphData.m_resmorphs.push_back(std::pair<string,float>(pMorphName,ThinValue));
    }
    else if( strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "neck_fat") == 0 )
    {
      morphData.m_resmorphs.push_back(std::pair<string,float>(pMorphName,NeckFatValue));
    }
    else if( strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "neck_thin") == 0 )
    {
      morphData.m_resmorphs.push_back(std::pair<string,float>(pMorphName,NeckThinValue));
    }
    //
    for (size_t s = 0; s < activeSwitches.size(); ++s)
    {
      if (strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), activeSwitches[s].c_str()) == 0)
      {
        morphData.m_resmorphs.push_back(std::pair<string,float>(pMorphName,1.0f));
      }
    }
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacter::UpdateMorphs()
{
  //
  if( WarnCached("UpdateMorphs ignored") )
  {
    return;
  }
  //
//  if( WarnCommitOnCache("UpdateMorphs ignored") )
//  {
//    return;
//  }
  //
  float FatValue = CLAMP(GetFatness(), 0, 1);
  float ThinValue = fabs(CLAMP(GetFatness(), -1, 0));
  //
  float NeckFatValue = CLAMP(GetNeckFatness(), 0, 1);
  float NeckThinValue = fabs(CLAMP(GetNeckFatness(), -1, 0));
  //
  // find all switches
  std::vector<string> activeSwitches;
  CollectActiveSwitches(activeSwitches);
  //
  //CryLog("UpdateMorphs: started");
  // process parts attachments
  for (size_t n = 0; n < m_pInstance->GetIAttachmentManager()->GetAttachmentCount(); ++n)
  {
    IAttachment* att = m_pInstance->GetIAttachmentManager()->GetInterfaceByIndex(n);
    if ( isMyAttachment(att) )
    {
      const ICharacterPart* pCharacterPart = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager()->GetPartByName(att->GetName());
      CCharacterPartAttachment* part = static_cast<CCharacterPartAttachment*>(att->GetIAttachmentObject());
      if (part == NULL) continue;
      if (!part->GetICharacterInstance()) continue;
      if( isServicePart(att->GetName()) )
      {
        continue;
      }
      //
      const IPartModel* pModel = pCharacterPart->GetModel(m_gender);
      // switch off all morphs
      IMorphing* pMorph = part->GetICharacterInstance()->GetIMorphing();
      if( pMorph )
      {
        pMorph->SetInstantMorphsApply(true);
        //
        pMorph->StopAllMorphs();
        //
        size_t numMophs = pMorph->GetNumMorphs();
        for(size_t morphIndex = 0;morphIndex<numMophs;morphIndex++)
        {
          bool bFound = false;
          //
          const char* pMorphName = pMorph->GetMorphName(morphIndex);
          for (size_t m = 0; m < pModel->GetNumMorphs(); ++m)
          {
            const char* morphSwitch = pModel->GetMorphByIndex(m)->GetSwitch();
            const char* morphTarget = pModel->GetMorphByIndex(m)->GetMorphTarget();
            //
            if( strcmp(morphTarget,pMorphName)==0 )
            {
              //
              bFound = true;
							{
								// special case fat/thin
								CryCharMorphParams Params;
								Params.m_fBlendIn		=	0;
								Params.m_fLength		=	0.0f;
								Params.m_fBlendOut	=	0;
								Params.m_fAmplitude = 0.0f;//!!!
								Params.m_fBalance = 0.0f;
								Params.m_fStartTime = 0.0f;
								Params.m_fSpeed = 1;
								Params.m_nFlags = CryCharMorphParams::FLAGS_NO_BLENDOUT;// | CryCharMorphParams::FLAGS_INSTANTANEOUS;
								if( strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "fat") == 0 || strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "thin") == 0 )
								{
									if( strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "fat") == 0 )
									{
										Params.m_fAmplitude = FatValue;
									}
									else
									{
										Params.m_fAmplitude = ThinValue;
									}
									//CryLog("StartMorph: %s %s %f",part->m_pCharacterPart->m_name.c_str(),pModel->m_morphs[m].m_target.c_str(),Params.m_fAmplitude);
									pMorph->StartMorph(pModel->GetMorphByIndex(m)->GetMorphTarget(), Params);
								}
								else if( strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "neck_fat") == 0 || strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "neck_thin") == 0 )
								{
									if( strcmp(pModel->GetMorphByIndex(m)->GetSwitch(), "neck_fat") == 0 )
									{
										Params.m_fAmplitude = NeckFatValue;
									}
									else
									{
										Params.m_fAmplitude = NeckThinValue;
									}
									//CryLog("StartMorph: %s %s %f",part->m_pCharacterPart->m_name.c_str(),pModel->m_morphs[m].m_target.c_str(),Params.m_fAmplitude);
									pMorph->StartMorph(pModel->GetMorphByIndex(m)->GetMorphTarget(), Params);
								}
							}
              // switches
              for (size_t s = 0; s < activeSwitches.size(); ++s)
              {
                if (strcmp(morphSwitch, activeSwitches[s].c_str()) == 0)
                {
                  //pMorph->StopMorph(pModel->m_morphs[m].m_target.c_str());
                  //
                  CryCharMorphParams Params;
                  Params.m_fBlendIn		=	0;
                  Params.m_fLength		=	0.0f;//(fBlendInTime+fBlendOutTime)/2;
                  Params.m_fBlendOut	=	0;
                  Params.m_fAmplitude = 1.0f;//!!!
                  Params.m_fBalance = 0.0f;
                  Params.m_fStartTime = 0.0f;
                  Params.m_fSpeed = 1;
                  Params.m_nFlags = CryCharMorphParams::FLAGS_NO_BLENDOUT;// | CryCharMorphParams::FLAGS_INSTANTANEOUS;
                  //CryLog("StartMorph: %s %s %f",part->m_pCharacterPart->m_name.c_str(),pModel->m_morphs[m].m_target.c_str(),Params.m_fAmplitude);
                  pMorph->StartMorph(pModel->GetMorphByIndex(m)->GetMorphTarget(), Params);
                }
              }
            }
          }
          //
          if( !bFound )
          {
            if( pMorphName[1]=='!' )
            {
              CryCharMorphParams Params;
              Params.m_fBlendIn		=	0;
              Params.m_fLength		=	0.0f;//(fBlendInTime+fBlendOutTime)/2;
              Params.m_fBlendOut	=	0;
              Params.m_fAmplitude = 1.0f;//!!!
              Params.m_fBalance = 0.0f;
              Params.m_fStartTime = 0.0f;
              Params.m_fSpeed = 1;
              Params.m_nFlags = CryCharMorphParams::FLAGS_NO_BLENDOUT;// | CryCharMorphParams::FLAGS_INSTANTANEOUS;
              //CryLog("StartMorph: %s %s %f",part->m_pCharacterPart->m_name.c_str(),pModel->m_morphs[m].m_target.c_str(),Params.m_fAmplitude);
              pMorph->StartMorph(pMorphName, Params);
            }
            else
            {
              CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Morph: %s %s not found",pCharacterPart->GetName(),pMorphName);
            }
          }
        }
      }
    }
  }
}
//-------------------------------------------------------------------------
const char* CCompoundCharacter::GetSurfaceType(const char* partName)
{
  CCompoundCharacter::SAttachedPart* pPart = GetAttachedPart(partName);
  if( !pPart )
    return NULL;
  return pPart->m_surface.c_str();
}
//-------------------------------------------------------------------------
void CCompoundCharacter::SetSurfaceType(const char* partName, const char* surface_type)
{
  //
  if( WarnCached("SetSurfaceType ignored") )
    return;
  //
  if( WarnCommitOnCache("SetSurfaceType ignored") )
    return;
  //
  IAttachmentManager* pManager = m_pInstance->GetIAttachmentManager();
  //
  std::vector<string> reattachPartsList;
  //
  int numAttached = pManager->GetAttachmentCount();
  for (int attach = 0; attach < numAttached; ++attach)
  {
    IAttachment* att = pManager->GetInterfaceByIndex(attach);
    if ( isMyAttachment(att) && strcmp(att->GetName(),partName)==0 )
    {
      CCharacterPartAttachment* part = static_cast<CCharacterPartAttachment*>(att->GetIAttachmentObject());
      if( part )
      {
        int surf_type = GetSurfaceTypeInternal(surface_type);
        if( surf_type!=-1 )
        {
          for(int n=0;n<part->m_addedPartsList.size();n++)
          {
            pe_params_part params;
            params.partid = part->m_addedPartsList[n].m_partId;
            int mtlMap[1];
            if( part->m_surface_type_id!=-1 )
            {
              params.nMats = 1;
              mtlMap[0] = surf_type;
              params.pMatMapping = mtlMap;
            }
            part->m_pRootPhysicalEntity->SetParams(&params);
          }
        }
      }
    }
  }
  // remember surf type
  CCompoundCharacter::SAttachedPart* pPart = GetAttachedPart(partName);
  if( pPart )
  {
    pPart->m_surface = surface_type;
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacter::UpdatePartModel(const char* partName)
{
  //
  if( WarnCached("UpdateMorphs ignored") )
  {
    return;
  }
  //
  if( WarnCommitOnCache("UpdateMorphs ignored") )
  {
    return;
  }

  string matName = GetAttachedPart(partName)->m_material_id;
  string surf_type = GetAttachedPart(partName)->m_surface;

  DetachPart(partName);
  AttachPart(partName, matName, surf_type);
}
//-------------------------------------------------------------------------
static void CleanupCache(IConsoleCmdArgs* /* pArgs */)
{
  CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->Cleanup(true);
}
//-------------------------------------------------------------------------
static void CCCacheAll(IConsoleCmdArgs* /* pArgs */)
{
  CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->CacheAll();
}
//-------------------------------------------------------------------------
struct SLevelSystemListener : public ILevelSystemListener
{
  //! callback on movie events
  virtual void OnLevelNotFound(const char *levelName){};
  virtual void OnLoadingStart(ILevelInfo *pLevel);
  virtual void OnLoadingComplete(ILevel *pLevel){};
  virtual void OnLoadingError(ILevelInfo *pLevel, const char *error){};
  virtual void OnLoadingProgress(ILevelInfo *pLevel, int progressAmount){};
};
//-------------------------------------------------------------------------
void SLevelSystemListener::OnLoadingStart(ILevelInfo *pLevel)
{
  // clean up
  CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->Cleanup(false);
}
//-------------------------------------------------------------------------
CCompoundCharacterCache::CCompoundCharacterCache()
{
  m_maxCacheSize = 100;
  m_cache = 0;
  m_verbose = 0;
  m_levelListener = new SLevelSystemListener();
  //
  CCryAction::GetCryAction()->GetILevelSystem()->AddListener(m_levelListener);
}
//-------------------------------------------------------------------------
CCompoundCharacterCache::~CCompoundCharacterCache()
{
  IGameFramework* pGF = gEnv->pGame->GetIGameFramework();
  pGF->GetILevelSystem()->RemoveListener(m_levelListener);
  // listeners
  delete m_levelListener;
}
//-------------------------------------------------------------------------
string CCompoundCharacterCache::GetCacheFolder()
{
  string chrFolder = "%USER%/ModelsCache";
  char path[ICryPak::g_nMaxPath];
  path[sizeof(path) - 1] = 0;
  gEnv->pCryPak->AdjustFileName(chrFolder, path, ICryPak::FLAGS_PATH_REAL | ICryPak::FLAGS_FOR_WRITING);
  if (!gEnv->pCryPak->MakeDir(path))
  {
	  CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "Cannot generate character part cache! Failed to create directory \"%s\".", chrFolder.c_str());
  }
  return path;
}
//-------------------------------------------------------------------------
void CCompoundCharacterCache::RegisterCVars()
{
  // register 
  IConsole *pConsole = gEnv->pConsole;
  pConsole->AddCommand( "cc_cleanupcache", CleanupCache, 0,"Clears characters cache");
  pConsole->Register( "cc_maxcachesize", &m_maxCacheSize, 100, 0, "Max character cache size in MB." );
  if( gEnv->pSystem->IsDedicated() )
  {
    pConsole->Register( "cc_cache", &m_cache, 0, 0, "Enables character cache." );
  }
  else
  {
    pConsole->Register( "cc_cache", &m_cache, 1, 0, "Enables character cache." );
  }
  pConsole->Register( "cc_verbose", &m_verbose, 0, 0, "verbose mode." );
  pConsole->AddCommand( "cc_cacheall", CCCacheAll, 0,"Cache All Compound Characters");

  pConsole->Register( "cc_lodGenNumLods", &cc_lodGenNumLods, 3, 0, "lods count." );

  pConsole->Register( "cc_lod1GenMaxAngle", &cc_lod1GenMaxAngle, 15.0f, 0, "Lod 1 generation max angle." );
  pConsole->Register( "cc_lod1GenTexSplits", &cc_lod1GenTexSplits, 0, 0, "Lod 1 handle texture splits." );
  pConsole->Register( "cc_lod1GenSkinning", &cc_lod1GenSkinning, 0, 0, "Lod 1 handle less accuracte skinning." );

  pConsole->Register( "cc_lod2GenMaxAngle", &cc_lod2GenMaxAngle, 25.0f, 0, "Lod 2 generation max angle." );
  pConsole->Register( "cc_lod2GenTexSplits", &cc_lod2GenTexSplits, 0, 0, "Lod 2 handle texture splits." );
  pConsole->Register( "cc_lod2GenSkinning", &cc_lod2GenSkinning, 0, 0, "Lod 2 handle less accuracte skinning." );

  pConsole->Register( "cc_lod3GenMaxAngle", &cc_lod3GenMaxAngle, 60.0f, 0, "Lod 3 generation max angle." );
  pConsole->Register( "cc_lod3GenTexSplits", &cc_lod3GenTexSplits, 1, 0, "Lod 3 handle texture splits." );
  pConsole->Register( "cc_lod3GenSkinning", &cc_lod3GenSkinning, 1, 0, "Lod 3 handle less accuracte skinning." );

  pConsole->Register( "cc_lod4GenMaxAngle", &cc_lod4GenMaxAngle, 70.0f, 0, "Lod 4 generation max angle." );
  pConsole->Register( "cc_lod4GenTexSplits", &cc_lod4GenTexSplits, 1, 0, "Lod 4 handle texture splits." );
  pConsole->Register( "cc_lod4GenSkinning", &cc_lod4GenSkinning, 1, 0, "Lod 4 handle less accuracte skinning." );

  pConsole->Register( "cc_lod5GenMaxAngle", &cc_lod5GenMaxAngle, 90.0f, 0, "Lod 5 generation max angle." );
  pConsole->Register( "cc_lod5GenTexSplits", &cc_lod5GenTexSplits, 1, 0, "Lod 5 handle texture splits." );
  pConsole->Register( "cc_lod5GenSkinning", &cc_lod5GenSkinning, 1, 0, "Lod 5 handle less accuracte skinning." );
}
//-------------------------------------------------------------------------
void CCompoundCharacterCache::UnRegisterCVars()
{
  // register 
  IConsole *pConsole = gEnv->pConsole;
  pConsole->RemoveCommand("cc_cleanup");
  pConsole->UnregisterVariable( "cc_maxcachesize", true );
  pConsole->UnregisterVariable( "cc_cache", true );
  pConsole->UnregisterVariable( "verbose", true );
  pConsole->UnregisterVariable( "cc_lodGenNumLods", true );
  pConsole->UnregisterVariable( "cc_lod1GenMaxAngle", true );
  pConsole->UnregisterVariable( "cc_lod1GenTexSplits", true );
  pConsole->UnregisterVariable( "cc_lod1GenSkinning", true );

  pConsole->UnregisterVariable( "cc_lod2GenMaxAngle", true );
  pConsole->UnregisterVariable( "cc_lod2GenTexSplits", true );
  pConsole->UnregisterVariable( "cc_lod2GenSkinning", true );

  pConsole->UnregisterVariable( "cc_lod3GenMaxAngle", true );
  pConsole->UnregisterVariable( "cc_lod3GenTexSplits", true );
  pConsole->UnregisterVariable( "cc_lod3GenSkinning", true );

  pConsole->UnregisterVariable( "cc_lod4GenMaxAngle", true );
  pConsole->UnregisterVariable( "cc_lod4GenTexSplits", true );
  pConsole->UnregisterVariable( "cc_lod4GenSkinning", true );

  pConsole->UnregisterVariable( "cc_lod5GenMaxAngle", true );
  pConsole->UnregisterVariable( "cc_lod5GenTexSplits", true );
  pConsole->UnregisterVariable( "cc_lod5GenSkinning", true );

  pConsole->RemoveCommand("cc_cacheall");
}

//-------------------------------------------------------------------------
static bool UDless ( const std::pair<string,_finddata_t>& elem1, const std::pair<string,_finddata_t>& elem2 )
{
  return elem1.second.time_write < elem2.second.time_write;
}
void CCompoundCharacterCache::Cleanup(bool bFullCleanup)
{
  //
  string cacheDir = GetCacheFolder();
  //
  if( bFullCleanup )
  {
    _finddata_t fd;
    intptr_t handle = gEnv->pCryPak->FindFirst( cacheDir+"/*.chr", &fd );
    if (handle >= 0)
    {
      do 
      {
        gEnv->pCryPak->RemoveFile(cacheDir+"/"+fd.name);
      } while (gEnv->pCryPak->FindNext( handle, &fd ) >= 0);
      gEnv->pCryPak->FindClose(handle);
    }
  }
  else
  {
    std::vector<std::pair<string,_finddata_t> > entries;
    uint64 totalSize = 0;
    //
    _finddata_t fd;
    intptr_t handle = gEnv->pCryPak->FindFirst( cacheDir+"/*.chr", &fd );
    if (handle >= 0)
    {
      do 
      {
        totalSize+=fd.size;
        entries.push_back(std::pair<string,_finddata_t>(cacheDir+"/"+fd.name,fd));
      } while (gEnv->pCryPak->FindNext( handle, &fd ) >= 0);
      gEnv->pCryPak->FindClose(handle);
    }
    long maxSize = 1024*1024*m_maxCacheSize;
    if( totalSize>maxSize)
    {
      std::sort(entries.begin(),entries.end(),UDless);
      for(size_t n=0;n<entries.size();n++)
      {
        if( totalSize>maxSize )
        {
          gEnv->pCryPak->RemoveFile(entries[n].first);
          totalSize-=entries[n].second.size;
        }
        else
        {
          break;
        }
      }
    }
  }
}
void CCompoundCharacterCache::CacheAll()
{
  CCharacterPartsManager* pIPartsManager = static_cast<CCharacterPartsManager*>(gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager());
  for(int n=0;n<pIPartsManager->m_registered.size();n++)
  {
    pIPartsManager->m_registered[n]->Cache(false);
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacterCache::Schedule(CCompoundCharacter* pCharacter,ICompoundCharacter::ECacheFlags flags)
{
  if( m_cache )
  {
    IMaterialMergePipe::EMergeMaterialsFlags mtlMergeFlags = (flags &  ICompoundCharacter::eCacheFlag_Verbose ? IMaterialMergePipe::mfVerbose : IMaterialMergePipe::mfDefaut);
    CCharacterPartsManager* pIPartsManager = static_cast<CCharacterPartsManager*>(gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager());
    // step1: remove all jobs for this character
    UnScheduleCharacterJobs(pCharacter);
    // step1a: allow cache based only on .chr 
    const char* ext = PathUtil::GetExt(pCharacter->GetCharacterInstance()->GetFilePath());
    if( stricmp(ext,"chr")!=0 )
      return;
    // step: bake physics
    if( !(flags & ICompoundCharacter::eCacheFlag_DoNotCachePhysics) )
    {
      Job job;
      job.m_jobType = Job::jtBakePhysics;
      job.m_mergePipeFlags = IMergePipe::mfPhysics;
      job.m_pCharacter = pCharacter;
      m_pendingJobs.push_back(job);
    }
    // step: bake morphs
    if( flags & ICompoundCharacter::eCacheFlag_CacheMorphsOnly )
    {
      int numParts = pCharacter->GetNumAttached();
      for(int n=0;n<numParts;n++)
      {
        //
        const char* name = pCharacter->GetAttachedPartName(n);
        if( name==NULL )
          continue;
        // morphs job
        Job job;
        job.m_jobType = Job::jtBakeMorphs;
        //job.m_mergePipeFlags = IMergePipe::mfPhysics;
        job.m_pCharacter = pCharacter;
        job.m_partName = name;
        m_pendingJobs.push_back(job);
      }
    }
    //
    if( !(flags & ICompoundCharacter::eCacheFlag_CacheMorphsOnly) )
    {
      // step4: bake third person and shadow
      {
        Job job;
        job.m_jobType = Job::jtBakeByVisibilityFlags;
        job.m_pCharacter = pCharacter;
        job.m_mergeVisFlags = VISIBLE_THIRD_PERSON | VISIBLE_SHADOW_GENERATION;
        job.m_mergeVisFlagsAnd = VISIBLE_THIRD_PERSON | VISIBLE_SHADOW_GENERATION;
        job.m_mergePipeFlags = (IMergePipe::EMergeFlags)(IMergePipe::mfSkin | IMergePipe::mfMergeMaterials);
        job.m_mergeMaterialFlags = (IMaterialMergePipe::EMergeMaterialsFlags)(mtlMergeFlags | IMaterialMergePipe::mfOnlyUsedSpace);
        job.m_bGenerateLod = true;
        if( (flags & ICompoundCharacter::eCacheFlag_DoNotMergeTextures) || pIPartsManager->m_cc_mergerDisableAtlas )
        {
          job.m_mergeMaterialFlags = (IMaterialMergePipe::EMergeMaterialsFlags)(job.m_mergeMaterialFlags | IMaterialMergePipe::mfDoNotMergeTextures);
        }
        if( pIPartsManager->m_cc_mergerUseDiffuseAtlas )
        {
          job.m_mergeMaterialFlags = (IMaterialMergePipe::EMergeMaterialsFlags)(job.m_mergeMaterialFlags | IMaterialMergePipe::mfUseDiffuseAtlas);
        }
        m_pendingJobs.push_back(job);
      }
      // step5: bake third person for first person
      {
        // just to bake morphs masks and equal materials
        Job job;
        job.m_jobType = Job::jtBakeByVisibilityFlags;
        job.m_pCharacter = pCharacter;
        job.m_mergeVisFlags = VISIBLE_FIRST_PERSON;
        job.m_mergeVisFlagsAnd = VISIBLE_FIRST_PERSON;
        job.m_mergePipeFlags = (IMergePipe::EMergeFlags)(IMergePipe::mfSkin | IMergePipe::mfMergeMaterials);
        job.m_mergeMaterialFlags = (IMaterialMergePipe::EMergeMaterialsFlags)(mtlMergeFlags | IMaterialMergePipe::mfDoNotMergeTextures);
        if( flags & ICompoundCharacter::eCacheFlag_DoNotMergeTextures || pIPartsManager->m_cc_mergerDisableAtlas )
        {
          job.m_mergeMaterialFlags = (IMaterialMergePipe::EMergeMaterialsFlags)(job.m_mergeMaterialFlags | IMaterialMergePipe::mfDoNotMergeTextures);
        }
        if( pIPartsManager->m_cc_mergerUseDiffuseAtlas )
        {
          job.m_mergeMaterialFlags = (IMaterialMergePipe::EMergeMaterialsFlags)(job.m_mergeMaterialFlags | IMaterialMergePipe::mfUseDiffuseAtlas);
        }
        m_pendingJobs.push_back(job);
      }
    }
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacterCache::RegisterCompoundCharacter(CCompoundCharacter* pCharacter)
{
  CCharacterPartsManager* pIPartsManager = static_cast<CCharacterPartsManager*>(gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager());
  std::vector<CCompoundCharacter*>::iterator it = std::find(pIPartsManager->m_registered.begin(),pIPartsManager->m_registered.end(),pCharacter);
  if( it==pIPartsManager->m_registered.end() )
  {
    pIPartsManager->m_registered.push_back(pCharacter);
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacterCache::UnScheduleCharacterJobs(CCompoundCharacter* pCharacter)
{
  // remove all jobs for this character
  for(int n=(int)m_pendingJobs.size()-1;n>=0;n--)
  {
    if( m_pendingJobs[n].m_pCharacter==pCharacter )
    {
      m_pendingJobs.erase(m_pendingJobs.begin()+n);
    }
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacterCache::UnRegisterCompoundCharacter(CCompoundCharacter* pCharacter)
{
  CCharacterPartsManager* pIPartsManager = static_cast<CCharacterPartsManager*>(gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager());
	stl::find_and_erase(pIPartsManager->m_registered,pCharacter);
	// remove all jobs for this character
	UnScheduleCharacterJobs(pCharacter);
}
string CCompoundCharacterCache::GetPhysicsCHRFileName(CCompoundCharacter* pCharacter)  const
{
  ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
  //
  string hashString = CACHE_VERSION;
  hashString+= string("CharacterPhysics!");
  //
  std::vector<string> partsNames;
  int numParts = pCharacter->GetNumAttached();
  for(int n=0;n<numParts;n++)
  {
    partsNames.push_back( pCharacter->GetAttachedPartName(n) );
  }
  // sort by names
  std::sort(partsNames.begin(),partsNames.end());
  //
  hashString+= string().Format("Gender!%d\n",(int)pCharacter->GetGender());
  //
  for(int n=0;n<numParts;n++)
  {
    string& partName = partsNames[n];
    hashString+= string("PartName:") + partName;
    const ICharacterPart* part =  pIPartsManager->GetPartByName(partName);
    // should never happens
    if( part==NULL )
    {
      return "";
    }
    //
    const IPartModel* pModel = part->GetModel(pCharacter->GetGender());
    string partCHRFileName=pCharacter->GetPartCHRFileName(partName);
    if( !gEnv->pCryPak->IsFileExist(partCHRFileName) )
    {
	    CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "CHR file is invalid for part:%s.",partName.c_str());
      return "";
    }
    //
    hashString+= string("FN!") + partCHRFileName;
    // fatness does not affect physics
    //    hashString+= string().Format("Fatness!%f\n",pCharacter->GetFatness());
    //    hashString+= string().Format("NeckFatness!%f\n",pCharacter->GetNeckFatness());
    unsigned char crc[16];
    gEnv->pCryPak->ComputeMD5(partCHRFileName,crc);
    hashString+= string("CRC!");
    for(int n2=0;n2<16;n2++)
    {
      hashString+= string().Format("%02X",crc[n2]);
    }
    hashString+="\n";
  }
  //
  MD5Context context;
  MD5Init(&context);
  MD5Update(&context,(unsigned char*)hashString.c_str(),hashString.size());
  unsigned char hash[16];
  MD5Final(hash,&context);
  //
  hashString="";
  for(int n=0;n<16;n++)
  {
    hashString+=string().Format("%02X",hash[n]);
  }
  //
  string chrFileName = "%USER%/ModelsCache/"+string("P")+hashString+".chr";
  //
  char path[ICryPak::g_nMaxPath];
  path[sizeof(path) - 1] = 0;
  gEnv->pCryPak->AdjustFileName(chrFileName, path, ICryPak::FLAGS_PATH_REAL | ICryPak::FLAGS_FOR_WRITING);
  return path;
  //
  //
}

string CCompoundCharacterCache::GetVisibilityFlagCHRFileName(CCompoundCharacter* pCharacter,uint32 visFlag)  const
{
  CCharacterPartsManager* pIPartsManager = static_cast<CCharacterPartsManager*>(gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager());
  //
  string hashString = CACHE_VERSION;
  hashString+= string("VisibilityFlags!");
  //
  std::vector<std::pair<string,string> > partsNames;
  int numParts = pCharacter->GetNumAttached();
  for(int n=0;n<numParts;n++)
  {
    const char* partName = pCharacter->GetAttachedPartName(n);
    CCompoundCharacter::SAttachedPart* pPart = pCharacter->GetAttachedPart(partName);
    if( pPart==NULL )
    {
      return "";
    }
    const char* partMaterial = pPart->m_material_name;
    partsNames.push_back( std::pair<string,string>(partName,partMaterial) );
  }
  // sort by names
  std::sort(partsNames.begin(),partsNames.end());
  //
  hashString+= string().Format("Gender:%d\n",(int)pCharacter->GetGender());
  hashString+= string().Format("Fatness:%f\n",pCharacter->GetFatness());
  hashString+= string().Format("NeckFatness:%f\n",pCharacter->GetNeckFatness());
  hashString+= string().Format("LodGenNumLods:%d\n",cc_lodGenNumLods);
  hashString+= string().Format("LodGen1MaxAngle:%f\n",cc_lod1GenMaxAngle);
  hashString+= string().Format("LodGen2MaxAngle:%f\n",cc_lod2GenMaxAngle);
  hashString+= string().Format("LodGen3MaxAngle:%f\n",cc_lod3GenMaxAngle);
  hashString+= string().Format("LodGen4MaxAngle:%f\n",cc_lod4GenMaxAngle);
  hashString+= string().Format("LodGen5MaxAngle:%f\n",cc_lod5GenMaxAngle);
  //
  hashString+= string().Format("LodGen1TexSplit:%d\n",cc_lod1GenTexSplits);
  hashString+= string().Format("LodGen2TexSplit:%d\n",cc_lod2GenTexSplits);
  hashString+= string().Format("LodGen3TexSplit:%d\n",cc_lod3GenTexSplits);
  hashString+= string().Format("LodGen4TexSplit:%d\n",cc_lod4GenTexSplits);
  hashString+= string().Format("LodGen5TexSplit:%d\n",cc_lod5GenTexSplits);
  //
  hashString+= string().Format("LodGen1Skinning:%d\n",cc_lod1GenSkinning);
  hashString+= string().Format("LodGen2Skinning:%d\n",cc_lod2GenSkinning);
  hashString+= string().Format("LodGen3Skinning:%d\n",cc_lod3GenSkinning);
  hashString+= string().Format("LodGen4Skinning:%d\n",cc_lod4GenSkinning);
  hashString+= string().Format("LodGen5Skinning:%d\n",cc_lod5GenSkinning);
  hashString+= string().Format("MergeAtlasDisabled:%d\n",pIPartsManager->m_cc_mergerDisableAtlas);
  hashString+= string().Format("MergerUseDiffuseAtlas:%d\n",pIPartsManager->m_cc_mergerUseDiffuseAtlas);
  hashString+= string().Format("sys_LowSpecPak:%d\n",gEnv->pConsole->GetCVar("sys_LowSpecPak")->GetIVal());
  //
  for(int n=0;n<numParts;n++)
  {
    string& partName = partsNames[n].first;
    string& mtlName = partsNames[n].second;
    //
	  hashString+= string().Format("PartName:%s\n",partName.c_str());
    //
    const ICharacterPart* part =  pIPartsManager->GetPartByName(partName);
    // should never happens
    if( part==NULL )
    {
      return "";
    }
    // part
    {
      const IPartModel* pModel = part->GetModel(pCharacter->GetGender());
      string partCHRFileName= pCharacter->GetPartCHRFileName(partName);

      if( !gEnv->pCryPak->IsFileExist(partCHRFileName) )
        return "";
      //
      hashString+= string("FN!") + partCHRFileName;
      unsigned char crc[16];
      if( !gEnv->pCryPak->ComputeMD5(partCHRFileName,crc) )
      {
	      CryLog("Cannot calculate md5 for file :%s",partCHRFileName.c_str());
        return "";
      }
      hashString+= string("CRC!");
      for(int n2=0;n2<16;n2++)
      {
        hashString+= string().Format("%02X",crc[n2]);
      }
      hashString+="\n";
    }
    // part material
    {
      bool bcustomHead=isCustomHead(partName,part,pCharacter->GetGender());
      if( !bcustomHead )
      {
        string mtlNameFull = mtlName+".mtl";
        if( !gEnv->pCryPak->IsFileExist(mtlNameFull) )
        {
          CryLog("Material does not exist:%s for part %s",mtlNameFull.c_str(),partName.c_str());
          return "";
        }
        //
        //
        hashString+= string("MN!") + mtlName;
        unsigned char crc[16];
        if( !gEnv->pCryPak->ComputeMD5(mtlNameFull,crc) )
        {
          CryLog("Cannot calculate md5 for file :%s",mtlNameFull.c_str());
          return "";
        }
        hashString+= string("CRC!");
        for(int k=0;k<16;k++)
        {
          hashString+= string().Format("%02X",crc[k]);
        }
      }
      else
      {
        string fname = pCharacter->getFaceFile();
        hashString+= string("custom_head!") + pCharacter->GetCustomHeadName();
        unsigned char crc[16];
        if( !gEnv->pCryPak->ComputeMD5(fname,crc) )
        {
          CryLog("Cannot calculate md5 for face file :%s",fname.c_str());
          return "";
        }
        hashString+= string("CRC!");
        for(int n2=0;n2<16;n2++)
        {
          hashString+= string().Format("%02X",crc[n2]);
        }
      }
      hashString+="\n";
    }
  }
  //
  MD5Context context;
  MD5Init(&context);
  MD5Update(&context,(unsigned char*)hashString.c_str(),hashString.size());
  unsigned char hash[16];
  MD5Final(hash,&context);
  //
  hashString="";
  for(int n=0;n<16;n++)
  {
    hashString+=string().Format("%02X",hash[n]);
  }
  //
  string chrFileName = "%USER%/ModelsCache/"+string().Format("V%04X",visFlag)+hashString+".chr";
  //
  char path[ICryPak::g_nMaxPath];
  path[sizeof(path) - 1] = 0;
  gEnv->pCryPak->AdjustFileName(chrFileName, path, ICryPak::FLAGS_PATH_REAL | ICryPak::FLAGS_FOR_WRITING);
  return path;
  //
  //
}

string CCompoundCharacterCache::GetCHRFileName(CCompoundCharacter* pCharacter,const char* partName) const
{
  ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
  const ICharacterPart* part =  pIPartsManager->GetPartByName(partName);
  // should never happens
  if( part==NULL )
  {
    return "";
  }
  const IPartModel* pModel = part->GetModel(pCharacter->GetGender());
  string partCHRFileName= pCharacter->GetPartCHRFileName(partName);
  if( !gEnv->pCryPak->IsFileExist(partCHRFileName) )
    return "";
  //
  string hashString = CACHE_VERSION;
  hashString+= string("CharacterPart!")+partName;
  hashString+= string("FN!") + partCHRFileName;
  hashString+= string().Format("Gender!%d",(int)pCharacter->GetGender());
  hashString+= string().Format("Fatness!%f",pCharacter->GetFatness());
  hashString+= string().Format("NeckFatness!%f",pCharacter->GetNeckFatness());
  unsigned char crc[16];
  gEnv->pCryPak->ComputeMD5(partCHRFileName,crc);
  hashString+= string("CRC!");
  for(int n=0;n<16;n++)
  {
    hashString+= string().Format("%02X",crc[n]);
  }
  //add morphs to hash
  CCompoundCharacter::MorphsData result;
  pCharacter->GetMorphsDataToBake(partName,result);
  for(int n=0;n<result.m_resmorphs.size();n++)
  {
	  hashString+=string().Format("%s%f",result.m_resmorphs[n].first.c_str(),result.m_resmorphs[n].second);
  }
  //
  MD5Context context;
  MD5Init(&context);
  MD5Update(&context,(unsigned char*)hashString.c_str(),hashString.size());
  unsigned char hash[16];
  MD5Final(hash,&context);
  //
  hashString="";
  for(int n=0;n<16;n++)
  {
    hashString+=string().Format("%02X",hash[n]);
  }
  //
  string chrFileName = "%USER%/ModelsCache/"+string("M")+hashString+".chr";
  //
  char path[ICryPak::g_nMaxPath];
  path[sizeof(path) - 1] = 0;
  gEnv->pCryPak->AdjustFileName(chrFileName, path, ICryPak::FLAGS_PATH_REAL | ICryPak::FLAGS_FOR_WRITING);
  return path;
  //
  //
}
string CCompoundCharacter::getFaceFile() const
{
  static ICVar* fg_Var=gEnv->pConsole->GetCVar("fg_default_preset");
  // Is 128 enough?
  CryFixedStringT<128> customHead="defaultMale";
  customHead=gEnv->pConsole->GetCVar("fg_default_preset")->GetString();
  if(!m_customHeadName.empty())
    customHead=m_customHeadName.c_str();
  //	fname+=gEnv->pConsole->GetCVar("fg_default_preset")->GetString();
  if(
    (customHead.length()>2)&&
    (customHead.at(0)=='#')&&
    (customHead.at(customHead.length()-1)=='#')
    )
  {
    string fname(gEnv->pCryPak->GetAlias("%USER%"));
    fname+="\\HeadsCache\\";
    fname+=customHead.substr(1,customHead.length()-2).c_str();
    fname+=".face";
    return fname;
  }
  else
  {
    string fname("Libs/Face/");
    fname+=customHead.c_str();
    fname+=".face";
    return fname;
  }
}
//-----------------------------------
string CCompoundCharacter::GetPartCHRFileName(const char* partName, IMaterial **material) const
{
  ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
  const ICharacterPart* part = pIPartsManager->GetPartByName(partName);
  // should never happens
  if( part==NULL )
  {
    return "";
  }
  const IPartModel* pModel = part->GetModel(GetGender());
  if( !gEnv->pCryPak->IsFileExist(pModel->GetFilename()) )
    return "";
  //
  bool bcustomHead=isCustomHead(partName,part,GetGender());
  bool hasFGTFile=strlen(pModel->GetFGTFilename())!=0;
  if( (!bcustomHead) && (!hasFGTFile))
  {
    return pModel->GetFilename();
  }
  string hashString = CACHE_VERSION;
  hashString+= string("CharacterPart!")+partName;
  hashString+= string("FN!") + pModel->GetFilename();
  hashString+= string().Format("Gender!%d",(int)GetGender());
  hashString+= string().Format("Fatness!%f",GetFatness());
  hashString+= string().Format("NeckFatness!%f",GetNeckFatness());
  unsigned char crc[16];
  gEnv->pCryPak->ComputeMD5(pModel->GetFilename(),crc);
  hashString+= string("CRC!");
  for(int n=0;n<16;n++)
  {
    hashString+= string().Format("%02X",crc[n]);
  }
  string fname = getFaceFile();
  gEnv->pCryPak->ComputeMD5(fname,crc);
  hashString+= string("CRC!");
  for(int n=0;n<16;n++)
  {
    hashString+= string().Format("%02X",crc[n]);
  }
/*
  //add morphs to hash
  CCompoundCharacter::MorphsData result;
  pCharacter->GetMorphsDataToBake(partName,result);
  for(int n=0;n<result.m_resmorphs.size();n++)
  {
  hashString+=string().Format("%s%f",result.m_resmorphs[n].first,result.m_resmorphs[n].second);
  }
*/
  //
  MD5Context context;
  MD5Init(&context);
  MD5Update(&context,(unsigned char*)hashString.c_str(),hashString.size());
  unsigned char hash[16];
  MD5Final(hash,&context);
  //
  hashString="";
  for(int n=0;n<16;n++)
  {
    hashString+=string().Format("%02X",hash[n]);
  }
  //
  string chrFileName = CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->GetCacheFolder()+string("/F")+hashString+".chr";
  //
  char path[ICryPak::g_nMaxPath];
  path[sizeof(path) - 1] = 0;
  gEnv->pCryPak->AdjustFileName(chrFileName, path, ICryPak::FLAGS_PATH_REAL | ICryPak::FLAGS_FOR_WRITING);
  //
  string mtlFN = PathUtil::ReplaceExtension(path,".mtl");
  if( gEnv->pCryPak->IsFileExist(path) && gEnv->pCryPak->IsFileExist(mtlFN) )
  {
    // material returned only for pure custom head
    if( material && isCustomHead(partName,part,GetGender()) )
    {
      *material = gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(mtlFN);
    }
    return path;
  }
  // gen face
  string headFgtFile = "Libs/Face/defaultMale.fgt";
  string fgtfile;
  if( !bcustomHead )
  {
    fgtfile=pModel->GetFGTFilename();
  }
  // material returned only for pure custom head
  if(gEnv->p3DEngine->SaveFaceCHR(pModel->GetFilename(),headFgtFile,fname,fgtfile.size() ? fgtfile.c_str() : NULL,path,isCustomHead(partName,part,GetGender()) ? material : NULL))
    return path;
  return "";
}

//-------------------------------------------------------------------------
struct CCMergePipe : IMergePipe
{
  CCMergePipe(CCompoundCharacter* pCC)
  {
    m_pCC = pCC;
    m_lodGenMaxAngle = 0.0f;
    m_lodGenTexSplits = 0;
    m_lodGenSkinning = 0;
    m_mergeMatFlags = IMaterialMergePipe::mfDefaut;
  }
  virtual int GetModelsCount() const
  {
    return chrFiles.size();
  }
  virtual const char* GetModelFilename(int index) const
  {
    return chrFiles[index];
  }
  virtual IMaterial* GetModelMaterial(int index) const
  {
    return chrMtls[index];
  }
  virtual float GetModelLodMaxAngle(int chrindex) const
  {
    return m_lodGenMaxAngle;
  }
  virtual ELodFlags GetModelLodFlags(int chrindex) const
  {
    ELodFlags flag = IMergePipe::elfNothing;
    if( m_lodGenTexSplits )
      flag = (IMergePipe::ELodFlags)(flag | IMergePipe::elfHandleSplits);
    if( m_lodGenSkinning )
      flag = (IMergePipe::ELodFlags)(flag | IMergePipe::elfHandleSkinning);
    return flag;
  }
  virtual const char* GetModelLogPrefix(int chrindex) const
  {
    return partNames[chrindex];
  }
  //
  // material merge
  virtual void SetMergedMaterial(IMaterial* pMergedMaterialIn)
  {
    pMergedMaterial = pMergedMaterialIn;
  }
  virtual const char* GetMaterialMergeTexmapName(int mtlIndex,int mtlSubsetIndex,int texmapIndex)
  {
    tmp_texmap.Format("%s_%02d_%02d_%02d%s.dds",chr_filename.c_str(),mtlIndex,mtlSubsetIndex,texmapIndex,texmapIndex==EFTT_BUMP ? "_ddn" : "");
    //
    tmp_texmap = "%USER%/ModelsCache/"+tmp_texmap;
    //
    char path[ICryPak::g_nMaxPath];
    path[sizeof(path) - 1] = 0;
    gEnv->pCryPak->AdjustFileName(tmp_texmap, path, ICryPak::FLAGS_PATH_REAL | ICryPak::FLAGS_FOR_WRITING);

    tmp_texmap = path;

    return tmp_texmap;
  }
  virtual IMaterialMergePipe::EMergeMaterialsFlags GetMaterialMergeFlags() const
  {
    return m_mergeMatFlags;
  }
  //
  //! returns name of newly created node
  virtual const char* GetModelName() const
  {
    return CC_PHYS_MODELNAME;
  }
  virtual SMergePhysicsBoneResult GetPhysicsBoneMergeInfo(int chrindex,const char* modelNameFull,const char* boneName,IMergePipeBackend* backend) const
  {
    //
    static int index = 0;
    static string phys_temp;
    //
    SMergePhysicsBoneResult res;
    //
    string modelName = modelNameFull;
    const char* pFirstChar = strchr(modelNameFull,'$');
    if( pFirstChar!=NULL )
    {
      modelName = string(modelNameFull,pFirstChar);
    }
    //
    const char * sn = boneName;
    // if joint name contains model name if format $MODELNAME$
    if( sn!=NULL && sn[0]=='$' && strnicmp(sn+1,modelName,modelName.size())==0 )
    {
      res.okToMerge = true;
      phys_temp.Format("%s:%s%d",CC_PHYS_PART_ATTACHMENT_NAME,partNames[chrindex].c_str(),index++);
      res.new_name = phys_temp.c_str();
      return res;
    }
		CRY_ASSERT(boneName);
    //
    if( strcmp(boneName,"Bip01 Groin Front")==0 || 
      strcmp(boneName,"Bip01 Groin Front_fat")==0 || 
      strcmp(boneName,"Bip01 Groin Front_thin")==0 || 
      strcmp(boneName,"Bip01 Groin Back")==0 ||
      strcmp(boneName,"Bip01 Groin Back_fat")==0 ||
      strcmp(boneName,"Bip01 Groin Back_thin")==0 )
    {
      // is slot is groin
      ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
      const ICharacterPart* part =  pIPartsManager->GetPartByName(partNames[chrindex]);
      if( part==NULL )
        return res;
      // check slot
      if( strcmp(part->GetSlot(),"Groin")!=0 )
        return res;
      //
      res.okToMerge = true;
      res.new_name = boneName;
      //
      return res;
    }
    //
    if( strcmp(boneName,"Bip01 HNeck")==0 )
    {
      // is slot is groin
      ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
      const ICharacterPart* part =  pIPartsManager->GetPartByName(partNames[chrindex]);
      if( part==NULL )
        return res;
      // check slot
      if( strcmp(part->GetSlot(),"Helmet")!=0 )
        return res;
      //
      res.okToMerge = true;
      res.new_name = boneName;
      //
      return res;
    }
    //
    // sockets
    {
      //
      sn = boneName;
      // if joint name contains model name if format $socket$MODELNAME$...
      if( sn==NULL || strncmp(sn,"$socket$", strlen("$socket$"))!=0 )
        return res;
      if( strnicmp(sn+strlen("$socket$"),partNames[chrindex],partNames[chrindex].size())!=0 )
        return res;
      const char* smChar  = sn+strlen("$socket$")+partNames[chrindex].size();
      if( smChar==NULL )
        return res;
      if( *(smChar+0)!='$' )
        return res;
      if( *(smChar+1)==NULL )
        return res;
      const char* name = smChar+1;
      //
      // find face attachment with name 
      CCompoundCharacter::SAttachedPart* pPart = m_pCC->GetAttachedPart(partNames[chrindex]);
      for(int a=0;a<pPart->m_faceAttachments.size();a++)
      {
        const char* face_att_name = pPart->m_faceAttachments[a].m_name;
        if( stricmp(face_att_name,name)==0 )
        {
          int face = pPart->m_faceAttachmentsData[a].m_FaceNr;
          Vec3 triangle[3];
          backend->GetFaceVerticies(chrindex,face,true,&triangle[0]);
          res.m_B2W.SetIdentity();
          res.m_B2W.SetTranslation((triangle[0]+triangle[1]+triangle[2])/3.0f);
          res.bOverrideB2W = true;
        }
      }
      res.okToMerge = true;
      res.new_name = boneName;
      //
      return res;
    }
    //
    return res;
  }
  //
  virtual float GetMorphTargetWeight(int chrIndex,const char* targetName) const
  {
    if( targetName[1]=='!' )
    {
      return 1.0f;
    }
    for(int n=0;n<morphs[chrIndex].m_resmorphs.size();n++)
    {
      if( morphs[chrIndex].m_resmorphs[n].first==targetName )
        return morphs[chrIndex].m_resmorphs[n].second;
    }
    return 0.0f;
  }
  string chr_filename;
  float m_lodGenMaxAngle;
  int   m_lodGenTexSplits;
  int   m_lodGenSkinning;
  CCompoundCharacter* m_pCC;
  std::vector<IMaterial*> chrMtls;
  std::vector<string> chrFiles;
  std::vector<string> partNames;
  std::vector<CCompoundCharacter::MorphsData> morphs;
  IMaterial* pMergedMaterial;
  IMaterialMergePipe::EMergeMaterialsFlags m_mergeMatFlags;
  string tmp_texmap;
};
//-------------------------------------------------------------------------
bool CCompoundCharacterCache::IsMergeValid(const char* chrFilename,bool bGenerateLod)
{
  string mtlName = PathUtil::ReplaceExtension(chrFilename,".mtl");
  if( !gEnv->pCryPak->IsFileExist(chrFilename) || !gEnv->pCryPak->IsFileExist(mtlName) )
    return false;
  // check material textures
  // check for valid textures
  IMaterial* pMtl = gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(mtlName,false);
  if( !pMtl )
    return false;
  //
  for(int m=0;m<pMtl->GetSubMtlCount();m++)
  {
    IMaterial* pSubMtl = pMtl->GetSubMtl(m);
    for(int i=0;i<EFTT_MAX;i++)
    {
      SShaderItem& si = pSubMtl->GetShaderItem(0);
      //
      for(int n=0;n<EFTT_MAX;n++)
      {
        //
        if( !si.m_pShaderResources->GetTexture(n) )
          continue;
        //
        if( si.m_pShaderResources->GetTexture(n)->m_Name.empty() )
          continue;
        //
        SEfResTexture* pTexture = si.m_pShaderResources->GetTexture(n);
        if( !gEnv->pCryPak->IsFileExist(pTexture->m_Name) )
          return false;
      }
    }
  }
  //
  if( bGenerateLod )
  {
    for(int n=0;n<cc_lodGenNumLods;n++)
    {
      string chrFileNameNoExt = PathUtil::GetPath(chrFilename)+PathUtil::GetFileName(chrFilename)+string().Format("_lod%d",n+1)+"."+CRY_CHARACTER_FILE_EXT;
      if( !gEnv->pCryPak->IsFileExist(chrFileNameNoExt) )
        return false;
    }
  }
  //
  return true;
}
//-------------------------------------------------------------------------
int CCompoundCharacterCache::Step()
{
  return StepInternal(NULL);
}
//-------------------------------------------------------------------------
int CCompoundCharacterCache::StepInternal(ICompoundCharacter* pCharacterIn)
{
  // 
  int numJobs = m_pendingJobs.size();
  if( numJobs && pCharacterIn==NULL )
  {
    numJobs = 1;
  }
  for(int j=0;j<numJobs;j++)
  {
    Job& job = m_pendingJobs.front();
    if( pCharacterIn && job.m_pCharacter!=pCharacterIn )
      continue;
    if( job.m_jobType==Job::jtBakeMorphs )
    {
      bool bWasCached = false;
      //
      string chrFileName;
      if( IsFullyCachedJob(job.m_pCharacter,job,&chrFileName) )
      {
        bWasCached = true;
      }
      else
      {
        CCompoundCharacter::MorphsData pMorphData;
        job.m_pCharacter->GetMorphsDataToBake(job.m_partName,pMorphData);
        ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
        const ICharacterPart* part =  pIPartsManager->GetPartByName(job.m_partName);
        if( part!=NULL )
        {
          const IPartModel* pModel = part->GetModel(job.m_pCharacter->GetGender());
          string cacheName = job.m_pCharacter->GetPartCHRFileName(job.m_partName);
          if( cacheName.size() )
          {
            if( gEnv->p3DEngine->SaveCHR(cacheName,chrFileName,&pMorphData) )
            {
              bWasCached = true;
            }
          }
        }
      }
      if( bWasCached )
      {
        job.m_pCharacter->ReloadPart(job.m_partName,true,true);
      }
      if( m_verbose )
      {
        CryLog("Character part cached:%s CHR:%s",job.m_partName.c_str(),chrFileName.c_str());
      }
    }
    else if( job.m_jobType==Job::jtBakePhysics )
    {
      bool bWasCached = false;
      string chrFileName;
      if( IsFullyCachedJob(job.m_pCharacter,job,&chrFileName) )
      {
        bWasCached = true;
      }
      else
      {
        if( !chrFileName.empty() )
        {
          ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
          //////////////////////////////////////////////////////////////////////////
          CCompoundCharacter *pCharacter = static_cast<CCompoundCharacter*>(job.m_pCharacter);
          if (pCharacter)
          {
            CCMergePipe pipe(pCharacter);
            //
            std::vector< std::vector<std::pair<string,float> > > in_morphs;
            //
            int numParts = pCharacter->GetNumAttached();
            for(int n=0;n<numParts;n++)
            {
              const char* partName = pCharacter->GetAttachedPartName(n);
              const ICharacterPart* part =  pIPartsManager->GetPartByName(partName);
              // should never happens
              if( part==NULL )
                continue;
              //
              CCompoundCharacter::SAttachedPart* pPart = static_cast<CCompoundCharacter*>(pCharacter)->GetAttachedPart(partName);
              if( pPart==NULL )
                continue;
              // part
              const IPartModel* pModel = part->GetModel(pCharacter->GetGender());
              if( !pModel )
                continue;
              //
              string chrFilename = job.m_pCharacter->GetPartCHRFileName(partName);
              if( chrFilename.empty() )
              {
                CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "CHR file is invalid for part:%s.",partName);
                continue;
              }
              pipe.chrFiles.push_back(job.m_pCharacter->GetPartCHRFileName(partName));
              pipe.partNames.push_back(partName);
              // mtl
              IMaterial* pMtl = gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(pPart->m_material_name);
              if( pMtl==NULL )
              {
                pMtl = gEnv->p3DEngine->GetMaterialManager()->GetDefaultMaterial();
              }
              pipe.chrMtls.push_back(pMtl);
              // required to recalc 
              CCompoundCharacter::MorphsData morphData;
              static_cast<CCompoundCharacter*>(pCharacter)->GetMorphsDataToBake(partName,morphData);
              pipe.morphs.push_back(morphData);
              //
            }
            // using base
            if( pipe.GetModelsCount() )
            {
              if( gEnv->p3DEngine->MergeCHRs(pCharacter->GetCharacterInstance()->GetFilePath(),&pipe,chrFileName,NULL,job.m_mergePipeFlags) )
              {
                bWasCached = true;
              }
            }
          }
        }
      }
      if( bWasCached )
      {
        if( m_verbose )
        {
          CryLog("Character physics cached CHR:%s file setup started",chrFileName.c_str());
        }
        //
        job.m_pCharacter->SetMergedPhysics(chrFileName);
        //
        if( m_verbose )
        {
          CryLog("Character physics CHR:%s file setup done",chrFileName.c_str());
        }
      }
    }
    else if( job.m_jobType==Job::jtBakeByVisibilityFlags )
    {
      //
      string hashString = "";
      string chrFileName = GetVisibilityFlagCHRFileName(job.m_pCharacter,job.m_mergeVisFlags);
      if( !chrFileName.empty() )
      {
        //
        IMaterial* pMergedMaterial = NULL;
        bool bWasCached = false;
        if ( !IsMergeValid(chrFileName,job.m_bGenerateLod) )
        {
          ICharacterPartsManager* pIPartsManager = gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager();
          //////////////////////////////////////////////////////////////////////////
          CCompoundCharacter *pCharacter = static_cast<CCompoundCharacter*>(job.m_pCharacter);
          if (pCharacter)
          {
            CCMergePipe pipe(pCharacter);
            pipe.m_lodGenMaxAngle = 0.0f;
            pipe.m_lodGenTexSplits = 0;
            pipe.m_lodGenSkinning = 0;
            pipe.m_mergeMatFlags = job.m_mergeMaterialFlags;
            //
            IMergeMaterialsResult* mergedMtls=NULL;
            std::vector< std::vector<std::pair<string,float> > > in_morphs;
            //
            int numParts = pCharacter->GetNumAttached();
            for(int n=0;n<numParts;n++)
            {
              const char* partName = pCharacter->GetAttachedPartName(n);
              const ICharacterPart* part =  pIPartsManager->GetPartByName(partName);
              // should never happens
              if( part==NULL )
                continue;
              //
              if( (part->GetVisibilityMask() & job.m_mergeVisFlagsAnd)!=job.m_mergeVisFlags )
                continue;
              //
              CCompoundCharacter::SAttachedPart* pPart = static_cast<CCompoundCharacter*>(pCharacter)->GetAttachedPart(partName);
              if( pPart==NULL )
                continue;
              // part
              const IPartModel* pModel = part->GetModel(pCharacter->GetGender());
              if( !pModel )
                continue;
              //
              string chrFilename = job.m_pCharacter->GetPartCHRFileName(partName);
              pipe.chrFiles.push_back(chrFilename);
              pipe.partNames.push_back(partName);
              // mtl
              IMaterial* pMtl = NULL;
              if( job.m_mergeVisFlags==VISIBLE_SHADOW_GENERATION )
              {
                pMtl = gEnv->p3DEngine->GetMaterialManager()->GetDefaultMaterial();
              }
              else
              {
                if( isCustomHead(partName,part,job.m_pCharacter->GetGender()) )
                {
                  //use .CHR mtl  from chr name
                  string mtlName = PathUtil::ReplaceExtension(chrFilename,".mtl");
                  pMtl = gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(mtlName);
                }
                else
                {
                  pMtl = gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(pPart->m_material_name);
                }
              }
              pipe.chrMtls.push_back(pMtl);
              //
              CCompoundCharacter::MorphsData morphData;
              static_cast<CCompoundCharacter*>(pCharacter)->GetMorphsDataToBake(partName,morphData);
              pipe.morphs.push_back(morphData);
              //
            }
            if( pipe.GetModelsCount() )
            {
              //
              pipe.chr_filename = PathUtil::GetFileName(chrFileName);
              if( m_verbose )
              {
                CryLog("CC:%s Merge started",pCharacter->GetName());
                CryLog("CC::LOD:%d",0);
              }
              if( gEnv->p3DEngine->MergeCHRs(pCharacter->GetCharacterInstance()->GetFilePath(),&pipe,chrFileName,&mergedMtls,job.m_mergePipeFlags) )
              {
                //
                pMergedMaterial = pipe.pMergedMaterial;
                //
                float angles[]= { cc_lod1GenMaxAngle, cc_lod2GenMaxAngle, cc_lod3GenMaxAngle, cc_lod4GenMaxAngle, cc_lod5GenMaxAngle };
                int texsplits[]= { cc_lod1GenTexSplits, cc_lod2GenTexSplits, cc_lod3GenTexSplits, cc_lod4GenTexSplits, cc_lod5GenTexSplits };
                int skinning[]= { cc_lod1GenSkinning, cc_lod2GenSkinning, cc_lod3GenSkinning, cc_lod4GenSkinning, cc_lod5GenSkinning };
                int numAngles = sizeof(angles)/sizeof(angles[0]);
                if( job.m_bGenerateLod && cc_lodGenNumLods>=1 )
                {
                  bWasCached = true;
                  int numLods = numAngles < cc_lodGenNumLods ? numAngles : cc_lodGenNumLods;
                  for(int n=0;n<numLods;n++)
                  {
                    pipe.m_lodGenMaxAngle = angles[n];
                    pipe.m_lodGenTexSplits = texsplits[n];
                    pipe.m_lodGenSkinning = skinning[n];
                    string chrFileNameLod = PathUtil::GetPath(chrFileName)+PathUtil::GetFileName(chrFileName)+string().Format("_lod%d",n+1)+"."+CRY_CHARACTER_FILE_EXT;
                    IMergePipe::EMergeFlags flags = (IMergePipe::EMergeFlags)((int)job.m_mergePipeFlags & ~((int)IMergePipe::mfMergeMaterials));
                    if( m_verbose )
                    {
                      CryLog("CC::LOD:%d",n+1);
                      flags = (IMergePipe::EMergeFlags)(flags | IMergePipe::mfVerbose_LOD);
                    }
                    if( !gEnv->p3DEngine->MergeCHRs(pCharacter->GetCharacterInstance()->GetFilePath(),&pipe,chrFileNameLod,&mergedMtls,flags) )
                    {
                      bWasCached = false;
                      break;
                    }
                  }
                }
                else
                {
                  bWasCached = true;
                }
                if( m_verbose )
                {
                  CryLog("CC:%s Merge done",pCharacter->GetName());
                }
                if( mergedMtls )
                  mergedMtls->Release();
              }
              else
              {
                CryLog("MergeCHR failed");
              }
            }
          }
        }
        else
        {
          string mtlName = PathUtil::ReplaceExtension(chrFileName,".mtl");
          pMergedMaterial = gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(mtlName, false);
          bWasCached = true;
        }
        if( bWasCached )
        {
          if( job.m_mergeVisFlags==VISIBLE_SHADOW_GENERATION )
          {
            pMergedMaterial = gEnv->p3DEngine->GetMaterialManager()->GetDefaultMaterial();
          }
          if( m_verbose )
          {
            CryLog("Character visibility flag cached CHR:%s file setup started",chrFileName.c_str());
          }
          //
          job.m_pCharacter->SetMergedVisibilityFlag(chrFileName,job.m_mergeVisFlagsAnd,job.m_mergeVisFlags,pMergedMaterial);
          //
          if( m_verbose )
          {
            CryLog("Character visibility flag cached CHR:%s file setup done",chrFileName.c_str());
          }
        }
      }
    }
    //
    m_pendingJobs.erase(m_pendingJobs.begin());
  }
  //
  return m_pendingJobs.size();
}

void CCompoundCharacter::AttachAllCacheOnCommit()
{
  // attach all parts
  for(int p=0;p<m_attachedParts.size();p++)
  {
    SAttachedPart& part = m_attachedParts[p];
    AttachPartInternal(part.m_name,part.m_material_id,part.m_surface,false,true,false);
    for(int s=0;s<part.m_sockets.size();s++)
    {
      AddSocketAttachment(part.m_name,part.m_sockets[s].m_name,part.m_sockets[s].m_filename);
    }
    for(int s=0;s<part.m_faceAttachments.size();s++)
    {
      AddFaceAttachment(part.m_name,part.m_faceAttachments[s].m_name,part.m_faceAttachments[s].m_filename,part.m_faceAttachments[s].m_qt,part.m_faceAttachments[s].m_bUseCenterPoint);
    }
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacter::Cache(bool bScheduleOnly,ICompoundCharacter::ECacheFlags flags)
{
  // allow cache based only on .chr 
  if( GetCharacterInstance() )
  {
    const char* ext = PathUtil::GetExt(GetCharacterInstance()->GetFilePath());
    if( stricmp(ext,"chr")!=0 )
    {
      CryWarning(VALIDATOR_MODULE_ENTITYSYSTEM, VALIDATOR_WARNING, "CC: Cache called for non .chr as base character:%s.",GetCharacterInstance()->GetFilePath());
      return;
    }
  }
  //
  if( CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->m_verbose )
  {
    CryLog("----- CC:%s -",GetName());
    CryLog("Gender:%d, Fatness:%f, Scale:%f",GetGender(),GetFatness(),GetScale());
    for(size_t n=0;n<m_attachedParts.size();n++)
    {
	    CryLog("PartName:%s, mtl:[%s,%s],srf:[%s]",m_attachedParts[n].m_name.c_str(), m_attachedParts[n].m_material_id.c_str(), m_attachedParts[n].m_material_name.c_str(),m_attachedParts[n].m_surface.c_str() );
    }
    CryLog("----");
  }
  //
  WarnCached("CCompoundCharacter::Cache");
  //
  if( !CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->m_cache )
  {
    if( this->IsCommitOnCache() )
    {
      AttachAllCacheOnCommit();
    }
    m_bCached = false;
    return;
  }
  if( !m_bCached )
  {
    // maybe good idea to add only this character jobs
    CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->Schedule(this,flags);
    if( !bScheduleOnly )
    {
      while(CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->Step())
      {
      };
    }
    else
    {
      if( this->IsCommitOnCache() )
      {
        bool bAttachAll = true;
        if( CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->IsFullyCached(this) )
        {
          CCryAction::GetCryAction()->GetCompoundCharacterCacheManager()->StepInternal(this);
          bAttachAll = false;
        }
        //
        if( bAttachAll )
        {
          AttachAllCacheOnCommit();
        }
      }
    }
    m_bCached = true;
  }
}
//
void CCompoundCharacter::CacheCHRs(bool bCache)
{
  if( !bCache )
  {
    m_cached_chrs.resize(0);
  }
  else
  {
    int numAttached = GetNumAttachedInternal();
    for (int attach = 0; attach < numAttached; attach++)
    {
      CCharacterPartAttachment* pPartAttachmentObject = static_cast<CCharacterPartAttachment*>(GetAttachedPartInfoInternal(attach));
      if( pPartAttachmentObject )
      {
        if( pPartAttachmentObject->GetICharacterInstance() )
        {
          m_cached_chrs.push_back(pPartAttachmentObject->GetICharacterInstance());
        }
      }
    }
  }
}
//-------------------------------------------------------------------------
void CCompoundCharacter::ExtractMaterialName(string& filename)
{
	PathUtil::RemoveExtension(filename);
}
//-------------------------------------------------------------------------
static int getLastPartID(IPhysicalEntity *pent,int afterIndex)
{
  int maxPartsId = -1;
  pe_status_nparts statusTmp;
  int nparts = pent->GetStatus(&statusTmp);
  for (int p=0; p<nparts; p++)
  {
    pe_params_part pp;
    pp.ipart = p;
    if (pent->GetParams(&pp))
    {
      if( pp.partid>=afterIndex )
      {
        maxPartsId = max(pp.partid,maxPartsId);
      }
    }
  }
  return maxPartsId;

}
//-------------------------------------------------------------------------
bool CCharacterPartAttachment::PhysicalizeAttachment( IAttachmentManager* pManager, int idx, int nLod, IPhysicalEntity *pent, const Vec3 &offset )
{
  if( !m_pCharInstance )
    return true;
  //
  IPhysicalWorld *pPhysWorld = gEnv->pPhysicalWorld;
  if (!pPhysWorld)
    return false;

  m_pRootPhysicalEntity = pent;
  //
  if(pent)
  {
    for(size_t n=0;n<m_addedPartsList.size();n++)
    {
      //pent->RemoveGeometry(part->m_addedPartsList[n].m_partId);
    }
    m_addedPartsList.resize(0);
    //return;
    pe_geomparams gp;
    pe_articgeomparams agp;
    pe_type pentype = pent->GetType();
    pe_geomparams *pgp = pentype==PE_ARTICULATED ? &agp:&gp;

    ICharacterInstance* pInstanceMaster = pManager->GetSkelInstance();
    ICharacterModel* pModelMaster = pInstanceMaster->GetICharacterModel();
    //std::vector<CModelJoint>& arrModelJointsMaster = pModelMaster->m_pModelSkeleton->m_arrModelJoints;
    uint32 numBonesMaster = pInstanceMaster->GetISkeletonPose()->GetJointCount();

    ICharacterModel* pModel = m_pCharInstance->GetICharacterModel();
    //std::vector<CModelJoint>& arrModelJoints = pModel->m_pModelSkeleton->m_arrModelJoints;
    uint32 numBones = m_pCharInstance->GetISkeletonPose()->GetJointCount();
    CRY_ASSERT(numBones);
    // physicalize only merged physics if exists
    bool bHasMergedPhysics = m_cc->GetInterfaceByName(CC_PHYS_PART_ATTACHMENT_NAME)!=NULL ? true : false;
    IAttachment* pAtt = pManager->GetInterfaceByIndex(idx);
    if( m_cc->isMyAttachment(pAtt) )
    {
      if( bHasMergedPhysics && strcmp(pAtt->GetName(),CC_PHYS_PART_ATTACHMENT_NAME)!=0 )
      {
        return true;
      }
    }
    //
    const char* modelNameFull = pModel->GetName();
    size_t modelNameFullLen = modelNameFull ? strlen(modelNameFull) : 0;
    if( modelNameFullLen!=0 )
    {
      string modelName = modelNameFull;
      const char* pFirstChar = strchr(modelNameFull,'$');
      if( pFirstChar!=NULL )
      {
        modelName = string(modelNameFull,pFirstChar);
      }
      //
      for (uint32 i=0; i <numBones; i++)
      {
        //CModelJoint& joint = pModel->m_pModelSkeleton->m_arrModelJoints[i];
        const char * sn = m_pCharInstance->GetISkeletonPose()->GetJointNameByID(i);
        if (pModel->GetJointPhysGeom(i,nLod)) 
        {
          bool bAvailInMasterSkeleton = false;
          for(uint32 m=0; m<numBonesMaster; m++) 
          {
            const char * mn = pInstanceMaster->GetISkeletonPose()->GetJointNameByID(m);
            if (stricmp(sn,mn) == 0) 
            {
              bAvailInMasterSkeleton = true;
              break;
            }
          }
          // if joint name contains model name if format $MODELNAME$
          if( sn==NULL || sn[0]!='$' || strnicmp(sn+1,modelName,modelName.size())!=0 )
            continue;
          // if not avail in parent skeleton
          if( bAvailInMasterSkeleton )
            continue;
          //
          // find parent bone
          //
          //int32 thisJointParentIndex = m_pCharInstance->GetISkeletonPose()->GetJointParentIndex(i);
          ISkeletonPose* pose = m_pCharInstance->GetISkeletonPose();
          int32 thisJointParentIndex = pose->GetParentIDByID(i);
          const char* jointParentName = pose->GetJointNameByID(thisJointParentIndex);
          /*	CryFixedStringT<32> childName = pose->GetJointNameByID(i);
          childName.replace('$', '#');
          CryLog("Physicalizing %s to parent %s", childName.c_str(), jointParentName);
          */
          QuatT par = pose->GetDefaultAbsJointByID(thisJointParentIndex);
          QuatT chld = pose->GetDefaultAbsJointByID(i);

          QuatT ofs = par.GetInverted()*chld;
          /*	CryLog("Offset in child char %.3f %.3f %.3f", ofs.t.x,ofs.t.y,ofs.t.z);*/
          //
          //int16 parentId = pInstanceMaster->GetISkeletonPose()->GetJointIDByName(jointParentName);
          /*
          // get rel in 
          CRY_ASSERT(parentId!=-1);
          QuatT absPosParent = pInstanceMaster->GetISkeletonPose()->GetDefaultAbsJointByID(parentId);
          int16 parentIdCur = m_pCharInstance->GetISkeletonPose()->GetJointIDByName(jointParentName);
          QuatT absPosThis = absPosParent*m_pCharInstance->GetISkeletonPose()->GetDefaultRelJointByID(parentIdCur);
          //
          pgp->pos = absPosThis.t;
          pgp->q = absPosThis.q;
          */
          //
          pgp->flags = geom_collides|geom_floats;
          //
          int16 parentId = pInstanceMaster->GetISkeletonPose()->GetJointIDByName(jointParentName);
          if( parentId>0 && !pManager->GetSkelInstance()->GetIgnoreScaleForJoint(parentId) )
          {
            pgp->scale = pManager->GetSkelInstance()->GetUniformScale();
          }
          //else
          //{
          //  pgp->scale = 1.0f;
          //}
          //
          int geomId = getLastPartID(pent,CA_PARTS_PHYSPARTS_START_INDEX);
          if( geomId==-1 )
          {
            geomId = CA_PARTS_PHYSPARTS_START_INDEX;
          }
          else
          {
            geomId++;
          }
          // figure out parent id for master and this instance (parent bone with physics)
          int idparentbody = pInstanceMaster->GetISkeletonPose()->getBonePhysParentOrSelfIndex(parentId,nLod);
          const char* parentBodyName = pInstanceMaster->GetISkeletonPose()->GetJointNameByID(idparentbody);
          int idParentPhys = pose->GetJointIDByName(parentBodyName);
          //
          agp.idbody = idparentbody;
          //if( pentype==PE_ARTICULATED )
          //{
          //  pgp->flags = geom_colltype_ray;
          //}
          //
          int surface_type = -1;
          if( m_partsSurfTypes.size() )
          {
            // looks like merged physics
            std::map<string,int>::iterator it = m_partsSurfTypes.begin();
            for(;it!=m_partsSurfTypes.end();it++)
            {
              if( strstr(sn,it->first.c_str())!=NULL )
              {
                surface_type = it->second;
                break;
              }
            }
          }
          else
          {
            surface_type = m_surface_type_id;
          }
          int mtlMap[1];
          if( surface_type!=-1 )
          {
            pgp->surface_idx = 0;
            pgp->nMats = 1;
            mtlMap[0] = surface_type;
            pgp->pMatMapping = mtlMap;
          }
          // calculate abs transform of our body
          QuatT parentAbs = pInstanceMaster->GetISkeletonPose()->GetDefaultAbsJointByID(parentId);
          QuatT localPos = m_pCharInstance->GetISkeletonPose()->GetDefaultRelJointByID(i);
          QuatT thisAbs = parentAbs*localPos;
          //
          pgp->pos = thisAbs.t;
          pgp->q = thisAbs.q;
          //
          int newPart = pent->AddGeometry(pModel->GetJointPhysGeom(i,nLod),pgp,geomId);
          if(newPart == -1)
          {
            CryFixedStringT<32> childName = pose->GetJointNameByID(i);
            childName.replace('$', '#');
            CryWarning(VALIDATOR_MODULE_PHYSICS, VALIDATOR_ERROR, "Error adding physical proxy for %s", childName.c_str());
          }
          else
          {
            // calculate rel transform from master phys bone in child coords
            QuatT childPhysBoneParent = m_pCharInstance->GetISkeletonPose()->GetDefaultAbsJointByID(idParentPhys);
            QuatT childPhysBone = m_pCharInstance->GetISkeletonPose()->GetDefaultAbsJointByID(i);
            QuatT childPhysBoneRel = childPhysBoneParent.GetInverted()*childPhysBone;
            //
            CCharacterPartAttachment::AddedPhysicsParts parts;
            parts.m_local = m_pCharInstance->GetISkeletonPose()->GetDefaultRelJointByID(i);
            parts.m_partId = newPart;
            parts.m_parentJointIndex = parentId;
            parts.m_bonePhysParentIndex = idparentbody;
            parts.m_bonePhysLocal = childPhysBoneRel;
            m_addedPartsList.push_back(parts);
          }
        }
      }
    }
  }
  return true;
}


//
namespace 
{
  //
  // Neck
  //
  static const char* neck_bones[] =
  {
    "Bip01 Head",
    "Bip01 Head",
    "Bip01 Head",
    "Bip01 Head",
    "Bip01 Head",
  };
  static const int neck_numPoints = CRY_ARRAYSIZE(neck_bones);
  static const Vec3 neck_bones_offset[neck_numPoints] =
  {
    Vec3(0,-0.2f,0),
    Vec3(0.3f,0,0),
    Vec3(0,0,0.2f),
    Vec3(0,0,-0.2f),
    Vec3(0,0,0),
  };
  static const float neck_invmasses[neck_numPoints] =
  {
    1,
    0,
    0,
    0,
    0,
  };
  static const Verlet::Constraint neck_constraints[] = 
  {
    Verlet::Constraint(0,1,Vec3(neck_UpperDamper_Default)),
    Verlet::Constraint(0,2,Vec3(neck_LowerDamper_Default)),
    Verlet::Constraint(0,3,Vec3(neck_LowerDamper_Default)),
  };
  static const int neck_numConstraints = CRY_ARRAYSIZE(neck_constraints);
  // 1 is for upper constraints
  static const int neck_constraints_flags[neck_numConstraints] = 
  {
    1,
    0,
    0,
  };
}


//-------------------------------------------------------------------------
bool CCharacterPartAttachment::UpdatePhysicalizedAttachment( IAttachmentManager* pManager, int idx, IPhysicalEntity *pent, const QuatT &offsetIn)
{
  QuatT offset = offsetIn;
  if( !m_pCharInstance )
    return true;
  //
  if( m_cc )
    m_cc->Simulate();
  //
  IAttachment* pAtt = pManager->GetInterfaceByIndex(idx);
  if( !m_cc->isMyAttachment(pAtt) )
    return true;
  //
  ICharacterModel* pModel = m_pCharInstance->GetICharacterModel();
  CCharacterPartsManager* pIPartsManager = static_cast<CCharacterPartsManager*>(gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager());
  //
  bool bUpdateGroin = false;
  bool bUpdateNeck = false;
  if( pIPartsManager->m_cc_groinSimulation || pIPartsManager->m_cc_neckSimulation )
  {
    // check if exists in manager
    if( strcmp(pAtt->GetName(),CC_PHYS_PART_ATTACHMENT_NAME)==0 )
    {
      bUpdateGroin = true;
      bUpdateNeck = true;
    }
    else
    {
      const ICharacterPart* pTargetPart = pIPartsManager->GetPartByName(pAtt->GetName());
      if (pTargetPart != NULL)
      {
        if( strcmp(pTargetPart->GetSlot(),"Groin")==0 )
        {
          bUpdateGroin = true;
        }
        if( strcmp(pTargetPart->GetSlot(),"Helmet")==0 )
        {
          bUpdateNeck = true;
        }
      }
    }
  }
  if( pIPartsManager->m_cc_groinSimulation && bUpdateGroin )
  {
    ICharacterInstance* pInstanceMaster = pManager->GetSkelInstance();
    //
    int16 pelvis = pInstanceMaster->GetISkeletonPose()->GetJointIDByName("Bip01 Pelvis");
    int16 lthigh = pInstanceMaster->GetISkeletonPose()->GetJointIDByName("Bip01 L Thigh");
    int16 rthigh = pInstanceMaster->GetISkeletonPose()->GetJointIDByName("Bip01 R Thigh");
    //
    //
    int16 igroin_front_base = pInstanceMaster->GetISkeletonPose()->GetJointIDByName("Bip01 Groin Front");
    int16 igroin_front_base_parent = pInstanceMaster->GetISkeletonPose()->GetParentIDByID(igroin_front_base);
    int16 igroin_front = m_pCharInstance->GetISkeletonPose()->GetJointIDByName("Bip01 Groin Front");
    int16 igroin_front_fat = m_pCharInstance->GetISkeletonPose()->GetJointIDByName("Bip01 Groin Front_fat");
    int16 igroin_front_thin = m_pCharInstance->GetISkeletonPose()->GetJointIDByName("Bip01 Groin Front_thin");
    //
    int16 igroin_back_base = pInstanceMaster->GetISkeletonPose()->GetJointIDByName("Bip01 Groin Back");
    int16 igroin_back_base_parent = pInstanceMaster->GetISkeletonPose()->GetParentIDByID(igroin_back_base);
    int16 igroin_back = m_pCharInstance->GetISkeletonPose()->GetJointIDByName("Bip01 Groin Back");
    int16 igroin_back_fat = m_pCharInstance->GetISkeletonPose()->GetJointIDByName("Bip01 Groin Back_fat");
    int16 igroin_back_thin = m_pCharInstance->GetISkeletonPose()->GetJointIDByName("Bip01 Groin Back_thin");
    //
    const QuatT& masterPelvisPos = pInstanceMaster->GetISkeletonPose()->GetAbsJointByID(pelvis);
    const QuatT& masterLTPos = pInstanceMaster->GetISkeletonPose()->GetAbsJointByID(lthigh);
    const QuatT& masterRTPos = pInstanceMaster->GetISkeletonPose()->GetAbsJointByID(rthigh);
    //
    int16 groin_bones[10] = 
    {
      igroin_front_base,igroin_front,igroin_front_fat,igroin_front_thin,igroin_front_base_parent,
      igroin_back_base,igroin_back,igroin_back_fat,igroin_back_thin,igroin_back_base_parent,
    };
    float clampVals[4] = { 0, DEG2RAD(90), DEG2RAD(-45), 0,};
    bool bMinOrMax[2] = { false,true };
    for(int n=0;n<2;n++)
    {
      int16 groin_base = groin_bones[n*5+0];
      int16 groin = groin_bones[n*5+1];
      int16 groin_fat = groin_bones[n*5+2];
      int16 groin_thin = groin_bones[n*5+3];
      int16 groin_base_parent = groin_bones[n*5+4];
      if( groin!=-1 && groin_fat!=-1 && groin_thin!=-1 )
      {
        QuatT pos = m_pCharInstance->GetISkeletonPose()->GetDefaultRelJointByID(groin);
        const QuatT& pos_fat = m_pCharInstance->GetISkeletonPose()->GetDefaultRelJointByID(groin_fat);
        const QuatT& pos_thin = m_pCharInstance->GetISkeletonPose()->GetDefaultRelJointByID(groin_thin);
        QuatT defPos = m_pCharInstance->GetISkeletonPose()->GetDefaultAbsJointByID(groin);
        //const QuatT& defPos_fat = m_pCharInstance->GetISkeletonPose()->GetDefaultAbsJointByID(groin_fat);
        //const QuatT& defPos_thin = m_pCharInstance->GetISkeletonPose()->GetDefaultAbsJointByID(groin_thin);
        //
        QuatT fatnessPos = pos;
        //
        float FatValue,ThinValue;
        GetFatThinRatios(m_cc->GetFatness(),FatValue,ThinValue);
        if( FatValue>0 )
        {
          fatnessPos = QuatT::CreateNLerp(pos,pos_fat,FatValue);
        }
        else
        {
          fatnessPos = QuatT::CreateNLerp(pos,pos_thin,ThinValue);
        }
        QuatT fatnessPosNoScale = fatnessPos;
        //
        if( !pInstanceMaster->GetIgnoreScaleForJoint(groin_base_parent) )
        {
          fatnessPos.t*=pManager->GetSkinInstance()->GetUniformScale();
          //pos.t*=pManager->GetSkinInstance()->GetUniformScale();
          //defPos.t*=pManager->GetSkinInstance()->GetUniformScale();
        }
        const QuatT& masterPos = pInstanceMaster->GetISkeletonPose()->GetAbsJointByID(groin_base);
        const QuatT& masterPosRel = pInstanceMaster->GetISkeletonPose()->GetRelJointByID(groin_base);
        const QuatT& masterPosParent = pInstanceMaster->GetISkeletonPose()->GetAbsJointByID(groin_base_parent);
        QuatT thisAbs = masterPosParent*fatnessPos;
        //ITimer* timer = gEnv->pGame->GetIGameFramework()->GetISystem()->GetITimer();
        //
        QuatT pelvisInv = masterPelvisPos.GetInverted();
        Vec3 gravityVec = Vec3(0,0,-1);
        Vec3 gravityVecPSpace = pelvisInv.q*gravityVec;
        //
        Vec3 pelvisDownPSpace = Vec3(-1,0,0);
        //Vec3 pelvisDown = masterPelvisPos*pelvisDownPSpace;
        //
        // pelvis space
        //     x (up) 
        //     ^
        //     |
        //     |
        //     |
        //     |
        //     0---------> y (front)
        //
        Vec3 pelvisFrontPSpace = Vec3(0,1,0);
        //Vec3 pelvisFront = masterPelvisPos*pelvisFrontPSpace;
        //
        //      Vec3 pelvisDownPSpace = pelvisInv.q*pelvisDown;
        //      Vec3 pelvisFront = masterPelvisPos*Vec3(0,1,0);
        //Vec3 pelvisDown = masterPelvisPos*Vec3(-1,0,0);
        Vec3 LTDown = masterLTPos.q*Vec3(1,0,0);
        //Vec3 LTDownCP = masterPelvisPos*Vec3(1,0,0);// control point
        Vec3 LTDownPSpace = pelvisInv.q*LTDown;//masterPelvisPos*Vec3(1,0,0);
        //
        Vec3 RTDown = masterRTPos.q*Vec3(1,0,0);
        Vec3 RTDownPSpace = pelvisInv.q*RTDown;//masterPelvisPos*Vec3(1,0,0);
        //
        //convert all to pelvis space
        //
        float constraintAngleCos = cosf(DEG2RAD(pIPartsManager->m_cc_groinMaxWeightAngle));
        float constraintAngleCosToZero = cosf(DEG2RAD(pIPartsManager->m_cc_groinMinWeightAngle));
        float constraintAngleFaloffDelta = constraintAngleCos-constraintAngleCosToZero;
        //
        Vec3 LTDownPSpaceYZ = LTDownPSpace;
        LTDownPSpaceYZ.x=0;
        LTDownPSpaceYZ.Normalize();
        float dotL = fabs(Vec3(0,1,0).Dot(LTDownPSpaceYZ));
        if( dotL<constraintAngleCos )
        {
          dotL = CLAMP((dotL-constraintAngleCosToZero)/constraintAngleFaloffDelta,0,1.0f);
        }
        else
        {
          dotL = 1.0f;
        }
        Vec3 RTDownPSpaceYZ = RTDownPSpace;
        RTDownPSpaceYZ.x=0;
        RTDownPSpaceYZ.Normalize();
        float dotR = fabs(Vec3(0,1,0).Dot(RTDownPSpaceYZ));
        if( dotR<constraintAngleCos )
        {
          dotR = CLAMP((dotR-constraintAngleCosToZero)/constraintAngleFaloffDelta,0,1.0f);

        }
        else
        {
          dotR = 1.0f;
        }
        //
        float angleL = atan2(LTDownPSpace.y,-LTDownPSpace.x);// 
        float angleR = atan2(RTDownPSpace.y,-RTDownPSpace.x);// 
        //      float angleR = atan2(RTDown.y,RTDown.x);// 
        //      float angleR = atan2(RTDown.y,RTDown.x);// 
        //float dotLDown = LTDownPSpace.GetNormalized().Dot(Vec3(-1,0,0));
        //      float dotFront = pelvisDown.GetNormalized().Dot(LTDown.GetNormalized());
        //      thisAbs.q.SetRotationX(cry_sinf(timer->GetAsyncCurTime()));
        //      pInstanceMaster->GetISkeletonPose()->SetAbsJointByID(groin_base,thisAbs);
        // calc rot in pelvis space
        Quat rot; 
        float angle = bMinOrMax[n] ? (min(angleL*dotL,angleR*dotR)) : max(angleL*dotL,angleR*dotR);
        angle = CLAMP(angle,clampVals[n*2+0],clampVals[n*2+1]);
        rot.SetRotationX(angle);
        //
        thisAbs.q = thisAbs.q*rot;//.SetRotationX(angleL);//cry_sinf(timer->GetAsyncCurTime()));
        //
        QuatT basePos = pos;
        if( !pInstanceMaster->GetIgnoreScaleForJoint(groin_base_parent) )
        {
          //basePos.t*=pManager->GetSkinInstance()->GetUniformScale();
        }
        QuatT tt = (fatnessPosNoScale.GetInverted())*(basePos);
        QuatT jointInv = (defPos*tt.GetInverted()).GetInverted();
        if( !pInstanceMaster->GetIgnoreScaleForJoint(groin_base_parent) )
        {
          //tt.t*=pManager->GetSkinInstance()->GetUniformScale();
          //defPos.t*=pManager->GetSkinInstance()->GetUniformScale();
//          thisAbs.t*=pManager->GetSkinInstance()->GetUniformScale();
//          jointInv.t*=pManager->GetSkinInstance()->GetUniformScale();
        }

        pInstanceMaster->GetISkeletonPose()->SetDefaultAbsInvJointByID(groin_base,jointInv);
        //
        //
        pInstanceMaster->GetISkeletonPose()->SetAbsJointByID(groin_base,thisAbs);
        //m_pCharInstance->GetISkeletonPose()->SetAbsJointByID(groin,thisAbs);
      }
    }
  }
  if( pIPartsManager->m_cc_neckSimulation && bUpdateNeck )
  {
    if( m_neckSim.m_verlet.m_particles.size()==0 )
    {
      m_neckSim.Initialize(m_cc->GetCharacterInstance(),neck_bones,neck_bones_offset,neck_invmasses,neck_numPoints,neck_constraints,neck_numConstraints);
    }
    if( m_neckSim.BeginSimulate() )
    {
      ICharacterInstance* pInstanceMaster = pManager->GetSkelInstance();
      //
      Vec3 animVecNonNorm = (m_neckSim.m_animated_pose[0]-m_neckSim.m_animated_pose[4]);
      Vec3 animVec = animVecNonNorm.GetNormalized();
      //
      if( neck_UpperDamper_Default!=pIPartsManager->m_cc_neck_upperDamper || 
        neck_LowerDamper_Default!=pIPartsManager->m_cc_neck_lowerDamper )
      {
        for(int n=0;n<m_neckSim.m_verlet.m_constraints.size();n++)
        {
          if( neck_constraints_flags[n]==1 )
          {
            m_neckSim.m_verlet.m_constraints[n].springDamper = Vec3(pIPartsManager->m_cc_neck_upperDamper);
          }
          else
          {
            m_neckSim.m_verlet.m_constraints[n].springDamper = Vec3(pIPartsManager->m_cc_neck_lowerDamper);
          }
        }
        neck_UpperDamper_Default = pIPartsManager->m_cc_neck_upperDamper;
        neck_LowerDamper_Default = pIPartsManager->m_cc_neck_lowerDamper;
      }
      //
      m_neckSim.Simulate();
      m_neckSim.EndSimulate(false,pIPartsManager->m_cc_neck_minDist,pIPartsManager->m_cc_neck_maxDist);
      //m_neckSim.DrawDebugLines();
      //
      // calc rot in pelvis space
      Vec3 animVecSimNonNorm = (m_neckSim.m_verlet.m_particles[0].p - m_neckSim.m_verlet.m_particles[4].p).GetNormalized();
      Vec3 animVecSim = animVecSimNonNorm.GetNormalized();
      //
      //IRenderAuxGeom* pRenAux = gEnv->pRenderer->GetIRenderAuxGeom();
      //if (pRenAux)
      //{
      //  pRenAux->DrawLine(m_neckSim.m_animated_pose[4],RGBA8(0xff,0x00,0xff,0xff),m_neckSim.m_animated_pose[4]+animVec,RGBA8(0xff,0x00,0xff,0xff),2);
      //  pRenAux->DrawLine(m_neckSim.m_animated_pose[4],RGBA8(0x00,0x00,0xff,0xff),m_neckSim.m_animated_pose[4]+animVecSim,RGBA8(0x00,0x00,0xff,0xff),2);
      //}
      //
      int16 headBone = pInstanceMaster->GetISkeletonPose()->GetJointIDByName("Bip01 Head");
      int16 headBoneParent = pInstanceMaster->GetISkeletonPose()->GetParentIDByID(headBone);
      int16 neckBone = pInstanceMaster->GetISkeletonPose()->GetJointIDByName("Bip01 Neck");
      //
      Vec3 hNeckTrDef = pInstanceMaster->GetISkeletonPose()->GetDefaultAbsJointByID(headBone)*neck_bones_offset[0];
      QuatT neckBoneTrDef = pInstanceMaster->GetISkeletonPose()->GetDefaultAbsJointByID(neckBone);
      float lenSpineHNeckDef = (neckBoneTrDef.t-hNeckTrDef).len();
      //
      Vec3 hNeckTrCur = pInstanceMaster->GetISkeletonPose()->GetAbsJointByID(headBone)*neck_bones_offset[0];
      QuatT neckBoneTrCur = pInstanceMaster->GetISkeletonPose()->GetAbsJointByID(neckBone);
      float lenSpineHNeckCur = (neckBoneTrCur.t-hNeckTrCur).len();
      //
      float delta = (animVecSim-animVec).len();
      delta*=pIPartsManager->m_cc_neckAngleMultilier;
      delta -= delta*0.5f;
      delta+=DEG2RAD(pIPartsManager->m_cc_neckAngleBias);
      //
      Quat headBoneCur = pInstanceMaster->GetISkeletonPose()->GetRelJointByID(headBone).q;
      Vec3 fwd = headBoneCur*Vec3(0,1,0);
      float angleUpDown = atan2f(fwd.x,fwd.y);
      delta-=angleUpDown;
      //
      Vec3 fwdLR = headBoneCur*Vec3(1,0,0);
      float angleLR = -atan2f(fwdLR.z,fwdLR.x);
      angleLR*=pIPartsManager->m_cc_neckAngleMultilierLR;
      //
      int16 neckHBone = pInstanceMaster->GetISkeletonPose()->GetJointIDByName("Bip01 HNeck");
      int16 neckHBone_parent = pInstanceMaster->GetISkeletonPose()->GetParentIDByID(neckHBone);
      //
      const QuatT& masterPosParent = pInstanceMaster->GetISkeletonPose()->GetAbsJointByID(neckHBone_parent);
      QuatT relPos = pInstanceMaster->GetISkeletonPose()->GetDefaultRelJointByID(neckHBone);
      //
      QuatT thisAbs = masterPosParent*relPos;
      //
      Quat rot; 
      rot.SetIdentity();
      float angle = delta;//DEG2RAD(delta);//cry_acosf(dot);
      rot.SetRotationX(angle);
      //
      Quat rotLR; 
      rotLR.SetRotationY(angleLR);
      //
      thisAbs.q = thisAbs.q*(rot*rotLR);//.SetRotationX(angleL);//cry_sinf(timer->GetAsyncCurTime()));
      //
      pInstanceMaster->GetISkeletonPose()->SetAbsJointByID(neckHBone,thisAbs);
    }
  }
  IPhysicalWorld *pPhysWorld = gEnv->pPhysicalWorld;
  if (!pPhysWorld)
    return true;

  if(pent && pManager->GetSkelInstance())
  {
    //   pe_status_pos entPos;
//    pent->GetStatus(&entPos);
//    pe_status_awake statusTmp;
//    int bPhysicsAwake = pent->GetStatus(&statusTmp);
    //
    pe_params_joint pj;
    pe_params_part partpos;
    ICharacterInstance* pInstanceMaster = pManager->GetSkelInstance();
    ICharacterModel* pModelMaster = pManager->GetSkinInstance()->GetICharacterModel();
    for(size_t n=0;n<m_addedPartsList.size();n++)
    {
      bool bOffset = true;
      //QuatT parentAbs = pInstanceMaster->GetISkeletonPose()->GetAbsJointByID(m_addedPartsList[n].m_parentJointIndex);
      //QuatT localPos = m_addedPartsList[n].m_local;
      //
      ///*
      bOffset = false;
      pe_status_pos pePos;
      pePos.flags = status_local;
      pePos.partid = m_addedPartsList[n].m_bonePhysParentIndex;
      pent->GetStatus(&pePos);
      QuatT parentAbs(pePos.pos,pePos.q);
      QuatT localPos = m_addedPartsList[n].m_bonePhysLocal;
      //
      //
      if( m_addedPartsList[n].m_parentJointIndex>0 && !pManager->GetSkelInstance()->GetIgnoreScaleForJoint(m_addedPartsList[n].m_parentJointIndex) )
      {
        localPos.t*=pManager->GetSkinInstance()->GetUniformScale();
      }
      QuatT thisAbs = parentAbs*localPos;
      if( bOffset )
      {
        thisAbs = offset*thisAbs;
      }
      //
/*
      IRenderAuxGeom* pRenAux = gEnv->pRenderer->GetIRenderAuxGeom();
      if (pRenAux)
      {
      pe_status_pos pePos;
      //pePos.flags = status_local;
      //pePos.partid = 1;
      pent->GetStatus(&pePos);
      //
      QuatT chrPos(pePos.pos,pePos.q);
      pRenAux->DrawLine(chrPos*thisAbs*Vec3(0,0,0),RGBA8(0xff,0x00,0x00,0xff),chrPos*thisAbs*Vec3(0,0,0)+Vec3(0,0,0.5f),RGBA8(0x00,0xff,0x00,0xff),20);
      }
*/
      //
      partpos.partid = m_addedPartsList[n].m_partId;
      //partpos.bRecalcBBox = i==j;
      partpos.pos = thisAbs.t;
      partpos.q = thisAbs.q;
      pent->SetParams(&partpos);
    }
  }
  return true;
}

void CCharacterPartAttachment::OnRemoveAttachment(IAttachmentManager* pManager, int idx)
{
  // remove physical entity
  IPhysicalWorld *pPhysWorld = gEnv->pPhysicalWorld;
  if (pPhysWorld)
  {
    if(m_pRootPhysicalEntity)
    {
      for(size_t n=0;n<m_addedPartsList.size();n++)
      {
        //part->m_pRootPhysicalEntity->RemoveGeometry(part->m_addedPartsList[n].m_partId);
      }
      m_addedPartsList.resize(0);
    }
  }
}
/*

int CCharacterPartAttachment::GetNumFaceAttachments() const 
{
  return m_faceAttachments.size(); 
}

const char* GetFaceAttachmentName(int index) const
{
  if (index >= 0 && index < (int)m_faceAttachments.size())
    return m_faceAttachments[index].m_name.c_str();
  return NULL;
}

*/
void CCharacterPartAttachment::AddFaceAttachment(const char* partName,const char* name, QuatT attRelativeDefault, QuatT attAbsoluteDefault, uint32 faceNr)
{
  CCompoundCharacter::SAttachedPart* pPart = m_cc->GetAttachedPart(partName);
  if( pPart )
  {
    pPart->AddFaceAttachmentDesc(partName,name,attRelativeDefault,attAbsoluteDefault,faceNr);
  }
}

void CCompoundCharacter::SAttachedPart::AddFaceAttachmentDesc(const char* partName, const char* name, QuatT attRelativeDefault, QuatT attAbsoluteDefault, uint32 faceNr)
{
  SFaceAttachmentData newAtt;
  newAtt.m_name = name;
  newAtt.m_AttRelativeDefault = attRelativeDefault;
  newAtt.m_AttAbsoluteDefault = attAbsoluteDefault;
  newAtt.m_FaceNr = faceNr;
  m_faceAttachmentsData.push_back(newAtt);
}

void CCharacterPartsManager::PrintDebugInfo() const
{
  if( m_cc_profile==1 )
  {
    int x = 0;
    int y = 60;
    int h = 5;
    std::set<string> texmaps_all;
    int texCount_all = 0;
    int texMemCount_all = 0;
    for(int i=0;i<m_registered.size();i++)
    {
      //
      int texCount = 0;
      int texMemCount = 0;
      std::set<string> texmaps;
      IAttachmentManager* pManager = m_registered[i]->GetCharacterInstance()->GetIAttachmentManager();
      for(int a=0;a<pManager->GetAttachmentCount();a++)
      {
        IAttachment* att = pManager->GetInterfaceByIndex(a);
        if( !m_registered[i]->isMyAttachment(att) )
          continue;
        //
        if( att->IsAttachmentHidden() )
          continue;
        //
        CCharacterPartAttachment* charAtt = static_cast<CCharacterPartAttachment*>(att->GetIAttachmentObject());
        if( !charAtt )
          continue;
        //
        std::vector<IMaterial*> mtlList;
        IMaterial* pMtl = charAtt->GetMaterial();
        if( pMtl )
        {
          mtlList.push_back(pMtl);
          // calc material state
          for(int m=0;m<pMtl->GetSubMtlCount();m++)
          {
            mtlList.push_back(pMtl->GetSubMtl(m));
          }
        }
        //
        for(int mtl=0;mtl<mtlList.size();mtl++)
        {
          SShaderItem& si = mtlList[mtl]->GetShaderItem(0);
          //
          for(int n=0;n<EFTT_MAX;n++)
          {
            //
            if( !si.m_pShaderResources->GetTexture(n) )
              continue;
            //
            SEfResTexture* pTexture = si.m_pShaderResources->GetTexture(n);
            if( pTexture )
            {
              STexSampler& sampler = pTexture->m_Sampler;
              ITexture* pTex = sampler.m_pITex;
              const char* dds_filename = pTex->GetName();
              if( texmaps.find(dds_filename)==texmaps.end() )
              {
                texmaps.insert(dds_filename);
                texMemCount += pTex->GetDeviceDataSize();
                texCount++;
              }
              if( texmaps_all.find(dds_filename)==texmaps_all.end() )
              {
                texmaps_all.insert(dds_filename);
                texMemCount_all += pTex->GetDeviceDataSize();
                texCount_all++;
              }
            }
          }
        }
      }
      //
      //
      const char* name = m_registered[i]->GetName();
			float col[4] = {1,1,1,1};
			gEnv->pRenderer->Draw2dLabel( (float)x,(float)y,1,col,false,"%-20s: | UTexCount:%4d, | UTexMem:%5d",name,texCount,texMemCount/1024);
      y+=2*h;
    }
    //
		float col[4] = {1,1,1,1};
    gEnv->pRenderer->Draw2dLabel((float)x,(float)y,1,col,false,"-----------------------------------------------------------------");
    gEnv->pRenderer->Draw2dLabel((float)x,(float)y+2*h,1,col,false,"%-20s: | UTexCount:%4d, | UTexMem:%5d","Total",texCount_all,texMemCount_all/1024);
  }
}

void CCharacterPartsManager::OnPostUpdate(float fDeltaTime)
{
  PrintDebugInfo();
}

bool CCompoundCharacterCache::IsFullyCachedJob(ICompoundCharacter* pCharacter,Job& job,string* out_chrfilename)
{
  if( job.m_jobType==Job::jtBakeMorphs )
  {
    //
    string chrFileName = GetCHRFileName(job.m_pCharacter,job.m_partName);
    if( out_chrfilename )
    {
      *out_chrfilename = chrFileName;
    }
    if( chrFileName.empty() )
    {
      return false;
    }
    if (!gEnv->pCryPak->IsFileExist(chrFileName))
    {
      return false;
    }
    return true;
  }
  else if( job.m_jobType==Job::jtBakePhysics )
  {
    //
    string chrFileName = GetPhysicsCHRFileName(job.m_pCharacter);
    if( out_chrfilename )
    {
      *out_chrfilename = chrFileName;
    }
    if( chrFileName.empty() )
    {
      return false;
    }
    if (!gEnv->pCryPak->IsFileExist(chrFileName))
    {
      return false;
    }
    return true;
  }
  return false;
}
bool CCompoundCharacterCache::IsFullyCached(ICompoundCharacter* pCharacter)
{
  // if all jobs is done before run all jobs in one pass
  bool bAllJobsIsCached = true;
  for(int j=0;j<m_pendingJobs.size();j++)
  {
    Job& job = m_pendingJobs[j];
    if( job.m_pCharacter!=pCharacter )
      continue;
    //
    if( !IsFullyCachedJob(pCharacter,job,NULL) )
    {
      bAllJobsIsCached = false;
      break;
    }
  }
  //
  if( bAllJobsIsCached )
    return true;
  return false;
}
//
namespace 
{
  static const char* breast_bones[] =
  {
    "Bip01 L Pec Control",
    "Bip01 R Pec Control",
    "Bip01 Neck",
    "#Bip01 R Pec Control",
    "#Bip01 R Pec Control",
    "#Bip01 R Pec Control",
  };
  static const int breast_numPoints = CRY_ARRAYSIZE(breast_bones);
  static Vec3 breast_bones_offset[breast_numPoints] =
  {
    Vec3(0,0,0),
    Vec3(0,0,0),
    Vec3(0,0,0),
    Vec3(0,-0.2f,0.2f),
    Vec3(0,-0.2f,-0.2f),
    Vec3(0,-0.1f,0),
  };
  static const float breast_invmasses[breast_numPoints] =
  {
    1,
    1,
    0,
    0,
    0,
  };
  static const Verlet::Constraint breast_constraints[] = 
  {
    Verlet::Constraint(0,2,Vec3(breast_UpperDamper_Default)),
    Verlet::Constraint(1,2,Vec3(breast_UpperDamper_Default)),
    Verlet::Constraint(0,3,Vec3(breast_LowerDamper_Default)),
    Verlet::Constraint(1,4,Vec3(breast_LowerDamper_Default)),
    Verlet::Constraint(0,1,Vec3(breast_LowerDamper_Default)),
    Verlet::Constraint(0,5,Vec3(breast_LowerDamper_Default)),
    Verlet::Constraint(1,5,Vec3(breast_LowerDamper_Default)),
  };
  static const int breast_numConstraints = CRY_ARRAYSIZE(breast_constraints);
  // 1 is for upper constraints
  static const int breast_constraints_flags[breast_numConstraints] = 
  {
    1,
    1,
    0,
    0,
    0,
    0,
    0,
  };
  //     b = 0.5
  //
  //             /
  //            /
  //           /
  //          /
  //         /
  //        /
  //
  //     b < 0.5 -> curve to right
  //     b > 0.5 -> curve to left
  static float bias( float b, float t )
  {
    if (b == 0.0f)
      return 0.0f;
    return cry_powf( t, cry_logf(b)/cry_logf(0.5f) );
  }
}
//
void CCompoundCharacter::InitSimulation()
{
  //
  // Breast
  //
  m_breastSim.Reset();
  // ignore male
  if( m_gender==GENDER_FEMALE )
  {
    m_breastSim.Initialize(m_pInstance,breast_bones,breast_bones_offset,breast_invmasses,breast_numPoints,breast_constraints,breast_numConstraints);
  }
}

void CCompoundCharacter::Simulate()
{
  //
  FUNCTION_PROFILER( GetISystem(),PROFILE_ANIMATION );
  //
  CCharacterPartsManager* pIPartsManager = static_cast<CCharacterPartsManager*>(gEnv->pGame->GetIGameFramework()->GetICharacterPartsManager());
  //
  // Optimize me: figure out this value not in every frame
  bool bSimBreast = true;
  for(int n=0;n<m_attachedParts.size();n++)
  {
    if( m_attachedParts[n].m_slot=="Vest" )
    {
      bSimBreast = false;
      break;
    }
  }
  if( pIPartsManager->m_cc_breast_simulation && bSimBreast )
  {
    //
    if( m_breastSim.BeginSimulate() )
    {
      // 
      if( breast_UpperDamper_Default!=pIPartsManager->m_cc_breast_upperDamper || 
        breast_LowerDamper_Default!=pIPartsManager->m_cc_breast_lowerDamper )
      {
        for(int n=0;n<m_breastSim.m_verlet.m_constraints.size();n++)
        {
          if( breast_constraints_flags[n]==1 )
          {
            m_breastSim.m_verlet.m_constraints[n].springDamper = Vec3(pIPartsManager->m_cc_breast_upperDamper);
          }
          else
          {
            m_breastSim.m_verlet.m_constraints[n].springDamper = Vec3(pIPartsManager->m_cc_breast_lowerDamper);
          }
        }
        breast_UpperDamper_Default = pIPartsManager->m_cc_breast_upperDamper;
        breast_LowerDamper_Default = pIPartsManager->m_cc_breast_lowerDamper;
      }
      // 
      if( breast_UpperRestRatio_Default!=pIPartsManager->m_cc_breast_upperRestRatio || 
        breast_LowerRestRatio_Default!=pIPartsManager->m_cc_breast_upperRestRatio )
      {
        for(int n=0;n<m_breastSim.m_verlet.m_constraints.size();n++)
        {
          float restLen = (m_breastSim.m_verlet.m_particles[m_breastSim.m_verlet.m_constraints[n].particleA].initp-m_breastSim.m_verlet.m_particles[m_breastSim.m_verlet.m_constraints[n].particleB].initp).len();
          if( breast_constraints_flags[n]==1 )
          {
            m_breastSim.m_verlet.m_constraints[n].restlength=restLen*pIPartsManager->m_cc_breast_upperRestRatio;
          }
          else
          {
            m_breastSim.m_verlet.m_constraints[n].restlength=restLen*pIPartsManager->m_cc_breast_lowerRestRatio;
          }
        }
        breast_UpperRestRatio_Default = pIPartsManager->m_cc_breast_upperRestRatio;
        breast_LowerRestRatio_Default = pIPartsManager->m_cc_breast_lowerRestRatio;
      }
      //
      m_breastSim.m_verlet.m_vGravity.Set(0,0,pIPartsManager->m_cc_breast_gravity);
      m_breastSim.Simulate();
      m_breastSim.EndSimulate(true,pIPartsManager->m_cc_breast_minDist,pIPartsManager->m_cc_breast_maxDist);
    }
  }
}
//
// integrate step
void Verlet::Intergrate() 
{
  float t2 = m_fTimeStep*m_fTimeStep;
  for(int i=0; i<m_particles.size(); i++) 
  {
    if( fabsf(m_particles[i].invmass)>0.0001f )
    {
      Vec3& x = m_particles[i].p;
      Vec3 temp = x;
      Vec3& oldx = m_particles[i].oldp;
      Vec3& a = m_particles[i].a;
      x += x-oldx+a*t2;
      oldx = temp;
    }
  }
}
//
bool CharacterVerletSystem::Initialize(ICharacterInstance* pInstance,const char* *bones,const Vec3 *offsets,const float* invmases,size_t pointsCount,const Verlet::Constraint* constraints,size_t numConstaints)
{
  m_pointsCount = pointsCount;
  m_bones = bones;
  m_offsets = offsets;
  m_invmases = invmases;
  m_pInstance = pInstance;
  m_lastSimFrame = INT_MAX;
  //
  m_verlet.Reset();
  // points
  // all calculations in character control space
  m_bones_ids.resize(pointsCount);
  m_animated_pose.resize(pointsCount);
  for(int i=0;i<pointsCount;i++)
  {
    if( bones[i][0]=='#' ) // use parent
    {
      int16 ctrl_id = m_pInstance->GetISkeletonPose()->GetJointIDByName(bones[i]+1);
      m_bones_ids[i] = m_pInstance->GetISkeletonPose()->GetParentIDByID(ctrl_id);
    }
    else
    {
      m_bones_ids[i] = m_pInstance->GetISkeletonPose()->GetJointIDByName(bones[i]);
    }
    if( m_bones_ids[i]==-1 )
    {
      m_verlet.Reset();
      CryLog("No bone found %s",bones[i]);
      return false;
    }
    QuatT tr = m_pInstance->GetISkeletonPose()->GetDefaultAbsJointByID(m_bones_ids[i]);
    Vec3 pos = tr*offsets[i];
    m_verlet.m_particles.push_back(Verlet::Particle(pos,Vec3(0,0,0),invmases[i]));
  }
  for(int i=0;i<numConstaints;i++)
  {
    // constraints
    Verlet::Constraint vconstraint;
    vconstraint.particleA = constraints[i].particleA;
    vconstraint.particleB = constraints[i].particleB;
    vconstraint.restlength = (m_verlet.m_particles[vconstraint.particleA].p-m_verlet.m_particles[vconstraint.particleB].p).len();
    vconstraint.springDamper = constraints[i].springDamper;
    m_verlet.m_constraints.push_back(vconstraint);
  }
  //
  m_verlet.m_vGravity.Set(0,0,-9.0f);
  return true;
}

bool CharacterVerletSystem::CanSimulate()
{
  //
  if( m_verlet.m_particles.size()==0 )
    return false;
  //
  ITimer::ETimer timer = gEnv->bEditor ? ITimer::ETIMER_UI : ITimer::ETIMER_GAME;
  int curFrame = gEnv->pRenderer->GetFrameID();
  if( curFrame==m_lastSimFrame )
    return false;
  return true;
}

bool CharacterVerletSystem::BeginSimulate()
{
  //
  if( !CanSimulate() )
    return false;
  // update control points poses
  for(int i=0;i<m_bones_ids.size();++i)
  {
    QuatT tr = m_pInstance->GetISkeletonPose()->GetAbsJointByID(m_bones_ids[i]);
    Vec3 pos = tr*m_offsets[i];
    m_animated_pose[i] = pos;
    //
    int16 boneId = m_bones_ids[i];
    if( fabsf(m_verlet.m_particles[i].invmass)<=0.0001f )
    {
      m_verlet.m_particles[i].p = pos;
    }
  }
  return true;
}
void CharacterVerletSystem::Simulate()
{
}
void CharacterVerletSystem::EndSimulate(bool applyResultToCharacter,float minDelta,float maxDelta)
{
/*
  for(int i=0;i<m_bones_ids.size();i++)
  {
  if( fabsf(m_verlet.m_particles[i].invmass)>=0.0001f )
  {
  Vec3& pos = m_verlet.m_particles[i].p;
  CryLog("BEFORE: %d %f %f %f",i,pos.x,pos.y,pos.z);
  }
  }
*/
  //
  float deltat = gEnv->pSystem->GetITimer()->GetFrameTime();
  m_verlet.TimeStep(deltat,3);
  //DrawDebugLines();
  //
  ITimer::ETimer timer = gEnv->bEditor ? ITimer::ETIMER_UI : ITimer::ETIMER_GAME;
  float frameStartTime = gEnv->pSystem->GetITimer()->GetFrameStartTime(timer).GetSeconds();
  deltat = gEnv->pSystem->GetITimer()->GetFrameTime();
  // 
  // float maxDelta = -FLT_MAX;
  for(int i=0;i<m_bones_ids.size();i++)
  {
    if( fabsf(m_verlet.m_particles[i].invmass)>=0.0001f )
    {
      Vec3& pos = m_verlet.m_particles[i].p;
//      CryLog("AFTER: %d %f %f %f",i,pos.x,pos.y,pos.z);
      //
      float delta = (pos-m_animated_pose[i]).len();
      //maxDelta = max(delta,maxDelta);
      // do not allow to move far away from static position
      if( delta>minDelta)
      {
        float t = (delta-minDelta)/(maxDelta-minDelta);
        if( t>1.0f )
        {
          t = 1.0f;
        }
        float blend = bias(0.2f,t);
        m_verlet.m_particles[i].p.SetLerp(pos,m_animated_pose[i],blend);
      }
      //m_verlet.m_particles[i].p = m_animated_pose[i];
    }
  }
  //
  if( applyResultToCharacter )
  {
    for(int i=0;i<m_bones_ids.size();i++)
    {
      int16 boneId = m_bones_ids[i];
      if( m_verlet.m_particles[i].invmass!=0.0f )
      {
        QuatT old = m_pInstance->GetISkeletonPose()->GetAbsJointByID(boneId);
        old.t = m_verlet.m_particles[i].p;
        m_pInstance->GetISkeletonPose()->SetAbsJointByID(boneId,old);
      }
    }
  }
  //
  m_lastSimFrame = gEnv->pRenderer->GetFrameID();
}

void CharacterVerletSystem::DrawDebugLines()
{
  IRenderAuxGeom* pRenAux = gEnv->pRenderer->GetIRenderAuxGeom();
  if (pRenAux)
  {
    for(int c=0;c<m_verlet.m_constraints.size();c++)
    {
      pRenAux->DrawLine(m_verlet.m_particles[m_verlet.m_constraints[c].particleA].p,RGBA8(0xff,0xff,0xff,0xff),m_verlet.m_particles[m_verlet.m_constraints[c].particleB].p,RGBA8(0xff,0xff,0xff,0xff),2);
    }
  }
}

#include UNIQUE_VIRTUAL_WRAPPER(ICharacterPart)
#include UNIQUE_VIRTUAL_WRAPPER(IPartModel)
#include UNIQUE_VIRTUAL_WRAPPER(IPartMaterial)
#include UNIQUE_VIRTUAL_WRAPPER(IPartSocket)
#include UNIQUE_VIRTUAL_WRAPPER(ICharacterPartTemplate)
#include UNIQUE_VIRTUAL_WRAPPER(IPartMorph)
#include UNIQUE_VIRTUAL_WRAPPER(ICompoundCharacter)
#include UNIQUE_VIRTUAL_WRAPPER(ICharacterPartAttachment)
#include UNIQUE_VIRTUAL_WRAPPER(ICharacterPartsManager)
#endif