/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2007-2008.
-------------------------------------------------------------------------
$Id$
$DateTime$
Description: Character parts manager.

-------------------------------------------------------------------------
History:
- 10:12:2007   15:30 : Created by Maxim Dyachenko

*************************************************************************/
#ifndef __CHARACTERPARTSMANAGER_H__
#define __CHARACTERPARTSMANAGER_H__

#pragma once

struct ILevelSystemListener;
class CCompoundCharacter;
class CCryAction;
class CCharacterPartsManager;

#include <ICharacterPartsManager.h>

//-------------------------------------------------------------------------
struct SPartMorph : public IPartMorph
{
  // Morph switch
  VIRTUAL const char* GetSwitch() const { return m_switch.c_str(); }
  VIRTUAL void SetSwitch(const char* switchName) { m_switch = switchName; }

  // Morph target inside model
  VIRTUAL const char* GetMorphTarget() const { return m_target.c_str(); }
  VIRTUAL void SetMorphTarget(const char* morphTarget) { m_target = morphTarget; }

protected:
  string m_switch;                  // switch to activate morph
  string m_target;                  // morph target name in .CHR
};

//-------------------------------------------------------------------------
struct SPartMaterial : public IPartMaterial
{
  //-------------------------------------------------------------------------
  // Material name
  VIRTUAL const char* GetName() const { return m_name.c_str(); }
  VIRTUAL void SetName(const char* name) { m_name = name; }

  //-------------------------------------------------------------------------
  // Material filename
  VIRTUAL const char* GetFilename() const { return m_filename.c_str(); }
  VIRTUAL void SetFilename(const char* filename) { m_filename = filename; }

protected:
  string m_name;                    // local material name
  string m_filename;                // filename of .MTL
};

//-------------------------------------------------------------------------
struct SPartSocket : public IPartSocket
{
  //-------------------------------------------------------------------------
  // Socket name
  VIRTUAL const char* GetName() const { return m_name.c_str(); }
  VIRTUAL void SetName(const char* name) { m_name = name; }

  //-------------------------------------------------------------------------
  // Edge of socket axis
  VIRTUAL int GetEdge() const { return m_edge; }
  VIRTUAL void SetEdge(int edge) { m_edge = edge; }

  //-------------------------------------------------------------------------
  // Limit angle
  VIRTUAL float GetAngle() const { return m_angle; }
  VIRTUAL void SetAngle(float angle) { m_angle = angle; }

  //-------------------------------------------------------------------------
  // Damping
  VIRTUAL float GetDamping() const { return m_damping; }
  VIRTUAL void SetDamping(float damping) { m_damping = damping; }

  //-------------------------------------------------------------------------
  // Allowed socket attachment types
  VIRTUAL int GetNumAllowedTypes() const { return m_allowed.size(); }

  VIRTUAL const char* GetAllowedTypeByIndex(int index) const
  {
    if (index >= 0 && index < m_allowed.size())
      return m_allowed[index].c_str();
    return NULL;
  }

  VIRTUAL void ClearAllowedTypes() { m_allowed.resize(0); }
  VIRTUAL void ReserveAllowedTypes(int size) { m_allowed.reserve(size); }
  VIRTUAL void AddAllowedType(const char* name) { m_allowed.push_back(name); }

protected:
  string  m_name;                     // socket name in .chr
  int     m_edge;                     // edge of socket axes
  float   m_angle;                    // limit angle
  float   m_damping;                  // limit angle
  std::vector<string> m_allowed;      // list of allowed types see file SocketAttachmentsTypes.xml
};

//-------------------------------------------------------------------------
struct SPartModel : public IPartModel
{
  //-------------------------------------------------------------------------
  // Model filename (.CHR)
  VIRTUAL const char* GetFilename() const { return m_filename.c_str(); }
  VIRTUAL void SetFilename(const char* filename) { m_filename = filename; }

	// Model FaceGenTemplate filename (.FGT)
	VIRTUAL const char* GetFGTFilename() const { return m_fgtfilename.c_str(); };
	VIRTUAL void SetFGTFilename(const char* filename) { m_fgtfilename=filename; };

  VIRTUAL const char* GetPocketsMaskFilename() const { return m_pmaskfilename.c_str(); }
  VIRTUAL void SetPocketsMaskFilename(const char* filename) { m_pmaskfilename = filename; }

  //-------------------------------------------------------------------------
  // Morphs
  VIRTUAL int GetNumMorphs() const { return m_morphs.size(); }

  VIRTUAL IPartMorph* GetMorphByIndex(int index)
  {
    if (index >= 0 && index < m_morphs.size())
      return &m_morphs[index];
    return NULL;
  }

  VIRTUAL const IPartMorph* GetMorphByIndex(int index) const
  {
    if (index >= 0 && index < m_morphs.size())
      return &m_morphs[index];
    return NULL;
  }

  VIRTUAL void ClearMorphs() { m_morphs.resize(0); }
  VIRTUAL void ReserveMorphs(int size) { m_morphs.reserve(size); }

  VIRTUAL IPartMorph* AddMorph(const char* switchName, const char* morphTarget)
  {
    SPartMorph newMorph;
    newMorph.SetSwitch(switchName);
    newMorph.SetMorphTarget(morphTarget);
    m_morphs.push_back(newMorph);

    return &m_morphs.back();
  }

  VIRTUAL void AddMorph(const IPartMorph* source)
  {
    SPartMorph newMorph;
    newMorph.Set(source);
    m_morphs.push_back(newMorph);
  }

  VIRTUAL void InsertMorphAtIndex(int index, const char* switchName, const char* morphTarget)
  {
    SPartMorph newMorph;
    newMorph.SetSwitch(switchName);
    newMorph.SetMorphTarget(morphTarget);
    m_morphs.insert(m_morphs.begin() + index, newMorph);
  }

  VIRTUAL void SetMorphAtIndex(int index, const char* switchName, const char* morphTarget)
  {
    if (index >= 0 && index < m_morphs.size())
    {
      if (switchName)
        m_morphs[index].SetSwitch(switchName);
      if (morphTarget)
        m_morphs[index].SetMorphTarget(morphTarget);
    }
  }

  VIRTUAL void RemoveMorphAtIndex(int index)
  {
    if (index >= 0 && index < m_morphs.size())
      m_morphs.erase(m_morphs.begin() + index);
  }

  VIRTUAL void SetMorphArrayPos(int oldPos, int newPos)
  {
    SPartMorph newEntry; newEntry.Set(&*(m_morphs.begin() + oldPos));
    m_morphs.erase(m_morphs.begin() + oldPos);
    m_morphs.insert(m_morphs.begin() + newPos, newEntry);

  }

  //-------------------------------------------------------------------------
  // Materials
  VIRTUAL int GetNumMaterials() const { return m_materials.size(); }

  VIRTUAL IPartMaterial* GetMaterialByIndex(int index)
  {
    if (index >= 0 && index < m_materials.size())
      return &m_materials[index];
    return NULL;
  }

  VIRTUAL const IPartMaterial* GetMaterialByIndex(int index) const
  {
    if (index >= 0 && index < m_materials.size())
      return &m_materials[index];
    return NULL;
  }

  VIRTUAL void ClearMaterials() { m_materials.resize(0); }
  VIRTUAL void ReserveMaterials(int size) { m_materials.reserve(size); }

  VIRTUAL IPartMaterial* AddMaterial(const char* name, const char* filename)
  {
    SPartMaterial newMaterial;
    newMaterial.SetName(name);
    newMaterial.SetFilename(filename);
    m_materials.push_back(newMaterial);

    return &m_materials.back();
  }

  VIRTUAL void AddMaterial(const IPartMaterial* source)
  {
    SPartMaterial newMaterial;
    newMaterial.Set(source);
    m_materials.push_back(newMaterial);
  }

  VIRTUAL void RemoveMaterialAtIndex(int index)
  {
    if (index >= 0 && index < m_materials.size())
      m_materials.erase(m_materials.begin() + index);
    else
      CryLog("Trying to erase material at wrong index %d.", index);
  }

  //-------------------------------------------------------------------------
  // Switches
  VIRTUAL int GetNumSwitches() const { return m_switches.size(); }

  VIRTUAL const char* GetSwitchByIndex(int index) const
  {
    if (index >= 0 && index < m_switches.size())
      return m_switches[index];
    return NULL;
  }

  VIRTUAL void ClearSwitches() { m_switches.resize(0); }
  VIRTUAL void ReserveSwitches(int size) { m_switches.reserve(size); }
  VIRTUAL void AddSwitch(const char* switchName) { m_switches.push_back(switchName); }

  VIRTUAL void InsertSwitchAtIndex(int index, const char* switchName)
  {
    m_switches.insert(m_switches.begin() + index, string(switchName));
  }

  VIRTUAL void SetSwitchAtIndex(int index, const char* switchName)
  {
    if (index >= 0 && index < m_switches.size())
      m_switches[index] = switchName;
  }

  VIRTUAL void RemoveSwitchAtIndex(int index)
  {
    if (index >= 0 && index < m_switches.size())
      m_switches.erase(m_switches.begin() + index);
  }

  //-------------------------------------------------------------------------
  // Sockets
  VIRTUAL int GetNumSockets() const { return m_sockets.size(); }

  VIRTUAL IPartSocket* GetSocketByIndex(int index)
  {
    if (index >= 0 && index < m_sockets.size())
      return &m_sockets[index];
    return NULL;
  }

  VIRTUAL const IPartSocket* GetSocketByIndex(int index) const
  {
    if (index >= 0 && index < m_sockets.size())
      return &m_sockets[index];
    return NULL;
  }

  VIRTUAL void ClearSockets() { m_sockets.resize(0); }
  VIRTUAL void ReserveSockets(int size) { m_sockets.reserve(size); }

  VIRTUAL IPartSocket* AddSocket(const char* name, int edge, float angle, float damping)
  {
    SPartSocket newSocket;
    newSocket.SetName(name);
    newSocket.SetEdge(edge);
    newSocket.SetAngle(angle);
    newSocket.SetDamping(damping);

    m_sockets.push_back(newSocket);
    return &m_sockets.back();
  }

  VIRTUAL void AddSocket(const IPartSocket* source)
  {
    SPartSocket newSocket;
    newSocket.Set(source);
    m_sockets.push_back(newSocket);
  }

protected:
  string m_filename;                      // filename of .CHR
	string m_fgtfilename;                   // filename of FaceGen template file .FGT
  string m_pmaskfilename;                 // pockets mask filename
  std::vector<SPartMorph> m_morphs;       // morphs list
  std::vector<SPartMaterial> m_materials; // materials list
  std::vector<string> m_switches;         // switches list activated by this part
  std::vector<SPartSocket> m_sockets;     // sockets list
};

//-------------------------------------------------------------------------
struct SCharacterPart : public ICharacterPart
{
  // Part name
  VIRTUAL const char* GetName() const { return m_name.c_str(); }
  VIRTUAL void SetName(const char* name) { m_name = name; }

  // Part slot
  VIRTUAL const char* GetSlot() const { return m_slot.c_str(); }
  VIRTUAL void SetSlot(const char* slot) { m_slot = slot; }

  // Part filename
  VIRTUAL const char* GetFilename() const { return m_filename.c_str(); }
  VIRTUAL void SetFilename(const char* filename) { m_filename = filename; }

	// Part template
  VIRTUAL const char* GetTemplateName() const { return m_template.c_str(); }
  VIRTUAL void SetTemplateName(const char* name) { m_template = name; }

  // Part visibility mask
  VIRTUAL uint32 GetVisibilityMask() const { return m_visibilityMask; }
  VIRTUAL void SetVisibilityMask(uint32 mask) { m_visibilityMask = mask; }

  // Part model data
  VIRTUAL IPartModel* GetModel(EGender gender) { return gender == GENDER_MALE ? &m_maleModel : &m_femaleModel; }
  VIRTUAL const IPartModel* GetModel(EGender gender) const { return gender == GENDER_MALE ? &m_maleModel : &m_femaleModel; }

protected:
  string m_name;            // part name
  string m_slot;            // part slot
	string m_filename;        // filename of chr file
  string m_template;        // filename of base template

  uint32 m_visibilityMask;  // visibility mask (see EVisibilityMask)

  SPartModel m_maleModel;
  SPartModel m_femaleModel;
};

//-------------------------------------------------------------------------
struct SCharacterPartTemplate : public ICharacterPartTemplate
{
  // Morphs
  VIRTUAL int GetNumMorphs(EGender gender) const { return (gender == GENDER_MALE) ? m_morphsMale.size() : m_morphsFemale.size(); }

  VIRTUAL IPartMorph* GetMorphByIndex(EGender gender, int index)
  {
    if (gender == GENDER_MALE)
    {
      if (index >= 0 && index < m_morphsMale.size())
        return &m_morphsMale[index];
    }
    else
    {
      if (index >= 0 && index < m_morphsFemale.size())
        return &m_morphsFemale[index];
    }

    return NULL;
  }

  VIRTUAL const IPartMorph* GetMorphByIndex(EGender gender, int index) const
  {
    if (gender == GENDER_MALE)
    {
      if (index >= 0 && index < m_morphsMale.size())
        return &m_morphsMale[index];
    }
    else
    {
      if (index >= 0 && index < m_morphsFemale.size())
        return &m_morphsFemale[index];
    }

    return NULL;
  }

  // Switches
  VIRTUAL int GetNumSwitches(EGender gender) const { return (gender == GENDER_MALE) ? m_switchesMale.size() : m_switchesFemale.size(); }

  VIRTUAL const char* GetSwitchByIndex(EGender gender, int index) const
  {
    if (gender == GENDER_MALE)
    {
      if (index >= 0 && index < m_switchesMale.size())
        return m_switchesMale[index];
    }
    else
    {
      if (index >= 0 && index < m_switchesFemale.size())
        return m_switchesFemale[index];
    }

    return NULL;
  }

  // Template filename
  VIRTUAL const char* GetFilename() const { return m_filename.c_str(); }

  // Template name
  VIRTUAL const char* GetName() const { return m_name.c_str(); }

  // Part visibility mask
  VIRTUAL uint32 GetVisibilityMask() const { return m_visibilityMask; }

  // Template reading
  bool ReadTemplate(const char* filename);

protected:
  std::vector<SPartMorph> m_morphsMale;
  std::vector<SPartMorph> m_morphsFemale;
  std::vector<string> m_switchesMale;
  std::vector<string> m_switchesFemale;
  string m_filename;
  string m_name;

  uint32 m_visibilityMask;   // visibility mask (see EVisibilityMask)
};

//-------------------------------------------------------------------------
struct Verlet 
{
  struct Particle
  {
    Particle(){};
    Particle(const Vec3 ip,const Vec3 ia,float i_invmass=0.0f)
    {
      initp = oldp = p = ip;
      a = ia;
      invmass = i_invmass;
    }
    Vec3    p;    // cur pos
    Vec3    oldp; // prev pos
    Vec3    a;    // forces
    Vec3    initp;    // init pos
    float   invmass;
  };
  // constraint
  struct Constraint 
  {
    Constraint(){}
    Constraint(int a,int b, const Vec3& damper)
    {
      particleA = a;
      particleB = b;
      springDamper = damper;
    }
    int particleA, particleB;
    float restlength;
    Vec3 springDamper; 
  };
  void Reset()
  {
    m_particles.resize(0);
    m_constraints.resize(0);
  }
  // integrate step
  void Intergrate();
  // forces
  void AccumulateForces()
  { 
    // add gravity
    for(int i=0; i<m_particles.size(); i++) 
    {
      m_particles[i].a = m_vGravity;
    }
  }
  // 
  void SatisfyConstraints(int numStepsForConstraints) 
  {
    for(int j=0; j<numStepsForConstraints; j++) 
    {
      for(int i=0; i<m_constraints.size(); i++) 
      {
        Constraint& c = m_constraints[i];
        Vec3& x1 = m_particles[c.particleA].p;
        Vec3& x2 = m_particles[c.particleB].p;
        Vec3 delta = (x2-x1);
        float deltalength = cry_sqrtf(delta*delta);
        float diff = (deltalength-c.restlength)/(deltalength*(m_particles[c.particleA].invmass+m_particles[c.particleB].invmass));
        x1 += (m_particles[c.particleA].invmass*delta*diff).CompMul(c.springDamper);
        x2 -= (m_particles[c.particleB].invmass*delta*diff).CompMul(c.springDamper);
      }
    }
  }
  //
  void TimeStep(float fTimeStep,int numStepsForConstraints) 
  {
    m_fTimeStep = fTimeStep;
    AccumulateForces();
    Intergrate();
    SatisfyConstraints(numStepsForConstraints);
  }
  //
  std::vector<Constraint> m_constraints;
  std::vector<Particle> m_particles;
  Vec3    m_vGravity;            // gravity
  float      m_fTimeStep;
  //
};
struct CharacterVerletSystem
{
  CharacterVerletSystem()
  {
    m_lastSimFrame = INT_MAX;
  }
  void Reset()
  {
    m_lastSimFrame = INT_MAX;
    m_verlet.Reset();
    m_bones_ids.resize(0);
    m_animated_pose.resize(0);
  }
  bool Initialize(ICharacterInstance* pInstance,const char* *bones,const Vec3 *offsets,const float* invmases,size_t pointsCount,const Verlet::Constraint* constraints,size_t numConstaints);
  bool BeginSimulate();
  void Simulate();
  void EndSimulate(bool applyResultToCharacter,float minDelta,float maxDelta);
  void DrawDebugLines();
  //
  ICharacterInstance* m_pInstance;
  Verlet m_verlet;
  int m_lastSimFrame;
  std::vector<int16> m_bones_ids;
  std::vector<Vec3> m_animated_pose;
  //
  size_t m_pointsCount;
  const char* *m_bones;
  const Vec3 *m_offsets;
  const float* m_invmases;
protected:
  bool CanSimulate();
};

//-------------------------------------------------------------------------
struct CCharacterPartAttachment : public ICharacterPartAttachment
{
  //-------------------------------------------------------------------------
  CCharacterPartAttachment(CCompoundCharacter* cc,uint32 visibilityMask,int surf_type_id) 
    : m_cc(cc), m_visibilityMask(visibilityMask)
  {
    m_surface_type_id = surf_type_id;
    m_pRootPhysicalEntity = NULL;
  }

  //-------------------------------------------------------------------------
  CCharacterPartAttachment(CCompoundCharacter* cc,const std::map<string,int>& partsSurfTypes)
    : m_cc(cc), m_partsSurfTypes(partsSurfTypes)
  {
    m_visibilityMask = 0;
    m_surface_type_id = -1;
    m_pRootPhysicalEntity = NULL;
    m_surface_type_id = -1;
  }

  //-------------------------------------------------------------------------
  // Visibility mask
  VIRTUAL uint32 GetVisibilityMask() const { return m_visibilityMask; }
	VIRTUAL void SetVisibilityMask(uint32 mask) { m_visibilityMask = mask; }

  //-------------------------------------------------------------------------
  // Realtime face attachments
  VIRTUAL void AddFaceAttachment(const char* partName,const char* name, QuatT attRelativeDefault, QuatT attAbsoluteDefault, uint32 faceNr);

  virtual bool PhysicalizeAttachment( IAttachmentManager* pManager, int idx, int nLod, IPhysicalEntity *pent, const Vec3 &offset );
  virtual bool UpdatePhysicalizedAttachment( IAttachmentManager* pManager, int idx, IPhysicalEntity *pent, const QuatT &offset);
  virtual void OnRemoveAttachment(IAttachmentManager* pManager, int idx);


public:
  IPhysicalEntity *m_pRootPhysicalEntity;// character physical entity
  //
  struct AddedPhysicsParts
  {
    int m_partId;
    int m_parentJointIndex;// in master skeleton
    int m_bonePhysParentIndex;
    QuatT m_bonePhysLocal;
    QuatT m_local;// local transform
  };
  std::vector<AddedPhysicsParts> m_addedPartsList;
  int m_surface_type_id;
  //
  //
  uint32 m_visibilityMask;
  std::map<string,int> m_partsSurfTypes;//[partName,surftype]
  CCompoundCharacter* m_cc;
  CharacterVerletSystem m_neckSim;
};
//-------------------------------------------------------------------------
class CCompoundCharacter : public ICompoundCharacter
{
  friend class CCharacterPartsManager;
	friend struct CCharacterPartAttachment;
public:
  struct SAttachedPart
  {
    SAttachedPart(){};
    SAttachedPart(const char* name,const char* material_id,const char* materialName,const char* surface,const char* slot);
    string m_name;
    string m_material_id;
    string m_material_name;
    string m_surface;
    string m_slot;
    struct SSocketAttachment
    {
      string m_name;
      string m_filename;
      Vec3 m_pos;
      QuatT m_qt;
      bool m_bUseCenterPoint;
    };
    struct SFaceAttachmentData
    {
      string  m_name;         // attachment name
      QuatT   m_AttRelativeDefault;		// 	this attachment matrix is relative to the facematrix in default pose;
      QuatT   m_AttAbsoluteDefault;		// 	this attachment matrix is absolute
      uint32  m_FaceNr;				// index of face in this attachment
    };
    std::vector<SSocketAttachment> m_sockets;
    std::vector<SSocketAttachment> m_faceAttachments;
    std::vector<SFaceAttachmentData> m_faceAttachmentsData;
    void AddFaceAttachmentDesc(const char* partName,const char* name, QuatT attRelativeDefault, QuatT attAbsoluteDefault, uint32 faceNr);
  };
public:
  CCompoundCharacter(ICharacterInstance* inst,const char* name,ICompoundCharacter::EFlags flags)
    : m_pInstance(inst), m_name(name)
  {
    m_gender = GENDER_MALE;
    m_fFatness = 0.0f;
    m_fNeckFatness = 0.0f;
    m_nRefCounter = 0;
    m_bCached = false;
    m_flags = flags;
    m_lastSimTime = FLT_MAX;
  }
  //
  VIRTUAL ICharacterInstance* GetCharacterInstance() { return m_pInstance; }
	virtual void SetCharacterInstance(ICharacterInstance* instance) {m_pInstance = instance;}
  //
  const char* GetName() const { return m_name.c_str(); }
  // gender
  VIRTUAL void SetGender(const EGender& gender);
  VIRTUAL EGender GetGender() const { return m_gender; };
	// scale
	VIRTUAL void SetScale(const float& fScale);
	VIRTUAL float GetScale() const;
  // fatness
  VIRTUAL void SetFatness(const float& fFatness);
  VIRTUAL float GetFatness() const { return m_fFatness; }
  // neck fatness
  VIRTUAL void SetNeckFatness(const float& fFatness);
  VIRTUAL float GetNeckFatness() const { return m_fFatness; }
  // Parts attachment and detachment
  VIRTUAL bool AttachPart(const char* partName, const char* material,const char* surface_type=NULL);
  virtual bool AttachPartInternal(const char* partName, const char* material,const char* surface_type, bool bUseCache,bool bForceAttach=false,bool bUpdateMorphs=true);
  // sockets
  VIRTUAL int GetNumSocketAttachments(const char* partName) const;
  VIRTUAL const char* GetSocketName(const char* partName,int socketIndex) const;
  VIRTUAL const char* GetSocketFilename(const char* partName,int socketIndex) const;
  VIRTUAL bool AddSocketAttachment(const char* partName,const char* socketName,const char* BindingPath);
  VIRTUAL bool HasFaceAttachment(const char* partName,const char* pocketName);
  VIRTUAL bool AddFaceAttachment(const char* partName,const char* pocketName,const char* BindingPath,const QuatT& pt,bool bUseCenterPointDist);
  VIRTUAL bool MoveFaceAttachment(const char* partName,const char* pocketName,const QuatT& pt,bool bUseCenterPointDist);
  // 
  VIRTUAL void DetachPart(const char* name);
  VIRTUAL void DetachAllParts();
  virtual void DetachPartInternal(const char* name,bool bDeletePhysics,bool bIgnoreCommitOnCache = false,bool bUpdateMorphs = true);
  //virtual void DetachPart(int index, EGender gender,IAttachmentManager* pManager) = 0;

  VIRTUAL void ReAttachPart(const char* name);
  virtual void ReloadPart(const char* name,bool bUseCache,bool bIgnoreCommitOnCache = false,bool bUpdateMorphs=false);
  virtual bool SetMergedPhysics(const char* name);
  virtual bool SetMergedVisibilityFlag(const char* filename,uint32 VisFlagsAnd,uint32 VisFlag,IMaterial* pMtl);

  VIRTUAL void ChangeAttachedPartMaterial(const char* partName, const char* material);
  VIRTUAL const char* GetAssignedMaterial(const char* partName);
  //virtual void ChangeAttachedPartMaterial(int index, const char* mtlSlot) = 0;
  VIRTUAL void SetSurfaceType(const char* partName, const char* surface_type);
  VIRTUAL const char* GetSurfaceType(const char* partName);

  VIRTUAL void ApplyVisibilityMask(int mask);

  // Attached parts info
  VIRTUAL int GetNumAttached() const;
//  virtual const ICharacterPartAttachment* GetAttachedPartInfo(const char* name) const;
//  virtual const ICharacterPartAttachment* GetAttachedPartInfo(int index) const;
  VIRTUAL const char* GetAttachedPartName(int index) const;
  //
  bool IsCommitOnCache();
  bool WarnCommitOnCache(const char* msg);
  bool WarnCached(const char* msg);
  bool isServicePart(const char* partName) const;
  //
  void AddRef() {	++m_nRefCounter; }
  virtual void Release() { delete this; }
  //
  VIRTUAL void Cache(bool bScheduleOnly,ECacheFlags flags=eCacheFlag_Default);
	VIRTUAL void CacheCHRs(bool bCache);
	VIRTUAL void SetCustomHeadName(const char* newCustomHeadName)
	{
		m_customHeadName=newCustomHeadName;
	}
	VIRTUAL const string& GetCustomHeadName() const
	{
		return m_customHeadName;
	}
  //
  void UpdateMorphs();
  //
  struct MorphsData : IMergeCHRMorphsPipe
  {
    virtual size_t getNumMorphs() const
    {
      return m_resmorphs.size();
    }
    virtual const char* getMorphName(size_t index) const
    {
      return m_resmorphs[index].first;
    }
    virtual float getMorphWeight(size_t index) const
    {
      return m_resmorphs[index].second;
    }
    //
    string m_partName;
    std::vector<std::pair<string,float> > m_resmorphs;
  };
  //
  void GetMorphsDataToBake(const char* partName,MorphsData& result);
  void UpdatePartModel(const char* partName);
  //
  SAttachedPart* GetAttachedPart(const char* name);
  const SAttachedPart* GetAttachedPart(const char* name) const;
	string GetPartCHRFileName(const char* partName, IMaterial **material=0) const;
  string getFaceFile() const;
  ICharacterPartAttachment* GetAttachedPartInfoInternal(const char* name) const;
  bool isCached() const
  {
    return m_bCached;
  }
  //
  void Simulate();
protected:
  ~CCompoundCharacter();
  void ApplyGenderToAttachedParts();
  //
  void CollectActiveSwitches(std::vector<string>& activeSwitches);
  static int GetSurfaceTypeInternal(const char* surface_type);
  virtual int GetNumAttachedInternal() const;
  ICharacterPartAttachment* GetAttachedPartInfoInternal(int index) const;
  void AttachAllCacheOnCommit();
  //
  void ExtractMaterialName(string& filename);
  //
  bool isMyAttachment(IAttachment* pAtt) const;
  IAttachment* GetInterfaceByName( const char* szName ) const;
  int32 GetIndexByName( const char* szName ) const;
  void RemoveAttachmentByName( const char* szName );
  //
  _smart_ptr<ICharacterInstance> m_pInstance;
  EGender m_gender;
  float   m_fFatness;
  float   m_fNeckFatness;
  int m_nRefCounter;	// Reference count.
  bool m_bCached;
  string m_name;
  //
  std::vector<SAttachedPart> m_attachedParts;
  ICompoundCharacter::EFlags m_flags;
  // this sim done in spine bone space
  float m_lastSimTime;
  void InitSimulation();
  std::vector<int16> m_bones_ids;
  CharacterVerletSystem m_breastSim;
	std::vector< _smart_ptr<ICharacterInstance> > m_cached_chrs;
	string m_customHeadName;
};
//-------------------------------------------------------------------------
class CCharacterPartsManager : public ICharacterPartsManager, IGameFrameworkListener
{
public:
  //-------------------------------------------------------------------------
  // Constructor
  CCharacterPartsManager(CCryAction* pAction);

  //-------------------------------------------------------------------------
  void ScanForParts(const char* folderName);
  void AddNewPart(const char* filename);

  //-------------------------------------------------------------------------
  // Global parts list
  const ICharacterPart* GetPartByName(const char* name) const;
  const ICharacterPart* GetPartByIndex(int index) const;
  int GetNumParts() const;
  VIRTUAL void ReloadPart(const char* name);
  VIRTUAL void RedefinePart(const char* name, const ICharacterPart* pNewPartData);
  int GetPartIndexByName(const char* name) const;

  //-------------------------------------------------------------------------
  int GetNumTemplates() const;
  const ICharacterPartTemplate* GetTemplateByIndex(int index) const;
  const ICharacterPartTemplate* GetTemplateByName(const char* name) const;

  //-------------------------------------------------------------------------
  VIRTUAL ICompoundCharacter* LoadCPF(const char* cpfFilename, ICharacterInstance* pInstance,bool bUseCache=false) const;
  VIRTUAL ICompoundCharacter* CreateCompoundCharacter(ICharacterInstance* pInstance,const char* dbgname,ICompoundCharacter::EFlags flags = ICompoundCharacter::eFlag_Default) const;
  //
  void RegisterCVars();
  void UnRegisterCVars();
  //
  void PrintDebugInfo() const;
  //
  int   m_cc_mergerDisableAtlas;
  int   m_cc_mergerUseDiffuseAtlas;
  //
  float m_cc_groinMaxWeightAngle;
  float m_cc_groinMinWeightAngle;
  int   m_cc_groinSimulation;
  int   m_cc_profile;
  float m_cc_breast_gravity;
  float m_cc_breast_minDist;
  float m_cc_breast_maxDist;
  float m_cc_breast_upperDamper;
  float m_cc_breast_lowerDamper;
  float m_cc_breast_upperRestRatio;
  float m_cc_breast_lowerRestRatio;
  int m_cc_breast_simulation;
  //
  int   m_cc_neckSimulation;
  float m_cc_neckAngleMultilier;
  float m_cc_neck_upperDamper;
  float m_cc_neck_lowerDamper;
  float m_cc_neck_minDist;
  float m_cc_neck_maxDist;
  float m_cc_neckAngleBias;
  float m_cc_neckAngleMultilierLR;
  std::vector<CCompoundCharacter*> m_registered;

  float cc_scale_head_factor;

  string m_fg_default_preset;

public:
  void OnPostUpdate(float fDeltaTime);
  void OnSaveGame(ISaveGame* pSaveGame){};
  void OnLoadGame(ILoadGame* pLoadGame){};
  void OnLevelEnd(const char* nextLevel){};
  void OnActionEvent(const SActionEvent& event){};
protected:
  void ScanForParts(const char* folderName, bool scanForTemplates);
  void LoadPart(const char* filename);
  // Properties
  std::vector<SCharacterPartTemplate> m_templates;
  std::vector<SCharacterPart> m_parts;
};

//-------------------------------------------------------------------------
class CCompoundCharacterCache
{
public:
  //-------------------------------------------------------------------------
  // Constructor
  CCompoundCharacterCache();
  ~CCompoundCharacterCache();
  //-------------------------------------------------------------------------
  void RegisterCompoundCharacter(CCompoundCharacter* pCharacter);
  void UnRegisterCompoundCharacter(CCompoundCharacter* pCharacter);
  //
  void UnScheduleCharacterJobs(CCompoundCharacter* pCharacter);
  //
  void Schedule(CCompoundCharacter* pCharacter,ICompoundCharacter::ECacheFlags flags);
  //! returns num pending operations
  int Step();
  int StepInternal(ICompoundCharacter* pCharacter);
  bool IsFullyCached(ICompoundCharacter* pCharacter);
  //
  void Cleanup(bool bFullCleanup);
  //
  void CacheAll();
  //
  void RegisterCVars();
  void UnRegisterCVars();
  //
  string GetCacheFolder();
  string GetCHRFileName(CCompoundCharacter* pCharacter,const char* partName) const;
  string GetVisibilityFlagCHRFileName(CCompoundCharacter* pCharacter,uint32 visFlag) const;
  string GetPhysicsCHRFileName(CCompoundCharacter* pCharacter) const;
  bool IsMergeValid(const char* mergeBase,bool bGenerateLod);
  //
  int m_verbose;
  int m_cache;
protected:
  //
  struct Job
  {
    enum EJobType
    {
      jtBakePhysics,
      jtBakeMorphs,
      jtBakeByVisibilityFlags,
    };

    Job()
    {
      m_mergePipeFlags = IMergePipe::mfDefault;
      m_mergeMaterialFlags = IMaterialMergePipe::mfDefaut;
      m_bGenerateLod = false;
    }

    EJobType m_jobType;
    CCompoundCharacter* m_pCharacter;
    string m_partName;
    uint32 m_mergeVisFlags;
    uint32 m_mergeVisFlagsAnd;// to exclude something
    IMergePipe::EMergeFlags m_mergePipeFlags;
    IMaterialMergePipe::EMergeMaterialsFlags m_mergeMaterialFlags;
    bool m_bGenerateLod; 
  };
  std::vector<Job> m_pendingJobs;
  int m_maxCacheSize;
  int cc_lodGenNumLods;
  float cc_lod1GenMaxAngle;
  int   cc_lod1GenTexSplits;
  int   cc_lod1GenSkinning;
  float cc_lod2GenMaxAngle;
  int   cc_lod2GenTexSplits;
  int   cc_lod2GenSkinning;
  float cc_lod3GenMaxAngle;
  int   cc_lod3GenTexSplits;
  int   cc_lod3GenSkinning;
  float cc_lod4GenMaxAngle;
  int   cc_lod4GenTexSplits;
  int   cc_lod4GenSkinning;
  float cc_lod5GenMaxAngle;
  int   cc_lod5GenTexSplits;
  int   cc_lod5GenSkinning;
  ILevelSystemListener* m_levelListener;
  bool IsFullyCachedJob(ICompoundCharacter* pCharacter,Job& job,string* chrfilename);
};

#endif // __CHARACTERPARTSMANAGER_H__
