#pragma once

#include "Cry_Vector3.h"
#include "Cry_Quat.h"
#include "Cry_Matrix.h"

#ifdef FG_STANDALONE

struct SBoneMod
{
	Vec3 trans;
	float scale;
	int bone;
	int knot;
	int vec;
	float w;

	struct SLink
	{
		float wts[3]; // max of 3 key points to link
		int   idx[3];
		SLink()
		{
			Reset();
		}
		void Reset()
		{
			wts[0]=wts[1]=wts[2]=0.0f;
			idx[0]=idx[1]=idx[2]=-1;
		}
	};

	SLink links[3]; // 3 coordinates
	SBoneMod():w(1.0f), scale(1.0f){}
	//		float softSelWeight;
};

class IFaceGen{
public:
	virtual void FindNearestUVVertexShift(const Vec3& pos, const Vec2& uv, Vec3& shift) const = 0;
	virtual void CalcVertexNormals(void) = 0;
	virtual void CalcVertexPositions(void) = 0;
	virtual void ScaleAttachments() = 0;
	virtual void Drop(void) = 0;
	virtual void CalcMinZ(void) = 0;
	virtual void AutoFaceBack(void) = 0;
	virtual int  Load(void const *,unsigned int) = 0;
	virtual void Destroy()=0;
	virtual SBoneMod* GetBoneMod(size_t index) = 0;
	virtual void SetBoneModTrans(size_t index, const Vec3& trans) = 0;
	virtual bool GetBoneModTrans(const char* name, size_t knot, size_t vec, Vec3& trans, float& scale)const = 0;
	virtual void FetchTransform(const IFaceGen* gf) = 0;
};

#else

#include <../CryCommon/IFaceGen.h>

#endif

struct SBoneInfo
{
	struct SBezierKnot{
		Vec3 pts[3]; // 0 -- pos, 1 -- in, 2 -- out
//		Vec3 in;
//		Vec3 out;
		unsigned int type; // 
		// KTYPE_AUTO=0,
		// KTYPE_CORNER=1,
		// KTYPE_BEZIER=2,
		// KTYPE_BEZIER_CORNER=KTYPE_CORNER|KTYPE_BEZIER,

		int N[3];
		//		int knotN;
		//		int knotInN;
		//		int knotOutN;
	};

	int isCurve;  // true (not 0) if bezier, false if matrix (int used instead of bool to keep 4 byte aliasing)
	float matrix[4][3]; // Max's Matrix3 is not used because of flags member
	// Matrix3 matrix; // Bone TM (used if bone is non-curve)
	int nSegments; // Number of segments in curve. Each one owns 3 points and last owns 3 if curve is closed or 4 otherwise
	int isClosed; // true (not 0) if curve is closed, false otherwise
	int nameStart;
	int nameLength;
	int nKnots; // Number in points in curve (mSegments*3 or mSegments*3+1 as explained above)
	std::vector<SBezierKnot>knots;
	string name;
};

struct SBind
{
	// I decided to store no normal, binding position and binding tangent. 
	// They are calculated.
	int nb;   // number of bone
	int ns;   // number of segment
	float d;  // param value inside segment
	float w;  // weight
};


struct SVertexPosAndBind{
	Vec3 p;   // position
	int boneN;  // Number of effective bones
	SBind s[8];
	SVertexPosAndBind(){memset(this,0,sizeof(*this));}
};


struct SVertexUV{
	Vec3 u;   // texture coordinates
};

struct SVertexVC{
	Vec3 c;   // vertex color
};

struct SVertex{
	unsigned int PBIndex; // index of Position and Binding item
	unsigned int UVIndex; // index of UV mapping item
	unsigned int VCIndex; // index of Vertex Color item
};

struct SWeightedNormal
{
	Vec3 n;
	float w;
};

struct SPos2
{
	bool actual;
	float x;
	float y;
};

struct SFaceFeatureKeyPoint
{
	int absolute; // if point is absolute to world coordinates, nt tied to some bone
	Vec3 pos;   // pos is valid if absolute!=0
	int boneN;
	int knotN;
	int view;
	string id;
	string feature;
	string text;
	SPos2 frontRef;
	SPos2 sideRef;
};

struct SFaceFeature
{
	std::vector<int> keyPoints; // keypoint indices
};

struct SBackLink
{
	int bone;
	int knot;
	int vec;
	std::vector<int>keyPointLinks;
	SBackLink():vec(0){};
};

class CFaceGen : public IFaceGen
{
// Persistent data
	std::vector<SVertexPosAndBind> VPBBuffer;
	std::vector<SVertexUV>         VUVBuffer;
	std::vector<SVertexVC>         VVCBuffer;
	std::vector<SVertex>           VBuffer;
	std::vector<unsigned int>      IBuffer;
	std::vector<SBoneInfo>         bones;

	std::vector<Vec3> VPositions;
	std::vector<Vec3> VNormals;

	std::vector<std::vector<Vec3> > undoBuffer;
	int undos;

	std::vector<SFaceFeatureKeyPoint> keyPoints;
	std::map<string,int>keyPointMap;
	std::vector<SBackLink> backLinks;

	int m_Mode;

	std::map<string, int> m_boneNamesMap;

public:

	int GetBoneNByName(const char* name) const
	{
		std::map<string, int>::const_iterator found=m_boneNamesMap.find(name);
		if(found==m_boneNamesMap.end())
			return -1;
		else
			return found->second;
	}

private:

	std::vector<SBoneMod>boneMods;

public:
	float minz;
	void SetMode(int newMode){m_Mode=newMode;};
	int GetMode()const{return m_Mode;};
	void NextMode(){m_Mode=(m_Mode+1)%4;};
	void PrevMode(){m_Mode=(m_Mode-1);if(m_Mode<0)m_Mode=3;};

	std::vector<SVertexPosAndBind>& GetVW(){return VPBBuffer;};

	int GetBoneCount()const {return (int) bones.size();};
	const SBoneInfo* GetBonePtr(int index)const 
	{
		return &bones[index];
	};
	std::vector<SBoneInfo>& GetBones(){return bones;};
	std::vector<SBoneMod>& GetBoneMods(){return boneMods;};
	size_t GetBoneModsCount(){return boneMods.size();};
	void CopyBoneMods(const std::vector<SBoneMod>&src)
	{
		boneMods=src;
	}
	int GetBoneModIndex(size_t bone,size_t knot,size_t vec) const
	{
		if(bone>=bones.size())
			return -1;
		if(knot>=bones[bone].knots.size())
			return -1;
		knot%=bones[bone].knots.size();
		if(vec>2)
			return -1;
		return bones[bone].knots[knot].N[vec];
	};
	const std::vector<unsigned int>& GetIndices()const{return IBuffer;};
	const std::vector<SVertex>& GetVertices()const{return VBuffer;};
	
	const std::vector<Vec3>& GetVNormals()const{return VNormals;};
	const std::vector<Vec3>& GetVPositions()const{return VPositions;};
	const std::vector<SVertexUV>& GetUVs()const{return VUVBuffer;};
	std::vector<SFaceFeatureKeyPoint>& GetKeyPoints(){return keyPoints;};
	std::vector<SBackLink>& GetBackLinks(){return backLinks;};
	int FindKeyPointByName(const string& name);
	int AddKeyPoint(const string& name, int index);

	void CalcCurveWeights();

	const Vec3& GetBP0(size_t b,size_t p, size_t i)
	{
		if(GetBones()[b].isCurve)
		{
			p=p%(int)GetBones()[b].knots.size();
			return GetBones()[b].knots[p].pts[i];
		}
		else
		{
			return GetBones()[b].knots[0].pts[0];
		}
	}

	const Vec3& GetBP0i(size_t i)
	{
		return GetBP0(boneMods[i].bone,boneMods[i].knot,boneMods[i].vec);
	}

	Vec3 CalcBP(int b,int p, int i)
	{
		if(GetBones()[b].isCurve)
		{
			p=p%(int)GetBones()[b].knots.size();
			return GetBones()[b].knots[p].pts[i]+boneMods[GetBones()[b].knots[p].N[i]].trans;
		}
		else
		{
			return GetBones()[b].knots[0].pts[0]+boneMods[GetBones()[b].knots[0].N[0]].trans;
		}
	}

	void SetBP(int b,int p, int i, const Vec3& pos);

	void ProcessBackLinks();

	Vec3 CalcBPi(int i)
	{
		return CalcBP(boneMods[i].bone,boneMods[i].knot,boneMods[i].vec);
	}

	size_t GetStateBufferSize()
	{
		return boneMods.size()*sizeof(boneMods[0].trans);
	}

	void SaveState(void* statebuffer);
	void LoadState(void* statebuffer);

	void BackUp();
	void Reset();
	void Undo();
	void Redo();

	void ClearKPMap()
	{
		keyPointMap.clear();
	}
	void ClearBackLinks()
	{
		for(size_t i=0;i<boneMods.size();i++)
		{
			SBoneMod& bm=boneMods[i];
			for(size_t j=0;j<3;j++)
				bm.links[j].Reset();
		}
	}

	unsigned int FindNearestVertex(const Vec3& pos) const;
	void FindNearestVertexShift(const Vec3& pos, Vec3& shift) const;

	unsigned int FindNearestUVVertex(const Vec2& uv) const;
	virtual void FindNearestUVVertexShift(const Vec3& pos, const Vec2& uv, Vec3& shift) const;

	virtual int Load(const void* mem, size_t length);
	virtual void CalcVertexNormals();
	virtual void CalcVertexPositions();
	virtual void ScaleAttachments();
	virtual void CalcMinZ();
	virtual void Drop();
	virtual void AutoFaceBack();
	virtual void FaceBack();
	virtual void Destroy();
	virtual SBoneMod* GetBoneMod(size_t index);
	virtual void SetBoneModTrans(size_t index, const Vec3& trans);
	virtual bool GetBoneModTrans(const char* name, size_t knot, size_t vec, Vec3& trans, float& scale)const;
	virtual void FetchTransform(const IFaceGen* gf);

	CFaceGen(void);
	~CFaceGen(void);
};
