/*************************************************************************
	Crytek Source File.
	Copyright (C), Crytek Studios, 2001-2004.
	-------------------------------------------------------------------------
	$Id: RealtimeRemoteUpdate.cpp,v 1.1 2009/01/03 10:45:15 Paulo Zaffari Exp wwwrun $
	$DateTime$
	Description:	This is the source file for the module Realtime remote 
					update. The purpose of this module is to allow data update to happen 
					remotely so that you can, for example, edit the terrain and see the changes
					in the console.
-------------------------------------------------------------------------
History:
- 03:01:2009   10:45: Created by Paulo Zaffari
- 23:09:2009   10:39: Merged c2 version to main and moved to the engine by Johnmichael Quinlan
 *************************************************************************/


#include "StdAfx.h"
#include "RealtimeRemoteUpdate.h"
#include "ISystem.h"
#include "I3DEngine.h"
#include <IEntitySystem.h>
#include "IGame.h"
#include "IViewSystem.h"
#include "IEntitySystem.h"
#include "IGameFramework.h"
#include "IGameRulesSystem.h"

#ifdef XENON
#include "Xtl.h"
#endif //XENON

//////////////////////////////////////////////////////////////////////////
CRealtimeRemoteUpdateListener&	CRealtimeRemoteUpdateListener::GetRealtimeRemoteUpdateListener()
{
	static CRealtimeRemoteUpdateListener oRealtimeUpdateListener;
	return oRealtimeUpdateListener;
}
//////////////////////////////////////////////////////////////////////////
bool CRealtimeRemoteUpdateListener::Enable(bool boEnable)
{
	if (!gEnv)
	{
		return false;
	}

	if (!gEnv->pSystem)
	{
		return false;
	}

	INotificationNetwork*	piNotificationNetwork=gEnv->pSystem->GetINotificationNetwork();
	if (!piNotificationNetwork)
	{
		return false;
	}

	if (boEnable)
	{
		m_boIsEnabled=piNotificationNetwork->ListenerBind("RealtimeUpdate",this);
	}
	else
	{
		piNotificationNetwork->ListenerRemove(this);
		m_boIsEnabled=false;
	}

	return m_boIsEnabled;
}
//////////////////////////////////////////////////////////////////////////
bool	CRealtimeRemoteUpdateListener::IsEnabled()
{
	if (!gEnv)
	{
		return false;
	}

	if (!gEnv->pSystem)
	{
		return false;
	}

	INotificationNetwork*	piNotificationNetwork=gEnv->pSystem->GetINotificationNetwork();
	if (!piNotificationNetwork)
	{
		return false;
	}

	// We should instead query the notification network here.
	return m_boIsEnabled;	
}

//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::AddGameHandler(IRealtimeUpdateGameHandler * handler)
{
	GameHandlerList::iterator item = m_gameHandlers.begin();
	GameHandlerList::iterator end = m_gameHandlers.end();

	for ( ; item != end; ++item )
	{
		if ( handler == (*item) )
		{
			return; //already present	
		}
	}

	m_gameHandlers.push_back(handler);
}
//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::RemoveGameHandler(IRealtimeUpdateGameHandler * handler)
{
	GameHandlerList::iterator item = m_gameHandlers.begin();
	GameHandlerList::iterator end = m_gameHandlers.end();
	for ( ; item != end; ++item )
	{
		if ( handler == (*item) )
		{
			break;
		}
	}

	m_gameHandlers.erase(item);
}


//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::OnNotificationNetworkReceive(const void *pBuffer, size_t length)
{
	char * szBuffer((char*)pBuffer);
	size_t nStringLenght(0);
	unsigned char *	chBinaryBuffer(NULL);
	XmlNodeRef oXmlNode = gEnv->pSystem->LoadXmlFromString(szBuffer);
	size_t nBinaryDataSize(0);

	// Currently, if we have no XML node this is not a well formed message and
	// thus we stop processing.
	if (!oXmlNode)
	{
		return;
	}

	nStringLenght=strlen(szBuffer)+1;
	chBinaryBuffer=(unsigned char*)(szBuffer+nStringLenght);

	if (strcmp(oXmlNode->getTag(),"SyncMessage")!=0)
	{
		return;
	}

	string oSyncType = oXmlNode->getAttr("Type");
	if (oSyncType.empty())
	{
		return;
	}

	if (!oXmlNode->getAttr("BinaryDataSize",nBinaryDataSize))
	{
		return;
	}

#ifdef XENON
	// We are, this way, reseting the timer for the screensaver.
	XEnableScreenSaver(FALSE);
	XEnableScreenSaver(TRUE);
#endif //XENON

	bool requiresFurtherProcessing = false;

	for ( GameHandlerList::iterator item = m_gameHandlers.begin(), end = m_gameHandlers.end(); item != end; ++item )
	{
		if ( (*item)->UpdateGameData(oXmlNode,chBinaryBuffer) )
		{
			requiresFurtherProcessing = true;
		}
	}

	if ( !requiresFurtherProcessing )
		return;

	static std::vector<struct IStatObj*> * pStatObjTable = NULL;
	static std::vector<IMaterial*> * pMatTable = NULL;

	if (oSyncType.compare("EngineTerrainData")==0)
	{
		gEnv->p3DEngine->LockCGFResources();

		if (nBinaryDataSize>0)
		{
			if(ITerrain * piTerrain = gEnv->p3DEngine->GetITerrain())
			{
				size_t nUncompressedBinarySize(nBinaryDataSize);
				unsigned char * szData = new unsigned char[nBinaryDataSize];

				gEnv->pSystem->DecompressDataBlock(chBinaryBuffer,length-nStringLenght,szData,nUncompressedBinarySize);

				SHotUpdateInfo * pExportInfo = (SHotUpdateInfo *)szData;

				// As messages of oSyncType "EngineTerrainData" always come before 
				// "EngineIndoorData" and are always paired togheter, and have 
				// inter-dependencies amongst themselves, the locking is done here
				// and the unlocking is done when we receive a "EngineIndoorData".
				// Currently if we, for any reason, don't receive the second message,
				// we should expect horrible things to happen.
				gEnv->p3DEngine->LockCGFResources();

				pStatObjTable = NULL;
				pMatTable = NULL;

				piTerrain->SetCompiledData((uint8*)szData+sizeof(SHotUpdateInfo),nBinaryDataSize-sizeof(SHotUpdateInfo),&pStatObjTable,&pMatTable,true,pExportInfo);
				SAFE_DELETE_ARRAY(szData);
			}
		}
	}
	else if (oSyncType.compare("EngineIndoorData")==0)
	{
		if (nBinaryDataSize>0)
		{
			if(IVisAreaManager * piIVisAreaManager = gEnv->p3DEngine->GetIVisAreaManager())
			{
				size_t nUncompressedBinarySize(nBinaryDataSize);
				unsigned char * szData = new unsigned char[nBinaryDataSize];

				gEnv->pSystem->DecompressDataBlock(chBinaryBuffer,length-nStringLenght,szData,nUncompressedBinarySize);

				SHotUpdateInfo * pExportInfo = (SHotUpdateInfo *)szData;

				if(piIVisAreaManager)
					piIVisAreaManager->SetCompiledData((uint8*)szData+sizeof(SHotUpdateInfo),nBinaryDataSize-sizeof(SHotUpdateInfo),&pStatObjTable,&pMatTable,true,pExportInfo);

				SAFE_DELETE_ARRAY(szData);
			}
		}

		gEnv->p3DEngine->UnlockCGFResources();

		pStatObjTable = NULL;
		pMatTable = NULL;
	}
	else if (oSyncType.compare("Voxel")==0)
	{
		if (nBinaryDataSize>0)
		{
			size_t nUncompressedBinarySize(nBinaryDataSize);
			unsigned char* szData=new unsigned char[nBinaryDataSize];

			gEnv->pSystem->DecompressDataBlock(chBinaryBuffer,length-nStringLenght,szData,nUncompressedBinarySize);

			SVoxTerrainInfo	* pHeader = (SVoxTerrainInfo*)szData;
			IVoxTerrain * pVoxTerrain = gEnv->p3DEngine->CreateVoxTerrain(*pHeader);

			pVoxTerrain->SetCompiledData((uint8*)szData+sizeof(SVoxTerrainInfo),nBinaryDataSize-sizeof(SVoxTerrainInfo),true,true,&pHeader->aabbTerrain);

			SAFE_DELETE_ARRAY(szData);
		}
	}
	else if (oSyncType.compare("Vegetation")==0)
	{
		XmlNodeRef oCurrentNode=oXmlNode->findChild("Vegetation");
	}
	else if (oSyncType.compare("DetailLayers")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("SurfaceTypes");
		if (oChildRootNode)
		{
			gEnv->p3DEngine->LoadTerrainSurfacesFromXML(oChildRootNode,true);
		}
	}
	else if (oSyncType.compare("Environment")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("Environment");
		if (oChildRootNode)
		{
			gEnv->p3DEngine->LoadEnvironmentSettingsFromXML( oChildRootNode );	
		}
	}
	
	else if (oSyncType.compare("TimeOfDay")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("TimeOfDay");
		if (oChildRootNode)
			LoadTimeOfDay( oChildRootNode );
	}
	else if (oSyncType.compare("Materials")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("Materials");
		if (oChildRootNode)
			LoadMaterials( oChildRootNode );
	}
	else if (oSyncType.compare("EntityArchetype")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("EntityPrototypes");
		if (oChildRootNode)
			LoadArchetypes( oChildRootNode );
	}
	else if (oSyncType.compare("ConsoleVariables")==0)
	{
		LoadConsoleVariables(oXmlNode);
	}
	else if (oSyncType.compare("Particles")==0)
	{		
		LoadParticles(oXmlNode);
	}
	else if (oSyncType.compare("LayerTexture")==0)
	{
		if (nBinaryDataSize>0)
		{
			size_t nUncompressedBinarySize(nBinaryDataSize);
			unsigned char*	szData=new unsigned char[nBinaryDataSize];

			if (!szData)
			{
				return;
			}

			if (!gEnv->pSystem->DecompressDataBlock(chBinaryBuffer,length-nStringLenght,szData,nUncompressedBinarySize))
			{
				SAFE_DELETE_ARRAY(szData);
				return;
			}

			LoadTerrainLayer(oXmlNode,szData);

			SAFE_DELETE_ARRAY(szData);
		}
	}
	else if (oSyncType.compare("Particle.Library")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("ParticleLibrary");
		if (oChildRootNode)
		{
			const char* szEffectName(NULL);

			oXmlNode->removeChild(oChildRootNode);

			if (!oChildRootNode->getAttr("Effect",&szEffectName))
			{
				return;
			}

			XmlNodeRef oEffectNode=oChildRootNode->findChild("Effect");
			if (!oEffectNode)
			{
				return;
			}

			gEnv->pParticleManager->LoadEffect(szEffectName,oEffectNode,true);
		}
	}
	else if (oSyncType.compare("ChangeLevel")==0)
	{
		if (oXmlNode->haveAttr("LevelName"))
		{
			const char*			szLevelName(NULL);
			string		strMapCommand("map ");

			oXmlNode->getAttr("LevelName",&szLevelName);
			strMapCommand+=szLevelName;

			gEnv->pConsole->ExecuteString(strMapCommand);
		}
	}
	else if (oSyncType.compare("Entities")==0)
	{
		XmlNodeRef root=oXmlNode->findChild("Entities");

		if ( !root )
			return;

		LoadEntities(root);
	}
	else if (oSyncType.compare("KeepAlive")==0)
	{
		// Here we reset the time counter for the keep alive message.
		m_lastKeepAliveMessageTime=gEnv->pTimer->GetAsyncTime();
	}
}
//////////////////////////////////////////////////////////////////////////
CRealtimeRemoteUpdateListener::CRealtimeRemoteUpdateListener():
	m_boIsEnabled(false),m_lastKeepAliveMessageTime((const int64)0)
{
}
//////////////////////////////////////////////////////////////////////////
CRealtimeRemoteUpdateListener::~CRealtimeRemoteUpdateListener()
{
}
//////////////////////////////////////////////////////////////////////////
void	CRealtimeRemoteUpdateListener::LoadArchetypes(XmlNodeRef &root)
{
	IEntitySystem		*pEntitySystem	= gEnv->pEntitySystem;

	// Remove Entities with ID`s from the list.
	for (int i = 0; i < root->getChildCount(); i++)
	{
		XmlNodeRef entityNode = root->getChild(i);
		if (entityNode->isTag("EntityPrototype"))
		{
			pEntitySystem->LoadEntityArchetype(entityNode);
		}
	}	
}

//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::LoadTimeOfDay( XmlNodeRef &root )
{
	gEnv->p3DEngine->GetTimeOfDay()->Serialize( root,true );
}

//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::LoadMaterials( XmlNodeRef &root )
{
	// Remove Entities with ID`s from the list.
	for (int i = 0; i < root->getChildCount(); i++)
	{
		XmlNodeRef mtlNode = root->getChild(i);
		if (mtlNode->isTag("Material"))
		{
			const char *mtlName = mtlNode->getAttr( "name" );
			gEnv->p3DEngine->GetMaterialManager()->LoadMaterialFromXml( mtlName,mtlNode );
		}
	}
}
//////////////////////////////////////////////////////////////////////////
void  CRealtimeRemoteUpdateListener::LoadConsoleVariables(XmlNodeRef &root )
{
	IConsole*	piConsole(NULL);
	char*			szKey(NULL);
	char*			szValue(NULL);
	ICVar*		piCVar(NULL);

	piConsole=gEnv->pConsole;
	if (!piConsole)
	{
		return;
	}

	// Remove Entities with ID`s from the list.
	for (int i = 0; i < root->getNumAttributes(); i++)
	{
		root->getAttributeByIndex(i,(const char**)&szKey,(const char**)&szValue);
		piCVar=piConsole->GetCVar(szKey);
		if (!piCVar)
		{
			continue;
		}
		piCVar->Set(szValue);
	}
}
//////////////////////////////////////////////////////////////////////////
void  CRealtimeRemoteUpdateListener::LoadParticles(XmlNodeRef &root )
{
	XmlNodeRef		oParticlesLibrary;
	XmlNodeRef		oLibrary;
	XmlString			strLibraryName;

	int						nCurrentChild(0);
	int						nNumberOfChildren(0);

	oParticlesLibrary=root->findChild("ParticlesLibrary");
	if (!oParticlesLibrary)
	{
		return;
	}

	nNumberOfChildren=oParticlesLibrary->getChildCount();
	for (nCurrentChild=0;nCurrentChild<nNumberOfChildren;++nCurrentChild)
	{
		oLibrary=oParticlesLibrary->getChild(nCurrentChild);
		if (oLibrary->isTag("Library"))
		{
			continue;
		}
		if (!oLibrary->getAttr("name",strLibraryName))
		{
			continue;
		}
		gEnv->pParticleManager->LoadLibrary((const char*)strLibraryName,oLibrary,true);
	}
}
//////////////////////////////////////////////////////////////////////////
void  CRealtimeRemoteUpdateListener::LoadTerrainLayer(XmlNodeRef &root, unsigned char*	uchData)
{
	int texId(0);
	int posx(0),posy(0);
	int w(0),h(0);
	int					nSourceFormat(0);
	ETEX_Format	eTFSrc(eTF_R8G8B8);

	if (!root->getAttr("Posx",posx))
	{
		return;
	}

	if (!root->getAttr("Posy",posy))
	{
		return;
	}

	if (!root->getAttr("w",w))
	{
		return;
	}

	if (!root->getAttr("h",h))
	{
		return;
	}

	if (!root->getAttr("ETEX_Format",nSourceFormat))
	{
		return;
	}

	eTFSrc=(ETEX_Format)nSourceFormat;

	if (gEnv->pRenderer&&gEnv->p3DEngine)
	{
		texId = gEnv->pRenderer->DownLoadToVideoMemory(uchData,w,h,eTFSrc,eTFSrc, 0, false, FILTER_NONE, 0, NULL, FT_USAGE_ALLOWREADSRGB);
		// Swapped x & y for historical reasons.
		gEnv->p3DEngine->SetTerrainSectorTexture(posy,posx,texId);		
	}
}

//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::LoadEntities( XmlNodeRef &root )
{
	

	IEntitySystem *pEntitySystem = gEnv->pEntitySystem;

	bool bTransformOnly(false);
	bool bDeleteOnly = false;
	bool bRemoveAllOld = true;

	gEnv->pSystem->SetThreadState(ESubsys_Physics,false);

	if (root->haveAttr("PartialUpdate"))
	{
		bRemoveAllOld = false;
	}
	if (root->haveAttr("Delete"))
	{
		bDeleteOnly = true;
	}

	//////////////////////////////////////////////////////////////////////////
	// Delete all entities except the unremovable ones and the local player.
	if (bRemoveAllOld)
	{
		IEntityItPtr pIt = pEntitySystem->GetEntityIterator();

		if ( ! gEnv->pGame )
			return;

		IGameFramework * piGameFramework(gEnv->pGame->GetIGameFramework());
		IEntity	* piRulesEntity(NULL);
		if (piGameFramework)
		{
			IGameRulesSystem * piGameRulesSystem(piGameFramework->GetIGameRulesSystem());
			if (piGameRulesSystem)
			{
				piRulesEntity=piGameRulesSystem->GetCurrentGameRulesEntity();
			}
		}

		pIt->MoveFirst();
		while (!pIt->IsEnd())
		{
			IEntity * pEntity = pIt->Next();
			IEntityClass * pEntityClass=pEntity->GetClass();
			uint32 nEntityFlags = pEntity->GetFlags();

			// Local player must not be deleted.
			if (nEntityFlags & ENTITY_FLAG_LOCAL_PLAYER)
				continue;

			// Rules should not be deleted as well.
			if (piRulesEntity)
			{
				if (pEntity->GetId()==piRulesEntity->GetId())
				{
					continue;
				}
			}

			// Specific for GDCE
			// This should ALWAYS be true
			if (pEntityClass)
			{
				// We don't want to sync guns now.
				string strClassName(pEntityClass->GetName());
				strClassName.MakeLower();
				if (strstr(strClassName.c_str(),"gun"))
				{
					continue;
				}
			}

#if 0
#ifdef XENON
			OutputDebugString(pEntity->GetName());
			OutputDebugString(":");
			OutputDebugString(pEntity->GetClass()->GetName());
			OutputDebugString("\n");
#endif //XENON
#endif //0

			pEntity->ClearFlags(ENTITY_FLAG_UNREMOVABLE);
			pEntitySystem->RemoveEntity( pEntity->GetId() );
		}
		// Force deletion of removed entities.
		pEntitySystem->DeletePendingEntities();
		//////////////////////////////////////////////////////////////////////////
	}
	else
	{
		// Remove Entities with ID`s from the list.
		for (int i = 0; i < root->getChildCount(); i++)
		{
			XmlNodeRef objectNode = root->getChild(i);
			if (objectNode->isTag("Entity"))
			{
				// reserve the id
				EntityId id;
				if (objectNode->getAttr( "EntityId", id ))
				{
					IEntity * pEntity = pEntitySystem->GetEntity(id);
					if (!pEntity)
					{
						pEntitySystem->RemoveEntity(id,true);
						continue;
					}

					if (!objectNode->getAttr("TransformOnly",bTransformOnly ))
					{
						pEntity->ClearFlags(ENTITY_FLAG_UNREMOVABLE);
						pEntitySystem->RemoveEntity(id,true);
						continue;
					}

					if (!bTransformOnly)
					{
						pEntity->ClearFlags(ENTITY_FLAG_UNREMOVABLE);
						pEntitySystem->RemoveEntity(id,true);
						continue;
					}

					//////////////////////////////////////////////////////////////////////////
					//TEMPORARY, should be removed after GDC09 Presentation
					XmlString strEntityClass;
					if (objectNode->getAttr("EntityClass",strEntityClass))
					{
						strEntityClass.MakeLower();
						if (strcmp(strEntityClass,"simplelight")!=0)
						{
							pEntity->ClearFlags(ENTITY_FLAG_UNREMOVABLE);
							pEntitySystem->RemoveEntity(id,true);
							continue;					
						}

						// GDCE specific: we don't want to sync guns.
						if (strstr(strEntityClass.c_str(),"gun"))
						{
							continue;
						}
					}
					//////////////////////////////////////////////////////////////////////////

					Vec3 oPos(0,0,0);
					Vec3 oScale(0,0,0);
					Quat oRotate(1,0,0,0);

					objectNode->getAttr("Pos",oPos );
					objectNode->getAttr("Rotate",oRotate );
					objectNode->getAttr("Scale",oScale );

					pEntity->SetPosRotScale(oPos,oRotate,oScale);

				}
			}
		}
		// Force deletion of removed entities.
		pEntitySystem->DeletePendingEntities();
	}

	if (!bDeleteOnly)
	{
		pEntitySystem->LoadEntities(root);
	}

	// you can't pass temporaries to non-const references, so objects on the stack must be created
	SEntityEvent LevelLoaded(ENTITY_EVENT_LEVEL_LOADED);
	SEntityEvent StartGame(ENTITY_EVENT_START_GAME);

	pEntitySystem->SendEventToAll(LevelLoaded);
	pEntitySystem->SendEventToAll(StartGame);
	gEnv->pSystem->SetThreadState(ESubsys_Physics,true);
}
//////////////////////////////////////////////////////////////////////////
bool CRealtimeRemoteUpdateListener::IsSyncingWithEditor()
{
	CTimeValue	oTimeValue(gEnv->pTimer->GetAsyncTime());
	oTimeValue-=m_lastKeepAliveMessageTime;

	return (fabs((oTimeValue).GetSeconds()) <= 30.0f);
}
//////////////////////////////////////////////////////////////////////////
