#ifndef _CRY_LIST_H_
#define _CRY_LIST_H_
#pragma once

#include "Allocator.h"

namespace CryList
{
	ILINE bool TestPtrValidity(void* pointerToTest)
	{
		assert(pointerToTest != NULL);
		return true;
	}
}

#define for_all_ptrs(Type, p, cont) \
	for (Type* p = (cont).begin(), *_e = (cont).end(), *_next; (p != _e) && CryList::TestPtrValidity((void*)p) && ((_next = (cont).next(p)) || true); p = _next)

#define for_rev_all_ptrs(Type, p, cont) \
	for (Type* p = (cont).rbegin(), *_e = (cont).rend(), *_prev; (p != _e) && CryList::TestPtrValidity((void*)p) && ((_prev = (cont).prev(p)) || true); p = _prev)

////////////////////////////////////////////////////////////////////////
// Bidirectional list

template< class T, class TAlloc = StdAllocator >
class List
{
protected:

	struct Node: T
	{
		Node*	pNext;
		Node*	pPrev;

		Node()
			{}

		template<typename I>
		Node(I const& i)
			: T(i) {}
				
	};

	TAlloc		m_Alloc;

public:

	typedef T value_type;

	// Constructors, using default allocator or allocator initialiser.
	List()
	{ reset(); }

	template<class I>
	List(const I& init)
		: m_Alloc(init)
		{ reset(); }

	~List()
		{ clear(); }

	bool operator +() const
		{ return head() != NULL; }
	bool operator !() const
		{ return head() == NULL; }
	bool empty() const
		{ return head() == NULL; }

	CONST_VAR_FUNCTION( T& front(),
		{ assert(!empty()); return *head(); } )
	CONST_VAR_FUNCTION( T& back(),
		{ assert(!empty()); return *tail(); } )

	// Slow: do not call often
	int size() const
	{
		int nSize = 0;
		for_all_ptrs (const T, p, *this)
			nSize++;
		return nSize;
	}

	//
	// Iteration
	//
	CONST_VAR_FUNCTION( T* begin(),
		{ return head(); } )
	CONST_VAR_FUNCTION( T* end(),
		{ return NULL; } )
	static T* next(const T* p)
		{ return p ? get_node(p)->pNext : NULL; }

	CONST_VAR_FUNCTION( T* rbegin(),
		{ return tail(); } )
	CONST_VAR_FUNCTION( T* rend(),
		{ return NULL; } )
	static T* prev(const T* p)
	{ return p ? get_node(p)->pPrev : NULL; }

	//
	// Add elements
	//
	void* push_back_new()
	{
		Node* pNode = (Node*)m_Alloc.Allocate(pNode);		
		if( m_pTail) m_pTail->pNext = pNode;
		pNode->pPrev = m_pTail;
		pNode->pNext = NULL;
		m_pTail = pNode;
		if( m_pHead == NULL ) m_pHead = pNode;
		assert(tail() == pNode);
		return pNode;
	}
	T* push_back()
	{ 
		return new(push_back_new()) Node(); 
	}
	template<class I>
	T* push_back(const I& ini)
	{ 
		return new(push_back_new()) Node(ini); 
	}

	void* push_front_new()
	{
		Node* pNode = (Node*)m_Alloc.Allocate();		
		if( m_pHead) m_pHead->pPrev = pNode;
		pNode->pNext = m_pHead;
		pNode->pPrev = NULL;
		m_pHead = pNode;
		if( m_pTail == NULL ) m_pTail = pNode;
		assert(head() == pNode);
		return pNode;
	}
	T* push_front()
	{ 
		return new(push_front_new()) Node(); 
	}
	template<class I>
	T* push_front(const I& ini)
	{ 
		return new(push_front_new()) Node(ini); 
	}

	//
	// Remove elements
	//
	void erase(T* p)
	{
		Node* pNode = get_node(p);
		
		if(  pNode == m_pHead && pNode == m_pTail) // only node in list
		{
			m_pHead = NULL;
			m_pTail = NULL;
		}
		else if( pNode == m_pHead ) // node is head node
		{
			m_pHead = pNode->pNext;
			m_pHead->pPrev = NULL;
		}
		else if( pNode == m_pTail ) // node is tail node
		{
			m_pTail = pNode->pPrev;
			m_pTail->pNext = NULL;
		}
		else // somewhere in the middel of the list
		{
			pNode->pPrev->pNext = pNode->pNext;
			pNode->pNext->pPrev= pNode->pPrev;
		}
		Delete(m_Alloc, pNode);	
	}

	void pop_back()
	{
		assert(!empty());
		erase(tail());
	}

	void pop_front()
	{
		assert(!empty());
		erase(head());
	}

	void clear()
	{
		// Destroy all elements, in reverse order
		for (Node* p = tail(); p != NULL; )
		{
			Node* pPrev = p->pPrev;
			Delete(m_Alloc, p);
			p = pPrev;
		}
		reset();
	}
	
	template<class TSizer>
	void GetMemoryUsage( TSizer* pSizer ) const
	{
		m_Alloc.GetMemoryUsage<Node>(pSizer);
		for_all_ptrs (const T, p, *this)
		{
			p->GetMemoryUsage(pSizer);				
		}
	}

protected:

	Node*	m_pHead;
	Node* m_pTail;

protected:
	
	CONST_VAR_FUNCTION( Node* head(),
		{ return m_pHead; } )
	CONST_VAR_FUNCTION( Node* tail(),
		{ return m_pTail; } )

	void reset()
	{
		m_pHead = m_pTail = NULL;
	}

	static Node* get_node(T* p)
	{
		Node* pNode = static_cast<Node*>(p);
		return pNode;
	}
	static const Node* get_node(const T* p)
	{
		const Node* pNode = static_cast<const Node*>(p);
		return pNode;
	}

};

#endif
