#ifndef _CryMTrace_h_
#define _CryMTrace_h_ 1

// This file is ps3 only!
#if !defined(PS3)
# error "This file should only be included on ps3"
#endif

#include <CryModuleDefs.h>
#include <CryThread.h>

namespace MTrace
{
  // Statistics gathered for each module seperately 
  struct ModuleStats
  {
    // The current memory footprint in bytes
    size_t current_memory; 
    // The peak memory footprint 
    size_t peak_memory; 
    // The system memory footprint for the module (text+bss+data)
    size_t system_memory; 
    // Total Ammount of memory allocated during trace
    uint64_t allocated;
    // Total Ammount of memory freed during trace 
    uint64_t freed;
    // Total number of memory allocations
    uint64_t num_allocations;
  }; 

  // allocations statistics gathered by mtrace  
  struct AllocationStats
  {
    // The number of bytes allocated 
    uint64_t allocated_memory; 
    // The number of bytes freed 
    uint64_t freed_memory; 
    // The number of calls to malloc()
    size_t malloc_calls;
    // The number of calls to calloc()
    size_t calloc_calls;
    // The number of calls to realloc()
    size_t realloc_calls;
    // The number of calls to memalign()
    size_t memalign_calls;
    // The number of calls to free()
    size_t free_calls;
    // The system local thread id 
    uint16_t thread_local_id;
    // The thread id (~0U for accumulated system wide allocation statistics)
    uint32_t thread_id;
    // The thread id (~0U for accumulated system wide allocation statistics)
    uint32_t thread_sys_id;
    // Flag describing if tracing is enabled for this thread 
    uint32_t thread_recording;
  }; 

  // system wide statistics gathered by mtrace 
  struct SystemStats
  {
    // The overall allocations statistics for allocations 
    AllocationStats allocations; 
    // The statistics for each single module
    ModuleStats modules[eCryM_Num];
    // Statistics for code that do not belong to a module
    ModuleStats unknown;
  };  

  // Structure containing all the required 
  struct Config 
  {
    // A specific address to be catched.  If this is not NULL and a memory
    // operation receives or returns the specfied address, then the global
    // function MTrace::Catch() is called.
    void* CatchAddr;

    // A specfied request size to be catched.  If this is not 0 and an
    // allocation or re-allocation operation receives the specfied size
    // parameter, then the global function MTrace::Catch() is called.
    size_t CatchSize;

    // Flag indicating if the MTrace module is enabled.
    //
    // Note that the MTrace is initially enabled because otherwise early
    // allocations (before 'launcher.cfg' is read) would be missed.  The
    // launcher will turn MTrace off if it is not needed.
    //
    // The early allocations are collected in the I/O buffer.  In case of an
    // overrun, an error message is dumped to the console and the MTrace module
    // is disabled (the launcher may _not_ turn it back on in this case).
    bool RecordingEnabled;

    // Flag indicating if the MTrace module is connected to an mtrace
    // server 
    bool Connected;

    // Flag indicating if the MTRACE module should print the memory
    // operations to stdout. Can be conveniently enabled/disabled from
    // the debugger 
    bool PrintToStdout; 

    // The numerical ip address of the mtrace server.
    char ServerAddress[16]; 

    // The port that should be used to communicate with the mtrace
    // server
    uint16_t ServerPort; 
  };
  extern Config g_Config;

  // Initialize mtrace
  bool Initialize();

  // Initialize mtrace
  bool Shutdown();

  // Establishes a connection to the mtrace server. Returns true if
  // the connection was succesfully establised.
  bool Connect();

  // Disconnects from the mtrace server. Returns true if the
  // connection was succesfully closed.
  bool Disconnect();

  // Create a snapshot. All memory operations after issueing this call
	// will be grouped under the name of the snapshot given to this
	// function. 
	void SnapShot(const char* name);

  // Process messages from mtrace server (if connected!)
  void Process(); 

  // Register a named thread to mtrace 
	void RegisterThreadName(uint32_t threadId, const char* name);

  // Enable thread allocation tracing for a specific thread 
	void EnableThreadTrace(uint32_t threadId);

  // Disable thread allocation tracing for a specific
	void DisableThreadTrace(uint32_t threadId);

  // Accumulates the statistics gathered by all threads 
  void AccumulateThreadStatistics(AllocationStats& results);

  // Retrieve the moving average of memory operations, and allocated/freed
  // memory per frame 
  void FrameStats(float &avgOps, float &avgAllocs, float &avgFreed);

  // Retrieve the system wide memory statistics 
  bool SystemStatistics(SystemStats& results);

  // Retrieve the memory statistics by thread id 
  bool ThreadStatistics(size_t threadId, AllocationStats& results);

  // Simple Scope based thread tracer 
  struct ThreadTraceSwitch 
  {
    ThreadTraceSwitch() { EnableThreadTrace(GetCurrentThreadId()); }
    ~ThreadTraceSwitch() { DisableThreadTrace(GetCurrentThreadId()); }
  };

  // The tracing functions. These should be called with the appropiate
  // parameters when the respective memory operation is performed.
  void trace_calloc(void* ptr, size_t size);
  void trace_malloc(void *ptr, size_t size);
  void trace_free(void* ptr, size_t size);
  void trace_realloc(void *old_ptr, void* new_ptr, size_t old_size, size_t new_size);
  void trace_memalign(void* ptr, size_t boundary, size_t size);
}; 

#endif
