////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek
// -------------------------------------------------------------------------
//  File name:   CryOnline.h
//  Created:     21/05/2009 by Alexey Vasilenko
//  Description: 
// -------------------------------------------------------------------------
//  History: 
//
////////////////////////////////////////////////////////////////////////////

#ifndef __CRYONLINE_H__
#define __CRYONLINE_H__

#if _MSC_VER > 1000
# pragma once
#endif

#ifdef SWIG
%module(directors="1") CryOnline
%feature("director") IOnlineConnectionListener;
%feature("director") IOnlineProfileListener;
%feature("director") IOnlineChatRoomListener;
%feature("director") IOnlineAccount;
%feature("director") IOnlineChatListener;
%feature("director") IOnlineQueryBinder;
#endif

struct ICryOnlineListener 
{
	virtual ~ICryOnlineListener() {}

	virtual void OnConnected() = 0;
	virtual void OnDisconnected() = 0;
	virtual void OnLogMessage(const char * msg) = 0;
};

struct ICryOnlineChannel
{
	virtual void SetOnlineId(const char * onlineId) = 0;
	virtual void SetPassword(const char * password) = 0;
	virtual const char * GetOnlineId() = 0;

	virtual void Connect() = 0;
	virtual void Disconnect() = 0;

	virtual void Subscribe(ICryOnlineListener * listener) = 0;
	virtual void Unsubscribe(ICryOnlineListener * listener) = 0;
};

struct ICryOnline
{
	virtual ICryOnlineChannel * CreateChannel() = 0;
	virtual void DestroyChannel(ICryOnlineChannel * channel) = 0;

	virtual void Subscribe(ICryOnlineListener * listener) = 0;
	virtual void Unsubscribe(ICryOnlineListener * listener) = 0;

	virtual void Tick() = 0;
};


enum EOnlineError
{
	eOnlineError_NoError,
	eOnlineError_StreamError,
	eOnlineError_StreamVersionError,  
	eOnlineError_StreamClosed,        
	eOnlineError_ProxyAuthRequired,   
	eOnlineError_ProxyAuthFailed,     
	eOnlineError_ProxyNoSupportedAuth,
	eOnlineError_IoError,             
	eOnlineError_ParseError,          
	eOnlineError_ConnectionRefused,   
	eOnlineError_DnsError,            
	eOnlineError_OutOfMemory,         
	eOnlineError_NoSupportedAuth,     
	eOnlineError_TlsFailed,           
	eOnlineError_TlsNotAvailable,     
	eOnlineError_CompressionFailed,   
	eOnlineError_AuthenticationFailed,
	eOnlineError_UserDisconnected,    
	eOnlineError_NotConnected,

	eOnlineError_UnknownError
};

enum EOnlineConnectionState
{
	eOnlineConnectionState_Disconnected,
	eOnlineConnectionState_Connecting,
	eOnlineConnectionState_Connected,

	eOnlineConnectionState_Unknown
};

enum EOnlineStanzaType
{
	eOnlineStanzaType_Get,
	eOnlineStanzaType_Set,
	eOnlineStanzaType_Result,
	eOnlineStanzaType_Error,

	eOnlineStanzaType_Unknown
};

enum EOnlinePresence
{
	eOnlinePresence_Unknown,
	eOnlinePresence_Available,
	eOnlinePresence_Chat,
	eOnlinePresence_Away,
	eOnlinePresence_Dnd,
	eOnlinePresence_Xa,
	eOnlinePresence_Unavailable
};

struct IOnlineConnection;

struct IOnlineConnectionListener
{
	virtual ~IOnlineConnectionListener() {}

	virtual void OnConnectionAvailable(IOnlineConnection * connection) = 0;
	virtual void OnConnectionLost(IOnlineConnection * connection) = 0;
	virtual void OnConnectionTick(IOnlineConnection * connection) = 0;
};

struct IOnlineQueryBinder : public IOnlineConnectionListener
{
	virtual void OnQueryCompleted(const char * xml, int queryId) = 0;
	virtual void OnRequest(const char * xml, IOnlineConnection * connection) = 0;
};

struct IOnlineDownloadBinder : public IOnlineConnectionListener
{
	virtual void OnDownloadCompleted(const char * filename, const void * data, int size) = 0;
};

struct IOnlineUploadBinder : public IOnlineConnectionListener
{
	virtual void OnUploadCompleted(const char * filename) = 0;
};

struct IOnlineProfileListener
{
	virtual ~IOnlineProfileListener() {}

	virtual void OnProfileLoaded() = 0;
	virtual void OnProfileSaved() = 0;
};

struct IOnlineProfile
{
	virtual void Load() = 0;
	virtual void Save() = 0;
	virtual void RegisterListener(IOnlineProfileListener * listener) = 0;
	virtual void UnregisterListener(IOnlineProfileListener * listener) = 0;
};

struct IOnlineConfiguration
{
	virtual ~IOnlineConfiguration() {};

	virtual const char * GetDomain() = 0;
	virtual void SetDomain(const char * domain) = 0;

	virtual const char * GetServer() = 0;
	virtual void SetServer(const char * server) = 0;
 
	virtual int GetServerPort() = 0;
	virtual void SetServerPort(int port) = 0;

	virtual const char * GetHost() = 0;
	virtual void SetHost(const char * host) = 0;

	virtual const char * GetResource() = 0;
	virtual void SetResource(const char * resource) = 0;

	virtual const char * GetOnlineId() = 0;
	virtual void SetOnlineId(const char * onlineId) = 0;

	virtual const char * GetFSProxy() = 0;
	virtual void SetFSProxy(const char * proxy) = 0;

	virtual int GetFSProxyPort() = 0;
	virtual void SetFSProxyPort(int port) = 0;

	virtual const char * GetFSServiceId() = 0;
	virtual void SetFSServiceId(const char * fsId) = 0;

	virtual const char * GetChatServiceId() = 0;
	virtual void SetChatServiceId(const char * chatId) = 0;
};

struct IOnlineAccount : public ICryOnlineListener
{
	virtual void Tick() = 0;
	virtual void SetCredentials(const char * onlineId, const char * password) = 0;
	virtual IOnlineConfiguration * GetConfiguration() = 0;
	virtual IOnlineConnection * GetConnection() = 0;
	virtual const char * GetCVar(const char * varName) = 0;
	virtual void SetCVar(const char * varName, const char * varVal) = 0;
};


struct IOnlineChatRoomListener
{
	virtual ~IOnlineChatRoomListener() {}

	virtual void OnChatRoomMessage(const char * nick, const char * message, bool isPrivate) = 0;
	virtual void OnChatRoomPresence(const char * nick, EOnlinePresence presence) = 0;
	virtual void OnChatRoomParticipant(const char * nick) = 0;
};

struct IOnlineChatRoom
{
	virtual ~IOnlineChatRoom() {}
	virtual bool RegisterListener(IOnlineChatRoomListener * listener) = 0;
	virtual bool UnregisterListener(IOnlineChatRoomListener * listener) = 0;
	virtual bool IsListenersEmpty() = 0;
	virtual void Join() = 0;
	virtual void Leave() = 0;
	virtual void EnumerateParticipants() = 0;
	virtual void Send(const char * message) = 0;
	virtual void SendPrivate(const char * nick, const char * message) = 0;
};

struct IOnlineChatListener
{
	virtual ~IOnlineChatListener() {}

	virtual void OnChatRoomDiscovered(const char * roomId) = 0;
};

struct IOnlineChat
{
	virtual bool RegisterListener(IOnlineChatListener * listener) = 0;
	virtual bool UnregisterListener(IOnlineChatListener * listener) = 0;
	virtual void DiscoverChatRooms() = 0;
	virtual IOnlineChatRoom * GetChatRoom(const char * roomId) = 0;
};

struct IOnlineConnection
{
	virtual int Query(const char * query, const char * receiverId, IOnlineQueryBinder * binder) = 0;
	virtual void Response(const char * xml) = 0;
	virtual void Download(const char * filename, IOnlineDownloadBinder * binder) = 0;
	virtual void Upload(const char * filename, const void * data, int size, const char * receiverId, const char* resource, IOnlineUploadBinder * binder) = 0;

	virtual IOnlineChat * GetChat() = 0;
};

struct IOnline 
{
	virtual void Tick() = 0;
	virtual void Shutdown() = 0;
	virtual bool Connect(IOnlineAccount * account, const char * password) = 0;
	virtual bool Disconnect(const char * domain) = 0;
	virtual bool RegisterConnectionListener(IOnlineConnectionListener * listener, const char * domain) = 0;
	virtual bool UnregisterConnectionListener(IOnlineConnectionListener * listener) = 0;
	virtual IOnlineConfiguration * CreateConfiguration() = 0;
	virtual void SetAccount(IOnlineAccount * account) = 0;
	virtual void RegisterQueryBinder(IOnlineQueryBinder * binder, const char * tag) = 0;
	virtual void UnregisterQueryBinder(IOnlineQueryBinder * binder) = 0;
};

#ifdef SWIG
#	define CRYONLINE_API
#elif defined(__GNUC__)
#	define CRYONLINE_API __attribute__ ((visibility("default")))
#else
#	ifdef CRYONLINE_EXPORTS
#		define CRYONLINE_API __declspec(dllexport)
#	else
#		define CRYONLINE_API __declspec(dllimport)
#	endif
#endif

#ifdef LINUX
extern struct IOnline* CryOnlineGetInstance();
extern "C" CRYONLINE_API struct IOnline * CryOnlineCreateInstance();
#else
extern "C" CRYONLINE_API void CryOnlineInit();
extern "C" CRYONLINE_API void CryOnlineShutdown();
extern "C" CRYONLINE_API struct IOnline * CryOnlineGetInstance();

extern "C" CRYONLINE_API void CryOnlineInit2();
extern "C" CRYONLINE_API void CryOnlineShutdown2();
extern "C" CRYONLINE_API struct ICryOnline * CryOnlineGetInstance2();
#endif

#endif // __CRYONLINE_H__