////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek Studios, 2008.
// -------------------------------------------------------------------------
//  File name:   IAnimAction.h
//  Created:     20/05/2008 by Timur.
//  Modified:    24/07/2008 by MarcoK
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef __IANIMACTION_H__
#define __IANIMACTION_H__

#pragma once

struct IAnimAction;
struct IAnimActionActor;
struct IAnimActionClass;
struct CAnimation;

//////////////////////////////////////////////////////////////////////////
// Description:
//    SAnimActionPropertyDesc struct
//////////////////////////////////////////////////////////////////////////
enum EAnimActionPropertyType
{
	eAAPT_null,
	eAAPT_string,
	eAAPT_float,
	eAAPT_int,
	eAAPT_bool,
	eAAPT_animation,
};

struct SAnimActionPropertyDesc
{
	EAnimActionPropertyType propertyType;
	const char* propertyName;
	const char* defaultValue;
	const char* displayName;
	const char* description;
};

struct SAnimActionAnimationDesc
{
	string animation;
	float  params[2];
};


//////////////////////////////////////////////////////////////////////////
// Description:
//   This structure represent a single Property of the animation action.
//////////////////////////////////////////////////////////////////////////
struct SAnimActionProperty
{
	struct SAnimationDesc
	{
		string aniamtion;
		float  params[2];
	};
	
	int nParamId;      // Id of the parameter
	EAnimActionPropertyType propertyType;

	string m_str;
	union
	{
		float m_float;
		int m_int;
		bool m_bool;
	};

	//////////////////////////////////////////////////////////////////////////
	// methods
	//////////////////////////////////////////////////////////////////////////
	const string& GetString() const { return m_str; }
	float GetFloat() const { return m_float; }
	int GetInt() const { return m_int; }
	bool GetBool() const { return m_bool; }

	void SetString( const string& value ) { m_str = value; }
	void SetFloat( float value ) { m_float = value; }
	void SetInt( int value ) { m_int = value; }
	void SetBool( bool value ) { m_bool = value; }

	string GetAsString( const SAnimActionPropertyDesc* propertyDesc ) const
	{
		assert( propertyDesc->propertyType != eAAPT_null );
		string result;
		switch ( propertyDesc->propertyType )
		{
		case eAAPT_string:
			return m_str;
		case eAAPT_float:
			result.Format( "%.8g", m_float );
			break;
		case eAAPT_int:
			result.Format( "%d", m_int );
			break;
		case eAAPT_bool:
			result = m_bool ? "1" : "0";
			break;
		default:
			assert( !"Processing a property of undefined type!" );
		}
		return result;
	}
	void SetAsString( const SAnimActionPropertyDesc* propertyDesc, const char* value )
	{
		assert( propertyDesc->propertyType != eAAPT_null );
		if ( value == NULL )
			value = propertyDesc->defaultValue;
		switch ( propertyDesc->propertyType )
		{
		case eAAPT_string:
			m_str = value;
			break;
		case eAAPT_float:
			m_float = (float) atof( value );
			break;
		case eAAPT_int:
			m_int = atoi( value );
			break;
		case eAAPT_bool:
			m_bool = atoi( value ) != 0;
			break;
		default:
			assert( !"Processing a property of undefined type!" );
		}
	}
};
typedef DynArray< SAnimActionProperty > TAnimActionPropertyValues;

//////////////////////////////////////////////////////////////////////////
// Description:
//    IAnimActionParameter interface
//////////////////////////////////////////////////////////////////////////
struct SAnimActionParameter
{
	int                      nParamId;      // Id of the parameter
	EAnimActionPropertyType  type;          // Type of this parameter.
	const char *name;         // Variable name.
	const char *displayName;  // Variable user readable name.
	unsigned int nFlags;      // Flags for this parameters.

	const char* defaultValue;
	const char* description;
};

struct SAnimActionAnimation
{
	string animation;
	float  fParam[3];  //3 Optional parameters per animation.
	TAnimActionPropertyValues properties;
	_smart_ptr< IAnimActionClass > prototype;

	SAnimActionAnimation() { fParam[0] = 0; fParam[1] = 1; }
};


// Some params for the future.
struct SAnimActionStartParams
{
	uint32 m_nFlags; // Flags for starting AnimAction
	int    m_nLayer; // Layer where anim action playing
	SAnimActionStartParams() : m_nFlags(0),m_nLayer(-1) {};
};



//////////////////////////////////////////////////////////////////////////
// Description:
//    IAnimActionClass interface
//    Every AnimAction belongs to a certain class
//    Example classes are Locomotion, Sequence, Random, etc...
//////////////////////////////////////////////////////////////////////////
struct IAnimActionClass : public _reference_target_t
{
	//////////////////////////////////////////////////////////////////////////
	// Returns the name of this AnimAction class
	virtual const char* GetAAClassName() const = 0;

	// Get animation layer preferred by this animation action.
	virtual uint32 GetDesiredLayer() const = 0;

	// returns an array of property descriptors (eAAPT_null terminated)
	virtual const SAnimActionPropertyDesc* GetPropertiesDesc() const = 0;

	// returns an array of property descriptors for child animation entries (eAAPT_null terminated)
	virtual const SAnimActionPropertyDesc* GetChildPropertiesDesc() const = 0;

	// Creates a new instance of the prototype class
	virtual IAnimActionClass* CreatePrototype() = 0;

	
	// Creates an instance of this AnimAction prototype
//	virtual IAnimAction* CreateAnimAction( IAnimActionActor* pActor ) = 0;

	//////////////////////////////////////////////////////////////////////////
	// For Prototypes:
	//////////////////////////////////////////////////////////////////////////
	// Get name of the prototype anim action
	virtual const char* GetActionName() const = 0;
	// Set name of the prototype anim action
	virtual void SetActionName( const char* sActionName ) = 0;
	// Return file name of the anim action prototype
	virtual const char* GetFileName() const = 0;
	// Set file name of the prototype anim action
	virtual void SetFileName( const char* sFilename ) = 0;

	//////////////////////////////////////////////////////////////////////////

	// Return number of animations in anim action prototype.
	int GetAnimationCount() const { return (int)m_childAnimations.size(); }
	// Return indexed animation from anim action prorotype.
	const SAnimActionAnimation& GetAnimation( int nIndex ) const { assert(nIndex>=0&&nIndex<(int)m_childAnimations.size()); return m_childAnimations[nIndex]; }

	// Return number of properties in anim action prototype.
	int GetPropertyCount() const { return (int)m_properties.size(); }
	// Return indexed property from anim action prototype.
	const SAnimActionProperty& GetProperty( int nIndex ) const 
	{ 
		uint32 numProps=m_properties.size();
		assert(nIndex>=0); 
		assert(nIndex<(int32)numProps); 
		return m_properties[nIndex]; 
	}

	//////////////////////////////////////////////////////////////////////////
	// Do not use
	//////////////////////////////////////////////////////////////////////////
	const TAnimActionPropertyValues& GetProperties() const { return m_properties; }
	TAnimActionPropertyValues& GetProperties() { return m_properties; }
	const DynArray<SAnimActionAnimation>& GetChildAnimations() const { return m_childAnimations; }
	DynArray<SAnimActionAnimation>& GetChildAnimations() { return m_childAnimations; }
	//////////////////////////////////////////////////////////////////////////

protected:
	TAnimActionPropertyValues m_properties;
	DynArray<SAnimActionAnimation> m_childAnimations;
};










template <typename T> class TAnimActionClass : public IAnimActionClass
{
public:

	virtual const char* GetActionName() const 
	{ 
		return m_actionName.c_str(); 
	};
	virtual void SetActionName( const char *sActionName ) 
	{ 
		m_actionName = sActionName; 
	};

	virtual const char* GetFileName() const { return m_filename.c_str(); };
	virtual void SetFileName( const char *sFilename ) { m_filename = sFilename; };

	// IAnimActionClass
	virtual const SAnimActionPropertyDesc* GetPropertiesDesc() const { return NULL; }

	virtual const char** GetNamedChildEntries() const { return NULL; }

	virtual const SAnimActionPropertyDesc* GetChildPropertiesDesc() const { return NULL; }

	virtual IAnimActionClass* CreatePrototype()
	{
		return new T;
	}

	// Get animation layer prefered by this animation action.
	virtual uint32 GetDesiredLayer() const { return 0; };


	//~ IAnimActionClass
	//////////////////////////////////////////////////////////////////////////

protected:
	string m_actionName;
	string m_filename;
};

// TEMP: Move above
#include <CryExtension/ICryUnknown.h>

// TEMP: This will become local to CryAction.
struct IAALocomotion :
	public ICryUnknown
{
	CRYINTERFACE_DECLARE(IAALocomotion, 0x1f32622c92504af9, 0xbea4dee35474ce6e)

	virtual f32 GetDesiredSpeed() const = 0;
	virtual const Vec2& GetWorldDesiredBodyDirection() const = 0;
	virtual const Vec2& GetWorldDesiredMoveDirection() const = 0;
	virtual const f32 GetDesiredSlope() const = 0;
	virtual const f32 GetDesiredSlopeMoveDir() const = 0;
	virtual void SetDesiredSpeed(f32 fDesiredSpeed) = 0;
	virtual void SetDesiredSlope(f32 fDesiredSlopeMoveDir,f32 fDesiredSlope) = 0;
	virtual void SetWorldDesiredBodyDirection(const Vec3& vWorldDesiredBodyDirection) = 0;
	virtual void SetWorldDesiredMoveDirection(const Vec3& vWorldDesiredMoveDirection) = 0;
};

//////////////////////////////////////////////////////////////////////////
// Description:
//    IAnimAction interface
//////////////////////////////////////////////////////////////////////////
struct IAnimAction :
	public ICryUnknown//, public _reference_target_t
{
	CRYINTERFACE_DECLARE(IAnimAction, 0xb9cc34837aac4959, 0xa2e1a7b7972a0502)

	struct SPlaybackContext
	{
		bool bCanRemove;
		bool bLastInQueue;
		bool bCanActivate;
		SAnimActionStartParams params;
	};
	enum EActionStatus
	{
		eActionStatusDisabled,
		eActionStatusEnabled,
		eActionStatusBlendIn,
		eActionStatusBlendOut,
	};

	virtual void Init( IAnimActionActor *pActor ) =0;

	virtual void AnimActionInit() =0;
	// AnimAction prototype which defines this AnimAction
	const IAnimActionClass* GetPrototype() const { return m_pPrototype; }
	// get the name of the class
	virtual const char* GetAAClassName() const =0;
	// Actor who owns this animation action
	IAnimActionActor* GetActor() const { return m_pAnimActionActor; }
	// must return true if still wants to stay in the queue
	virtual bool Update( const SPlaybackContext& context ) = 0;

	virtual void GetMemoryUsage(ICrySizer *pSizer) const = 0;
	//////////////////////////////////////////////////////////////////////////
	// Draw debug info about what happens inside anim action
	//////////////////////////////////////////////////////////////////////////
	struct SDebugDrawInfo {	float x,y,ystep;};
	virtual void DebugDraw( SDebugDrawInfo &info ) = 0;


	//////////////////////////////////////////////////////////////////////////
	IAnimActionActor* m_pAnimActionActor;
	SAnimActionStartParams params; // Params used when this anim action started.
	EActionStatus status;     // Status of this anim action.

	_smart_ptr<IAnimActionClass> m_pPrototype;
};

typedef cryshared_ptr<IAnimAction> IAnimActionPtr;

/////////////////////////////////// ///////////////////////////////////////
// Description:
//    IAnimActionActor interface
//    Created for each character, and manages all animation actions for this character
//////////////////////////////////////////////////////////////////////////
struct IAnimActionActor
{
/*
	virtual void Init( ICharacterInstance* pCharInstance ) =0;
	virtual void Update() =0;
*/
	virtual void SetAnimLocation(const QuatT& qtAnimLocation) =0;
	virtual const QuatT& GetAnimLocation() const = 0;
	virtual bool StartAnimAction( const char *sAnimAction,SAnimActionStartParams &params ) = 0;
	virtual bool StopAnimAction(uint32 nLayer) = 0;

	//////////////////////////////////////////////////////////////////////////
	virtual int32 GetActiveAnimAction() const = 0;
	virtual IAnimAction* GetActiveIAnimAction() const = 0;
};


//////////////////////////////////////////////////////////////////////////
// Description:
//    IAnimActionManager interface
//    Knows about all existing animations action classes
//////////////////////////////////////////////////////////////////////////
struct IAnimActionManager
{
	// Functions to enumerate all existing animation action classes.
	virtual size_t GetAAClassCount() const=0;
	// Get the interface to that class.
	virtual IAnimActionClass* GetAAClass(size_t nIndex)=0;

	// Creates a new prototype from the XML.
	// XML must be a valid AnimAction description xml or the call will return NULL.
	virtual void CreateAAPrototype( const char* szFilePath, const char* szAnimName,uint32 nGlobalAnimID)=0;
	virtual uint32 GetFlagsAAC(uint32 nGlobalAnimID)=0;
	virtual IAnimActionClass* CreatePrototypeFromXml( const char *sActionName,XmlNodeRef &node )=0;
	virtual int CreateGlobalMotionHeaderAAC(const char* strFilePath)=0;
};


#endif // __IANIMACTION_H__
