////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   IPlatformOS.h
//  Created:     18/12/2009 by Alex Weighell.
//  Description: Interface to the Platform OS
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include DEVIRTUALIZE_HEADER_FIX(IPlatformOS.h)

#ifndef __IPLATFORMOS_H_
#define __IPLATFORMOS_H_

#include <CryFixedString.h>
#include <CryExtension/CrySharedPtr.h>

#if defined(PS3)
#include <sysutil/sysutil_sysparam.h>
#include <sysutil/sysutil_savedata.h>
#include "ICryPak.h"
#elif defined(WIN32) || defined(WIN64)
//#include <Lmcons.h> // this causes issues when including other windows headers later by defining PASCAL
#endif

class CSysCallThread;

// Interface platform OS (Operating System) functionality
UNIQUE_IFACE struct IPlatformOS
{
	enum
	{
		Unknown_User = ~0,
		Invalid_User = ~0
	};

	enum
	{
#if defined(XENON)
		USER_MAX_NAME = XUSER_NAME_SIZE,
#elif defined(PS3)
		USER_MAX_NAME = CELL_SYSUTIL_SYSTEMPARAM_CURRENT_USERNAME_SIZE,
#elif defined(WIN32) || defined(WIN64)
		USER_MAX_NAME = 256+1,//UNLEN+1,
#else			
		USER_MAX_NAME = 256,
#endif
	};

	typedef CryFixedStringT<USER_MAX_NAME> TUserName;

	enum EFileOperationCode
	{
		eFOC_Success,							// no error
		eFOC_Failure,							// generic error
		eFOC_ErrorOpenRead,				// unable to open a file to read
		eFOC_ErrorOpenWrite,			// unable to open a file to write
		eFOC_ErrorRead,						// error reading from a file
		eFOC_ErrorWrite,					// error writing to a file
	};

	enum EMsgBoxResult
	{
		eMsgBox_OK,
		eMsgBox_Cancel,
		eMsgBoxNumButtons
	};

	// each entry fills in the relevant member of SStatAnyValue
	enum EUserProfilePreference
	{
		EUPP_CONTROLLER_INVERT_Y,		// eVT_Int (0,1)
		EUPP_CONTROLLER_SENSITIVITY,	// eVT_Float (0.0f..2.0f)
		EUPP_GAME_DIFFICULTY,			// eVT_Int (0,1,2)=(easy,medium,hard)
	};

	struct SUserProfileVariant
	{
		enum EVariantType
		{
			eVT_Invalid,
			eVT_Bool,
			eVT_Int,
			eVT_Float,
		};

		SUserProfileVariant() : m_type(eVT_Invalid), m_iValue(0) {}
		SUserProfileVariant(bool bBool) : m_type(eVT_Bool), m_bValue(bBool) {}
		SUserProfileVariant(int iValue) : m_type(eVT_Int), m_iValue(iValue) {}
		SUserProfileVariant(float fValue) : m_type(eVT_Float), m_fValue(fValue) {}

		EVariantType GetType() const { return m_type; }

		bool GetBool() const { assert(GetType() == eVT_Bool); return m_bValue; }
		int GetInt() const { assert(GetType() == eVT_Int); return m_iValue; }
		float GetFloat() const { assert(GetType() == eVT_Float); return m_fValue; }

	private:
		EVariantType m_type;

		union
		{
			bool	m_bValue;
			int		m_iValue;
			float	m_fValue;
		};
	};

	// Internal event handle passed to observers from listeners. Derive platform specific events from this
	struct SPlatformEvent 
	{
		SPlatformEvent(unsigned int user) // force user index on construction
			: m_user(user) {}

		unsigned int m_user;

		enum EEventType
		{
			eET_PlatformSpecific,

			// Platform agnostic events
			eET_SignIn,								// uses m_signIn - allows a listener to know a user has been signed in or out
			eET_StorageMounted,				// uses m_storageMounted - allows a listener to know a new save location is mounted for profiles etc.
			eET_FileError,						// uses m_fileError fired when a file error occurs (see CSaveWriter_Xenon::AppendBytes for example)
		} m_eEventType;

		union UEventParams
		{
			struct SSignIn
			{
				bool m_signedIn;				// sign in or sign out
			} m_signIn;

			struct SStorageMounted
			{
			} m_storageMounted;

			struct SFileError
			{
				EFileOperationCode m_errorType;
			} m_fileError;
		} m_uParams;
	};

	// Platform specific events now, these have to be here for access by the rest of the engine
#ifdef XENON

	struct SPlatformEventXenon : public SPlatformEvent
	{
		SPlatformEventXenon(unsigned int user)
			: SPlatformEvent(user) // required by SPlatformEvent
			, m_id(0)
			, m_param(0) {}
		DWORD		m_id;
		ULONG_PTR	m_param;
	};

#endif

#ifdef PS3

	struct SPlatformEventPS3 : public IPlatformOS::SPlatformEvent
	{
		SPlatformEventPS3(unsigned int user)
			: IPlatformOS::SPlatformEvent(user)
			, m_status(0)
			, m_param(0)
			, m_userdata(0) {}

		uint64_t	m_status;
		uint64_t	m_param;
		void*		m_userdata;
	};

#endif

	struct ISaveReader
	{
		enum ESeekMode
		{
			ESM_BEGIN,
			ESM_CURRENT,
			ESM_END,
		};

		template<class T>
		EFileOperationCode ReadItem(T& data)
		{
			return ReadBytes(reinterpret_cast<void*>(&data), sizeof(T));
		}

		virtual EFileOperationCode Seek(long seek, ESeekMode mode) = 0;
		virtual EFileOperationCode GetFileCursor(long& fileCursor) = 0;
		virtual EFileOperationCode ReadBytes(void* data, size_t numBytes) = 0;
		virtual EFileOperationCode GetNumBytes(size_t& numBytes) = 0;
		virtual EFileOperationCode Close() = 0;
		virtual IPlatformOS::EFileOperationCode LastError() const = 0;
	};
	typedef cryshared_ptr<ISaveReader> ISaveReaderPtr;

	struct ISaveWriter
	{
		template<class T>
		IPlatformOS::EFileOperationCode AppendItem(const T& data)
		{
			return AppendBytes(reinterpret_cast<const void*>(&data), sizeof(T));
		}

		template<class T>
		IPlatformOS::EFileOperationCode AppendItems(T *data, size_t elems)
		{
			return AppendBytes(static_cast<const void*>(data), sizeof(T) * elems);
		}

		virtual IPlatformOS::EFileOperationCode AppendBytes(const void* data, size_t length) = 0;
		virtual EFileOperationCode Close() = 0;
		virtual IPlatformOS::EFileOperationCode LastError() const = 0;
	};
	typedef cryshared_ptr<ISaveWriter> ISaveWriterPtr;

	struct IFileFinder
	{
		enum EFileState
		{
			eFS_NotExist,
			eFS_File,
			eFS_Directory,
			eFS_FileOrDirectory = -1, // for Win32 based bool return value we use this - see CFileFinderCryPak::FileExists
		};

		// FileExists:
		//   Determine if a file or path exists
		virtual EFileState FileExists(unsigned int user, const char* path) = 0;

		// FindFirst:
		//   Find the first file matching a specified pattern
		//   Returns: find handle to be used with FindNext() and FindClose()
		virtual intptr_t FindFirst(unsigned int user, const char* filePattern, _finddata_t* fd) = 0;

		// FindClose:
		//   Close and release resources associated with FindFirst/FindNext operations
		virtual int FindClose(intptr_t handle) = 0;

		// FindNext:
		//   Find the next file matching a specified pattern
		virtual int FindNext(intptr_t handle, _finddata_t* fd) = 0;
	};
	typedef cryshared_ptr<IFileFinder> IFileFinderPtr;


	// Call once to create and initialize. Use delete to destroy.
	static IPlatformOS* Create();
	virtual ~IPlatformOS() {}

	// Tick with each frame to determine if there are any system messages requiring handling
	virtual void Tick(float realFrameTime) = 0;

	// Local user profile functions to check/initiate user sign in:

	// UserGetMaximumSignedInUsers:
	//   Returns the maximum number of signed in users. EG: On Xbox is 4.
	virtual unsigned int 	UserGetMaximumSignedInUsers() const = 0;

	// UserIsSignedIn:
	//   Returns true if the user is signed in to the OS.
	virtual bool 			UserIsSignedIn(unsigned int userIndex) const = 0;

	// UserIsSignedIn:
	//   Returns true if a named user is signed in to the OS.
	virtual bool 			UserIsSignedIn(const IPlatformOS::TUserName& userName, unsigned int& outUserIndex) const = 0;

	// UserDoSignIn:
	//   Initiate signin dialog box in the OS. Returns true on success.
	virtual bool			UserDoSignIn(unsigned int numUsersRequested = 1) = 0;

	// UserGetPlayerIndex:
	//   Returns the player index of a signed in user or Unknown_User if the user is not signed in.
	virtual unsigned int UserGetPlayerIndex(const char* userName) const = 0;

	// UserGetName:
	//   Get the name of a user. Returns true on success.
	virtual bool 			UserGetName(unsigned int userIndex, IPlatformOS::TUserName& outName) const = 0;

	// UserSelectStorageDevice:
	//   Get the user to select a storage device location for save data.
	virtual bool			UserSelectStorageDevice(unsigned int userIndex) = 0;

	// GetUserProfilePreference:
	//   Get a specific preference for the signed in user
	virtual bool			GetUserProfilePreference(unsigned int user, EUserProfilePreference ePreference, SUserProfileVariant& outResult) const = 0;

	// Use the TCR/TRC compliant platform save/load API for save game data
	virtual bool			UsePlatformSavingAPI() const = 0;

	//[AlexMcC|12.02.10]: PS3 devirtualization needs ISaveReaderPtr (and writer) to be fully qualified)
	// SaveGetReader:
	//   Get a reader object to read from a save file. The file is automatically opened and closed.
	virtual IPlatformOS::ISaveReaderPtr SaveGetReader(const char* fileName, unsigned int user = IPlatformOS::Unknown_User) = 0;

	// SaveGetReader:
	//   Get a writer object to write to a save file. The file is automatically opened and closed.
	virtual IPlatformOS::ISaveWriterPtr SaveGetWriter(const char* fileName, unsigned int user = IPlatformOS::Unknown_User) = 0;

	// GetFindFile:
	//   Get a IFindFile interface for searching for files.
	virtual IPlatformOS::IFileFinderPtr GetFileFinder() = 0;

	// GetFirstSignedInUser:
	//   Returns the user ID of the first signed in user, or Unknown_User if no one is signed in.
	int						GetFirstSignedInUser() const;

	//////////////////////////////////////////////////////////////////////////
	//   Platform listener
	// 

	// Derive listeners from this interface and cast event to the appropriate platform event interface
	struct IPlatformListener
	{
		virtual void OnPlatformEvent(const IPlatformOS::SPlatformEvent& event) = 0;
	};

	virtual void AddListener(IPlatformListener* pListener, const char* szName) = 0;
	virtual void RemoveListener(IPlatformListener* pListener) = 0;
	virtual void NotifyListeners(SPlatformEvent& event) = 0;

	// DebugMessageBox:
	//   Displays an OS dialog box for debugging messages.
	//   A modal (blocking) dialog box with OK and Cancel options.
	// Arguments:
	//   body   -  text body of the message
	//   title	-  title text of the message
	//   flags  -  reserved for future use
	virtual IPlatformOS::EMsgBoxResult DebugMessageBox(const char* body, const char* title, unsigned int flags=0) const = 0;
	
	CSysCallThread* GetSysCallThread() const;

protected:
#if defined(PS3)
	CSysCallThread* m_pSysThread;
#endif

		class	 CFileFinderCryPak : public IFileFinder
		{
		public:
			VIRTUAL IFileFinder::EFileState FileExists(unsigned int user, const char* path) { return gEnv->pCryPak->IsFileExist(path) ? eFS_FileOrDirectory : eFS_NotExist; }
			VIRTUAL intptr_t FindFirst(unsigned int user, const char* filePattern, _finddata_t* fd) { return gEnv->pCryPak->FindFirst(filePattern, fd); }
			VIRTUAL int FindClose(intptr_t handle) { return gEnv->pCryPak->FindClose(handle); }
			VIRTUAL int FindNext(intptr_t handle, _finddata_t* fd) { return gEnv->pCryPak->FindNext(handle, fd); }
		};
};

ILINE int IPlatformOS::GetFirstSignedInUser() const
{
	unsigned int maximumSignedInUsers = UserGetMaximumSignedInUsers();
	for (unsigned int user = 0; user < maximumSignedInUsers; ++user)
	{
		if (UserIsSignedIn(user))
		{
			return static_cast<int>(user);
		}
	}

	return Unknown_User;
}

ILINE CSysCallThread* IPlatformOS::GetSysCallThread() const
{
#if defined(PS3)
	return m_pSysThread;
#else
	return NULL;
#endif	
}

#endif

