//////////////////////////////////////////////////////////////////////
//
//  Crytek (C) 2001
//
//  CrySound Source Code
//
//  File: ISound.h
//  Description: Sound interface.
// 
//  History:
//  - August 28, 2001: Created by Marco Corbetta
//  - Januar 10, 2005: Taken over by Tomas Neumann
//
//////////////////////////////////////////////////////////////////////
#include DEVIRTUALIZE_HEADER_FIX(ISound.h)

#ifndef CRYSOUND_ISOUND_H
#define CRYSOUND_ISOUND_H

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "Cry_Math.h"
#include "TimeValue.h"
#include "SerializeFwd.h"

//forward declarations

//
class		CCamera;
struct	IMusicSystem;
struct  IAudioDevice;
class		ICrySizer;
struct	IVisArea;
struct	ptParam;
struct	ISoundMoodManager;
struct	IReverbManager;
struct	ISound;
struct	ISoundBuffer;
struct	IMicrophone;
struct	INetworkSoundListener;
struct	IMicrophoneStream;
struct	IEntitySoundProxy;
struct  ISound_Extended;
struct  ISound_Deprecated;
struct	ISoundSystem_Extended;
struct	ISoundSystem_Deprecated;
struct	ISoundSystem_WorkInProgress;

struct IVisArea;

// Note:
//	 Need this explicit here to prevent circular includes to IEntity.
// Summary:
//	 Unique identifier for each entity instance.
typedef unsigned int EntityId;	


// Unique ID of a sound.
typedef uint32	tSoundID;
#define INVALID_SOUNDID 0

#define DECIBELTODISTANCE(_DB)	float(cry_powf(10,(_DB-11)/30.0f));

#define SOUND_MAX_SKIPPED_ENTITIES	5	// Maximum number of skipped entities for obstruction calculation. (Defines cache size)
#define SOUND_MAX_OBSTRUCTION_TESTS 10

#define SOUND_INDOOR_TAILNAME "indoor"
#define SOUND_OUTDOOR_TAILNAME "outdoor"

//////////////////////////////////////////////////////////////////////
#define MAX_SFX													1024

//////////////////////////////////////////////////////////////////////
// Valid Event Sound Flags (only use these on a event)
//////////////////////////////////////////////////////////////////////
//#define FLAG_SOUND_RELATIVE							BIT(6)	// Sound position moves relative to player.
//#define FLAG_SOUND_OUTDOOR							BIT(11)	// Play the sound only if the listener is in outdoor.
//#define FLAG_SOUND_INDOOR	 							BIT(12)	// Play the sound only if the listener is in indoor.
//#define FLAG_SOUND_CULLING		 					BIT(14)	// The sound uses sound occlusion (based on VisAreas).
//#define FLAG_SOUND_LOAD_SYNCHRONOUSLY					BIT(15)	// The loading of this sound will be synchronous (asynchronously by default).
//#define FLAG_SOUND_OBSTRUCTION						BIT(18)	// The sound uses sound obstruction (based on ray-world-intersects).
//#define FLAG_SOUND_SELFMOVING							BIT(19)	// Sounds will be automatically moved controlled by direction vector in m/sec.
//#define FLAG_SOUND_START_PAUSED						BIT(20)	// Start the sound paused, so an additional call to unpause is needed.
//#define FLAG_SOUND_VOICE      						BIT(21)	// Sound used as a voice (sub-titles and lip sync can be applied).

//////////////////////////////////////////////////////////////////////
// Valid Wave Sound Flags
//////////////////////////////////////////////////////////////////////
//#define FLAG_SOUND_LOOP								BIT(0)	 
//#define FLAG_SOUND_2D									BIT(1)	
//#define FLAG_SOUND_3D									BIT(2)	 
//#define FLAG_SOUND_STEREO								BIT(3)	 
//#define FLAG_SOUND_16BITS								BIT(4)	 
//#define FLAG_SOUND_STREAM								BIT(5)	// Streamed wav.
//#define FLAG_SOUND_RELATIVE							BIT(6)	// Sound position moves relative to player.
//#define FLAG_SOUND_DOPPLER							BIT(8)	// Use doppler effect for this sound.	
//#define FLAG_SOUND_NO_SW_ATTENUATION					BIT(9)	// Doesn't use SW attenuation for this sound.
//#define FLAG_SOUND_MUSIC								BIT(10)	// Pure music sound, to use to set pure music volume.
//#define FLAG_SOUND_OUTDOOR							BIT(11)	// Play the sound only if the listener is in outdoor.
//#define FLAG_SOUND_INDOOR	 							BIT(12)	// Play the sound only if the listener is in indoor.
//#define FLAG_SOUND_UNSCALABLE							BIT(13)	// For all sounds with this flag the volume can be scaled separately respect to the master volume.
//#define FLAG_SOUND_CULLING		 					BIT(14)	// The sound uses sound occlusion (based on VisAreas).
//#define FLAG_SOUND_LOAD_SYNCHRONOUSLY					BIT(15)	// The loading of this sound will be synchronous (asynchronously by default).
//#define FLAG_SOUND_MANAGED							BIT(16) // Managed sounds life time is controlled by the sound system, when sound stops it will be deleted.
//#define FLAG_SOUND_EDITOR								BIT(17)	// Mark sound as being only used within the Editor (eg. Facial Editor).
//#define FLAG_SOUND_OBSTRUCTION						BIT(18)	// The sound uses sound obstruction (based on ray-world-intersects).
//#define FLAG_SOUND_SELFMOVING							BIT(19)	// Sounds will be automatically moved controlled by direction vector in m/sec.
//#define FLAG_SOUND_START_PAUSED						BIT(20) // Start the sound paused, so an additional call to unpause is needed.
//#define FLAG_SOUND_VOICE      						BIT(21) // Sound used as a voice (sub-titles and lip sync can be applied).


//////////////////////////////////////////////////////////////////////
// Internal Sound Flags
//////////////////////////////////////////////////////////////////////

#define FLAG_SOUND_LOOP									BIT(0)	 
#define FLAG_SOUND_2D										BIT(1)	
#define FLAG_SOUND_3D										BIT(2)	 
#define FLAG_SOUND_STEREO								BIT(3)	 
#define FLAG_SOUND_16BITS								BIT(4)	 
#define FLAG_SOUND_STREAM								BIT(5)	// Streamed wav.
#define FLAG_SOUND_RELATIVE							BIT(6)	// Sound position moves relative to player.
#define FLAG_SOUND_RADIUS								BIT(7)	// Sound has a radius, custom attenuation calculation.
#define FLAG_SOUND_DOPPLER							BIT(8)	// Use doppler effect for this sound.
#define FLAG_SOUND_NO_SW_ATTENUATION		BIT(9)	// Doesn't use SW attenuation for this sound.
#define FLAG_SOUND_MUSIC								BIT(10)	// Pure music sound, to use to set pure music volume.
#define FLAG_SOUND_OUTDOOR							BIT(11)	// Play the sound only if the listener is in outdoor.
#define FLAG_SOUND_INDOOR								BIT(12)	// Play the sound only if the listener is in indoor.
#define FLAG_SOUND_UNSCALABLE						BIT(13)	// For all sounds with this flag the volume can be scaled separately respect to the master volume.
#define FLAG_SOUND_CULLING							BIT(14)	// The sound uses sound occlusion (based on VisAreas).
#define FLAG_SOUND_LOAD_SYNCHRONOUSLY		BIT(15)	// The loading of this sound will be synchronous (asynchronously by default).
#define FLAG_SOUND_MANAGED							BIT(16) // Managed sounds life time is controlled by the sound system, when sound stops it will be deleted.
#define FLAG_SOUND_EDITOR								BIT(17)	// Mark sound as being only used within the Editor (eg. Facial Editor).
#define FLAG_SOUND_OBSTRUCTION					BIT(18)	// The sound uses sound obstruction (based on ray-world-intersects).
#define FLAG_SOUND_SELFMOVING						BIT(19)	// Sounds will be automatically moved controlled by direction vector in m/sec.
#define FLAG_SOUND_START_PAUSED					BIT(20) // Start the sound paused, so an additional call to unpause is needed.
#define FLAG_SOUND_VOICE								BIT(21) // Sound used as a voice (sub-titles and lip sync can be applied).
#define FLAG_SOUND_EVENT								BIT(22) // This sound is a sound event.
#define FLAG_SOUND_PARAM_DOPPLER_ABS		BIT(23)	// This sound has a doppler parameter (relative to world).
#define FLAG_SOUND_PARAM_DOPPLER_REL		BIT(24)	// This sound has a doppler parameter (relative to listener).
#define FLAG_SOUND_PARAM_SPREAD					BIT(25) // This sound has a spread parameter.
#define FLAG_SOUND_PARAM_SQUELCH				BIT(26) // This sound has a radio squelch parameter.
#define FLAG_SOUND_PARAM_ENVIRONMENT		BIT(27) // This sound has a environment parameter.
#define FLAG_SOUND_PARAM_FRONTLEFTREAR	BIT(28) // This sound has a front_left_rear FLR_degree parameter, describing rotation around z.
#define FLAG_SOUND_PARAM_TOPREARDOWN		BIT(29) // This sound has a top_read_down TRD_degree parameter, describing rotation around x.
#define FLAG_SOUND_PARAM_TOPLEFTDOWN		BIT(30) // This sound has a top_left_down TLD_degree parameter, describing rotation around y.
#define FLAG_SOUND_MOVIE								BIT(31) // this is a movie sound


//#define FLAG_SOUND_DEFAULT_3D (FLAG_SOUND_3D | FLAG_SOUND_RADIUS | FLAG_SOUND_CULLING | FLAG_SOUND_OBSTRUCTION )

//
#define FLAG_SOUND_DEFAULT_3D ( FLAG_SOUND_CULLING | FLAG_SOUND_OBSTRUCTION )

#define FLAG_SOUND_ACTIVELIST	 	(FLAG_SOUND_RADIUS | FLAG_SOUND_CULLING | FLAG_SOUND_INDOOR | FLAG_SOUND_OUTDOOR)
#define SOUNDBUFFER_FLAG_MASK	(FLAG_SOUND_LOOP | FLAG_SOUND_2D | FLAG_SOUND_3D | FLAG_SOUND_STEREO | FLAG_SOUND_16BITS | FLAG_SOUND_STREAM | FLAG_SOUND_LOAD_SYNCHRONOUSLY)	// Flags affecting the sound-buffer, not its instance.


//////////////////////////////////////////////////////////////////////
// Precache Flags

#define FLAG_SOUND_PRECACHE_LOAD_SOUND				BIT(0)	 
#define FLAG_SOUND_PRECACHE_LOAD_GROUP				BIT(1)	 
#define FLAG_SOUND_PRECACHE_LOAD_PROJECT			BIT(2)
#define FLAG_SOUND_PRECACHE_STAY_IN_MEMORY		BIT(3)	
#define FLAG_SOUND_PRECACHE_UNLOAD_AFTER_PLAY	BIT(4)	
#define FLAG_SOUND_PRECACHE_UNLOAD_NOW				BIT(5)	


#define FLAG_SOUND_PRECACHE_EVENT_DEFAULT (FLAG_SOUND_PRECACHE_LOAD_SOUND)
#define FLAG_SOUND_PRECACHE_DIALOG_DEFAULT (FLAG_SOUND_PRECACHE_LOAD_SOUND | FLAG_SOUND_PRECACHE_UNLOAD_AFTER_PLAY)
#define FLAG_SOUND_PRECACHE_READABILITY_DEFAULT (FLAG_SOUND_PRECACHE_LOAD_SOUND)


//////////////////////////////////////////////////////////////////////
// Group stuff.


#define MAX_SOUNDSCALE_GROUPS		8
#define SOUNDSCALE_MASTER				0
#define SOUNDSCALE_SCALEABLE		1
#define SOUNDSCALE_DEAFNESS			2
#define SOUNDSCALE_UNDERWATER		3
#define SOUNDSCALE_MISSIONHINT	4


// Update rates in MS = 1000/Hz

#define UPDATE_SOUND_ASSET_MANAGER_IN_MS	250
#define UPDATE_SOUND_REVERB_MANAGER_IN_MS	250
#define UPDATE_AUDIO_DEVICE_IN_MS					10
#define UPDATE_SOUNDS_IN_MS								15
#define UPDATE_SOUND_AUDIODEVICE_IN_MS		15

#define SONIC_SPEED_METER_PER_SEC	343.0f

enum ESoundSystemErrorCode
{
	eSoundSystemErrorCode_None                    = 0x00000000,
	eSoundSystemErrorCode_ProjectNotReady         = BIT(0),
	eSoundSystemErrorCode_SoundCreateFailed       = BIT(1),
	eSoundSystemErrorCode_SoundIDRangeFull        = BIT(2),
	eSoundSystemErrorCode_SoundGotCulled          = BIT(3),
	eSoundSystemErrorCode_SoundNotKnown           = BIT(4),
	eSoundSystemErrorCode_SoundNotRegistered      = BIT(5),
	eSoundSystemErrorCode_SoundFlagsBad           = BIT(6),
	eSoundSystemErrorCode_NotCalledFromMainThread = BIT(7),
	eSoundSystemErrorCode_Max                     = BIT(31)
};

// Summary:
//	 Sound Semantics flags.
enum ESoundSemantic
{
	eSoundSemantic_None											= 0x00000000,
	eSoundSemantic_OnlyVoice								= BIT(0),
	eSoundSemantic_NoVoice									= BIT(1),
	eSoundSemantic_Sandbox									= BIT(2),
	eSoundSemantic_Unused1									= BIT(3),

	eSoundSemantic_Unused2									= BIT(4),
	eSoundSemantic_Unused3									= BIT(5),
	eSoundSemantic_Ambience									= BIT(6), //a
	eSoundSemantic_Ambience_OneShot					= BIT(7), //b

	eSoundSemantic_Physics_Collision        = BIT(8), //c
	eSoundSemantic_Dialog                   = BIT(9), //d
	eSoundSemantic_MP_Chat                  = BIT(10), //e
	eSoundSemantic_Physics_Footstep         = BIT(11), //f

	eSoundSemantic_Physics_General					= BIT(12), //g
	eSoundSemantic_HUD											= BIT(13), //h
	eSoundSemantic_Unused4									= BIT(14), //i
	eSoundSemantic_FlowGraph								= BIT(15), //j

	eSoundSemantic_Player_Foley_Voice       = BIT(16), //k
	eSoundSemantic_Living_Entity            = BIT(17), //l
	eSoundSemantic_Mechanic_Entity          = BIT(18), //m
	eSoundSemantic_NanoSuit                 = BIT(19), //n

	eSoundSemantic_SoundSpot                = BIT(20), //o
	eSoundSemantic_Particle                 = BIT(21), //p
	eSoundSemantic_AI_Pain_Death            = BIT(22), //q
	eSoundSemantic_AI_Readability           = BIT(23), //r

	eSoundSemantic_AI_Readability_Response  = BIT(24), //s
	eSoundSemantic_TrackView                = BIT(25), //t
	eSoundSemantic_Projectile               = BIT(26), //u
	eSoundSemantic_Vehicle                  = BIT(27), //v

	eSoundSemantic_Weapon                   = BIT(28), //w
	eSoundSemantic_Explosion                = BIT(29), //x
	eSoundSemantic_Player_Foley             = BIT(30), //y
	eSoundSemantic_Animation                = BIT(31) //z
};

enum ESoundActiveState
{
	eSoundState_None,
	eSoundState_Active,
	eSoundState_Ending,
	eSoundState_Inactive,
	eSoundState_Stopped,
	eSoundState_ProgrammerSoundReleased,
};

enum EFadeState
{
	eFadeState_None,
	eFadeState_FadingIn,
	eFadeState_FadingOut,
	eFadeState_JustFinished,
};

enum ESoundStopMode
{
	ESoundStopMode_AtOnce,
	ESoundStopMode_EventFade,
	// Note:
	//	 Fallback: will be stopped after 0.1 sec or set spSYNCTIMEOUTINSEC before.
	ESoundStopMode_OnSyncPoint, 
};

enum ESoundUpdateMode
{
	eSoundUpdateMode_None				= 0x00000000,
	eSoundUpdateMode_Listeners	= BIT(0),
	eSoundUpdateMode_Sounds			= BIT(1),
	eSoundUpdateMode_Rest				= BIT(2),
	eSoundUpdateMode_LostFocus	= BIT(3),
	eSoundUpdateMode_All				= 0x0000000F
};

enum EOutputHandle
{
	eOUTPUT_WINMM,			// Pointer to type HWAVEOUT is returned.
	eOUTPUT_DSOUND,			// Pointer to type DIRECTSOUND is returned.
	eOUTPUT_WASAPI,			// Pointer to type WASAPI is returned.
	eOUTPUT_OPENAL,			// Pointer to type OPENAL is returned.
	eOUTPUT_ASIO,				// NULL / 0 is returned.
	eOUTPUT_OSS,				// File handle is returned, (cast to int).
	eOUTPUT_ESD,				// Handle of type int is returned, as returned by so_esd_open_sound (cast to int). 
	eOUTPUT_ALSA,				// Pointer to type snd_pcm_t is returned.
	eOUTPUT_MAC,				// Handle of type SndChannelPtr is returned.
	eOUTPUT_Xbox,				// Pointer to type DIRECTSOUND is returned.
	eOUTPUT_Xbox360,		// Pointer to type IXAudio2 is returned.
	eOUTPUT_PS2,				// NULL / 0 is returned.
	eOUTPUT_PS3,				// NULL / 0 is returned.
	eOUTPUT_PSP,				// NULL / 0 is returned.
	eOUTPUT_GC,					// NULL / 0 is returned.
	eOUTPUT_WII,				// NULL / 0 is returned.
	eOUTPUT_NOSOUND,		// NULL / 0 is returned.
	eOUTPUT_WAVWRITER,	// NULL / 0 is returned.
	eOUTPUT_MAX,
};

enum EPrecacheResult
{
	ePrecacheResult_None,
	ePrecacheResult_OK,
	ePrecacheResult_Disabled,
	ePrecacheResult_ProjectNotReady,
	ePrecacheResult_Error,
	ePrecacheResult_Delayed,
	ePrecacheResult_Max,
};

enum ESoundLogType
{
	eSLT_Message,
	eSLT_Warning,
	eSLT_Error,
	eSLT_Always,
};


typedef struct 
{
	float fCurrentVolumeRatio;
	float fTargetVolumeRatio;

	float fCurrentObstructionRatio;
	float fTargetObstructionRatio;
	
	float fCurrentReverbFeed;
	float fTargetReverbFeed;
	
	float fTimeleft;
	bool  bHasChanged;
} SSoundGroupProperties;

typedef int32 ListenerID;

#define LISTENERID_STANDARD 0
#define LISTENERID_ALLACTIVE -1
#define LISTENERID_INVALID -2

#define MAX_VIS_AREAS 256 // Maximum of visarea cache.


typedef struct IListener 
{
	virtual ListenerID	GetID() const = 0;
	virtual EntityId		GetEntityID() const = 0;
	
	virtual bool GetActive() const = 0;
	virtual void SetActive(bool bActive) = 0;
	
	virtual void SetRecordLevel(float fRecord) = 0;
	virtual float GetRecordLevel() = 0;

	virtual Vec3 GetPosition() const = 0;
	virtual void SetPosition(const Vec3 Position) = 0;

	virtual Vec3 GetForward() const = 0;
	virtual Vec3 GetTop() const = 0;
	virtual Vec3 GetVelocity() const = 0;
	virtual void SetVelocity(Vec3 vVel) = 0;

	virtual void SetMatrix(const Matrix34 newTransformation) = 0;
	virtual Matrix34 GetMatrix() const = 0;

	virtual float GetUnderwater() const = 0;
	virtual void	SetUnderwater(const float fUnder) = 0;

	virtual IVisArea* GetVisArea() const = 0;
	virtual void SetVisArea(IVisArea* pVArea) = 0;

} IListener;


// Description:
//	 These values are used with CS_FX_Enable to enable DirectX 8 FX for a channel.
enum SOUND_FX_MODES
{
    S_FX_CHORUS,
    S_FX_COMPRESSOR,
    S_FX_DISTORTION,
    S_FX_ECHO,
    S_FX_FLANGER,
    S_FX_GARGLE,
    S_FX_I3DL2REVERB,
    S_FX_PARAMEQ,
    S_FX_WAVES_REVERB
};


// Description:
//	 Sound events sent to callback that can registered to every sound.
enum ESoundCallbackEvent
{
	SOUND_EVENT_ON_LOADED,					// Fired when sound is loaded.
	SOUND_EVENT_ON_LOAD_FAILED,				// Fired if sound loading is failed.
	SOUND_EVENT_ON_START,					// Fired when sound is started.
	SOUND_EVENT_ON_PLAYBACK_STARTED,		// Fired when sound's playback started.
	SOUND_EVENT_ON_PLAYBACK_UNPAUSED,		// Fired when sound's playback unpaused.
	SOUND_EVENT_ON_STOP,					// Fired when sound stops.
	SOUND_EVENT_ON_PLAYBACK_STOPPED,		// Fired when sound's playback stops.
	SOUND_EVENT_ON_PLAYBACK_PAUSED,			// Fired when sound's playback paused.
	SOUND_EVENT_ON_SYNCHPOINT				// Fired when sound reaches a syncpoint.
};

// Description:
//	 Soundsystem events sent to callback that registered to soundsystem.
// Note:
//	 Important the first elements need to be the same as in ESoundCallbackEvent
// See also:
//	 ESoundCallbackEvent
enum ESoundSystemCallbackEvent
{
	SOUNDSYSTEM_EVENT_ON_LOADED,						// Fired when sound is loaded.
	SOUNDSYSTEM_EVENT_ON_LOAD_FAILED,				// Fired if sound loading is failed.
	SOUNDSYSTEM_EVENT_ON_START,							// Fired when sound is started.
	SOUNDSYSTEM_EVENT_ON_PLAYBACK_STARTED,	// Fired when sound's playback started.
	SOUNDSYSTEM_EVENT_ON_PLAYBACK_UNPAUSED,	// Fired when sound's playback unpaused.
	SOUNDSYSTEM_EVENT_ON_STOP,							// Fired when sound stops.
	SOUNDSYSTEM_EVENT_ON_PLAYBACK_STOPPED,	// Fired when sound's playback stops.
	SOUNDSYSTEM_EVENT_ON_PLAYBACK_PAUSED,		// Fired when sound's playback paused.
	SOUNDSYSTEM_EVENT_ON_SYNCHPOINT					// Fired when sound reaches a syncpoint.
};

// Description:
//	 Structure for sound obstruction
//	 a single obstruction test.
typedef struct  
{
	Vec3 vOrigin;
	Vec3 vDirection;
	int  nHits;
	int  nPierceability;
	int  nTestForTest;
	tSoundID SoundID;
	float fDistance;
	bool bDirect;
	bool bResult;
} SObstructionTest;

// Summary:
//	 Obstruction test.
typedef struct SObstruction
{
	public: 

		// Constructor.
		SObstruction() 
		{
			fDirectOcclusion			= 0.0f;
			fDirectOcclusionAccu	= 0.0f;
			fReverbOcclusion			= 0.0f;
			fReverbOcclusionAccu	= 0.0f;
			nRaysShot							= 0;
			bProcessed						= false;
			bAssigned							= false;
			bDelayPlayback				= false;
			bFirstTime						= true;
			bDontAveragePrevious	= true;


			for (int i=0; i< SOUND_MAX_OBSTRUCTION_TESTS; ++i)
			{
				ObstructionTests[i].SoundID = INVALID_SOUNDID;
				ObstructionTests[i].vOrigin = Vec3(0);
				ObstructionTests[i].vDirection = Vec3(0);
				ObstructionTests[i].nHits = 0;
				ObstructionTests[i].nPierceability = 0;
				ObstructionTests[i].nTestForTest = 0;
				ObstructionTests[i].fDistance = 0.0f;
				ObstructionTests[i].bDirect = false;
				ObstructionTests[i].bResult = false;
			}

			nObstructionSkipEnts = 0;
			for (int i=0; i< SOUND_MAX_SKIPPED_ENTITIES; ++i)
				pOnstructionSkipEntIDs[i] = 0;
		}

		float GetDirect() const
		{
			return fDirectOcclusion;
		}

		void SetDirect(const float fObstruct)
		{
			fDirectOcclusion = min(max(fObstruct,0.0f), 1.0f);
		}

		void AddDirect(const float fObstruct)
		{
			fDirectOcclusionAccu += fObstruct;
		}

		float GetAccumulatedDirect() const
		{
			return fDirectOcclusionAccu;
		}

		void ResetDirect() 
		{
			fDirectOcclusion			= 0.0f;
			fDirectOcclusionAccu	= 0.0f;
		}

		float GetReverb() const
		{
			return fReverbOcclusion;
		}

		void SetReverb(const float fObstruct)
		{
			fReverbOcclusion = min(max(fObstruct,0.0f), 1.0f);
		}

		void AddReverb(const float fObstruct)
		{
			fReverbOcclusionAccu += fObstruct;
		}

		float GetAccumulatedReverb() const
		{
			return fReverbOcclusionAccu;
		}

		void ResetReverb() 
		{
			fReverbOcclusion			= 0.0f;
			fReverbOcclusionAccu	= 0.0f;
		}

	// Members

	int nRaysShot;

	SObstructionTest ObstructionTests[SOUND_MAX_OBSTRUCTION_TESTS];

	CTimeValue	LastUpdateTime;

	EntityId pOnstructionSkipEntIDs[SOUND_MAX_SKIPPED_ENTITIES];
	int nObstructionSkipEnts;
	bool bAddObstruction;
	bool bProcessed;
	bool bAssigned;
	bool bDelayPlayback;
	bool bFirstTime;
	bool bDontAveragePrevious;

private:
	float fDirectOcclusion;				// level of obstruction from 0 = none, to 1 = full obstructed
	float fDirectOcclusionAccu;		// accumulates level of obstruction from 0 = none, to 1 = full obstructed
	float fReverbOcclusion;				// level of obstruction from 0 = none, to 1 = full obstructed
	float fReverbOcclusionAccu;		// accumulates level of obstruction from 0 = none, to 1 = full obstructed


} SObstruction;


// Summary:
//	 Listener interface for the sound.
struct ISoundEventListener
{
	// Summary:
	//	 Callback event.
	virtual void OnSoundEvent( ESoundCallbackEvent event,ISound *pSound ) = 0;
};

// Summary:
//	 Listener interface for the soundsystem.
struct ISoundSystemEventListener
{
	// Summary:
	//	 Callback event.
	virtual void OnSoundSystemEvent( ESoundSystemCallbackEvent event,ISound *pSound ) = 0;
};

struct SSoundEventListenerInfo
{
	ISoundEventListener	*pListener;
//#ifdef _DEBUG

	//
	char sWho[64];


//#endif
};

// Note:
//	 Marco's NOTE: this is a redefine of the EAX preset OFF, since it seems
//	 that audigy cards are having problems when the default EAX off preset.
#define MY_CS_PRESET_OFF  {0,	1.0f,	0.00f, -10000, -10000, -10000,   0.1f,  0.1f, 0.1f,  -10000, 0.0f, { 0.0f,0.0f,0.0f }, -10000, 0.0f, { 0.0f,0.0f,0.0f }, 0.0750f, 0.00f, 0.04f, 0.000f, 0.0f, 1000.0f, 20.0f, 0.0f,   0.0f,   0.0f, 0 }


// Summary:
//	 Wavebank interface.
struct IWavebank
{
	struct SWavebankInfo
	{
	public: 
		SWavebankInfo()
		{
			nTimesAccessed			= 0;
			nFileSize						= 0;
			nMemCurrentlyInByte = 0;
			nMemPeakInByte			= 0;
		}
		
		uint32 nTimesAccessed;
		uint32 nFileSize;
		uint32 nMemCurrentlyInByte;
		uint32 nMemPeakInByte;
	};

	virtual const char *GetName() = 0;
	virtual const char* GetPath() = 0;
	virtual void SetPath(const char* sWavebankPath) = 0;
	virtual IWavebank::SWavebankInfo* GetInfo() = 0;
	virtual void AddInfo(SWavebankInfo &WavebankInfo) = 0;
};

UNIQUE_IFACE struct ISoundProfileInfo
{
	struct SSoundProfileInfo
	{
	public: 
		SSoundProfileInfo()
		{
			nTimesPlayed					= 0;
			nTimesPlayedOnChannel	= 0;
			nMemorySize						= 0;
			nPeakSpawn						= 0;
			nChannelsUsed					= 0;
			nSoundIDsUsed					= 0;
			bWavebanksLogged			= false;
		}

		uint32 nTimesPlayed;
		uint32 nTimesPlayedOnChannel;
		uint32 nMemorySize;
		uint32 nPeakSpawn;
		uint32 nChannelsUsed;
		uint32 nSoundIDsUsed;
		bool   bWavebanksLogged;
	};
	virtual const char *GetName() = 0;
	virtual ISoundProfileInfo::SSoundProfileInfo* GetInfo() = 0;
	virtual void AddInfo(SSoundProfileInfo &SoundInfo) = 0;
};

// Summary:
//	 Sound system interface.
struct ISoundSystem
{
	//////////////////////////////////////////////////////////////////////////
	// CORE FUNCTIONALITY
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Has to be called regularly.	
	virtual void Update(ESoundUpdateMode UpdateMode) = 0;

	// Summary:
	//	 Saves and restores the state of the complete SoundSystem
	virtual void Serialize(TSerialize ser) = 0;

	// Summary:
	//	 Centrally outputs log information using an internal Spam filter
	virtual void Log( const ESoundLogType eType, const char *szFormat,... ) = 0;

	// Summary:
	//	 Queries the SoundSystem for the last thrown errors, eg. after getting a NULL pointer from CreateSound
	virtual ESoundSystemErrorCode GetError() = 0;
	
	// Summary:
	//	 Releases the SoundSystem
	virtual void Release() = 0;

	// Summary:
	//	 Registers listener to the sound.
	virtual void AddEventListener( ISoundSystemEventListener *pListener, bool bOnlyVoiceSounds ) = 0;

	// Summary:
	//	 Removes listener to the sound.
	virtual void RemoveEventListener( ISoundSystemEventListener *pListener ) = 0;

	// Summary:
	//	 Locks all sound buffer resources to prevent them from unloading (when restoring checkpoint).
	virtual void LockResources() = 0;
	// Summary:
	//	 Unlocks all sound buffer resources to prevent them from unloading.
	virtual void UnlockResources() = 0;

	// Summary:
	//	 Reset the sound system, unloading all loaded asset
	//   Usually called at unloading of the level
	virtual void Reset() = 0;

	// Summary:
	//   Sets a global parameter value to all events which have a parameter with this name
	virtual void SetGlobalParameter( const char *sGlobalParameter, const float fValue) = 0;

	//////////////////////////////////////////////////////////////////////////
	// SOUND CREATION & HANDLING
	//////////////////////////////////////////////////////////////////////////

	// Description:
	//	 Creates a sound object by combining the definition of the sound and the buffer of sound data.
	// Arguments:
	//	 nFlags - Sound flags.
	// Return Value:
	//	 ISound* is a Ptr to a Sound, this will change to the unique Sound ID later.
	virtual ISound* CreateSound			( const char *sGroupAndSoundName, uint32 nFlags ) = 0;

	// Description:
	//	 Creates a sound object by combining the definition of the sound and the buffer of sound data.
	// Arguments:
	//	 nFlags - Sound flags.
	// Return Value:
	//	 ISound* is a Ptr to a Sound, this will change to the unique Sound ID later.
	virtual ISound* CreateLineSound		( const char *sGroupAndSoundName, uint32 nFlags, const Vec3 &vStart, const Vec3 &VEnd ) = 0;

	// Description:
	//	 Creates a sound object by combining the definition of the sound and the buffer of sound data.
	// Arguments:
	//	 nFlags - Sound flags.
	// Return Value:
	//	 ISound* is a Ptr to a Sound, this will change to the unique Sound ID later.
	virtual ISound* CreateSphereSound	( const char *sGroupAndSoundName, uint32 nFlags, const float fRadius ) = 0;

	// Summary:
	//	 Gets a sound interface from the sound system.
	// Arguments:
	//	 nSoundId - Sound id.
	virtual struct ISound* GetSound(tSoundID nSoundID) const = 0;

	// Description:
	//	 Precaches the definition and the buffer of a group, subgroup or a sound.
	// Arguments:
	//	 sGroupAndSoundName - Name of group (with /path/) and name of sound split by ":".
	//	 nSoundFlags		- some Flags that can be set, needs to be defined yet.
	//	 nPrecacheFlags		- Some Flags that can be set to change precaching behaviour.
	// Return Value:
	//	 EPrecacheResult result code.
	virtual EPrecacheResult Precache( const char *sGroupAndSoundName, uint32 nSoundFlags, uint32 nPrecacheFlags ) = 0;

	//////////////////////////////////////////////////////////////////////////
	// VOLUME CONTROL
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Sets master volume.
	// Arguments:
	//	 fVol - Volume (0.0f - 1.0f)		
	virtual void SetMasterVolume(float fVol) = 0;

	// Summary:
	//	 Sets the volume scale for all sounds with FLAG_SOUND_SCALABLE.
	// Arguments:
	//	 fScale - Volume scale (default 1.0)			
	virtual void SetMasterVolumeScale(float fScale, bool bForceRecalc=false) = 0;

	// Summary:
	//	 Gets SFX volume.
	// Arguments:
	//	 fVol - Volume (0.0f - 1.0f)
	virtual float GetSFXVolume() = 0;

	/*! SetMovieFadeoutVolume
	@param movieFadeoutVolume fadeout volume (0.0f - 1.0f)
	*/
	virtual void SetMovieFadeoutVolume(const float movieFadeoutVolume) = 0;

	/*! GetMovieFadeoutVolume
	@param movieFadeoutVolume fadeout volume (0.0f - 1.0f)
	*/
	virtual float GetMovieFadeoutVolume() const = 0;

	// Description:
	//	 Sets parameters for directional attenuation (for directional microphone effect); set fConeInDegree to 0 to disable the effect.
	virtual void CalcDirectionalAttenuation(const Vec3 &Pos, const Vec3 &Dir, const float fConeInRadians) = 0;

	// Description:
	//	 Returns the maximum sound-enhance-factor to use it in the binoculars as "graphical-equalizer"...
	virtual float GetDirectionalAttenuationMaxScale() = 0;

	// Summary:
	//	 Returns if directional attenuation is used.
	virtual bool UsingDirectionalAttenuation() = 0;

	//////////////////////////////////////////////////////////////////////////
	// PLAYBACK FUNCTIONALITY
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Pauses all sounds.
	virtual void	Pause(bool bPause, bool bResetVolume=false, unsigned int nFadeOutInMS=0) = 0; 
	// Summary:
	//	 Checks if the SoundSystem is paused.
	virtual bool  IsPaused() = 0;

	// Summary:
	//	 Sets Muted/unmuted all sounds.
	virtual void	Mute(bool bMute)=0;

	// Summary:
	//	 Stops all sounds and music.
	virtual bool	Silence(bool bStopLoopingSounds, bool bUnloadData) = 0;

	//////////////////////////////////////////////////////////////////////////
	// Listener
	//////////////////////////////////////////////////////////////////////////

	// Description:
	//	 Set the listener transformation matrix, velocity and record level for a special listener.
	// Arguments:
	//	 nListenerID	- ListenerID of the listener.
	//	 matOrientation - Transformation matrix.
	//	 vVel			- Velocity vector.
	//	 fRecordLevel	- Level of input sensitivity.
	// Return Value:
	//	 Bool value if listener was successfully found and updated.
	virtual bool SetListener(const ListenerID nListenerID, const Matrix34 &matOrientation, const Vec3 &vVel, bool bActive, float fRecordLevel)=0;

	virtual void SetListenerEntity( ListenerID nListenerID, EntityId nEntityID ) = 0;

	// Summary:
	//	 Creates a new listener.
	// Return Value:
	//	 ListenerID of new created Listener or INVALID_LISTENERID on failure.
	virtual ListenerID CreateListener() = 0;

	// Summary:
	//	 Removes a listener.
	// Arguments:
	//	 nListenerID - ListenerID of the listener which should be removed.
	// Return Value:
	//	 Bool if Listener was removed successfully .
	virtual bool RemoveListener(ListenerID nListenerID) = 0;

	// Description:
	//	 Retrieves the ListenerID of the closest listener to a 3D Point.
	// Arguments:
	//	 vPosition - Position in 3D
	// Return Value:
	//	 nListenerID - ListenerID of the active listener which is closest.
	virtual ListenerID	GetClosestActiveListener(Vec3 vPosition) const = 0;

	// Description:
	//	 Retrieves a pointer to a Listener struct of the listener defined by ListenerID.
	// Arguments:
	//	 nListenerID - ListenerID of a valid listener.
	// Return Value:
	//	 IListener* Pointer to listener struct or NULL if not found.
	virtual	IListener*	GetListener(ListenerID nListenerID) = 0;

	// Description:
	//	 Retrieves a pointer to a listener struct of the listener defined by EntityID.
	// Arguments:
	//	 nEntityID - EntityID of a valid Listener.
	// Return Value:
	//	 IListener* Pointer to Listener struct or NULL if not found.
	virtual	IListener*	GetListener(EntityId nEntityID) = 0;

	// Description:
	//	 Retrieves a pointer to a listener struct of the listener who has a higher ID than provided.
	// Arguments:
	//	 nListenerID - ListenerID or LISTENERID_STANDARD to start.
	// Return Value:
	//	 IListener* Pointer to listener struct or NULL if no listener has a higher ID.
	virtual IListener*	GetNextListener(ListenerID nListenerID) = 0;

	// Description:
	//	 Retrieves the number of active listeners.
	// Return Value:
	//	 uint32 Number of active Listeners
	virtual uint32			GetNumActiveListeners() const = 0;

	//////////////////////////////////////////////////////////////////////////
	// VOIP
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Checks for multiple record devices and expose them.
	virtual bool GetRecordDeviceInfo(const int nRecordDevice, char* sName, int nNameLength) = 0;

	// Summary:
	//	 Creates a new microphone.
	virtual IMicrophone* CreateMicrophone(
		const unsigned int nRecordDevice,
		const unsigned int nBitsPerSample, 
		const unsigned int nSamplesPerSecond,
		const unsigned int nBufferSizeInSamples) = 0;

	// Summary:
	//	 Removes a microphone.
	virtual bool RemoveMicrophone( IMicrophone *pMicrophone ) = 0;

	// Description:
	//	 Creates a 3D sound emitting the talk of PlayerName that belongs to PlayerMicro.
	//	 Probably it even gets routed to the headset's speaker, so we need to identify target player.
	virtual ISound* CreateNetworkSound(	INetworkSoundListener *pNetworkSoundListener,
		const unsigned int nBitsPerSample, 
		const unsigned int nSamplesPerSecond,
		const unsigned int nBufferSizeInSamples,
		EntityId PlayerID) = 0;

	// Description:
	//	 Removes this network sound, probably ISound:Release can replace this later.
	virtual void RemoveNetworkSound (ISound *pSound) = 0;

	//////////////////////////////////////////////////////////////////////////
	// Interfaces
	//////////////////////////////////////////////////////////////////////////

	virtual ISoundSystem_Extended* GetInterfaceExtended() = 0;
	virtual ISoundSystem_Deprecated* GetInterfaceDeprecated() = 0;
	virtual ISoundSystem_WorkInProgress* GetInterfaceWorkInProgress() = 0;

	// Summary:
	//	 Returns IAudioDevice Interface.
	virtual IAudioDevice*		GetIAudioDevice()	const = 0;

	// Summary:
	//	 Returns ISoundMoodManager Interface.
	virtual ISoundMoodManager* GetIMoodManager() const = 0;

	// Summary:
	//	 Returns IReverbManager Interface.
	virtual IReverbManager* GetIReverbManager() const = 0;

};

// Extended ISoundSystem Interface for use in Sandbox
// Functions calls in the Extended Interface should only be used after fully understanding the impact
// and the functionality. When using the data-driven SoundSystem most of these calls should not be called
// by the game
struct ISoundSystem_Extended : public ISoundSystem
{
	//////////////////////////////////////////////////////////////////////////
	// SYSTEM CONTROL
	//////////////////////////////////////////////////////////////////////////

	// Description:
	//	 Need to call this after the AudioDevice was set and initialized.
	virtual bool Init() = 0; 

	// Description:
	// Creates a music-system. You should only create one music-system at a time.
	virtual IMusicSystem* CreateMusicSystem() = 0;

	// Summary:
	//	 Added to set and get music effect value for volume.
	virtual void SetMusicEffectsVolume(float v ) = 0;

	// Summary:
	//	 Gets music-volume.
	virtual float GetMusicVolume() const = 0;

	// Summary:
	//	 Sets master pitch.
	// Arguments:
	//	 fPitch - Pitch (in octaves).	
	virtual void SetMasterPitch(float fPitch) = 0;

	// Description:
	//	 Called to be able to overwrite sound files and clear any loaded dependency.
	virtual bool DeactivateAudioDevice() = 0;
	// Description:
	//	 Called to be able to overwrite sound files and clear any loaded dependency.
	virtual bool ActivateAudioDevice() = 0;

	//////////////////////////////////////////////////////////////////////////
	// PROFILING
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 This one is to debug some of FMOD memory allocation.
	virtual void TraceMemoryUsage(int nMemUsage) = 0;

	// Summary:
	//	 Gets memory usage.
	virtual void GetMemoryUsage(class ICrySizer* pSizer) const = 0;

	// Summary:
	//	 Gets memory usage info.
	virtual void	GetSoundMemoryUsageInfo(int *nCurrentMemory,int *nMaxMemory)const = 0;

	// Summary:
	//	 Gets memory usage in MB.
	virtual int  GetMemoryUsageInMB(bool bGetPoolSize = false) = 0;

	// Summary:
	//	 Gets number of voices playing.
	virtual int	GetUsedVoices() const = 0;

	// Summary:
	//	 Gets cpu-usuage.
	virtual float	GetCPUUsage() const = 0;

	// Summary:
	//		Profiling sounds.
	virtual ISoundProfileInfo* GetSoundInfo(int nIndex) = 0;
	virtual ISoundProfileInfo* GetSoundInfo(const char* sSoundName) = 0;
	virtual int GetSoundInfoCount() = 0;
	
	//////////////////////////////////////////////////////////////////////////
	// INFORMATION
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Returns true if sound is being debugged.
	virtual bool DebuggingSound() = 0;	
	virtual bool IsNullImplementation() const = 0;

	//////////////////////////////////////////////////////////////////////////
	// VIDEO PLAYBACK
	//////////////////////////////////////////////////////////////////////////

	// Return Value:
	//	 The output handle if possible, else NULL.
	//	 The MasterVoice handle on Xbox360 if possible, else NULL.
	// Arguments:
	//	 HandleType		- Will specify if this a pointer to DirectX LPDIRECTSOUND or a WINMM handle
	//					  Will be eOUTPUT_MAX if invalid
	virtual void	GetOutputHandle( void **pHandle, void **pHandle2, EOutputHandle *HandleType)	const = 0;


	//////////////////////////////////////////////////////////////////////////
	// FUNCTIONALITY
	//////////////////////////////////////////////////////////////////////////
	
	// Description:
	//	 Sets minimal priority for sounds to be played.
	//	 Sound`s with priority less then that will not be played.
	// Return Value:
	//	 Previous minimal priority..
	virtual int SetMinSoundPriority( int nPriority ) = 0;
	
	// Description:
	//	 To be called when something changes in the environment which could affect
	//	 sound occlusion, for example a door closes etc.
	// Arguments:
	//	 bRecomputeListener - Recomputes the listener vis area.
	//	 bForceRecompute	- Forces to recompute the vis area connections even if
	// 	 the listener didn't move (useful for moving objects that can occlude).
	virtual void	RecomputeSoundOcclusion(bool bRecomputeListener, bool bForceRecompute, bool bReset=false) = 0;

	// Description:
	//	 Converts a dialog sound name reference to a file into a dialog key syntax
	//	 which is compatible with the LocalizationManager
	// Return Value:
	//	 name of the dialog key
	virtual const char* MakeDialogKey(const char* sDialogFileName) = 0;
};

// ISoundSystem Interface extention with functions which will be deprecated
struct ISoundSystem_Deprecated
{
	// Description:
	//	 Sets the scaling factor for a specific scale group (0-7).
	virtual bool SetGroupScale(int nGroup, float fScale) = 0;

	virtual void SetSoundActiveState(ISound *pSound, ESoundActiveState State) = 0;

	// Summary:
	//	 Checks for EAX support.
	virtual bool IsEAX(void) = 0;
};

// ISoundSystem Interface extention with functions which are in development and are not fully supported yet
struct ISoundSystem_WorkInProgress
{

	// Summary:
	//	 Sets the weather condition that affect acoustics.
	virtual bool SetWeatherCondition(float fWeatherTemperatureInCelsius, float fWeatherHumidityAsPercent, float fWeatherInversion) = 0;

	// Summary:
	//	 Gets the weather condition that affect acoustics.
	virtual bool GetWeatherCondition(float &fWeatherTemperatureInCelsius, float &fWeatherHumidityAsPercent, float &fWeatherInversion) = 0;


};

// Summary:
//	 Special asset ParamSemantics.
enum enumSoundParamSemantics
{
	// review and clean this list, check implementation

	spNONE,
	spINITIALIZED,
	spSOUNDID,
	spSOUNDTYPE, // ?
	spSAMPLETYPE, //?
	spFREQUENCY,
	spPITCH,

	spVOLUME,
	spISPLAYING,	
	spPAUSEMODE,
	spLOOPMODE,
	spLENGTHINMS,
	spLENGTHINBYTES,
	spLENGTHINSAMPLES,
	spSAMPLEPOSITION,
	spTIMEPOSITION,
	sp3DPOSITION,
	sp3DVELOCITY,
	spMINRADIUS,
	spMAXRADIUS,
	spPRIORITY,
	spFXEFFECT,
	spAMPLITUDE,
	spSPEAKERPAN,
	spREVERBWET,
	spREVERBDRY,
	spSYNCTIMEOUTINSEC
};

enum eUnloadDataOptions
{
	sbUNLOAD_NONE,
	sbUNLOAD_ALL_DATA,
	sbUNLOAD_ALL_DATA_NOW,
	sbUNLOAD_UNNEEDED_DATA,
	sbUNLOAD_UNNEEDED_DATA_NOW
};

// Summary
// Basic Interface to work with a Sound. All functionality listed here is sufficient for Game Code
struct ISound
{
	virtual void GetMemoryUsage( ICrySizer *pSizer ) const=0;

	//////////////////////////////////////////////////////////////////////////
	// CORE FUNCTIONALITY
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Every Sound has to have a Semantic assigned to it before it can be played back
	virtual void	SetSemantic(const ESoundSemantic eSemantic) = 0;
	virtual const ESoundSemantic	GetSemantic()const = 0;

	// Summary:
	//	 Registers event listener to the sound.
	virtual void AddEventListener( ISoundEventListener *pListener, const char *sWho ) = 0;

	// Summary:
	//	 Remove event listener to the sound.
	virtual void RemoveEventListener( ISoundEventListener *pListener ) = 0;

	// Summary:
	//	 Adds / removes reference counts.
	// Note:
	//	 This is used in the smart_ptr automatically.
	virtual int	AddRef() = 0;
	virtual int	Release() = 0;

	//////////////////////////////////////////////////////////////////////////
	// PLAYBACK CONTROL
	//////////////////////////////////////////////////////////////////////////

	virtual void Play(const float fVolumeScale=1.0f, const bool bForceActiveState=true, const bool bSetRatio=true, IEntitySoundProxy *pEntitySoundProxy=NULL) = 0;
	virtual void Stop(const ESoundStopMode eStopMode=ESoundStopMode_EventFade) = 0; // Sound will be invalid after stopped, unless being static on a soundproxy
	
	virtual void SetPaused(const bool bPaused) = 0;
	virtual const bool GetPaused()const = 0;

	// Summary:
	//	 Fading In/Out - 0.0f is Out 1.0f is In.
	virtual void				SetFade(const float fFadeGoal, const int nFadeTimeInMS) = 0;
	virtual EFadeState	GetFadeState() const = 0;

	//////////////////////////////////////////////////////////////////////////
	// SPATIALIZATION
	//////////////////////////////////////////////////////////////////////////
	
	// Summary:
	//	 Sets sound source position.
	// Note:
	//	 Position gets automatically updated if the sound is played on a SoundProxy
	virtual void	SetPosition(const Vec3 &pos) = 0;

	// Summary:
	//	 Gets sound source position.
	virtual Vec3 GetPosition() const = 0;	
	
	virtual void			SetMatrix(const Matrix34 newTransformation) = 0;
	virtual Matrix34	GetMatrix() const = 0;
	virtual void			SetDirection(const Vec3 &vDir) = 0; // overwrites transformation matrix
	virtual Vec3			GetForward() const = 0;
	
	//////////////////////////////////////////////////////////////////////////
	// MANIPULATION
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Gets parameter defined in the enumAssetParam list.
	virtual bool			GetParam(enumSoundParamSemantics eSemantics, ptParam* pParam) const = 0;
	// Summary:
	//	 Sets parameter defined in the enumAssetParam list.
	virtual bool			SetParam(enumSoundParamSemantics eSemantics, ptParam* pParam) = 0;

	// Summary:
	//	 Gets parameter defined by string and float value, returns the index of that parameter
	virtual int				GetParam(const char *sParameter, float *fValue, bool bOutputWarning=true) const = 0;
	// Summary:
	//	 Sets parameter defined by string and float value, returns the index of that parameter
	virtual int				SetParam(const char *sParameter, float fValue, bool bOutputWarning=true) = 0;

	// Summary:
	//	 Gets parameter defined by index and float value.
	virtual bool			GetParam(int nIndex, float* const pfValue, float* const pfRangeMin=NULL, float* const pfRangeMax=NULL, char const** const ppcParameterName=NULL, bool const bOutputWarning=true) const = 0;
	// Summary:
	// Sets parameter defined by index and float value.
	virtual bool			SetParam(int nIndex, float fValue, bool bOutputWarning=true) = 0;

	// Description:
	//	 Sets a distance multiplier so sound event's distance can be tweak (sadly pretty hack feature).
	virtual void			SetDistanceMultiplier(const float fMultiplier) = 0;

	//////////////////////////////////////////////////////////////////////////
	// INFORMATION
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Gets unique id of sound.
	virtual const tSoundID	GetId()const = 0;

	// Summary:
	//	 Gets name of sound file.
	virtual const char*	GetName() = 0;

	// Summary:
	//	 Gets the Sound Flags to query information about the sound
	virtual uint32	GetFlags() const = 0;

	virtual bool IsPlaying() const = 0;
	virtual bool IsPlayingVirtual() const = 0;

	// Summary:
	//	 Returns true if sound is now in the process of asynchronous loading of sound buffer.
	virtual bool IsLoading()const = 0;
	// Summary:
	//	 Returns true if sound have already loaded sound buffer.
	virtual bool IsLoaded() const = 0;

	// Summary:
	//	 Returns the size in ms.
	// Return Value:
	//	 0 on looping sound or if buffer is not loaded (dialog).
	virtual int GetLengthMs() const = 0; 

	// Summary:
	//	 Returns the size of the stream in bytes.
	virtual int GetLengthInBytes() const = 0; // will soon change to:
	//virtual int GetLengthInBytes()=0;

	// Summary:
	//	 Retrieves the maximum distance of a sound.
	virtual float GetMaxDistance() const = 0;

	virtual bool IsRelative() const = 0;

	//////////////////////////////////////////////////////////////////////////
	// EXTENDED CONTROL
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Modifies a line sound.
	virtual void SetLineSpec(const Vec3 &vStart, const Vec3 &vEnd) = 0;
	virtual bool GetLineSpec(  Vec3 &vStart,   Vec3 &vEnd) = 0;

	// Summary:
	//	 Modifies a sphere sound
	virtual void SetSphereSpec(const float fRadius) = 0;
	
	// Summary:
	//	 Sets to-be-skipped entities for obstruction calculation.
	virtual void SetPhysicsToBeSkipObstruction(EntityId *pSkipEnts,int nSkipEnts) = 0;

	// Summary:
	//	 Unloads the soundbuffer of this sound.
	virtual bool UnloadBuffer(eUnloadDataOptions UnloadOption) = 0;


	//////////////////////////////////////////////////////////////////////////
	// INTERFACES
	//////////////////////////////////////////////////////////////////////////

	virtual ISound_Extended* GetInterfaceExtended() = 0;
	virtual ISound_Deprecated* GetInterfaceDeprecated() = 0;

};



// Extended ISound Interface for use in Sandbox
// Functions calls in the Extended Interface should only be used after fully understanding the impact
// and the functionality. When using the data-driven SoundSystem most of these calls should not be called
// by the game
struct ISound_Extended : public ISound
{
	//////////////////////////////////////////////////////////////////////////
	// PLAYBACK CONTROL
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Sets sound volume. Better control pitch using a parameter
	// Note:
	//	 Range: 0-1
	virtual	void	SetVolume( const float fVolume ) = 0;
	
	// Summary:
	//	 Gets sound volume. 
	virtual	float	GetVolume() const = 0;
	
	// Summary:
	//	 Sets the Sound Flags, SetFlags will overwrite all old Flags!
	virtual void	SetFlags(uint32 nFlags) = 0;

	// Summary:
	//	 Sets sound pitch. Better control pitch using a parameter
	// Note:
	//	 1000 is default pitch.
	virtual void SetPitch(int nPitch) = 0; // TODO internal

	// Summary:
	//	 Overwrites Velocity which is calculated by the SoundSystem based on Sound's movement
	virtual void	SetVelocity(const Vec3 &vel) = 0;
	
	// Summary:
	//	 Gets sound source velocity.
	virtual Vec3	GetVelocity( void ) const = 0;

	// Summary:
	//	 Sets the volume ratio.
	virtual void SetRatio(float fRatio) = 0;
	
	//////////////////////////////////////////////////////////////////////////
	// MANIPULATION
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Set the currently played sample-pos in bytes or milliseconds.
	virtual void SetCurrentSamplePos(unsigned int nPos,bool bMilliSeconds) = 0;

	// Summary:
	//	 Retrieves the currently played sample-pos, in milliseconds or bytes.
	virtual unsigned int GetCurrentSamplePos(bool bMilliSeconds=false) const = 0;

	// Summary:
	//	 Sets sound priority (0-255).
	virtual void	SetSoundPriority(int nSoundPriority) = 0;	
	
	// Summary:
	//	 Defines sound cone.
	// Note:
	//	 Angles are in degrees, in range 0-360.
	virtual void SetConeAngles(const float fInnerAngle,const float fOuterAngle) = 0;

	// Summary:
	//	 Gets sound cone.
	virtual void GetConeAngles(float &fInnerAngle, float &fOuterAngle) = 0;

	virtual bool Preload() = 0;

	// Summary:
	//	 Sets panning values (-1.0 left, 0.0 center, 1.0 right).
	virtual void	SetPan(float fPan) = 0; // TODO legacy
	virtual float GetPan() const = 0; // TODO legacy

	// Description:
	//	 Sets 3d panning values (0.0 no directional, 1.0 full 3d directional effect).
	virtual void	Set3DPan(float f3DPan) = 0; // TODO verify
	virtual float	Get3DPan() const = 0; // TODO verify

	//////////////////////////////////////////////////////////////////////////
	// INFORMATION
	//////////////////////////////////////////////////////////////////////////

	// Summary:
	//	 Gets sound obstruction struct.
	virtual SObstruction* GetObstruction( void ) = 0;

	virtual bool IsInCategory(const char* sCategory) = 0;

};

// ISound Interface extention with functions which will be deprecated
struct ISound_Deprecated
{

	// Summary:
	//	 Sets automatic pitching amount (0-1000).
	virtual void SetPitching(float fPitching) = 0;  // TODO legacy

	// Description:
	//	 Set Minimal/Maximal distances for sound.
	//	 Sound is not attenuated below minimal distance and not eared outside of max distance.
	virtual void SetMinMaxDistance(float fMinDist, float fMaxDist) = 0; // TODO legacy

	// Summary:
	//	 Enable fx effects for this sound.
	// Note:
	//	 Must be called after each play
	virtual	void	FXEnable(int nEffectNumber)=0;

	virtual	void	FXSetParamEQ(float fCenter,float fBandwidth,float fGain)=0;

	// group stuff

	// Summary:
	//	 Adds sound to specific sound-scale-group (0-31).
	virtual void AddToScaleGroup(int nGroup) = 0;
	// Summary:
	//	 Removes sound from specific sound-scale-group (0-31).
	virtual void RemoveFromScaleGroup(int nGroup) = 0;
	// Summary:
	//	 Sets sound-scale-groups by bitfield.
	virtual void SetScaleGroup(unsigned int nGroupBits) = 0;

	// Summary:
	//	 Sets looping mode of sound.
	virtual void SetLoopMode(bool bLoop) = 0;

	// Summary:
	//	 Returns frequency of sound.
	virtual int	 GetFrequency() const = 0; // TODO legacy

};


// Summary:
//	 A network sound listener.
UNIQUE_IFACE struct INetworkSoundListener
{
	// Summary:
	//	 Asks for buffer to be filled with data.
	virtual bool FillDataBuffer(	unsigned int nBitsPerSample,
																unsigned int nSamplesPerSecond,		
																unsigned int nNumSamples, 
																void* pData ) = 0;

	virtual void OnActivate( bool active ) = 0;



	// Description:
	//	 Sets the distance at which sound playback switches between 2d and 3d.
	//	 Closer than max3dDistance: sound plays in 3d
	//	 Further than min2dDistance: sound plays in 2d
	//	 Between the two: linear transition.
	// Note:
	//	 NB: asserts max3dDistance >= min2dDistance
	virtual void SetSoundPlaybackDistances(float max3dDistance, float min2dDistance) = 0;

	// Summary:
	//	 Recalculate the sound's 3d pan based on position.
	virtual void UpdateSound3dPan() = 0;
};

// Summary:
// A single microphone structure:
UNIQUE_IFACE struct IMicrophone
{
	// Summary:
	//	 Starts recording and calling ReadDataBuffer on IMicrophoneStream.
	virtual bool Record(const unsigned int nRecordDevice, const unsigned int nBitsPerSample, 
											const unsigned int nSamplesPerSecond, const unsigned int nBufferSizeInSamples) = 0;

	// Summary:
	//	 Stops recording and calling the callback. Can restart with Record.
	virtual bool Stop() = 0;

	// Summary:
	//	 Regular call to fill internal buffer.
	virtual void Update() = 0;

	// Summary:
	//	 Returns size of data in the internal buffer(valid until next call to Update()).
	virtual int GetDataSize()=0;

	// Summary:
	//	 Returns pointer to internal buffer (valid until next call to Update()).
	virtual int16* GetData()=0;

	// Summary:
	//	 Will tear down this microphone.
	virtual void Release() = 0;

};


//////////////////////////////////////////////////////////////////////////////////////////////
typedef ISoundSystem* (*PFNCREATESOUNDSYSTEM)(struct ISystem*, void*);

#ifdef CRYSOUNDSYSTEM_EXPORTS
	#define CRYSOUND_API DLL_EXPORT
#else
	#define CRYSOUND_API DLL_IMPORT
#endif

extern "C"
{
	CRYSOUND_API ISoundSystem* CreateSoundSystem(struct ISystem*, void *pInitData);
}

//////////////////////////////////////////////////////////////////////////
// interface that passes all parameters via a structures
// this allows having stable interface methods and flexible default arguments system
// see physics code for the original idea, tried to do this a bit nicer
//////////////////////////////////////////////////////////////////////////
// Generic ptParam structure that offers Get/SetValue, which returns bool
// if that query was successful and the type checking was ok
//
// Use this by implementing an abstract interface to get/set parameter
// bool	GetParam(enumParamSemantics eSemantics, ptParam* pParam);
// bool	SetParam(enumParamSemantics eSemantics, ptParam* pParam);
// 
// ptParamINT32 NewParam(pspNumberOfSomething, 42);
// m_pMyImpl->SetParam(&NewParam);
//
// Then internally use the Semantics to tell what the value means
// switch (eSemantics)
//
//	case pspNumberOfSomething:
//		int32 nTemp;
//		if (!(pParam->GetValue(nTemp))) return (false);
//		if (!CS_SetSomethingToFmod(m_nChannel,nTemp)) return (false);
//		break;
//////////////////////////////////////////////////////////////////////////

// Summary:
//	 Some generic parameter types.
enum enumParamType
{
	ptBOOL,
	ptVOIDP,
	ptF32,
	ptINT32,
	ptCRYSTRING,
	ptVEC3F32
};

////////////////////////// Params structures /////////////////////

struct ptParamBOOL;
struct ptParamVOIDP;
struct ptParamF32;
struct ptParamINT32;
struct ptParamCRYSTRING;
struct ptParamVEC3F32;

template<typename T> struct Map2ptParam;
template<> struct Map2ptParam<bool>			{ typedef bool Tkey; typedef ptParamBOOL Tval; enum { TypeId = ptBOOL }; };
template<> struct Map2ptParam<void*>		{ typedef void* Tkey; typedef ptParamVOIDP Tval; enum { TypeId = ptVOIDP }; };
template<> struct Map2ptParam<f32>			{ typedef f32 Tkey; typedef ptParamF32 Tval; enum { TypeId = ptF32 }; };
template<> struct Map2ptParam<int32>		{ typedef int32 Tkey; typedef ptParamINT32 Tval; enum { TypeId = ptINT32 }; };
template<> struct Map2ptParam<string>		{ typedef string Tkey; typedef ptParamCRYSTRING Tval; enum { TypeId = ptCRYSTRING }; };
template<> struct Map2ptParam<Vec3>			{ typedef Vec3 Tkey; typedef ptParamVEC3F32 Tval; enum { TypeId = ptVEC3F32 }; };

// Todo:
//	 Rename.
struct ptParam
{
protected:
	ptParam(const enumParamType eType) : m_eType(eType) {}

	enumParamType m_eType;

public:
	// Summary:
	//	 Template class to the given type with the value and return if type check was ok.
	template<typename T> bool GetValue(T &outVal) const
	{
		if (m_eType == (enumParamType)Map2ptParam<T>::TypeId)
		{
			outVal = ( *static_cast<const typename Map2ptParam<T>::Tval*>(this) ).m_val;
			return true;
		}
		return false; // type check failed
	}

	// Summary:
	//	 Template class to set the given type with the value and return if type check was ok.
	template<typename T> bool SetValue(const T &inVal)
	{
		if (m_eType == (enumParamType)Map2ptParam<T>::TypeId)
		{
			( *static_cast<typename Map2ptParam<T>::Tval*>(this) ).m_val = inVal;
			return true;
		}
		return false; // type check failed
	}
};

struct ptParamBOOL : ptParam 
{ 
	friend struct ptParam;
	ptParamBOOL(bool newVal) : ptParam(ptBOOL), m_val(newVal) {}
private:
	bool m_val;
};

struct ptParamVOIDP : ptParam
{
	friend struct ptParam;
	ptParamVOIDP(void * newVal) : ptParam(ptVOIDP), m_val(newVal) {}
private:
	void* m_val;
};

struct ptParamF32 : ptParam 
{ 
	friend struct ptParam;
	ptParamF32(f32 newVal) : ptParam(ptF32), m_val(newVal) {} 
private:
	f32 m_val;
};

struct ptParamINT32 : ptParam 
{ 
	friend struct ptParam;
	ptParamINT32(int32 newVal) : ptParam(ptINT32), m_val(newVal) {}
private:
	int32 m_val;
};

struct ptParamCRYSTRING : ptParam 
{ 
	friend struct ptParam;
	ptParamCRYSTRING(const string& newVal) : ptParam(ptCRYSTRING), m_val(newVal) {}
private:
	string m_val;
};

struct ptParamVEC3F32 : ptParam 
{ 
	friend struct ptParam;
	ptParamVEC3F32(const Vec3& newVal) : ptParam(ptVEC3F32), m_val(newVal) {}
private:
	Vec3 m_val;
};

#endif // CRYSOUND_ISOUND_H
