#ifdef PS3

#include <sys/process.h>

#define eCryModule eCryM_Launcher
#define _LAUNCHER
#include <CryModuleDefs.h>
#include <platform.h>
#include "IPlatformOS.h"		//Used to pass messages to the PlatformOS, nevessary for cellSysUtil event listening

#if defined(_DEBUG)
  #define STARTUP_MAIN_THREAD_STACK_SIZE (MAIN_THREAD_STACK_SIZE * 4 * 1024)
#else
	#define STARTUP_MAIN_THREAD_STACK_SIZE (MAIN_THREAD_STACK_SIZE * 1024)
#endif

static string g_HomeDir[2];

const int MAX_PATH_LEN = 1024;

//#define SUPP_MTRACE
//#define SUPP_LOAD_SEQ
//#define SUPP_PROFILE_LOADING
//#define SUPP_LOAD_HEARTBEAT

#include <platform_impl.h> 
 
#include <IGameStartup.h>
#include <IEntity.h>
#include <IGameFramework.h>
#include <IConsole.h>

#ifdef USE_LOADSEQ
  #include <LoadSeq.h>
#endif

#include <sys/dbg.h>
#include <fenv.h>

// MTrace support
#include "CryMTrace.h"


#ifdef _LIB
	// Manually instantiate templates as needed here.
	#include "Common_TypeInfo.h"
	STRUCT_INFO_T_INSTANTIATE(Vec2_tpl, <float>)
  STRUCT_INFO_T_INSTANTIATE(Vec2_tpl, <int>)
//#ifdef XENON_INTRINSICS
#if 0
	template struct Vec3_tpl<float>;
#else
	STRUCT_INFO_T_INSTANTIATE(Vec3_tpl, <float>)
#endif
	STRUCT_INFO_T_INSTANTIATE(Vec3_tpl, <int>)
	STRUCT_INFO_T_INSTANTIATE(Ang3_tpl, <float>)
	STRUCT_INFO_T_INSTANTIATE(Quat_tpl, <float>)
	STRUCT_INFO_T_INSTANTIATE(Plane_tpl, <float>)
	STRUCT_INFO_T_INSTANTIATE(Matrix33_tpl, <float>)
	STRUCT_INFO_T_INSTANTIATE(Color_tpl, <float>)
	STRUCT_INFO_T_INSTANTIATE(Color_tpl, <uint8>)
#endif

#ifdef XENON_INTRINSICS
	const XMVECTOR g_XMIdentityR0 = {1.0f, 0.0f, 0.0f, 0.0f};
	const XMVECTOR g_XMIdentityR1 = {0.0f, 1.0f, 0.0f, 0.0f};
	const XMVECTOR g_XMIdentityR2 = {0.0f, 0.0f, 1.0f, 0.0f};
	const XMVECTOR g_XMIdentityR3 = {0.0f, 0.0f, 0.0f, 1.0f};
  const XMVECTOR  g_XMZero      = {0.0f, 0.0f, 0.0f, 0.0f};
  const XMVECTOR  g_XMOne       = {1.0f, 1.0f, 1.0f, 1.0f};
#endif

#ifdef USE_LOADSEQ
  #define LOAD_SEQ_PATH "loadseq"
#endif

#ifdef PS3_PROFILE_LOCKS
	int g_nLockOpCount = 0;
	int g_nLockContentionCount = 0;
	int g_nLockContentionCost = 0;
#endif

#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
	namespace NMemCont
	{
		extern void InitMemContainer();
		extern void DestroyMemContainer();
	}
#endif 

//static memory the SPU code will be pasted into by JobGen
DEFINE_SPUREPOSITORY

#include <sdk_version.h>

#include <netex/net.h>
#include <netex/errno.h>
#include <cell/fs/cell_fs_errno.h>
#include <cell/fs/cell_fs_file_api.h>

#ifndef USE_SYSTEM_THREADS
	#include <pthread.h>
#endif

#include <sys/types.h>
#include <sys/memory.h>
#include <cell/sysmodule.h>
#include <sysutil/sysutil_common.h>
#include <sysutil/sysutil_sysparam.h>
#include <sys/synchronization.h>
#include <sys/ppu_thread.h>
#include <sys/spu_initialize.h>
#include <sys/paths.h>

#ifndef _LIB
#include <InitPRX.h>
#endif

#include <IJobManSPU.h>

SYS_PROCESS_PARAM(THREAD_PRIORITY_NORMAL,STARTUP_MAIN_THREAD_STACK_SIZE)

extern uint32 g_ForceStopSPUs;
extern void DXPSDebugPrinterRelease();

//#define SPU_CHECK_THREAD

#if defined(SPU_CHECK_THREAD)
	#include <IRenderer.h>
	bool g_UseCheckThread = false;
	void SPUCheckThread(uint64_t)
	{
		uint32 lastFrameCounter = 0;
		while(1)
		{
			Sleep(4000);
			if(gEnv->pRenderer)
			{
				uint32 newFrameID = gEnv->pRenderer->GetFrameID();
				if(lastFrameCounter == newFrameID && IsSPUEnabled())
				{
					//check run control of all spus
					if(0!=GetIJobManSPU()->VerifySPUs(true))
						snPause();
				}
				lastFrameCounter = newFrameID;
			}
		}
	}
#endif

#if defined SUPP_LOAD_HEARTBEAT
__attribute__((noinline)) void exec_heartbeat()
{ 
  for (int i=0; i<1024; ++i); 
}
void loading_heartbeat_thread(uint64_t param)
{
  do 
  { 
    exec_heartbeat();
    sys_timer_usleep(1000*1000);
  } while (true);
}
#endif 

// do not change it, also used for directory creation in cp_mastercd
const int MAX_GAME_FOLDER_PATH =128;
static char g_GameDirName[MAX_GAME_FOLDER_PATH] = "game";
static char g_HDDDirName[MAX_GAME_FOLDER_PATH] = "GAME02";
static char g_TitleIDName[MAX_GAME_FOLDER_PATH] = "Crysis";

static int g_DeleteShaderCache = 0;
static int g_CopyShaders = 1;
#ifdef USE_LOADSEQ
  static int g_DeleteLoadSequFiles = 0;
#endif

static int g_use_bdvd = 0;	//force disc mode
static int g_use_app = 0;	//force app_home mode
static int g_isDisc = 0;		//indicates if we run from disc
	
static int g_cpMasterCD = 1;//0: no copy of mastercd at startup, 1: copy update mode, 2: force update mode
static int g_DXPSDebugPrint = 0; // Use dxps debug to display messages onscreen during loading

static int g_VirtMemSysSize = 512*1024;			//8x64 KB pages cached as default
static int g_VirtMemRSXSize = 32*1024*1024;	//24 MB RSX as default

THREADLOCAL size_t __thread_stacksize;
THREADLOCAL void*  __thread_stackptr;

//------------------------------------start of cp_mastercd----------------------------------------

struct SCellGameDataStatGet
{
	char *pContentInfoPath;
	char *pUsrdirPath;
	void Allocate(){pContentInfoPath = new char[CELL_GAME_PATH_MAX];pUsrdirPath = new char[CELL_GAME_PATH_MAX];}
	void Destroy(){delete [] pContentInfoPath; delete [] pUsrdirPath; pContentInfoPath = pUsrdirPath = NULL;}
} g_GameStat;

void DXPSDebugPrinterInit();
void DXPSDebugPrint(const char* pText);
void DebugPrint(const char* pFormat,...)
{
# if !defined(_RELEASE)
  char Text[4*1024];
  va_list va;
  va_start(va,pFormat); 
  vsprintf(Text,pFormat,va);
  printf(Text);
  if (g_DXPSDebugPrint != 0) 
		DXPSDebugPrint(Text);
# endif 
}

typedef struct SCopyStats
{
	int filesDeleted, fileErrors, filesCopied, filesUpdated, filesLeft;
	std::vector<std::string> fileErrorVec;
	std::map<std::string, int64_t> fileMapOutput;//map strings to modify times, written at the end
	std::map<std::string, int64_t> fileMap;//map strings to modify times, read at the beginning
	SCopyStats() : filesDeleted(0),fileErrors(0), filesCopied(0), filesUpdated(0), filesLeft(0){}
} SCopyStats;

enum
{
	MODE_LEAVE = 0,
	MODE_COPY,
	MODE_UPDATE
};

inline void AssignLowerString(std::string& rDest, const char*const cpSrc, const int cStartIndex = 0)
{
	rDest = cpSrc;
	const int cLen = rDest.size();
	char* pString = (char*)rDest.c_str();
	for(int i=cStartIndex; i<cLen; ++i)
	{
		*pString = tolower(*pString);
		++pString;
	}
}

inline int64_t atoi64(const char * const cpBuf)
{
	const char* cpWorkBuf = cpBuf;
	while (isspace((int)*cpWorkBuf))
		++cpWorkBuf;
	int c = (int)*cpWorkBuf++;
	int sign = c;
	if (c == '-' || c == '+')
		c = (int)*cpWorkBuf++;
	int64_t value = 0;
	while (isdigit(c))
	{
		value = 10 * value + (c - '0');
		c = (int)*cpWorkBuf++;
	}
	if (sign == '-')
		return -value;
	else
		return value;
}

inline void ListAllFiles(std::vector<std::string>& rFileVec, const char *cpDir)
{
	CellFsErrno ret;
	int dir = 0;
	CellFsDirent dirent;
	uint64_t numRead;
	char path[MAX_PATH_LEN];
	ret = cellFsOpendir(cpDir, &dir);
	if(ret != CELL_FS_SUCCEEDED)
	{
		DebugPrint("copy_mastercd: cellFsOpendir(%s) failed, 0x%x\n", cpDir, ret);
		return;
	}
	while(CELL_FS_SUCCEEDED == cellFsReaddir(dir, &dirent, &numRead)) 
	{
		if(0 == numRead) 
			break;
		switch(dirent.d_type) 
		{
		case CELL_FS_TYPE_REGULAR:
			{
				std::string fileName = cpDir;
				fileName += dirent.d_name;
				rFileVec.push_back(fileName);
				break;
			}
		case CELL_FS_TYPE_DIRECTORY:
			if(0 != strcmp(dirent.d_name, ".") && 0 != strcmp(dirent.d_name, ".."))
			{
				snprintf(path, sizeof(path), "%s/%s", cpDir, dirent.d_name);
				//copy recursively
				ListAllFiles(rFileVec, path);
			}
			break;
		default:
			break;
		}
	}
	cellFsClosedir(dir);
}

//loads file map from disk
int LoadFileMap(const char *cpFileMapFileName, std::map<std::string, int64_t>& rFileMap)
{
	int rd = 0;
	const CellFsErrno cRet = cellFsOpen(cpFileMapFileName, CELL_FS_O_RDONLY, &rd, NULL, 0);
	if(0 > cRet)
	{
		//failed to open the file modification file
		g_cpMasterCD = 2;
		return 0;
	}
	CellFsStat stats;
	if(cellFsStat(cpFileMapFileName, &stats) != CELL_FS_SUCCEEDED)
	{
		//failed to read the file modification stats
		if(rd)
			cellFsClose(rd);
		return -1;
	}
	char *pFileBuf = new char[stats.st_size];
	uint64_t numRead;
	const CellFsErrno cReadRet = cellFsRead(rd, pFileBuf, stats.st_size, &numRead);
	if(0 > cReadRet)
	{
		//failed to read the file modification contents
		if(rd)
			cellFsClose(rd);
		return -2;
	}
	if(numRead != stats.st_size)
	{
		//size read: %d is not equal to the size of the file modification file
		if(rd)
			cellFsClose(rd);
		return -3;
	}
	char itemBuf[MAX_PATH_LEN];
	const char* const cpEnd = &pFileBuf[stats.st_size];
	const char *cpCur = pFileBuf;
	const char *cpLastCur = cpCur;
	int mode = 0;
	std::string cmpString;	cmpString.reserve(MAX_PATH_LEN);
	int64_t mTime = 0;
	while(true)
	{
		//seek to next entry start
		while(cpCur != cpEnd && (*cpCur == '\0' || *cpCur == '\n')) 
			++cpCur;
		if(cpCur == cpEnd)
			break;
		cpLastCur = cpCur;
		//seek to end of next entry
		while(cpCur != cpEnd && (*cpCur != '\0' && *cpCur != '\n')) 
			++cpCur;
		const int cLen = (int)cpCur - (int)cpLastCur;
		if(cLen >= MAX_PATH_LEN-1)
			continue;
		memcpy(itemBuf, cpLastCur, cLen);
		itemBuf[cLen] = '\0';
		if(!mode)
			AssignLowerString(cmpString, itemBuf);
		else
			mTime = atoi64(itemBuf);
		if(mode)
			rFileMap.insert(std::make_pair(cmpString, mTime));
		mode ^= 1;
		if(cpCur == cpEnd)
			break;
	}
	//add all files to input map which were created by the PS3 itself (rsync behavior)
	std::vector<std::string> fileVec;	fileVec.reserve(1024);
	ListAllFiles(fileVec, g_GameStat.pUsrdirPath);
	const std::vector<std::string>::const_iterator cFileEnd = fileVec.end();
	for(std::vector<std::string>::const_iterator fileIt = fileVec.begin(); fileIt!=cFileEnd; ++fileIt)
	{
		const std::map<std::string, int64_t>::const_iterator cRes = rFileMap.find(*fileIt);
		if(cRes == rFileMap.end()) 
		{ 
			//add file with a dummy file time
			rFileMap.insert(std::make_pair(*fileIt, 0));
		}
	}
	if(rd)
		cellFsClose(rd);
	delete [] pFileBuf;
	return 0;
}

inline void CopySingleFile
(
	const char *cpSrc, 
	const char *cpDst, 
	const char *cpFileName,
	char* const __restrict pFileBuf,
	const uint32 cFileBufSize,
	SCopyStats& rCopyStats
)
{
	std::string cmpString;		cmpString.reserve(MAX_PATH_LEN);
	std::string lowerString;	

	AssignLowerString(lowerString, cpFileName);

	char path[MAX_PATH_LEN];
	char path2[MAX_PATH_LEN];

	snprintf(path, sizeof(path), "%s/%s", cpSrc, cpFileName);
	snprintf(path2, sizeof(path2), "%s/%s", cpDst, lowerString.c_str());
	std::string checkPath( path );

	// check against blacklist to spare some unwanted files from copying 
	// finding matches is done the easy way with std::string
	std::string lowerPath;
	AssignLowerString(lowerPath, path);

	CellFsErrno ret = 0;
	int rd = 0, wr = 0;
	int copyMode = MODE_COPY;

	bool outputProgress = false;
	int sizeDiffOutput = 0;
	int64_t lastSize = 0;
	int64_t curSize = 0;

	uint64_t numRead;
	char outputPath[MAX_PATH_LEN];

	ret = cellFsOpen(path, CELL_FS_O_RDONLY, &rd, NULL, 0);
	if(ret != CELL_FS_SUCCEEDED) 
	{
		++rCopyStats.fileErrors;
		return;
	}

	strcpy(outputPath, &path[g_HomeDir[g_isDisc].size()]);
	AssignLowerString(cmpString, path2);
	ret = cellFsOpen(path2, CELL_FS_O_APPEND, &wr, NULL, 0);
	if(ret != CELL_FS_SUCCEEDED && ret != CELL_FS_EEXIST && ret != CELL_FS_ENOENT)
		++rCopyStats.fileErrors;

	CellFsStat statsSrc, statsDst;
	if(cellFsStat(path, &statsSrc) != CELL_FS_SUCCEEDED)
	{
		++rCopyStats.fileErrors;
		goto done;
	}
	{
		outputProgress = (statsSrc.st_size > 4 * 1024 * 1024);
		if(outputProgress)
			sizeDiffOutput = statsSrc.st_size / 10;//output 10 dots

		const std::map<std::string, int64_t>::const_iterator cRes = rCopyStats.fileMap.find(cmpString);

		if(CELL_FS_SUCCEEDED == ret)
		{
			copyMode = MODE_UPDATE;
			if(g_cpMasterCD != 2 && cellFsStat(path2, &statsDst) == CELL_FS_SUCCEEDED)
			{
				if(statsDst.st_size == statsSrc.st_size) 
				{	
					//size is equal
					if(cRes != rCopyStats.fileMap.end())
					{
						if((cRes->second & ~1) == (statsSrc.st_mtime & ~1))
							copyMode = MODE_LEAVE;//timestamp matches too
					}
				}
			}
			cellFsClose(wr);
			wr = 0;
		}

		rCopyStats.fileMapOutput.insert(std::make_pair(cmpString, statsSrc.st_mtime));		
		if(copyMode == MODE_LEAVE)
			goto done;
		if(copyMode == MODE_UPDATE)
		{
			//unlink first
			cellFsUnlink(path2);
		}
		ret = cellFsOpen(path2, CELL_FS_O_CREAT | CELL_FS_O_TRUNC | CELL_FS_O_WRONLY, &wr, NULL, 0);
		if(ret != CELL_FS_SUCCEEDED) 
		{
			++rCopyStats.fileErrors;
			goto done;
		}
		if(copyMode == MODE_UPDATE)
		{
			++rCopyStats.filesUpdated;
			DebugPrint("updating: %s", outputPath);
		}
		else
		{
			++rCopyStats.filesCopied;
			DebugPrint("copying: %s", outputPath);
		}
		fflush(stdout);
		do
		{
			ret = cellFsRead(rd, pFileBuf, cFileBufSize, &numRead);
			if(ret != CELL_FS_SUCCEEDED) 
			{
				DebugPrint("copy_mastercd: cellFsRead() failed, 0x%x\n", ret);
				break;
			}
			if(numRead == 0) 
				break;
			ret = cellFsWrite(wr, pFileBuf, numRead, NULL);
			if(ret != CELL_FS_SUCCEEDED)  
			{
				DebugPrint("copy_mastercd: cellFsWrite() failed, 0x%x\n", ret);
				break;
			}
			if(outputProgress)
			{
				curSize += numRead;
				if((int)(curSize - lastSize) > sizeDiffOutput)
				{
					DebugPrint(".");
					fflush(stdout);
					lastSize = curSize;
				}
			}
		} 
		while(numRead > 0);
		//set the file modification time to match the host-pc side
		if(strstr(path2, "/shaders/"))
		{
			CellFsUtimbuf timeBuf;
			timeBuf.actime	= statsSrc.st_mtime;
			timeBuf.modtime = statsSrc.st_mtime;
			const CellFsErrno fileErr = cellFsUtime(path2, &timeBuf);//local host file
			if(fileErr != CELL_FS_SUCCEEDED)
			{
				char errBuf[512];
				DebugPrint(errBuf, "cellFsUTime failed for: %s\n",path2);
				rCopyStats.fileErrorVec.push_back(std::string(errBuf));
			}
		}
		DebugPrint("\n");
		fflush(stdout);
	}
done:
	if(rd) 
		cellFsClose(rd);
	if(wr)
		cellFsClose(wr);
}

//copy all regular files from srcDir to dstDir
inline void CopyDir
(
	const char *cpSrc, 
	const char *cpDst, 
	const int cRecLevel,
	char* const __restrict pFileBuf,
	const uint32 cFileBufSize,
	SCopyStats& rCopyStats
)
{	
	if(cRecLevel > 8)
	{
		char errBuf[512];
		snprintf(errBuf, sizeof(errBuf), "Max recursion level exceeded for dir: %s\n",cpSrc);
		rCopyStats.fileErrorVec.push_back(std::string(errBuf));
		return;
	}
	CellFsErrno ret;
	int dir = 0;
	CellFsDirent dirent;
	uint64_t numRead;
	char pathDst[MAX_PATH_LEN];
	char pathSrc[MAX_PATH_LEN];
	std::string lowerString;	lowerString.reserve(MAX_PATH_LEN);

	ret = cellFsOpendir(cpSrc, &dir);
	if(ret != CELL_FS_SUCCEEDED) 
	{
		DebugPrint("copy_mastercd: cellFsOpendir(%s) failed, 0x%x\n", cpSrc, ret);
		goto done;
	}

	while(CELL_FS_SUCCEEDED == cellFsReaddir(dir, &dirent, &numRead)) 
	{
		if(0 == numRead) 
			break;
		switch(dirent.d_type) 
		{
		case CELL_FS_TYPE_REGULAR:
			CopySingleFile(cpSrc, cpDst, dirent.d_name, pFileBuf, cFileBufSize, rCopyStats);
			break;
		case CELL_FS_TYPE_DIRECTORY:
			if(0 != strcmp(dirent.d_name, ".") && 0 != strcmp(dirent.d_name, ".."))
			{
				AssignLowerString(lowerString, dirent.d_name);
				snprintf(pathDst, sizeof(pathDst), "%s/%s", cpDst, lowerString.c_str());
				snprintf(pathSrc, sizeof(pathSrc), "%s/%s", cpSrc, dirent.d_name);
				//do not copy shader cache
				if(strstr(cpDst, "shaders") && (lowerString == "cache" || lowerString == "saved_from_ps3"))
					break;
				//create directory if not existing
				const CellFsErrno cRet = cellFsMkdir(pathDst, CELL_FS_DEFAULT_CREATE_MODE_1);
				if(cRet != CELL_FS_SUCCEEDED && cRet != CELL_FS_EEXIST)
				{
					DebugPrint("copy_mastercd: cannot create directory: %s, error code: ",pathDst);
					switch(cRet)
					{
					case CELL_FS_ENOTMOUNTED:
						DebugPrint("CELL_FS_ENOTMOUNTED\n");break;
					case CELL_FS_ENOENT:
						DebugPrint("CELL_FS_ENOENT\n");break;
					case CELL_FS_ENOTSUP:
						DebugPrint("CELL_FS_ENOTSUP\n");break;
					case CELL_FS_EINVAL:
						DebugPrint("CELL_FS_EINVAL\n");break;
					case CELL_FS_EFSSPECIFIC:
						DebugPrint("CELL_FS_EFSSPECIFIC\n");break;
					case CELL_FS_EFAULT:
						DebugPrint("CELL_FS_EFAULT\n");break;
					case CELL_FS_EIO:
						DebugPrint("CELL_FS_EIO\n");break;
					case CELL_FS_ENOMEM:
						DebugPrint("CELL_FS_ENOMEM\n");break;
					case CELL_FS_EPERM:
						DebugPrint("CELL_FS_EPERM\n");break;
					case CELL_FS_ENAMETOOLONG:
						DebugPrint("CELL_FS_ENAMETOOLONG\n");break;
					case CELL_FS_EROFS:
						DebugPrint("CELL_FS_EROFS\n");break;
					case CELL_FS_ENOTDIR:
						DebugPrint("CELL_FS_ENOTDIR\n");break;
					default:
						DebugPrint("UNKNOWN\n");
					}
					break;
				}
				//copy recursively
				CopyDir(pathSrc, pathDst, cRecLevel+1, pFileBuf, cFileBufSize, rCopyStats);
			}
			break;
		default:
			break;
		}
	}

done:
	if(dir) 
		cellFsClosedir(dir);
}

//delete the files being found in the input map but not in the output map
inline void CleanUpFiles(SCopyStats& rCopyStats)
{	
	// make sure that the used gamefolder is enclosed in '/' to ensure only paths are matched and no files
	std::string sCurGameFolder = g_GameDirName;
	sCurGameFolder += "/";
	std::map<std::string, int64_t> tmp_map;
	const std::map<std::string, int64_t>::const_iterator cInputEnd = rCopyStats.fileMap.end();
	const std::map<std::string, int64_t>::const_iterator cEnd = rCopyStats.fileMapOutput.end();
	for(std::map<std::string, int64_t>::const_iterator it = rCopyStats.fileMap.begin(); it != cInputEnd; ++it)
	{
		const std::map<std::string, int64_t>::const_iterator cRes = rCopyStats.fileMapOutput.find(it->first);		
		if(cRes == cEnd) 
		{ 
			// only delete files if the subdirectory matches the game directory
			std::string folder = it->first;
			int pos = folder.find("/usrdir/");	// only check folder after userdir			
			if( folder.find( sCurGameFolder, pos ) == std::string::npos )
			{
				tmp_map.insert(std::make_pair(it->first, it->second) );
				continue;
			}
			char buffer[MAX_PATH_LEN];
			strcpy(buffer, (it->first).c_str());
			//replace common path (its has all become lower cased)
			memcpy(buffer, g_GameStat.pUsrdirPath, strlen(g_GameStat.pUsrdirPath));
			if(CELL_FS_SUCCEEDED == cellFsUnlink(buffer))
			{
				DebugPrint("deleting: \"%s\"\n", buffer);
				++rCopyStats.filesDeleted;
			}
		}
	}
	// add all skipped files from other games to the output map(or else there are not found when switching the game and all files are transfered again)
	for( std::map<std::string, int64_t>::const_iterator it = tmp_map.begin(); it != tmp_map.end(); ++it )
		rCopyStats.fileMapOutput.insert( std::make_pair(it->first, it->second) );
}

inline void SaveFileMap(const char *cpFileMapFileName, const SCopyStats& crCopyStats)
{
	char fileBuf[MAX_PATH_LEN];
	int wr = 0;
	const std::map<std::string, int64_t>::const_iterator cEnd = crCopyStats.fileMapOutput.end();
	cellFsUnlink(cpFileMapFileName);
	const CellFsErrno cRet = 
		cellFsOpen(cpFileMapFileName, CELL_FS_O_CREAT | CELL_FS_O_TRUNC | CELL_FS_O_WRONLY, &wr, NULL, 0);
	if(0 > cRet)
		return;
	for(std::map<std::string, int64_t>::const_iterator it = crCopyStats.fileMapOutput.begin(); it != cEnd; ++it)
	{
		snprintf(fileBuf, sizeof(fileBuf), "%s\n",(it->first).c_str());
		int ret = cellFsWrite(wr, fileBuf, strlen(fileBuf), NULL);
		if(ret != CELL_FS_SUCCEEDED) 
			break;
		snprintf(fileBuf, sizeof(fileBuf), "%lld\n",it->second);
		ret = cellFsWrite(wr, fileBuf, strlen(fileBuf), NULL);
		if(ret != CELL_FS_SUCCEEDED)  
			break;
	}
	if(wr)
		cellFsClose(wr);
}

//delete all files in directory, does not remove dir itself
//return 0 if succeeded
static int DeleteDir(const char * const cpDir, const bool cRecurse = false)
{
	int retVal = 0;
	int dir = 0;
	CellFsDirent dirent;
	uint64_t numRead;
	CellFsErrno ret = cellFsOpendir(cpDir, &dir);
	if(CELL_FS_SUCCEEDED != ret)
	{
#ifdef USE_LOADSEQ
		if(!strstr(cpDir, LOAD_SEQ_PATH))
			printf("cellFsOpendir(%s) failed, 0x%x\n", cpDir, ret);
#endif
		return -1;
	}
	char filePath[MAX_PATH_LEN];
	strcpy(filePath, cpDir);
	char *pRelFilePath = &filePath[strlen(cpDir)];
	*pRelFilePath++ = '/';
	while(CELL_FS_SUCCEEDED == cellFsReaddir(dir, &dirent, &numRead)) 
	{
		if(0 == numRead) 
			break;
		switch(dirent.d_type) 
		{
		case CELL_FS_TYPE_REGULAR:
			strcpy(pRelFilePath, dirent.d_name);
			retVal = cellFsUnlink(filePath);
			break;
		case CELL_FS_TYPE_DIRECTORY:
			if(cRecurse && 0 != strcmp(dirent.d_name, ".") && 0 != strcmp(dirent.d_name, ".."))
			{
				strcpy(pRelFilePath, dirent.d_name);
				retVal |= DeleteDir(filePath, cRecurse);//delete recursively
			}
			break;
		default:
			break;
		}
	}
	cellFsClosedir(dir);
	return retVal;
}

//copies mastercd
//  expects initialized file system, usage of harddrive and initialized game data
inline void CopyMasterCD()
{
	//create dst dir
	char dstDir[MAX_PATH_LEN];
	snprintf(dstDir, sizeof(dstDir), "%s/%s", g_GameStat.pUsrdirPath, g_GameDirName);
	int hDst;
	if(cellFsOpendir(dstDir, &hDst) != CELL_FS_SUCCEEDED)
	{
		cellFsMkdir(dstDir, CELL_FS_DEFAULT_CREATE_MODE_1);
		g_cpMasterCD = 2;
	}
	else
		cellFsClose(hDst);

	//if g_cpMasterCD is set to 2, delete hdd0 recursively
	if(g_cpMasterCD == 2)
		DeleteDir(gPS3Env->pCurDirHDD0, true);

	SCopyStats copyStats;
	if(g_cpMasterCD != 0)
	{
#   if !defined(_RELEASE)
		if (g_DXPSDebugPrint != 0) 
      DXPSDebugPrinterInit();
#   endif 
		//create and open src dir
		char srcDir[MAX_PATH_LEN];
		snprintf(srcDir, sizeof(srcDir), "%s/%s", g_HomeDir[g_isDisc].c_str(),g_GameDirName);
		int hSrc;		
		if(cellFsOpendir(srcDir, &hSrc) != CELL_FS_SUCCEEDED)
		{
			DebugPrint("copy_mastercd: cannot open source data dir: %s\n",srcDir);
			return;
		}
		cellFsClose(hSrc);
	
		//load file map
		char fileContPath[MAX_PATH_LEN];
		snprintf(fileContPath, sizeof(fileContPath), "%s/filecontents_%s", g_GameStat.pContentInfoPath,g_GameDirName);

		int ret = LoadFileMap(fileContPath, copyStats.fileMap);
		if(ret != 0)
			DebugPrint("copy_mastercd: failed to load mod.cont %s\n",fileContPath);
		DebugPrint("copy_mastercd: %s from \"%s\" to \"%s\"\n",(g_cpMasterCD==2)?"copying":"updating",srcDir,dstDir);

		//copy game folder recursively, save file map and delete obsolete files
		const uint32 cFileBufSize = 512 * 1024;
		char *const __restrict pFileBuf = new char[cFileBufSize];
		CopyDir(srcDir, dstDir, 0, pFileBuf, cFileBufSize, copyStats);
		
		// also copy engine folder
		snprintf(srcDir, sizeof(srcDir), "%s/engine", g_HomeDir[g_isDisc].c_str());
		snprintf(dstDir, sizeof(dstDir), "%s/engine", g_GameStat.pUsrdirPath );	
		if(cellFsOpendir(dstDir, &hDst) != CELL_FS_SUCCEEDED)
		{
			cellFsMkdir(dstDir, CELL_FS_DEFAULT_CREATE_MODE_1);
		}
		else
			cellFsClose(hDst);
		CopyDir(srcDir, dstDir, 0, pFileBuf, cFileBufSize, copyStats);
		
		delete [] pFileBuf;

		CleanUpFiles(copyStats);
		SaveFileMap(fileContPath, copyStats);

		DebugPrint("copy_mastercd: files copied: %d   updated: %d   deleted: %d   errors: %d\n",
			copyStats.filesCopied, copyStats.filesUpdated, copyStats.filesDeleted, copyStats.fileErrors);
		const std::vector<std::string>::const_iterator cErrEnd = copyStats.fileErrorVec.end();
		for(std::vector<std::string>::const_iterator it = copyStats.fileErrorVec.begin(); it != cErrEnd; ++it)
			DebugPrint("%s",it->c_str());
	}
	// create a game dir for the engine.cfg file, so that it can be transfered successfully even when only using game04
	char engineConfigFolder[MAX_PATH_LEN];
	snprintf(engineConfigFolder, sizeof(engineConfigFolder), "%s/game", g_GameStat.pUsrdirPath );
	cellFsMkdir( engineConfigFolder, CELL_FS_DEFAULT_CREATE_MODE_1 );
	
	// create user/modelscache dir to prevent error message about not being createable
	char modelscache[MAX_PATH_LEN];
	snprintf(modelscache, sizeof(modelscache), "%s/user/", g_GameStat.pUsrdirPath );
	cellFsMkdir( modelscache, CELL_FS_DEFAULT_CREATE_MODE_1 );
		
	snprintf(modelscache, sizeof(modelscache), "%s/user/modelscache", g_GameStat.pUsrdirPath );
	cellFsMkdir( modelscache, CELL_FS_DEFAULT_CREATE_MODE_1 );
}

//------------------------------------end of cp_mastercd----------------------------------------


//necessary to be able to dma from stack to spu
#define MAIN_STACK_HEAP_SIZE ((MAIN_STACK_SIZE) * 1024)
#define USE_HEAP_AS_STACK 1

static char g_CmdLine[1024];

static void ps3LoadLauncherConfig(void);


//////////////////////////////////////////////////////////////////////////

extern "C" IGameStartup* CreateGameStartup();

extern "C" void abort()
{
	__asm__ volatile ( "tw 31,1,1" );
	while (true);
};

// This is code supporting the profiling of the startup and loading process.
// The idea is to have a fairly low-resolution interrupt stopping all threads
// (by trapping into the debugger).  A script driving the debugger will then
// iterate over all threads and record the stack traces into a file.
//
// This feature is enabled using the 'profile_loading' option in launcher.cfg.
// The debugger script is responsible for setting g_LoadProfileInterval to a
// non-negative value to trigger the thread stopping.  While
// g_LoadProfileInterval is negative, the profiling thread will poll this
// variable in a 500ms interval (so it is safe to leave 'profile_loading'
// enabled in the launcher.cfg).
#ifdef SUPP_PROFILE_LOADING

	int g_LoadProfileInterval = -1; // Will be set by the profiler script.
	bool g_ProfileLoading = false;
	__attribute__ ((noinline)) static void LoadProfileThreadStop()
	{
		__asm__ __volatile__ ("tw 31, 1, 1\n");
	}

	static void LoadProfileThread(uint64_t)
	{
		while (true)
		{
			if (g_LoadProfileInterval > 0)
			{
				if (g_LoadProfileInterval > 10)
				{
					Sleep(5);
					Sleep(g_LoadProfileInterval - 5);
				}
				else
					Sleep(g_LoadProfileInterval);
				LoadProfileThreadStop();
			}
			else
			{
				// Wait for the debugger to set g_LoadProfileInterval without hogging
				// the CPU.
				Sleep(500);
			}
		}
	}

	static void StartLoadProfileThread()
	{
		sys_ppu_thread_t loadProfileThread = 0;
		int err = sys_ppu_thread_create(
				&loadProfileThread,
				LoadProfileThread,
				reinterpret_cast<uint64_t>(""),
				0, 4096,
				SYS_PPU_THREAD_CREATE_JOINABLE,
				"LoadProfileThread");
		if (err != 0)
			printf("Cannot create LoadProfileThread, err=%d\n", err);
	}
#endif//SUPP_PROFILE_LOADING

//#define ENABLE_MEM_GUARD
#if defined(ENABLE_MEM_GUARD)
	static sys_mutex_t g_MemGuardMutex;
	static sys_cond_t g_MemGuardCond;
	static const uint32 scMaxElemSize = 1024;
	static volatile uint32 g_CurMaxUsedIndex = 0;
	static sys_ppu_thread_t g_MemGuardThread;
	//buffer of all addresses to track
	volatile uint32 g_AddrToTrack[scMaxElemSize];
	volatile int g_AddrLastRun[scMaxElemSize];

	void RunMemGuard(uint64_t)
	{
		//wait initially til the first mem address was added
		int err = sys_mutex_lock(g_MemGuardMutex, 0);
		if(err != CELL_OK)
			printf("Failed to lock mutex in RunMemGuard: 0x%08x\n",err);
		while(g_CurMaxUsedIndex == 0)
		{
			err = sys_cond_wait(g_MemGuardCond, 0);
			if (err != CELL_OK)
				printf("Failed to wait for condition in RunMemGuard: 0x%08x\n", err);
		}
		sys_mutex_unlock(g_MemGuardMutex);

		printf("RunMemGuard: start checking memory...\n"); 

		while(1)
		{
			for(int i=0; i<g_CurMaxUsedIndex; ++i)
			{
				if(g_AddrToTrack[i] != 0 && g_AddrLastRun[i] != 0xBADF00D)
				{
					if(g_AddrLastRun[i] != *(volatile int*)g_AddrToTrack[i])
					{
						__asm__ volatile ( "tw 31,1,1" );
						g_AddrLastRun[i] = *(int*)g_AddrToTrack[i];
					}
				}
			}
	//		sys_timer_usleep(1);
		}
	}

	void InitMemGuard()
	{
		g_MemGuardThread = 0;
		memset((void*)g_AddrToTrack, 0, sizeof(g_AddrToTrack));
		for(int i=0; i<scMaxElemSize; ++i)
			g_AddrLastRun[i] = 0xBADF00D;
		sys_mutex_attribute_t mutexAttr;
		sys_cond_attribute_t condAttr;
		sys_mutex_attribute_initialize(mutexAttr);
		sys_cond_attribute_initialize(condAttr);
		int err = sys_mutex_create(&g_MemGuardMutex, &mutexAttr);
		if(err != CELL_OK)
			printf("Failed to create mem guard mutex: %08x\n", err);
		err = sys_cond_create(&g_MemGuardCond, g_MemGuardMutex, &condAttr);
		if(err != CELL_OK)
			printf("Failed to create mem guard condition: %08x\n", err);
		err = sys_ppu_thread_create
		(
			&g_MemGuardThread,
			reinterpret_cast<void (*)(uint64_t)>(RunMemGuard),
			(uint64)(UINT_PTR)"",
			1,
			64*1024,
			0,
			"MemGuardThread"
		);
		if(err != CELL_OK)
			printf("Failed to create Mem Guard thread\n");
	}

	void DeleteMemBreakPoint(const uint32 cAddr)
	{
		int curIndex = 0;
		while(curIndex < g_CurMaxUsedIndex && g_AddrToTrack[curIndex] != cAddr)
			++curIndex;
		if(g_AddrToTrack[curIndex] == cAddr)
		{
			g_AddrToTrack[curIndex] = 0;
			g_AddrLastRun[curIndex] = 0xBADF00D;
			if(curIndex == g_CurMaxUsedIndex && g_CurMaxUsedIndex > 0)
				--g_CurMaxUsedIndex;
		}
	}

	void AddMemBreakPoint(uint32 cAddr)
	{
		cAddr &= ~0x3;//align to 4 byte boundaries
		int curIndex = 0;
		while(curIndex < g_CurMaxUsedIndex)
		{
			if(g_AddrToTrack[curIndex] == cAddr)
				return;
			if(g_AddrToTrack[curIndex] == 0)
			{
				g_AddrToTrack[curIndex] = cAddr;
				g_AddrLastRun[curIndex] = *(volatile int32*)cAddr;
				return;
			}
			++curIndex;
		}
		if(g_CurMaxUsedIndex < scMaxElemSize-1)
		{
			g_AddrToTrack[g_CurMaxUsedIndex] = cAddr;
			g_AddrLastRun[g_CurMaxUsedIndex] = *(volatile int32*)cAddr;
			++g_CurMaxUsedIndex;
			if(g_CurMaxUsedIndex == 1)
			{
				int err = sys_mutex_lock(g_MemGuardMutex, 0);
				if(err != CELL_OK)
					printf("Failed to lock mutex in AddMemBreakPoint: 0x%08x\n",err);
				err = sys_cond_signal(g_MemGuardCond);
				if (err != CELL_OK)
					printf("Failed to signal condition in AddMemBreakPoint: 0x%08x\n", err);
				sys_mutex_unlock(g_MemGuardMutex);
			}
			return;
		}
	}
#endif//ENABLE_MEM_GUARD

#define RunGame_EXIT(exitCode) return (exitCode)

static void InitCellFs()
{
	static bool sLoaded = false;
	if(!sLoaded)
	{
		int cRet = 0;
#if !defined(_RELEASE)
 		cRet = cellSysmoduleLoadModule(CELL_SYSMODULE_LV2DBG);
		if(cRet != CELL_OK) 
			printf("Fatal error: cellSysmoduleLoadModule(CELL_SYSMODULE_LV2DBG) = 0x%08x\n",cRet);
#endif
		cRet = cellSysmoduleLoadModule(CELL_SYSMODULE_FS);
		if(cRet != CELL_OK) 
		{
			printf("Fatal error: cellSysmoduleLoadModule(CELL_SYSMODULE_FS) = 0x%08x\n",cRet);
			exit(EXIT_FAILURE);
		}
		sLoaded = true;
	}
}

static void UnMountFileSystems()
{
  if ( g_isDisc && g_use_bdvd )
	  cellFsAioFinish(SYS_DEV_BDVD);
	else  
	if(g_use_app)
	  cellFsAioFinish(SYS_APP_HOME);
	else
//  if (!g_use_bdvd)
	  cellFsAioFinish(SYS_DEV_HDD0);
}

static void MountFileSystems()
{
  // Initialize async I/O
  if ( g_isDisc && g_use_bdvd )
  {
	  CellFsErrno err = cellFsAioInit(SYS_DEV_BDVD);
	  if (err != CELL_OK)
	  {
			printf("Fatal error: AIO initialization failed for %s: 0x%08x\n",
				SYS_DEV_BDVD, static_cast<unsigned>(err));
			exit(EXIT_FAILURE);
	  }
  }
	else
  if(g_use_app)
  {
	  CellFsErrno err = cellFsAioInit(SYS_APP_HOME);
	  if (err != CELL_OK)
	  {
		printf("Fatal error: AIO initialization failed for %s: 0x%08x\n",
			SYS_APP_HOME, static_cast<unsigned>(err));
		exit(EXIT_FAILURE);
	  }
  }
	else
//  if (!g_use_bdvd)
  {
	  CellFsErrno err = cellFsAioInit(SYS_DEV_HDD0);
	  if (err != CELL_OK)
	  {
		printf("Fatal error: AIO initialization failed for %s: 0x%08x\n",
			SYS_DEV_HDD0, static_cast<unsigned>(err));
		exit(EXIT_FAILURE);
	  }
  }
}

int CopyFileHostHDD(const char *cpSrc, const char *cpDst, char* const __restrict pFileBuf, const uint32 cFileBufSize)
{
	int rd, wr;
	int ret = cellFsOpen(cpSrc, CELL_FS_O_RDONLY, &rd, NULL, 0);
	if(0 > ret)
	{
//		printf("cellFsOpen(%s) failed\n", cpSrc);
		return -1;
	}
	ret = cellFsOpen(cpDst, CELL_FS_O_CREAT | CELL_FS_O_TRUNC | CELL_FS_O_WRONLY, &wr, NULL, 0);
	if(0 > ret)
	{
//		printf("cellFsOpen(%s) failed\n", cpSrc);
		return -1;
	}

	CellFsStat statsSrc;
	if(cellFsStat(cpSrc, &statsSrc) != CELL_FS_SUCCEEDED)
	{
//		printf("cellFsStat for source file %s failed\n", cpSrc);
		return -1;	
	}
	
	uint64_t numRead;
	do
	{
		ret = cellFsRead(rd, pFileBuf, cFileBufSize, &numRead);
		if(0 > ret)
		{
//			printf("cellFsRead() failed, 0x%x\n", ret);
			break;
		}
		if(numRead == 0) 
			break;
		ret = cellFsWrite(wr, pFileBuf, numRead, NULL);
		if(0 > ret) 
		{
//			printf("cellFsWrite() failed, 0x%x\n", ret);
			break;
		}
	} 
	while(numRead > 0);

	cellFsClose(rd);
	cellFsClose(wr);
	return 0;
}


void DeleteShaderCache()
{
	char pathBuf[MAX_PATH_LEN];
	
	strcpy(pathBuf, gPS3Env->pCurDirHDD0);
	char *pPathBufBase = &pathBuf[gPS3Env->nCurDirHDD0Len];		

	sprintf(pPathBufBase, "/user/shaders/cache" );
	int suc = DeleteDir(pathBuf, true);
	printf("%s deleting shader cache(dir: %s)\n",(suc==0)?"Done":"Failed",pathBuf);
	
	// also delete shadercache.pak and its timestamp if both are present
	int handle = -1;
	bool bDeleteShaderCachePak = false;
	bool bDeleteShaderCachePakTimeStamp = false;
	char pathShaderCachePak[MAX_PATH_LEN];		
	char pathShaderCachePakTimeStamp[MAX_PATH_LEN];		
			
	sprintf(pathShaderCachePak, "%sshadercache.pak",gPS3Env->pCurDirHDD0);		
	sprintf(pathShaderCachePakTimeStamp, "%sshadercachepak_timestamp",gPS3Env->pCurDirHDD0);
	
	if(cellFsOpen(pathShaderCachePak, CELL_FS_O_RDONLY, &handle, NULL, 0) == CELL_FS_SUCCEEDED )
	{
		bDeleteShaderCachePak = true;
	}
	
	cellFsClose(handle);
	
	if(cellFsOpen(pathShaderCachePakTimeStamp, CELL_FS_O_RDONLY, &handle, NULL, 0) == CELL_FS_SUCCEEDED )
	{
		bDeleteShaderCachePakTimeStamp = true;
	}
	
	cellFsClose(handle);
	
	if( bDeleteShaderCachePak ) cellFsUnlink(pathShaderCachePak);
	if( bDeleteShaderCachePakTimeStamp ) cellFsUnlink(pathShaderCachePakTimeStamp);
}

//updates a file if the file time has changed
static const uint32 UpdateFile(const char* const cpSrc, const char* const cpDst, char* const __restrict pFileBuf, const uint32 cFileBufSize, bool bCreate = false, bool verbose = true)
{
	int updateFile = 0;
	CellFsStat statsSrc, statsDst;
	if(cellFsStat(cpDst, &statsDst) == CELL_FS_SUCCEEDED)
	{
		if(cellFsStat(cpSrc, &statsSrc) == CELL_FS_SUCCEEDED)
		{
			if((statsSrc.st_mtime & ~1) != (statsDst.st_mtime & ~1))
				updateFile = 1;
		}
	}
	else
	{
		if (bCreate)
			updateFile = 1;
	}

	if(updateFile)
	{
		if(0 != CopyFileHostHDD(cpSrc, cpDst, pFileBuf, cFileBufSize))
			return 0;
		if(verbose)
			printf("Updating \"%s\"\n",cpDst);
		CellFsUtimbuf timeBuf;
		timeBuf.actime	= statsSrc.st_mtime;
		timeBuf.modtime = statsSrc.st_mtime;
		cellFsUtime(cpDst, &timeBuf);
		return 1;
	}
	else
	{
		// If the file's been removed from the home directory, remove it from the console.
		CellFsStat dummy;
		if ((cellFsStat(cpDst, &dummy) == CELL_FS_SUCCEEDED) && (cellFsStat(cpSrc, &dummy) == CELL_FS_ENOENT))
		{
			cellFsUnlink(cpDst);
			return 1;
		}
	}

	return 0;
}

static const uint32 UpdateFiles(const char* const cpSrcDir, const char* const cpDstDir, char* const __restrict pFileBuf, const uint32 cFileBufSize)
{
	//checks if file does already exists and if so, it checks if the modification time is different
	uint32 updatedCount = 0;
	int destDir = 0, srcDir = 0;
	CellFsDirent dirent;
	uint64_t numRead;
	CellFsErrno ret = cellFsOpendir(cpSrcDir, &srcDir);
	if(0 > ret)
	{
//		printf("Update files: cellFsOpendir(%s) failed, 0x%x\n", cpSrcDir, ret);
		return 0;
	}
	ret = cellFsOpendir(cpDstDir, &destDir);
	if(0 > ret)
	{
		printf("Update files: cellFsOpendir(%s) failed, 0x%x\n", cpDstDir, ret);
		return 0;
	}

	char fileSrcPath[MAX_PATH_LEN];
	strcpy(fileSrcPath, cpSrcDir);
	char *pRelSrcFilePath = &fileSrcPath[strlen(cpSrcDir)];
	*pRelSrcFilePath++ = '/';
	char fileDstPath[MAX_PATH_LEN];
	strcpy(fileDstPath, cpDstDir);
	char *pRelDstFilePath = &fileDstPath[strlen(cpDstDir)];
	*pRelDstFilePath++ = '/';
	while(CELL_FS_SUCCEEDED == cellFsReaddir(srcDir, &dirent, &numRead)) 
	{
		if(0 == numRead) 
			break;
		switch(dirent.d_type) 
		{
		case CELL_FS_TYPE_REGULAR:
		{
			strcpy(pRelSrcFilePath, dirent.d_name);
			const int cStrLen = strlen(dirent.d_name);
			char *pCurDest = pRelDstFilePath;
			const char *__restrict pCurSrc = dirent.d_name;
			for(int i=0; i<cStrLen; ++i)
				*pCurDest++ = tolower(*pCurSrc++);
			*pCurDest = '\0';
			//check if file does exist and if so, check modification time
			int updateFile = 1;
			CellFsStat statsSrc, statsDst;
			int dstExists = 0;
			if(cellFsStat(fileDstPath, &statsDst) == CELL_FS_SUCCEEDED)
			{
				dstExists = 1;
				//get modification time of source and dest
				if(cellFsStat(fileSrcPath, &statsSrc) == CELL_FS_SUCCEEDED)
				{
					if((statsSrc.st_mtime & ~1) == (statsDst.st_mtime & ~1) )//lack of precision
						updateFile = 0;
				}
			}
			if(updateFile)
			{
				if(dstExists)
					cellFsUnlink(fileDstPath);
				CopyFileHostHDD(fileSrcPath, fileDstPath, pFileBuf, cFileBufSize);
				//apply same modification time
				CellFsUtimbuf timeBuf;
				timeBuf.actime	= statsSrc.st_mtime;
				timeBuf.modtime = statsSrc.st_mtime;
				if(cellFsUtime(fileDstPath, &timeBuf) == CELL_FS_SUCCEEDED)
					printf("Updating file: \"%s\"\n",dirent.d_name);
				++updatedCount;
			}
		}
		break;
		default:
			break;
		}
	}
	if(srcDir) 
		cellFsClosedir(srcDir);
	if(destDir) 
		cellFsClosedir(destDir);
	return updatedCount;
}

//copies all source files if they do not exist or were modified (checking file time)
//return number of updated files
const uint32 CopyShaderSources()
{
	if(g_use_bdvd||g_isDisc||g_use_app)
		return 0;
	
	uint32 count = 0;
	char pathBuf[MAX_PATH_LEN];
	char pathBuf2[MAX_PATH_LEN];

	strcpy(pathBuf, gPS3Env->pCurDirHDD0);
	sprintf(&pathBuf[gPS3Env->nCurDirHDD0Len], "%s/user/shaders/", g_GameDirName);

	cellFsMkdir(pathBuf, CELL_FS_DEFAULT_CREATE_MODE_1);
	sprintf(pathBuf2, "%s/%s/shaders", g_HomeDir[g_isDisc].c_str(),g_GameDirName);
	const uint32 cFileBufSize = 32*1024;
	char fileBuf[cFileBufSize];
	count += UpdateFiles(pathBuf2, pathBuf, fileBuf, cFileBufSize);

	sprintf(&pathBuf[gPS3Env->nCurDirHDD0Len], "%s/shaders/hwscripts", g_GameDirName);

	cellFsMkdir(pathBuf, CELL_FS_DEFAULT_CREATE_MODE_1);
	sprintf(&pathBuf[gPS3Env->nCurDirHDD0Len], "%s/shaders/hwscripts/cryfx", g_GameDirName);
	cellFsMkdir(pathBuf, CELL_FS_DEFAULT_CREATE_MODE_1);
	sprintf(pathBuf2, "%s/%s/shaders/hwscripts/cryfx", g_HomeDir[g_isDisc].c_str(),g_GameDirName);
	count += UpdateFiles(pathBuf2, pathBuf, fileBuf, cFileBufSize);
	return count;
}

void InitNetwork()
{
	int ret = cellSysmoduleLoadModule(CELL_SYSMODULE_NET);	
	if(ret < 0) 
	{
		fprintf(stderr, "PS3 network initialization failed: cellSysmoduleLoadModule(), error %i\n", ret);
		exit(EXIT_FAILURE);
	}
	ret = sys_net_initialize_network();
	if(ret < 0) 
	{
		fprintf(stderr, "PS3 network initialization failed: sys_net_initialize_network(), error %i\n", ret);
		cellSysmoduleUnloadModule(CELL_SYSMODULE_NET);
		exit(EXIT_FAILURE);
	}

	ret = cellSysmoduleLoadModule(CELL_SYSMODULE_SYSUTIL_NP2);
	if (ret < 0)
	{
		fprintf(stderr, "PS3 network lobby initialization failed: cellSysmoduleLoadModule(), error %i\n", ret);
		cellSysmoduleUnloadModule(CELL_SYSMODULE_NET);
		exit(EXIT_FAILURE);
	}

	ret = cellSysmoduleLoadModule(CELL_SYSMODULE_SYSUTIL_AVCHAT2);
	if (ret < 0) 
	{
		fprintf(stderr, "PS3 network lobby initialization failed: cellSysmoduleLoadModule(), error %i\n", ret);
		cellSysmoduleUnloadModule(CELL_SYSMODULE_SYSUTIL_NP2);
		cellSysmoduleUnloadModule(CELL_SYSMODULE_NET);
		exit(EXIT_FAILURE);
	}

	ret = cellSysmoduleLoadModule(CELL_SYSMODULE_SYSUTIL_NP_TROPHY);
	if (ret < 0) 
	{
		fprintf(stderr, "PS3 trophy initialization failed: cellSysmoduleLoadModule(), error %i\n", ret);
		cellSysmoduleUnloadModule(CELL_SYSMODULE_SYSUTIL_AVCHAT2);
		cellSysmoduleUnloadModule(CELL_SYSMODULE_SYSUTIL_NP2);
		cellSysmoduleUnloadModule(CELL_SYSMODULE_NET);
		exit(EXIT_FAILURE);
	}

#if defined(SUPP_MTRACE)
	// Initialize the mtrace framework at this point 
	if (!MTrace::Connect())
    fprintf(stderr, "could not connect to mtrace server!\n");
#endif
}

void InitFileSystem()
{
	int ret = cellSysmoduleLoadModule(CELL_SYSMODULE_SYSUTIL_GAME);
	if (ret != CELL_OK)
	{
		printf("cellSysmoduleLoadModule(CELL_SYSMODULE_SYSUTIL_GAME) failed (0x%x)\n", ret);
		exit(EXIT_FAILURE);
	}
	g_GameStat.Allocate();

	CellGameContentSize size;
	memset(&size, 0x00, sizeof(CellGameContentSize));
	unsigned int type = 0;
	char hddDirName[CELL_GAME_DIRNAME_SIZE];
	ret = cellGameBootCheck( &type, &gPS3Env->bootAttributes, &size, hddDirName );
	if (ret != CELL_GAME_RET_OK)
	{
		printf("cellGameBootCheck() Error: 0x%x\n",ret);
		exit(EXIT_FAILURE);
	}
	gPS3Env->bBootAttributesInitialised = 1;

	CellGameSetInitParams gameParam;
	memset(&gameParam, 0x00, sizeof(gameParam));
	ret = cellGameGetParamString(CELL_GAME_PARAMID_TITLE, gameParam.title, sizeof(gameParam.title) );
	if (ret != CELL_GAME_RET_OK)
	{
		// PARAM.SFO is not used so we will use the following hard coded values
		printf("PARAM.SFO parameters cannot be found - using app specified values\n");
		strcpy(gameParam.title, g_TitleIDName);
		strcpy(gameParam.titleId, g_HDDDirName);
		strcpy(gameParam.version, "01.00");
	}
	else
	{
		// PARAM.SFO is used
		printf("PARAM.SFO parameters have been found - game title is %s\n", gameParam.title);
		ret = cellGameGetParamString(CELL_GAME_PARAMID_TITLE_ID, gameParam.titleId, sizeof(gameParam.titleId) );
		
		if (ret == CELL_GAME_RET_OK)
		{
 			strcpy( g_HDDDirName, gameParam.titleId );
		}
		else
		{
			printf("cellGameGetParamString(CELL_GAME_PARAMID_TITLE_ID) Error: 0x%x\n",ret);
			exit(EXIT_FAILURE);
		}
		ret = cellGameGetParamString(CELL_GAME_PARAMID_VERSION, gameParam.version, sizeof(gameParam.version) );
		
		if (ret != CELL_GAME_RET_OK)
		{
			printf("cellGameGetParamString(CELL_GAME_PARAMID_VERSION) Error: 0x%x\n",ret);
			strcpy(gameParam.version, "01.00");
		}
		printf("Game titleid: %s version %s\n", gameParam.titleId, gameParam.version);
	}
	// Need to call cellGameContentPermit to keep Mr Cell happy
	ret = cellGameContentPermit( g_GameStat.pContentInfoPath, g_GameStat.pUsrdirPath );
	if (ret != CELL_GAME_RET_OK)
	{
		printf("cellGameContentPermit() Error: 0x%x\n", ret);
		exit(EXIT_FAILURE);
	}

	memset(&size, 0x00, sizeof(CellGameContentSize));
	ret = cellGameDataCheck(CELL_GAME_GAMETYPE_GAMEDATA, g_HDDDirName, &size);
	if(CELL_GAME_GAMETYPE_HDD != ret && CELL_GAME_RET_OK != ret)
	{
		printf("cellGameDataCheck() Error: 0x%x\n",ret);
		exit(EXIT_FAILURE);
	}

	ret = cellGameCreateGameData(&gameParam, g_GameStat.pContentInfoPath, g_GameStat.pUsrdirPath);
	if(ret != CELL_GAME_RET_OK && ret != CELL_GAME_ERROR_EXIST)
	{
		printf("cellGameCreateGameData() Error: 0x%x\n", ret);
		exit(EXIT_FAILURE);
	}
	else 
	{
		ret = cellGameContentPermit(g_GameStat.pContentInfoPath, g_GameStat.pUsrdirPath);
		if(ret != CELL_GAME_RET_OK)
		{
			printf("cellGameContentPermit() Error: 0x%x\n", ret);
//			exit(EXIT_FAILURE);
		}

		if(cellGameDataCheck(CELL_GAME_GAMETYPE_DISC, NULL, NULL) == CELL_GAME_GAMETYPE_DISC)
			g_isDisc = 1;

		if(g_isDisc)
			printf("running from disc\n");

		MountFileSystems();
	
		if(g_use_bdvd && g_isDisc)
		{
			printf("InitFileSystem: device %s\n",g_HomeDir[g_isDisc].c_str());
			strcpy(fopenwrapper_basedir, g_HomeDir[g_isDisc].c_str());
			gPS3Env->pCurDirHDD0 = fopenwrapper_basedir;
			gPS3Env->nCurDirHDD0Len = strlen(fopenwrapper_basedir);
		}
		else 
		if(g_use_app)
		{
			printf("InitFileSystem: device %s\n",g_HomeDir[0].c_str());
			strcpy(fopenwrapper_basedir, g_HomeDir[0].c_str());
			gPS3Env->pCurDirHDD0 = fopenwrapper_basedir;
			gPS3Env->nCurDirHDD0Len = strlen(fopenwrapper_basedir);
		}
		else
		{
			int nCurDirHDD0Len = strlen(g_GameStat.pUsrdirPath);
			assert(nCurDirHDD0Len > 1);
			if(g_GameStat.pUsrdirPath[nCurDirHDD0Len-1] != '/')
			{	
				g_GameStat.pUsrdirPath[nCurDirHDD0Len++] = '/';
				g_GameStat.pUsrdirPath[nCurDirHDD0Len] = '\0';
			}
			gPS3Env->nCurDirHDD0Len = nCurDirHDD0Len;
			strcpy(fopenwrapper_basedir, g_GameStat.pUsrdirPath);
			gPS3Env->pCurDirHDD0 = fopenwrapper_basedir;
			printf("InitFileSystem: device %s\n",fopenwrapper_basedir);

			CopyMasterCD();
		}
	}
}

void InitSystemCache()
{
	const char* shaderfolders[] =
	{
		"/user",
		"/user/shaders",
		"/user/shaders/cache",
		"/user/shaders/cache/d3d10",
		"/user/shaders/cache/d3d10/cgpshaders",
		"/user/shaders/cache/d3d10/cgvshaders",
		"/user/shaders/cache/d3d10/fxerror"
	};
}

#define CGSERVER_HOSTNAME "192.168.4.121"
#define CGSERVER_HOSTNAME_MAXSIZE (128)
#define CGSERVER_PORT (4455)

char ps3_cgserver_hostname_buf[CGSERVER_HOSTNAME_MAXSIZE]
	= CGSERVER_HOSTNAME;
char *ps3_cgserver_hostname = ps3_cgserver_hostname_buf;
int ps3_cgserver_port = CGSERVER_PORT;

inline void strip(char *s)
{
	char *p = s, *p_end = s + strlen(s);

	while (*p && (isspace(*p) || *p=='\"')) ++p;//ignore ' ' and '"'
	if (p > s) { memmove(s, p, p_end - s + 1); p_end -= p - s; }
	for (p = p_end; p > s && (isspace(p[-1]) || p[-1] == '\"'); --p);
	*p = 0;
}

#ifdef USE_DMALLOC
// Configuration string for the dmalloc memory debugger.  When not linking
// against dmalloc, then this buffer is ignored.
char ps3_dmalloc_options[1024] = "";
#define AVOID_FOPEN 1
#endif

static void ps3LoadLauncherConfig(void)
{
	gPS3Env->spuEnabled = 1;
	gPS3Env->spuMax			= NPPU::scMaxSPU;
	char line[1024], *eq = 0;
	int n = 0;
	static bool configLoaded = false;
	bool disableCgc = false;
	bool enableMemReplay = false;
	bool disableLog = false;
  bool enableMTrace = false;
	bool enableDevMode = true;

	if (configLoaded)	return;

	char autoLoadName[256] = "";

	string configFileName(g_HomeDir[g_isDisc]);	configFileName += "/system.cfg";

#ifdef AVOID_FOPEN
	char configBuffer[32768];
	memset(configBuffer, 0, sizeof configBuffer);
	const char *configBufferP = configBuffer;
	{
		int fd = -1;
		int fsErr = cellFsOpen(
				configFileName.c_str(),
				CELL_FS_O_RDONLY,
				&fd,
				NULL,
				0);
		if (fsErr != CELL_FS_SUCCEEDED)
			return;
		uint64_t nRead = 0;
		fsErr = cellFsRead(
				fd,
				configBuffer,
				sizeof configBuffer,
				&nRead);
		if (fsErr != CELL_FS_SUCCEEDED)
			return;
		configBuffer[sizeof configBuffer - 1] = 0;
		if (nRead < sizeof configBuffer)
			configBuffer[nRead] = 0;
		cellFsClose(fd);
	}
#else
	FILE *fp = fopen(configFileName.c_str(), "r");
	if (!fp)
		return;
#endif

	printf("ps3LoadLauncherConfig : load %s\n",configFileName.c_str());

	while (true)
	{
		++n;
#ifdef AVOID_FOPEN
		{
			if (!*configBufferP)
				break;
			const char *lineEnd = strchr(configBufferP, '\n');
			if (lineEnd == NULL)
				lineEnd = configBufferP + strlen(configBufferP);
			size_t lineLen = std::min(
					static_cast<size_t>(lineEnd - configBufferP),
					static_cast<size_t>(sizeof line - 1));
			memcpy(line, configBufferP, lineLen);
			line[lineLen] = 0;
			configBufferP = lineEnd;
			while (*configBufferP && isspace(*configBufferP))
				++configBufferP;
		}
#else
		if (!fgets(line, sizeof line - 1, fp))
			break;
		line[sizeof line - 1] = 0;
#endif
		strip(line);
		//check for outcommented lines: '#', '--', ';', '//'
		if (!line[0]
			|| line[0] == '#'
			|| (line[0] == '-' && line[1] && (line[1] == '-'))
			|| (line[0] == '/' && line[1] && (line[1] == '/'))
			|| line[0] == ';')
			continue;
		eq = strchr(line, '=');
		if (!eq)
		{
			fprintf(stderr, "system.cfg: syntax error in line %i (%s)\n",	n, line);
			continue;
		}
		*eq = 0;
		strip(line);
		strip(++eq);
		if (!strcasecmp(line, "cgserver_host"))
		{
			if (strlen(eq) >= CGSERVER_HOSTNAME_MAXSIZE)
			{
				fprintf(stderr, "system.cfg, line %i: cgserver_host value too long(%s)\n", n, eq);
				continue;
			}
			strcpy(ps3_cgserver_hostname, eq);
		}
		else if (!strcasecmp(line, "cgserver_port"))
		{
			int port = atoi(eq);
			if (port < 1 || port > 65535)
			{
				fprintf(stderr, "system.cfg, line %i: invalid cgserver_port %i\n", n, port);
				continue;
			}
			ps3_cgserver_port = port;
		}
		else if (!strcasecmp(line, "devmode"))
		{
			if (strlen(eq) > 0 && eq[0] != ' ' && (atoi(eq) == 0) )
				enableDevMode = false;
		}
		else if (!strcasecmp(line, "spu_stop"))
		{
			if(atoi(eq) != 0)
				g_ForceStopSPUs = 1;
		}
#if defined(SPU_CHECK_THREAD)
		else if (!strcasecmp(line, "spu_checkthread"))
		{
			if(atoi(eq) != 0)
				g_UseCheckThread = true;
		}
#endif
		else if (!strcasecmp(line, "autoload"))
		{
			if (strlen(eq) > 0 && eq[0] != ' ' && (strstr(g_CmdLine, "+map") == NULL) ) // don't use autoload if +map was specified on cmd line
				strcpy(autoLoadName, eq);
		}
		else if (!strcasecmp(line, "disable_cgc"))
		{
			if(atoi(eq) != 0)
				disableCgc = true;
		}
		else if (!strcasecmp(line, "memreplay"))
		{
			if(atoi(eq) != 0)
				enableMemReplay = true;
		}
		else if (!strcasecmp(line, "disable_log"))
		{
			if(atoi(eq) != 0)
				disableLog = true;
		}
		else if (!strcasecmp(line, "copy_shaders"))
		{
			if(atoi(eq) == 0)
				g_CopyShaders = 0;
		}
		else if (!strcasecmp(line, "delete_shader_cache"))
		{
			if(atoi(eq) != 0)
				g_DeleteShaderCache = 1;
		}
#if !defined(_RELEASE)
		else if (!strcasecmp(line, "spu_enable"))
		{
			if(atoi(eq) == 0)
				gPS3Env->spuEnabled = 0;
		}
		else if (!strcasecmp(line, "spu_max"))
		{
			int maxVal = atoi(eq);
			gPS3Env->spuMax = max(2, min(maxVal, NPPU::scMaxSPU));//2..5, 1 used for DXPS, 1 at least for other jobs
		}
#endif
#ifdef USE_LOADSEQ
		else if (!strcasecmp(line, "delete_load_sequ_files"))
		{
			if(atoi(eq) != 0)
				g_DeleteLoadSequFiles = 1;
		}
#endif
#ifdef USE_DMALLOC
		else if (!strcasecmp(line, "dmalloc_options"))
		{
			strncpy(ps3_dmalloc_options, eq, sizeof ps3_dmalloc_options);
			ps3_dmalloc_options[sizeof ps3_dmalloc_options - 1] = 0;
		}
#endif
#ifdef SUPP_PROFILE_LOADING
    else if (!strcasecmp(line, "profile_loading"))
    {
      if (atoi(eq) != 0)
        g_ProfileLoading = true;
    }
#endif
#if defined(SUPP_MTRACE)
    else if(!strcasecmp(line, "mtrace_server"))
    {
      // skip whitespaces after eq 
      for (; *eq && std::isspace(*eq); ++eq); 
      if (std::strlen(eq) <= sizeof(MTrace::g_Config.ServerAddress))
      {
        memcpy(MTrace::g_Config.ServerAddress,eq,sizeof(MTrace::g_Config.ServerAddress));
        MTrace::g_Config.ServerAddress[sizeof(MTrace::g_Config.ServerAddress)-1] = '\0';
      }
    }
    else if(!strcasecmp(line, "mtrace_port"))
    {
      MTrace::g_Config.ServerPort = std::strtol(eq, NULL, 10);
    }
    else if (!strcasecmp(line, "enable_mtrace"))
    {
      if (atoi(eq) != 0)
        enableMTrace = true;
    }
#endif
    else if (!strcasecmp(line, "sys_game_folder"))
    {
			if (strlen(eq) > 1)
			{
				strncpy(g_GameDirName, eq, MAX_GAME_FOLDER_PATH);
				strlwr(g_GameDirName);
				printf("Using game folder: \"%s\"\n",g_GameDirName);
			}
    }
    else if (!strcasecmp(line, "sys_title_id"))
    {
			if (strlen(eq) > 1)
			{
				strncpy(g_TitleIDName, eq, MAX_GAME_FOLDER_PATH);
				printf("Using title: \"%s\"\n",g_TitleIDName);
			}
    }
    else if (!strcasecmp(line, "sys_hdd_folder"))
    {
			if (strlen(eq) > 1)
			{
				strncpy(g_HDDDirName, eq, MAX_GAME_FOLDER_PATH);
				printf("Using custom hdd folder: \"%s\"\n",g_HDDDirName);
			}
    }
		else if( !strcasecmp(line, "cmd_line"))
		{
			if (strlen(eq) > 1)
				strcat(g_CmdLine, eq);
		}
		else if( !strcasecmp(line, "use_bdvd"))
		{
		  if (atoi(eq) != 0)
				g_use_bdvd = 1;
		}
		else if( !strcasecmp(line, "use_sysapp"))
		{
		  if (atoi(eq) != 0)
				g_use_app = 1;
		}
		else if( !strcasecmp(line, "copy_mastercd"))
		{
			g_cpMasterCD = atoi(eq);
		}
    else if (!strcasecmp(line, "dxps_debugprint"))
    {
      g_DXPSDebugPrint = atoi(eq);
    }
		else if (!strcasecmp(line, "ps3_virt_mem_sys_size_kb"))
		{
			g_VirtMemSysSize = atoi(eq) << 10;
		}
		else if (!strcasecmp(line, "ps3_virt_mem_rsx_size_kb"))
		{
			g_VirtMemRSXSize = atoi(eq) << 10;
		}
	}
	if(enableDevMode)
	{
		strcat(g_CmdLine, " -devmode ");
		printf("Enabling devmode...\n");
	}
#ifndef AVOID_FOPEN
	fclose(fp);
#endif
	if(enableMemReplay)
	{
		strcat(g_CmdLine, " -memReplay ");
		bool bPaused = strstr(g_CmdLine, "-memReplayPaused") != 0;
		CryGetIMemReplay()->ReplayStart(bPaused);
	}
	if(strlen(autoLoadName) > 0)
	{
		strcat(g_CmdLine, " +map ");
		strcat(g_CmdLine, autoLoadName);
		if(enableMemReplay)
			strcat(g_CmdLine, " +memReplayDumpSymbols ");
	}
	gPS3Env->pCgSrvHostname = ps3_cgserver_hostname;
	gPS3Env->nCgSrvPort			= ps3_cgserver_port;
	gPS3Env->bDisableCgc		= disableCgc;
	gPS3Env->bDisableLog		= disableLog;
//	configLoaded = true;
}

extern "C" const char *getenv(const char *name)
{
#ifdef USE_DMALLOC
	InitCellFs();
	MountFileSystems();
	if (!strcmp(name, "DMALLOC_OPTIONS"))
	{
		ps3LoadLauncherConfig();
		return ps3_dmalloc_options;
	}
#endif
	return 0;
}

int RunGame(const char * /*commandLine*/)
{
	int exitCode = 0;
	
	int ret = cellSysmoduleLoadModule(CELL_SYSMODULE_AVCONF_EXT);
	if(ret < 0) 
	{
		fprintf(stderr, "PS3 AVCONF (gamma) initialization failed: cellSysmoduleLoadModule(), error %i\n", ret);
		RunGame_EXIT(1);
	}

/*	ret = cellSysmoduleLoadModule(CELL_SYSMODULE_JPGDEC);
	if(ret < 0) 
	{
		fprintf(stderr, "PS3 JPEG decoder initialization failed: cellSysmoduleLoadModule(), error %i\n", ret);
		RunGame_EXIT(1);
	}	
*/

	ret = cellSysmoduleLoadModule( CELL_SYSMODULE_RTC );
	if (ret < 0)
	{
		fprintf(stderr, "PS3 Sys Module RTC initialization failed: cellSysmoduleLoadModule(), error %i\n", ret);
		RunGame_EXIT(1);
	}
	
	string logFileName(g_HomeDir[g_isDisc]); logFileName += "/game.log";

#if defined(SPU_CHECK_THREAD)
	if(g_UseCheckThread)
	{
		sys_ppu_thread_t spuCheckThread = 0;
		int err;
		if(0 != (err=sys_ppu_thread_create(&spuCheckThread, SPUCheckThread, (uint64)(UINT_PTR)"", 0, 4096,	SYS_PPU_THREAD_CREATE_JOINABLE,"SPUCheckThread")))
			printf("Cannot create SPUCheckThread, err=%d\n",err);
		else
			printf("Created SPUCheckThread\n");
	}
#endif

#if defined(ENABLE_MEM_GUARD)
	InitMemGuard(); 
#endif

#ifdef SUPP_PROFILE_LOADING
  if (g_ProfileLoading) 
		StartLoadProfileThread();
#endif

	SSystemInitParams startupParams;
	startupParams.hInstance = 0;
	startupParams.sLogFileName = logFileName.c_str();

  if (strlen(g_CmdLine)+1 > sizeof(startupParams.szSystemCmdLine))
  {
    fprintf(stderr, "ERROR: cmdline %s too long\n", g_CmdLine);
    return 0; 
  }

	strncpy(startupParams.szUserPath, "user", sizeof(startupParams.szUserPath));
	strncpy(startupParams.szSystemCmdLine, g_CmdLine, sizeof(startupParams.szSystemCmdLine));

	// create the startup interface
	IGameStartup* pGameStartup = CreateGameStartup();

	if (!pGameStartup)
	{
		fprintf(stderr, "ERROR: Failed to create the GameStartup Interface!\n");
		RunGame_EXIT(1);
	}

	g_GameStat.Destroy();//not needed anymore
	// run the game
	IGame *game = pGameStartup->Init(startupParams);
#if defined(SUPP_MTRACE)
	MTrace::SnapShot("Engine_Init");
#endif
	if (game)
	{
#   if !defined(_RELEASE)
		if (g_DXPSDebugPrint != 0) 
      DXPSDebugPrinterRelease();
#   endif 

		// Check for signature file.
		if (!gEnv->pCryPak->IsFileExist("config/config.dat"))
			return -1;

		exitCode = pGameStartup->Run(NULL);
		pGameStartup->Shutdown();
		pGameStartup = 0;
		RunGame_EXIT(exitCode);
	}

	// if initialization failed, we still need to call shutdown
	pGameStartup->Shutdown();
	pGameStartup = 0;
	sys_net_finalize_network();
	cellSysmoduleUnloadModule(CELL_SYSMODULE_NET);
	cellSysmoduleUnloadModule(CELL_SYSMODULE_FS);
	if( !snIsDebuggerRunning() )
	{
		cellSysmoduleUnloadModule(CELL_SYSMODULE_LV2DBG);
	}
	fprintf(stderr, "ERROR: Failed to initialize the GameStartup Interface!\n");
	RunGame_EXIT(exitCode);

	// Not reached.
	return 0;
}

PS3SystemEnvironment gPS3Env;
SSystemGlobalEnvironment gEnv;
struct SRegFactoryNode* g_pHeadToRegFactories = 0;

int PS3Break() { fputs("break\n", stderr); return 0; }

#if defined __GPROF__
// The checkpoint function is intended as a profiling helper for profiling
// startup and level-loading times.  Should remain empty in the main branch.
int PS3Checkpoint(const char *message)
{
  // Add local checkpoint code here!
}
#endif

// PS3 TRC R080: It is possible for the game termination request event to be issued immediately after boot due to user
// operation or a disc ejection. Register the exit callback for handling the game termination request event
// as soon as possible after boot and poll cellSysutilCheckCallback() at regular intervals to enable
// termination processing from immediately after the start of the application.
const int EARLYSYSCALLTHREAD_PRIORITY = 1002;
const int EARLYSYSCALLTHREAD_STACK_SIZE = 32*1024;
const char EARLYSYSCALLTHREAD_NAME[] = "EarlySysCallThread";
const int SYSUTIL_CALLBACK_SLOT	= 0;	// Use this slot to avoid clashing with other users of the callbacks.

// PS3 TRC R080: This callback runs its own code until the PlatformOS_PS3 is initialised and can handle the events itself
void cellSysUtilGameThreadCallback(uint64_t status, uint64_t param, void * userdata)
{
	IPlatformOS::SPlatformEventPS3 platformEvent(0);
	IPlatformOS *pPlatform;
	if(gPS3Env->tickCellSysUtilCallbacksState==TICKCELLSYSUTIL_EARLY)
	{
		//Process here
		if(status==CELL_SYSUTIL_REQUEST_EXITGAME)	//This is the only request we need to handle to meet TRC R080
		{
			//A shutdown has been initiated before PlatformOS_PS3 is online and ticking. This is bad, but unlikely. Set a variable here so that the PlatformOS_PS3
			//can safely shut down on the game thread when it is initialised further down the line. Don't call the shutdown routine now, this is being ticked from a
			//separate thread.
			gPS3Env->bEarlyQuitMessageDetected = 1;
		}
	}
	else if(gPS3Env->tickCellSysUtilCallbacksState==TICKCELLSYSUTIL_GAME_THREAD)
	{
		//PlatformOS_PS3 has been initialised, use its handler
		assert(gEnv&&gEnv->pSystem);
		//the pData for this platformOS event is always the PlatformOS itself
		pPlatform = gEnv->pSystem->GetPlatformOS();
		platformEvent.m_status = status;
		platformEvent.m_param = param;
		platformEvent.m_userdata = pPlatform;
		pPlatform->NotifyListeners(platformEvent);
	}
	//else...
	//Processing is taking place on the Network thread; don't do anything here as the CCryPSNSupport object can invoke the handler on the PlatformOS_PS3 object
}

// PS3 TRC R080: Threaded function to tick the cellSysEvent queue until the PlatformOS_PS3 object is ready, at which point the thread is retired and the
// ticking is handled by the PlatformOS_PS3 object in the game thread or by the CryPSNSupport object in the Networking thread.
void EarlySysUtilThreadedTick(uint64_t arg)
{
	while(gPS3Env->tickCellSysUtilCallbacksState == TICKCELLSYSUTIL_EARLY)
	{
		cellSysutilCheckCallback();
		CrySleep(500);
	}
	sys_ppu_thread_exit(0);
}

#define STDOUT_BUF_SIZE 1024
#define STDERR_BUF_SIZE 512
char g_PrintBuf[STDOUT_BUF_SIZE];

// This is the first function called by main().
static void InitPS3()
{
	// PS3 TRC R080: Register a listener and create a thread to tick the cellSysEvent queue until the PlatformOS_PS3 object is ready
	gPS3Env->tickCellSysUtilCallbacksState = TICKCELLSYSUTIL_EARLY;
	int ret = cellSysutilRegisterCallback(SYSUTIL_CALLBACK_SLOT, cellSysUtilGameThreadCallback, NULL);
	sys_ppu_thread_t threadID;
	ret = sys_ppu_thread_create(&threadID,
									EarlySysUtilThreadedTick,
									NULL,
									EARLYSYSCALLTHREAD_PRIORITY,
									EARLYSYSCALLTHREAD_STACK_SIZE,
									0,
									EARLYSYSCALLTHREAD_NAME);
	assert(ret == CELL_OK);

	setvbuf(stdout, g_PrintBuf, _IOLBF, STDOUT_BUF_SIZE);
	setvbuf(stderr, NULL, _IOLBF, STDERR_BUF_SIZE);
	
	gPS3Env->staticMemUsedKB = 0;
	sys_memory_info_t memInfo;
	if(sys_memory_get_user_memory_size(&memInfo) == CELL_OK)
		gPS3Env->staticMemUsedKB = (memInfo.total_user_memory - memInfo.available_user_memory) >> 10;//in KB
#if defined(_LIB) && !defined(NOT_USE_CRY_MEMORY_MANAGER)
	extern long g_TotalAllocatedMemory;
	gPS3Env->staticMemUsedKB -= g_TotalAllocatedMemory >> 10;
	gPS3Env->staticMemUsedKB -= 1024;//initial pools
#endif
	gPS3Env->fnBreak = PS3Break;
#if defined __GPROF__
	gPS3Env->fnCheckpoint = PS3Checkpoint;
#endif

	// Note: If the entire application is linked statically, _LIB will be
	// defined, so we can use this as a test if PRX initialization code is
	// required.
#ifndef _LIB
	// Force initialization of the memory manager.
	_CryMemoryManagerPoolHelper::Init();
#endif // !_LIB

	// Initialize the fopen() wrapper buffers.
	InitFOpenWrapper();
}

#ifdef USE_LOADSEQ
inline void DeleteRecords()
{
	char pathBuf[MAX_PATH_LEN];
	strcpy(pathBuf, gPS3Env->pCurDirHDD0);
	char *pPathBufBase = &pathBuf[gPS3Env->nCurDirHDD0Len];
	strcpy(pPathBufBase, LOAD_SEQ_PATH);
	DeleteDir(pathBuf);
//	printf("Deleting load sequencer records...\n");
}
#endif

// Function to update configuration files like systems.cfg
const uint32 UpdateConfigFiles()
{
	char pathBuf[MAX_PATH_LEN];
	char pathBufSrc[MAX_PATH_LEN];
	strcpy(pathBuf, gPS3Env->pCurDirHDD0);
	char *pPathBufBase = &pathBuf[gPS3Env->nCurDirHDD0Len];

	uint32 filesUpdated = 0;

	//copy config files to be able to edit them and use the updated version on reload	
	//the system try to read system.cfg in this order
	// 1) /root/system.cfg
	// 2) /root/game/system.cfg
	// 3) /root/game/config/system.cfg

	const char *configfilelist[]=
	{
		"system.cfg",
		"user.cfg",
		"engine.cfg",
		"autotest.cfg",
		"chainlevels.txt",
		"test_chainlevels.txt"
	};

	const uint32 cFileBufSize = 8*1024;
	char fileBuf[cFileBufSize];
	for ( int i=0;i<sizeof(configfilelist)/sizeof(configfilelist[0]) ; i++ )
	{
		sprintf(pathBufSrc,"%s/%s",g_HomeDir[g_isDisc].c_str(),configfilelist[i]);
		sprintf(pPathBufBase, "%s",configfilelist[i]);
		filesUpdated += UpdateFile(pathBufSrc, pathBuf, fileBuf, cFileBufSize, true);
	}

	return filesUpdated;
}

//-------------------------------------------------------------------------------------
// Name: main()
// Desc: The application's entry point
//-------------------------------------------------------------------------------------
void main_func(void*)
{	
  printf("Main size(KB): thread = %d    stack = %d\n", STARTUP_MAIN_THREAD_STACK_SIZE/1024, MAIN_STACK_HEAP_SIZE/1024); 

  // Set the thread local variables for the main thread after it's
  // stack has hauled over to the heap.
  __thread_stacksize = MAIN_THREAD_STACK_SIZE;
  __thread_stackptr = gPS3Env->pMainStack; 

	// check if this is a full blue ray version( means assets and binary are both on BD )
	InitCellFs();

	ps3LoadLauncherConfig();

	if (!MTrace::Initialize())
    fprintf(stderr, "could not setup mtrace! tracing will be disabled!\n");

	InitNetwork();
	InitFileSystem();	
	InitSystemCache();

#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
	NMemCont::InitMemContainer();
#endif
	if(g_VirtMemSysSize > 0 && g_VirtMemRSXSize > 0)
		NVirtualMem::Init(g_VirtMemSysSize, g_VirtMemRSXSize);
	else
	{
		printf("Virtual memory: disabled\n");
	}
	uint32 cSystemFilesUpdated = 0;
	if (!(g_use_bdvd && g_isDisc) && !g_use_app)
		cSystemFilesUpdated = UpdateConfigFiles();
	if(g_DeleteShaderCache)
	{
		if(gPS3Env->bDisableCgc)
		{
			printf("Warning: delete shader cache and having gPS3Env->bDisableCgc = true\n");
//			gPS3Env->bDisableCgc = false;
		}
		DeleteShaderCache();
	}
		
	uint32 shaderFilesUpdated = 0;
	if(!gPS3Env->bDisableCgc && g_CopyShaders)
		shaderFilesUpdated += CopyShaderSources();
#ifdef USE_LOADSEQ
	//if shader files were updated -> binaries will be written -> no record, no playback
	if(shaderFilesUpdated > 0 || g_DeleteShaderCache)
	{
		//delete record files
		DeleteRecords();
		LoadSeq::Mode(LoadSeq::PASSTHROUGH);
	}
	else
	if(g_DeleteLoadSequFiles || cSystemFilesUpdated > 0)
	{
		DeleteRecords();
		LoadSeq::Mode(LoadSeq::PLAYBACK);
	}
	else
		LoadSeq::Mode(LoadSeq::PLAYBACK); //no files have been changed -> playback if record exists
#endif

	RunGame("");

	UnMountFileSystems();

	NVirtualMem::Shutdown();
}


#if !defined(USE_DMALLOC)
	extern "C" void *mwprivate2_memalign(size_t alignment, size_t size, ECryModule eCM = eCryModule)
	{
#if defined(_LIB)
		_CryMemoryManagerPoolHelper::allocatedMemory[eCryModule] += size;
#endif
		#undef memalign
		return memalign(alignment, size);
	}
#else
	extern "C" void *mwprivate2_memalign(size_t alignment, size_t size, ECryModule);
	extern "C" void mwprivate2_freealign(void *p, ECryModule eCM);
#endif

//main needs to put the stack via asm to the heap to be able to DMA from/to the stack by the SPUs
int main(int argc, char *argv[])
{
#if defined(SUPP_MTRACE)	
  MTrace::SnapShot("static_initialization");
#endif 
#if defined(USE_HEAP_AS_STACK)
	//	#if defined(_DEBUG)
	uint8 *pStack = (uint8*)mwprivate2_memalign(64, MAIN_STACK_HEAP_SIZE, eCryModule);
	//	#else
	//		uint8 *pStack = (uint8*)memalign(64, MAIN_STACK_HEAP_SIZE);
	//	#endif
#endif
	g_HomeDir[0] = SYS_APP_HOME"/";
	g_HomeDir[1] = SYS_DEV_BDVD"/PS3_GAME/USRDIR/";
	//TODO: must be removed for TRC
	if(argc >= 1 && NULL != strstr(argv[0],SYS_DEV_BDVD))
		g_isDisc = 1;
	
	if(argc > 1 && NULL != strstr(argv[1],"net"))
		g_use_app = 1;

	// build and store commandline to pass to engine(other things from system.cfg can also append stuff there like autoload)
  //strcpy(g_CmdLine, "ps3launcher +map gdce +exec autotesttimedemo");
	strcpy(g_CmdLine, "ps3launcher ");
	for( int i = 1 ; i < argc ; ++i )
	{		
		strcat(g_CmdLine, argv[i]);
		strcat(g_CmdLine, " ");
	}

	g_ThreadIndexVec.push_back((uint32)GetCurrentThreadId());

/*	if (strstr(g_CmdLine, "-memReplay"))
	{
		bool bPaused = strstr(g_CmdLine, "-memReplayPaused") != 0;
		CryGetIMemReplay()->ReplayStart(bPaused);
	}
*/
	InitPS3();

# if defined(SUPP_LOAD_HEARTBEAT)
	sys_ppu_thread_t heartBeatThread = 0;
	sys_ppu_thread_create
	(
		&heartBeatThread,
		reinterpret_cast<void (*)(uint64_t)>(loading_heartbeat_thread),
		uint64_t(),
		1000,
		4096,
		SYS_PPU_THREAD_CREATE_JOINABLE,
		"snTunerHeartBeat"
	);
#endif 

#if defined(USE_HEAP_AS_STACK)
/*	sys_ppu_thread_t currentThreadID;
	sys_ppu_thread_get_id(&currentThreadID);
	sys_ppu_thread_set_priority(currentThreadID, 600);
*/	
	gPS3Env->nMainStackSize = MAIN_STACK_HEAP_SIZE;
	gPS3Env->pMainStack  = pStack + MAIN_STACK_HEAP_SIZE;
	InvokeOnLinkedStack(main_func, NULL, (void*)pStack, MAIN_STACK_HEAP_SIZE);
	//force system free
	#undef free
	std::free(pStack);
#else
	sys_ppu_thread_t gameThread = 0;
	int err = sys_ppu_thread_create
	(
		&gameThread,
		reinterpret_cast<void (*)(uint64_t)>(main_func),
		(uint64)"",
		600,
		MAIN_STACK_HEAP_SIZE,
		SYS_PPU_THREAD_CREATE_JOINABLE,
		"MainThread"
	);
	assert(err == CELL_OK);
	uint64_t result = 0;
	err = sys_ppu_thread_join(gameThread, &result);
	assert(err == CELL_OK);
#endif

#if defined(ENABLE_MEM_GUARD)
	sys_cond_destroy(g_MemGuardCond);	
	sys_mutex_destroy(g_MemGuardMutex);
#endif

	//PS3 TRC R080: Unregister the early event listener
	cellSysutilUnregisterCallback(SYSUTIL_CALLBACK_SLOT);

  MTrace::Shutdown(); 
	return 0; 
}

#if defined(EMULATE_XENON_INTRINSICS) && !defined(IsXenonNumberValid)
bool IsXenonNumberValid(const float& x)
{
	//return ((FloatU32(x) & FloatU32ExpMask) != FloatU32ExpMask);

	uint32 i = FloatU32(x);
	uint32 expmask = FloatU32ExpMask;
	uint32 iexp = i & expmask;
	bool invalid = (iexp == expmask);

	if (invalid)
	{
		union f32_u
		{
			uint32 uintVal;
			f32 floatVal;
		};
		f32_u u;	u.uintVal = 0x7F800001;
		float fpe = u.floatVal;
	}

	return !invalid;
}
#endif

#include "console/Cry_PS3_Math.cpp"

#endif//PS3

// vim:ts=2:sw=2:expandta
