#include "StdAfx.h"

#include "CryMTrace.h"
#include <sys/memory.h>
#include <VectorSet.h>

// This file should never be compiled on a different platform
#if !defined(PS3)
# error "CryMTrace.cpp is ps3 only!" 
#endif

#include <pthread.h>
#include <CryThread.h>
#include <netex/net.h>
#include <netex/errno.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <mspace.h>

#define PS3_WRAP_NODEALLOC

#if defined(PS3_CRYSIZER_HEAP_TRAVERSAL) && defined(PS3_WRAP_NODEALLOC)
	#undef PS3_WRAP_NODEALLOC
#endif

#if !defined(_RELEASE)
//# define MEMSET_FREE_PATTERN (0xef)
#endif 

#if defined(PS3_WRAP_NODEALLOC)
#define VIRTUAL_ALLOC_SIZE 524288
#include "CryMemoryAllocator.h"
  node_alloc<eCryMallocCryFreeCRTCleanup, true, 524288> g_GlobPageBucketAllocator;
#undef CRY_MEMORY_ALLOCATOR
#endif 

#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
	#define PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
#endif
typedef uint16 TPairedMemIndex;

static void PrintSysErrorMessage(int ret, const char* cpOp)
{
#ifndef _RELEASE
	switch(ret)
	{
	case EINVAL: printf("%s: flags is invalid\n",cpOp); break;
	case EALIGN: printf("%s: size and alignment is invalid\n",cpOp); break;
	case ENOMEM: printf("%s: the requested area cannot be allocated because of user address space shortage\n",cpOp); break;
	case EAGAIN: printf("%s: the requested area cannot be allocated because of kernel resource shortage\n",cpOp); break;
	case EFAULT: printf("%s: alloc_addr is invalid\n",cpOp); break;
	case ESRCH:	 printf("%s: start_addr is invalid (not the multiple of page size)\n",cpOp);
	case EBUSY:	 printf("%s: specified address range overlaps with an area which is already mapped\n",cpOp);break;
	default:		 printf("%s: unknwon error code(0x%08x)\n",cpOp,ret); break;
	}
#endif
}

#undef calloc
#undef malloc
#undef free
#undef realloc
#undef reallocalign
#undef memalign

extern "C"
{
	void *__REAL__calloc(size_t, size_t); 
	void *__REAL__malloc(size_t);
	void __REAL__free(void *);
	void *__REAL__realloc(void *, size_t);
	void *__REAL__reallocalign(void *, size_t, size_t);
	void *__REAL__memalign(size_t, size_t);
	void __REAL___malloc_init(void);
	void __REAL___malloc_finalize(void);
}

#if !defined(_RELEASE)
//	#define ENABLE_DEBUG
#endif

#ifdef ENABLE_DEBUG
	#undef ILINE
	#define ILINE NO_INLINE
	#define SNPAUSE snPause()
#else
	#define SNPAUSE
#endif

static void PrintAllocFailureMessage(uint32 size)
{
#ifndef _RELEASE
	MEMORYSTATUS MemoryStatus;
	GlobalMemoryStatus(&MemoryStatus);
	float memInMB = (float) (MemoryStatus.dwTotalPhys - MemoryStatus.dwAvailPhys) / (1024.0f * 1024.0f);
	printf("[%.1f] **** failed to allocate %d bytes ****\n", memInMB, size );
#endif
	gEnv->bIsOutOfMemory = true;		
}

size_t _msize_nosystem(void *_p);

namespace PS3CrySizerHeapChecker
{
#if defined(PS3_CRYSIZER_HEAP_TRAVERSAL)
	void AddMemory( void *, uint32 size);
	void RemoveMemory(void*);
#else
	ILINE void AddMemory( void *, uint32 size){}
	ILINE void RemoveMemory(void*){}
#endif
}

namespace NMemCont
{
	static sys_addr_t	g_BaseAddress;
	static const uint32 MEMORY_ADDR_RANGE = 0x20000000UL;

	static const uint32 MEMORY_ALLOC_UPPER_BOUNDARY = 24*1024*1024;
	static const uint32 MEMORY_ALLOC_GRANULARITY = 64*1024;//do not change
	static const uint32 MEMORY_DEFAULT_ALIGN = 8;

	static const uint32 MEMORY_ADDR_IDX_0_CNT  = 1024;
	static const uint32 MEMORY_ADDR_IDX_1_CNT  = 64;
	static const uint32 MEMORY_ADDR_IDX_2_CNT  = 16;
	static const uint32 MEMORY_ADDR_IDX_3_CNT  = 4;
	static const uint32 MEMORY_MAX_ALLOC_AMOUNT = (MEMORY_ADDR_IDX_0_CNT + MEMORY_ADDR_IDX_1_CNT + MEMORY_ADDR_IDX_2_CNT + MEMORY_ADDR_IDX_3_CNT);

	static const uint32 MEMORY_ADDR_IDX_0_SIZE  = 256*1024;
	static const uint32 MEMORY_ADDR_IDX_1_SIZE  = 1024*1024;
	static const uint32 MEMORY_ADDR_IDX_2_SIZE  = 4*1024*1024;
	static const uint32 MEMORY_ADDR_IDX_3_SIZE  = MEMORY_ALLOC_UPPER_BOUNDARY;

	static const uint32 MEMORY_ALLOC_EXTERN_MIN_SIZE = 8*1024;

	static const uint32 MEMORY_ALLOC_WASTE_IGN_INIT = 16*1024;
	static const uint32 MEMORY_ALLOC_BOUNDARY_INIT	= MEMORY_ALLOC_GRANULARITY-MEMORY_ALLOC_WASTE_IGN_INIT;

	static const uint32 MEMORY_ALLOC_WASTE_IGN_SMALL = 38*1024;
	static const uint32 MEMORY_ALLOC_BOUNDARY_SMALL	= 22*1024;
	static const uint32 MEMORY_ALLOC_WASTE_IGN_LARGE = 24*1024;
	static const uint32 MEMORY_ALLOC_BOUNDARY_LARGE	= 45*1024;

	static uint32 MEMORY_ALLOC_BOUNDARY = MEMORY_ALLOC_BOUNDARY_INIT;
	static uint32 MEMORY_ALLOC_WASTE_IGN = MEMORY_ALLOC_WASTE_IGN_INIT;

	static const uint32 MEMORY_ALLOC_SMALL_LIMIT = MEMORY_ALLOC_GRANULARITY-MEMORY_ALLOC_BOUNDARY_SMALL;

	static const uint32 MEMORY_ALLOC_WASTE_IGN_BOUNDARY = 3*MEMORY_ALLOC_GRANULARITY;

	//activated via load seq macro: LOADING_DONE to only help lower fragmentation
	uint32 g_Activated = 0;
	//initialized via InitMemContainer()
	static int g_Initialized = 0;

	CryCriticalSectionNonRecursive g_MemCritSec;

	void Activate()
	{
		AUTO_LOCK_T(CryCriticalSectionNonRecursive, g_MemCritSec);
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
		MEMORY_ALLOC_BOUNDARY		= MEMORY_ALLOC_BOUNDARY_SMALL;
		MEMORY_ALLOC_WASTE_IGN	= MEMORY_ALLOC_WASTE_IGN_SMALL;
#else
		MEMORY_ALLOC_BOUNDARY		= MEMORY_ALLOC_BOUNDARY_LARGE;
		MEMORY_ALLOC_WASTE_IGN	= MEMORY_ALLOC_WASTE_IGN_LARGE;
#endif
		g_Activated = 1;
	}

	ILINE uint32 Activated(){return g_Activated;}

	ILINE uint32 AdjustSize(uint32 size)
	{
		//round up to multiple of 64 KB
		return (size + MEMORY_ALLOC_GRANULARITY - 1) & ~(MEMORY_ALLOC_GRANULARITY-1);
	}

	static std::vector<sys_addr_t> g_Addresses[4];	//addresses available for assigning
#if !defined(_RELEASE)
	uint32 g_Allocated;
	uint32 g_Used;
	uint32 g_NodeAllocated;
	uint32 g_AllocatedCount;
#endif

#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
	struct SMemHolder
	{
	#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
		SMemHolder():pairedMemEntryStorageIdx((TPairedMemIndex)-1){}
		SMemHolder(uint32 s, sys_addr_t p, TPairedMemIndex idx = (TPairedMemIndex)-1) : size(s),ptr(p),pairedMemEntryStorageIdx(idx){}
		TPairedMemIndex PairedMemEntryStorageIdx()const{return pairedMemEntryStorageIdx;}
		void PairedMemEntryStorageIdx(TPairedMemIndex idx){pairedMemEntryStorageIdx = idx;}
		bool HasPairedMemEntryStorageIdx()const{return pairedMemEntryStorageIdx != (TPairedMemIndex)-1;}
		explicit SMemHolder(void* p) : ptr((sys_addr_t)p),pairedMemEntryStorageIdx(-1){}
		uint32 Size(void* p)const;
	#else
		SMemHolder(uint32 s, sys_addr_t p) : size(s),ptr(p){}
		SMemHolder(){}
		explicit SMemHolder(void* p) : ptr((sys_addr_t)p){}
		ILINE uint32 Size(void*)const{return size;}
		TPairedMemIndex PairedMemEntryStorageIdx()const{return (TPairedMemIndex)-1;}
	#endif
		ILINE sys_addr_t Addr(){return ptr;}
		inline bool operator<(const SMemHolder& o)const{return ptr < o.ptr;}
	private:
		uint32 size;
		sys_addr_t ptr;
	#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
		TPairedMemIndex pairedMemEntryStorageIdx;
	#endif
	};
	typedef VectorSet<SMemHolder> TMappedAddrHolder;
	static TMappedAddrHolder g_MappedAddrSet;

#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
	//hold the info about the block pairs
	//idea: store 2 entries into 1 or 2 64K system blocks
	//use best fit to waste as little as possible
	struct SPairedMemEntry
	{
		static uint32 AlignSmall(uint32 s){return (s + 127) & ~127;}
		static const uint32 eBlockSizeShift = 30;
		static const uint32 eBlockSizeMask	= 1<<31 | 1<<30;
		uint32 available;
		int32 size0;
		int32 size1;
		sys_addr_t ptr;//also used to decide which one to reset

		ILINE uint32 Available()const
		{
			if(MEMORY_ALLOC_SMALL_LIMIT < MEMORY_ALLOC_GRANULARITY)return available;
			return available & ~eBlockSizeMask;
		}
		ILINE void Available(uint32 size)
		{
			if(MEMORY_ALLOC_SMALL_LIMIT < MEMORY_ALLOC_GRANULARITY)
				available = size;
			else
				available = available & eBlockSizeMask | size;
		}
		ILINE void AddAvailable(int32 add)
		{
			if(MEMORY_ALLOC_SMALL_LIMIT < MEMORY_ALLOC_GRANULARITY)
				available += add;
			else
				available = available & eBlockSizeMask | (uint32)((int)Available()+add);
		}
		ILINE uint32 BlockSizeUsed()const
		{
			if(MEMORY_ALLOC_SMALL_LIMIT < MEMORY_ALLOC_GRANULARITY)return MEMORY_ALLOC_GRANULARITY;//compile time constant
			return (1+(available>>eBlockSizeShift))*MEMORY_ALLOC_GRANULARITY;
		}
		ILINE uint32 SizeUsed()const
		{
			uint32 s=0; 
			s += (size0!=-1)?size0:0;
			s += (size1!=-1)?size1:0;
			return s;
		}
		ILINE uint32 BlockSizeUsed(uint32 size)
		{
			size = AdjustSize(size);
			available = size;
			if(MEMORY_ALLOC_SMALL_LIMIT >= MEMORY_ALLOC_GRANULARITY)
			{
				available |= (((size-MEMORY_ALLOC_GRANULARITY)/MEMORY_ALLOC_GRANULARITY)<<eBlockSizeShift);
				return size;
			}
			return MEMORY_ALLOC_GRANULARITY;
		}
		ILINE uint32 Elems()const
		{
			uint32 c=0; 
			c += (size0!=-1)?1:0;
			c += (size1!=-1)?1:0;
			return c;
		}
		uint32 Size(sys_addr_t p)
		{
			if(p < ptr && p >= ptr + BlockSizeUsed())SNPAUSE;
			return (p == ptr)?size0 : size1;
		}
		ILINE void Reset()
		{
			size0 = size1 = -1;
			ptr = available = 0;
		}
		ILINE bool CanFree(){return (size0 + size1 == -2);}
		SPairedMemEntry(){Reset();}
		ILINE bool operator<(const SPairedMemEntry& o)const 
		{
			uint32 avail0 = Available();
			uint32 avail1 = o.Available();
			if(avail0 != avail1)
				return avail0 < avail1; 
			return ptr < o.ptr;
		}
		uint32 FreeSmall(sys_addr_t p)
		{
			if(size0 == -1 && size1 == -1) SNPAUSE;
			uint32 s=0;
			if(ptr == p)
			{
				s = size0;
				AddAvailable(size0);
				size0 = -1;
			}
			else
			{
				s = size1;
				AddAvailable(size1);
				size1 = -1;
			}
			if(s==0)SNPAUSE;
			return s;
		}
		void* AllocateSmall(uint32 size)
		{
			size = AlignSmall(size);//ensure basic alignment
			if(size > Available() || (size0 != -1 && size1 != -1))SNPAUSE;
			AddAvailable(-size);
			void *pRet = NULL;
			if(size0 == -1){pRet = (void*)ptr; size0 = size;}
			else {size1 = size; pRet = (void*)(ptr+size0);}
			if(size0 != -1 && size1 != -1) Available(0);
			return pRet;
		}
	};
	SPairedMemEntry g_PairedMemEntryStorage[MEMORY_MAX_ALLOC_AMOUNT+1];
	std::vector<TPairedMemIndex> g_PairedMemEntryFreelist;

#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
	uint32 SMemHolder::Size(void* p)const
	{
		if(pairedMemEntryStorageIdx == (TPairedMemIndex)-1)return size;
		return g_PairedMemEntryStorage[pairedMemEntryStorageIdx].Size((sys_addr_t)p);
	}
#endif

	struct SPairEntrySortHelper
	{
		TPairedMemIndex index;
		ILINE bool operator <(const SPairEntrySortHelper& o)const
		{
			return g_PairedMemEntryStorage[index] < g_PairedMemEntryStorage[o.index];
		}

		ILINE const SPairedMemEntry& Entry() const{return g_PairedMemEntryStorage[index];}
	};
	typedef std::vector<SPairEntrySortHelper> TPairedMemEntrySorter;
	TPairedMemEntrySorter g_PairedMemEntrySorter;

	void* AllocateSmallBlock(sys_addr_t& virtAddr, uint32 size);
	void FreeSmallBlock(const SPairedMemEntry& entry);

	void* AllocateFromSmallPool(uint32 size, bool locked = false)
	{
		//get lower bound of allocation size, sorted by available amount
    if (!locked) g_MemCritSec.Lock(); 
		SPairedMemEntry& rHelper = g_PairedMemEntryStorage[MEMORY_MAX_ALLOC_AMOUNT];
		SPairEntrySortHelper helper = {MEMORY_MAX_ALLOC_AMOUNT};//index of rHelper
		rHelper.Available(size-1);
		TPairedMemEntrySorter::iterator curEnd = g_PairedMemEntrySorter.end();
		TPairedMemEntrySorter::iterator it = std::upper_bound(g_PairedMemEntrySorter.begin(),curEnd,helper);
#if !defined(_RELEASE)
		++g_AllocatedCount;
		g_Used += SPairedMemEntry::AlignSmall(size);
#endif
		void* pRet;
		if(it == curEnd)
		{
			//need to allocate a new block
			if(g_PairedMemEntryFreelist.empty())SNPAUSE;
			TPairedMemIndex newEntryIdx = *g_PairedMemEntryFreelist.rbegin();
			g_PairedMemEntryFreelist.pop_back();
			SPairedMemEntry& entry = g_PairedMemEntryStorage[newEntryIdx];
			sys_addr_t virtAddr;
			void *pAllocatedBlock = AllocateSmallBlock(virtAddr, entry.BlockSizeUsed(size));
			if(pAllocatedBlock == NULL)SNPAUSE;
			g_MappedAddrSet.insert(SMemHolder(entry.BlockSizeUsed(),virtAddr,newEntryIdx));
			SPairEntrySortHelper newHelperEntry = {newEntryIdx};
			g_PairedMemEntrySorter.push_back(newHelperEntry);
			entry.ptr = (sys_addr_t)pAllocatedBlock;
#if !defined(_RELEASE)
			g_Allocated += entry.BlockSizeUsed();
#endif
			pRet = entry.AllocateSmall(size);
		}
		else
		{
			SPairedMemEntry& entry = g_PairedMemEntryStorage[it->index];
			pRet = entry.AllocateSmall(size);
		}
		std::sort(g_PairedMemEntrySorter.begin(),g_PairedMemEntrySorter.end());
    if (!locked) g_MemCritSec.Unlock(); 
		return pRet;
	}

	void FreeFromSmallPool(void* p, TPairedMemIndex pairedMemEntryStorageIdx)
	{
		SPairedMemEntry& entry = g_PairedMemEntryStorage[pairedMemEntryStorageIdx];
		uint32 size = entry.FreeSmall((sys_addr_t)p);
#if !defined(_RELEASE)
		g_Used -= size;
#endif
		if(entry.CanFree())
		{
#if !defined(_RELEASE)
			g_Allocated -= entry.BlockSizeUsed();
#endif
			FreeSmallBlock(entry);
			entry.Reset();
			g_PairedMemEntryFreelist.push_back(pairedMemEntryStorageIdx);
			const TPairedMemEntrySorter::iterator end = g_PairedMemEntrySorter.end();
			TPairedMemEntrySorter::iterator it=g_PairedMemEntrySorter.begin();
			for(;it!=end;++it)
			{
				if(it->index == pairedMemEntryStorageIdx)
				{
					g_PairedMemEntrySorter.erase(it);
					break;
				}
			}
			if(it == end)SNPAUSE;
		}
		else
			std::sort(g_PairedMemEntrySorter.begin(),g_PairedMemEntrySorter.end());
#if !defined(_RELEASE)
		--g_AllocatedCount;
#endif
	}
#endif//PS3_USE_SYSTEM_MEM_CONTAINER_SMALL

	ILINE bool IsSmallSystemAlloc(uint32 size)
	{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
		return size <= MEMORY_ALLOC_SMALL_LIMIT && Activated();
#else
		return false;
#endif
	}

#endif
	void InitMemContainer()
	{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
	#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
		g_PairedMemEntrySorter.reserve(MEMORY_MAX_ALLOC_AMOUNT);
		g_PairedMemEntryFreelist.resize(MEMORY_MAX_ALLOC_AMOUNT);
		for(uint32 i=0; i<MEMORY_MAX_ALLOC_AMOUNT; ++i)
		{
			g_PairedMemEntryStorage[i].Reset();
			g_PairedMemEntryFreelist[i] = MEMORY_MAX_ALLOC_AMOUNT-i-1;//for better cache usage, apply decrementalwise
		}
		g_PairedMemEntryStorage[MEMORY_MAX_ALLOC_AMOUNT].ptr = 0xFFFFFFFF;//dummy object
#endif
		g_MappedAddrSet.reserve(MEMORY_MAX_ALLOC_AMOUNT);//must not resize as it would get here
		int ret = sys_mmapper_allocate_address
		(	
			MEMORY_ADDR_RANGE, SYS_MEMORY_GRANULARITY_64K | SYS_MEMORY_ACCESS_RIGHT_PPU_THR | 
			SYS_MEMORY_ACCESS_RIGHT_SPU_THR | SYS_MEMORY_ACCESS_RIGHT_RAW_SPU,
			0x0UL, &g_BaseAddress
		);
		if( ret != CELL_OK ) 
		{
			PrintSysErrorMessage(ret,"sys_mmapper_allocate_address");
			SNPAUSE;
		}
		else
		{
			//generate address tables
			g_Addresses[0].reserve(MEMORY_ADDR_IDX_0_CNT);
			g_Addresses[1].reserve(MEMORY_ADDR_IDX_1_CNT);
			g_Addresses[2].reserve(MEMORY_ADDR_IDX_2_CNT);
			g_Addresses[3].reserve(MEMORY_ADDR_IDX_3_CNT);
			sys_addr_t curAddr = g_BaseAddress;
			for(uint32 i=0; i<MEMORY_ADDR_IDX_0_CNT; ++i)
			{
				g_Addresses[0].push_back(curAddr);	curAddr += MEMORY_ADDR_IDX_0_SIZE;
			}			
			for(uint32 i=0; i<MEMORY_ADDR_IDX_1_CNT; ++i)
			{
				g_Addresses[1].push_back(curAddr);	curAddr += MEMORY_ADDR_IDX_1_SIZE;
			}			
			for(uint32 i=0; i<MEMORY_ADDR_IDX_2_CNT; ++i)
			{
				g_Addresses[2].push_back(curAddr);	curAddr += MEMORY_ADDR_IDX_2_SIZE;
			}			
			for(uint32 i=0; i<MEMORY_ADDR_IDX_3_CNT; ++i)
			{
				g_Addresses[3].push_back(curAddr);	curAddr += MEMORY_ADDR_IDX_3_SIZE;
			}			
			if(curAddr - g_BaseAddress > MEMORY_ADDR_RANGE)SNPAUSE;
			printf("Initialized system allocator (0x%08x ... 0x%08x)\n",g_BaseAddress,curAddr);
			g_Initialized = 1;
		}
#endif
	}

	ILINE uint32 AddressVecIndex(uint32 size)
	{
		return (size <= MEMORY_ADDR_IDX_0_SIZE)?0:(size <= MEMORY_ADDR_IDX_1_SIZE)?1:(size <= MEMORY_ADDR_IDX_2_SIZE)?2:3;
	}
		
	void DestroyMemContainer()
	{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
		if(g_Initialized)
			sys_mmapper_free_address(g_BaseAddress);
		g_Activated = g_Initialized = 0;
#endif
	}
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
	ILINE bool CanUse()
	{
		return g_Initialized;
	}

	ILINE bool AllocateByCont(uint32 size)
	{
		if(size < MEMORY_ALLOC_BOUNDARY || size > MEMORY_ALLOC_UPPER_BOUNDARY ||	
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
			(!IsSmallSystemAlloc(size) && size <= MEMORY_ALLOC_WASTE_IGN_BOUNDARY && (AdjustSize(size)-size) >= MEMORY_ALLOC_WASTE_IGN) ||
#else
			(size <= MEMORY_ALLOC_WASTE_IGN_BOUNDARY && (AdjustSize(size)-size) >= MEMORY_ALLOC_WASTE_IGN) ||
#endif
			g_MappedAddrSet.size() >= MEMORY_MAX_ALLOC_AMOUNT)
			return false;
		return true;
	}

	ILINE TMappedAddrHolder::iterator FindMappedAddr(void *ptr)
	{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
		TMappedAddrHolder::iterator it = std::lower_bound(g_MappedAddrSet.begin(),g_MappedAddrSet.end(),SMemHolder((void*)((sys_addr_t)ptr+1)));
		if(it == g_MappedAddrSet.begin())SNPAUSE;
		TMappedAddrHolder::iterator ret = it-1;
	#ifdef ENABLE_DEBUG
		if((uint32)ptr != ret->Addr())
		{
			if(!ret->HasPairedMemEntryStorageIdx())SNPAUSE;
			if((uint32)ptr > (uint32)g_PairedMemEntryStorage[ret->PairedMemEntryStorageIdx()].ptr + (uint32)g_PairedMemEntryStorage[ret->PairedMemEntryStorageIdx()].BlockSizeUsed())SNPAUSE;
		}
	#endif
		return ret;
#else
		return g_MappedAddrSet.find(SMemHolder(ptr));
#endif
	}

	ILINE bool InRange(void *ptr)
	{
		return (sys_addr_t)ptr >= g_BaseAddress && (sys_addr_t)ptr < g_BaseAddress + MEMORY_ADDR_RANGE;
	}

	ILINE bool InRange(void *ptr, uint32& rSize)
	{
		if(InRange(ptr))
		{
			TMappedAddrHolder::iterator it = FindMappedAddr(ptr);
			rSize = (it->Size(ptr));
			return true;
		}
		return false;
	}

	ILINE void* AllocateMappedAddr(sys_addr_t addr, uint32 size)
	{
		sys_memory_t	memory_id;
		int ret = sys_mmapper_allocate_memory(size, SYS_MEMORY_GRANULARITY_64K, &memory_id);
		if( ret != CELL_OK ) 
		{
			PrintSysErrorMessage(ret,"sys_mmapper_allocate_memory");
//			gEnv->bIsOutOfMemory = true;
			SNPAUSE;
			return NULL;
		}
		ret = sys_mmapper_map_memory(addr, memory_id, SYS_MEMORY_PROT_READ_WRITE);
		if( ret != CELL_OK ) 
		{
			PrintSysErrorMessage(ret,"sys_mmapper_map_memory");
		  sys_mmapper_free_memory(memory_id);
//			gEnv->bIsOutOfMemory = true;
			SNPAUSE;
			return NULL;
		}	
		return (void*)addr;
	}

	ILINE void FreeMappedAddr(sys_addr_t addr, uint32 idx)
	{
		sys_memory_t	memory_id;
		int ret = sys_mmapper_unmap_memory(addr, &memory_id);
		if(ret != CELL_OK)
		{
			printf("FreeMappedAddr(0x%08x) sys_mmapper_unmap_memory failed(0x%08x):\n",addr,ret);
			SNPAUSE;
			return;
		}
		ret = sys_mmapper_free_memory(memory_id);
		if( ret != CELL_OK ) 
		{
			printf("FreeMappedAddr(0x%08x) sys_mmapper_free_memory failed(0x%08x):\n",addr,ret);
			SNPAUSE;
			return;
		}
		g_Addresses[idx].push_back((sys_addr_t)addr);
	}

#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
	ILINE void* AllocateSmallBlock(sys_addr_t& virtAddr, uint32 size)
	{
		int idx = 0;
		//fetch address
		uint32 s = g_Addresses[idx].size();
		if(s == 0)SNPAUSE;
		virtAddr = g_Addresses[idx][s-1];
		void* pRet = AllocateMappedAddr(virtAddr, size);
		if(pRet)
			g_Addresses[idx].pop_back();
		return pRet;
	}

	ILINE void FreeSmallBlock(const SPairedMemEntry& entry)
	{
		void *p = (void*)entry.ptr;
		TMappedAddrHolder::iterator it = FindMappedAddr(p);
		FreeMappedAddr((sys_addr_t)p, 0);
		g_MappedAddrSet.erase(it);
	}
#endif

	inline void* Allocate(uint32 size, uint32 alignment, bool locked = false)
	{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
		if(IsSmallSystemAlloc(size))
			return AllocateFromSmallPool(size, locked);//alignment always 128 bytes
#endif
		//alignment is only used in case we fail to map an address
		uint32 sizeAdjusted = AdjustSize(size);
		int idx = AddressVecIndex(size);
		//fetch address
    if (!locked) g_MemCritSec.Lock(); 
		uint32 s = g_Addresses[idx].size();
		if(s == 0)
		{
			printf("sys_mmapper_allocate_address runs out of addresses, defaulting to sys malloc(size=%d kb)\n",sizeAdjusted>>10);
			if (!locked) g_MemCritSec.Unlock(); 
			return (alignment <= MEMORY_DEFAULT_ALIGN)?__REAL__malloc(size) : __REAL__memalign(alignment, size);
		}
		sys_addr_t virtAddr;
		virtAddr = g_Addresses[idx][s-1];
		void *pRet = AllocateMappedAddr(virtAddr, sizeAdjusted);
		if(pRet)
		{
			g_Addresses[idx].pop_back();
			g_MappedAddrSet.insert(SMemHolder(size,virtAddr));
#if !defined(_RELEASE)
			++g_AllocatedCount;
			g_Allocated += sizeAdjusted;
			g_Used += size;
#endif
      if (!locked) g_MemCritSec.Unlock(); 
			return pRet;
		}
		else
			pRet =	__REAL__memalign(alignment, size);
    if (!locked) g_MemCritSec.Unlock(); 
		return pRet;
	}

  void* ExternAllocate(uint32 size, uint32 alignment)
	{ 
		void *ret = NULL;
		if(size >= NMemCont::MEMORY_ALLOC_EXTERN_MIN_SIZE)
    {
      ret = Allocate(size, alignment); 
#if defined(SUPP_MTRACE)
      MTrace::trace_malloc(ret, size);
#endif 
    }
		else
			ret = (alignment > NMemCont::MEMORY_DEFAULT_ALIGN)?__REAL__memalign(alignment, size) : __REAL__malloc(size);			

		PS3CrySizerHeapChecker::AddMemory( ret, size );
		return ret;
	}

	inline bool Free(void *p)
	{
    AUTO_LOCK_T(CryCriticalSectionNonRecursive, g_MemCritSec);
    if(InRange(p))
    {
      TMappedAddrHolder::iterator it = FindMappedAddr(p);
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
      if(it->HasPairedMemEntryStorageIdx())
        FreeFromSmallPool(p,it->PairedMemEntryStorageIdx());
      else
#endif
      {
        const uint32 s = it->Size(p);
        FreeMappedAddr((sys_addr_t)p, AddressVecIndex(s));
#if !defined(_RELEASE)
        uint32 sizeAdjusted = AdjustSize(s);
        g_Allocated -= sizeAdjusted;
        g_Used -= s;
        --g_AllocatedCount;
#endif
        g_MappedAddrSet.erase(it);
      }
      return true;
    }
    return false; 
	}

  bool ExternFree(void *p)
  { 
    if(CanUse())
    {
      uint32_t s = 0; 
      if(InRange(p, s))
      {
#if defined(SUPP_MTRACE)
        MTrace::trace_free(p, s);
#endif
      }
    }
    PS3CrySizerHeapChecker::RemoveMemory( p );
    return Free(p);
  }

	inline void* Realloc(void *ptr, size_t size, size_t alignment = MEMORY_DEFAULT_ALIGN)
	{
		AUTO_LOCK_T(CryCriticalSectionNonRecursive, g_MemCritSec);
		if(!NMemCont::InRange(ptr))
		{
			if(!NMemCont::AllocateByCont(size) || alignment > NMemCont::MEMORY_ALLOC_GRANULARITY)
				return (alignment <= MEMORY_DEFAULT_ALIGN)?__REAL__realloc(ptr,size) : __REAL__reallocalign(ptr,size,alignment);
			uint32 existSize = malloc_usable_size(ptr);
			void *pNewPtr = Allocate(size,alignment, true);
			if(existSize != 0 && pNewPtr)
				memcpy(pNewPtr, ptr, std::min(existSize,size));
#     if defined(MEMSET_FREE_PATTERN)
      memset(ptr, MEMSET_FREE_PATTERN, existSize);
#     endif 
			__REAL__free(ptr);
			return pNewPtr;
		}
		else
		{
			TMappedAddrHolder::iterator it = FindMappedAddr(ptr);
			uint32 existSize = it->Size(ptr);
			void *pNewPtr = NULL;
			TPairedMemIndex pairedMemEntryStorageIdx = it->PairedMemEntryStorageIdx();
			if(pairedMemEntryStorageIdx == (TPairedMemIndex)-1)//otherwise erased inside FreeFromSmallPool
				g_MappedAddrSet.erase(it);//do it before it becomes invalid due to insertion op
			if(!NMemCont::AllocateByCont(size) || alignment > NMemCont::MEMORY_ALLOC_GRANULARITY)
				pNewPtr = (alignment <= NMemCont::MEMORY_DEFAULT_ALIGN)?__REAL__malloc(size) : __REAL__memalign(alignment,size);
			else
				pNewPtr = Allocate(size, alignment, true);
			if(pNewPtr != NULL)
				memcpy(pNewPtr, ptr, std::min(existSize,size)); // copy only min, for the case that a memory area is shrinked to prevent a memory overwrite
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER_SMALL
			if(pairedMemEntryStorageIdx != (TPairedMemIndex)-1)
				FreeFromSmallPool(ptr,pairedMemEntryStorageIdx);
			else
#endif
			{
				FreeMappedAddr((sys_addr_t)ptr, AddressVecIndex(existSize));
	#if !defined(_RELEASE)
				uint32 sizeAdjusted = AdjustSize(existSize);
				g_Allocated -= sizeAdjusted;
				g_Used -= existSize;
				--g_AllocatedCount;
	#endif
			}
			return pNewPtr;
		}
	}
#endif
}

// enforce symbol linkage by empty asm statements 
namespace
{
	static struct EnforceSymbolLinkage
	{
		EnforceSymbolLinkage()
		{
			__asm__ __volatile__ ("" : : "r" (__REAL__calloc) :);
			__asm__ __volatile__ ("" : : "r" (__REAL__malloc) :);
			__asm__ __volatile__ ("" : : "r" (__REAL__free) : );
			__asm__ __volatile__ ("" : : "r" (__REAL__realloc) :);
			__asm__ __volatile__ ("" : : "r" (__REAL__reallocalign) :);
			__asm__ __volatile__ ("" : : "r" (__REAL__memalign) :);
			__asm__ __volatile__ ("" : : "r" (__REAL___malloc_init) :);
			__asm__ __volatile__ ("" : : "r" (__REAL___malloc_finalize) :);
		}
	} _EnforceSymbolLinkage;
};

size_t _msize_nosystem(void *_p)
{
#if !defined(PS3_WRAP_NODEALLOC)
	return _msize(_p);
#else
	if (!_p) return 0; 
	size_t size = g_GlobPageBucketAllocator.getSizeEx(_p); 
	if (size) 
		return size;
	return malloc_usable_size(_p);
#endif
	return 0;
}

size_t _msize(void *_p)
{
#if !defined(PS3_WRAP_NODEALLOC)
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
	if(NMemCont::CanUse())
	{
		AUTO_LOCK_T(CryCriticalSectionNonRecursive, NMemCont::g_MemCritSec);
		uint32 s;
		if(NMemCont::InRange(_p, s))
			return s;
	}
#endif
  return malloc_usable_size(_p);
#else
	if (!_p) return 0; 
	size_t size = g_GlobPageBucketAllocator.getSizeEx(_p); 
	if (size) 
		return size;
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
	if(NMemCont::CanUse())
	{
		AUTO_LOCK_T(CryCriticalSectionNonRecursive, NMemCont::g_MemCritSec);
		uint32 s;
		if(NMemCont::InRange(_p, s))
			return s;
	}
#endif
  return malloc_usable_size(_p);
#endif
	return 0;
}

extern "C" 	void _malloc_init(void)
{
  __REAL___malloc_init();
}

extern "C" void _malloc_finalize(void)
{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
	NMemCont::DestroyMemContainer();
#endif
  __REAL___malloc_finalize();
}

extern "C" void *calloc(size_t nmemb, size_t size)
{
	void *ptr = NULL;
#if defined(PS3_WRAP_NODEALLOC)
  if ((nmemb*size) > (size_t)__MAX_BYTES)
#endif
	{
		ptr = __REAL__calloc(nmemb, size);
	}
#if defined(PS3_WRAP_NODEALLOC)
	else
	{
		size = size==0?1:size;
		ptr = g_GlobPageBucketAllocator.alloc((nmemb*size)); 
		std::memset(ptr, 0, (nmemb*size));
	}
#endif
	if (!ptr) 
		PrintAllocFailureMessage(nmemb*size);
#if defined(SUPP_MTRACE)
	MTrace::trace_calloc(ptr, _msize(ptr));		
#endif 

	// remember allocated memory for crysizer check
	PS3CrySizerHeapChecker::AddMemory( ptr, nmemb*size );

	return ptr;
}

extern "C" void *malloc_non_system(size_t size)
{
#if !defined(_RELEASE)
	NMemCont::g_NodeAllocated += size;
#endif
	//bypass system container if possible
	void *ptr = __REAL__malloc(size);
	if (!ptr) 
	{ 
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
		//we haven't got any block form the dlmalloc 1 MB pages, try again with system container
		if(NMemCont::CanUse() && NMemCont::AllocateByCont(size))
			ptr = NMemCont::Allocate(size,NMemCont::MEMORY_DEFAULT_ALIGN);
		if (!ptr)
#endif
			PrintAllocFailureMessage(size);
	}
#if defined(SUPP_MTRACE)
	MTrace::trace_malloc(ptr, _msize(ptr));
#endif
	PS3CrySizerHeapChecker::AddMemory( ptr, size );
	return ptr;
}

extern "C" void *malloc(size_t size)
{
	void *ptr = NULL;
#if defined(PS3_WRAP_NODEALLOC) 
	if (size > (size_t)__MAX_BYTES)
#endif
	{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
		if(NMemCont::CanUse() && NMemCont::AllocateByCont(size))
		{
			ptr = NMemCont::Allocate(size,NMemCont::MEMORY_DEFAULT_ALIGN);
		}
		else
#endif
			ptr = __REAL__malloc(size);
	}
#if defined(PS3_WRAP_NODEALLOC) 
	else
	{
		ptr = g_GlobPageBucketAllocator.alloc(size==0?1:size); 
	}
#endif
	if (!ptr) 
		PrintAllocFailureMessage(size);
#if defined(SUPP_MTRACE)
	MTrace::trace_malloc(ptr, _msize(ptr));
#endif
	PS3CrySizerHeapChecker::AddMemory( ptr, size );
	return ptr;
}

extern "C" void free(void *ptr)
{
	PS3CrySizerHeapChecker::RemoveMemory( ptr );

#if defined(SUPP_MTRACE)
	MTrace::trace_free(ptr, _msize(ptr));
#endif
	if (ptr == NULL) return; 
#if defined(PS3_WRAP_NODEALLOC) 
	size_t size = g_GlobPageBucketAllocator.getSizeEx(ptr); 
	if (size) 
	{
		g_GlobPageBucketAllocator.dealloc(ptr,size); 
	}
	else
#endif
	{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
    bool freed = false; 
		if (NMemCont::CanUse())
    {
      freed = NMemCont::Free(ptr);
    }
    if (!freed)
    {
#     if defined(MEMSET_FREE_PATTERN)
      memset(ptr, MEMSET_FREE_PATTERN, malloc_usable_size(ptr));
#     endif 
      __REAL__free(ptr);
    }
#else
#   if defined(MEMSET_FREE_PATTERN)
    memset(ptr, MEMSET_FREE_PATTERN, malloc_usable_size(ptr));
#   endif 
		__REAL__free(ptr);
#endif 
	}
}

extern "C" void *realloc(void *ptr, size_t size)
{
	if (ptr == NULL)
		return malloc(size); 

	if (size == 0) 
	{ 
		free(ptr);
		return NULL;
	}
	void* ptr2 = NULL;
#if defined(PS3_WRAP_NODEALLOC) 
	size_t oldsize = g_GlobPageBucketAllocator.getSizeEx(ptr); 
	if (oldsize) 
	{ 
		ptr2 = malloc(size); 
		memmove(ptr2, ptr, (oldsize<size) ? oldsize : size);
		free(ptr); 
	} 
	else
#endif
	{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
    const size_t old_size = (long)_msize(ptr);
		PS3CrySizerHeapChecker::RemoveMemory( ptr );
    if (NMemCont::CanUse())
    {
      ptr2 = NMemCont::Realloc(ptr, size);
    }
    if (!ptr2) 
    { 
      ptr2 = __REAL__realloc(ptr, size);
    }
#else
		const size_t old_size = (long)_msize(ptr);
		ptr2 = __REAL__realloc(ptr, size);
#endif
#if defined(SUPP_MTRACE)
		MTrace::trace_realloc(ptr, ptr2, old_size, _msize(ptr2))
#endif
	}
	if (!ptr2) 
		PrintAllocFailureMessage(size);

	PS3CrySizerHeapChecker::AddMemory( ptr2, size );
	return ptr2;
}

extern "C" void *reallocalign(void* ptr, size_t size, size_t alignment)
{
	if (ptr == NULL)
		return (alignment > NMemCont::MEMORY_DEFAULT_ALIGN)?memalign(alignment, size) : malloc(size);

	if (size == 0) 
	{
		free(ptr);
		return NULL; 
	}
	void *ptr2 = NULL; 
#if defined(PS3_WRAP_NODEALLOC) 
	size_t oldsize = g_GlobPageBucketAllocator.getSizeEx(ptr);
	if (oldsize) 
	{
		ptr2 = (alignment > NMemCont::MEMORY_DEFAULT_ALIGN)?memalign(alignment, size) : malloc(size);
		memmove(ptr2, ptr, (oldsize<size) ? oldsize : size);
		free(ptr); 
	} 
	else  
#endif
	{		
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
    const size_t old_size = (long)_msize(ptr);
		PS3CrySizerHeapChecker::RemoveMemory( ptr );	
    if (NMemCont::CanUse()) 
    { 
      ptr2 = NMemCont::Realloc(ptr, size, alignment);
    }
    else 
    { 
      ptr2 = (alignment > NMemCont::MEMORY_DEFAULT_ALIGN)?__REAL__reallocalign(ptr, size, alignment) : __REAL__realloc(ptr, size);
    } 
#else
		const size_t old_size = (long)_msize(ptr);
		ptr2 = (alignment > NMemCont::MEMORY_DEFAULT_ALIGN)?__REAL__reallocalign(ptr, size, alignment) : __REAL__realloc(ptr, size);
#endif
#if defined(SUPP_MTRACE)
		MTrace::trace_realloc(ptr, ptr2, old_size, _msize(ptr2));
#endif
	}
	if (!ptr2) 
		PrintAllocFailureMessage(size);
	PS3CrySizerHeapChecker::AddMemory( ptr2, size );
	return ptr2;
}

extern "C" void *memalign(size_t boundary, size_t size)
{
	if(boundary <= NMemCont::MEMORY_DEFAULT_ALIGN)
		return malloc(size);

	void *ptr = NULL;
#if defined(PS3_WRAP_NODEALLOC) 
	if (size > __MAX_BYTES || boundary > _ALIGNMENT)
#endif
	{
#ifdef PS3_USE_SYSTEM_MEM_CONTAINER
		if(NMemCont::CanUse() && NMemCont::AllocateByCont(size) && boundary <= NMemCont::MEMORY_ALLOC_GRANULARITY)
		{
			ptr = NMemCont::Allocate(size, boundary);
		}
		else
#endif
			ptr = __REAL__memalign(boundary, size);
	}
#if defined(PS3_WRAP_NODEALLOC) 
	else 
	{
		ptr = g_GlobPageBucketAllocator.alloc(size==0?1:size); 
	}
#endif
	if (!ptr) 
		PrintAllocFailureMessage(size);
#if defined(SUPP_MTRACE)
	MTrace::trace_memalign(ptr, boundary, _msize(ptr));
#endif
	PS3CrySizerHeapChecker::AddMemory( ptr, size );
	return ptr;
}

extern "C" size_t memGetFreeInBucketAllocator(void)
{
#if !defined(PS3_DONT_USE_NODEALLOC) && defined(PS3_WRAP_NODEALLOC)
	return g_GlobPageBucketAllocator._S_get_free();
#else
	return 0;
#endif
}

namespace NMemCont
{
	uint32 WastedInNodeBlocks()
	{
#if defined(PS3_USE_SYSTEM_MEM_CONTAINER_SMALL) && defined(ENABLE_DEBUG)
		printf("---------------------------SMALL BLOCKs---------\n");
		const TPairedMemEntrySorter::iterator end = g_PairedMemEntrySorter.end();
		TPairedMemEntrySorter::iterator it=g_PairedMemEntrySorter.begin();
		uint32 allocated = 0,used = 0,count=0,countFull=0;
		for(;it!=end;++it)
		{
			const SPairedMemEntry& e = it->Entry();
			uint32 u = e.SizeUsed();
			printf("num=%d  block=%d  KB  used=%d KB\n",e.Elems(),e.BlockSizeUsed()>>10,u>>10);
			allocated += e.BlockSizeUsed()>>10;
			used += u>>10;
			countFull += e.Elems();
			++count;
		}
		printf("\ncount=%d(%d)  alloc=%d  used=%d KB\n\n",count,countFull,allocated,used);
		allocated = 0;
		used=0;
		count = 0;
		printf("---------------------------LARGE BLOCKs---------\n");
		for(TMappedAddrHolder::iterator it = g_MappedAddrSet.begin(); it!=g_MappedAddrSet.end();++it)
		{
			const SMemHolder& e = *it;
			if(!e.HasPairedMemEntryStorageIdx())
			{
				uint32 size = e.Size((void*)NULL)>>10;
				printf("large=%d KB\n",size);
				used += size;
				allocated += (size + 63)&~63;
				++count;
			}
		}
		printf("\ncount=%d  alloc=%d  used=%d KB\n\n",count,allocated,used);
#endif
#if defined(PS3_WRAP_NODEALLOC) 
		return g_GlobPageBucketAllocator.get_wasted_in_blocks();
#else
		return 0;
#endif
	}
}


namespace PS3CrySizerHeapChecker 
{
#if defined(PS3_CRYSIZER_HEAP_TRAVERSAL)
	// structure to generate and store a callstack
	struct CallStack
	{
		void Init()
		{		
			nCallStackEntries = 0;
			memset( arrCallStack, 0, sizeof(arrCallStack));

			register uint64_t sp, ip;

			// Fetch current stack pointer.
			__asm__ __volatile__ (
				"    mr %[sp], 1\n"
				: [sp] "=r" (sp));

			while (true)
			{
				// Locate previous frame.
				uint64_t prevSp = *(uint64_t *)(uint32_t)(sp);
				if (prevSp == 0) break;
				// Load return address from previous frame.
				ip = *((uint64_t *)(uint32_t)prevSp + 2);
				sp = prevSp;

				arrCallStack[nCallStackEntries++] = static_cast<uint32_t>(ip);
				if( nCallStackEntries == MAX_CALLSTACK_DEPTH )
					break;
			}
		}

		bool operator<( const CallStack &rOther ) const
		{			
			// two different callstack should rarely sum up to the exact same value :)
			uint32 sumA = 0;
			uint32 sumB = 0;
			for ( int i = 0 ; i < nCallStackEntries ; ++i)
				sumA += arrCallStack[i];
			for ( int i = 0 ; i < rOther.nCallStackEntries ; ++i)
				sumB += rOther.arrCallStack[i];

			return sumA < sumB;
		}

		static const uint MAX_CALLSTACK_DEPTH =30;
		uint32 nCallStackEntries;
		uint32 arrCallStack[MAX_CALLSTACK_DEPTH];
	};

	// struct to represent an allocated memory block
	struct Entry
	{

		void Init( void *ptr, uint32 size, uint32 id, const CallStack *callStack )
		{
			nPtr = (uint32)ptr;
			nSize = size;
			pCallStack = callStack;
			pUserData = NULL;
			nId = id;
			nTimeDiff = -1;
		}

		// Init without callstack (uses as a search key)
		void InitFast( void *ptr  )
		{
			nPtr = (uint32)ptr;
			nSize = 0;
			pCallStack = NULL;
		}

		bool operator<(const Entry &rOther ) const
		{
			return nPtr < rOther.nPtr;
		}

		uint32 nPtr;
		uint32 nSize;
		uint32 nId;
		int nTimeDiff;
		const CallStack *pCallStack;
		void *pUserData;		
	};

	// util struct to generate freelists in the allocators
	struct FreeList
	{
		FreeList *pNext;
	};

	// base extra heap allocator, used to allocated 1MB pages for the other allocators
	namespace  BaseAllocator
	{	
		static sys_addr_t	m_BaseAddress =0;
		static sys_addr_t	m_CurAddress =0;		
		static bool bIsInitialied = false;
		void Init()
		{			
			if( bIsInitialied )
				return;

			// get address space for log array
			if( sys_mmapper_allocate_address(	0x10000000UL, SYS_MEMORY_GRANULARITY_1M | SYS_MEMORY_ACCESS_RIGHT_PPU_THR | 
				SYS_MEMORY_ACCESS_RIGHT_SPU_THR | SYS_MEMORY_ACCESS_RIGHT_RAW_SPU, 0x0UL, &m_BaseAddress ) != CELL_OK )
			{
				snPause();
			}

			m_CurAddress = m_BaseAddress;
			bIsInitialied= true;
		}

		void *Allocate()
		{
			BaseAllocator::Init();
			int ret = 0;
			sys_memory_t	memory_id;
			// get memory space for log array
			ret = sys_mmapper_allocate_memory(1*1024*1024, SYS_MEMORY_GRANULARITY_1M, &memory_id);
			if( ret != CELL_OK )
			{
				PrintSysErrorMessage(ret,"sys_mmapper_allocate_memory");
				snPause();
			}
			// map memory
			ret = sys_mmapper_map_memory(m_CurAddress, memory_id, SYS_MEMORY_PROT_READ_WRITE);
			if( ret != CELL_OK ) snPause();

			void *retPtr = (void*)m_CurAddress;
			m_CurAddress += 1*1024*1024;
			return retPtr;
		}	
	} 


//__________________________________________CrySizer Pool Tool_______________________________________________________



	// special allocator, using its own range(shared over all containers, to allow allocation memory without going over
	// the normal memory allocation functions
	template <class T> 
	class PrivateHeapAllocator 
	{
	public:
		typedef size_t    size_type;
		typedef ptrdiff_t difference_type;
		typedef T*        pointer;
		typedef const T*  const_pointer;
		typedef T&        reference;
		typedef const T&  const_reference;
		typedef T         value_type;
		template <class U> struct rebind { typedef PrivateHeapAllocator<U>
		other; };

		PrivateHeapAllocator() throw() : m_pFreeList(NULL), m_pCurBlock(NULL), m_pCurPos(NULL) { }
		PrivateHeapAllocator(const PrivateHeapAllocator&) throw(): m_pFreeList(NULL), m_pCurBlock(NULL), m_pCurPos(NULL) {}
		template <class U> PrivateHeapAllocator(const PrivateHeapAllocator<U>&) throw() : m_pFreeList(NULL), m_pCurBlock(NULL), m_pCurPos(NULL) {}
		~PrivateHeapAllocator() throw(){}

		//pointer address(reference x) const;
		//const_pointer address(const_reference x) const;

		pointer allocate(size_type n)
		{
			void *ret = NULL;
			if( m_pFreeList ) // can we use the freelist
			{
				ret = m_pFreeList;
				m_pFreeList = m_pFreeList->pNext;
			}
			else if( m_pCurBlock && (m_pCurPos + sizeof(T)) < m_pCurBlock + 1 *1024*1024 ) // free space in the block
			{
				ret = m_pCurPos;
				m_pCurPos += sizeof(T);
			}
			else
			{
				m_pCurBlock = (char*)BaseAllocator::Allocate();
				m_pCurPos = m_pCurBlock + sizeof(T);
				ret = m_pCurBlock;
			}

			return (pointer)ret;				
		}

		void deallocate(pointer p, size_type n)
		{
			if( n != 1 ) snPause();

			FreeList *pFree = (FreeList*)p;

			pFree->pNext = m_pFreeList;
			m_pFreeList = pFree;
		}

		//size_type max_size() const throw();

		void construct(pointer p, T& val) { new(p) T(val); }
		void destroy(pointer p) { p->~T(); }

		FreeList *m_pFreeList;
		char *m_pCurBlock;
		char *m_pCurPos;
	};

	// global sets to track memory ops and callstacks
	typedef std::set<CallStack, std::less<CallStack>, PrivateHeapAllocator<CallStack> > CallStackSetT;
	typedef std::set<CallStack, std::less<CallStack>, PrivateHeapAllocator<CallStack> >::iterator CallStackIt;
	typedef std::set<Entry, std::less<Entry>, PrivateHeapAllocator<Entry> > EntrySetT;
	typedef std::set<Entry, std::less<Entry>, PrivateHeapAllocator<Entry> >::iterator EntryIt;

	CallStackSetT& CallStacks()
	{
		static CallStackSetT singleton;
		return singleton;
	}

	EntrySetT& MemoryOps()
	{
		static EntrySetT singleton;
		return singleton;
	}

	EntrySetT& MemoryOpsTemp()
	{
		static EntrySetT singleton;
		return singleton;
	}

	CryCriticalSection g_Lock;

	// some counter to track which memory got removed
	static uint32 nRemovedMemRefs = 0;
	static uint32 nRemovedMemory = 0;
	static uint32 g_nHoldedMemory = 0;
	static uint32 nAllocationCounter = 0;
	static bool bCrySizerTraversel = false;

	// log memory allocator
	void AddMemory( void *ptr, uint32 nSize )
	{
		AUTO_LOCK_T(CryCriticalSection, PS3CrySizerHeapChecker::g_Lock);
		
		// don't log memory during CrySizer traversal
		if( bCrySizerTraversel ) return;

		CallStack callstack = {0};
		callstack.Init();

		CallStackIt callStackIt = CallStacks().find(callstack);
		if( callStackIt == CallStacks().end() )
		{		
			CallStacks().insert(callstack);
			callStackIt = CallStacks().find(callstack);
		}

		PS3CrySizerHeapChecker::Entry entry = {0};
		entry.Init( ptr, nSize, nAllocationCounter, &(*callStackIt) );		
		MemoryOps().insert(entry);
		nAllocationCounter++;

		g_nHoldedMemory += entry.nSize;
	}

	void RemoveMemory( void *ptr ) 
	{
		// log memory deallocation
		AUTO_LOCK_T(CryCriticalSection, PS3CrySizerHeapChecker::g_Lock);
	
		// don't log memory during CrySizer traversal
		if( bCrySizerTraversel ) return;

		// find entry for this ptr
		Entry entry = {0};
		entry.InitFast(ptr);

		EntryIt entryIt = MemoryOps().find(entry);		
		if( entryIt == MemoryOps().end() ) return;
		MemoryOps().erase( entryIt );		

		g_nHoldedMemory -= entryIt->nSize;
	}

	// Util functions for the CrySizer
	void CrySizerStart()
	{
		g_Lock.Lock();
		bCrySizerTraversel = true;

		// create copy of all memory ops to work on		
		MemoryOpsTemp() = MemoryOps();
	}

	void DumpAllOpenAllocations();
	void CrySizerStop()
	{
		// comment in to activate dumping of all remaining open allocation into the tty after the traversel
		//DumpAllOpenAllocations(); 
		MemoryOpsTemp().clear();
		bCrySizerTraversel = false;
		g_Lock.Unlock();
	}

	// base function to find the memory op entry for an ptr, will
	// do a range check
	EntryIt GetEntryIteratorExact( void *ptr )
	{
		uint32 nPtr = (uint32)ptr;
		Entry entry = {0};
		entry.InitFast(ptr);
		EntryIt entryIt = MemoryOpsTemp().lower_bound( entry );

		// out of range ptr(proberly not valid)
		if( entryIt == MemoryOpsTemp().end() )
			return MemoryOpsTemp().end();

		if( entryIt->nPtr == nPtr )
			return entryIt;

		// if ptr < first entry and not the same, return the end iterator to not match wrongly
		if( entryIt == MemoryOpsTemp().begin() )
			return MemoryOpsTemp().end();

		// get the previous element and do a range check
		entryIt--;

		uint32 nRangeStart = entryIt->nPtr;
		uint32 nRangeEnd = entryIt->nPtr + entryIt->nSize;

		if( nPtr >= nRangeStart && nPtr < nRangeEnd &&
				nPtr < nRangeStart + 32 )
		{				
			return entryIt;		
		}

		// nothing found, return end
		return MemoryOpsTemp().end();

	}

	EntryIt GetEntryIteratorRange( void *ptr )
	{
		uint32 nPtr = (uint32)ptr;
		Entry entry = {0};
		entry.InitFast(ptr);
		EntryIt entryIt = MemoryOpsTemp().lower_bound( entry );

		// out of range ptr(proberly not valid)
		if( entryIt == MemoryOpsTemp().end() )
			return MemoryOpsTemp().end();

		if( entryIt->nPtr == nPtr )
			return entryIt;

		// if ptr < first entry and not the same, return the end iterator to not match wrongly
		if( entryIt == MemoryOpsTemp().begin() )
			return MemoryOpsTemp().end();

		// get the previous element and do a range check
		entryIt--;

		uint32 nRangeStart = entryIt->nPtr;
		uint32 nRangeEnd = entryIt->nPtr + entryIt->nSize;

		if( nPtr >= nRangeStart && nPtr < nRangeEnd )
		{				
			return entryIt;		
		}

		// nothing found, return end
		return MemoryOpsTemp().end();

	}

	EntryIt GetEntryIteratorReference( void *ptr )
	{
		uint32 nPtr = (uint32)ptr;
		Entry entry = {0};
		entry.InitFast(ptr);
		EntryIt entryIt = MemoryOps().lower_bound( entry );

		// out of range ptr(proberly not valid)
		if( entryIt == MemoryOps().end() )
			return MemoryOps().end();

		if( entryIt->nPtr == nPtr )
			return entryIt;

		// if ptr < first entry and not the same, return the end iterator to not match wrongly
		if( entryIt == MemoryOps().begin() )
			return MemoryOps().end();

		// get the previous element and do a range check
		entryIt--;

		uint32 nRangeStart = entryIt->nPtr;
		uint32 nRangeEnd = entryIt->nPtr + entryIt->nSize;

		if( nPtr >= nRangeStart && nPtr < nRangeEnd )
		{				
			return entryIt;		
		}

		// nothing found, return end
		return MemoryOps().end();

	}

	bool HasPtrExact( void *ptr )
	{
		return GetEntryIteratorExact(ptr) != MemoryOpsTemp().end();		
	}

	bool HasPtrRange( void *ptr )
	{
		return GetEntryIteratorRange(ptr) != MemoryOpsTemp().end();		
	}


	uint32 GetPtrSizeExact( void *ptr )
	{
		EntryIt entryIt = GetEntryIteratorExact(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
			return entryIt->nSize;
		else
			return 0;
	}

	uint32 GetPtrSizeRange( void *ptr )
	{
		EntryIt entryIt = GetEntryIteratorRange(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
			return entryIt->nSize;
		else
			return 0;
	}

	uint32 GetPtrSizeReference( void *ptr )
	{
		EntryIt entryIt = GetEntryIteratorReference(ptr);

		if( entryIt !=  MemoryOps().end() )
			return entryIt->nSize;
		else
		{
			printf("Could not find size for 0x%x\n",ptr);
			return 0;
		}
	}

	void *GetRangeStartPtrExact(void*ptr)
	{
		EntryIt entryIt = GetEntryIteratorExact(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
			return (void*)entryIt->nPtr;
		else
			return NULL;
	}

	void *GetRangeStartPtrRange(void*ptr)
	{
		EntryIt entryIt = GetEntryIteratorRange(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
			return (void*)entryIt->nPtr;
		else
			return NULL;
	}

	void *GetRangeStartPtrReference(void*ptr)
	{
		EntryIt entryIt = GetEntryIteratorReference(ptr);

		if( entryIt !=  MemoryOps().end() )
			return (void*)entryIt->nPtr;
		else
		{
			printf("Could not find range for 0x%x\n",ptr);
			return NULL;
		}
	}

	void RemovePtrExact( void * ptr )
	{
		EntryIt entryIt = GetEntryIteratorExact(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
		{
			nRemovedMemRefs++;
			nRemovedMemory += entryIt->nSize;
			MemoryOpsTemp().erase(entryIt);
		}
	}

	void RemovePtrRange( void * ptr )
	{
		EntryIt entryIt = GetEntryIteratorRange(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
		{
			nRemovedMemRefs++;
			nRemovedMemory += entryIt->nSize;
			MemoryOpsTemp().erase(entryIt);
		}
	}

	int GetNumObjects()
	{
		return MemoryOpsTemp().size();
	}

	void DumpAllOpenReferencesStats()
	{
		printf("Open references %d, removed refs %d, tracked memory %d KB / %d KB \n",MemoryOpsTemp().size(), nRemovedMemRefs, nRemovedMemory/1024, g_nHoldedMemory/1024 );
	}

	void DumpCallStack( void *ptr )
	{
		EntryIt entryIt = GetEntryIteratorReference(ptr);
		
		if( entryIt !=  MemoryOps().end() )
		{
			printf("Callstack for 0x%x(%d):\t", (uint32)ptr,entryIt->nSize);
			const CallStack *pCallStack = entryIt->pCallStack;
			for( int i = 0 ; i < pCallStack->nCallStackEntries ; ++i)
			{
				printf("0x%x, ", pCallStack->arrCallStack[i] );
			}
			printf("\n");
		}
	}

	uint32 GetAllocationIdReference( void *ptr )
	{
		EntryIt entryIt = GetEntryIteratorReference(ptr);

		if( entryIt !=  MemoryOps().end() )
		{
			return entryIt->nId;
		}
		printf("Could not find id for 0x%x\n",ptr);
		return 0;
	}

	void *GetAllocationUserDataExact( void*ptr)
	{		
		EntryIt entryIt = GetEntryIteratorExact(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
		{
			return entryIt->pUserData;
		}
		return NULL;
	}

	uint32 GetAllocationTimeDiffExact(void*ptr)
	{
		EntryIt entryIt = GetEntryIteratorExact(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
		{
			return entryIt->nTimeDiff;
		}
		return 0;
	}

	void SetAllocationUserDataExact(void *ptr, void *userData)
	{
		EntryIt entryIt = GetEntryIteratorExact(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
		{
			Entry tmp = *entryIt;
			tmp.pUserData = userData;
			MemoryOpsTemp().erase(entryIt);
			MemoryOpsTemp().insert(tmp);
		}
	}

	void SetAllocationTimeDiffExact( void*ptr, int timediff)
	{
		EntryIt entryIt = GetEntryIteratorExact(ptr);

		if( entryIt !=  MemoryOpsTemp().end() )
		{
			Entry tmp = *entryIt;
			tmp.nTimeDiff = timediff;
			MemoryOpsTemp().erase(entryIt);
			MemoryOpsTemp().insert(tmp);
		}
	}

	static EntryIt gCurEntryPos;

	void StartMemIteration()
	{
		gCurEntryPos = MemoryOpsTemp().begin();
	}

	void *GetNextParentAllocation( void *&current)
	{
		for( ; gCurEntryPos != MemoryOpsTemp().end() ; ++gCurEntryPos )
		{
			if( gCurEntryPos->pUserData != NULL )
			{
				current = (void*)gCurEntryPos->nPtr;
				void *ret=  gCurEntryPos->pUserData;
				gCurEntryPos++;
				return ret;
			}
		}
		return NULL;
	}

	struct EntrySizeSum
	{
		uint32 nSize;
		uint32 nCount;
		CallStack *pCallStack;							
	};

	struct SizeSorter { bool operator()( const EntrySizeSum &rA, const EntrySizeSum &rB ) const { return rA.nSize < rB.nSize; } };
	struct CallStackSorter { bool operator()( const EntrySizeSum &rA, const EntrySizeSum &rB ) const { return (uint32)rA.pCallStack < (uint32)rB.pCallStack; } };

	void DumpAllOpenAllocations()
	{
		 std::set<EntrySizeSum, CallStackSorter, PrivateHeapAllocator<EntrySizeSum> > tmpSet;
		 for( EntryIt it = MemoryOpsTemp().begin() ; it != MemoryOpsTemp().end(); ++it )
		 {
				EntrySizeSum key = {0, 0, it->pCallStack };
				if( tmpSet.find(key) == tmpSet.end() )
				{
					EntrySizeSum value = {it->nSize, 1, it->pCallStack };
					tmpSet.insert(value);
				}
				else
				{
					std::set<EntrySizeSum, CallStackSorter, PrivateHeapAllocator<EntrySizeSum> >::iterator tmpIt = tmpSet.find(key);
					EntrySizeSum value = {tmpIt->nSize + it->nSize,tmpIt->nCount + 1, tmpIt->pCallStack };
					tmpSet.erase(tmpIt);
					tmpSet.insert(value);
				}
		 }

		 std::set<EntrySizeSum, SizeSorter, PrivateHeapAllocator<EntrySizeSum> > tmpSizeSet;
		 for( std::set<EntrySizeSum, CallStackSorter, PrivateHeapAllocator<EntrySizeSum> >::iterator it = tmpSet.begin() ; it != tmpSet.end(); ++it )
		 {
				tmpSizeSet.insert(*it);
		 }

		 for( std::set<EntrySizeSum, SizeSorter, PrivateHeapAllocator<EntrySizeSum> >::iterator it = tmpSizeSet.begin() ; it != tmpSizeSet.end(); ++it )
		 {
			 printf("Size %d(%d)\tCallStack:\t", it->nSize, it->nCount);
			 const CallStack *pCallStack = it->pCallStack;
			 for( int i = 0 ; i < pCallStack->nCallStackEntries ; ++i)
			 {
				 printf("0x%x, ", pCallStack->arrCallStack[i] );
			 }
			 printf("\n");
		 }

		 printf("Unique CallStacks %d, AllMissingCallstack %d, AllocatedMemory %d KB, RemovedMemory %d\n",tmpSizeSet.size(), MemoryOpsTemp().size(), g_nHoldedMemory/1024, nRemovedMemory/1024 );
	}
#endif
}

# include <CryPool/PoolAlloc.h>
namespace NVirtualMem
{	
	//pool allocator using virtual memory
	static const uint32 g_MaxVirtualAllocs = 2048;
	static uint32 g_CurVirtualCount = 0;
	static const uint32 g_VirtualAllocAlign = 16;
	typedef NCryPoolAlloc::CBestFit< NCryPoolAlloc::CReferenced< NCryPoolAlloc::CMemoryDynamic, g_MaxVirtualAllocs, false >, NCryPoolAlloc::CListItemReference > TMTSafeVirtualPool;
	static TMTSafeVirtualPool g_VirtPoolAlloc;
	static int g_VirtPoolInitialized;
	static CryCriticalSection	g_VirtPoolAllocCritSec;

	//enables constant sys cache usage, currently changed during loading to decrease loading time
//	#define FORCE_CONSTANT_SYS_CACHE_USE
//	#define DISPLAY_LARGE_ALLOCS

	/*
		class implements virtual memory on PS3 
			- a fixed memory block is allocated on RSX (via CryMallocRSX)
			- the same size as range is queried by OS to be mapped
			- a 1:1 connection is established between this range and the RSX range
			- a fixed amount of system pages is allocated and mapped according to some map LRU
			- until RSX memory is available, the full range is mapped to system and unmapped once available
			- if a 64K page needs to be mapped, a system callback arises and we need to 
					- check if a page is available -> map the physical page to the address
					- if none are available
							- check which one was mapped least recently
							- upload to its RSX storage (via CryMemcpyRSX)
							- download from RSX the content of the miss page
							- unmap physical page
							- map physical page to the miss page
	*/
	struct SVirtualUploadData
	{
		uint32 srcAddr;
		uint32 dstAddr;
		sys_memory_t *pMemoryId;
		uint32 faultAddr;
		uint32 backupAddr;
	};

	struct SPageData
	{
		sys_memory_t	memoryID;
		uint32				curAddr;
		SPageData() : memoryID((sys_memory_t)~0),curAddr(0){}
	};

	#define RETURN_FAILED SNPAUSE;m_Initialized=eVSFailed;return false;
	#define BREAK_FAILED  SNPAUSE;m_Initialized=eVSFailed;goto SwitchToVirtualFailed;
	#define USE_RANDOM

	void UploadToRSX(uint32);

	class CVirtMemMapper
	{
	public:
		typedef enum{eVSUnitialized = 0,eVSFailed,eVSSystemInitialized,eVSRSXInitialized,eVSRSXShutdown} EInitState;

		//systemMemSize size of system cache (physically mapped)
		//virtualMemSize size of address range to map and RSX memory allocated
		bool Init(uint32 systemMemSize, uint32 virtualMemSize)
		{
			//make sure both sizes fit multiple of 64K
			systemMemSize		= AlignSize(systemMemSize);
			virtualMemSize	= AlignSize(virtualMemSize);
			static const uint32 scMaxVirtSize = 256*1024*1024;
			if(virtualMemSize > scMaxVirtSize)
			{
				printf("CVirtMemMapper::Allocate: max virtual memory to map exceeded(max %d KB)\n",scMaxVirtSize>>10);
				RETURN_FAILED
			}
			m_VirtualMemSize	= m_VirtualMemSizeLeft = virtualMemSize;
			m_SystemPageCount = m_SystemMaxPageCount = virtualMemSize/NMemCont::MEMORY_ALLOC_GRANULARITY;
			m_SystemCachePageCount = systemMemSize/NMemCont::MEMORY_ALLOC_GRANULARITY;
#ifdef FORCE_CONSTANT_SYS_CACHE_USE
			m_SystemPageCount = m_SystemCachePageCount;
#endif
			//need to allocate address space with at least 256 MB
			int ret = sys_mmapper_allocate_address(256*1024*1024, SYS_MEMORY_GRANULARITY_64K | SYS_MEMORY_ACCESS_RIGHT_PPU_THR, 0x0UL, &m_BaseAddress);
			if( ret != CELL_OK ) 
			{
				PrintSysErrorMessage(ret,"CVirtMemMapper::sys_mmapper_allocate_address");
				RETURN_FAILED
			}
			ret = sys_mmapper_allocate_address(256*1024*1024, SYS_MEMORY_GRANULARITY_64K | SYS_MEMORY_ACCESS_RIGHT_PPU_THR,0x0UL, &m_BaseBackupAddress);
			if( ret != CELL_OK ) 
			{
				PrintSysErrorMessage(ret,"CVirtMemMapper::sys_mmapper_allocate_address for backup");
				RETURN_FAILED
			}
			m_CurUploadData.backupAddr = m_BaseBackupAddress;
			if(!gPS3Env->bRSXMemManInit)
			{
				//initialize the full system memory to map til we obtain rsx memory
				sys_memory_t	memoryID;
				ret = sys_mmapper_allocate_memory(m_VirtualMemSize, SYS_MEMORY_GRANULARITY_64K, &memoryID);
				if( ret != CELL_OK ) 
				{
					PrintSysErrorMessage(ret,"CVirtMemMapper::Init::sys_mmapper_allocate_memory");
					RETURN_FAILED
				}
				ret = sys_mmapper_map_memory(m_BaseAddress, memoryID, SYS_MEMORY_PROT_READ_WRITE);
				if( ret != CELL_OK ) 
				{
					PrintSysErrorMessage(ret,"CVirtMemMapper::Init::sys_mmapper_map_memory");
					RETURN_FAILED
				}
				m_Initialized	= eVSSystemInitialized;
			}
			else
				SwitchToVirtual();
			uint32 virtAddr = 0;
			Allocate(virtualMemSize, virtAddr);
			g_VirtPoolAlloc.InitMem(virtualMemSize, (uint8*)virtAddr);
			g_VirtPoolInitialized = 1;
			printf("Virtual memory: initialized %d KB(sys)-> %d KB(rsx)\n",systemMemSize>>10, virtualMemSize>>10);
			return true;
		}

		bool IsVirtualAlloc(void *p)
		{
			return ((uint32)p >= m_BaseAddress) && ((uint32)p < (m_BaseAddress+m_VirtualMemSize));
		}
		
		void SwitchToVirtual()
		{
			AUTO_LOCK(m_MemCritSec);
			int ret;
			if(gPS3Env->bRSXMemManInit && m_VirtualMemSize > 0)
			{
				if(InitRSXMem())
				{
					if(m_Initialized == eVSSystemInitialized)
					{
						//enable page fault notification
						if(!StartEventThread())
						{
							BREAK_FAILED
						}
						sys_memory_t memoryId;
						ret = sys_mmapper_unmap_memory(m_BaseAddress, &memoryId);
						if(ret != CELL_OK)
						{
							PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToVirtual::sys_mmapper_unmap_memory");
							SNPAUSE;
						}
						if(m_VirtualMemSize != m_VirtualMemSizeLeft)
						{
							//map to backup area
							ret = sys_mmapper_map_memory(m_BaseBackupAddress, memoryId, SYS_MEMORY_PROT_READ_ONLY);
							if( ret != CELL_OK )
							{
								PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToVirtual(backup) sys_mmapper_map_memory");
								SNPAUSE;
							}
							crymemcpy16(m_pRSXMem,(void*)m_BaseBackupAddress,m_VirtualMemSize-m_VirtualMemSizeLeft);
							ret = sys_mmapper_unmap_memory(m_BaseBackupAddress, &memoryId);
							if(ret != CELL_OK)
							{
								PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToVirtual::sys_mmapper_unmap_memory");
								SNPAUSE;
							}
						}
						ret = sys_mmapper_free_memory(memoryId);
						if( ret != CELL_OK ) 
						{
							PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToVirtual::sys_mmapper_free_memory");
							SNPAUSE;
						}
					}
					//initialize system cache pages, allocate for possible maximum
					m_pPageData = new SPageData[m_SystemMaxPageCount];
					m_Initialized = eVSRSXInitialized;
					printf("Virtual memory: enabled (0x%08x..0x%08x -> 0x%08x)\n",m_BaseAddress,m_BaseAddress+m_VirtualMemSize,(uint32)m_pRSXMem);
					return;
				}
			}
SwitchToVirtualFailed:
			{
				m_MemoryPresent = true;
				m_Initialized = eVSFailed;
			}
		}

		uint32 Available() const
		{
			return m_VirtualMemSizeLeft;
		}

		bool Allocate(uint32 size, uint32& rMem)
		{	
			if(size)
			{
				if(m_Initialized == eVSFailed)
				{
					void *pRet = malloc(size);
					rMem = (uint32)pRet;
					return true;
				}
				size = AlignSize(size);
				if(m_VirtualMemSizeLeft < size)
				{
					printf("CVirtMemMapper::Allocate: not enough virtual memory left(requested %d KB, left: %d KB)\n",size>>10,m_VirtualMemSizeLeft>>10);
					SNPAUSE;
					rMem = 0;
					return false;
				}
				rMem = m_BaseAddress + (m_VirtualMemSize - m_VirtualMemSizeLeft);
				m_VirtualMemSizeLeft -= size;
				return true;
			}
			return false;
		}

		void Shutdown()
		{
			if(m_Initialized == eVSFailed)
			{
				sys_memory_t memoryId;
				sys_mmapper_unmap_memory(m_BaseAddress, &memoryId);
				sys_mmapper_free_memory(memoryId);
				sys_mmapper_free_address(m_BaseAddress);
				sys_mmapper_free_address(m_BaseBackupAddress);
			}
			else
			if(m_Initialized == eVSRSXShutdown)
			{
				sys_memory_t memoryId;
				int ret = sys_mmapper_unmap_memory(m_BaseAddress, &memoryId);
				if(ret != CELL_OK)
				{
					PrintSysErrorMessage(ret,"CVirtMemMapper::Shutdown::sys_mmapper_unmap_memory(base)");
					SNPAUSE;
				}
				ret = sys_mmapper_free_memory(memoryId);
				if( ret != CELL_OK ) 
				{
					PrintSysErrorMessage(ret,"CVirtMemMapper::Shutdown::sys_mmapper_free_memory(base)");
					SNPAUSE;
				}
			}
			else
			if(m_Initialized > eVSFailed )
			{
				sys_event_port_send(m_TerminateEvent, 0, 0, 0 );
				sys_event_queue_destroy(m_QueueID,SYS_EVENT_QUEUE_LOCAL);
				if(m_Initialized == eVSRSXInitialized && gPS3Env->bRSXMemManInit && m_RSXHandle != ~0 && m_pRSXMem != NULL)
				{
					CryFreeRSX(m_RSXHandle);						//free rsx resource
					m_RSXHandle = ~0;
					m_pRSXMem		= NULL;
				}
				//unmap all memory pages
				int ret;
				for(uint32 i=0;i<m_SystemPageCount; ++i)
				{
					if(m_pPageData[i].curAddr != 0)
					{
						ret = sys_mmapper_unmap_memory(m_pPageData[i].curAddr, &m_pPageData[i].memoryID);
						if(ret != CELL_OK)
						{
							PrintSysErrorMessage(ret,"CVirtMemMapper::Shutdown::sys_mmapper_unmap_memory");
							SNPAUSE;
						}
					}
					ret = sys_mmapper_free_memory(m_pPageData[i].memoryID);
					if( ret != CELL_OK ) 
					{
						PrintSysErrorMessage(ret,"CVirtMemMapper::Shutdown::sys_mmapper_free_memory");
						SNPAUSE;
					}
				}
				sys_mmapper_free_address(m_BaseAddress);
				sys_mmapper_free_address(m_BaseBackupAddress);
			}
			delete [] m_pPageData;
			g_VirtPoolInitialized = 0;
			m_Initialized = eVSUnitialized;
		}

		CVirtMemMapper() : m_Initialized(eVSUnitialized),m_Thread((sys_ppu_thread_t)~0),m_MemoryPresent(false),
			m_pRSXMem(NULL),m_RSXHandle((uint32)~0),m_BaseAddress((sys_addr_t)~0),m_BaseBackupAddress((sys_addr_t)~0),
			m_PageReplIndex(0), m_pPageData(NULL)
		{
			m_CurUploadData.srcAddr = m_CurUploadData.dstAddr = 0;
		}

		static void	ProcessingThread(uint64 arg)
		{
			sys_event_t	  		event;
			sys_event_queue_t queue = (sys_event_queue_t)arg;
			do
			{
				sys_event_queue_receive(queue, &event, SYS_NO_TIMEOUT);
				if(SYS_MEMORY_PAGE_FAULT_GET_CAUSE(event) == SYS_MEMORY_PAGE_FAULT_CAUSE_NON_MAPPED)
				{
					sys_ppu_thread_t target_thr_id = SYS_MEMORY_PAGE_FAULT_GET_TARGET_ID(event);
					VirtMemMapper()->MapMemory(SYS_MEMORY_PAGE_FAULT_GET_ADDRESS(event),(uint32)target_thr_id);
					sys_ppu_thread_recover_page_fault(target_thr_id);
				}
				else
				{
					SNPAUSE;
				}
			}
			while (1);
		}

		static void	TerminateThread(uint64 eventId, uint64 data1, uint64 data2){sys_ppu_thread_exit(0);}

		void MapMemory(uint32 faultAddr, uint32 threadID)
		{
			/*algorithm:
					- use queue to find index of least recently mapped page
					- unmap page to backup area
					- map available page to fault address
					- initiate rsx memcpy
					- memcpy in  parallel to rsx
					- update queue
					- unmap from backup area
			*/
			if(m_Initialized == eVSRSXShutdown)
				return;
			AUTO_LOCK(m_MemCritSec);
			if(faultAddr < m_BaseAddress || faultAddr > m_BaseAddress+m_VirtualMemSize)
			{
				SNPAUSE;
			}
			faultAddr &= ~(NMemCont::MEMORY_ALLOC_GRANULARITY-1);
			uint32 oldIndex			= m_PageReplIndex;
			SPageData& page2Map	= m_pPageData[oldIndex];
			if(HasPageBeenMapped(faultAddr))
				return;
			++m_VMMisses;
			m_PageReplIndex						= SelectReplacementPage();
			m_CurUploadData.dstAddr		= m_CurUploadData.srcAddr = 0;//reset
			//any failure here cannot be handled
			int ret;
			m_CurUploadData.srcAddr = m_CurUploadData.dstAddr = 0;
			if(page2Map.curAddr)
			{
				ret = sys_mmapper_unmap_memory(page2Map.curAddr, &page2Map.memoryID);
				if(ret != CELL_OK)
				{
					PrintSysErrorMessage(ret,"CVirtMemMapper::MapMemory: sys_mmapper_unmap_memory");
					SNPAUSE;
				}
				//map to backup area
				ret = sys_mmapper_map_memory(m_BaseBackupAddress, page2Map.memoryID, SYS_MEMORY_PROT_READ_ONLY);
				if( ret != CELL_OK )
				{
					PrintSysErrorMessage(ret,"CVirtMemMapper::MapMemory:(backup) sys_mmapper_map_memory");
					SNPAUSE;
				}
				//setup data for rsx upload
				m_CurUploadData.srcAddr = m_BaseBackupAddress;
				m_CurUploadData.dstAddr = (uint32)m_pRSXMem + (page2Map.curAddr - m_BaseAddress);
			}
			//now start rsx upload and rsx download
			m_CurUploadData.pMemoryId		= &page2Map.memoryID;
			m_CurUploadData.faultAddr		= faultAddr;
			page2Map.curAddr						= faultAddr;
			CryMemcpyRSX((void*)m_BaseBackupAddress,(void*)((uint32)m_pRSXMem + (faultAddr - m_BaseAddress)),NMemCont::MEMORY_ALLOC_GRANULARITY,&UploadToRSX,false);

			ret = sys_mmapper_unmap_memory(m_BaseBackupAddress, &page2Map.memoryID);
			if(ret != CELL_OK)
			{
				PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToVirtual::sys_mmapper_unmap_memory");
				SNPAUSE;
			}
			ret = sys_mmapper_map_memory(faultAddr, page2Map.memoryID, SYS_MEMORY_PROT_READ_WRITE);
			if( ret != CELL_OK )
			{
				PrintSysErrorMessage(ret,"CVirtMemMapper::MapMemory: sys_mmapper_map_memory");
				SNPAUSE;
			}
		}

		size_t VirtualMemorySize()
		{
			return m_VirtualMemSize;
		}

		size_t GetMemoryUsage()
		{
			size_t s = sizeof(*this) + m_SystemPageCount * sizeof(SPageData) + sizeof(g_VirtPoolAlloc);
			if(m_Initialized == eVSFailed)
				s += m_VirtualMemSize;
			else
			{
				s += m_SystemPageCount * NMemCont::MEMORY_ALLOC_GRANULARITY;
			}
			return s;
		}

		SVirtualUploadData* UploadData() {return &m_CurUploadData;}

		void ShutdownRSX()
		{
			AUTO_LOCK(m_MemCritSec);
			//first unlink are currently allocated pages
			SwitchToSystemCache(true);
			//allocate and copy full rsx memory to here
			sys_memory_t	memoryID;
			int ret = sys_mmapper_allocate_memory(m_VirtualMemSize, SYS_MEMORY_GRANULARITY_64K, &memoryID);
			if( ret != CELL_OK ) 
				PrintSysErrorMessage(ret,"CVirtMemMapper::ShutdownRSX::sys_mmapper_allocate_memory");
			else
			{
				ret = sys_mmapper_map_memory(m_BaseAddress, memoryID, SYS_MEMORY_PROT_READ_WRITE);
				if( ret != CELL_OK ) 
					PrintSysErrorMessage(ret,"CVirtMemMapper::ShutdownRSX::sys_mmapper_map_memory");
				else
				{
					//now copy here
					memcpy((void*)m_BaseAddress,(void*)m_pRSXMem,m_VirtualMemSize);//slow rsx path
					m_Initialized = eVSRSXShutdown;
				}
			}
		}

		void SwitchToSystemFull()
		{
			AUTO_LOCK(m_MemCritSec);
			//increase the number of system pages matching the actual rsx amount
			//the page faults will automatically make all pages allocate
			if(m_Initialized == eVSRSXInitialized && m_SystemPageCount != m_SystemMaxPageCount)
			{
				printf("Virtual memory: switching to full system coverage\n");
				m_PageReplIndex		= m_SystemPageCount;
				m_SystemPageCount = m_SystemMaxPageCount;
			}
		}

		void SwitchToSystemCache(bool shutdown = false)
		{
			AUTO_LOCK(m_MemCritSec);
			//decrease the number of system pages matching the requested cache size
			if(m_Initialized == eVSRSXInitialized && (m_SystemCachePageCount != m_SystemPageCount) || shutdown)
			{
				printf("Virtual memory: switching to system cache\n");
				int ret;
				for(uint32 i=shutdown?0:m_SystemCachePageCount; i<m_SystemPageCount;++i)
				{
					SPageData& rPage = m_pPageData[i];
					if(rPage.curAddr)
					{
						//unmap page
						ret = sys_mmapper_unmap_memory(rPage.curAddr, &rPage.memoryID);
						if(ret != CELL_OK)
						{
							PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToSystemCache: sys_mmapper_unmap_memory");
							SNPAUSE;
						}
						//map to backup area
						ret = sys_mmapper_map_memory(m_BaseBackupAddress, rPage.memoryID, SYS_MEMORY_PROT_READ_ONLY);
						if( ret != CELL_OK )
						{
							PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToSystemCache:(backup) sys_mmapper_map_memory");
							SNPAUSE;
						}
						//upload to rsx
						crymemcpy16((void*)((uint32)m_pRSXMem + (rPage.curAddr - m_BaseAddress)), (void*)m_BaseBackupAddress, NMemCont::MEMORY_ALLOC_GRANULARITY);
						//unmap from backup area
						ret = sys_mmapper_unmap_memory(m_BaseBackupAddress, &rPage.memoryID);
						if(ret != CELL_OK)
						{
							PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToSystemCache::sys_mmapper_unmap_memory");
							SNPAUSE;
						}
					}
					//free to system
					if(rPage.memoryID != (sys_memory_t)~0)
					{
						ret = sys_mmapper_free_memory(rPage.memoryID);
						if( ret != CELL_OK ) 
						{
							PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToSystemCache::sys_mmapper_free_memory");
							SNPAUSE;
						}
					}
					//reset page
					rPage.curAddr		= 0;
					rPage.memoryID	= (sys_memory_t)~0;
				}
				m_PageReplIndex		= 0;
				m_SystemPageCount = shutdown?0:m_SystemCachePageCount;
			}
		}

		void FlagVMMissFallback()
		{
			m_VMMissFallback += 10;
		}

		uint32 VMMisses()
		{
			static const uint32 scMaxFrameAcc = 64;
			static uint32 scAccumBuffer[scMaxFrameAcc] = {0};
			static uint32 sCurIdx = 0;
			scAccumBuffer[sCurIdx] = m_VMMisses;
			sCurIdx = (sCurIdx+1) % scMaxFrameAcc;
			uint32 misses = 0;
			for(uint32 i=0; i<scMaxFrameAcc; ++i)
				misses += m_VMMisses;
			m_VMMisses = 0;
			gPS3Env->bInjectionMiss = (m_VMMissFallback > 0)?1:0;
			--m_VMMissFallback;
			if(m_VMMissFallback <= 0)m_VMMissFallback=0;
			return misses / scMaxFrameAcc;
		}

	private:
		CryCriticalSection			m_MemCritSec;
		SVirtualUploadData			m_CurUploadData;
		uint32									m_VirtualMemSize;
		uint32									m_VirtualMemSizeLeft;
		uint32									m_VMMisses;
		int											m_VMMissFallback;
		uint32									m_SystemPageCount;
		uint32									m_SystemCachePageCount;
		uint32									m_SystemMaxPageCount;
		sys_ppu_thread_t				m_Thread;
		uint32									m_RSXHandle;
		void*										m_pRSXMem;
		sys_addr_t							m_BaseAddress;
		sys_addr_t							m_BaseBackupAddress;
		EInitState							m_Initialized;
		bool										m_MemoryPresent;
		sys_event_queue_t				m_QueueID;
		sys_event_port_t				m_TerminateEvent;
		SPageData*							m_pPageData;
		uint32									m_PageReplIndex;

		uint32 SelectReplacementPage()
		{
			int minIdx = (m_PageReplIndex+1)%m_SystemPageCount;
#if defined(USE_RANDOM)
			if(m_SystemPageCount != m_SystemMaxPageCount)//do not use in case of full mapping
			{
				uint32 r = cry_rand() % m_SystemPageCount;
				if(r != m_PageReplIndex)
					minIdx = r;
			}
#endif
			return minIdx;
		}

		bool HasPageBeenMapped(uint32 addr) const
		{
			for(uint32 i=0; i<m_SystemPageCount; ++i)
			{
				if(m_pPageData[i].curAddr	== addr)
					return true;
			}
			return false;
		}

		uint32 AlignSize(uint32 size)
		{
			return (size+NMemCont::MEMORY_ALLOC_GRANULARITY-1) & ~(NMemCont::MEMORY_ALLOC_GRANULARITY-1);
		}
			
		bool InitRSXMem()
		{
			m_RSXHandle	= CryMallocRSX(m_VirtualMemSize,m_pRSXMem);
			if(m_RSXHandle == ~0 || m_pRSXMem==NULL)
			{
				printf("Error in CVirtMemMapper: cannot allocate RSX Memory");
				m_Initialized = eVSFailed;//will fallback to system allocator
				memset(m_pRSXMem,0,m_VirtualMemSize);
				return false;
			}
			return true;
		}

		bool StartEventThread()
		{
			sys_event_queue_attribute_t queueAttr;
			sys_event_queue_attribute_initialize(queueAttr);
			sys_event_queue_create(&m_QueueID, &queueAttr, SYS_EVENT_QUEUE_LOCAL, 32);
			int ret = sys_mmapper_enable_page_fault_notification(m_BaseAddress, m_QueueID);
			if( ret != CELL_OK ) 
			{
				PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToVirtual::sys_mmapper_enable_page_fault_notification");
				return false;
			}
			ret = sys_ppu_thread_create(&m_Thread, ProcessingThread, (uint64_t)m_QueueID, 0/*prio*/, 8*1024/*stack*/, 0, "VirtMemEventThread");
			if(ret != CELL_OK)
			{
				printf("Error in CVirtMemMapper: sys_ppu_thread_create failed(0x%08x)",ret);
				m_Initialized = eVSFailed;//will fallback to system allocator
				return false;
			}
			ret = sys_event_port_create(&m_TerminateEvent, SYS_EVENT_PORT_LOCAL, (uintptr_t)TerminateThread);
			if(ret != CELL_OK)
			{
				printf("Error in CVirtMemMapper: sys_event_port_create failed(0x%08x)",ret);
				m_Initialized = eVSFailed;//will fallback to system allocator
				return false;
			}
			return true;
		}
	};

	CVirtMemMapper * VirtMemMapper()
	{
		static CVirtMemMapper sVirtMemMapper;
		return &sVirtMemMapper;
	}

	void UploadToRSX(uint32 stage)
	{
		assert(pArg);
		SVirtualUploadData *const pUploadData = VirtMemMapper()->UploadData();
		bool uploadRequested = pUploadData->dstAddr && pUploadData->srcAddr;
		if(uploadRequested && (stage == 0))
		{
			//stage 0 is uploading to rsx
			crymemcpy16((void*)pUploadData->dstAddr, (void*)pUploadData->srcAddr, NMemCont::MEMORY_ALLOC_GRANULARITY);
		}
		if(stage == 1)
		{
			//stage 1 is mapping of page as preparation for the actual memory transfer inside CryMemcpyRSX
			int ret;
			if(uploadRequested)
			{
				//unmap page
				ret = sys_mmapper_unmap_memory(pUploadData->backupAddr, pUploadData->pMemoryId);
				if(ret != CELL_OK)
				{
					PrintSysErrorMessage(ret,"CVirtMemMapper::SwitchToVirtual::sys_mmapper_unmap_memory");
					SNPAUSE;
				}
			}
			if(*pUploadData->pMemoryId == (sys_memory_t)~0)
			{
				//allocate on demand
				ret = sys_mmapper_allocate_memory(NMemCont::MEMORY_ALLOC_GRANULARITY, SYS_MEMORY_GRANULARITY_64K, pUploadData->pMemoryId);
				if( ret != CELL_OK )
				{
					PrintSysErrorMessage(ret,"CVirtMemMapper::UploadToRSX::sys_mmapper_allocate_memory");
					SNPAUSE;
				}
			}
			//map page to fault address
			ret = sys_mmapper_map_memory(pUploadData->backupAddr, *pUploadData->pMemoryId, SYS_MEMORY_PROT_READ_WRITE);
			if( ret != CELL_OK )
			{
				PrintSysErrorMessage(ret,"CVirtMemMapper::MapMemory: sys_mmapper_map_memory");
				SNPAUSE;
			}
			pUploadData->dstAddr = pUploadData->srcAddr = 0;
		}
	}

	bool Init(uint32 systemMemSize, uint32 virtualMemSize)
	{
		VirtMemMapper()->Init(systemMemSize, virtualMemSize);
	}

	void Shutdown()
	{
		VirtMemMapper()->Shutdown();
	}

	bool Allocate(uint32 size, uint32& rMem)
	{
		return VirtMemMapper()->Allocate(size, rMem);
	}

	uint32 Available()
	{
		return VirtMemMapper()->Available();
	}

	void SwitchToVirtual()
	{
		VirtMemMapper()->SwitchToVirtual();
	}

	size_t GetMemoryUsage()
	{
		return VirtMemMapper()->GetMemoryUsage();
	}

	void SwitchToSystemFull()
	{
#ifndef FORCE_CONSTANT_SYS_CACHE_USE
		VirtMemMapper()->SwitchToSystemFull();
#endif
	}

	void SwitchToSystemCache()
	{
#ifndef FORCE_CONSTANT_SYS_CACHE_USE
		VirtMemMapper()->SwitchToSystemCache();
#endif
	}

	bool IsVirtualMemUsed(){return g_VirtPoolInitialized;}

	uint32 VirtualAllocCount()
	{
		return g_CurVirtualCount;		
	}

	void *AllocVirtualMem(uint32 s)
	{
		uint32 handle;
		return AllocVirtualMem(s, handle);
	}

	void *AllocVirtualMem(uint32 s, uint32& handle)
	{
		handle = 0;
		if(g_VirtPoolInitialized)
		{
			AUTO_LOCK(g_VirtPoolAllocCritSec);
			handle = g_VirtPoolAlloc.Allocate<size_t>(s, g_VirtualAllocAlign);
			if(handle != 0)
			{
				++g_CurVirtualCount;
				if(g_CurVirtualCount < g_MaxVirtualAllocs)
				{
					void *pRet = g_VirtPoolAlloc.Resolve<void*>(handle);
#ifdef DISPLAY_LARGE_ALLOCS
					if(s>500000)
						printf("AllocVirtualMem(%d KB) = 0x%08x, cur count=%d\n",s>>10,(uint32)pRet,g_CurVirtualCount);
#endif
					if(pRet)
						return pRet;
				}
			}
#ifdef DISPLAY_LARGE_ALLOCS
			printf("Virtual memory: alloc failed(%d KB, alloc count=%d/%d, use: %d KB), defaulting to system alloc...\n",s>>10,g_CurVirtualCount,g_MaxVirtualAllocs,(g_VirtPoolAlloc.MemSize()-g_VirtPoolAlloc.MemFree())>>10);
#endif
		}
		printf("Virtual memory: alloc failed, defaulting to system alloc...\n");
		return malloc(s);
	}

	void FreeVirtualMem(void* p, uint32 handle)
	{
		if(p && handle && g_VirtPoolInitialized && VirtMemMapper()->IsVirtualAlloc(p))
		{
			AUTO_LOCK(g_VirtPoolAllocCritSec);
			--g_CurVirtualCount;
#ifdef DISPLAY_LARGE_ALLOCS
			uint32 memSizebefore = g_VirtPoolAlloc.MemFree();
			g_VirtPoolAlloc.Free(handle);
			uint32 memSizeafter = g_VirtPoolAlloc.MemFree();
			uint32 s = memSizeafter-memSizebefore;
			if(s>500000)
				printf("	FreeVirtualMem(%d KB) = 0x%08x, cur count=%d\n",s>>10,(uint32)p,g_CurVirtualCount);
#else
			g_VirtPoolAlloc.Free(handle);
#endif
		}
		else
			free(p);
	}

	void FreeVirtualMem(void* p)
	{
		if(p && g_VirtPoolInitialized && VirtMemMapper()->IsVirtualAlloc(p))
		{
			 AUTO_LOCK(g_VirtPoolAllocCritSec);
			 --g_CurVirtualCount;
#ifdef DISPLAY_LARGE_ALLOCS
			uint32 memSizebefore = g_VirtPoolAlloc.MemFree();
			size_t h=g_VirtPoolAlloc.AddressToHandle(p);
			g_VirtPoolAlloc.Free(h);
			uint32 memSizeafter = g_VirtPoolAlloc.MemFree();
			uint32 s = memSizeafter-memSizebefore;
			if(s>500000)
				printf("	FreeVirtualMem(%d KB) = 0x%08x, cur count=%d\n",s>>10,(uint32)p,g_CurVirtualCount);
#else
			 g_VirtPoolAlloc.Free(g_VirtPoolAlloc.AddressToHandle(p));
#endif
		}
		else
			free(p);
	}

	size_t VirtualMemorySize()
	{
		return VirtMemMapper()->VirtualMemorySize();
	}

	size_t VirtualMemoryUsed()
	{
		return g_VirtPoolAlloc.MemSize()-g_VirtPoolAlloc.MemFree();
	}

	uint32 VMMisses()
	{
		return VirtMemMapper()->VMMisses();
	}

	bool IsVirtualAlloc(void* p)
	{
		return VirtMemMapper()->IsVirtualAlloc(p);
	}

	void FlagVMMissFallback()
	{
		VirtMemMapper()->FlagVMMissFallback();
	}

	void ShutdownRSX()
	{
		VirtMemMapper()->ShutdownRSX();
	}
}
