//////////////////////////////////////////////////////////////////////
//
//	Crytek Common Source code
//
//	File: Cry_PS3_Math.cpp
//	Description: Implementation of fast trigonometric funcs for PS3
//
//	History:
//	-May 29,2008: Created by MichaelG
//
//////////////////////////////////////////////////////////////////////


#if defined(PS3) && defined(PS3OPT) && !defined(__SPU__) && !defined(SUPP_FP_EXC)
	#ifndef ILINE
		#ifdef NO_ILINE
			#define ILINE inline
		#else
			#define ILINE __attribute__((always_inline)) inline
		#endif
	#endif
namespace
{
	//sin Taylor series coefficients
	static const double SC_double[]    =
	{
		-1.6666666666666666666667e-01, // 1/3!
		+8.3333333333333333333333e-03, // 1/5!
		-1.9841269841269841269841e-04, // 1/7!
		+2.7557319223985890652557e-06, // 1/9!
		-2.5052108385441718775052e-08, // 1/11!
		+1.6059043836821614599392e-10, // 1/13!
		-7.6471637318198164759011e-13, // 1/15!
		+2.8114572543455207631989e-15, // 1/17!
	};
	//cos Taylor series coefficients
	static const double CC_double[]    =
	{
		-5.0000000000000000000000e-01, // 1/2!
		+4.1666666666666666666667e-02, // 1/4!
		-1.3888888888888888888889e-03, // 1/6!
		+2.4801587301587301587302e-05, // 1/8!
		-2.7557319223985890652557e-07, // 1/10!
		+2.0876756987868098979210e-09, // 1/12!
		-1.1470745597729724713852e-11, // 1/14!
		+4.7794773323873852974382e-14, // 1/16!   
	};
	//asin/acos numerator coefficients
	static const double PC_double[]    =
	{
		-2.7368494524164255994e+01,
		+5.7208227877891731407e+01,
		-3.9688862997404877339e+01,
		+1.0152522233806463645e+01,
		-6.9674573447350646411e-01,
	};
	//asin/acos denominator coefficients
	static const double QC_double[]    =
	{
		-1.6421096714498560795e+02,
		+4.1714430248260412556e+02,
		-3.8186303361750149284e+02,
		+1.5095270841030604719e+02,
		-2.3823859153670238830e+01, 
	};
	//atan numerator coefficients
	static const double TQC_double[]   =
	{
		+4.1066306682575781263e+01,
		+8.6157349597130242515e+01,
		+5.9578436142597344465e+01,
		+1.5024001160028576121e+01,
	};
	//atan denominator coefficients
	static const double TPC_double[]   =
	{
		-1.3688768894191926929e+01,
		-2.0505855195861651981e+01,
		-8.4946240351320683534e+00,
		-8.3758299368150059274e-01,
	};
	//atan magic
	static const double TA_double[4]   =
	{
		0.0,
		+5.2359877559829887308e-01,
		+1.5707963267948966192e+00,
		+1.0471975511965977462e+00
	};
	static const float SC_float[]    =
	{
		-1.6666666666666666666667e-01f, // 1/3!
		+8.3333333333333333333333e-03f, // 1/5!
		-1.9841269841269841269841e-04f, // 1/7!
		+2.7557319223985890652557e-06f, // 1/9!
		-2.5052108385441718775052e-08f, // 1/11!
		+1.6059043836821614599392e-10f, // 1/13!
		-7.6471637318198164759011e-13f, // 1/15!
		+2.8114572543455207631989e-15f, // 1/17!
	};
	static const float QC_float[]    =
	{
		-1.6421096714498560795e+02f,
		+4.1714430248260412556e+02f,
		-3.8186303361750149284e+02f,
		+1.5095270841030604719e+02f,
		-2.3823859153670238830e+01f, 
	};
	static const float PC_float[]    =
	{
		-2.7368494524164255994e+01f,
		+5.7208227877891731407e+01f,
		-3.9688862997404877339e+01f,
		+1.0152522233806463645e+01f,
		-6.9674573447350646411e-01f,
	};
	//atan numerator coefficients
	static const float TQC_float[]   =
	{
		+4.1066306682575781263e+01f,
		+8.6157349597130242515e+01f,
		+5.9578436142597344465e+01f,
		+1.5024001160028576121e+01f,
	};
	//atan denominator coefficients
	static const float TPC_float[]   =
	{
		-1.3688768894191926929e+01f,
		-2.0505855195861651981e+01f,
		-8.4946240351320683534e+00f,
		-8.3758299368150059274e-01f,
	};
	//atan magic
	static const float TA_float[4]   =
	{
		0.0f,
		+5.2359877559829887308e-01f,
		+1.5707963267948966192e+00f,
		+1.0471975511965977462e+00f
	};
	static const float CC_float[]    =
	{
		-5.0000000000000000000000e-01f, // 1/2!
		+4.1666666666666666666667e-02f, // 1/4!
		-1.3888888888888888888889e-03f, // 1/6!
		+2.4801587301587301587302e-05f, // 1/8!
		-2.7557319223985890652557e-07f, // 1/10!
		+2.0876756987868098979210e-09f, // 1/12!
		-1.1470745597729724713852e-11f, // 1/14!
		+4.7794773323873852974382e-14f, // 1/16!   
	};

//=============================================================================

	#undef __fsel
#ifndef __SNC__
	//forced intrinsics(compiler does not always use them otherwise)
	//rounds x to integer.
	static ILINE double __fctid(double x)
	{
		double r;
		__asm__ ("fctid %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	static ILINE float __fctid(float x)
	{
		float r;
		__asm__ ("fctid %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	//rounds x to integer (towards zero)
	static ILINE double __fctidz(double x)
	{
		double r;
		__asm__ ("fctidz %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	static ILINE float __fctidz(float x)
	{
		float r;
		__asm__ ("fctidz %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	//converts x to double
	static ILINE double __fcfid(double x)
	{
		double r;
		__asm__ ("fcfid %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	static ILINE float __fcfid(float x)
	{
		float r;
		__asm__ ("fcfid %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	//computes |x|
	static ILINE double __fabs_(double x)
	{
		double r;
		__asm__ ("fabs %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	//computes |x|
	static ILINE float __fabs_(float x)
	{
		float r;
		__asm__ ("fabs %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	//computes an estimate of 1.0 / sqrt(x)
	static ILINE double __frsqrte(double x)
	{
		double r;
		__asm__ ("frsqrte %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	static ILINE float __frsqrte(float x)
	{
		float r;
		__asm__ ("frsqrte %0, %1"
			: "=f"(r) : "f"(x));
		return r;
	}

	//compute an estimate of 1.0 / x
	static ILINE double __fres(double x)
	{
	double r;
	__asm__ ("fres %0, %1"
	: "=f"(r) : "f"(x));
	return r;
	}

	//compute an estimate of 1.0 / x
	static ILINE float __fres(float x)
	{
	float r;
	__asm__ ("fres %0, %1"
	: "=f"(r) : "f"(x));
	return r;
	}

	//does a float selection
	#define __fsel(c, gte, lt)	({\
		float ret;\
		__asm__ ("fsel %0, %1, %2, %3" : "=f"(ret) : "f"(c), "f"(gte), "f"(lt));\
		ret;})

	#define __fsel_double(c, gte, lt)	({\
		double ret;\
		__asm__ ("fsel %0, %1, %2, %3" : "=f"(ret) : "f"(c), "f"(gte), "f"(lt));\
		ret;})

#else	//__SNC__
	#include <ppu_intrinsics.h>
	#define __fsel_double __fsel
	#define __fabs_ __fabs
	#define __frsqrte __builtin_frsqrte
	#define __fctidz __builtin_fctidz
	#define __fres __builtin_fre
#endif

	//computes the square root of x
	static double __fsqrt_(double x)
	{
		// Estimate, and then three iterations of Newton-Raphson.
		double x0 = __frsqrte(x);
		double x0x = x0 * x;
		const double HALFD = 0.5;
		double x0_2 = x0 * HALFD;
		x0 = (x0x*x0_2 - HALFD)*-x0 + x0;

		double x1 = x0;
		double x1x = x1 * x;
		double x1_2 = x1 * HALFD;
		x1 = (x1x*x1_2 - HALFD)*-x1 + x1;

		double x2 = x1;
		double x2x = x2 * x;
		double x2_2 = x2 * HALFD;
		x2 = (x2x*x2_2 - HALFD)*-x2 + x2;

		return __fsel_double(__fabs_(x) - 1e-50, x * x2, 0.0);
	}

	//computes the square root of x
	static float __fsqrt_(float _x)
	{
		const double x = (double)_x;
		// Estimate, and then three iterations of Newton-Raphson.
		double x0 = __frsqrte(x);
		double x0x = x0 * x;
		const double HALFD = 0.5;
		double x0_2 = x0 * HALFD;
		x0 = (x0x*x0_2 - HALFD)*-x0 + x0;

		double x1 = x0;
		double x1x = x1 * x;
		double x1_2 = x1 * HALFD;
		x1 = (x1x*x1_2 - HALFD)*-x1 + x1;

		double x2 = x1;
		double x2x = x2 * x;
		double x2_2 = x2 * HALFD;
		x2 = (x2x*x2_2 - HALFD)*-x2 + x2;

		return __fsel(__fabs_(_x) - 1e-30f, _x * (float)x2, 0.0f);
	}

	//rounds x to nearest integer.
	static ILINE double __round(double x)
	{
		return __fcfid(__fctid(x));
	}

	static ILINE float __round(float x)
	{
		return __fcfid(__fctid(x));
	}

	//truncates x (round to zero).
	static ILINE double __trunc(double x)
	{
#ifdef __SNC__
		return __builtin_fctiwz(x);
#else
		return __fcfid(__fctidz(x));
#endif
	}

	static ILINE float __trunc(float x)
	{
#ifdef __SNC__
		return __builtin_fctiwz(x);
#else
		return __fcfid(__fctidz(x));
#endif
	}

	//divide a by b
	static ILINE double __fdiv(double a, double b)
	{
		double x0 = __fres(b);      // Estimate
		// Two steps of Newton-Raphson
		double b_inv = ((b * x0 - 1.0) * -x0) + x0;
		b_inv = ((b * b_inv - 1.0) * -b_inv) + b_inv;
		return a * b_inv;
	}

	//divide a by b
	static ILINE float __fdiv(float a, float b)
	{
		float x0 = __fres(b);      // Estimate
		// Two steps of Newton-Raphson
		float b_inv = ((b * x0 - 1.0f) * -x0) + x0;
		b_inv = ((b * b_inv - 1.0f) * -b_inv) + b_inv;
		return a * b_inv;
	}
};//anon.namespace

//=============================================================================

//actual trigonometric funcs

//computes the sine of x
ILINE double ps3_sin(double x)
{
	double absx = __fabs_(x);
	double sign = __fsel_double(x, (double)1.0, (double)-1.0);
	const double INV_PI  = (double)1.0 / (double)3.14159265358979323846;
	double num_cycles = __round(absx * INV_PI);
	const double HALFD = (double)0.5;
	double is_even = __trunc(num_cycles * HALFD) - (num_cycles * HALFD);
	const double PI      = (double)3.14159265358979323846;
	double norm_x = absx - PI * num_cycles;
	double y = norm_x * norm_x;
	double taylor = ((((((((SC_double[7] * y + SC_double[6])
								 * y + SC_double[5])
								 * y + SC_double[4])
								 * y + SC_double[3])
								 * y + SC_double[2])
								 * y + SC_double[1])
								 * y + SC_double[0])
								 * y);
								 sign = __fsel_double(is_even, sign, -sign);
								 double result = norm_x + norm_x * taylor;
								 result *= sign;
								 return result;
}

ILINE float ps3_sin(float x)
{
	float absx = __fabs_(x);
	float sign = __fsel(x, (float)1.0, (float)-1.0);
	const float INV_PI  = (float)1.0 / (float)3.14159265358979323846;
	float num_cycles = __round(absx * INV_PI);
	const float HALFD = (float)0.5;
	float is_even = __trunc(num_cycles * HALFD) - (num_cycles * HALFD);
	const float PI      = (float)3.14159265358979323846;
	float norm_x = absx - PI * num_cycles;
	float y = norm_x * norm_x;
	float taylor = ((((((((SC_float[7] * y + SC_float[6])
		* y + SC_float[5])
		* y + SC_float[4])
		* y + SC_float[3])
		* y + SC_float[2])
		* y + SC_float[1])
		* y + SC_float[0])
		* y);
	sign = __fsel(is_even, sign, -sign);
	float result = norm_x + norm_x * taylor;
	result *= sign;
	return result;
}

//computes the cosine of x
ILINE double ps3_cos(double x)
{
	const double HALF_PI = 3.14159265358979323846 / 2.0;
	return ps3_sin(x + HALF_PI);
}

//computes the cosine of x
ILINE float ps3_cos(float x)
{
	const float HALF_PI = 3.14159265358979323846f / 2.0f;
	return ps3_sin(x + HALF_PI);
}

//computes the sine and cosine of x
void ps3_sincos(double x, double *s, double *c)
{
	double absx = __fabs_(x);
	const double HALF_PI = (double)3.14159265358979323846 / (double)2.0;
	double cabsx = absx + HALF_PI;
	double sign = __fsel_double(x, (double)1.0, (double)-1.0);
	const double HALFD = (double)0.5;
	const double INV_PI  = (double)1.0 / (double)3.14159265358979323846;
	double num_cycles  = __round(absx * INV_PI);
	double is_even = __trunc(num_cycles * HALFD) - (num_cycles * HALFD);
	const double PI      = (double)3.14159265358979323846;
	double norm_x  = absx - PI * num_cycles;
	double norm_cx = cabsx - PI * num_cycles - HALF_PI;
	double y  = norm_x * norm_x;
	double cy = norm_cx * norm_cx;
	double taylor = ((((((((SC_double[7] * y + SC_double[6])
		* y + SC_double[5])
		* y + SC_double[4])
		* y + SC_double[3])
		* y + SC_double[2])
		* y + SC_double[1])
		* y + SC_double[0])
		* y);
	double taylorc = ((((((((CC_double[7] * cy + CC_double[6])
		* cy + CC_double[5])
		* cy + CC_double[4])
		* cy + CC_double[3])
		* cy + CC_double[2])
		* cy + CC_double[1])
		* cy + CC_double[0])
		* cy);
	sign = __fsel_double(is_even, sign, -sign);
	double cos_sign = __fsel_double(is_even, (double)1.0, (double)-1.0);
	double sine = norm_x + norm_x * taylor;
	double cosine = (double)1.0 + taylorc;
	sine *= sign;
	cosine *= cos_sign;
	*s = sine;
	*c = cosine;
}

void ps3_sincos(float x, float *s, float *c)
{
	float absx = __fabs_(x);
	const float HALF_PI = (float)3.14159265358979323846 / (float)2.0;
	float cabsx = absx + HALF_PI;
	float sign = __fsel(x, (float)1.0, (float)-1.0);
	const float HALFD = (float)0.5;
	const float INV_PI  = (float)1.0 / (float)3.14159265358979323846;
	float num_cycles  = __round(absx * INV_PI);
	float is_even = __trunc(num_cycles * HALFD) - (num_cycles * HALFD);
	const float PI      = (float)3.14159265358979323846;
	float norm_x  = absx - PI * num_cycles;
	float norm_cx = cabsx - PI * num_cycles - HALF_PI;
	float y  = norm_x * norm_x;
	float cy = norm_cx * norm_cx;
	float taylor = ((((((((SC_float[7] * y + SC_float[6])
		* y + SC_float[5])
		* y + SC_float[4])
		* y + SC_float[3])
		* y + SC_float[2])
		* y + SC_float[1])
		* y + SC_float[0])
		* y);
	float taylorc = ((((((((CC_float[7] * cy + CC_float[6])
		* cy + CC_float[5])
		* cy + CC_float[4])
		* cy + CC_float[3])
		* cy + CC_float[2])
		* cy + CC_float[1])
		* cy + CC_float[0])
		* cy);
	sign = __fsel(is_even, sign, -sign);
	float cos_sign = __fsel(is_even, (float)1.0, (float)-1.0);
	float sine = norm_x + norm_x * taylor;
	float cosine = (float)1.0 + taylorc;
	sine *= sign;
	cosine *= cos_sign;
	*s = sine;
	*c = cosine;
}

//computes the tangent of x
ILINE double ps3_tan(double x)
{
	double s, c;
	ps3_sincos(x, &s, &c);
	double absc = __fabs_(c);
	double is_cos_not_zero = absc - 1e-50;
	c = __fsel_double(is_cos_not_zero, c, 1e-50);
	return __fdiv(s,c);
}

//computes the tangent of x
ILINE float ps3_tan(float x)
{
	float s, c;
	ps3_sincos(x, &s, &c);
	float absc = __fabs_(c);
	float is_cos_not_zero = absc - 1e-30f;
	c = __fsel(is_cos_not_zero, c, 1e-30f);
	return __fdiv(s,c);
}

//computes the inverse sine of x
ILINE double ps3_asin(double x)
{
	//a branchless implementation of the algorithm found
	//in "Software Manual for the Elementary Functions"
	const double HALF_PI = 3.14159265358979323846 / 2.0;
	double abs_x = __fabs_(x);
	const double HALFD = 0.5;
	double g = __fsel_double(abs_x - HALFD, (1.0 - abs_x) * HALFD, abs_x * abs_x);
	double y = __fsel_double(abs_x - HALFD, -2.0 * __fsqrt_(g), abs_x);
	double P = (((((PC_double[4] * g + PC_double[3]) * g + PC_double[2]) * g + PC_double[1]) * g + PC_double[0]) * g);
	double Q = (((((g + QC_double[4]) * g + QC_double[3])	* g + QC_double[2]) * g + QC_double[1]) * g + QC_double[0]);
	double R = y + y * __fdiv(P, Q);
	double res = __fsel_double(abs_x - HALFD, HALF_PI + R, R);
	return __fsel_double(x, res, -res);
}

//computes the inverse sine of x
ILINE float ps3_asin(float x)
{
	//a branchless implementation of the algorithm found
	//in "Software Manual for the Elementary Functions"
	const float HALF_PI = 3.14159265358979323846f / 2.0f;
	float abs_x = __fabs_(x);
	const float HALFD = 0.5f;
	float g = __fsel(abs_x - HALFD, (1.0f - abs_x) * HALFD, abs_x * abs_x);
	float y = __fsel(abs_x - HALFD, -2.0f * __fsqrt_(g), abs_x);
	float P = (((((PC_float[4] * g + PC_float[3]) * g + PC_float[2]) * g + PC_float[1]) * g + PC_float[0]) * g);
	float Q = (((((g + QC_float[4]) * g + QC_float[3])	* g + QC_float[2]) * g + QC_float[1]) * g + QC_float[0]);
	float R = y + y * __fdiv(P, Q);
	float res = __fsel(abs_x - HALFD, HALF_PI + R, R);
	return __fsel(x, res, -res);
}

//computes the inverse cosine of x
ILINE double ps3_acos(double x)
{
	//a branchless implementation of the algorithm found
	//in "Software Manual for the Elementary Functions"
	double abs_x = __fabs_(x);
	const double HALFD = 0.5;
	const double HALF_PI = 3.14159265358979323846 / 2.0;
	const double PI_4    = 3.14159265358979323846 / 4.0;
	double g = __fsel_double(abs_x - HALFD, (1.0 - abs_x) * HALFD, abs_x * abs_x);
	double y = __fsel_double(abs_x - HALFD, -2.0 * __fsqrt_(g), abs_x);
	double b = __fsel_double(abs_x - HALFD, HALF_PI, PI_4);
	double a = __fsel_double(abs_x - HALFD, 0.0, PI_4);
	double P = (((((PC_double[4] * g + PC_double[3]) * g + PC_double[2])	* g + PC_double[1]) * g + PC_double[0])	* g);
	double Q = (((((g + QC_double[4]) * g + QC_double[3])	* g + QC_double[2]) * g + QC_double[1])	* g + QC_double[0]);
	double R = y + y * __fdiv(P, Q);
	return __fsel_double(x, 2.0 * a - R, 2.0 * b + R);
}

ILINE float ps3_acos(float x)
{	
	//a branchless implementation of the algorithm found
	//in "Software Manual for the Elementary Functions"
	float abs_x = __fabs_(x);
	const float HALFD = 0.5f;
	const float HALF_PI = 3.14159265358979323846f / 2.0f;
	const float PI_4    = 3.14159265358979323846f / 4.0f;
	float g = __fsel(abs_x - HALFD, (1.0f - abs_x ) * HALFD, abs_x * abs_x);	
	float y = __fsel(abs_x - HALFD, -2.0f * __fsqrt_(g), abs_x);
	float b = __fsel(abs_x - HALFD, HALF_PI, PI_4);
	float a = __fsel(abs_x - HALFD, 0.0f, PI_4);
	float P = (((((PC_float[4] * g + PC_float[3]) * g + PC_float[2])	* g + PC_float[1]) * g + PC_float[0])	* g);
	float Q = (((((g + QC_float[4]) * g + QC_float[3])	* g + QC_float[2]) * g + QC_float[1])	* g + QC_float[0]);
	float R = y + y * __fdiv(P, Q);	
	return __fsel(x, 2.0f * a - R, 2.0f * b + R);	
}

//computes the inverse tangent of x
//a branchless implementation of the algorithm found
//in "Software Manual for the Elementary Functions".
ILINE double ps3_atan(double x)
{
	double abs_x = __fabs_(x);
	double lt_one = abs_x - (double)1.0;
	const double ONE = (double)1.0;
	double f = __fsel_double(lt_one, __fdiv(ONE, abs_x), abs_x);
	const double ROOT3   = (double)1.732050807568877293527;
	double A = ROOT3 - ONE;
	double lt_2_minus_root3 = f - (double)2.0 + ROOT3;
	f = __fsel_double(lt_2_minus_root3, __fdiv(((A * f - ONE) + f), (f + ROOT3)), f);
	double g = f * f;
	double P = ((((TPC_double[3] * g + TPC_double[2])	* g + TPC_double[1])	* g + TPC_double[0])	* g);
	double Q = ((((g + TQC_double[3]) * g + TQC_double[2]) * g + TQC_double[1]) * g + TQC_double[0]);
	double R = f * __fdiv(P,Q) + f;
	R = __fsel_double(lt_one, -R, R);
	double case0 = __fsel_double(lt_one, -ONE, ONE);
	case0 = __fsel_double(case0, -lt_2_minus_root3, -ONE);
	case0 = __fsel_double(case0, ONE, -ONE);
	double case1 = __fsel_double(lt_one, -ONE, ONE);
	case1 = __fsel_double(case1, lt_2_minus_root3, -ONE);
	case1 = __fsel_double(case1, ONE, -ONE);
	double case2 = __fsel_double(-lt_one, -ONE, ONE);
	case2 = __fsel_double(case2, -lt_2_minus_root3, -ONE);
	case2 = __fsel_double(case2, ONE, -ONE);
	double a = __fsel_double(case0, TA_double[0], TA_double[3]);
	a = __fsel_double(case1, TA_double[1], a);
	a = __fsel_double(case2, TA_double[2], a);
	R += a;
	return __fsel_double(x, R, -R);
}

ILINE float ps3_atan(float x)
{
	float abs_x = __fabs_(x);
	float lt_one = abs_x - (float)1.0;
	const float ONE = (float)1.0;
	float f = __fsel(lt_one, __fdiv(ONE, abs_x), abs_x);
	const float ROOT3   = (float)1.732050807568877293527;
	float A = ROOT3 - ONE;
	float lt_2_minus_root3 = f - (float)2.0 + ROOT3;
	f = __fsel(lt_2_minus_root3, __fdiv(((A * f - ONE) + f), (f + ROOT3)), f);
	float g = f * f;
	float P = ((((TPC_float[3] * g + TPC_float[2])	* g + TPC_float[1])	* g + TPC_float[0])	* g);
	float Q = ((((g + TQC_float[3]) * g + TQC_float[2]) * g + TQC_float[1]) * g + TQC_float[0]);
	float R = f * __fdiv(P,Q) + f;
	R = __fsel(lt_one, -R, R);
	float case0 = __fsel(lt_one, -ONE, ONE);
	case0 = __fsel(case0, -lt_2_minus_root3, -ONE);
	case0 = __fsel(case0, ONE, -ONE);
	float case1 = __fsel(lt_one, -ONE, ONE);
	case1 = __fsel(case1, lt_2_minus_root3, -ONE);
	case1 = __fsel(case1, ONE, -ONE);
	float case2 = __fsel(-lt_one, -ONE, ONE);
	case2 = __fsel(case2, -lt_2_minus_root3, -ONE);
	case2 = __fsel(case2, ONE, -ONE);
	float a = __fsel(case0, TA_float[0], TA_float[3]);
	a = __fsel(case1, TA_float[1], a);
	a = __fsel(case2, TA_float[2], a);
	R += a;
	return __fsel(x, R, -R);
}

//compute arc tangent of x / @a y, with the signs of x and y
ILINE double ps3_atan2(double x, double y)
{
	double absy = __fabs_(y);
	double is_y_not_zero = absy - 1e-50;
	const double PI = 3.14159265358979323846;
	const double HALF_PI = PI / 2.0;
	double res = __fsel_double(is_y_not_zero, ps3_atan(__fdiv(x, y)), __fsel_double(x, HALF_PI, -HALF_PI));
	return __fsel_double(y, res,	__fsel_double(is_y_not_zero,__fsel_double(x,res + PI,res - PI),	res));
}

//compute arc tangent of x / @a y, with the signs of x and y
ILINE float ps3_atan2(float x, float y)
{
	float absy = __fabs_(y);
	float is_y_not_zero = absy - 1e-30f;
	const float PI = 3.14159265358979323846f;
	const float HALF_PI = PI / 2.0;
	float res = __fsel(is_y_not_zero, ps3_atan(__fdiv(x, y)), __fsel(x, HALF_PI, -HALF_PI));
	return __fsel(y, res,	__fsel(is_y_not_zero,	__fsel(x,	res + PI,	res - PI), res));
}

//=============================================================================
#ifndef __SNC__
//wrappers for linker symbol replacement

double __wrap_sin(double) asm("__wrap_sin");
//a wrapper function for symbol replacement
double __wrap_sin(double x)
{
	return ps3_sin(x);
}

double __wrap_cos(double) asm("__wrap_cos");
//a wrapper function for symbol replacement
double __wrap_cos(double x)
{
	return ps3_cos(x);
}

double __wrap_tan(double) asm("__wrap_tan");
//a wrapper function for symbol replacement
double __wrap_tan(double x)
{
	return ps3_tan(x);
}

double __wrap_asin(double) asm("__wrap_asin");
//a wrapper function for symbol replacement
double __wrap_asin(double x)
{
	return ps3_asin(x);
}

double __wrap_acos(double) asm("__wrap_acos");
//a wrapper function for symbol replacement
double __wrap_acos(double x)
{
	return ps3_acos(x);
}

double __wrap_atan(double) asm("__wrap_atan");
//a wrapper function for symbol replacement
double __wrap_atan(double x)
{
	return ps3_atan(x);
}

double __wrap_atan2(double, double) asm("__wrap_atan2");
//a wrapper function for symbol replacement
double __wrap_atan2(double x, double y)
{
	return ps3_atan2(x, y);
}

float __wrap_sinf(float) asm("__wrap_sinf");
//a wrapper function for symbol replacement
float __wrap_sinf(float x)
{
	return ps3_sin(x);
}

float __wrap_cosf(float) asm("__wrap_cosf");
//a wrapper function for symbol replacement
float __wrap_cosf(float x)
{
	return ps3_cos(x);
}

float __wrap_tanf(float) asm("__wrap_tanf");
//a wrapper function for symbol replacement
float __wrap_tanf(float x)
{
	return ps3_tan(x);
}

float __wrap_asinf(float) asm("__wrap_asinf");
//a wrapper function for symbol replacement
float __wrap_asinf(float x)
{
	return ps3_asin(x);
}

float __wrap_acosf(float) asm("__wrap_acosf");
//a wrapper function for symbol replacement
float __wrap_acosf(float x)
{
	return ps3_acos(x);
}

float __wrap_atanf(float) asm("__wrap_atanf");
//a wrapper function for symbol replacement
float __wrap_atanf(float x)
{
	return ps3_atan(x);
}

float __wrap_atan2f(float, float) asm("__wrap_atan2f");
//a wrapper function for symbol replacement
float __wrap_atan2f(float x, float y)
{
	return ps3_atan2(x, y);
}
#endif

#endif//PS3+PS3OPT