#ifndef __CRYMATCHMAKING_H__
#define __CRYMATCHMAKING_H__

#pragma once

#include "ICryMatchMaking.h"
#include "Lobby/CryLobby.h"
#include "Socket/NetAddress.h"

const ECryLobbyError eCLE_Pending = eCLE_NumErrors;
const uint32 CryMatchMakingConnectionTimeOut = 10000;
const uint32 CryMatchMakingHostMigrationDefaultTimeout = 10000;

#define MAX_MATCHMAKING_SESSION_USER_DATA	64
#define MAX_MATCHMAKING_SESSIONS					4
#define MAX_MATCHMAKING_TASKS							4
#define MAX_MATCHMAKING_PARAMS						3


typedef uint32 CryMatchMakingConnectionUID;
const CryMatchMakingConnectionUID CryMatchMakingInvalidConnectionUID = 0;

typedef uint32 CryMatchMakingTaskID;
const CryMatchMakingTaskID CryMatchMakingInvalidTaskID = 0xffffffff;

typedef uint32 CryMatchMakingConnectionID;
const CryMatchMakingConnectionID CryMatchMakingInvalidConnectionID = 0xffffffff;

class CCryMatchMaking : public CMultiThreadRefCount, public ICryMatchMaking
{
public:
																	CCryMatchMaking(CCryLobby* lobby, CCryLobbyService* service);

					ECryLobbyError					Initialise();
					ECryLobbyError					Terminate();
					void										OnPacket(const TNetAddress& addr, const uint8* data, uint32 length);
					void										OnError(const TNetAddress& addr, ESocketError error, CryLobbySendID sendID);
					void										OnSendComplete(const TNetAddress& addr, CryLobbySendID sendID);
	virtual void										CancelTask(CryLobbyTaskID lTaskID);
					void										SessionDisconnectRemoteConnection(CrySessionHandle gh, const TNetAddress& addr);
	virtual void										SessionJoinFromConsole(void);
	virtual bool										IsDead()const	{ return false; }

	// The CrySessionHandle passed to the game consists of
	// the local session handle used to identify the session in other session calls
	// a uid generated by the host to identify the connection used.
	// This id is passed around when the host receives a connection so any peer can identify a connection from this.
	CrySessionHandle								CreateGameSessionHandle(CrySessionHandle h, CryMatchMakingConnectionUID uid);
	CryMatchMakingConnectionUID			CreateConnectionUID();
	CrySessionHandle								GetSessionHandleFromGameSessionHandle(CrySessionHandle gh);
	CryMatchMakingConnectionUID			GetConnectionUIDFromGameSessionHandle(CrySessionHandle gh);

	bool														GetAddressFromChannelID(TNetChannelID channelID, TNetAddress& address);
	bool														GetChannelIDFromAddress(const TNetAddress& address, TNetChannelID* pChannelID);

	CrySessionHandle								GetCurrentSessionHandle(void) const;
	CryMatchMakingConnectionUID			GetLocalConnectionUID(void) const;
	CryMatchMakingConnectionUID			GetHostConnectionUID(void) const;

#if NETWORK_HOST_MIGRATION
	virtual void										HostMigrationInitiate(void);
	virtual ECryLobbyError					HostMigrationServer(void);
	virtual bool										IsHostMigrationFinished(void);
	virtual bool										GetNewHostAddress(char* address) = 0;
	virtual void										HostMigrationSetNewServerAddress(TNetAddress& address);
#endif

protected:
	struct  SSession
	{
		struct SLConnection
		{
			CryMatchMakingConnectionUID	uid;
			uint8												numUsers;
		}*														localConnection;

		struct SRConnection 
		{
			bool													TimerStarted()			{ return timerStarted; }
			void													StartTimer()				{ timerStarted = true; timer = g_time; }
			void													StopTimer()					{ timerStarted = false; }
			int64													GetTimer()					{ return g_time.GetMilliSecondsAsInt64() - timer.GetMilliSecondsAsInt64(); }

			CTimeValue									timer;
			CryLobbyConnectionID				connectionID;
			CryMatchMakingConnectionUID	uid;
			uint8												numUsers;
			bool												used;
			bool												timerStarted;
		}*														remoteConnection[MAX_LOBBY_CONNECTIONS];

		CryMatchMakingConnectionID		hostConnectionID;
		bool													used;
		bool													host;
	};

	struct  STask
	{
		bool													TimerStarted()			{ return timerStarted; }
		void													StartTimer()				{ timerStarted = true; timer = g_time; }
		int64													GetTimer()					{ return g_time.GetMilliSecondsAsInt64() - timer.GetMilliSecondsAsInt64(); }

		CTimeValue										timer;
		CryLobbyTaskID								lTaskID;
		ECryLobbyError								error;
		CryMatchMakingTaskID					returnTaskID;
		uint32												startedTask;
		uint32												subTask;
		CrySessionHandle							session;
		void*													cb;
		void*													cbArg;
		CryLobbySendID								sendID;
		ECryLobbyError								sendStatus;
		TMemHdl												params[MAX_MATCHMAKING_PARAMS];
		uint32												numParams[MAX_MATCHMAKING_PARAMS];
		bool													used;
		bool													running;
		bool													timerStarted;
		bool													canceled;
	};

	void														Tick(CTimeValue tv);
	ECryLobbyError									StartTask(uint32 etask, bool startRunning, CryMatchMakingTaskID* mmTaskID, CryLobbyTaskID* lTaskID, CrySessionHandle h, void* cb, void* cbArg);
	void														StartSubTask(uint32 etask, CryMatchMakingTaskID mmTaskID);
	void														FreeTask(CryMatchMakingTaskID mmTaskID);
	ECryLobbyError									CreateTaskParam(CryMatchMakingTaskID mmTaskID, uint32 param, const void* paramData, uint32 numParams, size_t paramSize);
	ECryLobbyError									CreateSessionHandle(CrySessionHandle *h, bool host, int numUsers);
	void														FreeSessionHandle(CrySessionHandle h);
	CryMatchMakingConnectionID			AddRemoteConnection(CrySessionHandle h, CryLobbyConnectionID connectionID, CryMatchMakingConnectionUID uid, uint32 numUsers);
	virtual void										FreeRemoteConnection(CrySessionHandle h, CryMatchMakingConnectionID id);
	bool														FindLocalConnectionFromUID(CryMatchMakingConnectionUID uid, CrySessionHandle* h);
	bool														FindConnectionFromUID(CryMatchMakingConnectionUID uid, CrySessionHandle* h, CryMatchMakingConnectionID* id);
	bool														FindConnectionFromLobbyConnectionID(CryLobbyConnectionID lobbyConnectionID, CryMatchMakingConnectionUID* puid);
	CryMatchMakingTaskID						FindTaskFromSendID(CryLobbySendID sendID);
	CryMatchMakingTaskID						FindTaskFromTaskConnectionID(uint32 task, CryLobbyConnectionID id);
	CryMatchMakingTaskID						FindTaskFromTaskTaskID(uint32 task, CryMatchMakingTaskID mmTaskID);
	void														UpdateTaskError(CryMatchMakingTaskID mmTaskID, ECryLobbyError error);
	void														SessionDisconnectRemoteConnectionViaNub(EDisconnectionCause cause, CrySessionHandle gh, const char* reason);

	ESocketError										Send(CryMatchMakingTaskID mmTaskID, const uint8* buffer, size_t length, const TNetAddress& to, bool reliable);
	ESocketError										Send(CryMatchMakingTaskID mmTaskID, const uint8* buffer, size_t length, CrySessionHandle h, CryMatchMakingConnectionID connectionID, bool reliable);
	void														SendToAll(CryMatchMakingTaskID mmTaskID, const uint8* buffer, size_t length, CrySessionHandle h, CryMatchMakingConnectionID skipID, bool reliable);

	void														ProcessSessionDeleteRemoteConnection(const TNetAddress& addr, const uint8* data, uint32 length);

#if NETWORK_HOST_MIGRATION
	TNetAddress											m_newHostAddress;
#endif
	CryMatchMakingConnectionUID			m_connectionUIDCounter;
	SSession*												m_sessions[MAX_MATCHMAKING_SESSIONS];
	STask*													m_task[MAX_MATCHMAKING_TASKS];
	CCryLobby*											m_lobby;
	CCryLobbyService*								m_pService;
	bool														m_hostMigrationFinished;
#if NETWORK_HOST_MIGRATION
	bool														m_newHostAddressValid;
#endif
};

const uint32 PacketCryMatchMakingConnectionUIDSize	= PacketUINT32Size;
inline void AddPacketCryMatchMakingConnectionUID(uint8* buffer, uint32 bufferSize, uint32* bufferPos, CryMatchMakingConnectionUID data)
{
	AddPacketUINT32(buffer, bufferSize, bufferPos, data);
}

inline void RemovePacketCryMatchMakingConnectionUID(const uint8* buffer, uint32 bufferSize, uint32* bufferPos, CryMatchMakingConnectionUID* data)
{
	RemovePacketUINT32(buffer, bufferSize, bufferPos, data);
}

#endif // __CRYMATCHMAKING_H__

