#include "StdAfx.h"

// This file should never be compiled on a different platform
#if defined(PS3)

#include "CryMTrace.h"
#include <pthread.h>
#include <CryThread.h>
#include <netex/net.h>
#include <netex/errno.h>
#include <sys/socket.h>
#include <sys/ppu_thread.h>
#include <sys/sys_time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <cstring>

// The default path to the mtrace ipconfig file 
#define MTRACE_IPCONFIG_PATH "/app_home/mtrace_ipconfig.txt"

// Define this to verify the default
//#define MTRACE_VERIFY_ALLOCSTATS 1
#undef MTRACE_VERIFY_ALLOCSTATS

#if defined(MTRACE_VERIFY_ALLOCSTATS)
#  define MTRACE_CHECK_ALLOC { \
  SystemStats stats;\
  MTrace::SystemStatistics(stats);\
  if (stats.allocations.freed_memory > stats.allocations.allocated_memory)\
    snPause();\
  ; }
#else 
#  define MTRACE_CHECK_ALLOC 
#endif 

// Enable debugging of the data stream sent to the mtrace server. If
// this is enabled, the conncection will append the stream debug tag
// at the end of each single binary blob of data sent over the
// network to aide in finding errors when the stream of data sent has
// consistency errors!
//#define ENABLE_STREAM_DEBUGGING 1
#if defined(ENABLE_STREAM_DEBUGGING)
#  define STREAM_DEBUG_TAG (0xdeadbeefU)
#endif

// The default port to which mtrace connects 
#define MTRACE_DEFAULT_PORT ((uint16_t)(33333))

// The default address for that is connected 
#define MTRACE_DEFAULT_ADDRESS "192.168.9.146"

// The maximum number of traced threads 
#define MTRACE_MAX_THREAD_COUNT (32)

// The number of seconds without activity until a socket conncection
// is considered dead. This is not exactly measured in seconds but
// rather in retries with the timeout between successive retries has
// the granularity of one second. 
#define MTRACE_SOCKET_TIMEOUT (16)

// The maximum size (in bytes) that the CChunk helper structure can
// allocate on the stack. Warning: Do not make this structure insanely
// large as the stack size of every thread is not known a-priori! 
// Todo: implement a stack overflow check in each constructor call!
#define MTRACE_MAX_CHUNKSIZE (2*256)

// Define this to enable ring buffer debug assertions
#define MTRACE_RINGBUFFER_DEBUG 

// Define MTRACE_RINGBUFFER Assertions 
#if defined(MTRACE_RINGBUFFER_DEBUG)
#  define MTRACE_RINGBUFFER_ASSERT(x)           \
  do { if(!(x)) snPause(); } while(false);
#else
#  define MTRACE_RINGBUFFER_ASSERT(x)
#endif 

// Required for thread name support.
extern const char* GetRegisteredThreadName(uint32);

// We have to undefine these here or the compiler will go haywire like
// you have never seen before.
#undef calloc
#undef malloc
#undef free
#undef realloc
#undef memalign

namespace MTrace
{
  // The global memory operation id 
  uint32_t g_memory_operation_id = 0;
}

namespace
{ 
	// Tags sent by the ps3 application
	enum Function
	{
		FN_NONE     = 0,
		FN_calloc   = 1,
		FN_malloc   = 2,
		FN_free     = 4,
		FN_realloc  = 8,
		FN_memalign = 16,
		FN_END      = 0x7fffffff
	};

  struct MemoryOperation 
  {
    uint32_t operation_id; 
    uint16_t operation_type; 
    uint16_t thread_id;
    void*    primary;
    void*    secondary;
    uint32_t size; 
  }; 

  // The buffer into which the mtrace server will send back data 
  static MTrace::SystemStats g_traced_stats; 

  // The frame statistics 
  struct GlobalFrameStats
  {
    int32_t frame_counter;
    uint64_t init_ops;
    uint64_t init_allocs;
    uint64_t init_freed;
  };
  static GlobalFrameStats g_frame_stats = 
  { -255, 0UL, 0UL, 0UL };

  // The local statistics gathered by each thread is contained as a
  // thread-local pointer to avoid expensive operations 
  THREADLOCAL MTrace::AllocationStats* thread_alloc_stats = 0;

  // The array of thread local statistics dependin. For each thread a
  // pointer to an instance to this array is stored in thread local
  // storage so that updates can occur without expensive mutual
  // exclusion or equally inefficient atomic operations. 
  static MTrace::AllocationStats thread_alloc_table[MTRACE_MAX_THREAD_COUNT];

  // The index to the next free thread alloc stats entry in the thread
  // alloc table. 
  static uint16_t next_thread_alloc_index = 0;

  // Increment the global memory operation id 
  static inline uint32_t IncrementOpId() 
  { 
    return (uint32_t)CryInterlockedIncrement(
      (int volatile*) &MTrace::g_memory_operation_id);
  } 

  // The index to the next free thread alloc stats entry in the thread
  // alloc table. 
  static int AllocStats_SpinLock = 0; 

  // Call ShutDown at exit 
  void AtExit() { MTrace::Shutdown(); }

  // Retrieve a pointer to the allocations statistics object for the
  // current thread. A new thread allocation structure is assigned to
  // the current thread if the pointer to the current thread's assigned
  // allocation structure was not set previously. 
  static __attribute__((always_inline)) 
  MTrace::AllocationStats* GetAllocationStats()
   {
    if (!thread_alloc_stats) 
    { 
      CrySpinLock(&AllocStats_SpinLock, 0, 1); 

      // In case the debugger ever stops here, Please increase 
      // MTRACE_MAX_THREAD_COUNT at the top of the file 
      if (next_thread_alloc_index == MTRACE_MAX_THREAD_COUNT)
        abort();
      thread_alloc_stats = &thread_alloc_table[next_thread_alloc_index++];
      thread_alloc_stats->thread_local_id = next_thread_alloc_index-1;
      thread_alloc_stats->thread_id = GetCurrentThreadId();
      thread_alloc_stats->thread_sys_id = GetCurrentThreadSystemId();
      thread_alloc_stats->thread_recording = 1; 

      CryReleaseSpinLock(&AllocStats_SpinLock, 0);
    } 
    return thread_alloc_stats;
  }

  // A stack based small utility buffer. Chunks can accumulate writes
  // on the stack so that the gathered data can get written in one
  // large chunk.
  template<size_t Size> 
  class CChunk_t
  {
    // The current position within the chunk
    size_t m_CurrentPosition; 

    // The actual chunk buffer 
    char m_Buffer[Size]; 

    // Ensure that the write operation fits into the buffer 
    inline bool CheckSize(size_t size) const
    {
      return (m_CurrentPosition + size < Size);
    }

  public:
    // Constructor
    CChunk_t() : m_CurrentPosition() {}; 

    // Write a value to the buffer, flush if the buffer is full.
    template<typename T> 
    inline void Write(T value)
    {
      if (CheckSize(sizeof value)) 
      { 
        memcpy(m_Buffer + m_CurrentPosition, &value, sizeof(value));
        m_CurrentPosition += sizeof(value);
      }
    }

    // Retrieve a pointer to the data in the chunk buffer 
    const char* GetData() const { return m_Buffer; }

    // Retrieve the size of the data 
    size_t GetSize() const { return m_CurrentPosition; }

    // Retrieve the size of the data 
    size_t GetTotalSize() const { return Size; }

  }; 
  typedef CChunk_t<MTRACE_MAX_CHUNKSIZE> CChunk;

	// Dump a stack trace to a chunked buffer.
	// The data stream containing the trace is terminated by 0xffffffff.
	__attribute__ ((always_inline))
	static inline void Stacktrace(CChunk& chunk)
	{
		register uint64_t sp, ip;

		// Fetch current stack pointer.
		__asm__ __volatile__ (
				"    mr %[sp], 1\n"
				: [sp] "=r" (sp));

		while (true)
		{
			// Locate previous frame.
			uint64_t prevSp = *(uint64_t *)(uint32_t)(sp);
			if (prevSp == 0 || 
        chunk.GetSize() == chunk.GetTotalSize() - sizeof(uint32_t) ) 
        break;
			// Load return address from previous frame.
			ip = *((uint64_t *)(uint32_t)prevSp + 2);
			sp = prevSp;
			chunk.Write(static_cast<uint32_t>(ip));
		}
		chunk.Write(static_cast<uint32_t>(0xffffffffU));
	}

  // Address Catching 
  __attribute__ ((noinline))
  void Catch(const char *fn)
  {
    fprintf(
      stderr, "CatchAddr: addr = 0x%08x, fn = %s\n",
      reinterpret_cast<uint32_t>(MTrace::g_Config.CatchAddr), fn);
    __asm__ volatile ( "tw 31,  1, 1\n" : : : );
  }


  // Thread-Safe size invariant ring buffer.  
  // Blocks on writes if there is not enough space left to complete
  // the write. Reads never block, but may be return incomplete reads
  // if there is not enough data present.
  template<size_t BlockSize>
  class IOBuffer 
  {
    // The local lock variable variabke 
    pthread_mutex_t m_Lock; 

    // The local condition variable 
    pthread_cond_t  m_Cond;

    // The read count in bytes. 
    // This variable may freely wrap back to 0.
    size_t m_read_count; 

    // The write count in bytes. 
    // This variable may freely wrap back to 0.
    size_t m_write_count;

    // The begin (read) cursor.
    char* m_begin;

    // The end (write) cursor.
    char* m_end; 

    // The actual data 
    char  m_data[BlockSize];

    // The flag describing if the buffer is active 
    bool m_Active; 

    // boolean depicting if the data has been initialized 
    bool  m_Initialized;  

    // Verifies that the buffer is completely within it's bounds 
    __attribute__((always_inline)) 
    bool Verify()
    {
      return (m_end   >= m_data && m_end   <= m_data+BlockSize && 
              m_begin >= m_data && m_begin <= m_data+BlockSize);
    } 

  public:
    // Constructor 
    IOBuffer()
      : m_read_count(),
        m_write_count(),
        m_begin(m_data), 
        m_end(m_data), 
        m_Active(true),
        m_Initialized(false)
    {}

    // Writes data into the ring buffer. 
    // 
    // If there is not enough space left in the ring buffer to
    // complete the write, the call will block on the condition
    // variable until there is enough room in the buffer.
    // 
    // Note: If a write blocks because of insufficient space, there is
    // the possibility that a write from a different thread may
    // complete before the blocked write has a chance to finish. In
    // other words: the order of writes is not guaranteed!
    void Write(const char* buffer, size_t length, bool failsafe = true)
    {
      if (m_Initialized) pthread_mutex_lock(&m_Lock); 
      MTRACE_RINGBUFFER_ASSERT(Verify());
      while (true)
      {
        size_t distance = 0;
        if (!m_Active) break;
        if (m_write_count - m_read_count >= BlockSize) 
          goto write_wait; 
        if (m_end < m_begin)
        {
          distance = m_begin - m_end; 
          if (distance < length) 
            goto write_wait; 
          distance = (distance < length) ? distance : length;
          MTRACE_RINGBUFFER_ASSERT(
            (m_end+distance)<=(m_data+BlockSize));
          memcpy(m_end, buffer, distance); 
          m_end += distance; 
          if (m_end == m_data+BlockSize)
            m_end = m_data;
          m_write_count += distance; 
          break;
        }
        else 
        {
          distance = (m_data+BlockSize) - m_end; 
          if (distance < length && 
              (m_begin-m_data) < (length-distance)) 
            goto write_wait; 
          distance = (distance < length) ? distance : length;
          MTRACE_RINGBUFFER_ASSERT(
            (m_end+distance)<=(m_data+BlockSize));
          memcpy(m_end, buffer, distance); 
          m_end  += distance; 
          buffer += distance; 
          length -= distance; 
          m_write_count += distance; 
          if (m_end == m_data+BlockSize)
            m_end = m_data;
          if (length) 
          { 
            MTRACE_RINGBUFFER_ASSERT(
              (m_end+length)<=(m_data+BlockSize));
            memcpy(m_end, buffer, length); 
            m_end += length; 
            m_write_count += length; 
            if (m_end == m_data+BlockSize)
              m_end = m_data;
          } 
          break; 
        }
        if (false) 
        { 
        write_wait:
          if (!failsafe) break; 
          if (!m_Initialized) 
          { 
            fprintf(stderr, "mtrace: output buffer too small, discarding data\n");
            break; 
          }
          int result = pthread_cond_wait(&m_Cond, &m_Lock);
          switch (result)
          {
          case 0: continue;
          case EINVAL:
          case EPERM:
            abort();
          }
        }
      } 
      MTRACE_RINGBUFFER_ASSERT(Verify());
      if (m_Initialized) pthread_mutex_unlock(&m_Lock); 
    }

    // Read data from  the ring buffer. 
    // 
    // If there is not enough space left in the ring buffer to
    // complete the read, the number of consecutive bytes read will be
    // returned. 
    size_t Read(char* buffer, size_t length)
    {
      if (m_Initialized) pthread_mutex_lock(&m_Lock); 
      MTRACE_RINGBUFFER_ASSERT(Verify());
      if (m_write_count - m_read_count == 0) 
      {
        if (m_Initialized) pthread_mutex_unlock(&m_Lock); 
        return 0; 
      }
      size_t read = 0;
      if (m_begin < m_end)
      {
        read = m_end - m_begin; 
        MTRACE_RINGBUFFER_ASSERT((m_begin+read) <= m_end);
      }
      else 
      {
        read = (m_data+BlockSize) - m_begin; 
        MTRACE_RINGBUFFER_ASSERT((m_begin+read) <= m_data+BlockSize);
      }
      read = (read < length) ? read : length;
      if (read != 0) 
      {
        memcpy(buffer, m_begin, read); 
        m_begin += read; 
        m_read_count += read; 
        if (m_begin == m_data+BlockSize)
          m_begin = m_data;
      }
      if (m_Initialized) 
      { 
        pthread_mutex_unlock(&m_Lock); 
        pthread_cond_broadcast(&m_Cond); 
      } 
      MTRACE_RINGBUFFER_ASSERT(Verify());
      return read; 
    }

    // Clears the data within the ringbuffer - resets it to an initial
    // state
    void Clear()
    {
      if (m_Initialized) { pthread_mutex_lock(&m_Lock); }
      m_read_count  = 0;
      m_write_count = 0; 
      m_begin       = m_data;
      m_end         = m_data; 
      if (m_Initialized)
      {
        pthread_mutex_unlock(&m_Lock);
        pthread_cond_broadcast(&m_Cond); 
      }
    }

    // Disable writing to the buffer 
    void Disable() 
    {
      if (m_Initialized) pthread_mutex_lock(&m_Lock);
      m_Active = false;
      if (m_Initialized) 
      {
        pthread_mutex_unlock(&m_Lock);
        pthread_cond_broadcast(&m_Cond); 
      }
    }

    // Enable writing to the buffer 
    void Activate() 
    {
      if (m_Initialized) pthread_mutex_lock(&m_Lock);
      m_Active = true;
      if (m_Initialized) 
      {
        pthread_mutex_unlock(&m_Lock);
        pthread_cond_broadcast(&m_Cond); 
      }
    }

    // Initializes the locks within the buffer 
    bool Initialize()
    {
      if (m_Initialized) 
      {
        fprintf(stderr, "mtrace cannot be initialized twice\n");
        return false; 
      }
      pthread_mutexattr_t attr;
      pthread_mutexattr_init(&attr);
      pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);
      pthread_mutex_init(&m_Lock, &attr);
      pthread_mutexattr_destroy(&attr);
      pthread_cond_init(&m_Cond, NULL); 
      // Lock and unlock the mutex and condition variable because
      // these buggers *may* allocate on usage and create deadlocks!
      pthread_mutex_lock(&m_Lock);
      timespec ts; 
      ts.tv_sec  = 0; 
      ts.tv_nsec = 1;
      pthread_cond_timedwait(&m_Cond, &m_Lock, &ts);
      pthread_mutex_unlock(&m_Lock);

      m_Initialized = true; 
      return true; 
    }; 

    // Release all aquired locks 
    bool Shutdown()
    {
      pthread_cond_destroy(&m_Cond); 
      pthread_mutex_destroy(&m_Lock); 
      m_Initialized = false; 
      return true; 
    } 
  }; 
#if defined(SUPP_MTRACE)
  // Static helper methods for the input and output buffers. 
  typedef IOBuffer<256<<10> OutputBuffer_t; 
  static OutputBuffer_t& OutputBuffer() 
  {
    static OutputBuffer_t _buffer; 
    return _buffer; 
  }
  typedef IOBuffer<8<<10> InputBuffer_t; 
  static InputBuffer_t& InputBuffer()
  {
    static InputBuffer_t _buffer;
    return _buffer; 
  }
#endif 

};
  
// Helper method for address catching 
#define MTRACE_CHECK(ADDR, SIZE) do {                                   \
    if (MTrace::g_Config.CatchAddr != NULL || MTrace::g_Config.CatchSize != 0)          \
    {                                                                   \
      bool _catch = true;                                               \
      if (MTrace::g_Config.CatchAddr != NULL && MTrace::g_Config.CatchAddr != (ADDR))   \
        _catch = false;                                                 \
      else if (MTrace::g_Config.CatchSize != 0 && MTrace::g_Config.CatchSize != (SIZE)) \
        _catch = false;                                                 \
      if (_catch)                                                       \
        Catch(__FUNCTION__);                                            \
    }                                                                   \
  } while (false)

namespace MTrace
{
  // Default mtrace configuration 
  Config g_Config =   
  { 
    NULL,                   // No catch address set 
    0x0,                    // No catch size set 
    true,                    // Recording enabled by default  
    false,                  // Not connected on startup 
    false,                  // Printing to standard error disabled by default
    MTRACE_DEFAULT_ADDRESS, // Default server address 
    MTRACE_DEFAULT_PORT     // Default server port 
  }; 

  // Threaded NetPump that acts as an asynchronous loop forwarding data
  // to the mtrace server and asynchronously recieving results 
  class CNetPump : public CryThread<CNetPump>
  {
    // The state in which the network pump can reside
    enum State
    {
      STATE_IDLE,       // Idle - waits for a connect 
      STATE_CONNECT,    // Connection wants to be established
      STATE_DISCONNECT, // Should disconnect from server
      STATE_IO,         // IO pump loop 
    };

    // The state in which the thread is supposed to be 
    State m_State; 

    // The socket used for the connection with the mtrace server 
    int m_Socket; 

    // Flag depicting if the thread execution should be canceled
    bool m_bCanceled; 

    // Flag describing if currently connected to a mtrace server 
    bool m_bConnected; 

    // The local send buffer 
    char m_SendBuffer[16<<10]; 

    // The local send buffer 
    char m_RecieveBuffer[16<<10]; 

    // The number of bytes still to be sent
    size_t m_ToBeSent; 

    // The local send buffer 
    char* m_SendBufferPtr; 

    // The local send buffer 
    char* m_RecieveBufferPtr; 

    // The actual io pump loop 
    void State_IO(); 

    // State invoked when the thread is informed to connect to a
    // mtrace server
    void State_Connect();

    // State invoked when the thread is informed to disconnect from
    // the mtrace server 
    void State_Disconnect(); 

    // Private constructor - constructed automatically when needed 
    CNetPump();

  public:
    // The run method 
    void Run(); 

    // The cancel method 
    void Cancel(); 

    // Connect to mtrace server  method 
    bool Connect();

    // Disconnect from mtrace server 
    bool Disconnect();

    // Retrieve the static instance 
    static CNetPump& instance();
  }; 

  // Initialize mtrace 
  bool Initialize() 
  {
#if defined(SUPP_MTRACE)
    //if no specific mtrace server address has been set, start remote server
    if(0 == strcmp(MTrace::g_Config.ServerAddress,MTRACE_DEFAULT_ADDRESS))
    {
      int fd, fdc;
      cellFsUnlink(MTRACE_IPCONFIG_PATH);
      int ret = cellFsOpen("/app_home/EXEC:exec_mtrace.bat", 0, &fd, NULL, 0);
      if( ret == EFSSPECIFIC || ret == CELL_FS_SUCCEEDED)
      {
        system_time_t start = sys_time_get_system_time();
        CellFsStat stat; 
        do 
        { 
          if (cellFsStat(MTRACE_IPCONFIG_PATH, &stat) != CELL_FS_ENOENT)
            break; 
        } while (sys_time_get_system_time() - start < 1000 * 1000 * 30); 
        if (cellFsStat(MTRACE_IPCONFIG_PATH, &stat) == CELL_FS_ENOENT) 
        {
          fprintf(stderr, "mtraceip config file not found\n");
          return false; 
        }
        //read ipconfig.txt file
        ret = cellFsOpen(MTRACE_IPCONFIG_PATH, CELL_FS_O_RDONLY, &fdc, NULL, 0);
        if( ret == CELL_FS_SUCCEEDED )
        {
          char buffer[512];
          uint64_t readBytes = 0;
          cellFsRead(fdc, buffer, sizeof(buffer), &readBytes);
          cellFsClose(fdc);
          if(readBytes > 0)
          {
            const char* IPString = "IP Address. . . . . . . . . . . . : ";
            string ipConfig(buffer);
            string::size_type pos = ipConfig.find(IPString);
            if(pos != string::npos)
            {
              char *pDst = MTrace::g_Config.ServerAddress;
              const char *pSrc = &ipConfig[pos+strlen(IPString)];
              while(*pSrc && *pSrc != '\n' && *pSrc != 13) *pDst++ = *pSrc++;
              *pDst = '\0';
            }
          }
          fprintf(stdout, "set mtrace ip to %s\n", MTrace::g_Config.ServerAddress);
        }
        cellFsClose(fd);
      }
    }

    if (!OutputBuffer().Initialize())
      return false; 
    if (!InputBuffer().Initialize())
      return false; 
#endif
    std::atexit(AtExit);
    return true; 
  }

  // Shutdown mtrace
  bool Shutdown()
  {
#if defined(SUPP_MTRACE)
    CNetPump::instance().Disconnect();
    CNetPump::instance().Cancel();
    
    if (!OutputBuffer().Shutdown())
      return false; 
    if (!InputBuffer().Shutdown())
      return false; 
#endif 
    return true; 
  }

  // Connect to the given mtrace server 
  bool Connect()
  {
#if defined(SUPP_MTRACE)
    return CNetPump::instance().Connect();
#endif
  }

  // Disconnects from the mtrace server. Returns true if the
  // connection was closed 
  bool Disconnect()
  {
#if defined(SUPP_MTRACE)
    return CNetPump::instance().Disconnect();
#endif
  }

	// Generate a snapshot request. Into the current stream a snapshot
	// request is inserted into the recording stream so that later
	// analysis of the stream can be partitioned into 'snapshots'. A
	// snapshot is defined as containing all memory operations from the
	// last snapshot till the snapshot tag has been inserted into the
	// recording stream. 
	void SnapShot(const char* name) 
	{ 
#if defined(SUPP_MTRACE)
    if (!g_Config.RecordingEnabled || !name)
      return; 
    CChunk chunk;
    chunk.Write('S'); 
    while(*name != '\0') 
    {
      chunk.Write(*name); 
      ++name; 
    }
    chunk.Write('\0');
#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif
    OutputBuffer().Write(chunk.GetData(), chunk.GetSize());
#endif 
	} 

  // Register a named thread to mtrace 
	void RegisterThreadName(uint32_t threadId, const char* name)
  {
#if defined(SUPP_MTRACE)
    if (!g_Config.RecordingEnabled || !name)
      return; 
    CChunk chunk; 
    chunk.Write('T');
    chunk.Write(threadId); 
    while(*name != '\0') 
    {
      chunk.Write(*name); 
      ++name; 
    }
    chunk.Write('\0');
#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif
    OutputBuffer().Write(chunk.GetData(), chunk.GetSize());    
#endif 
  }

  // Enable thread allocation tracing for a specific thread 
	void EnableThreadTrace(uint32_t threadId)
  {
    AllocationStats* stats = GetAllocationStats(); 
    ++stats->thread_recording;        
  }

  // Disable thread allocation tracing for a specific thread
	void DisableThreadTrace(uint32_t threadId)
  {
    AllocationStats* stats = GetAllocationStats(); 
    if (stats->thread_recording > 0)
      --stats->thread_recording;
  }

  // Accumulates the statistics gathered by all threads 
  void AccumulateThreadStatistics(AllocationStats& alloc_results)
  {
    // Retrieve the overall statistics system wide 
    for (uint16_t i=0; i<next_thread_alloc_index; ++i) 
    {
      const AllocationStats& alloc_stats = thread_alloc_table[i]; 
      alloc_results.allocated_memory += alloc_stats.allocated_memory; 
      alloc_results.freed_memory     += alloc_stats.freed_memory; 
      alloc_results.malloc_calls     += alloc_stats.malloc_calls; 
      alloc_results.calloc_calls     += alloc_stats.calloc_calls; 
      alloc_results.realloc_calls    += alloc_stats.realloc_calls; 
      alloc_results.memalign_calls   += alloc_stats.memalign_calls; 
      alloc_results.free_calls       += alloc_stats.free_calls;
    }
  }

  // Retrieve the moving average of memory operations, and allocated/freed
  // memory per frame 
  void FrameStats(float &avgOps, float &avgAllocs, float &avgFreed)
  {
    GlobalFrameStats& frameStats = g_frame_stats;
    AllocationStats results; 
    AccumulateThreadStatistics(results);

    if (++frameStats.frame_counter > 0)
    {
      const uint64_t accum_operations = 
        (results.malloc_calls +
          results.calloc_calls +
          results.realloc_calls +
          results.memalign_calls +
          results.free_calls) - frameStats.init_ops;
      const uint64_t accum_allocs = 
        results.allocated_memory - frameStats.init_allocs;
      const uint64_t accum_freed = results.freed_memory - frameStats.init_freed;

      const float reci = 1.f / static_cast<float>(frameStats.frame_counter);
      avgOps = static_cast<float>(accum_operations) * reci;
      avgAllocs = static_cast<float>(accum_allocs) * reci;
      avgFreed = static_cast<float>(accum_freed) * reci;
    }
    else
    {
      frameStats.init_ops = 
        results.malloc_calls +
        results.calloc_calls +
        results.realloc_calls +
        results.memalign_calls +
        results.free_calls;
      frameStats.init_allocs = results.allocated_memory;
      frameStats.init_freed = results.freed_memory;
    }
  }


  // Retrieves the accumulated system statistics. System statistics are
  // gathered by thread in a thread local storage area to avoid the
  // necessity to lock on every single allocation and in are
  // accumulated in the function below. 
  bool SystemStatistics(SystemStats& results)
  {
    // Accumulate the statistics gathered by each thread 
    memset(&results, 0, sizeof(results));
    AllocationStats& alloc_results = results.allocations; 
    alloc_results.thread_id = ~0U; 
    AccumulateThreadStatistics(alloc_results);

    // Copy System wide allocation 
    for (uint16_t i=0; i<static_cast<uint16_t>(eCryM_Num); ++i)
      results.modules[i] = g_traced_stats.modules[i];
    results.unknown = g_traced_stats.unknown;

#if defined(SUPP_MTRACE)
    if (g_Config.RecordingEnabled)
    {
      CChunk chunk; 
      chunk.Write('I');
#if defined(ENABLE_STREAM_DEBUGGING)
      chunk.Write(STREAM_DEBUG_TAG);
#endif

      OutputBuffer().Write(chunk.GetData(), chunk.GetSize());
      Process();
    }
#endif

    return true; 
  }

  // Retrieve the memory statistics gathered by a specific thread 
  bool ThreadStatistics(size_t threadId, AllocationStats& results)
  {
    if (!(threadId < next_thread_alloc_index))
        return false;
    const AllocationStats& alloc_stats = thread_alloc_table[threadId]; 
    results.allocated_memory += alloc_stats.allocated_memory; 
    results.freed_memory     += alloc_stats.freed_memory; 
    results.malloc_calls     += alloc_stats.malloc_calls; 
    results.calloc_calls     += alloc_stats.calloc_calls; 
    results.realloc_calls    += alloc_stats.realloc_calls; 
    results.memalign_calls   += alloc_stats.memalign_calls; 
    results.free_calls       += alloc_stats.free_calls;
  }

  // Process recieved messages from mtrace server 
  void Process()
  {
#if defined(SUPP_MTRACE)
    while(true) 
    { 
      char c, buffer[8192];
      uint32_t tag; 
      size_t read = InputBuffer().Read(&c, sizeof(c));
      if (read == 0) break; 
      switch (c) 
      { 
      case 'I':
        read = 0; 
        while (read < sizeof(SystemStats))
        {
          read += InputBuffer().Read(
            &buffer[read], 
            sizeof(SystemStats)-read);
        }
        memcpy(&g_traced_stats, buffer, sizeof(SystemStats));
#if defined(ENABLE_STREAM_DEBUGGING)
        read = 0; 
        while (read < sizeof(tag))
          read += InputBuffer().Read(&buffer[read], sizeof(tag)-read);
        memcpy(&tag, buffer, sizeof(tag));
        if (tag != STREAM_DEBUG_TAG)
        { abort(); }
#endif
        break; 
      default:
        fprintf(stderr, "mtrace process: "
                "recieved unknown command!\n");
        break; 
      } 
    }
#endif 
  }

  // Trace function for recording calloc calls. 
  void trace_calloc(
    void* ptr, 
    size_t size)
  {
    // Retrieve the allocation statisitics instance for the current
    // thread and update the appropiate fields 
    AllocationStats* stats = GetAllocationStats(); 
    stats->allocated_memory += size; 
    ++stats->calloc_calls;

    MTRACE_CHECK_ALLOC;

#if defined(SUPP_MTRACE)

    MTRACE_CHECK(ptr, size);

    // Print the allocation request if printing to standard error is
    // requested
    if (g_Config.PrintToStdout) 
    {
      g_Config.PrintToStdout = false; 
      fprintf(stderr,
              "mtrace: calloc 0x%x (size %d bytes)\n", 
              (uint32_t)ptr, size); 
      g_Config.PrintToStdout = true; 
    }

    // If recording is not enabled globally or for the thread 
    // return here
    if (!g_Config.RecordingEnabled || !stats->thread_recording) 
      return;

    CChunk chunk; 
    chunk.Write('M');

    MemoryOperation mOp; 
    mOp.operation_id = IncrementOpId(); 
    mOp.operation_type = FN_calloc; 
    mOp.thread_id = stats->thread_local_id; 
    mOp.primary = ptr; 
    mOp.secondary = 0;
    mOp.size = (size>0) ? size : 1; 
    chunk.Write(mOp);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    Stacktrace(chunk);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    OutputBuffer().Write(chunk.GetData(), chunk.GetSize());
#endif
  }

  // Trace function for recording malloc calls
  void trace_malloc(
    void *ptr, 
    size_t size)
  {
    // Retrieve the allocation statisitics instance for the current
    // thread and update the appropiate fields 
    AllocationStats* stats = GetAllocationStats(); 
    stats->allocated_memory += size; 
    ++stats->malloc_calls;

    MTRACE_CHECK_ALLOC;

#if defined(SUPP_MTRACE)

    MTRACE_CHECK(ptr, size);

    // Print the allocation request if printing to standard error is
    // requested
    if (g_Config.PrintToStdout) 
    {
      g_Config.PrintToStdout = false; 
      fprintf(stderr,"mtrace: malloc 0x%x (size %d bytes)\n",
              (uint32_t)ptr, size); 
      g_Config.PrintToStdout = true; 
    }

    // If recording is not enabled globally or for the thread 
    // return here
    if (!g_Config.RecordingEnabled || !stats->thread_recording) 
      return;

    CChunk chunk; 
    chunk.Write('M');

    MemoryOperation mOp; 
    mOp.operation_id = IncrementOpId(); 
    mOp.operation_type = FN_malloc; 
    mOp.thread_id = stats->thread_local_id; 
    mOp.primary = ptr; 
    mOp.secondary = 0; 
    mOp.size = (size>0) ? size : 1; 
    chunk.Write(mOp);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    Stacktrace(chunk);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    OutputBuffer().Write(chunk.GetData(), chunk.GetSize());
#endif
  }

  // Trace function for recording calls to free()
  void trace_free(
    void* ptr, 
    size_t size) 
  { 
    // Retrieve the allocation statisitics instance for the current
    // thread and update the appropiate fields 
    AllocationStats* stats = GetAllocationStats(); 
    stats->freed_memory += size; 
    ++stats->free_calls;

    MTRACE_CHECK_ALLOC;

#if defined(SUPP_MTRACE)

    MTRACE_CHECK(ptr, size);

    // Return the allocation request if printing to standard error is
    // requested
    if (g_Config.PrintToStdout) 
    {
      g_Config.PrintToStdout = false; 
      fprintf(stderr,"mtrace: free 0x%x (%d bytes)\n",
              (uint32_t)ptr, size); 
      g_Config.PrintToStdout = true; 
    }

    // If recording is not enabled globally or for the thread 
    // return here
    if (!g_Config.RecordingEnabled || !stats->thread_recording) 
      return;

    CChunk chunk; 
    chunk.Write('M');

    MemoryOperation mOp; 
    mOp.operation_id = IncrementOpId(); 
    mOp.operation_type = FN_free; 
    mOp.thread_id = stats->thread_local_id; 
    mOp.primary = ptr; 
    mOp.secondary = 0; 
    mOp.size = 0; 
    chunk.Write(mOp);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    Stacktrace(chunk);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    OutputBuffer().Write(chunk.GetData(), chunk.GetSize());
#endif
  } 

  // Trace function for recording calls to realloc()
  void trace_realloc(
    void *old_ptr, 
    void* new_ptr, 
    size_t old_size, 
    size_t new_size)
  {

    // Retrieve the allocation statisitics instance for the current
    // thread and update the appropiate fields 
    AllocationStats* stats = GetAllocationStats(); 
    stats->allocated_memory += new_size; 
    stats->freed_memory += old_size; 
    ++stats->realloc_calls;

    MTRACE_CHECK_ALLOC;

#if defined (SUPP_MTRACE)

    MTRACE_CHECK(new_ptr, new_size);

    // Return the allocation request if printing to standard error is
    // requested 
    if (g_Config.PrintToStdout) 
    {
      g_Config.PrintToStdout = false; 
      fprintf(
        stderr, "mtrace: "
        "realloc 0x%x 0x%x (new %d bytes, old %d bytes)\n", 
        (uint32_t)new_ptr, (uint32_t)old_ptr, new_size, old_size); 
      g_Config.PrintToStdout = true; 
    }

    // If recording is not enabled globally or for the thread 
    // return here
    if (!g_Config.RecordingEnabled || !stats->thread_recording) 
      return;

    CChunk chunk; 
    chunk.Write('M');

    MemoryOperation mOp; 
    mOp.operation_id = IncrementOpId(); 
    mOp.operation_type = FN_realloc;
    mOp.thread_id = stats->thread_local_id; 
    mOp.primary = new_ptr; 
    mOp.secondary = old_ptr; 
    mOp.size = new_size; 
    chunk.Write(mOp);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    Stacktrace(chunk);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    OutputBuffer().Write(chunk.GetData(), chunk.GetSize());
#endif
  }

  // Trace function for recording calls to memalign 
  void trace_memalign(
    void* ptr, 
    size_t boundary, 
    size_t size)
  {
    // Retrieve the allocation statisitics instance for the current
    // thread and update the appropiate fields 
    AllocationStats* stats = GetAllocationStats(); 
    stats->allocated_memory += size; 
    ++stats->memalign_calls;

    MTRACE_CHECK_ALLOC;

#if defined(SUPP_MTRACE)

    MTRACE_CHECK(ptr, size);

    // Print the allocation request if printing to standard error is
    // requested 
    if (g_Config.PrintToStdout) 
    {
      g_Config.PrintToStdout = false; 
      fprintf(stderr,"mtrace: memalign 0x%x (align: %d) (%d bytes)\n", 
              (uint32_t)ptr, boundary, size); 
      g_Config.PrintToStdout = true; 
    }

    // If recording is not enabled globally or for the thread 
    // return here
    if (!g_Config.RecordingEnabled || !stats->thread_recording) 
      return;

    CChunk chunk; 
    chunk.Write('M');

    MemoryOperation mOp; 
    mOp.operation_id = IncrementOpId(); 
    mOp.operation_type = FN_memalign;
    mOp.thread_id = stats->thread_local_id; 
    mOp.primary = ptr; 
    mOp.secondary = 0; 
    mOp.size = size;
    chunk.Write(mOp);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    Stacktrace(chunk);

#if defined(ENABLE_STREAM_DEBUGGING)
    chunk.Write(STREAM_DEBUG_TAG);
#endif

    OutputBuffer().Write(chunk.GetData(), chunk.GetSize());
#endif
  }
}


// The below code is only enabled when supp-mtrace is enabled 
#if defined(SUPP_MTRACE)

using namespace MTrace;

CNetPump::CNetPump()
  : CryThread<CNetPump>(), 
    m_State(STATE_IDLE), 
    m_Socket(-1), 
    m_bCanceled(false),
    m_bConnected(false),
    m_ToBeSent(0), 
    m_SendBufferPtr(m_SendBuffer)
{ 
  Start(0, "MTrace NetPump", THREAD_PRIORITY_NORMAL+25, 
        2*SIMPLE_THREAD_STACK_SIZE_KB*1024); 
}

void CNetPump::Run()
{
  while(!m_bCanceled)
  {
    // Sleep on a condition if we are not connected 
    switch (m_State) 
    {
    case STATE_IDLE:
      Lock(); 
      Wait();
      Unlock(); 
      break; 
    case STATE_CONNECT:
      State_Connect();
      break; 
    case STATE_DISCONNECT:
      State_Disconnect();
      break; 
    case STATE_IO:
      State_IO(); 
      break; 
    }; 
  }
}

void CNetPump::State_IO()
{
  while(m_bConnected)
  {
    // Pump IO Loop 
    timeval timeout; 
    timeout.tv_sec = 1;
    timeout.tv_usec = 0; 

    fd_set socket_read, socket_write, socket_except; 
    FD_ZERO(&socket_read); 
    FD_ZERO(&socket_write);
    FD_ZERO(&socket_except);
    FD_SET(m_Socket, &socket_read); 
    FD_SET(m_Socket, &socket_write); 
    FD_SET(m_Socket, &socket_except); 
       
    // Determine if the socket has data that can be read,
    // is ready to be written to or has recieved out-ouf-band
    // messages
    signed ready = socketselect(
      m_Socket+1, &socket_read, &socket_write, &socket_except, &timeout);
    if (ready < 0)
    {
      // Blocking call was interrupted - simply try again 
      if (sys_net_errno == SYS_NET_EINTR)
      {
        Lock();
        if (m_State != STATE_IO)
          break; 
        Unlock();
        continue; 
      }
      // Invalid file descriptor passed to select 
      if (sys_net_errno == SYS_NET_EBADF)
      {
        fprintf(
          stderr, "mtrace NetPump: bad socket descriptor passed\n");
        goto NetPump_exit; 
      }
      // Invalid timeout time (should never happen!) 
      if (sys_net_errno == SYS_NET_EINVAL) 
      {
        fprintf(
          stderr, "mtrace NetPump: bad socket descriptor passed\n");
        goto NetPump_exit; 
      }
      fprintf(
        stderr, "mtrace NetPump: unknown socket error\n");
      goto NetPump_exit; 
    }
    if (ready > 0) 
    {
      // Check if there is data to be read from the socket 
      if (FD_ISSET(m_Socket, &socket_read))
      {
        char buffer[1024];
        signed read = 0;
        while (true) 
        { 
          read = recv( m_Socket, buffer, sizeof(buffer), 0);
          // Error handling 
          if (read < 0) 
          { 
            switch (sys_net_errno)
            {
            case SYS_NET_EINTR:
              continue; 
            case SYS_NET_EAGAIN:
              // case SYS_NET_EWOULDBLOCK:
              fprintf(
                stderr, "mtrace NetPump: "
                "read would block - exiting\n");
              goto NetPump_exit; 
            case SYS_NET_EBADF:
              fprintf(
                stderr, "mtrace NetPump: "
                "Invalid socket number specified\n");
              goto NetPump_exit; 
            case SYS_NET_EINVAL:
              fprintf(
                stderr, "mtrace NetPump: "
                "Invalid argument or function call\n");
              goto NetPump_exit; 
            case SYS_NET_ECONNABORTED:
              fprintf(
                stderr, "mtrace NetPump: "
                "Connection was closed\n");
              goto NetPump_exit; 
            };
          }
          // If read returns 0 then the other side of the conncection has
          // most probably disconnected.
          if (read == 0) 
          { 
            fprintf(stderr, "mtrace NetPump: server has disconnected!\n"); 
            goto NetPump_exit; 
          } 
          break; 
        }
        InputBuffer().Write(buffer, read, false);
      }
      // Check if the socket is ready to send data. Retrieves the next
      // chunk of data 
      if (FD_ISSET(m_Socket, &socket_write))
      { 
        signed  wrote = 0; 
        if (m_ToBeSent == 0) 
        { 
          size_t length = OutputBuffer().Read(m_SendBuffer, sizeof(m_SendBuffer)); 
          if (length == 0) 
          {
            // Prevent the thread from burning too much cpu cycles when
            // sending data to the mtrace server 
            sys_ppu_thread_yield(); 
            continue;
          }
          m_ToBeSent = length; 
        }
        wrote = send( m_Socket, m_SendBufferPtr, m_ToBeSent, MSG_DONTWAIT);
        if (wrote < 0) 
        { 
          switch(sys_net_errno)
          {
          case SYS_NET_EINTR:
          case SYS_NET_EAGAIN:
            /*case SYS_NET_EWOULDBLOCK: //duplicate! */
            fprintf(
              stderr, "mtrace NetPump: "
              "send would block (should never happen)\n");
            continue;
          case SYS_NET_EBADF:
            fprintf(
              stderr, "mtrace NetPump: "
              "Invalid socket number specified\n");
            goto NetPump_exit;
          case SYS_NET_EINVAL:
            fprintf(
              stderr, "mtrace NetPump: "
              "Invalid argument or function call\n");
            goto NetPump_exit;
          case SYS_NET_EPIPE:
            fprintf(
              stderr, "mtrace NetPump: "
              "The writing side of the socket has "
              "already been closed\n");
            goto NetPump_exit;
          case SYS_NET_EMSGSIZE:
            fprintf(
              stderr, "mtrace NetPump: "
              "Message size is too large\n");
            goto NetPump_exit;
          case SYS_NET_EHOSTDOWN:
            fprintf(
              stderr, "mtrace NetPump: "
              "Other end is down and unreachable\n");
            goto NetPump_exit;
          case SYS_NET_ENETUNREACH:
            fprintf(
              stderr, "mtrace NetPump: "
              "Destination is unreachable\n");
            goto NetPump_exit;
          case SYS_NET_ECONNRESET:
            fprintf(
              stderr, "mtrace NetPump: "
              "Connection was reset (TCP only)\n");
            goto NetPump_exit;
          case SYS_NET_ENOTCONN:
            fprintf(
              stderr, "mtrace NetPump: "
              "Specified connection is not established\n");
            goto NetPump_exit;
          };
        } 
        // If send returns 0 written bytes this usually means that the
        // connection has been lost. 
        if (wrote == 0) 
        { 
          fprintf(stderr,"mtrace NetPump: server has disconnected?\n");
          continue; 
        } 
        m_ToBeSent -= wrote; 
        if (m_ToBeSent == 0) 
          m_SendBufferPtr = m_SendBuffer; 
        else
          m_SendBufferPtr += wrote; 
        // Prevent the thread from burning too much cpu cycles when
        // sending data to the mtrace server 
        sys_ppu_thread_yield(); 
      }
    }
    // Error Condition 
    if (false) 
    { 
    NetPump_exit: 
      Disconnect(); 
      break; 
    }
  }
}

void CNetPump::State_Connect()
{
  int nonBlock, noDelay, s; 
  Lock(); 
  // Don't do anything if the socket is already connected to a host
  if (m_Socket != -1)
  {
    fprintf(stderr, "mtrace NetPump: already connected to mtrace server\n");
    goto connect_error;
  }

  // Create the socket 
  s = socket(AF_INET, SOCK_STREAM, 0); 
  if (s < 0)
  {
    fprintf(stderr,"mtrace NetPump: could not create socket!");
    goto connect_error;
  }

  // Resolve the address of the mtrace server 
  sockaddr_in serv_addr;
  std::memset(&serv_addr, 0, sizeof serv_addr);
  serv_addr.sin_family = AF_INET;
  serv_addr.sin_port = htons(g_Config.ServerPort);
  inet_pton(AF_INET, g_Config.ServerAddress, &serv_addr.sin_addr);

  // Connect to the mtrace server 
  if(connect(s, (sockaddr*)&serv_addr, sizeof(serv_addr)) != 0) 
  { 
    fprintf(stderr,"mtrace NetPump: could not connect to client %s:%d\n", 
      g_Config.ServerAddress, g_Config.ServerPort);
    goto connect_error;
  } 

  // Set the socket to non blocking mode after the connection has been
  // established 
  nonBlock = 1; 
  if (setsockopt(s, SOL_SOCKET, SO_NBIO, &nonBlock, sizeof(nonBlock))
      != 0) 
  { 
    switch(sys_net_errno)
    {
    case SYS_NET_EBADF:
      fprintf(stderr,"mtrace NetPump: "
              "Invalid socket number specified\n");
      break;
    case SYS_NET_EINVAL:
      fprintf(stderr,"mtrace NetPump: "
              "Invalid value specified\n");
      break;
    case SYS_NET_ENOPROTOOPT:
      fprintf(stderr,"mtrace NetPump: "
              "Invalid combination of level (level) "
              "and option (optname)\n");
      break;
    case SYS_NET_EADDRNOTAVAIL:
      fprintf(stderr,"mtrace NetPump: "
              "Invalid address was specified\n");
      break;
    case SYS_NET_ETOOMANYREFS:
      fprintf(stderr,"mtrace NetPump: "
              "Too many multicast addresses specified\n");
      break;
    }
    goto connect_error;
  } 

  // Set the socket the socket have nagling disabled so that no high latency
  // connection is being used
  noDelay = 1;
	if (setsockopt(s, IPPROTO_TCP, TCP_NODELAY, &noDelay, sizeof(noDelay)) != 0)
	{
    switch(sys_net_errno)
    {
    case SYS_NET_EBADF:
      fprintf(stderr,"mtrace NetPump: "
              "Invalid socket number specified\n");
      break;
    case SYS_NET_EINVAL:
      fprintf(stderr,"mtrace NetPump: "
              "Invalid value specified\n");
      break;
    case SYS_NET_ENOPROTOOPT:
      fprintf(stderr,"mtrace NetPump: "
              "Invalid combination of level (level) "
              "and option (optname)\n");
      break;
    case SYS_NET_EADDRNOTAVAIL:
      fprintf(stderr,"mtrace NetPump: "
              "Invalid address was specified\n");
      break;
    case SYS_NET_ETOOMANYREFS:
      fprintf(stderr,"mtrace NetPump: "
              "Too many multicast addresses specified\n");
      break;
    }
    goto connect_error;
	}

  m_Socket = s; 
  m_State = STATE_IO;
  m_bConnected = true;
  g_Config.RecordingEnabled = true; 
  g_Config.Connected = true; 
  Unlock();
  return; 

connect_error:
  m_State = STATE_IDLE;
  m_bConnected = false;
  OutputBuffer().Disable();
  InputBuffer().Disable();
  g_Config.RecordingEnabled = false; 
  g_Config.Connected = false; 
  Unlock(); 
  return; 
}

void CNetPump::State_Disconnect()
{
  Lock(); 
  
  // Close the socket so we can possibly resuse it later 
  if (m_Socket != -1 && socketclose(m_Socket) == -1)
  {
    fprintf(stderr, "mtrace NetPump: could not shutdown connection\n");
  }

  m_Socket = -1; 
  m_State = STATE_IDLE;
  OutputBuffer().Clear();
  InputBuffer().Clear();
  OutputBuffer().Disable();
  InputBuffer().Disable();
  g_Config.RecordingEnabled = false; 
  g_Config.Connected = false; 
  Unlock();
}

void CNetPump::Cancel()
{
  Lock();
  m_bCanceled = true;
  Unlock();
  Notify();
}

bool CNetPump::Connect()
{
  Lock();
  m_State = STATE_CONNECT;
  Unlock(); 
  Notify(); 
  return true; 
}

bool CNetPump::Disconnect()
{
  Lock();
  m_State = STATE_DISCONNECT;
  Unlock(); 
  sys_net_abort_socket(m_Socket,SYS_NET_ABORT_STRICT_CHECK); 
  return true; 
}

CNetPump& CNetPump::instance()
{
  static CNetPump _instance; 
  return _instance;
}
#endif // SUPP_MTRACE 

#endif

