#include "StdAfx.h"
#include <ITimer.h>
#include <CrySizer.h>
#include "CrySizerImpl.h"

CrySizerImpl::CrySizerImpl() :m_pResourceCollector(0)
{
	m_nFlags = 0;
	m_nTotalSize = 0;
	// to avoid reallocations during walk through the memory tree, reserve the space for the names
	clear();
}

CrySizerImpl::~CrySizerImpl()
{
}

void CrySizerImpl::Push (const char* szComponentName)
{
	m_stackNames.push_back (getNameIndex(getCurrentName(), szComponentName));
	// if the depth is too deep, something is wrong, perhaps an infinite loop
	assert (m_stackNames.size() < 128);
}

void CrySizerImpl::PushSubcomponent (const char* szSubcomponentName)
{
	Push (szSubcomponentName);
}


void CrySizerImpl::Pop ()
{
	if (!m_stackNames.empty())
		m_stackNames.pop_back();
	else
		assert (0);
}

// returns the index of the current name on the top of the name stack
size_t CrySizerImpl::getCurrentName()const
{
	assert(!m_stackNames.empty());
	return m_stackNames.empty()?0:m_stackNames.back();
}



// searches for the name in the name array; adds the name if it's not there and returns the index
size_t CrySizerImpl::getNameIndex (size_t nParent, const char* szComponentName)
{
	NameArray::const_iterator it = m_arrNames.begin(), itEnd = it + m_arrNames.size();
	for (; it != itEnd; ++it)
#if defined(LINUX)
		if (!strcasecmp(it->strName.c_str(), szComponentName) && it->nParent == nParent)
#else
		if (!strcmp(it->strName.c_str(), szComponentName) && it->nParent == nParent)
#endif
			return (size_t)(it - m_arrNames.begin());//it-m_arrNames.begin();

	size_t nNewName = m_arrNames.size();
	m_arrNames.resize(nNewName+1);

	m_arrNames[nNewName].assign(szComponentName, nParent);

	m_arrNames[nParent].arrChildren.push_back(nParent);

	return nNewName;
}





IResourceCollector &CrySizerImpl::GetResourceCollector()
{ 
	if(m_pResourceCollector)
		return *m_pResourceCollector; 
	else
	{
		static IResourceCollector NullCollector; 

		return NullCollector;
	}
}

void CrySizerImpl::Reset()
{
	clear();

	m_nTotalSize=0;

	//m_arrNames.resize(0);
	//m_arrNames.push_back("TOTAL"); // the default name, with index 0
	//m_LastObject.clear();
	////m_nFlags;
	//m_nTotalSize=0;
	//if (m_pResourceCollector)
	//{
	//	m_pResourceCollector->Reset();
	//}
	//m_setObjects->clear();
	//m_stackNames.resize(0);
	//m_stackNames.push_back(0);
}

#if defined(PS3_CRYSIZER_HEAP_TRAVERSAL)
namespace PS3CrySizerHeapChecker
{
	// these function do an exact match with start of an allocated block
	bool HasPtrExact( void *ptr );	
	uint32 GetPtrSizeExact( void *ptr );
	void RemovePtrExact( void * ptr );
	void *GetRangeStartPtrExact(void*);
	uint32 GetAllocationIdReference( void *ptr );
	uint32 GetAllocationTimeDiffExact(void*ptr);
	void SetAllocationUserDataExact(void *, void*);
	void SetAllocationTimeDiffExact( void*, int );
	void* GetAllocationUserDataExact( void*);

	// these function perform a range check with each allocated range
	bool HasPtrRange( void *ptr );			
	uint32 GetPtrSizeRange( void *ptr );
	void RemovePtrRange( void * ptr );
	void *GetRangeStartPtrRange(void*);

	// these functions work on the original unaltered data
	uint32 GetPtrSizeReference( void *ptr );	
	void* GetRangeStartPtrReference( void *ptr );	
	uint32 GetAllocationIdReference( void *ptr );

	// util functions
	void DumpCallStack( void *ptr );
	int GetNumObjects();
	void* GetNextParentAllocation( void *&);
	void StartMemIteration();
}
#endif

// adds an object identified by the unique pointer (it needs not be
// the actual object position in the memory, though it would be nice,
// but it must be unique throughout the system and unchanging for this object)
// RETURNS: true if the object has actually been added (for the first time)
//          and calculated
bool CrySizerImpl::AddObject (const void* pIdentifier, size_t sizeBytes,int nCount)
{
	if (!pIdentifier || !sizeBytes)
		return false; // we don't add the NULL objects

#if defined(PS3_CRYSIZER_HEAP_TRAVERSAL) // ask the allocated memory for the correct ptr and its size
	const void* pIdentifierNew = PS3CrySizerHeapChecker::GetRangeStartPtrRange((void*)pIdentifier);
	size_t sizeBytesNew = PS3CrySizerHeapChecker::GetPtrSizeRange((void*)pIdentifier);

	// only used the correct ones if we passed a valid ptr
	// note: an invalid ptr can also ne representive for allocated memory like a global variable
	if ( pIdentifierNew != NULL && sizeBytesNew != 0)
	{
		pIdentifier = pIdentifierNew;
		sizeBytes = sizeBytesNew;
	}
	else
	{
		return;
	}
#endif

	Object NewObject(pIdentifier, sizeBytes, getCurrentName());

	// check if the last object was the same
	if (NewObject == m_LastObject)
	{
		assert (m_LastObject.nSize == sizeBytes);
		return false;
	}

	ObjectSet& rSet = m_setObjects[getHash(pIdentifier)];	
	ObjectSet::iterator it = rSet.find (NewObject);
	if (it == rSet.end())
	{
		// there's no such object in the map, add it
		rSet.insert (NewObject);
		ComponentName& CompName = m_arrNames[getCurrentName()];
		CompName.numObjects += nCount;
		CompName.sizeObjects += sizeBytes;

		m_nTotalSize += sizeBytes;
		return true;
	}
	else
	{
		Object *pObj = const_cast<Object*>(&(*it));	

		// if we do an heap check, don't accept the same object twice
#if defined(PS3_CRYSIZER_HEAP_TRAVERSAL)
		if (sizeBytes != pObj->nSize)
		{
			// if the following assert fails:
//			assert (0);
			// .. it means we have one object that's added two times with different sizes; that's screws up the whole idea
			// we assume there are two different objects that are for some reason assigned the same id
			pObj->nSize += sizeBytes; // anyway it's an invalid situation
			ComponentName& CompName = m_arrNames[getCurrentName()];
			CompName.sizeObjects += sizeBytes;
			return true; // yes we added the object, though there were an error condition
		}
#endif
		return false;
	}
}

size_t CrySizerImpl::GetObjectCount() 
{
	size_t count = m_stackNames.size();
	for (int i = 0; i < g_nHashSize; i++)
		count += m_setObjects[i].size();
	return count;
}


// finalizes data collection, should be called after all objects have been added
void CrySizerImpl::end()
{
	// find all not found references
	this->CompareWithHeap();

	// clean up the totals of each name
	int i;
	for (i = 0; i < m_arrNames.size(); ++i)
	{
		assert (i == 0 || ((int)m_arrNames[i].nParent < i && m_arrNames[i].nParent >=0));
		m_arrNames[i].sizeObjectsTotal = m_arrNames[i].sizeObjects;
	}

	// add the component's size to the total size of the parent.
	// for every component, all their children are put after them in the name array
	// we don't include the root because it doesn't belong to any other parent (nowhere further to add)
	for (i = m_arrNames.size() - 1; i > 0; --i)
	{
		// the parent's total size is increased by the _total_ size (already calculated) of this object
		m_arrNames[m_arrNames[i].nParent].sizeObjectsTotal += m_arrNames[i].sizeObjectsTotal;
	}
}


void CrySizerImpl::clear()
{
	for (unsigned i = 0 ; i < g_nHashSize; ++i)
		m_setObjects[i].clear();

	m_arrNames.clear();
	m_arrNames.push_back("TOTAL"); // the default name, with index 0
	m_stackNames.clear();
	m_stackNames.push_back(0);
	m_LastObject.pId = NULL;

	if (m_pResourceCollector)
	{
		m_pResourceCollector->Reset();
	}
}

// hash function for an address; returns value 0..1<<g_nHashSize
unsigned CrySizerImpl::getHash (const void* pId)
{
	//return (((unsigned)pId) >> 4) & (g_nHashSize-1);
	
	// pseudorandomizing transform
	ldiv_t _Qrem = (ldiv_t)ldiv(((uint32)(UINT_PTR)pId >> 2), 127773);
	_Qrem.rem = 16807 * _Qrem.rem - 2836 * _Qrem.quot;
	if (_Qrem.rem < 0)
		_Qrem.rem += 2147483647; // 0x7FFFFFFF
	return ((unsigned)_Qrem.rem) & (g_nHashSize-1);
	
}

void CrySizerImpl::FindSubAllocations( const ObjectSet &rInPutObjectSet, ObjectSet &rOutputObjectSet )
{
#if defined(PS3_CRYSIZER_HEAP_TRAVERSAL)
	// first pass: Assign each found allocation the parent allocation with the nearest ID
	for( ObjectSet::const_iterator it = rInPutObjectSet.begin() ; it != rInPutObjectSet.end() ; ++it )
	{				
		// make sure we only operate on valid allocated memory ranges for the heap traversel
		uint32 nPtr =  (uint32)PS3CrySizerHeapChecker::GetRangeStartPtrReference( (void*)it->pId );
		uint32 nSize = PS3CrySizerHeapChecker::GetPtrSizeReference( (void*)it->pId ); 

		// traverse the object by assuming each member is a ptr to something and checking if this pointer was really allocated
		// could generate false positives
		uint32 nParentId = PS3CrySizerHeapChecker::GetAllocationIdReference((void*)nPtr);
		void **pPtr = (void**)nPtr;
		for( uint32 j = 0 ; j < nSize/4  ; ++ j)
		{				
			if( PS3CrySizerHeapChecker::HasPtrExact( pPtr[j] ) )
			{
				//printf("Parent Id %d, allocationId %d, old diff %d, new diff %d\n",(int)PS3CrySizerHeapChecker::GetAllocationIdReference(pPtr[j]),(int)nParentId,(uint32)PS3CrySizerHeapChecker::GetAllocationTimeDiffExact(pPtr[j]),(uint32)std::abs((int)PS3CrySizerHeapChecker::GetAllocationIdReference(pPtr[j]) - (int)nParentId ));
				if( (uint32)std::abs((int)PS3CrySizerHeapChecker::GetAllocationIdReference(pPtr[j]) - (int)nParentId ) < (uint32)PS3CrySizerHeapChecker::GetAllocationTimeDiffExact(pPtr[j])) // this allocation is *nearer*
				{					
					PS3CrySizerHeapChecker::SetAllocationUserDataExact(pPtr[j],(void*)it->pId );
					PS3CrySizerHeapChecker::SetAllocationTimeDiffExact(pPtr[j], std::abs((int)PS3CrySizerHeapChecker::GetAllocationIdReference(pPtr[j]) - (int)nParentId ));
				}
			}
		}			
	}

	// second pass: collect all found allocation and assign these to the CrySizer
	PS3CrySizerHeapChecker::StartMemIteration();
	void *pCurrent = NULL;
	void *pParent = PS3CrySizerHeapChecker::GetNextParentAllocation( pCurrent );
	while( pParent != NULL )
	{
		// since we have to do a range check, get the allocated size and ptr
		uint32 nFoundSize = PS3CrySizerHeapChecker::GetPtrSizeExact( pCurrent );
		void *pFoundPtr = PS3CrySizerHeapChecker::GetRangeStartPtrExact( pCurrent );

		ObjectSet &rParentSet = m_setObjects[getHash(pParent)];	
		ObjectSet::iterator it = rParentSet.find(pParent);
		char buffer[512] = {0};
		size_t nParent = it->nName;

		size_t nNameIndex = 0;

		if( strstr(m_arrNames[nParent].strName.c_str(), "_Unknown") != NULL)
		{
			nNameIndex = nParent;
		} else {
			// build and add name for unknown sub component
			strcpy(buffer,  m_arrNames[nParent].strName.c_str() );
			strcpy( &buffer[strlen(m_arrNames[nParent].strName.c_str())], "_Unknown");
			nNameIndex = getNameIndex( nParent, buffer);
		}
		Object NewObject(pFoundPtr, nFoundSize, nNameIndex );						
		ObjectSet& rSet = m_setObjects[getHash(pFoundPtr)];	
		rSet.insert (NewObject);
		ComponentName& CompName = m_arrNames[nNameIndex];
		CompName.numObjects += 1;
		CompName.sizeObjects += nFoundSize;
		m_nTotalSize += nFoundSize;

		// store newly found subobject to speed up later iterations
		rOutputObjectSet.insert(NewObject);

		PS3CrySizerHeapChecker::RemovePtrExact( pCurrent );

		pParent = PS3CrySizerHeapChecker::GetNextParentAllocation( pCurrent );
	}
#endif
}

void CrySizerImpl::CompareWithHeap()
{
#if defined(PS3_CRYSIZER_HEAP_TRAVERSAL)
	// to find untracked references the following algorihtm is used:
	// 1. Remove all objects which are tracked from a set of all allocated memory ranges
	// 2. thread each correctly tracked object as an array of void* and then check if this is an valid allocation
	//		if yes, add as subobject
	// repeat still no new matches are found
	// Optimizations: Instead of checking all tracked memory each iteration, remeber the matches and just compare these
	
	typedef typename ObjectSet::const_iterator SetIterT;
	
	// remove all correct tracked ptr
	for( int i = 0 ; i < g_nHashSize ; ++i )
	{
		const ObjectSet &rObjectSet = m_setObjects[i];
		for( SetIterT it = rObjectSet.begin() ; it != rObjectSet.end() ; ++it )
		{
			uint32 nPtr = (uint32)it->pId;
			if( PS3CrySizerHeapChecker::HasPtrRange( (void*)it->pId ) )
			{				
				PS3CrySizerHeapChecker::RemovePtrRange( (void*)it->pId );
			}			
		}
	}

	ObjectSet tempObjectSets[2];
	uint32 nCurTempObjectSet = 0;

	// first pass, use all CrySizer found objects as a base		
	gEnv->pLog->Log("Initial Iteration, %d not assigned objects\n", PS3CrySizerHeapChecker::GetNumObjects());		
	for( int i = 0 ; i < g_nHashSize ; ++i )
	{		
		FindSubAllocations(m_setObjects[i],tempObjectSets[0]);
	}		
	
	// since we can assume that after one full traversal, no new matches can be found in the old data, we only traverse over the new data
	uint32 nIteration = 0;
	uint32 nOldObjectCount = 0;
	do 
	{
		gEnv->pLog->Log("Iteration %d, %d not assigned objects\n", nIteration, PS3CrySizerHeapChecker::GetNumObjects());		
		// store number of objects before iteration to compare how many we removed
		nOldObjectCount = PS3CrySizerHeapChecker::GetNumObjects();
		nIteration += 1;

		FindSubAllocations(tempObjectSets[(nCurTempObjectSet+0)%2],tempObjectSets[(nCurTempObjectSet+1)%2]);

		tempObjectSets[(nCurTempObjectSet+0)%2].clear();
		// switch temp object sets
		nCurTempObjectSet += 1;

		// stop if we have more than 20 iterations or didn't remove more than 50 object in one iteration
	} while (nIteration < 20 && (nOldObjectCount - PS3CrySizerHeapChecker::GetNumObjects()) > 50);

	
#if 0 // the following part is debug code to check false positives
	printf("Char Instances\n");		
	for( int i = 0 ; i < g_nHashSize ; ++i )
	{
		const ObjectSet &rObjectSet = m_setObjects[i];
		for( SetIterT it = rObjectSet.begin() ; it != rObjectSet.end() ; ++it )
		{
			if(	strstr(m_arrNames[it->nName].strName.c_str(), "_Unknown") != NULL &&
				strstr(m_arrNames[it->nName].strName.c_str(), "Character instance") != NULL )
			{				
				PS3CrySizerHeapChecker::DumpCallStack(it->pId);
			}			
		}
	}
#endif

#endif
}