//////////////////////////////////////////////////////////////////////
//
//	Crytek CryENGINE Source code
//
//	File:FileReadSequencer.cpp
//  Description: Loading file data ahead for fast load from DVD
//    Usually all CVars has to be set into release state for proper work of file sequencer:
//
//	History:
//	  -May 24,2010: Created by Vladimir Kajalin
//
//////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "CryPak.h"
#include <ILog.h>
#include <ISystem.h>
#include <IPlatformOS.h>
#include <IConsole.h>
#include <ITimer.h>
#include <IPerfHud.h>
#include "zlib/zlib.h"				// crc32()
#include "md5.h"

#if defined(PS3) || defined(LINUX)
#include "System.h"
#include <unistd.h>
#include <sys/stat.h>					// fstat, fileno
#endif
#ifdef XENON
#include "System.h"
#include "xfilecache.h"
#endif

#if defined(PS3) || defined(XENON)
const char * pSeqFileNameWrite = "e:\\TestResults\\%s.seq";
const char * pSeqFileNameRead = "d:\\user\\%s.seq";
#else
const char * pSeqFileNameWrite = "user\\%s.seq";
const char * pSeqFileNameRead = "user\\%s.seq";
#endif

float CFileReadSequencer::fTimeInMainThreadRead=0;
int CFileReadSequencer::nBytesSequnced=0;
int CFileReadSequencer::nReadsNumSequnced=0;
int CFileReadSequencer::nBytesSequncedSkip=0;
int CFileReadSequencer::nReadsNumSequncedSkip=0;
int CFileReadSequencer::nBytesSequncedNone=0;
int CFileReadSequencer::nReadsNumSequncedNone=0;
int CFileReadSequencer::nSequencerMode=0;
string CFileReadSequencer::sSectionName;
DWORD CFileReadSequencer::dwTargetThread=0;
CryCriticalSection CFileReadSequencer::csFread;
FILE * CFileReadSequencer::hSeqFile=0;
class CReadSequencerThread * CFileReadSequencer::pReadThread = 0;
bool CFileReadSequencer::m_bRestart = true;
std::map<FILE*,string> CFileReadSequencer::mapOpenFiles;
CryCriticalSection CFileReadSequencer::csMapOpenFiles;
CTimeValue CFileReadSequencer::sectionStartTime;
int CFileReadSequencer::nMessageOrder = 0; 

//#define FS_DEBUG

struct SSeqHeader
{
  SSeqHeader() { Reset(); } 
  void Reset() 
  { 
#ifdef FS_DEBUG
    szFileName[0]=szCallStack[0]=0;
#endif
    sMarker[0]=nSize=nCount=nItemsRead=0; nSeek=-1; 
  } 
  char sMarker[4];
  int nSize;
  int nCount;
  int nItemsRead;
  int nSeek;

#ifdef FS_DEBUG
  char szCallStack[1024];
  char szFileName[1024];
#endif
} ;

#define RS_BUFFER_SIZE (40*1024*1024)
#define RS_CHUNK_SIZE (4*1024*1024)

class CReadSequencerThread : public CrySimpleThread<>
{
public:

  CReadSequencerThread(const char * szFileName)
  {	
    bContinue=true;
    bDone=false;
    m_nLoadedStart=0;
    m_nLoadedEnd=0;
    m_nSeek=0;
    m_bm_nLoadedEOF=false;

    m_pBuffer = new byte[RS_BUFFER_SIZE];
    m_pBufferAsync = new byte[RS_CHUNK_SIZE];

    sFileName = szFileName;

    Start(0, "ReadSequencerThread");
  }

  ~CReadSequencerThread()
  {
    Stop();
    Cancel();
    WaitForThread();
    while(!bDone)
      CrySleep(10);
    Stop();
    WaitForThread();

    delete [] m_pBuffer;
    delete [] m_pBufferAsync;
  }

  int Fread(void *pData, size_t nSize, size_t nCount, FILE * hFile)
  {
		LOADING_TIME_PROFILE_SECTION;

    int nBytesToRead = nSize*nCount;

    int nNotReadyCount = 0;

    const int nSleep = 20;

    while(1)
    {
      if(m_bm_nLoadedEOF && m_nSeek+nBytesToRead > m_nLoadedEnd)
      {
        AUTO_LOCK(m_csRead);

        if(m_bm_nLoadedEOF && m_nSeek+nBytesToRead > m_nLoadedEnd)
        {
          m_nSeek = m_nLoadedEnd;
          return 0;
        }
      }

      if(m_nSeek >= m_nLoadedStart && m_nSeek+nBytesToRead <= m_nLoadedEnd)
      {
        AUTO_LOCK(m_csRead);

        if(m_nSeek >= m_nLoadedStart && m_nSeek+nBytesToRead <= m_nLoadedEnd)
        {
          if(nNotReadyCount)
            CryLogAlways( "FILE_SEQUENCER(%2d, %.1f s): Data not ready stall: %d ms, %.1f MB, %d, %s ", 
            CFileReadSequencer::nMessageOrder++, CFileReadSequencer::GetSectionTime(), 
            nNotReadyCount*nSleep, float(nBytesToRead)/1024.f/1024.f, m_nSeek, CFileReadSequencer::GetFileName(hFile) );

          if(pData)
            memcpy(pData, &m_pBuffer[m_nSeek-m_nLoadedStart], nBytesToRead);
          m_nSeek += nBytesToRead;

          return nCount;
        }
      }

      CrySleep(nSleep);

      nNotReadyCount++;
    }
  }

private:

  virtual void Run()
  {
    CryThreadSetName( ~0, "ReadSequencerThread" );

    FILE * pFile = ::fopen(sFileName,"rbS");

    int nAsyncReady = 0;

    int nReadChunkSize = RS_CHUNK_SIZE/256;

    while(bContinue && pFile)
    {       
      if(!nAsyncReady)
        nAsyncReady = ::fread(m_pBufferAsync, 1, nReadChunkSize, pFile);
      else
        CrySleep(10);

      if((m_nSeek >= m_nLoadedStart+RS_CHUNK_SIZE || m_nLoadedEnd<RS_BUFFER_SIZE) && nAsyncReady)
      {
        AUTO_LOCK(m_csRead);

        if((m_nSeek >= m_nLoadedStart+RS_CHUNK_SIZE || m_nLoadedEnd<RS_BUFFER_SIZE) && nAsyncReady)
        {
          if(m_nLoadedEnd<RS_BUFFER_SIZE)
          {
            // append new data
            memcpy(&m_pBuffer[m_nLoadedEnd], m_pBufferAsync, nAsyncReady);        
            m_nLoadedEnd += nAsyncReady;
          }
          else
          {
            // shift data
            memcpy(&m_pBuffer[0],&m_pBuffer[RS_CHUNK_SIZE],RS_BUFFER_SIZE-RS_CHUNK_SIZE);

            // append new data
            memcpy(&m_pBuffer[RS_BUFFER_SIZE-RS_CHUNK_SIZE], m_pBufferAsync, nAsyncReady);        

            // update loaded range
            m_nLoadedStart += RS_CHUNK_SIZE;
            m_nLoadedEnd += nAsyncReady;
          }

          // detect EOF
          if(nAsyncReady != nReadChunkSize)
          {
            m_bm_nLoadedEOF = true;
            break;
          }

          if(m_nLoadedEnd >= RS_CHUNK_SIZE && nReadChunkSize != RS_CHUNK_SIZE)
          {
//            CryLogAlways( "FILE_SEQUENCER(%2d, %.1f s): ReadChunkSize is %d MB", 
  //            CFileReadSequencer::nMessageOrder++, CFileReadSequencer::GetSectionTime(), RS_CHUNK_SIZE/1024/1024);

            nReadChunkSize = RS_CHUNK_SIZE;
          }

          // request next read
          nAsyncReady = 0;
        }
      }
    }

    m_bm_nLoadedEOF = true;

    if(pFile)
      ::fclose(pFile);

    bDone = true;
  }

  virtual void Cancel()
  {
    bContinue = false;
  }

  bool bContinue;
  bool bDone;

  byte * m_pBuffer, * m_pBufferAsync;
  int m_nLoadedStart;
  int m_nLoadedEnd;
  int m_nSeek;
  bool m_bm_nLoadedEOF;

  CryCriticalSection m_csRead;

  string sFileName;
};


void CFileReadSequencer::OnFopen(FILE * f, const char * file, const char * mode)
{
  AUTO_LOCK(csMapOpenFiles);

  mapOpenFiles[f] = file;
}

int CFileReadSequencer::Fread(void *pData, size_t nSize, size_t nCount, FILE *hFile)
{
  if(!CFileReadSequencer::nSequencerMode)
    return 0;

  if(GetCurrentThreadId() != dwTargetThread)
  {
    nBytesSequncedNone += nCount*nSize;
    nReadsNumSequncedNone ++;
    return 0;
  }

  float fStartTime = gEnv->pTimer->GetAsyncCurTime();

  AUTO_LOCK(csFread);

  if(GetCurrentThreadId() != dwTargetThread)
  {
    nBytesSequncedNone += nCount*nSize;
    nReadsNumSequncedNone ++;
    return 0;
  }

  if(nSize*nCount > RS_CHUNK_SIZE)
  {
    const char * pFileName = GetFileName(hFile);

    CryLogAlways( "FILE_SEQUENCER(%2d, %.1f s): Block is bigger than %.1f MB: %.1f MB, %s", 
      nMessageOrder++, GetSectionTime(), 
      float(RS_CHUNK_SIZE)/1024.f/1024.f, float(nSize*nCount)/1024.f/1024.f, pFileName );
  }

  int nItemsRead=0;

  if(nSequencerMode==2) // create sequencer file
  {
    int nSeek = ::ftell(hFile);

    nItemsRead = ::fread(pData, nSize, nCount, hFile);

#ifdef XENON
		HRESULT hr = DmMapDevkitDrive();
#endif

    if(m_bRestart)
    {
      if(hSeqFile)
        ::fclose(hSeqFile);
      char szFileName[512];
      sprintf_s(szFileName,pSeqFileNameWrite,CFileReadSequencer::sSectionName.c_str());
      hSeqFile = ::fopen(szFileName,"wb");      
      m_bRestart = false;
    }

    if(hSeqFile)
    {
      char * pMarker = "CRY";
      ::fwrite(pMarker, 4, 1, hSeqFile);

      ::fwrite(&nSize, sizeof(nSize), 1, hSeqFile);
      ::fwrite(&nCount, sizeof(nCount), 1, hSeqFile);
      ::fwrite(&nItemsRead, sizeof(nItemsRead), 1, hSeqFile);

      ::fwrite(&nSeek, sizeof(nSeek), 1, hSeqFile);

#ifdef FS_DEBUG
      static char szCallStack[1024]="";
      strcpy_s(szCallStack, sizeof(szCallStack), gEnv->pSystem->GetLoadingProfilerCallstack());
      ::fwrite(szCallStack, sizeof(szCallStack), 1, hSeqFile);
      static char szFileName[1024]="";
      strcpy_s(szFileName, sizeof(szFileName), GetFileName(hFile));
      ::fwrite(szFileName, sizeof(szFileName), 1, hSeqFile);
#endif

      ::fwrite(pData, nSize, nCount, hSeqFile);
      ::fflush(hSeqFile);

			nBytesSequnced += nCount*nSize;
      nReadsNumSequnced ++;
    }
  }

  if(nSequencerMode==1) // load from sequencer file
  {
    if(m_bRestart)
    {
      SAFE_DELETE(pReadThread);
      char szFileName[512];
      sprintf_s(szFileName,pSeqFileNameRead,CFileReadSequencer::sSectionName.c_str());
      pReadThread = new CReadSequencerThread(szFileName);
      m_bRestart = false;
    }

    int nMissesNum = 0;

    SSeqHeader seqHeaderMiss;

    int nSeek = ::ftell(hFile);

    while(pReadThread)
    {
      SSeqHeader seqHeader;

      if(pReadThread->Fread(&seqHeader, sizeof(seqHeader), 1, hFile) != 1)
      {
        LogStatus("STREAM_ENDED", hFile);
        SAFE_DELETE(pReadThread);
      }

      if(pReadThread)
        if(strncmp(seqHeader.sMarker, "CRY", sizeof(seqHeader.sMarker)))
      {
        LogStatus("STREAM_MARKER_ERROR", hFile);
        SAFE_DELETE(pReadThread);
      }

      bool bFound = (pReadThread!=0);
      if(pReadThread)
        if(seqHeader.nSize != nSize || seqHeader.nCount != nCount || seqHeader.nSeek != nSeek)
      {
        if(!seqHeaderMiss.nSize)
          seqHeaderMiss = seqHeader;

        bFound = false;

        if(!nMissesNum)
        {
          const char * pFileName = GetFileName(hFile);
          CryLogAlways( "FILE_SEQUENCER(%2d, %.1f s): Miss detected: %d bytes, file: %s", 
            nMessageOrder++, GetSectionTime(), nCount*nSize, pFileName );
        }

        nMissesNum++;
      }

      if(pReadThread)
        nItemsRead = seqHeader.nItemsRead;

      if(pReadThread)
      {
        if(pReadThread->Fread(bFound ? pData : NULL, seqHeader.nSize, seqHeader.nCount, hFile) != seqHeader.nCount)
        {
          LogStatus("STREAM_READ_ERROR", hFile);
          SAFE_DELETE(pReadThread);
        }

        if(pReadThread)
        {
          if(bFound)
          {
            nBytesSequnced += nItemsRead*seqHeader.nSize;
            nReadsNumSequnced++;
          }
          else
          {
            nBytesSequncedSkip += nItemsRead*seqHeader.nSize;
            nReadsNumSequncedSkip++;
          }
        }
      }

      if(bFound)
      {
        if(nMissesNum)
        {
          seqHeaderMiss.Reset(); // miss fixed

          const char * pFileName = GetFileName(hFile);

          CryLogAlways( "FILE_SEQUENCER(%2d, %.1f s): Miss fixed: %d blocks, %d bytes, file: %s", 
            nMessageOrder++, GetSectionTime(), nMissesNum, nCount*nSize, pFileName );
        
          nMissesNum = 0;
        }

        break;
      }
    }

    if(pReadThread && nItemsRead)
    {
      ::fseek(hFile, nSize*nCount, SEEK_CUR);
    }
    else if(seqHeaderMiss.nSize)
    {
      const char * pFileName = GetFileName(hFile);

      if(nMissesNum)
      {
        seqHeaderMiss.Reset(); // miss fixed

        const char * pFileName = GetFileName(hFile);

        CryLogAlways( "FILE_SEQUENCER(%2d, %.1f s): DATA NOT FOUND, Miss not fixed: %d blocks, %d bytes, file: %s", 
          nMessageOrder++, GetSectionTime(), nMissesNum, nCount*nSize, pFileName );

        nMissesNum = 0;
      }
      else
        CryLogAlways( "FILE_SEQUENCER(%2d, %.1f s): DATA NOT FOUND, No miss, File: %s", 
        nMessageOrder++, GetSectionTime(), pFileName );
    }
  }

  CFileReadSequencer::fTimeInMainThreadRead += max(0.f, gEnv->pTimer->GetAsyncCurTime() - fStartTime);

  return nItemsRead;
}

void CFileReadSequencer::CloseSeqFile()
{
  // close file operations
  SAFE_DELETE(pReadThread);
  if(hSeqFile)
  {
    ::fclose(hSeqFile);
    hSeqFile = 0;
  }
}

const char * CFileReadSequencer::GetFileName(FILE * hFile)
{
  AUTO_LOCK(csMapOpenFiles);

  static string szFileName;

  std::map<FILE*,string>::iterator iter = mapOpenFiles.find(hFile);
  if(iter == mapOpenFiles.end())
    szFileName = "";
  else
    szFileName = iter->second;

  return szFileName.c_str();
}

void CFileReadSequencer::LogStatus(const char * pCommandName, FILE * hFile)
{
  const char * pFileName = GetFileName(hFile);

  CryLogAlways( "FILE_SEQUENCER(%2d, %.1f s): %s, %s, %s, %.1f (%.1f/%.1f) MBytes, %d (%d/%d) reads, %.1f sec, %s", 
    nMessageOrder++, GetSectionTime(),
    pCommandName, sSectionName.c_str(), (nSequencerMode == 2) ? "CREATE" : "READ", 
    float(nBytesSequnced)/1024/1024, float(nBytesSequncedSkip)/1024/1024, float(nBytesSequncedNone)/1024/1024,
    nReadsNumSequnced, nReadsNumSequncedSkip, nReadsNumSequncedNone, 
    fTimeInMainThreadRead, pFileName );
}

void CFileReadSequencer::StartSection( const char * pSectionName )
{
  AUTO_LOCK(csFread);

  sectionStartTime = gEnv->pTimer->GetAsyncTime();

  static ICVar * pVar = NULL;
  if(pVar==NULL)
  {
    pVar = gEnv->pConsole->GetCVar("sys_FileReadSequencer");
    nSequencerMode = pVar->GetIVal();
  }

  assert(pSectionName);

  if(strstr(pSectionName,"/"))
    sSectionName = strstr(pSectionName,"/")+1;
  else
    sSectionName = pSectionName;

  if(!nSequencerMode)
    return;

  dwTargetThread = GetCurrentThreadId();

  CloseSeqFile();

  m_bRestart = true;

  if(nSequencerMode==1) // load from sequencer file
  {
    if(m_bRestart)
    {
      SAFE_DELETE(pReadThread);
      char szFileName[512];
      sprintf_s(szFileName,pSeqFileNameRead,CFileReadSequencer::sSectionName.c_str());
      pReadThread = new CReadSequencerThread(szFileName);
      m_bRestart = false;
    }
  }

  // reset stats
  fTimeInMainThreadRead = 0;
  nReadsNumSequnced = nBytesSequnced = 0;
  nReadsNumSequncedSkip = nBytesSequncedSkip = 0;
  nReadsNumSequncedNone = nBytesSequncedNone = 0;

  LogStatus("SECTION_START", NULL);
}

void CFileReadSequencer::UpdateCurrentThread( )
{
  AUTO_LOCK(csFread);

  if(!nSequencerMode)
    return;

  dwTargetThread = GetCurrentThreadId();
}

float CFileReadSequencer::GetSectionTime()
{
  return (gEnv->pTimer->GetAsyncTime()-sectionStartTime).GetSeconds();
}

void CFileReadSequencer::EndSection( )
{
  AUTO_LOCK(csFread);

  if(!nSequencerMode)
    return;

  CloseSeqFile();

  if(nSequencerMode==2 && strcmp(sSectionName.c_str(), "SYSTEM"))
  {
#ifdef XENON
    HRESULT hr = DmMapDevkitDrive();
#endif

    char szFileNameDone[256]="";
    sprintf(szFileNameDone,pSeqFileNameWrite,"DONE");
    if(FILE * fD = ::fopen(szFileNameDone,"wb"))
      ::fclose(fD);
  }

  LogStatus("SECTION_END", NULL);
}

IFileReadSequencer * CFileReadSequencer::GetFileReadSequencer()
{
  static CFileReadSequencer s;
  return &s;
}
