////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2004.
// -------------------------------------------------------------------------
//  File name:   PS3MemoryManagement
//  Version:     v1.00
//  Created:     8/8/2009 by Chris Raine
//  Compilers:   GCC 4.1.1
//  Description: System Memory Management routines for ps3
// -------------------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////

#if !defined(PS3MEMORYMANAGEMENT_H) && defined(PS3)
#define PS3MEMORYMANAGEMENT_H

#include <stdint.h>

#undef calloc
#undef malloc
#undef free
#undef realloc
#undef reallocalign
#undef memalign

// Memory can be mapped in chunks of 64 kb or in chunks of 1 mb
enum PAGESIZE
{
    PAGESIZE_64KB = 64<<10,
    PAGESIZE_1M   = 1<<20,
};

// Allocator statistics 
struct allocator_stats
{
  // The bytes allocated within the allocator
  size_t allocated; 

  // The bytes actually in use within the allocator
  size_t in_use;    

  // The bytes wasted due to internal structure overhead
  size_t wasted_internal;    

  // The largest possible allocation possible that would not trigger a
  // request for more physical memory from the operating system. 
  size_t largest_free_block; 

  // The largest possible allocation 
  size_t aquired_physical_mem; 

  // Reclaimable physical memory from this allocator 
  size_t reclaimable_physical_mem; 
}; 

// Physical memory statistics 
struct physcal_memory_stats
{
  // The number of 64kb physical pages currently acquired 
  size_t pages_64kb_aquired; 

  // The number of 1mb physical pages currently acquired 
  size_t pages_1mb_aquired; 

  // The number of bytes reclaimable
  size_t mem_bytes_in_use; 

  // The number of bytes reclaimable
  size_t mem_bytes_reclaimable; 
}; 

////////////////////////////////////////////////////////////////////////////////
// Scratch memory allocator for non persistent allocations 
////////////////////////////////////////////////////////////////////////////////

class CScratchMemoryAllocator
{
  struct internals; 
  internals* m_internals; 

  // The scratch memory constructor. Instances of
  // CScratchMemoryAllocator are non-assignable and
  // non-copy-constructable.
  // Please use the static create method.
  CScratchMemoryAllocator() {};
  CScratchMemoryAllocator(CScratchMemoryAllocator&);
  CScratchMemoryAllocator& operator= (CScratchMemoryAllocator&);
  
public:
  // Retrieve the 
  void get_statistics(allocator_stats&) const; 

  // Allocate a chunk of memory 
  void* allocate(size_t size, size_t boundary);

  // Deallocate a chunk of memory 
	void deallocate(void* _ptr);

  // Resize an existing allocation to a new boundary 
	void* update(void* _ptr, size_t size, size_t boundary);

  // Returns a non-null handle if the allocator contains the
  // allocation (e.g is responsible for freeing it)
  void* contains(void* _ptr) const;

  // Returns the memory size of an allocation if it is contained
  // within the allocator.
  size_t msize(void* _ptr) const;

  // Create a scratch memory allocator for the system 
  static CScratchMemoryAllocator* create(PAGESIZE);
}; 


////////////////////////////////////////////////////////////////////////////////
// System interface. The below functions mimic the space headers for
// scratch allocation and 
////////////////////////////////////////////////////////////////////////////////
namespace sys
{
  // Initialize the memory management subsystem. Returns true on
  // success
  bool initialize(); 

  // Finalize (shutdown) the memory management subsystem.  
  bool finalize();

  // Memory Interface 
  size_t _msize(void*);
  void* calloc(size_t, size_t);
	void* malloc(size_t);
	void  free(void *);
	void* realloc(void *, size_t);
	void* reallocalign(void *, size_t, size_t);
	void* memalign(size_t, size_t);
}; 

#endif //#if !defined(PS3MEMORYMANAGEMENT_H) && defined(PS3)
