////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   PlatformOS_PC.cpp
//  Created:     11/02/2010 by Alex McCarthy.
//  Description: Implementation of the IPlatformOS interface for PC
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include <StdAfx.h>

#if !defined(XENON) && !defined(PS3)

#if defined(WIN32) || defined(WIN64)
#define PLATFORM_WINDOWS
#endif

#include "PlatformOS_PC.h"
#include "SaveReaderWriter_CryPak.h"
#include "ICryPak.h"


IPlatformOS* IPlatformOS::Create()
{
	return new CPlatformOS_PC();
}

CPlatformOS_PC::CPlatformOS_PC()
: m_listeners(4)
, m_bSignedIn(false)
{
}

bool CPlatformOS_PC::GetUserProfilePreference(unsigned int user, EUserProfilePreference ePreference, SUserProfileVariant& outResult) const
{
	return false;
}

unsigned int CPlatformOS_PC::UserGetMaximumSignedInUsers() const
{
	return 1;
}

bool CPlatformOS_PC::UserIsSignedIn(unsigned int userIndex) const
{
	return m_bSignedIn;
}

bool CPlatformOS_PC::UserDoSignIn(unsigned int numUsersRequested)
{
	if(!m_bSignedIn)
	{
		m_bSignedIn = true;

		// Tell the system that we are signed in
		IPlatformOS::SPlatformEvent event(0);
		event.m_eEventType = SPlatformEvent::eET_SignIn;
		event.m_uParams.m_signIn.m_signedIn = true;
		NotifyListeners(event);

		// Tell the system that storage is mounted - required for CGame
		event.m_eEventType = SPlatformEvent::eET_StorageMounted;
		NotifyListeners(event);
	}
	return true;
}

bool CPlatformOS_PC::UserGetName(unsigned int userIndex, IPlatformOS::TUserName& outName) const
{
#ifdef PLATFORM_WINDOWS
	DWORD numChars = outName.MAX_SIZE;
	BOOL e = GetUserNameA(outName.m_strBuf, &numChars);
	return e ? true : false;
#else
	outName.assign( gEnv->pSystem->GetUserName() );
	return true;
#endif
}

IPlatformOS::ISaveReaderPtr CPlatformOS_PC::SaveGetReader(const char* fileName, unsigned int /*user*/)
{
	CSaveReader_CryPakPtr	pSaveReader(new CSaveReader_CryPak(fileName));

	if(!pSaveReader || pSaveReader->LastError() != IPlatformOS::eFOC_Success)
	{
		return CSaveReader_CryPakPtr(NULL);
	}
	else
	{
		return pSaveReader;
	}
}

IPlatformOS::ISaveWriterPtr CPlatformOS_PC::SaveGetWriter(const char* fileName, unsigned int /*user*/)
{
	CSaveWriter_CryPakPtr	pSaveWriter(new CSaveWriter_CryPak(fileName));

	if(!pSaveWriter || pSaveWriter->LastError() != IPlatformOS::eFOC_Success)
	{
		return CSaveWriter_CryPakPtr(NULL);
	}
	else
	{
		return pSaveWriter;
	}
}


/* 
--------------------- Listener -----------------------
*/

void CPlatformOS_PC::AddListener(IPlatformListener* pListener, const char* szName)
{
	m_listeners.Add(pListener, szName);
}

IPlatformOS::EMsgBoxResult
CPlatformOS_PC::DebugMessageBox( const char* body, const char* title, unsigned int flags ) const
{
#ifdef PLATFORM_WINDOWS
	int winresult = CryMessageBox( body, title, MB_OKCANCEL );
	return (winresult == IDOK) ? eMsgBox_OK : eMsgBox_Cancel;
#else
	CRY_ASSERT_MESSAGE(false, "DebugMessageBox not implemented on non-windows platforms!");
	return eMsgBox_OK; // [AlexMcC|30.03.10]: Ok? Cancel? Dunno! Uh-oh :( This is only used in CryPak.cpp so far, and for that use it's better to return ok
#endif
}

void CPlatformOS_PC::RemoveListener(IPlatformListener* pListener)
{
	m_listeners.Remove(pListener);
}

void CPlatformOS_PC::NotifyListeners(SPlatformEvent& event)
{
	for (CListenerSet<IPlatformListener*>::Notifier notifier(m_listeners); notifier.IsValid(); notifier.Next())
		notifier->OnPlatformEvent(event);
}

#endif // !defined(XENON) && !defined(PS3)

