////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   PlatformOS_PS3.cpp
//  Created:     11/02/2010 by Alex McCarthy.
//  Description: Implementation of the IPlatformOS interface for PS3
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include <StdAfx.h>

#ifdef PS3

#include "PlatformOS_PS3.h"
#include "SaveReaderWriter_CryPak.h"
#include "SysCallThread_PS3.h"
#include "../System.h"
#include "PS3Specific.h"

#include <sysutil/sysutil_msgdialog.h>

class CPlatformOS_PS3_Impl : public CPlatformOS_PS3
{
	CSysCallThread	m_sysThreadObject;
public:
	ILINE CPlatformOS_PS3_Impl()
	{
		m_pSysThread = &m_sysThreadObject;
	}
};

const char SAVEDATATHREAD_NAME[] = "SaveDataThread";
const int SAVEDATATHREAD_STACK_SIZE = 32*1024;
const int SAVEDATATHREAD_PRIORITY = 1001;
const int SAVEDATATHREAD_EVENT_QUEUE_DEPTH = 8;
const int SAVEDATATHREAD_DATA_QUEUE_DEPTH = 8;

const char SAVEDATA_PARAMSFO_TITLE[] = "Crysis 2 Save Title";
const char SAVEDATA_PARAMSFO_SUBTITLE[] = "Crysis 2 Save Subtitle";
const char SAVEDATA_PARAMSFO_DETAIL[] = "Created by Crysis 2";

const int SAVEDATA_FILE_LIST_SIZE = 8;
const int SAVEDATA_DIR_LIST_SIZE = 4;

const int SAVEDATA_WRITE_CHUNK_SIZE = 1024;

const byte QUEUE_ATTRIBS_ELEMENT_SIZE_ALIGN = 16; //Queue structures should be 16 byte aligned

#define SAVEDATA_BUFFER_SIZE (MAX(SAVEDATA_DIR_LIST_SIZE * sizeof(CellSaveDataDirList), SAVEDATA_FILE_LIST_SIZE * sizeof(CellSaveDataFileStat)))

static char g_setBufBuffer[SAVEDATA_BUFFER_SIZE];

static void SaveDataSaveFileCB(CellSaveDataCBResult *cbResult,	CellSaveDataFileGet *get, CellSaveDataFileSet *set);
static void SaveDataSaveStatCB(CellSaveDataCBResult *cbResult, CellSaveDataStatGet *get, CellSaveDataStatSet *set);

enum ESaveDataFileType
{
	eSDFT_Icon0 = 0,
	eSDFT_Icon1,
	eSDFT_Pic1,
	eSDFT_Snd0,
	eSDFT_DataFile,

	eSDFT_Invalid = 0x7FFFFFFF
};

enum ESaveDataState
{
	eSDS_Idle = 0,
	eSDS_SaveInit,
	eSDS_SaveDataStatus,
	eSDS_SaveFileOp,
	eSDS_LoadInit,
	eSDS_LoadDataStatus,
	eSDS_LoadFileOp,

	eSDS_Invalid = 0x7FFFFFFF
};

struct SSaveDataEvent
{
	enum ESaveDataEventType
	{
		eSDET_Save = 0,
		eSDET_Load,
		eSDET_Exit,

		eSDET_Invalid = 0x7FFFFFFF
	};

	ESaveDataEventType m_type;
	IPlatformOS::ISaveWriterPtr m_saveWriter;

	unsigned char padding[QUEUE_ATTRIBS_ELEMENT_SIZE_ALIGN - (sizeof(ESaveDataEventType) + sizeof(IPlatformOS::ISaveWriterPtr))];
};

struct SSaveFileEntry
{
	CSaveWriter_PS3* m_saveWriter;
	char m_fileName[CELL_SAVEDATA_FILENAME_SIZE];
	ESaveDataFileType m_fileType;
};

struct SGlobalSaveData
{
	SSaveFileEntry m_saveFiles[SAVEDATA_FILE_LIST_SIZE];
	int m_saveFileCount;
	int m_currentSaveFileIndex;

	char m_saveDataDirPrefix[CELL_SAVEDATA_DIRNAME_SIZE];
	char m_saveDataDirFull[CELL_SAVEDATA_DIRNAME_SIZE];

	CellSaveDataDirList m_modifiedDirList[SAVEDATA_DIR_LIST_SIZE];

	CellSaveDataListNewData m_saveDataDef;
	CellSaveDataSystemFileParam m_saveDataParamSFO;

	ESaveDataState m_state;

	byte *m_saveDataBuffer;

} g_saveData;

int TranslateSaveDataFileType(ESaveDataFileType ft)
{
	int translateArray[] =
	{
		CELL_SAVEDATA_FILETYPE_CONTENT_ICON0,
		CELL_SAVEDATA_FILETYPE_CONTENT_ICON1,
		CELL_SAVEDATA_FILETYPE_CONTENT_PIC1,
		CELL_SAVEDATA_FILETYPE_CONTENT_SND0,
		CELL_SAVEDATA_FILETYPE_NORMALFILE
	};

	return(translateArray[ft]);
}

IPlatformOS* IPlatformOS::Create()
{
	return new CPlatformOS_PS3_Impl();
}

CPlatformOS_PS3::CPlatformOS_PS3()
:	m_listeners(4)
, m_bSignedIn(false)
, m_threadID(0)
, m_saveDataEventQueue(0)
, m_eventQueueBuffer(0)
, m_bIsSaving(false)
,	m_bIsQuitting(false)
{
	gPS3Env->tickCellSysUtilCallbacksState = TICKCELLSYSUTIL_GAME_THREAD;

	//Check for an early quit event
	if(gPS3Env->bEarlyQuitMessageDetected==1)
	{
		//Set the quit flag that the game loop uses to exit
		gEnv->pSystem->Quit();
		//As we're quitting, there's no need to carry on initialising this object
		return;
	}

	AddListener(this, "CPlatformOS_PS3");
	
	// load system module for CellSync2Queue
	// NOTE: This loads prx with ~150kb, please refactor to not use this functionality
	int cRet = cellSysmoduleLoadModule(CELL_SYSMODULE_SYNC2);
	if(cRet != CELL_OK) 
	{
		printf("Fatal error: cellSysmoduleLoadModule(CELL_SYSMODULE_SYNC2) = 0x%08x\n",cRet);
		exit(EXIT_FAILURE);
	}

	// set up event queue
	CellSync2QueueAttribute queueAttribs;
	cellSync2QueueAttributeInitialize(&queueAttribs);

	queueAttribs.depth = SAVEDATATHREAD_EVENT_QUEUE_DEPTH;
	//has to be 16*N
	queueAttribs.elementSize = sizeof(SSaveDataEvent);
	queueAttribs.maxPopWaiters = 16;
	queueAttribs.maxPushWaiters = 16;

	size_t queueBufSize = 0;
	int ret = cellSync2QueueEstimateBufferSize(&queueAttribs, &queueBufSize);
	assert(ret == CELL_OK);
	m_eventQueueBuffer = CryModuleMemalign(queueBufSize, CELL_SYNC2_QUEUE_BUFFER_ALIGN);

	m_saveDataEventQueue = reinterpret_cast<CellSync2Queue *>(CryModuleMemalign(CELL_SYNC2_QUEUE_SIZE, CELL_SYNC2_QUEUE_ALIGN));
	ret = cellSync2QueueInitialize(m_saveDataEventQueue, m_eventQueueBuffer, &queueAttribs);
	assert(ret == CELL_OK);

	g_saveData.m_saveDataBuffer = new byte[SAVEDATA_WRITE_CHUNK_SIZE];
	assert(g_saveData.m_saveDataBuffer);

	if (UsePlatformSavingAPI())
	{
		ret = sys_ppu_thread_create(&m_threadID,
									CPlatformOS_PS3::SaveDataThreadFunc,
									(uint64_t)this,
									SAVEDATATHREAD_PRIORITY,
									SAVEDATATHREAD_STACK_SIZE,
									0,
									SAVEDATATHREAD_NAME);

		assert(ret == CELL_OK);
	}
	
	const char* productCode = "CRYS201234";	// TODO: replace with correct product ID
	const char* dirPrefix = "C2SAVE";
	int saveVersion = 0;

	sprintf_s(g_saveData.m_saveDataDirPrefix, "%s%s%02d", productCode, dirPrefix, saveVersion);
}

CPlatformOS_PS3::~CPlatformOS_PS3()
{
	delete [] g_saveData.m_saveDataBuffer;
	CryModuleFree(m_eventQueueBuffer);

	//Dispatch the exit event to the thread
	SSaveDataEvent someData;
	someData.m_type = SSaveDataEvent::eSDET_Exit;

	cellSync2QueuePush(m_saveDataEventQueue, &someData, 0);

	RemoveListener(this);
}

bool CPlatformOS_PS3::UserDoSignIn(unsigned int numUsersRequested)
{
	if(!m_bSignedIn)
	{
		m_bSignedIn = true;

		// Tell the system that we are signed in
		IPlatformOS::SPlatformEvent event(0);
		event.m_eEventType = SPlatformEvent::eET_SignIn;
		event.m_uParams.m_signIn.m_signedIn = true;
		NotifyListeners(event);

		// Tell the system that storage is mounted - required for CGame
		event.m_eEventType = SPlatformEvent::eET_StorageMounted;
		NotifyListeners(event);
	}
	return true;
}


void CPlatformOS_PS3::Tick(float realFrameTime)
{
	//Only check the cellSys events in this thread if we're listening for them
	if(gPS3Env->tickCellSysUtilCallbacksState == TICKCELLSYSUTIL_GAME_THREAD)
	{
		cellSysutilCheckCallback();
	}
	//Check for a deferred quit event
	if( m_bIsQuitting == true && m_bIsSaving == false )
	{
		//Set the quit flag that the game loop listens for
		gEnv->pSystem->Quit();
		//For future proofing, don't call that routine more than once, it might not be safe to do so
		m_bIsQuitting = false;
	}
}

bool CPlatformOS_PS3::UserGetName(unsigned int userIndex, IPlatformOS::TUserName& outName) const
{
	int e = cellSysutilGetSystemParamString(CELL_SYSUTIL_SYSTEMPARAM_ID_CURRENT_USERNAME, outName.m_strBuf, outName.MAX_SIZE);
	return e == CELL_OK;
}

bool CPlatformOS_PS3::FileExists(unsigned int user, const char* path)
{
	return gEnv->pCryPak->IsFileExist(path);
}

IPlatformOS::ISaveReaderPtr CPlatformOS_PS3::SaveGetReader(const char* fileName, unsigned int user)
{
	string tempFileName(fileName);
	bool bIsSave = tempFileName.find(CRY_SAVEGAME_FILE_EXT) != string::npos;

	//TO_DO : remove bIsSave, handle non-savegames through API
	if(bIsSave && UsePlatformSavingAPI())
	{
		assert(0);//unimplemented
		return(ISaveReaderPtr(0));
	}
	else
	{
		CSaveReader_CryPakPtr	pSaveReader(new CSaveReader_CryPak(fileName));

		if(!pSaveReader || pSaveReader->LastError() != IPlatformOS::eFOC_Success)
		{
			return CSaveReader_CryPakPtr(NULL);
		}
		else
		{
			return pSaveReader;
		}
	}
}

IPlatformOS::ISaveWriterPtr CPlatformOS_PS3::SaveGetWriter(const char* fileName, unsigned int user)
{
	if(UsePlatformSavingAPI())
	{
		SSaveDataEvent someData;
		someData.m_type = SSaveDataEvent::eSDET_Save;
		someData.m_saveWriter = ISaveWriterPtr(new CSaveWriter_PS3(fileName));

		m_bIsSaving = true;	//Set this to defer calling the quit routine while saving
		cellSync2QueuePush(m_saveDataEventQueue, &someData, 0);

		return (someData.m_saveWriter);
	}
	else
	{
		CSaveWriter_CryPakPtr	pSaveWriter(new CSaveWriter_CryPak(fileName));

		if(!pSaveWriter || pSaveWriter->LastError() != IPlatformOS::eFOC_Success)
		{
			return CSaveWriter_CryPakPtr(NULL);
		}
		else
		{
			return pSaveWriter;
		}
	}
}

void CPlatformOS_PS3::AddListener(IPlatformOS::IPlatformListener* pListener, const char* szName)
{
	m_listeners.Add(pListener, szName);
}

IPlatformOS::IFileFinderPtr CPlatformOS_PS3::GetFileFinder()
{
	//if(UsePS3SaveReaderWriter())
	//	return IFileFinderPtr(new CFileFinderPS3(this)); // TODO: implement CFileFinderPS3
	return IFileFinderPtr(new CFileFinderCryPak);
}


bool CPlatformOS_PS3::UsePlatformSavingAPI() const
{
	return g_cvars.sys_usePlatformSavingAPI != 0;
}

void CPlatformOS_PS3::RemoveListener(IPlatformOS::IPlatformListener* pListener)
{
	m_listeners.Remove(pListener);
}

void CPlatformOS_PS3::NotifyListeners(IPlatformOS::SPlatformEvent& event)
{
	//This should always be called from the game thread, either directly or via the TO_GAME macro from the networking thread
	assert(gEnv->mMainThreadId == GetCurrentThreadId());
	for(CListenerSet<IPlatformOS::IPlatformListener*>::Notifier notifier(m_listeners); notifier.IsValid(); notifier.Next())
	{
		notifier->OnPlatformEvent(event);
	}
}

void CPlatformOS_PS3::OnPlatformEvent(const IPlatformOS::SPlatformEvent& event)
{
	//Handle system events such as termination requests here; unless they have registered listeners elsewhere in the codebase
	const SPlatformEventPS3& PS3Event = static_cast<const SPlatformEventPS3&>(event);
	switch(PS3Event.m_status)
	{
	case CELL_SYSUTIL_REQUEST_EXITGAME:
		//Call the same routine that the game console uses. As the PS3 handles shutdown well, Alex W has suggested calling _Exit() and bypassing the destructors if
		//shutdown is riddled with PS3 specific problems. However, some threads are only closed down by destructors so we'd get thread sync finalise errors.
		if( m_bIsSaving == true )
		{
			//Don't quit now, defer and check in the tick
			m_bIsQuitting = true;
		}
		else
		{
			//Not saving, can quit now
			gEnv->pSystem->Quit();
		}
		break;
	default:
		//Unhandled event
		break;
	}
}

void CPlatformOS_PS3::SaveDataThreadFunc(uint64_t arg)
{
	CPlatformOS_PS3 *thisPtr = (CPlatformOS_PS3*)arg;

	SSaveDataEvent poppedData;
	poppedData.m_type = SSaveDataEvent::eSDET_Invalid;

	bool dontExit = true;

	while(dontExit)
	{
		// this is the blocking version by design
		cellSync2QueuePop(thisPtr->m_saveDataEventQueue, &poppedData, 0);

		switch (poppedData.m_type)
		{
			case SSaveDataEvent::eSDET_Save:
			{
				thisPtr->PerformSave(poppedData.m_saveWriter);
				thisPtr->m_bIsSaving = false;	//Set this to false, we may now shut down the game without risking a partial save
			} break;

			case SSaveDataEvent::eSDET_Load:
			{
				thisPtr->PerformLoad();
			} break;

			case SSaveDataEvent::eSDET_Exit:
			{
				dontExit = false;
			} break;

			default:
			{

			} break;
		}
	}

	CryModuleFree(thisPtr->m_saveDataEventQueue);

	sys_ppu_thread_exit(0);
}

static void SaveDataSaveFileCB(	CellSaveDataCBResult *cbResult,	CellSaveDataFileGet *get, CellSaveDataFileSet *set )
{
	static int lastOpWriteSize = 0;

	if (g_saveData.m_state == eSDS_SaveFileOp)
	{
		if (get->excSize != lastOpWriteSize)
		{
			assert(get->excSize == lastOpWriteSize);
			cbResult->result = CELL_SAVEDATA_CBRESULT_ERR_FAILURE;
			return;
		}
	}

	g_saveData.m_state = eSDS_SaveFileOp;
	int dataSize = 0;
	bool isLast = false;

	if (g_saveData.m_saveFiles[g_saveData.m_currentSaveFileIndex].m_fileType == eSDFT_DataFile)
	{
		while(dataSize == 0 && isLast == false)
		{
			g_saveData.m_saveFiles[g_saveData.m_currentSaveFileIndex].m_saveWriter->ReadData(g_saveData.m_saveDataBuffer, SAVEDATA_WRITE_CHUNK_SIZE, dataSize);
			isLast = g_saveData.m_saveFiles[g_saveData.m_currentSaveFileIndex].m_saveWriter->IsComplete();
		}

		if (dataSize)
		{
			memset(set, 0, sizeof(CellSaveDataFileSet));

			set->fileOperation = CELL_SAVEDATA_FILEOP_WRITE;

			set->fileType = TranslateSaveDataFileType(g_saveData.m_saveFiles[g_saveData.m_currentSaveFileIndex].m_fileType);

			set->fileName = g_saveData.m_saveFiles[g_saveData.m_currentSaveFileIndex].m_fileName;
			set->fileBufSize = dataSize;
			set->fileSize = dataSize;
			set->fileBuf = reinterpret_cast<char*>(g_saveData.m_saveDataBuffer);
			lastOpWriteSize = set->fileSize;

			cbResult->result = CELL_SAVEDATA_CBRESULT_OK_NEXT;
		}

		if(isLast)
		{
			cbResult->result = CELL_SAVEDATA_CBRESULT_OK_LAST_NOCONFIRM;
		}
	}
}

void dumpCellSaveDataStatGet(CellSaveDataStatGet *get)
{
	printf("Dump CellSaveDataStatGet in CellSaveDataStatCallback--------------------\n" );
	printf("\tget->dir.dirName : %s\n", get->dir.dirName );
	printf("\tget->isNewData: %d\n", get->isNewData );
	printf("\tget->hddFreeSizeKB 0x%x\n", get->hddFreeSizeKB);
	printf("\tget->sizeKB : 0x%x\n", get->sizeKB);
	printf("\tget->sysSizeKB : 0x%x\n", get->sysSizeKB);
	printf("\tget->bind : %d\n", get->bind);
	printf("\tget->dir : dirName : %s  atime : %lld mtime : %lld ctime : %lld\n", get->dir.dirName, get->dir.st_atime, get->dir.st_mtime, get->dir.st_ctime );
	printf("\tget->fileListNum: %d\n", get->fileListNum );

	for( unsigned int i = 0; i < get->fileListNum; i++ )
	{
		printf("\t%3d  FILENAME: %s   type : %d  size : %lld  atime : %lld mtime : %lld ctime : %lld\n", i,
			get->fileList[i].fileName,
			get->fileList[i].fileType,
			get->fileList[i].st_size,
			get->fileList[i].st_atime,
			get->fileList[i].st_mtime,
			get->fileList[i].st_ctime );
	}
	printf("\tget->fileNum: %d\n", get->fileNum );
	printf("\n" );
	printf("\tPARAM.SFO:TITLE: %s\n", get->getParam.title );
	printf("\tPARAM.SFO:SUB_TITLE: %s\n", get->getParam.subTitle );
	printf("\tPARAM.SFO:DETAIL: %s\n", get->getParam.detail );
	printf("\tPARAM.SFO:ATTRIBUTE: %d\n", get->getParam.attribute );
	printf("\tPARAM.SFO:LIST_PARAM: %s\n", get->getParam.listParam );
	printf("\n" );
}

static void SaveDataSaveStatCB(CellSaveDataCBResult *cbResult, CellSaveDataStatGet *get, CellSaveDataStatSet *set)
{
	memset(set, 0, sizeof(CellSaveDataStatSet));

	// do free size checks here

	dumpCellSaveDataStatGet(get);

	set->setParam = &g_saveData.m_saveDataParamSFO;
	set->reCreateMode = CELL_SAVEDATA_RECREATE_NO_NOBROKEN;

	cbResult->result = CELL_SAVEDATA_CBRESULT_OK_NEXT;
}

int SetupSaveDataParameters(const SSaveFileEntry *fileList, const int fileCount, const char* title, const char* subTitle, const char* detail)
{

	if (fileCount > SAVEDATA_FILE_LIST_SIZE)
	{
		// error
		return(-1);
	}

	g_saveData.m_saveFileCount = fileCount;

	for (int i=0; i < fileCount; ++i)
	{
		g_saveData.m_saveFiles[i] = fileList[i];
	}

	// set up PARAM.SFO
	memset(&g_saveData.m_saveDataParamSFO, 0, sizeof(CellSaveDataSystemFileParam));

	strncpy_s(g_saveData.m_saveDataParamSFO.title, title, CELL_SAVEDATA_SYSP_TITLE_SIZE);
	strncpy_s(g_saveData.m_saveDataParamSFO.subTitle, subTitle, CELL_SAVEDATA_SYSP_SUBTITLE_SIZE);
	strncpy_s(g_saveData.m_saveDataParamSFO.detail, detail, CELL_SAVEDATA_SYSP_DETAIL_SIZE);
	g_saveData.m_saveDataParamSFO.attribute = CELL_SAVEDATA_ATTR_NORMAL;
	strncpy_s(g_saveData.m_saveDataParamSFO.listParam, "0000000", CELL_SAVEDATA_SYSP_LPARAM_SIZE);

	return(0);
}

int CPlatformOS_PS3::PerformSave(IPlatformOS::ISaveWriterPtr saveWriter)
{
	g_saveData.m_state = eSDS_SaveInit;

	CellSaveDataSetBuf setBuf;

	memset(&setBuf,0,sizeof(CellSaveDataSetBuf));
	setBuf.dirListMax = SAVEDATA_DIR_LIST_SIZE;
	setBuf.fileListMax = SAVEDATA_FILE_LIST_SIZE;
	setBuf.bufSize = SAVEDATA_BUFFER_SIZE;
	setBuf.buf = g_setBufBuffer;

	SSaveFileEntry fileEntry;

	fileEntry.m_saveWriter = static_cast<CSaveWriter_PS3*>(saveWriter.get());
	fileEntry.m_fileType = eSDFT_DataFile;
	const char *pFileName = "TEST.CSF"; //fileEntry.m_saveWriter->getFileName();
	CRY_ASSERT(pFileName);
	if(pFileName)
		strncpy(fileEntry.m_fileName, pFileName, CELL_SAVEDATA_FILENAME_SIZE);

	int ret = SetupSaveDataParameters(&fileEntry, 1, SAVEDATA_PARAMSFO_TITLE, SAVEDATA_PARAMSFO_SUBTITLE, SAVEDATA_PARAMSFO_DETAIL);

	if(ret)
	{
		assert(ret == 0);// if somebody actually hits this, they've made a big mistake
		return(ret);
	}

	sprintf_s(g_saveData.m_saveDataDirFull, "%s%02d", g_saveData.m_saveDataDirPrefix, 0);
	g_saveData.m_currentSaveFileIndex = 0;

	ret = cellSaveDataAutoSave2(CELL_SAVEDATA_VERSION_CURRENT, g_saveData.m_saveDataDirFull, CELL_SAVEDATA_ERRDIALOG_ALWAYS, &setBuf, SaveDataSaveStatCB, SaveDataSaveFileCB, SYS_MEMORY_CONTAINER_ID_INVALID, 0);

	return(ret);
}

void CPlatformOS_PS3::PerformLoad()
{

}

static void flip()
{
	IRenderer* pRenderer = gEnv->pRenderer;
	pRenderer->BeginFrame();
	ColorF black( Col_Black );
	pRenderer->ClearBuffer(FRT_CLEAR | FRT_CLEAR_IMMEDIATE, &black);
	pRenderer->EndFrame();
}


IPlatformOS::EMsgBoxResult
CPlatformOS_PS3::DebugMessageBox( const char* body, const char* title, unsigned int flags ) const
{
	struct SHandler
	{
		SHandler(): m_done(false) { }
		static void Callback(int button, void* user)
		{
			SHandler* self = static_cast<SHandler*>(user);
			self->m_done = true;
			self->m_button = button;
		}
		static void None(uint64 status, uint64 param, void* user) { }
		int 	m_button;
		bool 	m_done;
	};
	char msg[CELL_MSGDIALOG_STRING_SIZE];
	SHandler handler;
	const unsigned int type = CELL_MSGDIALOG_TYPE_SE_TYPE_ERROR | CELL_MSGDIALOG_TYPE_BUTTON_TYPE_OK;

	snprintf(msg, sizeof(msg), "%s\n\n%s", title, body);
	msg[CELL_MSGDIALOG_STRING_SIZE - 1] = '\0';

	int status = cellMsgDialogOpen2( type, msg, handler.Callback, &handler, NULL );

	if (status == 0)
	{
		cellSysutilRegisterCallback(1, handler.None, NULL);
		while (!handler.m_done)
		{
			cellSysutilCheckCallback();
			sys_timer_usleep(16600);
			flip();
		}
		cellSysutilUnregisterCallback(1);
	}
	return (handler.m_button == CELL_MSGDIALOG_BUTTON_OK) ? eMsgBox_OK : eMsgBox_Cancel;
}

#include UNIQUE_VIRTUAL_WRAPPER(IPlatformOS)

#endif //PS3
