////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   PlatformOS_Xenon.cpp
//  Created:     11/02/2010 by Alex McCarthy.
//  Description: Implementation of the IPlatformOS interface for Xbox 360
// -------------------------------------------------------------------------
//  History:
//    06/04/2010 Implemented by Ian Masters.
//
////////////////////////////////////////////////////////////////////////////

#include <StdAfx.h>

#ifdef XENON

#include "PlatformOS_Xenon.h"
#include "SaveReaderWriter_CryPak.h"
#include "SaveReaderWriter_Xenon.h"
#include "SaveReaderWriter_Memory.h"
#include <xfilecache.h>
#include "../System.h"


// TODO: localize these and all Message UI strings
static const WCHAR* SAVE_GAME_DISPLAY_NAME = L"Crysis 2";
static const WCHAR* SAVE_GAME_NAME = L"Crysis2";

static const char* const SAVE_ROOT_NAME = "cry";	// cry0 for cry0, cry1 for user1 etc.
static const size_t SAVE_ROOT_NAME_LEN = strlen(SAVE_ROOT_NAME) + 3; // so we can safely fit names into FILE_MAX_NAME/XCONTENT_MAX_FILENAME_LENGTH. +2 for "0:\\" / "1:\\" etc.
static const size_t s_uniqueNameDigits = 5; // this will allow for 10000 unique names when file names conflict in remapping

const char* CPlatformOS_Xenon::SUser::s_fileMapName = "~~xenon~filemap~~";



IPlatformOS* IPlatformOS::Create()
{
	return new CPlatformOS_Xenon();
}


CPlatformOS_Xenon::CPlatformOS_Xenon()
: m_listeners(4)
, m_bFirstSignin(false)
, m_lastFileError(eFOC_Success)
, m_pMemoryFileSystem(IMemoryFileSystem::CreateFileSystem())
{
	// Event listener
	m_listener = XNotifyCreateListener(XNOTIFY_SYSTEM | XNOTIFY_LIVE | XNOTIFY_PARTY);
	CRY_ASSERT(m_listener != NULL);

	for(DWORD user=0; user<MAX_USERS; ++user)
	{
		m_user[user].Reset();

		// Xenon sends user states to the application on startup, however we need to know immediately who is signed in
		// Force a signin if not using the save API
		m_user[user].m_signinState = UsePlatformSavingAPI() ? XUserGetSigninState(user) : eXUserSigninState_NotSignedIn;
	}

	AddListener(this, "Xenon");
}

CPlatformOS_Xenon::~CPlatformOS_Xenon()
{
	RemoveListener(this);

	if (m_listener)
	{
		XCloseHandle(m_listener);
		m_listener = NULL;
	}
}

/* 
--------------------- Listener -----------------------
*/

void CPlatformOS_Xenon::AddListener(IPlatformListener* pListener, const char* szName)
{
	m_listeners.Add(pListener, szName);
}

void CPlatformOS_Xenon::RemoveListener(IPlatformListener* pListener)
{
	m_listeners.Remove(pListener);
}

void CPlatformOS_Xenon::NotifyListeners(SPlatformEvent& event)
{
	for (CListenerSet<IPlatformListener*>::Notifier notifier(m_listeners); notifier.IsValid(); notifier.Next())
		notifier->OnPlatformEvent(event);
}


/*
--------------------- Events -----------------------
*/

void CPlatformOS_Xenon::Tick(float realFrameTime)
{
	// Unless otherwise stated events are platform specific
	SPlatformEventXenon event(0);	// notification message passed to all observers
	event.m_eEventType = SPlatformEvent::eET_PlatformSpecific;

	if (XNotifyGetNext(m_listener, 0, &event.m_id, &event.m_param))
		NotifyListeners(event);

	ProcessPendingRequests();
}



void CPlatformOS_Xenon::OnPlatformEvent(const IPlatformOS::SPlatformEvent& _event)
{
	const SPlatformEventXenon& xenonEvent = static_cast<const SPlatformEventXenon&>(_event);

	switch(xenonEvent.m_eEventType)
	{
	case SPlatformEvent::eET_PlatformSpecific:
		{
			switch(xenonEvent.m_id)
			{
			case XN_SYS_SIGNINCHANGED:
				{
					DWORD numUsersSignedIn = 0;

					// m_param = DWORD dwValid : Bit specifying which players are valid.
					DWORD valid = static_cast<DWORD>(xenonEvent.m_param);
					for(DWORD user=0; user<MAX_USERS; ++user)
					{
						//if(valid & (1 << user)) // The XDK is ambiguous about this bit so lets check all users
						{
							XUSER_SIGNIN_STATE previousSigninState = m_user[user].m_signinState;
							XUSER_SIGNIN_STATE newSigninState = XUserGetSigninState(user);

							// When not using save API things are handled a little differently
							if(!UsePlatformSavingAPI())
							{
								if(newSigninState == previousSigninState) // anything to do?
									continue;

								// We don't allow sign out when not using the save API
								if(newSigninState == eXUserSigninState_NotSignedIn && previousSigninState != eXUserSigninState_NotSignedIn)
									continue;
							}

							m_user[user].m_signinState = newSigninState;

							// Agnostic signin/signout xenonEvent
							SPlatformEvent signinEvent(user);
							signinEvent.m_eEventType = SPlatformEvent::eET_SignIn;
							signinEvent.m_uParams.m_signIn.m_signedIn = m_user[user].m_signinState != eXUserSigninState_NotSignedIn;

							// Broadcast the agnostic xenonEvent - do it before resetting internal state
							if(previousSigninState != m_user[user].m_signinState)
								NotifyListeners(signinEvent);

							if(m_user[user].m_signinState != eXUserSigninState_NotSignedIn)
							{
								++numUsersSignedIn;

								m_user[user].m_userIndex = user;

								// XDK: The title should also call XUserGetXUID and compare the XUID against the previously known offline XUID. The dwValue parameter should be ignored
								XUID previousXUID = m_user[user].m_userXUID;
								XUserGetXUID(user, &m_user[user].m_userXUID);

								// Select storage device if the user is different
								if(previousXUID != m_user[user].m_userXUID)
									UserSelectStorageDevice(user);
							}
							else if(previousSigninState != eXUserSigninState_NotSignedIn)
							{
								// The user was signed in, close everything and reset
								m_user[user].Reset();
							}
						}
					}

					// Single player - check if user is signed in
					ICVar* pCVar = gEnv->pConsole->GetCVar("g_enableGameSwitchMenu");
					if(!gEnv->bMultiplayer && (!pCVar || !pCVar->GetIVal()))
					{
						if(UsePlatformSavingAPI() && numUsersSignedIn == 0 && m_bFirstSignin)
						{
							m_user[0].m_pendingRequests |= ePR_CheckAskConfirmNoSignIn;
							m_bFirstSignin = false;
						}
					}

					break;
				}

			case XN_SYS_STORAGEDEVICESCHANGED:
				{
					// Indicates that a storage device has been inserted or removed
					break;
				}

			case XN_SYS_INPUTDEVICESCHANGED:
				{
					// Indicates that a controller has been connected or disconnected from the system
					break;
				}
			}
			break; // eET_PlatformSpecific
		}
	}
}

bool CPlatformOS_Xenon::WaitResetOverlapped(XOVERLAPPED& overlapped)
{
	if(!XHasOverlappedIoCompleted(&overlapped))
		return false;
	memset(&overlapped, 0, sizeof(XOVERLAPPED));
	return true;
}


/*
--------------------- User -----------------------
*/

unsigned int CPlatformOS_Xenon::UserGetMaximumSignedInUsers() const
{
	return MAX_USERS;
}

bool CPlatformOS_Xenon::UserIsSignedIn(unsigned int user) const
{
	CRY_ASSERT(user < MAX_USERS);
	if(!(user < MAX_USERS))
		return false;
	return m_user[user].m_signinState != eXUserSigninState_NotSignedIn;
}

bool CPlatformOS_Xenon::UserIsSignedIn(const IPlatformOS::TUserName& userName, unsigned int& outUserIndex) const
{
	if(userName.empty()) return false;
	outUserIndex = UserGetPlayerIndex(userName);
	return outUserIndex != Unknown_User;
}

bool CPlatformOS_Xenon::UserDoSignIn(unsigned int numUsersRequested)
{
	// TODO: By god the hacks are piling up. This business is to handle signing in a default profile.
	// They should be removed once we have a in memory storage device of some description handling saves where there is no save device.
	bool signinDefaultUser = numUsersRequested == 0;

	// XDK - XShowSigninUI : The cPanes parameter determines the layout of the sign-in UI and must have a value of either 1, 2, or 4. 
	CRY_ASSERT(signinDefaultUser || (numUsersRequested >= 1 && numUsersRequested <= 4));
	if(!signinDefaultUser && !(numUsersRequested >= 1 && numUsersRequested <= 4))
		return false;

	if(signinDefaultUser)
	{
		// Tell the system that we are signed in
		SPlatformEvent event(Unknown_User);
		event.m_eEventType = SPlatformEvent::eET_SignIn;
		event.m_uParams.m_signIn.m_signedIn = true;
		NotifyListeners(event);

		// Tell the system that storage is mounted - we will be using SaveReaderWriter_Memory
		event.m_eEventType = SPlatformEvent::eET_StorageMounted;
		NotifyListeners(event);
		return true;
	}

	if(!UsePlatformSavingAPI())
	{
		for(unsigned int user = 0; user < numUsersRequested; ++user)
		{
			if(m_user[user].m_signinState == eXUserSigninState_NotSignedIn)
			{
				// Fake it so the system can setup the default profile
				if(!signinDefaultUser)
				{
					m_user[user].m_signinState = eXUserSigninState_SignedInLocally;
					m_user[user].m_userIndex = user;
				}

				// Tell the system that we are signed in
				SPlatformEvent event(user);
				event.m_eEventType = SPlatformEvent::eET_SignIn;
				event.m_uParams.m_signIn.m_signedIn = true;
				NotifyListeners(event);

				// Tell the system that storage is mounted - required for CGame
				event.m_eEventType = SPlatformEvent::eET_StorageMounted;
				NotifyListeners(event);
			}
		}
		return true;
	}

	m_bFirstSignin = true;

	DWORD numPanes = numUsersRequested == 3 ? 4 : numUsersRequested;
	DWORD e = XShowSigninUI(numPanes, XSSUI_FLAGS_LOCALSIGNINONLY);
	if(e != ERROR_SUCCESS)
	{
		// Request login when available
		for(unsigned int user=0; user<numUsersRequested; ++user)
			if(user < numUsersRequested)
				m_user[user].m_pendingRequests |= ePR_SignIn;
	}
	else
		m_user[0].m_pendingRequests |= ePR_CheckAskConfirmNoSignIn; // TODO: check this for multiple signed in users

	return e == ERROR_SUCCESS;
}

unsigned int CPlatformOS_Xenon::UserGetPlayerIndex(const char* userName) const
{
	TUserName name;
	for(unsigned int user=0; user<MAX_USERS; ++user)
	{
		if(UserGetName(user, name))
		{
			if(name == userName)
				return user;
		}
	}
	return Unknown_User;
}

bool CPlatformOS_Xenon::UserGetName(unsigned int user, IPlatformOS::TUserName& outName) const
{
	CRY_ASSERT(user == Unknown_User || user < MAX_USERS);

	// Handle default user
	if(user == Unknown_User && GetFirstSignedInUser() == Unknown_User)
	{
		outName = "Player1";
		return true;
	}

	if(user >= MAX_USERS)
		return false;

	if(!UsePlatformSavingAPI() || m_user[user].m_signinState == eXUserSigninState_NotSignedIn)
	{
		outName.Format("Player%d", user + 1);
		return true;
	}
	char name[XUSER_NAME_SIZE];
	DWORD e = XUserGetName(user, name, XUSER_NAME_SIZE);
	if(e == ERROR_SUCCESS)
		outName = name;
	return e == ERROR_SUCCESS;
}

bool CPlatformOS_Xenon::UserGetNameW(unsigned int user, wstring& userNameW)
{
	TUserName userName;
	if(UserGetName(user, userName))
	{
		// Convert to WCHAR
		StringToWString(userName, userNameW);
		return true;
	}
	return false;
}

unsigned int CPlatformOS_Xenon::UserFromFileName(const char* fileName)
{
	string path = string(fileName).MakeLower().replace('\\', '/');
	static const char* key = "%user%/";
	static const size_t keyLen = strlen(key);
	size_t offset = path.find(key);
	if(offset != string::npos)
	{
		offset += keyLen;
		offset = path.find("/", offset); // skip Profiles or SaveGames
		if(offset != string::npos)
		{
			++offset;
			size_t offset2 = path.find("/", offset);
			if(offset2 != string::npos)
			{
				string userName = string(fileName).substr(offset, offset2-offset);
				unsigned int user;
				if(UserIsSignedIn(userName.c_str(), user))
				{
					return user;
				}
			}
		}
	}
	//CRY_ASSERT_TRACE(false, ("Unable to get user from filename with: %s", fileName));
	// TODO: This needs to be fixed, hopefully by removing this function completely and passing real user indices into save/load functions throughout the engine.
	return 0; //Unknown_User;
}

bool CPlatformOS_Xenon::GetUserProfilePreference(unsigned int user, EUserProfilePreference ePreference, SUserProfileVariant& outResult) const
{
	bool success = false;
	DWORD osPreference;
	DWORD resultsSize = 0;

	// Map EUserProfilePreference to an XPROFILE preference
	switch(ePreference)
	{
	case EUPP_CONTROLLER_INVERT_Y:
		osPreference = XPROFILE_GAMER_YAXIS_INVERSION;
		break;

	case EUPP_CONTROLLER_SENSITIVITY:
		osPreference = XPROFILE_GAMER_CONTROL_SENSITIVITY;
		break;

	case EUPP_GAME_DIFFICULTY:
		osPreference = XPROFILE_GAMER_DIFFICULTY;
		break;

	default:
		CRY_ASSERT_MESSAGE(false, "Invalid EUserProfilePreference");
		return false;
	}

	DWORD error = XUserReadProfileSettings(0, user, 1, &osPreference, &resultsSize, NULL, NULL);

	CRY_ASSERT(error == ERROR_INSUFFICIENT_BUFFER);
	CRY_ASSERT(resultsSize > 0);

	if(error == ERROR_INSUFFICIENT_BUFFER && resultsSize > 0)
	{
		// Allocate the buffer using the size returned by the first call to XUserReadProfileSettings.
		CRY_ASSERT(resultsSize % sizeof(XUSER_READ_PROFILE_SETTING_RESULT) == 0);
		size_t numResults = resultsSize / sizeof(XUSER_READ_PROFILE_SETTING_RESULT);
		std::vector<XUSER_READ_PROFILE_SETTING_RESULT> results;
		results.resize(numResults);

		// Call XUserReadProfileSettings again, using the allocated buffer instead of NULL.
		error = XUserReadProfileSettings(0, user, 1, &osPreference, &resultsSize, &results.front(), 0);

		CRY_ASSERT_TRACE(error == ERROR_SUCCESS, ("Error %u reading profile settings", error));
		if(error == ERROR_SUCCESS)
		{
			switch(results[0].pSettings->dwSettingId)
			{
			case XPROFILE_GAMER_YAXIS_INVERSION:
				switch(results[0].pSettings->data.nData)
				{
				case XPROFILE_YAXIS_INVERSION_OFF: outResult = SUserProfileVariant(0L); break;
				case XPROFILE_YAXIS_INVERSION_ON: outResult = SUserProfileVariant(1L); break;
				}
				success = true;
				break;

			case XPROFILE_GAMER_CONTROL_SENSITIVITY:
				switch(results[0].pSettings->data.nData)
				{
				case XPROFILE_CONTROL_SENSITIVITY_LOW: outResult = SUserProfileVariant(0.0f); break;
				case XPROFILE_CONTROL_SENSITIVITY_MEDIUM: outResult = SUserProfileVariant(1.0f); break;
				case XPROFILE_CONTROL_SENSITIVITY_HIGH: outResult = SUserProfileVariant(2.0f); break;
				}
				success = true;
				break;

			case XPROFILE_GAMER_DIFFICULTY:
				switch(results[0].pSettings->data.nData)
				{
				case XPROFILE_GAMER_DIFFICULTY_EASY: outResult = SUserProfileVariant(0); break;
				case XPROFILE_GAMER_DIFFICULTY_NORMAL: outResult = SUserProfileVariant(1); break;
				case XPROFILE_GAMER_DIFFICULTY_HARD: outResult = SUserProfileVariant(2); break;
				}
				success = true;
				break;

			default:
				CRY_ASSERT_MESSAGE(false, "Missing entry in result for GetUserProfilePreference");
				break;
			}
		}
	}

	return success;
}

/*
--------------------- SUser -----------------------
*/
void CPlatformOS_Xenon::SUser::Reset()
{
	m_userIndex = Unknown_User;

	// Xenon will send us a user sign-in for each user signed in, we need to see that the user has changed.
	m_userXUID = 0;

	m_pendingRequests = 0;

	// Device selection
	m_saveDeviceID = XCONTENTDEVICE_ANY;
	memset(&m_overlapped, 0, sizeof(m_overlapped));

	memset(&m_saveGame, 0, sizeof(m_saveGame));
	m_saveGameOpenCount = 0;
	m_bWritingContent = false;

	m_fileMap.clear();
}

bool CPlatformOS_Xenon::SUser::MapFileName(CPlatformOS_Xenon* pOS, const char* fileName, TFileName& mappedFileName, bool bCreateMapping)
{
	// For map search ensure parity with lower case and back slashes
	string path = string(fileName).replace('/', '\\');

	// Remove the player name from the path or things can get lost if someone changes their XBOX profile name
	IPlatformOS::TUserName userName;
	pOS->UserGetName(m_userIndex, userName);
	const string userPart = string("\\") + userName.c_str() + "\\";
	size_t offset = path.find(userPart);
	if(offset != string::npos)
		path = path.Left(offset) + path.Right(path.length() - offset - userPart.length() + 1);

	string workingName;

	// Does it exist?
	FileMap::iterator it = m_fileMap.begin();
	for(FileMap::iterator itEnd = m_fileMap.end(); it != itEnd; ++it)
	{
		if(0 == it->first.compareNoCase(path))
		{
			mappedFileName = it->second;
			return true;
		}
	}

	if(bCreateMapping)
	{
		// Map all directory paths so FileExists works
		TFileName dummy;
		offset = path.find('\\');
		for(; offset != string::npos && path[offset+1] != '\0'; )
		{
			string p = path.substr(0, offset + 1); // include the trailing '\\' to indicate a directory
			MapFileName(pOS, p, dummy, bCreateMapping);

			offset = path.find('\\', offset + 1);
		}

		if(path[path.length()-1] == '\\') // directory?
		{
			workingName = path.c_str();
		}
		else
		{
			offset = path.rfind("\\");
			workingName = offset == string::npos ? path.c_str() : path.substr(offset + 1).c_str();

			// Scan all entries matching this filename and append a suffix if required to ensure the mapped file is unique
			// - Ensure pattern matching will still work through the FindFile interface
			string suffix;
			int nSuffix = -1;
			offset = workingName.rfind('.');
			size_t length = offset != string::npos ? offset : workingName.length();
			string mappedFileBase = workingName.substr(0, length);
			string mappedFileExt = offset == string::npos ? "" : workingName.substr(offset);

			// Ensure mappedFileBase + mappedFileExt will fit into FILE_MAX_NAME chars
			const size_t clampLength = FILE_MAX_NAME - s_uniqueNameDigits - SAVE_ROOT_NAME_LEN;
			if(mappedFileBase.length() + mappedFileExt.length() > clampLength)
			{
				mappedFileBase = mappedFileBase.substr(0, clampLength - mappedFileExt.length());
				workingName = mappedFileBase + mappedFileExt;
				length = mappedFileBase.length();
			}

			for(it = m_fileMap.begin(); it != m_fileMap.end(); ++it)
			{
				const string& mappedCompare = it->second;
				if(0 == mappedCompare.substr(0, length).compareNoCase(mappedFileBase))
				{
					offset = mappedCompare.rfind('.');
					string mappedCompareExt = offset == string::npos ? "" : mappedCompare.substr(offset);
					if(0 == mappedCompareExt.compareNoCase(mappedFileExt))
					{
						size_t suffixOffset = mappedCompare.find('_', length);
						if(suffixOffset == string::npos)
							suffixOffset = offset;

						size_t count = offset == string::npos ? string::npos : offset - suffixOffset;
						int n = 0;
						if(count && count != string::npos)
						{
							CRY_ASSERT(count > 1); // double check we don't have a dodgy state here
							suffix = mappedCompare.substr(suffixOffset, count);
							n = atoi(&suffix[1]); // if the text after the suffix underscore is
						}
						if(n > nSuffix)
						{
							nSuffix = n + 1;
							//suffix.Format("_%d", nSuffix);
							//workingName = mappedFileBase.c_str();
							//workingName = workingName + suffix.c_str() + mappedFileExt.c_str();
							workingName.Format("%s_%d%s", mappedFileBase.c_str(), nSuffix, mappedFileExt.c_str());
						}
					}
				}
			}
		}

		mappedFileName = workingName.c_str();
		m_fileMap[path] = mappedFileName;
	}

	return false;
}

IPlatformOS::EFileOperationCode CPlatformOS_Xenon::SUser::SaveFileMap(CPlatformOS_Xenon* os)
{
	const IPlatformOS::EFileOperationCode noError = IPlatformOS::eFOC_Success;

	IPlatformOS::ISaveWriterPtr writer = os->SaveGetWriter(s_fileMapName, m_userIndex);
	if(writer != NULL)
	{
		// Write the number of files
		size_t numFiles = m_fileMap.size();
		writer->AppendItem(numFiles);

		// Write each file mapping
		FileMap::iterator iter = m_fileMap.begin(), iterEnd = m_fileMap.end();
		for(; iter != iterEnd && writer->LastError() == noError; ++iter)
		{
			// Filename key
			size_t fileNameLen = iter->first.length();
			writer->AppendItem(fileNameLen);
			writer->AppendBytes(&iter->first[0], fileNameLen);

			// Mapped filename
			fileNameLen = iter->second.length();
			writer->AppendItem(fileNameLen);
			writer->AppendBytes(iter->second.c_str(), fileNameLen);
		}
		writer->Close();
		return writer->LastError();
	}
	return noError;
}

IPlatformOS::EFileOperationCode CPlatformOS_Xenon::SUser::LoadFileMap(CPlatformOS_Xenon* os)
{
	// Clear the file map in case the storage device was removed and a new device has been selected
	m_fileMap.clear();

	string fileNameKey;
	TFileName fileNameValue;

	// In order to determine if the file actually exists on the physical media, we need to create the mapping
	// for the OS file system to check the physical media for the file map file. (SaveGetReader doesn't create a mapping, SaveGetWriter does).
	// Clear the file map afterward because the file map may not have existed on the physical media.
	IPlatformOS::IFileFinderPtr fileMap = os->GetFileFinder();
	static_cast<CFileFinderXenon*>(fileMap.get())->MapFileName(m_userIndex, s_fileMapName, fileNameValue, true);
	IPlatformOS::ISaveReaderPtr reader = os->SaveGetReader(s_fileMapName, m_userIndex);
	m_fileMap.clear();

	if(reader == NULL)
		return eFOC_Failure;

	// Read the number of files
	size_t numFiles;
	reader->ReadItem(numFiles);

	std::vector<char> bytes;
	bytes.reserve(FILE_MAX_NAME);

	// Read each file mapping
	for(size_t i=0; i<numFiles; ++i)
	{
		// Filename key
		size_t fileNameLen;
		reader->ReadItem(fileNameLen);
		bytes.resize(fileNameLen + 1);
		reader->ReadBytes(&bytes[0], fileNameLen);
		bytes[fileNameLen] = 0;
		fileNameKey = &bytes[0];

		// Mapped filename
		reader->ReadItem(fileNameLen);
		CRY_ASSERT(fileNameLen <= FILE_MAX_NAME);
		bytes.resize(fileNameLen + 1);
		reader->ReadBytes(&bytes[0], fileNameLen);
		bytes[fileNameLen] = 0;
		fileNameValue = &bytes[0];

		CRY_ASSERT(m_fileMap.find(fileNameKey) == m_fileMap.end());
		m_fileMap[fileNameKey] = fileNameValue;
	}
	reader->Close();

	CRY_ASSERT_MESSAGE(m_fileMap.find(s_fileMapName) != m_fileMap.end() && m_fileMap.find(s_fileMapName)->second == s_fileMapName,
		"The file map should be IN FACT the file map and not a remapped file that just happens to have the same filename prefix");

#ifdef _DEBUG
	//LogFileMap();
#endif

	return reader->LastError();
}

void CPlatformOS_Xenon::SUser::LogFileMap()
{
	CryLog("Mounted save data for %s:", m_saveGame.szFileName);
	for(FileMap::iterator iter = m_fileMap.begin(); iter != m_fileMap.end(); ++iter)
		CryLog(" - %s -> %s", iter->first.c_str(), iter->second.c_str());
}

/*
--------------------- Storage -----------------------
*/

string CPlatformOS_Xenon::GetSaveRootName(unsigned int user)
{
	string saveRoot;
	saveRoot.Format("%s%d", SAVE_ROOT_NAME, user);
	return saveRoot;
}

bool CPlatformOS_Xenon::GetAnyReadySaveDevice(unsigned int user)
{
	CRY_ASSERT(user < MAX_USERS);
	if(!(user < MAX_USERS))
		return false;

	m_user[user].m_saveDeviceID = XCONTENTDEVICE_ANY;
	return MountSaveFile(user);
}

bool CPlatformOS_Xenon::UserSelectStorageDevice(unsigned int user)
{
	if(!UsePlatformSavingAPI())
		return false;

	CRY_ASSERT(user < MAX_USERS);
	if(!(user < MAX_USERS))
		return false;

	if(GetAnyReadySaveDevice(user))
	{
		SPlatformEvent event(user);
		event.m_eEventType = SPlatformEvent::eET_StorageMounted;
		NotifyListeners(event);
		return true;
	}

	// Explicitly set this flag, if the request is unavailable we need to retry next frame
	m_user[user].m_pendingRequests |= ePR_SaveDeviceSelector;

	// If the user signed in?
	if(m_user[user].m_signinState == eXUserSigninState_NotSignedIn)
	{
		// Will retry next frame
		return true;
	}

	// Wait for any currently pending overlapped before continuing
	if(WaitResetOverlapped(m_user[user].m_overlapped))
	{
		ULARGE_INTEGER requiredBytes;
		requiredBytes.QuadPart = XContentCalculateSize(
			0,				// approximate amount of space required - we're only enumerating
			0);				// number of directories required

		m_user[user].m_saveDeviceID = XCONTENTDEVICE_ANY;

		// XDK: If only one storage device is available, and it has enough space to store the data, this function
		//      will return the identifier of that device in pDeviceID without displaying the selector interface.

		DWORD showDeviceResult = XShowDeviceSelectorUI(
			user,
			XCONTENTTYPE_SAVEDGAME,			// only interested in save games
			XCONTENTFLAG_MANAGESTORAGE,		// Allows the gamer to select a device even if the device has no free space
			requiredBytes,
			&m_user[user].m_saveDeviceID,
			&m_user[user].m_overlapped);

		//CryLogAlways("showDeviceResult: %x\n", showDeviceResult);

		// Clear the pending flag if successful
		if(showDeviceResult == ERROR_IO_PENDING)
		{
			m_user[user].m_pendingRequests &= ~ePR_SaveDeviceSelector;
			m_user[user].m_pendingRequests |= ePR_SaveDevicePending;
		}
		else if(showDeviceResult != ERROR_ACCESS_DENIED)
		{
			// ERROR_ACCESS_DENIED means there is currently a MessageBox on screen which
			// is actually quite likely since we may have just signed in through the UI.
			CRY_ASSERT_TRACE(false, ("Unknown XShowDeviceSelectorUI result %u", showDeviceResult));
			return false;
		}
	}
	// return true since this is an async operation and the request is now complete or pending
	return true;
}

//XCONTENTDEVICEID CPlatformOS_Xenon::GetUserSaveDeviceID(unsigned int user) const
//{
//	CRY_ASSERT(user < MAX_USERS);
//	return m_saveDeviceID[user];
//}


IPlatformOS::ISaveReaderPtr CPlatformOS_Xenon::SaveGetReader(const char* fileName, unsigned int user)
{
	ISaveReaderPtr saveReader;

	if(user == Unknown_User && GetFirstSignedInUser() == Unknown_User)
	{
		user = 0;
		saveReader = ISaveReaderPtr(new CSaveReader_Memory(m_pMemoryFileSystem, fileName, user));
	}
	else
	{
		if (UsePlatformSavingAPI())
		{
			if(user == Unknown_User) user = UserFromFileName(fileName);
			if(user == Unknown_User) return ISaveReaderPtr(NULL);
			saveReader = ISaveReaderPtr(new CSaveReader_Xenon(fileName, user));
		}
		else
		{
			saveReader = ISaveReaderPtr(new CSaveReader_CryPak(fileName));
		}
	}

	if(!saveReader || saveReader->LastError() != eFOC_Success)
		return ISaveReaderPtr(NULL);

	return saveReader;
}

IPlatformOS::ISaveWriterPtr CPlatformOS_Xenon::SaveGetWriter(const char* fileName, unsigned int user)
{
	ISaveWriterPtr saveWriter;

	if(user == Unknown_User && GetFirstSignedInUser() == Unknown_User)
	{
		user = 0;
		saveWriter = ISaveWriterPtr(new CSaveWriter_Memory(m_pMemoryFileSystem, fileName, user));
	}
	else
	{
		if (UsePlatformSavingAPI())
		{
			if(user == Unknown_User) user = UserFromFileName(fileName);
			if(user == Unknown_User) return ISaveWriterPtr(NULL);
			saveWriter = ISaveWriterPtr(new CSaveWriter_Xenon(fileName, user));
		}
		else
		{
			saveWriter = ISaveWriterPtr(new CSaveWriter_CryPak(fileName));
		}
	}

	if(!saveWriter || saveWriter->LastError() != eFOC_Success)
		return ISaveWriterPtr(NULL);

	return saveWriter;
}

bool CPlatformOS_Xenon::MountSaveFile(unsigned int user)
{
	IPlatformOS::EFileOperationCode fileResult = IPlatformOS::eFOC_Failure;
	bool error = false;

	// Enumerate saves first
	HANDLE enumerator;
	DWORD size;

	DWORD result = XContentCreateEnumerator(
		user,
		m_user[user].m_saveDeviceID,
		XCONTENTTYPE_SAVEDGAME,
		XCONTENTFLAG_ENUM_EXCLUDECOMMON,
		1,	// enumerate one at a time
		&size,
		&enumerator);

	if(result == ERROR_SUCCESS)
	{
		if(size != 0)
		{
			assert(size % sizeof(XCONTENT_DATA) == 0);

			while(fileResult != IPlatformOS::eFOC_Success && !error && result == ERROR_SUCCESS)
			{
				DWORD numItemsEnumerated;
				memset(&m_user[user].m_saveGame, 0, sizeof(m_user[user].m_saveGame));
				result = XEnumerate(enumerator, &m_user[user].m_saveGame, sizeof(XCONTENT_DATA), &numItemsEnumerated, NULL);
				if(result == ERROR_SUCCESS)
				{
					DWORD disposition;
					DWORD licenseMask;

					string saveRootName = GetSaveRootName(user);

					result = XContentCreate(user,
						saveRootName,
						&m_user[user].m_saveGame,
						XCONTENTFLAG_OPENEXISTING,
						&disposition,
						&licenseMask,
						NULL);

					switch(result)
					{
					case ERROR_SUCCESS:
						{
							XContentClose(saveRootName, NULL); // only required on successful XContentCreate

							if(0 == wcscmp(m_user[user].m_saveGame.szDisplayName, SAVE_GAME_DISPLAY_NAME))
							{
								fileResult = m_user[user].LoadFileMap(this);
								if(fileResult == IPlatformOS::eFOC_Success)
								{
									// Store the device ID we're now using
									m_user[user].m_saveDeviceID = m_user[user].m_saveGame.DeviceID;
								}
								else
								{
									error = true;
									m_user[user].m_pendingRequests |= ePR_ShowSaveGameError;
								}
							}
						}
						break;

					case ERROR_FILE_CORRUPT:
						error = true;
						m_user[user].m_pendingRequests |= ePR_ShowSaveGameCorrupt;
						break;

					case ERROR_DISK_CORRUPT:
						error = true;
						m_user[user].m_pendingRequests |= ePR_ShowSaveDiskCorrupt;
						break;

					default:
						error = true;
						CRY_ASSERT_TRACE(false, ("Unhandled XContentCreate error %d", result));
						break;
					}
				}
			}
		}

		XCloseHandle(enumerator);
	}

	// If we have success here then we have opened an existing save

	// Otherwise attempt to create a new save on the selected storage device
	if(m_user[user].m_saveDeviceID != XCONTENTDEVICE_ANY && fileResult != IPlatformOS::eFOC_Success && !error)
	{
		DWORD disposition;
		DWORD licenseMask;

		IPlatformOS::TUserName userName;
		if(UserGetName(user, userName))
		{
			SYSTEMTIME sysTime;
			GetSystemTime(&sysTime);

			CRY_ASSERT(m_user[user].m_saveDeviceID != XCONTENTDEVICE_ANY);
			m_user[user].m_saveGame.dwContentType = XCONTENTTYPE_SAVEDGAME;
			m_user[user].m_saveGame.DeviceID = m_user[user].m_saveDeviceID;
			wcscpy_s(m_user[user].m_saveGame.szDisplayName, SAVE_GAME_DISPLAY_NAME);
			strcpy_s(m_user[user].m_saveGame.szFileName, userName.c_str());

			result = XContentCreate(user,
				GetSaveRootName(user),
				&m_user[user].m_saveGame,
				XCONTENTFLAG_CREATEALWAYS | XCONTENTFLAG_NOPROFILE_TRANSFER,
				&disposition,
				&licenseMask,
				NULL);

			if(result != ERROR_SUCCESS)
			{
				// On any result other than success clear the save game information before taking the next action
				memset(&m_user[user].m_saveGame, 0, sizeof(m_user[user].m_saveGame));
			}

			switch(result)
			{
			case ERROR_SUCCESS:
				result = XContentClose(GetSaveRootName(user), NULL);

				// We're good to go
				fileResult = m_user[user].SaveFileMap(this);

				m_user[user].m_pendingRequests |= ePR_SaveCreated;
				break;

			case ERROR_DISK_CORRUPT:
				m_user[user].m_pendingRequests |= ePR_ShowSaveDiskCorrupt;
				break;
			}
		}
	}

	return fileResult == eFOC_Success;
}

bool CPlatformOS_Xenon::OpenSaveGameContent(unsigned int user, bool bWrite)
{
	// Is the content available?
	if(m_user[user].m_saveGame.DeviceID == 0)
		return false;

	m_user[user].m_bWritingContent |= bWrite;

	CRY_ASSERT(m_user[user].m_saveGameOpenCount >= 0);
	if(m_user[user].m_saveGameOpenCount > 0)
	{
		++m_user[user].m_saveGameOpenCount;
		return true;
	}

	// Open it
	DWORD disposition, licenseMask;

	DWORD error = XContentCreate(user,
		GetSaveRootName(user),
		&m_user[user].m_saveGame,
		XCONTENTFLAG_OPENEXISTING,// | XCONTENTFLAG_NOPROFILE_TRANSFER,
		&disposition,
		&licenseMask,
		NULL);

	if(error == ERROR_SUCCESS)
	{
		++m_user[user].m_saveGameOpenCount;
		return true;
	}
	else if(error == ERROR_DEVICE_NOT_AVAILABLE || error == ERROR_DEVICE_NOT_CONNECTED)
	{
		// TCR # 051 - Games must notify the player if the in-use storage device is unavailable
		m_user[user].m_pendingRequests |= ePR_AskDeviceRemoved;
		return false;
	}

	return false;
}

bool CPlatformOS_Xenon::CloseSaveGameContent(unsigned int user)
{
	CRY_ASSERT(user < MAX_USERS);
	CRY_ASSERT(m_user[user].m_saveGameOpenCount > 0);

	//if(m_user[user].m_saveGameOpenCount <= 0)
	//	return false;

	// If there's been a write then write the file map table also
	if(m_user[user].m_saveGameOpenCount == 1 && m_user[user].m_bWritingContent)
	{
		m_user[user].SaveFileMap(this);
		m_user[user].m_bWritingContent = false;
	}

	if(--m_user[user].m_saveGameOpenCount > 0)
		return true;

	DWORD error = XContentClose(GetSaveRootName(user), NULL);
	return error == ERROR_SUCCESS;
}

bool CPlatformOS_Xenon::UsePlatformSavingAPI() const
{
#ifdef _RELEASE
	// Can't save to the DVD in release, and saving to the HDD is 20-life
	return true;
#else
	return g_cvars.sys_usePlatformSavingAPI != 0;
#endif
}

/* 
--------------------- Pending Requests -----------------------
*/

void CPlatformOS_Xenon::ProcessPendingRequests()
{
	// NOTE: Be careful to process ONLY ONE pending request at once.
	//       Continue the loop after calling a function in here to prevent overlapped IO from being trampled.

	for(DWORD user=0; user<MAX_USERS; ++user)
	{
		SUser& theUser = m_user[user];

		if(WaitResetOverlapped(theUser.m_overlapped)) // ensure overlapped IO is complete before processing any pending requests
		{
			// Process all state checking messages first

			if(theUser.m_pendingRequests & ePR_ConfirmNoSignIn)
			{
				CRY_ASSERT(user == 0);
				Pending_ConfirmNoSignIn(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_CheckAskConfirmNoSignIn)
			{
				CRY_ASSERT(user == 0);
				Pending_CheckAskConfirmNoSignIn(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_SaveDevicePending)
			{
				Pending_SaveDevice(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_ConfirmNoSaveDevice)
			{
				Pending_ConfirmNoSaveDevice(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_SaveGameError)
			{
				Pending_SaveGameError(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_SaveGameCorrupt)
			{
				Pending_SaveGameCorrupt(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_SaveDiskCorrupt)
			{
				Pending_SaveDiskCorrupt(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_DeleteSaveFile)
			{
				Pending_DeleteSaveFile(user);
				continue;
			}

			// Process pending messages that fire off UI last so we don't potentially get two firing at once and things out of order

			if(theUser.m_pendingRequests & ePR_SaveDeviceSelector)
			{
				UserSelectStorageDevice(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_AskConfirmNoSaveDevice)
			{
				Pending_AskConfirmNoSaveDevice(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_AskDeviceRemoved)
			{
				Pending_AskDeviceRemoved(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_DeviceRemoved)
			{
				Pending_DeviceRemoved(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_SaveCreated)
			{
				Pending_SaveCreated(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_ShowSaveGameError)
			{
				Pending_ShowSaveGameError(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_ShowSaveGameCorrupt)
			{
				Pending_ShowSaveGameCorrupt(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_SaveDiskCorrupt)
			{
				Pending_ShowSaveDiskCorrupt(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_ShowDeleteSaveFile)
			{
				Pending_ShowDeleteSaveFile(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_ShowDeleteSuccess)
			{
				Pending_ShowDeleteSuccess(user);
				continue;
			}

			if(theUser.m_pendingRequests & ePR_ShowDeleteError)
			{
				Pending_ShowDeleteError(user);
				continue;
			}

			if(m_user[user].m_pendingRequests & ePR_SignIn)
			{
				CRY_ASSERT(user == 0);
				Pending_DoSignIn();
				continue;
			}
		}
	}
}

void CPlatformOS_Xenon::Pending_DoSignIn()
{
	// Count the number of users requesting sign-in and ask again
	DWORD numUsersRequested = 0;
	for(DWORD user=0; user<MAX_USERS; ++user)
	{
		if(m_user[user].m_pendingRequests & ePR_SignIn)
		{
			++numUsersRequested;
			m_user[user].m_pendingRequests &= ~ePR_SignIn;
		}
	}
	UserDoSignIn(numUsersRequested);
}

void CPlatformOS_Xenon::Pending_CheckAskConfirmNoSignIn(DWORD user)
{
	// Check if we have any users signed in
	DWORD numUsersSignedIn = 0;
	for(DWORD i=0; i<MAX_USERS; ++i)
		if(m_user[i].m_signinState != eXUserSigninState_NotSignedIn)
			++numUsersSignedIn;

	if(numUsersSignedIn > 0)
	{
		// User(s) are signed in - we're done here
		for(DWORD i=0; i<MAX_USERS; ++i)
			m_user[i].m_pendingRequests &= ~ePR_CheckAskConfirmNoSignIn;
	}
	else
	{
		// TODO: localize all of these strings
		static const WCHAR* buttons[] = { L"Yes", L"No" };

		DWORD error = XShowMessageBoxUI(user,
			L"Sign in.",
			L"If no profile is signed in game data will not be saved and all progress will be lost.\n"
			L"\n"
			L"Are you sure you want to proceed without selecting a profile?",
			ARRAYSIZE(buttons),
			buttons,
			1, // No (retry)
			XMB_WARNINGICON,
			&m_user[user].m_messageBoxResult,
			&m_user[user].m_overlapped);

		if(error != ERROR_ACCESS_DENIED)
		{
			m_user[user].m_pendingRequests &= ~ePR_CheckAskConfirmNoSignIn;
			m_user[user].m_pendingRequests |= ePR_ConfirmNoSignIn;
		}
	}
}

void CPlatformOS_Xenon::Pending_ConfirmNoSignIn(DWORD user)
{
	m_user[user].m_pendingRequests &= ~ePR_ConfirmNoSignIn;

	// Ask the user again to select a device if required
	if(m_user[user].m_messageBoxResult.dwButtonPressed != 0) // can be No or Canceled with B
		UserDoSignIn(1); // TODO: for future reference this may need to come from a cached state
	else
		UserDoSignIn(0); // Tell the system that we're signing in but without a user (default profile)
}

void CPlatformOS_Xenon::Pending_SaveDevice(DWORD user)
{
	// Request may have been completed or cancelled, either way keep m_saveDeviceID[user] whatever it is.
	m_user[user].m_pendingRequests &= ~ePR_SaveDevicePending;
	// m_saveDeviceID[user] is now valid or empty

	// TCR # 050 - if cancelled we should notify the user that no game data will saved.
	if(m_user[user].m_saveDeviceID == XCONTENTDEVICE_ANY)
	{
		m_user[user].m_pendingRequests |= ePR_AskConfirmNoSaveDevice;
	}
	else
	{
		// We have a valid deviceID - create a new save here
		if(MountSaveFile(user))
		{
			SPlatformEvent event(user);
			event.m_eEventType = SPlatformEvent::eET_StorageMounted;
			NotifyListeners(event);
		}
		else
		{
			// Invalidate the device for this user if mounting didn't succeed
			m_user[user].m_saveDeviceID = XCONTENTDEVICE_ANY;
			// TODO: show error file dialog or similar at this point
		}
	}
}

void CPlatformOS_Xenon::Pending_AskConfirmNoSaveDevice(DWORD user)
{
	// TODO: localize all of these strings
	static const WCHAR* buttons[] = { L"Yes", L"No" };

	wstring text;
	UserGetNameW(user, text);
	text += L" - No Storage Device";

	DWORD error = XShowMessageBoxUI(user,
		text.c_str(),
		L"If no storage device is selected no game save will take place and all progress will be lost.\n"
		L"\n"
		L"Are you sure you want to proceed without saving?",
		ARRAYSIZE(buttons),
		buttons,
		1, // no/retry
		XMB_WARNINGICON,
		&m_user[user].m_messageBoxResult,
		&m_user[user].m_overlapped);

	if(error != ERROR_ACCESS_DENIED)
	{
		m_user[user].m_pendingRequests &= ~ePR_AskConfirmNoSaveDevice;
		m_user[user].m_pendingRequests |= ePR_ConfirmNoSaveDevice;
	}
}

void CPlatformOS_Xenon::Pending_ConfirmNoSaveDevice(DWORD user)
{
	m_user[user].m_pendingRequests &= ~ePR_ConfirmNoSaveDevice;

	// Ask the user again to select a device if required
	switch(m_user[user].m_messageBoxResult.dwButtonPressed) // can be No or Canceled with B
	{
	case 0:
		break;

	case XMB_CANCELID:
	default:
		UserSelectStorageDevice(user);
	}
}

void CPlatformOS_Xenon::Pending_AskDeviceRemoved(DWORD user)
{
	wstring text;
	UserGetNameW(user, text);
	text.append(L" - Device Removed");

	static const WCHAR* buttons[] = { L"Retry", L"Select Device", L"Continue without saving" };

	DWORD error = XShowMessageBoxUI(user,
		text.c_str(),
		L"Storage device has been removed.\n"
		L"\n"
		L"Please reinsert the storage device, select another device, or you may continue without saving.",
		ARRAYSIZE(buttons),
		buttons,
		0, // Retry
		XMB_ALERTICON,
		&m_user[user].m_messageBoxResult,
		&m_user[user].m_overlapped);

	if(error != ERROR_ACCESS_DENIED)
	{
		m_user[user].m_pendingRequests &= ~ePR_AskDeviceRemoved;
		m_user[user].m_pendingRequests |= ePR_DeviceRemoved;
	}
}

void CPlatformOS_Xenon::Pending_DeviceRemoved(DWORD user)
{
	m_user[user].m_pendingRequests &= ~ePR_DeviceRemoved;

	switch(m_user[user].m_messageBoxResult.dwButtonPressed)
	{
	case 0: // retry / cancel with B button
		break;

	case XMB_CANCELID:
	case 1: // select device
		UserSelectStorageDevice(user);
		break;

	case 2: // continue without saving
		break;
	}
	// Ask the user again to select a device if required
}

void CPlatformOS_Xenon::Pending_SaveCreated(DWORD user)
{
	static const WCHAR* buttons[] = { L"Continue" };

	wstring text;
	UserGetNameW(user, text);

	DWORD error = XShowMessageBoxUI(user,
		text.c_str(),
		L"Save game has been created.",
		ARRAYSIZE(buttons),
		buttons,
		0, // Continue
		XMB_NOICON,
		&m_user[user].m_messageBoxResult,
		&m_user[user].m_overlapped);

	if(error != ERROR_ACCESS_DENIED)
	{
		m_user[user].m_pendingRequests &= ~ePR_SaveCreated;
	}
}

void CPlatformOS_Xenon::Pending_ShowSaveGameError(DWORD user)
{
	wstring text;
	UserGetNameW(user, text);
	text.append(L" - Save Error");

	static const WCHAR* buttons[] = { L"Select new device", L"Delete invalid save", L"Continue without saving" };

	DWORD error = XShowMessageBoxUI(user,
		text.c_str(),
		L"The same game data on this device is invalid. You may:\n"
		L"\n"
		L"- Select another device,\n"
		L"\n"
		L"- Delete the invalid file and create a new save on this device,\n"
		L"\n"
		L"- Continue without saving - you will lose any unsaved data.",
		ARRAYSIZE(buttons),
		buttons,
		0, // Select new device
		XMB_ERRORICON,
		&m_user[user].m_messageBoxResult,
		&m_user[user].m_overlapped);

	if(error != ERROR_ACCESS_DENIED)
	{
		m_user[user].m_pendingRequests &= ~ePR_ShowSaveGameError;
		m_user[user].m_pendingRequests |= ePR_SaveGameError;
	}
}

void CPlatformOS_Xenon::Pending_SaveGameError(DWORD user)
{
	m_user[user].m_pendingRequests &= ~ePR_SaveGameError;

	// Ask the user again to select a device if required
	switch(m_user[user].m_messageBoxResult.dwButtonPressed)
	{
	case XMB_CANCELID:
	case 0:
		// Select new device
		UserSelectStorageDevice(user);
		break;

	case 1:
		// Delete invalid save
		m_user[user].m_pendingRequests |= ePR_ShowDeleteSaveFile;
		break;

	case 2:
		// Continue without saving
		break;
	}
}

void CPlatformOS_Xenon::Pending_ShowSaveGameCorrupt(DWORD user)
{
	wstring text;
	UserGetNameW(user, text);
	text.append(L" - Save Corrupt");

	static const WCHAR* buttons[] = { L"Select new device", L"Delete corrupt save", L"Continue without saving" };

	DWORD error = XShowMessageBoxUI(user,
		text.c_str(),
		L"The same game data on this device is corrupt. You may:\n"
		L"\n"
		L"- Select another device,\n"
		L"\n"
		L"- Delete the corrupt file and create a new save on this device,\n"
		L"\n"
		L"- Continue without saving - you will lose any unsaved data.",
		ARRAYSIZE(buttons),
		buttons,
		0, // Select new device
		XMB_ERRORICON,
		&m_user[user].m_messageBoxResult,
		&m_user[user].m_overlapped);

	if(error != ERROR_ACCESS_DENIED)
	{
		m_user[user].m_pendingRequests &= ~ePR_ShowSaveGameCorrupt;
		m_user[user].m_pendingRequests |= ePR_SaveGameCorrupt;
	}
}

void CPlatformOS_Xenon::Pending_SaveGameCorrupt(DWORD user)
{
	m_user[user].m_pendingRequests &= ~ePR_SaveGameCorrupt;

	// Ask the user again to select a device if required
	switch(m_user[user].m_messageBoxResult.dwButtonPressed)
	{
	case XMB_CANCELID:
	case 0:
		// Select new device
		UserSelectStorageDevice(user);
		break;

	case 1:
		// Delete corrupt save
		m_user[user].m_pendingRequests |= ePR_ShowDeleteSaveFile;
		break;

	case 2:
		// Continue without saving
		break;
	}
}

void CPlatformOS_Xenon::Pending_ShowSaveDiskCorrupt(DWORD user)
{
	wstring text;
	UserGetNameW(user, text);
	text.append(L" - Disk Corrupt");

	static const WCHAR* buttons[] = { L"Select Device", L"Continue without saving" };

	DWORD error = XShowMessageBoxUI(user,
		text.c_str(),
		L"The selected save data is corrupt.\n"
		L"\n"
		L"Please select another device\n"
		L"or\n"
		L"\n"
		L"Continue without saving - you will lose any progress made.",
		ARRAYSIZE(buttons),
		buttons,
		0, // Select Device
		XMB_ERRORICON,
		&m_user[user].m_messageBoxResult,
		&m_user[user].m_overlapped);

	if(error != ERROR_ACCESS_DENIED)
	{
		m_user[user].m_pendingRequests &= ~ePR_ShowSaveDiskCorrupt;
		m_user[user].m_pendingRequests |= ePR_SaveDiskCorrupt;
	}
}

void CPlatformOS_Xenon::Pending_SaveDiskCorrupt(DWORD user)
{
	m_user[user].m_pendingRequests &= ~ePR_SaveDiskCorrupt;

	// Ask the user again to select a device if required
	switch(m_user[user].m_messageBoxResult.dwButtonPressed) // can be Select Device / Continue without saving
	{
	case XMB_CANCELID:
	case 0:
		UserSelectStorageDevice(user);
		break;
	}
}

void CPlatformOS_Xenon::Pending_ShowDeleteSaveFile(DWORD user)
{
	// Check the save game associated with this user is valid and belongs to the user
	string saveRootName = GetSaveRootName(user);
	DWORD result = XContentCreate(user, saveRootName, &m_user[user].m_saveGame, XCONTENTFLAG_OPENEXISTING, NULL, NULL, NULL);
	if(result == ERROR_SUCCESS)
	{
		result = XContentClose(saveRootName, NULL);
		CRY_ASSERT(result == ERROR_SUCCESS);

		// Verify the content belongs to this user
		BOOL isCreator;
		result = XContentGetCreator(user, &m_user[user].m_saveGame, &isCreator, &m_user[user].m_userXUID, NULL);
		if(result == ERROR_SUCCESS && !isCreator)
		{
			m_user[user].m_pendingRequests &= ~ePR_ShowDeleteSaveFile;
			return;
		}
	}

	wstring text;
	UserGetNameW(user, text);
	text.append(L" - Delete ");
	if(result == ERROR_FILE_CORRUPT)
		text.append(L"corrupt ");
	else if(result == ERROR_FILE_INVALID)
		text.append(L"invalid ");
	text.append(L"save game");

	static const WCHAR* buttons[] = { L"Confirm Delete", L"Cancel" };

	DWORD error = XShowMessageBoxUI(user,
		text.c_str(),
		L"Deleted save games are completely irretrievable.\n\n"
		L"Are you absolutely sure you want to delete this save file?\n",
		ARRAYSIZE(buttons),
		buttons,
		1, // Cancel
		XMB_WARNINGICON,
		&m_user[user].m_messageBoxResult,
		&m_user[user].m_overlapped);

	if(error != ERROR_ACCESS_DENIED)
	{
		m_user[user].m_pendingRequests &= ~ePR_ShowDeleteSaveFile;
		m_user[user].m_pendingRequests |= ePR_DeleteSaveFile;
	}
}

void CPlatformOS_Xenon::Pending_DeleteSaveFile(DWORD user)
{
	m_user[user].m_pendingRequests &= ~ePR_DeleteSaveFile;

	switch(m_user[user].m_messageBoxResult.dwButtonPressed)
	{
	case XMB_CANCELID:
	case 1:
		UserSelectStorageDevice(user);
		break;

	case 0: // Confirm
		{
			// Nuke it - after triple checking that it belongs to this user and is still available
			string saveRootName = GetSaveRootName(user);
			DWORD result = XContentCreate(user, saveRootName, &m_user[user].m_saveGame, XCONTENTFLAG_OPENEXISTING, NULL, NULL, NULL);
			if(result == ERROR_FILE_CORRUPT)
			{
				result = ERROR_SUCCESS;
			}
			else if(result == ERROR_SUCCESS)
			{
				result = XContentClose(saveRootName, NULL);
				CRY_ASSERT(result == ERROR_SUCCESS);

				// Verify the content belongs to this user
				BOOL isCreator;
				result = XContentGetCreator(user, &m_user[user].m_saveGame, &isCreator, &m_user[user].m_userXUID, NULL);
				if(result == ERROR_SUCCESS && !isCreator)
				{
					m_user[user].m_pendingRequests &= ~ePR_ShowDeleteSaveFile;
					return;
				}
				result = ERROR_SUCCESS;
			}

			if(result == ERROR_SUCCESS)
			{
				result = XContentDelete(user, &m_user[user].m_saveGame, NULL);
				if(result == ERROR_SUCCESS)
				{
					m_user[user].m_pendingRequests |= ePR_ShowDeleteSuccess;
				}
				else
				{
					m_user[user].m_pendingRequests |= ePR_ShowDeleteError;
				}
			}
			break;
		}
	}
}

void CPlatformOS_Xenon::Pending_ShowDeleteSuccess(DWORD user)
{
	static const WCHAR* buttons[] = { L"Continue" };

	wstring text;
	UserGetNameW(user, text);

	DWORD error = XShowMessageBoxUI(user,
		text.c_str(),
		L"The save data was successfully deleted.",
		ARRAYSIZE(buttons),
		buttons,
		0, // Select device
		XMB_NOICON,
		&m_user[user].m_messageBoxResult,
		&m_user[user].m_overlapped);

	if(error != ERROR_ACCESS_DENIED)
	{
		m_user[user].m_pendingRequests &= ~ePR_ShowDeleteSuccess;
		UserSelectStorageDevice(user);
	}
}

void CPlatformOS_Xenon::Pending_ShowDeleteError(DWORD user)
{
	static const WCHAR* buttons[] = { L"Select another device" };

	wstring text;
	UserGetNameW(user, text);

	DWORD error = XShowMessageBoxUI(user,
		text.c_str(),
		L"There was an error deleting the save data.",
		ARRAYSIZE(buttons),
		buttons,
		0, // Select another device
		XMB_ERRORICON,
		&m_user[user].m_messageBoxResult,
		&m_user[user].m_overlapped);

	if(error != ERROR_ACCESS_DENIED)
	{
		m_user[user].m_pendingRequests &= ~ePR_ShowDeleteError;
		UserSelectStorageDevice(user);
	}
}

/* 
--------------------- File finder -----------------------
*/

static inline int64 makeint64(DWORD high, DWORD low)
{
	return (static_cast<int64>(high) << 32) | static_cast<int64>(low);
}


IPlatformOS::IFileFinderPtr CPlatformOS_Xenon::GetFileFinder()
{
	// Handle default user
	if(GetFirstSignedInUser() == Unknown_User)
		return IFileFinderPtr(new CFileFinderMemory(m_pMemoryFileSystem));
	if(UsePlatformSavingAPI())
		return IFileFinderPtr(new CFileFinderXenon(this));
	return IFileFinderPtr(new CFileFinderCryPak);
}

bool CPlatformOS_Xenon::CFileFinderXenon::MapFileName(unsigned int user, const char* fileName, TFileName& mappedFileName, bool bCreateMapping)
{
	CRY_ASSERT(user >=0 && user < MAX_USERS);
	bool existed = m_os->m_user[user].MapFileName(m_os, fileName, mappedFileName, bCreateMapping);
	mappedFileName = (m_os->GetSaveRootName(user) + ":\\" + mappedFileName.c_str()).c_str();
	return existed;
}

intptr_t CPlatformOS_Xenon::CFileFinderXenon::FindFirst(unsigned int userIndex, const char* filePattern, _finddata_t* fd)
{
	CRY_ASSERT(userIndex >=0 && userIndex < MAX_USERS);
	if(m_os->OpenSaveGameContent(userIndex, false))
	{
		m_user = userIndex;
		TFileName root;

		// Save m_fileMap before doing the search because MapFileName() will internally create
		// mappings to the directory structure of the input path.
		SUser::FileMap fileMapSave = m_os->m_user[userIndex].m_fileMap;
		MapFileName(userIndex, filePattern, root, true);
		m_os->m_user[userIndex].m_fileMap = fileMapSave;

		m_handle = FindFirstFileA(root, &m_findFileData);
		if(m_handle != INVALID_HANDLE_VALUE)
		{
			FillFindData(fd);
			return reinterpret_cast<intptr_t>(this); // casting to generic pointer sized int for passing an unknown handle around
		}
		m_os->CloseSaveGameContent(userIndex);
	}
	return -1;
}

int CPlatformOS_Xenon::CFileFinderXenon::FindNext(intptr_t handle, _finddata_t* fd)
{
	CRY_ASSERT(reinterpret_cast<CFileFinderXenon*>(handle) == this); // casting to generic pointer sized int for passing an unknown handle around
	BOOL result = FindNextFileA(m_handle, &m_findFileData);

	if(result)
		FillFindData(fd);

	return result ? 0 : -1;
}

int CPlatformOS_Xenon::CFileFinderXenon::FindClose(intptr_t handle)
{
	if(handle)
	{
		CRY_ASSERT(reinterpret_cast<CFileFinderXenon*>(handle) == this); // casting to generic pointer sized int for passing an unknown handle around
		if(m_handle != INVALID_HANDLE_VALUE)
		{
			BOOL result = ::FindClose(m_handle);
			m_handle = INVALID_HANDLE_VALUE;
			m_os->CloseSaveGameContent(m_user);
			return result ? 0 : -1;
		}
	}
	return 0;
}

void CPlatformOS_Xenon::CFileFinderXenon::FillFindData(_finddata_t* fd)
{
	// Find the filename in the user filemap and return the actual filename
	SUser::FileMap::const_iterator it = m_os->m_user[m_user].m_fileMap.begin();
	SUser::FileMap::const_iterator itEnd = m_os->m_user[m_user].m_fileMap.end();
	for(; it != itEnd; ++it)
	{
		if(0 == it->second.compareNoCase(m_findFileData.cFileName))
		{
			string filename = PathUtil::GetFile(it->first);
			strcpy_s(fd->name, filename);
			break;
		}
	}
	CRY_ASSERT(it != itEnd);

	fd->attrib = m_findFileData.dwFileAttributes;
	fd->size = m_findFileData.nFileSizeLow;
	fd->time_access = makeint64(m_findFileData.ftLastAccessTime.dwHighDateTime, m_findFileData.ftLastAccessTime.dwLowDateTime);
	fd->time_create = makeint64(m_findFileData.ftCreationTime.dwHighDateTime, m_findFileData.ftCreationTime.dwLowDateTime);
	fd->time_write  = makeint64(m_findFileData.ftLastWriteTime.dwHighDateTime, m_findFileData.ftLastWriteTime.dwLowDateTime);
}

IPlatformOS::IFileFinder::EFileState CPlatformOS_Xenon::CFileFinderXenon::FileExists(unsigned int user, const char* path)
{
	TFileName fileSystemPath;
	if(!path || !path[0])
		return eFS_NotExist;
	bool exists = MapFileName(user, path, fileSystemPath, false);
	if(!exists)
	{
		// Check for folders without trailing backslashes
		char lastChar = path[strlen(path)-1];
		if(lastChar != '/' && lastChar != '\\')
		{
			string p = path;
			p += "\\";
			exists = MapFileName(user, p, fileSystemPath, false);
		}
	}
	if(exists)
		return fileSystemPath[fileSystemPath.length()-1] == '\\' ? eFS_Directory : eFS_File;
	return eFS_NotExist;
}


/* 
--------------------- Miscellaneous -----------------------
*/

void CPlatformOS_Xenon::StringToWString(const string& str, wstring& wstr)
{
	size_t numChars = str.length() + 1; // must include null terminator
	std::vector<wchar_t> buffer;
	buffer.resize(numChars);
	int converted = MultiByteToWideChar(CP_UTF8, 0, str.c_str(), -1, &buffer[0], numChars);
	CRY_ASSERT(converted == numChars);
	wstr = &buffer[0];
}


static void flip()
{
	IRenderer* pRenderer = gEnv->pRenderer;
	pRenderer->BeginFrame();
	ColorF black( Col_Black );
	pRenderer->ClearBuffer(FRT_CLEAR | FRT_CLEAR_IMMEDIATE, &black);
	pRenderer->EndFrame();
}

IPlatformOS::EMsgBoxResult
CPlatformOS_Xenon::DebugMessageBox( const char* body, const char* title, unsigned int flags ) const
{
	enum { _TITLEMAX = 128, _TEXTMAX = 300 };
	const wchar_t* buttonText[eMsgBoxNumButtons] = {0};
	wchar_t uniTitle[_TITLEMAX];
	wchar_t uniBody[_TEXTMAX];

 	buttonText[eMsgBox_OK] = L"OK";
	buttonText[eMsgBox_Cancel] = L"Break";
	mbstowcs( uniTitle, title, _TITLEMAX );
	uniTitle[_TITLEMAX-1] = 0;
	mbstowcs( uniBody, body, _TEXTMAX );
	uniBody[_TEXTMAX-1] = 0;

	const unsigned numButtons = sizeof(buttonText) / sizeof(buttonText[0]);
	MESSAGEBOX_RESULT msgResult = {0};
	XOVERLAPPED overlapped = {0};
	int status = 0;

	for (int i = 0; i < 100; i++)
	{
		status = XShowMessageBoxUI( 0, uniTitle, uniBody, numButtons, buttonText,
																0, XMB_ALERTICON, &msgResult, &overlapped );
		if (status == ERROR_IO_PENDING)
		{
			break;
		}
		Sleep(16);
		flip();
	}
	if (status == ERROR_IO_PENDING)
	{
		while (!XHasOverlappedIoCompleted( &overlapped ))
		{
			Sleep(16);
			flip();
		}
		status = XGetOverlappedResult( &overlapped, NULL, TRUE );
	}
	else
	{
		OutputDebugString( "ERROR in XShowMessageBoxUI\n\n" );
		OutputDebugString( title );
		OutputDebugString( "\n-----------------------------------------------\n" );
		OutputDebugString( body );
		OutputDebugString( "\n" );
		CryWarning( VALIDATOR_MODULE_SYSTEM, VALIDATOR_ERROR, "ERROR in XShowMessageBoxUI %d\n\n%s\n\n%s\n", status, title, body );
	}
	return (EMsgBoxResult)msgResult.dwButtonPressed;
}

#endif //XENON
