////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   IPlatformOS_Xenon.h
//  Created:     18/12/2009 by Alex Weighell.
//  Description: Interface to the Platform OS
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef __PLATFORMOS_XENON_H_
#define __PLATFORMOS_XENON_H_

#ifdef XENON

#include "IPlatformOS.h"
#include <CryListenerSet.h>

struct IMemoryFileSystem;


class CPlatformOS_Xenon : public IPlatformOS, IPlatformOS::IPlatformListener
{
public:
	enum { FILE_MAX_NAME = XCONTENT_MAX_FILENAME_LENGTH };
	typedef CryFixedStringT<FILE_MAX_NAME> TFileName;

private:
	enum { MAX_USERS = 4 };

	// For asynchronous operations we need to queue up various requests until others are done.
	enum EPendingRequests
	{
		// Sign in
		ePR_SignIn										= BIT(0),			// request user sign in
		ePR_CheckAskConfirmNoSignIn		= BIT(1),			// throw confirm no profile sign-in message box - if required
		ePR_ConfirmNoSignIn						= BIT(2),			// message box being displayed to ask about no sign-in
		// Save device
		ePR_SaveDeviceSelector				= BIT(3),			// throw save device selector as soon as a user is signed in - don't set this manually - it is set in UserSelectStorageDevice()
		ePR_SaveDevicePending					= BIT(4),			// save device selector has been successfully requested, poll for completion
		ePR_AskConfirmNoSaveDevice		= BIT(5),			// request message box to be displayed to confirm no save device for this user
		ePR_ConfirmNoSaveDevice				= BIT(6),			// message box being displayed to confirm no save device for this user
		ePR_AskDeviceRemoved					= BIT(7),			// storage device has been removed - ask the player what to do
		ePR_DeviceRemoved							= BIT(8),			// message box being displayed to request action after device is removed
		ePR_SaveCreated								= BIT(9),			// display message box to confirm a save has been created
		// Error and corruption
		ePR_ShowSaveGameError					= BIT(10),		// show save game error message box
		ePR_SaveGameError							= BIT(11),		// act on game error message box
		ePR_ShowSaveGameCorrupt				= BIT(12),		// show save game corrupt message box
		ePR_SaveGameCorrupt						= BIT(13),		// act on save game corrupt message box
		ePR_ShowSaveDiskCorrupt				= BIT(14),		// show disk corrupt message box
		ePR_SaveDiskCorrupt						= BIT(15),		// act on disk corrupt message box
		ePR_ShowDeleteSaveFile				= BIT(16),		// show delete the current save associated selected with this user
		ePR_DeleteSaveFile						= BIT(17),		// delete the current save associated selected with this user
		ePR_ShowDeleteSuccess					= BIT(18),		// inform the user there the save file was deleted
		ePR_ShowDeleteError						= BIT(19),		// inform the user there was a problem deleting the save file
	};

	struct SUser
	{
		void Reset();
		IPlatformOS::EFileOperationCode SaveFileMap(CPlatformOS_Xenon* os);
		IPlatformOS::EFileOperationCode LoadFileMap(CPlatformOS_Xenon* os);

		// MapFileName
		// Description:
		//   Create an internal file mapping for safely saving files to a Xenon save file
		// See Also: CFileFinderXenon::MapFileName
		// Arguments:
		//   const char* fileName - raw engine file path
		//   TFileName& mappedFileName - output mapped Xenon file system filename
		//   bool bCreateMapping - true to actually create a mapping, false to check if a mapping exists for fileName
		// Return:
		//   bool - true if the mapping existed before the call
		bool MapFileName(CPlatformOS_Xenon* pOS, const char* fileName, TFileName& mappedFileName, bool bCreateMapping);	// return the save name (on physical media) for a filename

	private:
		void LogFileMap();

	public:
		unsigned int m_userIndex;			// sign in index

		// Device selection
		XOVERLAPPED m_overlapped; // overlapped IO may be used for more than just MessageBox UI functions
		XCONTENTDEVICEID m_saveDeviceID;
		MESSAGEBOX_RESULT m_messageBoxResult;

		// Bit field of EPendingRequests
		DWORD m_pendingRequests;

		// Sign-in state for all four users, maintained through the internal event listener
		XUSER_SIGNIN_STATE m_signinState;
		XUID m_userXUID;

		// Current save game or empty
		XCONTENT_DATA m_saveGame;
		int m_saveGameOpenCount;
		bool m_bWritingContent;

		// File map
		static const char* s_fileMapName;
		typedef std::map<string, TFileName> FileMap;	// lower case, original_filename -> saved_filename
		FileMap m_fileMap;										
	};

public:

	class CFileFinderXenon : public IPlatformOS::IFileFinder
	{
		CPlatformOS_Xenon* m_os;
		unsigned int m_user;
		HANDLE m_handle;
		WIN32_FIND_DATA m_findFileData;

	public:
		CFileFinderXenon(CPlatformOS_Xenon* os) : m_os(os), m_handle(INVALID_HANDLE_VALUE) {}
		~CFileFinderXenon() { FindClose(reinterpret_cast<intptr_t>(this)); }

		VIRTUAL EFileState		FileExists(unsigned int user, const char* path);
		VIRTUAL intptr_t			FindFirst(unsigned int userIndex, const char* filePattern, _finddata_t* fd);
		VIRTUAL int						FindNext(intptr_t handle, _finddata_t* fd);
		VIRTUAL int						FindClose(intptr_t handle);

		bool					MapFileName(unsigned int user, const char* fileName, TFileName& fileNameSave, bool bCreateMapping);

	private:
		void FillFindData(_finddata_t* fd);
	};

public:

	~CPlatformOS_Xenon();
	CPlatformOS_Xenon();

	// IPlatformOS
	// Called each frame to update the platform listener
	VIRTUAL void Tick(float realFrameTime);

	// Local user profile functions to check/initiate user sign in:
	// See IPlatformOS.h for documentation.
	VIRTUAL unsigned int	UserGetMaximumSignedInUsers() const;
	VIRTUAL bool					UserIsSignedIn(unsigned int user) const;
	VIRTUAL bool 					UserIsSignedIn(const IPlatformOS::TUserName& userName, unsigned int& outUserIndex) const;
	VIRTUAL bool					UserDoSignIn(unsigned int numUsersRequested);
	VIRTUAL unsigned int	UserGetPlayerIndex(const char* userName) const;
	VIRTUAL bool					UserGetName(unsigned int user, IPlatformOS::TUserName& outName) const;
	VIRTUAL bool					UserSelectStorageDevice(unsigned int user);
	VIRTUAL bool					GetUserProfilePreference(unsigned int user, IPlatformOS::EUserProfilePreference ePreference, SUserProfileVariant& outResult) const;

	VIRTUAL bool					UsePlatformSavingAPI() const;
	VIRTUAL IPlatformOS::ISaveReaderPtr SaveGetReader(const char* fileName, unsigned int user);
	VIRTUAL IPlatformOS::ISaveWriterPtr SaveGetWriter(const char* fileName, unsigned int user);
	VIRTUAL IPlatformOS::IFileFinderPtr GetFileFinder();

	VIRTUAL void					AddListener(IPlatformListener* pListener, const char* szName);
	VIRTUAL void					RemoveListener(IPlatformListener* pListener);
	VIRTUAL void					NotifyListeners(SPlatformEvent& event);
	// ~IPlatformOS

	bool OpenSaveGameContent(unsigned int user, bool bWrite);
	bool CloseSaveGameContent(unsigned int user);

protected:

	VIRTUAL IPlatformOS::EMsgBoxResult DebugMessageBox(const char* body, const char* title, unsigned int flags=0) const;
	// IPlatformOS::IPlatformListener
	VIRTUAL void OnPlatformEvent(const IPlatformOS::SPlatformEvent& _event);
	// ~IPlatformOS::IPlatformListener

private:

	bool GetAnyReadySaveDevice(unsigned int user);
	bool MountSaveFile(unsigned int user);
	string GetSaveRootName(unsigned int user);
	unsigned int UserFromFileName(const char* fileName);

	static void StringToWString(const string& str, wstring& wstr);

	// WaitResetOverlapped
	// Description:
	//   Checks the overlapped to determine if it is still in progress and resets it ready for next use if it isn't in use.
	// See Also: 
	// Arguments:
	//   XOVERLAPPED & overlapped - the overlapped structure to check and reset
	// Return:
	//   bool - false if the overlapped is still in use; true if it has been reset and ready for use
	static bool WaitResetOverlapped(XOVERLAPPED& overlapped);

	void ProcessPendingRequests();
	bool UserGetNameW(unsigned int user, wstring& userNameW);

	void Pending_DoSignIn();
	void Pending_CheckAskConfirmNoSignIn(DWORD user);
	void Pending_ConfirmNoSignIn(DWORD user);
	void Pending_SaveDevice(DWORD user);
	void Pending_AskConfirmNoSaveDevice(DWORD user);
	void Pending_ConfirmNoSaveDevice(DWORD user);
	void Pending_AskDeviceRemoved(DWORD user);
	void Pending_DeviceRemoved(DWORD user);
	void Pending_SaveCreated(DWORD user);
	void Pending_ShowSaveGameError(DWORD user);
	void Pending_ShowSaveGameCorrupt(DWORD user);
	void Pending_ShowSaveDiskCorrupt(DWORD user);
	void Pending_SaveGameError(DWORD user);
	void Pending_SaveGameCorrupt(DWORD user);
	void Pending_SaveDiskCorrupt(DWORD user);
	void Pending_ShowDeleteSaveFile(DWORD user);
	void Pending_DeleteSaveFile(DWORD user);
	void Pending_ShowDeleteSuccess(DWORD user);
	void Pending_ShowDeleteError(DWORD user);

private:
	bool m_bFirstSignin;

	// OS internal event listener
	HANDLE m_listener;
	CListenerSet<IPlatformOS::IPlatformListener*> m_listeners;

	SUser m_user[MAX_USERS];
	friend class CFileFinderXenon;

	// Error handling
	IPlatformOS::EFileOperationCode m_lastFileError;

	cryshared_ptr<IMemoryFileSystem> m_pMemoryFileSystem;
};

#endif
#endif
