////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   SaveReaderWriter_CryPak.cpp
//  Created:     15/02/2010 by Alex McCarthy.
//  Description: Implementation of the ISaveReader and ISaveWriter
//               interfaces using CryPak
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include <StdAfx.h>

#include "SaveReaderWriter_CryPak.h"

static const int INVALID_SEEK = -1;

int TranslateSeekMode(IPlatformOS::ISaveReader::ESeekMode mode)
{
	COMPILE_TIME_ASSERT(INVALID_SEEK != SEEK_SET);
	COMPILE_TIME_ASSERT(INVALID_SEEK != SEEK_CUR);
	COMPILE_TIME_ASSERT(INVALID_SEEK != SEEK_END);

	switch(mode)
	{
		case IPlatformOS::ISaveReader::ESM_BEGIN: return SEEK_SET;
		case IPlatformOS::ISaveReader::ESM_CURRENT: return SEEK_CUR;
		case IPlatformOS::ISaveReader::ESM_END: return SEEK_END;
		default:
			{
				CRY_ASSERT_TRACE(false, ("Unrecognized seek mode %i", static_cast<int>(mode)));
				return INVALID_SEEK;
			}
	}
}

////////////////////////////////////////////////////////////////////////////

CCryPakFile::CCryPakFile(const char* fileName, const char* szMode)
: m_pFile(gEnv->pCryPak->FOpen(fileName, szMode))
{
}

CCryPakFile::~CCryPakFile()
{
	CloseImpl();
}

IPlatformOS::EFileOperationCode CCryPakFile::CloseImpl()
{
	CRY_ASSERT(gEnv);
	CRY_ASSERT(gEnv->pCryPak);

	if (m_pFile && (gEnv->pCryPak->FClose(m_pFile) == 0))
	{
		return IPlatformOS::eFOC_Success;
	}
	m_eLastError = IPlatformOS::eFOC_Failure;
	return m_eLastError;
}

////////////////////////////////////////////////////////////////////////////

CSaveReader_CryPak::CSaveReader_CryPak(const char* fileName)
: CCryPakFile(fileName, "rbx") // x=don't cache full file
{
	m_eLastError = m_pFile == NULL ? IPlatformOS::eFOC_ErrorOpenRead : IPlatformOS::eFOC_Success;
}

IPlatformOS::EFileOperationCode CSaveReader_CryPak::Seek(long seek, ESeekMode mode)
{
	CRY_ASSERT(gEnv);
	CRY_ASSERT(gEnv->pCryPak);

	int translatedMode = TranslateSeekMode(mode);
	CRY_ASSERT(translatedMode != INVALID_SEEK);
	if (translatedMode == INVALID_SEEK)
	{
		return IPlatformOS::eFOC_Failure;
	}

	const bool bSuccess = (gEnv->pCryPak->FSeek(m_pFile, seek, translatedMode) == 0);
	return bSuccess ? IPlatformOS::eFOC_Success : IPlatformOS::eFOC_Failure;
}

IPlatformOS::EFileOperationCode CSaveReader_CryPak::GetFileCursor(long& fileCursor)
{
	CRY_ASSERT(gEnv);
	CRY_ASSERT(gEnv->pCryPak);

	fileCursor = gEnv->pCryPak->FTell(m_pFile);

	return IPlatformOS::eFOC_Success;
}

IPlatformOS::EFileOperationCode CSaveReader_CryPak::ReadBytes(void* data, size_t numBytes)
{
	CRY_ASSERT(gEnv);
	CRY_ASSERT(gEnv->pCryPak);

	size_t readBytes = gEnv->pCryPak->FReadRaw(data, 1, numBytes, m_pFile); // TODO: do we need to do an endian swap?
	
	if(numBytes != readBytes)
	{
		m_eLastError = IPlatformOS::eFOC_ErrorRead;
		return m_eLastError;
	}
	return IPlatformOS::eFOC_Success;
}

IPlatformOS::EFileOperationCode CSaveReader_CryPak::GetNumBytes(size_t& numBytes)
{
	CRY_ASSERT(gEnv);
	CRY_ASSERT(gEnv->pCryPak);

	numBytes = gEnv->pCryPak->FGetSize(m_pFile);

	return IPlatformOS::eFOC_Success;
}

////////////////////////////////////////////////////////////////////////////

CSaveWriter_CryPak::CSaveWriter_CryPak(const char* fileName)
: CCryPakFile(fileName, "wb")
{
	m_eLastError = m_pFile == NULL ? IPlatformOS::eFOC_ErrorOpenWrite : IPlatformOS::eFOC_Success;
}

IPlatformOS::EFileOperationCode CSaveWriter_CryPak::AppendBytes(const void* data, size_t length)
{
	CRY_ASSERT(gEnv);
	CRY_ASSERT(gEnv->pCryPak);

	if (m_pFile && length != 0)
	{
		if (gEnv->pCryPak->FWrite(data, 1, length, m_pFile) != length)
		{
			m_eLastError = IPlatformOS::eFOC_ErrorWrite;
			return m_eLastError;
		}
	}
	return IPlatformOS::eFOC_Success;
}
