////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 2010.
// -------------------------------------------------------------------------
//  File name:   SaveReaderWriter_Memory.cpp
//  Created:     30/4/2010 by Ian Masters.
//  Description: Simple in-memory file system for saving/loading files.
// -------------------------------------------------------------------------
//  History:
//  
//  TODO:
//  - Add transfer capability so memory saves can be copied to a real
//    storage device once one is available.
////////////////////////////////////////////////////////////////////////////

#include <StdAfx.h>
#include "SaveReaderWriter_Memory.h"


enum { CHUNK_SIZE = 128*1024 };
static int amatch(const char *str, const char *p);


////////////////////////////////////////////////////////////////////////////
// CMemoryFile
////////////////////////////////////////////////////////////////////////////

class CMemoryFile
{
	friend class CMemoryFileSystem;
	MemoryFileSystemPtr m_pFileSystem;

public:
	CMemoryFile(const char* fileName, unsigned int userIndex)
		: m_user(userIndex)
		, m_filePath(fileName)
		, m_filePtr(0)
	{
		m_filePath.replace('\\', '/');
		Truncate();
	}

	size_t GetLength() const { return m_fileBytes.size(); }

	void Truncate()
	{
		m_fileBytes.resize(0);
		m_filePtr = 0;
#if defined(PS3) || defined(LINUX)
		time_t unixtime;
		time(&unixtime);
		// Borrowed from UnixTimeToFileTime in LocalizedStringManager.cpp
		#define Int32x32To64(a, b) ((uint64)((uint64)(a)) * (uint64)((uint64)(b)))
		LONGLONG longlong = Int32x32To64(unixtime, 10000000) + 116444736000000000LL; 
		m_fileTime.dwLowDateTime = (DWORD)longlong;
		m_fileTime.dwHighDateTime = (DWORD)(longlong >> 32);
#else
		SYSTEMTIME sysTime;
		GetSystemTime(&sysTime);
		SystemTimeToFileTime(&sysTime, &m_fileTime);
#endif
	}

	bool operator < (const CMemoryFile& file) const
	{
		return m_user < file.m_user || strcmp(m_filePath.c_str(), file.m_filePath.c_str()) < 0;
	}

	const char* GetFile() const
	{
			size_t offset = m_filePath.rfind('/');
			if(offset != string::npos)
				return &m_filePath[offset + 1];
			return m_filePath.c_str();
	}

	static inline int64 makeint64(DWORD high, DWORD low)
	{
		return (static_cast<int64>(high) << 32) | static_cast<int64>(low);
	}

	void FillFindData(_finddata_t& fd)
	{
		fd.attrib = FILE_ATTRIBUTE_NORMAL;
		fd.time_access = fd.time_create = fd.time_write = makeint64(m_fileTime.dwHighDateTime, m_fileTime.dwLowDateTime);
		fd.size = m_fileBytes.size();
		strcpy_s(fd.name, GetFile());
	}

	bool operator == (const CMemoryFile& file) const
	{
		return m_user == file.m_user && !m_filePath.compareNoCase(file.m_filePath.c_str());
	}

	unsigned int m_user;
	string m_filePath;
	std::vector<char> m_fileBytes;
	size_t m_filePtr;
	FILETIME m_fileTime;
};


////////////////////////////////////////////////////////////////////////////
// CMemoryFileSystem
////////////////////////////////////////////////////////////////////////////

class CMemoryFileSystem : public IMemoryFileSystem
{
	typedef std::vector<MemoryFilePtr> FileList;
	FileList m_files;

	struct mem_finddata_t
	{
		FileList::const_iterator it;
		string filePattern;
	};

public:

	VIRTUAL void LinkFile(MemoryFilePtr file)
	{
		// If the file already exists, delete the old one and overwrite
		FileList::iterator it = std::find(m_files.begin(), m_files.end(), file);
		if(it != m_files.end())
			m_files.erase(it);
		m_files.push_back(file);
	}

	VIRTUAL bool UnlinkFile(MemoryFilePtr file)
	{
		return stl::find_and_erase(m_files, file);
	}

	VIRTUAL MemoryFilePtr GetFile(const char* fileName, unsigned int user) const
	{
		string fname(fileName);
		fname.replace('\\', '/');
		for(FileList::const_iterator it = m_files.begin(); it != m_files.end(); ++it)
		{
			if((*it)->m_user == user && !(*it)->m_filePath.compareNoCase(fname.c_str()))
			{
				return *it;
			}
		}
		return MemoryFilePtr(NULL);
	}

	intptr_t FindFirst(unsigned int userIndex, const char* filePattern, _finddata_t* fd)
	{
		assert(fd != NULL);
		mem_finddata_t* find = new mem_finddata_t;
		find->filePattern = filePattern;
		for(find->it = m_files.begin(); find->it != m_files.end(); ++find->it)
		{
			CMemoryFile* file = find->it->get();
			int match = amatch(file->m_filePath, filePattern);
			if(match)
			{
				file->FillFindData(*fd);
				return reinterpret_cast<intptr_t>(find);
			}
		}
		delete find;
		return -1;
	}

	int FindNext(intptr_t handle, _finddata_t* fd)
	{
		assert(fd != NULL);
		mem_finddata_t* find = reinterpret_cast<mem_finddata_t*>(handle);
		for(++find->it; find->it != m_files.end(); ++find->it)
		{
			CMemoryFile* file = find->it->get();
			int match = amatch(file->m_filePath, find->filePattern);
			if(match)
			{
				file->FillFindData(*fd);
				return 0;
			}
		}
		return -1;
	}

	int FindClose(intptr_t handle)
	{
		mem_finddata_t* find = reinterpret_cast<mem_finddata_t*>(handle);
		delete find;
		return 0;
	}
};

IMemoryFileSystem* IMemoryFileSystem::CreateFileSystem()
{
	return new CMemoryFileSystem;
}

////////////////////////////////////////////////////////////////////////////
// CSaveWriter_Memory
////////////////////////////////////////////////////////////////////////////

CSaveWriter_Memory::CSaveWriter_Memory(const MemoryFileSystemPtr& pFileSystem, const char* fileName, unsigned int userIndex)
: m_pFileSystem(pFileSystem)
, m_eLastError(IPlatformOS::eFOC_Success)
{
	m_file = pFileSystem->GetFile(fileName, userIndex);
	if(m_file != NULL)
		m_file->Truncate();
	else
	{
		m_file = MemoryFilePtr(new CMemoryFile(fileName, userIndex));
		m_file->m_fileBytes.reserve(CHUNK_SIZE);
		pFileSystem->LinkFile(m_file);
	}
#ifdef _DEBUG
	CryLog("[CSaveWriter_Memory] \"%s\" opened for write", fileName);
#endif
}

CSaveWriter_Memory::~CSaveWriter_Memory()
{
	std::vector<char>(m_file->m_fileBytes.begin(), m_file->m_fileBytes.end()).swap(m_file->m_fileBytes);
}

IPlatformOS::EFileOperationCode CSaveWriter_Memory::AppendBytes(const void* data, size_t length)
{
	size_t offset = m_file->m_fileBytes.size();
	if(offset + length > m_file->m_fileBytes.capacity())
		m_file->m_fileBytes.reserve(m_file->m_fileBytes.capacity() + CHUNK_SIZE);
	m_file->m_fileBytes.resize(offset + length);
	memcpy(&m_file->m_fileBytes[offset], data, length);
	m_file->m_filePtr = offset;
	return IPlatformOS::eFOC_Success;
}

////////////////////////////////////////////////////////////////////////////
// CSaveReader_Memory
////////////////////////////////////////////////////////////////////////////

CSaveReader_Memory::CSaveReader_Memory(const MemoryFileSystemPtr& pFileSystem, const char* fileName, unsigned int userIndex)
: m_pFileSystem(pFileSystem)
, m_eLastError(IPlatformOS::eFOC_Success)
{
	m_file = pFileSystem->GetFile(fileName, userIndex);
	if(m_file != NULL)
	{
		m_file->m_filePtr = 0;
#ifdef _DEBUG
		CryLog("[CSaveReader_Memory] \"%s\" opened for read", fileName);
#endif
	}
	else
		m_eLastError = IPlatformOS::eFOC_ErrorOpenRead;
}

IPlatformOS::EFileOperationCode CSaveReader_Memory::Seek(long seek, ESeekMode mode)
{
	long fileSize = static_cast<long>(m_file->m_fileBytes.size());

	switch(mode)
	{
	case ESM_BEGIN:
		break;

	case ESM_CURRENT:
		seek = static_cast<long>(m_file->m_filePtr) + seek;
		break;

	case ESM_END:
		seek = static_cast<long>(m_file->m_fileBytes.size()) + seek;
		break;

	default:
		assert(!"Invalid seek mode");
		return IPlatformOS::eFOC_Failure;
	}

	if(seek < 0 || seek > fileSize)
		return IPlatformOS::eFOC_Failure;

	m_file->m_filePtr = seek;
	return IPlatformOS::eFOC_Success;

}

IPlatformOS::EFileOperationCode CSaveReader_Memory::GetFileCursor(long& fileCursor)
{
	fileCursor = static_cast<long>(m_file->m_filePtr);
	return IPlatformOS::eFOC_Success;
}

IPlatformOS::EFileOperationCode CSaveReader_Memory::ReadBytes(void* data, size_t numBytes)
{
	assert(m_file->m_filePtr + numBytes <= m_file->m_fileBytes.size());
	long count = static_cast<long>(numBytes);
	count = min(count, m_file->m_fileBytes.size() - m_file->m_filePtr);
	if(count < numBytes)
	{
		m_eLastError = IPlatformOS::eFOC_ErrorRead;
		return m_eLastError;
	}
	memcpy(data, &m_file->m_fileBytes[m_file->m_filePtr], count);
	m_file->m_filePtr += count;
	return IPlatformOS::eFOC_Success;
}

IPlatformOS::EFileOperationCode CSaveReader_Memory::GetNumBytes(size_t& numBytes)
{
	numBytes = m_file->GetLength();
	return IPlatformOS::eFOC_Success;
}



intptr_t CFileFinderMemory::FindFirst(unsigned int userIndex, const char* filePattern, _finddata_t* fd)
{
	return m_pFileSystem->FindFirst(userIndex, filePattern, fd);
}

int CFileFinderMemory::FindNext(intptr_t handle, _finddata_t* fd)
{
	return m_pFileSystem->FindNext(handle, fd);
}

int CFileFinderMemory::FindClose(intptr_t handle)
{
	return m_pFileSystem->FindClose(handle);
}

IPlatformOS::IFileFinder::EFileState CFileFinderMemory::FileExists(unsigned int user, const char* path)
{
	MemoryFilePtr file = m_pFileSystem->GetFile(path, user);
	return file ? eFS_File : eFS_NotExist;
}






/*
 * robust glob pattern matcher
 * ozan s. yigit/dec 1994
 * public domain
 *
 * glob patterns:
 *	*	matches zero or more characters
 *	?	matches any single character
 *	[set]	matches any character in the set
 *	[^set]	matches any character NOT in the set
 *		where a set is a group of characters or ranges. a range
 *		is written as two characters separated with a hyphen: a-z denotes
 *		all characters between a to z inclusive.
 *	[-set]	set matches a literal hypen and any character in the set
 *	[]set]	matches a literal close bracket and any character in the set
 *
 *	char	matches itself except where char is '*' or '?' or '['
 *	\char	matches char, including any pattern character
 *
 * examples:
 *	a*c		ac abc abbc ...
 *	a?c		acc abc aXc ...
 *	a[a-z]c		aac abc acc ...
 *	a[-a-z]c	a-c aac abc ...
 *
 * $Log: glob.c,v $
 * Revision 1.3  1995/09/14  23:24:23  oz
 * removed boring test/main code.
 *
 * Revision 1.2  94/12/11  10:38:15  oz
 * cset code fixed. it is now robust and interprets all
 * variations of cset [i think] correctly, including [z-a] etc.
 * 
 * Revision 1.1  94/12/08  12:45:23  oz
 * Initial revision
 * 
 * 10/05/04 ian
 * Copied and slightly modified from http://www.cse.yorku.ca/~oz/glob.bun
 */

#undef NEGATE
#define NEGATE	'^'			/* std cset negation char */

static int amatch(const char *str, const char *p)
{
	int negate;
	int match;
	int c;

	while (*p) {
		if (!*str && *p != '*')
			return FALSE;

		switch (c = *p++) {

		case '*':
			while (*p == '*')
				p++;

			if (!*p)
				return TRUE;

			if (*p != '?' && *p != '[' && *p != '\\')
				while (*str && *p != *str)
					str++;

			while (*str) {
				if (amatch(str, p))
					return TRUE;
				str++;
			}
			return FALSE;

		case '?':
			if (*str)
				break;
			return FALSE;

		// set specification is inclusive, that is [a-z] is a, z and
		// everything in between. this means [z-a] may be interpreted
		// as a set that contains z, a and nothing in between.
 
		case '[':
			if (*p != NEGATE)
				negate = FALSE;
			else {
				negate = TRUE;
				p++;
			}

			match = FALSE;

			while (!match && (c = *p++)) {
				if (!*p)
					return FALSE;
				if (*p == '-') {	/* c-c */
					if (!*++p)
						return FALSE;
					if (*p != ']') {
						if (*str == c || *str == *p ||
						    (*str > c && *str < *p))
							match = TRUE;
					}
					else {		/* c-] */
						if (*str >= c)
							match = TRUE;
						break;
					}
				}
				else {			/* cc or c] */
					if (c == *str)
						match = TRUE;
					if (*p != ']') {
						if (*p == *str)
							match = TRUE;
					}
					else
						break;
				}
			}

			if (negate == match)
				return FALSE;

			// if there is a match, skip past the cset and continue on
			while (*p && *p != ']')
				p++;
			if (!*p++)	/* oops! */
				return FALSE;
			break;

		case '\\':
			if (*p)
				c = *p++;
		default:
			if (c != *str)
				return FALSE;
			break;

		}
		str++;
	}

	return !*str;
}
