////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   SaveReaderWriter_PS3.cpp
//  Created:     19/03/2010 by Stuart Merry.
//  Description: Implementation of the ISaveReader and ISaveWriter
//               interfaces using PS3 savedata API.
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include <StdAfx.h>

#ifdef PS3

#include "SaveReaderWriter_PS3.h"
#include "PlatformOS_PS3.h"
#include "../System.h"

////////////////////////////////////////////////////////////////////////////
class SReadWriteLock
{
public:
	SReadWriteLock(CellSync2Semaphore* pSemaphore) : m_pSemaphore(pSemaphore)
	{
		cellSync2SemaphoreAcquire(m_pSemaphore, 1, 0);
	}

	~SReadWriteLock()
	{
		cellSync2SemaphoreRelease(m_pSemaphore, 1, 0);
	}

private:
	CellSync2Semaphore* m_pSemaphore;
};

////////////////////////////////////////////////////////////////////////////

CPS3File::CPS3File(const char* fileName, EFileMode eFileMode)
: m_szRootName(fileName)
, m_eLastError(IPlatformOS::eFOC_Success)
, m_fileOpen(true)
{


}

IPlatformOS::EFileOperationCode CPS3File::CloseImpl()
{
	m_fileOpen = false;
	return(IPlatformOS::eFOC_Failure);
}

////////////////////////////////////////////////////////////////////////////

CSaveReader_PS3::CSaveReader_PS3(const char* fileName, DWORD userIndex)
: CPS3File(fileName, eFM_Write)
{
	// set up data write semaphore
	CellSync2SemaphoreAttribute semaphoreAttribs;
	int ret = cellSync2SemaphoreAttributeInitialize(&semaphoreAttribs);
	assert(ret == CELL_OK);

	m_saveDataReadSemaphore = reinterpret_cast<CellSync2Semaphore *>(CryModuleMemalign(CELL_SYNC2_SEMAPHORE_SIZE, CELL_SYNC2_SEMAPHORE_ALIGN));
	ret = cellSync2SemaphoreInitialize(m_saveDataReadSemaphore, 0, 0, &semaphoreAttribs);
	assert(ret == CELL_OK);
}

CSaveReader_PS3::~CSaveReader_PS3()
{
	CryModuleMemalignFree(m_saveDataReadSemaphore);
}
////////////////////////////////////////////////////////////////////////////

CSaveWriter_PS3::CSaveWriter_PS3(const char* fileName)
: CPS3File(fileName, eFM_Write)
, m_dataBuffer(0)
, m_dataStart(0)
, m_dataEnd(0)
, m_dataSize(0)
, m_validDataSize(0)
, m_saveDataWriteSemaphore(0)
{
	m_dataBuffer = new byte[DATA_BUFFER_MAX_SIZE];
	assert(m_dataBuffer != 0);

	// set up data write semaphore
	CellSync2SemaphoreAttribute semaphoreAttribs;
	semaphoreAttribs.maxWaiters = 2;
	int ret = cellSync2SemaphoreAttributeInitialize(&semaphoreAttribs);
	assert(ret == CELL_OK);

	m_saveDataWriteSemaphore = reinterpret_cast<CellSync2Semaphore *>(CryModuleMemalign(CELL_SYNC2_SEMAPHORE_SIZE, CELL_SYNC2_SEMAPHORE_ALIGN));
	ret = cellSync2SemaphoreInitialize(m_saveDataWriteSemaphore, 0, 0, &semaphoreAttribs);
	assert(ret == CELL_OK);
}

CSaveWriter_PS3::~CSaveWriter_PS3()
{
	CryModuleMemalignFree(m_saveDataWriteSemaphore);
}

IPlatformOS::EFileOperationCode CSaveWriter_PS3::AppendBytes(const void* data, size_t length)
{
	size_t currentLength = length;
	size_t readOffset = 0;
	const byte* byteData = reinterpret_cast<const byte*>(data);

	CPlatformOS_PS3* platformOS = static_cast<CPlatformOS_PS3*>(GetISystem()->GetPlatformOS());

	IPlatformOS::EFileOperationCode result = IPlatformOS::eFOC_Success;

	while(currentLength > 0 && result == IPlatformOS::eFOC_Success)
	{
		const byte* readPtr = &(byteData[readOffset]);

		int lengthWritten = 0;
		result = WriteData(readPtr, currentLength, lengthWritten);

		currentLength -= lengthWritten;
		readOffset += lengthWritten;
	}

	return result;
}

IPlatformOS::EFileOperationCode CSaveWriter_PS3::WriteData(const byte* dataPtr, const int dataLength, int& lengthWritten)
{
	SReadWriteLock lock(m_saveDataWriteSemaphore);

	int iDataLength = dataLength;
	iDataLength = CLAMP(iDataLength, iDataLength, DATA_BUFFER_MAX_SIZE-1);

	//don't lap the reader
	bool bLappingReader = (m_dataEnd < m_dataStart && (m_dataEnd + iDataLength > m_dataStart));
	//assert(!bLappingReader);
	if(bLappingReader || m_validDataSize + dataLength >= DATA_BUFFER_MAX_SIZE) 
	{
		lengthWritten = 0;
		return IPlatformOS::eFOC_Success;
	}

	int iDataEnd = m_dataEnd;
	//this is a cyclic buffer, when filled, start from 0 again
	if(iDataEnd + iDataLength >= DATA_BUFFER_MAX_SIZE)
	{
		if(m_dataStart == 0) //extreme case
		{
			lengthWritten = 0;
			return IPlatformOS::eFOC_Success;
		}
		else
		{
			iDataLength = DATA_BUFFER_MAX_SIZE - iDataEnd;
			m_dataEnd = 0;
		}
	}
	else //else increment write index
	{
		m_dataEnd += iDataLength;
	}

	byte * pWritePtr = &(m_dataBuffer[iDataEnd]);
	m_validDataSize += iDataLength;
	lengthWritten = iDataLength;

	memcpy(pWritePtr, dataPtr, iDataLength);

	assert(m_validDataSize >= 0);
	assert(m_validDataSize < DATA_BUFFER_MAX_SIZE);

	return IPlatformOS::eFOC_Success;
}

IPlatformOS::EFileOperationCode CSaveWriter_PS3::ReadData(byte* dataPtr, const int dataLength, int& lengthRead)
{
	SReadWriteLock lock(m_saveDataWriteSemaphore);

	//return if there is no data to read
	if(m_validDataSize == 0)
	{
		lengthRead = 0;
		return IPlatformOS::eFOC_Failure;
	}

	byte *pReadPtr = &(m_dataBuffer[m_dataStart]);
	//make sure to read only data that has been written before
	int iReadLength = dataLength;
	if(m_dataEnd >= m_dataStart)
	{
		iReadLength = MIN(iReadLength, m_dataEnd - m_dataStart);

		//increment reading index
		assert(!(m_dataStart < m_dataEnd && (m_dataStart + iReadLength > m_dataEnd)));
		m_dataStart += iReadLength;
	}
	else if(m_dataStart + iReadLength >= DATA_BUFFER_MAX_SIZE)
	{
		iReadLength = DATA_BUFFER_MAX_SIZE - m_dataStart;
		m_dataStart = 0;
	}
	else
		m_dataStart += iReadLength;

	memcpy(dataPtr, pReadPtr, iReadLength);

	m_validDataSize -= iReadLength;
	assert(m_validDataSize >= 0);

	int iDataSize = m_dataSize + iReadLength;
	lengthRead = iReadLength;
	m_dataSize = iDataSize;

	return IPlatformOS::eFOC_Success;
}


#endif
