////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   SaveReaderWriter_Xenon.h
//  Created:     17/02/2010 by Alex McCarthy.
//  Description: Implementation of the ISaveReader and ISaveWriter
//               interfaces using Xenon API calls
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include <StdAfx.h>

#ifdef XENON

#include "SaveReaderWriter_Xenon.h"
#include "PlatformOS_Xenon.h"
#include "../System.h"
#include <xbox.h>


static const int INVALID_SEEK = -1;
static const DWORD INVALID_OPEN_USER = ~0;

DWORD TranslateSeekModeXenon(IPlatformOS::ISaveReader::ESeekMode mode)
{
	COMPILE_TIME_ASSERT(INVALID_SEEK != SEEK_SET);
	COMPILE_TIME_ASSERT(INVALID_SEEK != SEEK_CUR);
	COMPILE_TIME_ASSERT(INVALID_SEEK != SEEK_END);

	switch(mode)
	{
		case IPlatformOS::ISaveReader::ESM_BEGIN: return SEEK_SET;
		case IPlatformOS::ISaveReader::ESM_CURRENT: return SEEK_CUR;
		case IPlatformOS::ISaveReader::ESM_END: return SEEK_END;
		default:
			{
				CRY_ASSERT_TRACE(false, ("Unrecognized seek mode %i", static_cast<int>(mode)));
				return INVALID_SEEK;
			}
	}
}

////////////////////////////////////////////////////////////////////////////

CXenonFile::CXenonFile(const char* fileName, DWORD userIndex, EFileMode eFileMode)
: m_hFile(INVALID_HANDLE_VALUE)
, m_contentOpenUser(INVALID_OPEN_USER)
, m_eLastError(IPlatformOS::eFOC_Success)
{
	assert(eFileMode == eFM_Read || eFileMode == eFM_Write);

	CPlatformOS_Xenon* platformOS = static_cast<CPlatformOS_Xenon*>(GetISystem()->GetPlatformOS());

	if(!platformOS->OpenSaveGameContent(userIndex, eFileMode == eFM_Write))
	{
		m_eLastError = eFileMode == eFM_Read ?
			IPlatformOS::eFOC_ErrorOpenRead : 
			IPlatformOS::eFOC_ErrorOpenWrite;

		CryLogAlways("Error opening save game content for %s", eFM_Read ? "reading" : "writing");
		return;
	}
	m_contentOpenUser = userIndex;

	DWORD fileError = ERROR_SUCCESS;

	CPlatformOS_Xenon::TFileName path;
	IPlatformOS::IFileFinderPtr findFinder = platformOS->GetFileFinder();
	bool existed = static_cast<CPlatformOS_Xenon::CFileFinderXenon*>(findFinder.get())->MapFileName(userIndex, fileName, path, eFileMode == eFM_Write);
	if(eFileMode == eFM_Read && !existed)
	{
		fileError = ERROR_FILE_NOT_FOUND;
	}
	else
	{
		if(eFileMode == eFM_Read)
		{
			m_hFile = CreateFile(path.c_str(), GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
			fileError = GetLastError();
		}
		else
		{
			m_hFile = CreateFile(path.c_str(), GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
			fileError = GetLastError();

			// If the path isn't setup yet attempt to create all directories back to root
			if(m_hFile == INVALID_HANDLE_VALUE && fileError == ERROR_PATH_NOT_FOUND)
			{
				size_t offset = path.find('\\');
				for(;;)
				{
					offset = path.find('\\', offset + 1);
					if(offset == string::npos) // reached the end of the path, try to create the file again
					{
						m_hFile = CreateFile(path.c_str(), GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
						fileError = GetLastError();
						break;
					}
					string p = path.substr(0, offset);
					if(!CreateDirectory(p.c_str(), NULL))
					{
						m_eLastError = IPlatformOS::eFOC_Failure;
						break;
					}
				}
			}
		}
	}
	if (m_hFile == INVALID_HANDLE_VALUE)
	{
		if(m_eLastError == IPlatformOS::eFOC_Success)
			m_eLastError = eFileMode == eFM_Read ? IPlatformOS::eFOC_ErrorOpenRead : IPlatformOS::eFOC_ErrorOpenWrite;
		CryWarning(VALIDATOR_MODULE_SYSTEM, VALIDATOR_WARNING, "Error %i opening file %s (%s)", fileError, fileName, path.c_str());
	}
#ifdef _DEBUG
	else
	{
		CryLogAlways("[CXenonFile] \"%s\" (%s) opened for %s", fileName, path.c_str(), eFileMode == eFM_Read ? "read" : "write");
	}
#endif
}

CXenonFile::~CXenonFile()
{
	CloseImpl();
}

IPlatformOS::EFileOperationCode CXenonFile::CloseImpl()
{
	if (m_hFile != INVALID_HANDLE_VALUE)
	{
		if (CloseHandle(m_hFile) == 0) // returns true on success
		{
			m_eLastError = IPlatformOS::eFOC_Failure;
		}
		m_hFile = INVALID_HANDLE_VALUE;
	}

	if(m_contentOpenUser != INVALID_OPEN_USER)
	{
		CPlatformOS_Xenon* platformOS = static_cast<CPlatformOS_Xenon*>(GetISystem()->GetPlatformOS());
		platformOS->CloseSaveGameContent(m_contentOpenUser);
	}

	if(m_eLastError != IPlatformOS::eFOC_Success)
	{
		IPlatformOS::SPlatformEvent event(m_contentOpenUser);
		event.m_eEventType = IPlatformOS::SPlatformEvent::eET_FileError;
		event.m_uParams.m_fileError.m_errorType = m_eLastError;
		GetISystem()->GetPlatformOS()->NotifyListeners(event);
	}

	m_contentOpenUser = INVALID_OPEN_USER;
	return m_eLastError;
}

////////////////////////////////////////////////////////////////////////////

CSaveReader_Xenon::CSaveReader_Xenon(const char* fileName, DWORD userIndex)
: CXenonFile(fileName, userIndex, eFM_Read)
{
}

IPlatformOS::EFileOperationCode CSaveReader_Xenon::Seek(long seek, ESeekMode mode)
{
	DWORD moveMethod = TranslateSeekModeXenon(mode);
	DWORD newPointer = SetFilePointer(m_hFile, seek, NULL, moveMethod);
	if(newPointer == INVALID_SET_FILE_POINTER)
	{
		return IPlatformOS::eFOC_Failure;
	}
	return IPlatformOS::eFOC_Success;
}

IPlatformOS::EFileOperationCode CSaveReader_Xenon::GetFileCursor(long& fileCursor)
{
	DWORD pos = SetFilePointer(m_hFile, 0, NULL, FILE_CURRENT);
	if(pos != INVALID_SET_FILE_POINTER)
	{
		fileCursor = static_cast<long>(pos);
		return IPlatformOS::eFOC_Success;
	}
	return IPlatformOS::eFOC_Failure;
}

IPlatformOS::EFileOperationCode CSaveReader_Xenon::ReadBytes(void* data, size_t numBytes)
{	
	DWORD numRead;
	BOOL success = ReadFile(m_hFile, data, numBytes, &numRead, NULL);
	return success && numRead == numBytes ? IPlatformOS::eFOC_Success : IPlatformOS::eFOC_ErrorRead;
}

IPlatformOS::EFileOperationCode CSaveReader_Xenon::GetNumBytes(size_t& numBytes)
{
	DWORD fileSize = GetFileSize(m_hFile, NULL);
	if(fileSize == -1) return IPlatformOS::eFOC_Failure;
	numBytes = fileSize;
	return IPlatformOS::eFOC_Success;
}

////////////////////////////////////////////////////////////////////////////

CSaveWriter_Xenon::CSaveWriter_Xenon(const char* fileName, DWORD userIndex)
: CXenonFile(fileName, userIndex, eFM_Write)
{
}

IPlatformOS::EFileOperationCode CSaveWriter_Xenon::AppendBytes(const void* data, size_t length)
{
	if(m_eLastError != IPlatformOS::eFOC_Success)
		return m_eLastError;

	DWORD bytesWritten = 0;
	if (WriteFile(m_hFile, data, length, &bytesWritten, NULL) == 0) // returns true on success
	{
		m_eLastError = IPlatformOS::eFOC_ErrorWrite;
	}

	if (bytesWritten != length)
	{
		m_eLastError = IPlatformOS::eFOC_ErrorWrite;
	}
	return m_eLastError;
}

#endif //XENON
