////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   SysCallThread_PS3.cpp
//  Created:     22/03/2010 by Alex Weighell.
//  Description:
//  	This is a shared thread for PS3 operating system utilities that
//  	need to be on their own thread:
//
//   	Trophies, savedata, gamedata etc.
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#if defined(PS3)

#include <StdAfx.h>

#include "SysCallThread_PS3.h"

#include <sys/timer.h>
#include <sys/event.h>

#define SYS_THREAD_PRIO		    	(1200)
#define SYS_THREAD_STACK_SIZE	  (16*1024)
#define SYS_EVENT_QUEUE_SIZE	 	(32)

// Some debugging support
#ifdef _DEBUG
#define SYSTHREAD_DEBUGGING
#endif

#ifdef SYSTHREAD_DEBUGGING
#define DebugAssertMsg(condition, format) do \
		if (!(condition)) { \
			CryFatalError format ; \
			CryFatalError("\n"); \
			DebugBreak(); \
		} \
	while(0)
#else
#define DebugAssertMsg(condition, format)
#endif

void CSysCallThread::SysProcessingThread(uint64 arg)
{
	int		    				status;
	sys_event_t	  		event;
	bool 							keepGoing = true;
	sys_event_queue_t	queue;

	{
		// Do not access pState below due to being deleted
		CSysCallThread* pSysThread = reinterpret_cast<CSysCallThread*>(arg);

		queue = pSysThread->m_queue;
		pSysThread->m_threadHasStarted = true;
	}

	do
	{
		status = sys_event_queue_receive(queue, &event, SYS_NO_TIMEOUT);

		// Recieved an event (we have work to do)
		printf("receiver: source = %llx, data1 = %llx, data2 = %llx, data3 = %llx : %d\n", event.source, event.data1, event.data2, event.data3, status);

		Handler callback = reinterpret_cast<CSysCallThread::Handler>(event.source);

		if (callback)
		{
			callback(event.data1, event.data2, event.data3);
		}
		else
		{
			keepGoing = false; // NULL callback: thread terminate signal
		}
	}
	while (keepGoing);

	sys_ppu_thread_exit(0);
}

CSysCallThread::CSysCallThread()
{
	m_threadHasStarted = false;

	int 												status;
	sys_event_queue_attribute_t	queueAttr;

	// Create an event queue for doing stuff

	sys_event_queue_attribute_initialize(queueAttr);
	status = sys_event_queue_create(&m_queue, &queueAttr, SYS_EVENT_QUEUE_LOCAL, SYS_EVENT_QUEUE_SIZE);
	DebugAssertMsg(status==CELL_OK,("sys_event_queue_create() failed in osSysThreadInitialise %08x",status));

	// Spawn a thread. it will only wake when it has work to do

	status = sys_ppu_thread_create(&m_thread, SysProcessingThread, (uintptr_t)this, SYS_THREAD_PRIO, SYS_THREAD_STACK_SIZE, 0, "CrySysCallThread");
	DebugAssertMsg(status==CELL_OK,("sys_ppu_thread_create() failed in CSysCallThread::CSysCallThread %08x",status));

	m_terminateEvent = RegisterHandler( NULL );
}

CSysCallThread::~CSysCallThread()
{
	while (!m_threadHasStarted)
	{
		sys_timer_usleep(100);
	}
	QueueEvent( m_terminateEvent, 0, 0, 0 );
}

void CSysCallThread::QueueEvent(CSysCallThread::TEvent event, uint64 eventType, uint64 data1, uint64 data2)
{
	int status;

	status = sys_event_port_send(event, eventType, data1, data2);
	DebugAssertMsg(status==CELL_OK, ("Error in osSysThreadQueueEvent - message will be lost : %08x : event = %d", status, event));
}

CSysCallThread::TEvent CSysCallThread::RegisterHandler(CSysCallThread::Handler callback)
{
	int 							status;
	sys_event_port_t 	event;

	// Create an event port (this is used for signalling the termination of the thread
	status = sys_event_port_create(&event, SYS_EVENT_PORT_LOCAL, (uintptr_t)callback);
	DebugAssertMsg(status==CELL_OK,("sys_event_port_create() failed in osSysThreadRegisterHandler: %08x : callback = %p",status,callback));

	status = sys_event_port_connect_local(event, m_queue);
	DebugAssertMsg(status==CELL_OK,("sys_event_port_connect_local() failed in osSysThreadRegisterHandler: %08x : event = %08x",status,event));

	return event;
}

#endif // defined(PS3)

