////////////////////////////////////////////////////////////////////////////
//
//  Crytek Source File.
//  Copyright (C), Crytek Studios, 2009.
// -------------------------------------------------------------------------
//  File name:   ProtocolBuilder.h
//  Version:     v1.00
//  Created:     13/08/2009 by Younggi Lim
//  Compilers:   Visual Studio.NET
//  Description: make a simple protocol based on key/value string
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////
#ifndef __protocolbuilder_h__
#define __protocolbuilder_h__

#pragma once

#ifdef USING_LICENSE_PROTECTION

#include <map>

class DataTypeConvertor
{
public:
	template<typename T>
	static void ToNumeric(const char* text, T& numeric) 
	{}

	template<>
	static void ToNumeric(const char* text, string& numeric) 
	{ 
		numeric = text; 
	}

	template<>
	static void ToNumeric(const char* text, int& numeric) 
	{ 
		numeric = atoi(text); 
	}

	template<>
	static void ToNumeric(const char* text, char& numeric) 
	{ 
		numeric = atoi(text); 
	}

	template<>
	static void ToNumeric(const char* text, unsigned char& numeric) 
	{ 
		numeric = atoi(text);	
	}

	template<>
	static void ToNumeric(const char* text, short& numeric) 
	{ 
		numeric = atoi(text); 
	}

	template<>
	static void ToNumeric(const char* text, unsigned short& numeric) 
	{ 
		numeric = atoi(text);	
	}

	template<>
	static void ToNumeric(const char* text, uint32& numeric) 
	{ 
		numeric = atoi(text);	
	}

	template<>
	static void ToNumeric(const char* text, float& numeric)
	{ 
		numeric = (float)atof(text); 
	}

	template<>
	static void ToNumeric(const char* text, double& numeric)
	{ 
		numeric = atof(text); 
	}

	template<>
	static void ToNumeric(const char* text, bool& numeric) 
	{ 
		numeric = (strcmp(text, "true")==0);
	}

public:
	template<typename T>
	static void ToString(const T numeric, char* str, uint32 bufferSize)
	{}

	template<>
	static void ToString(const string numeric, char* str, uint32 bufferSize) 
	{ 
		sprintf_s(str, bufferSize, "%s", numeric.c_str());
	}

	template<>
	static void ToString(const char* numeric, char* str, uint32 bufferSize) 
	{ 
		sprintf_s(str, bufferSize, "%s", numeric);
	}

	template<>
	static void ToString(const int numeric, char* str, uint32 bufferSize) 
	{
		_itoa_s(numeric, str, bufferSize, 10);
	}

	template<>
	static void ToString(const char numeric, char* str, uint32 bufferSize) 
	{
		_itoa_s(numeric, str, bufferSize, 10);
	}

	template<>
	static void ToString(const unsigned char numeric, char* str, uint32 bufferSize) 
	{
		_ultoa_s(numeric, str, bufferSize, 10);
	}

	template<>
	static void ToString(const short numeric, char* str, uint32 bufferSize) 
	{
		_ltoa_s(numeric, str, bufferSize, 10);
	}

	template<>
	static void ToString(const unsigned short numeric, char* str, uint32 bufferSize) 
	{
		_ultoa_s(numeric, str, bufferSize, 10);
	}

	template<>
	static void ToString(const uint32 numeric, char* str, uint32 bufferSize) 
	{
		_ultoa_s(numeric, str, bufferSize, 10);
	}

	template<>
	static void ToString(const float numeric, char* str, uint32 bufferSize) 
	{
		sprintf_s(str, bufferSize, "%.4f", numeric);
	}

	template<>
	static void ToString(const double numeric, char* str, uint32 bufferSize) 
	{
		sprintf_s(str, bufferSize, "%.4f", numeric);
	}

	template<>
	static void ToString(const bool numeric, char* str, uint32 bufferSize) 
	{
		if (numeric)
			sprintf_s(str, bufferSize, "true");
		else
			sprintf_s(str, bufferSize, "false");
	}
};

class CAesCryptography;

class CProtocolBuilder
{
public:
	typedef std::map<string, string>									StringKeyValueMap;
	typedef StringKeyValueMap::iterator								StringMapIterator;
	typedef StringKeyValueMap::const_iterator					StringMapConstIterator;
	typedef StringKeyValueMap::const_reverse_iterator StringMapReverseIterator;

	struct SProtocolHeader
	{
		uint8		type;
		uint32	totalLength;
		uint32	checkSum;
	};

public:
	CProtocolBuilder();
	CProtocolBuilder(const CProtocolBuilder& other);
	virtual ~CProtocolBuilder();

	template<typename T>
	bool					Add(const string& key, const T& value)
	{
		const int MaxValueLength = 64;
		char strValue[MaxValueLength] = {0,};
		DataTypeConvertor::ToString(value, strValue, MaxValueLength);
		return AddImpl(key, strValue);
	}

	bool					Remove(const string& key);
	void					Clear();

	template<typename T>
	T		GetValue(const string& key) const
	{
		string result = GetValueImpl(key);
		T value;
		DataTypeConvertor::ToNumeric(result.c_str(), value);
		return value;
	}

	////////////////////////////////////////////////////////////////////////
	// Description:
	//   Parsing and Add to container 
	//   i.e.) name=alice;level=40; => ["name", "alice"], ["level", "40"]
	////////////////////////////////////////////////////////////////////////
	bool		Parse(const string& source);
	bool		Parse(char* buffer, uint32 &offset, const uint32 bufferLength);
	string	MakeString() const;
	uint32	MakeBuffer(char* buffer, uint32 length) const;
	void		EncryptKey(unsigned char* key, unsigned char keyLen);


protected:
	bool		AddImpl(const string& key, const string& value);
	////////////////////////////////////////////////////////////////////////
	// Description:
	//   If there is no value, return empty string("")
	////////////////////////////////////////////////////////////////////////
	string	GetValueImpl(const string& key) const;

	uint32	MakeHeader(char* buffer, uint32 dataLength, uint32 checkSum) const;
	bool		GetHeader(const char* buffer, uint32 offset, uint32 bufferLength, SProtocolHeader& header) const;

private:
	StringKeyValueMap m_container;
	CAesCryptography*	m_cryptObj;
	bool							m_encrypt;
};

#endif // USING_LICENSE_PROTECTION

#endif // __protocolbuilder_h__

