////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2009.
// -------------------------------------------------------------------------
//  File name:   VisRegTest.cpp
//  Version:     v1.00
//  Created:     07/07/2009 by Nicolas Schulz.
//  Description: Visual Regression Test
// -------------------------------------------------------------------------
//  History:
//	07/07/2009		Created by Nicolas Schulz.
//	30/04/2010		Continued by Christian Helmich.
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "VisRegTest.h"

#include "ISystem.h"
#include "I3DEngine.h"
#include "IRenderer.h"
#include "IConsole.h"
#include "ITimer.h"
#include "IInput.h"
#include "IEntitySystem.h"

#define VISREG_LOG_POS(pos, ang)	"%.3f %.3f %.3f %.3f %.3f %.3f", pos.x, pos.y, pos.z, ang.x, ang.y, ang.z
#define VISREG_GOTO_POS(pos, ang)	"%f %f %f %f %f %f", pos.x, pos.y, pos.z, ang.x, ang.y, ang.z
#define VISREG_READ_POS(pos, ang)	"%f %f %f %f %f %f", &pos.x, &pos.y, &pos.z, &ang.x, &ang.y, &ang.z

#define VISREG_ENTITY_CLASSNAME		"VisRegPoint"

#if defined(PS3)
#define VISREG_BASE_DIR		SYS_APP_HOME
#else
#define VISREG_BASE_DIR		gEnv->pSystem->GetRootFolder()
#endif //defined(PS3)
#define VISREG_OUT_SUBDIR	"/VisualRegression/results/"

#define VISREG_DEBUG(inStr, ...)	CryLogAlways("[VisRegTest debug] " inStr, ## __VA_ARGS__)
#define VISREG_LOG(inStr, ...)		CryLogAlways("[VisRegTest] " inStr, ## __VA_ARGS__)
#define VISREG_WARNING(inStr, ...)	CryWarning( VALIDATOR_MODULE_SYSTEM, VALIDATOR_WARNING, "[VisRegTest] " inStr, ## __VA_ARGS__)
#define VISREG_ERROR(inStr, ...)	CryWarning( VALIDATOR_MODULE_SYSTEM, VALIDATOR_ERROR, "[VisRegTest] " inStr, ## __VA_ARGS__)

CVisRegTest::CVisRegTest(const char* filename):
	m_waitFrames( 0 ),
	m_fileName( filename ),
	m_csvName( PathUtil::Make(VISREG_BASE_DIR, filename, ".csv") ),
	m_screenshotPath(PathUtil::Make(VISREG_BASE_DIR, VISREG_OUT_SUBDIR, "")),
	m_deterministic( true ),
	m_fixedTimeStep( 0.033333f ),
	m_timeScale( 0.0f ),
	m_attachCamCmdName("g_detachCamera 0"),
	m_detachCamCmdName("g_detachCamera 1")	
{
	VISREG_LOG( "Enabled visual regression tests" );
	VISREG_DEBUG( "fileName: %s", m_fileName.c_str() );
	VISREG_DEBUG( "cvsName: %s", m_csvName.c_str() );
	VISREG_DEBUG( "screenshotPath: %s", m_screenshotPath.c_str() );
}


void CVisRegTest::Init()
{
	// Fill cmd buffer
	if( !LoadConfig() )
	{
		VISREG_ERROR( "Failed to load %s from game folder", m_fileName.c_str() );
		return;
	}

	//gEnv->pConsole->ExecuteString( "hud_startPaused 0" );

	gEnv->pTimer->SetTimeScale( m_timeScale );
	GetISystem()->GetISystemEventDispatcher()->OnSystemEvent( ESYSTEM_EVENT_RANDOM_SEED, 0, 0 );
	srand( 0 );

	// Disable features that are known to cause problems (that are non-deterministic)
	if(m_deterministic)
	{
		gEnv->pConsole->ExecuteString( "e_ParticlesThread 0" );
		gEnv->pConsole->ExecuteString( "r_multithreaded 0" );
	}
}


void CVisRegTest::AfterRender()
{
	ExecCommands();
}

//////////////////////////////////////////////////////////////////////////

bool CVisRegTest::LoadConfig()
{
	VISREG_LOG( "loading XML '%s'", m_fileName.c_str() );
	XmlNodeRef rootNode = GetISystem()->LoadXmlFile( PathUtil::GetGameFolder() + "/" + m_fileName.c_str() );
	if( !rootNode )
		return false;

	bool configLoadedOk = false;
	if(rootNode->isTag( "VisRegTest" ) ) 
		configLoadedOk = LoadConfig_v1(rootNode);
	else
		configLoadedOk = LoadConfig_v2(rootNode);

	assert( configLoadedOk );
	if( !configLoadedOk )
		return false;

	assert( m_cmdBufCallStack.empty() );	//we are in big trouble if this is false
	if( !m_cmdBufCallStack.empty() )
		return false;

	m_cmdBufCallStack.push( std::make_pair(m_cmdBuf.begin(), &m_cmdBuf) );		//push the default command buffer and its initial iterator
	return true;
}




//////////////////////////////////////////////////////////////////////////

bool CVisRegTest::LoadConfig_v1( XmlNodeRef rootNode )
{
	string levelMap = rootNode->getAttr( "map" );
	
	m_cmdBuf.clear();
	m_cmdBuf.push_back( SCmd( eCMDInit, "" ) );
	m_cmdBuf.push_back( SCmd( eCMDMap, levelMap ) );
	m_cmdBuf.push_back( SCmd( eCMDPostMap, "" ) );

	for( int i = 0; i < rootNode->getChildCount(); ++i )
	{
		XmlNodeRef node = rootNode->getChild( i );
		SCmd cmd;

		if( node->isTag( "WaitFrames" ) )
		{
			LoadNodeWaitFrames(node, cmd);
		}
		else if( node->isTag( "ConsoleCmd" ) )
		{
			LoadNodeConsoleCmd(node, cmd);
		}
		else if( node->isTag( "Goto" ) )
		{
			LoadNodeGoto(node, cmd);
		}
		else if( node->isTag( "GotoEntity" ) )
		{
			LoadNodeGotoEntity(node, cmd);
		}
		else if( node->isTag( "Screenshot" ) )
		{
			LoadNodeScreenshot(node, cmd);
		}
		else
		{
			continue;
		}

		m_cmdBuf.push_back( cmd );
	}

	m_cmdBuf.push_back( SCmd( eCMDFinish, ""  ) );

	return true;
}

//////////////////////////////////////////////////////////////////////////

bool CVisRegTest::LoadConfig_v2( XmlNodeRef rootNode )
{
	assert(rootNode);
	if( !rootNode )
		return false;

	if( !rootNode->isTag( "VisualRegression" ) )
		return false;

	string version( rootNode->getAttr( "version" ));
	if(version != string("2.0"))	//note for later: if update needed, change version number and add loader here
		return false;

	m_cmdBuf.clear();
	m_cmdBuf.push_back( SCmd( eCMDInit, "" ) );

	for( int lvl1 = 0; lvl1 < rootNode->getChildCount(); ++lvl1 )
	{
		LoadConfig_v2_Node_Settings( rootNode->getChild( lvl1 ) );
		LoadConfig_v2_Node_Macros( rootNode->getChild( lvl1 ) );
		LoadConfig_v2_Node_Tests( rootNode->getChild( lvl1 ) );
		LoadConfig_v2_Node_Test( rootNode->getChild( lvl1 ) );		//TODO: remove in next revision
	}

	m_cmdBuf.push_back( SCmd( eCMDConsoleCmd, "quit" ) );
	m_cmdBuf.push_back( SCmd( eCMDFinish, ""  ) );

	return true;
}


bool CVisRegTest::LoadConfig_v2_Node_Settings( XmlNodeRef node )
{
	assert(node);
	if( !node )
		return false;

	if( !node->isTag( "Settings" ) )
		return false;

	for( int lvl2 = 0; lvl2 < node->getChildCount(); ++lvl2 )
	{
		if( !LoadConfig_v2_ApplySetting(node->getChild( lvl2 )) )
			continue;
	}
	return true;
}


bool CVisRegTest::LoadConfig_v2_ApplySetting( XmlNodeRef node )
{
	assert(node);
	if( !node )
		return false;

	if( node->isTag("ConsoleVar") )
	{
		assert( node->haveAttr("cmd") );
		stack_string cvar( node->getAttr("cmd") );
		gEnv->pConsole->ExecuteString( cvar.c_str() );
	}
	else if( node->isTag("FixedTimeStep") )
	{
		assert( node->haveAttr("value") );
		m_fixedTimeStep = (float)atof( node->getAttr("value") );
	}
	else if( node->isTag("TimeScale") )
	{
		assert( node->haveAttr("value") );
		m_timeScale = (float)atof( node->getAttr("value") );
	}
	else if( node->isTag("Deterministic") )
	{
		assert( node->haveAttr("value") );
		stack_string val( node->getAttr("value") );

		if(val == "True")
			m_deterministic = true;
		else if(val == "False")
			m_deterministic = false;
		else
			return false;
	}
	else if( node->isTag("CameraControl") )
	{
		assert( node->haveAttr("attach") );
		if( !node->haveAttr("attach") )
			return false;

		assert( node->haveAttr("detach") );
		if( !node->haveAttr("detach") )
			return false;
		
		m_attachCamCmdName = node->getAttr("attach");
		m_detachCamCmdName = node->getAttr("detach");		
	}
	//extend here
	else
	{
		return false;
	}

	return true;
}

//////////////////////////////////////////////////////////////////////////

bool CVisRegTest::LoadConfig_v2_Node_Macros( XmlNodeRef node )
{
	assert(node);
	if(!node)
		return false;

	if( !node->isTag( "Macros" ) )
		return false;

	for( int lvl2 = 0; lvl2 < node->getChildCount(); ++lvl2 )
	{
		if( !LoadConfig_v2_CreateMacro(node->getChild( lvl2 )) )
			continue;
	}
	return true;
}


bool CVisRegTest::LoadConfig_v2_CreateMacro( XmlNodeRef node )
{
	assert(node);
	if(!node)
		return false;

	if( !node->isTag( "Macro" ) )
		return false;

	assert( node->haveAttr( "name" ) );
	if( !node->haveAttr( "name" ) )
		return false;

	string macroName( node->getAttr( "name" ) );

	TCmdBuffer& macroCmdBuf = m_macroBuf[macroName];

	return LoadConfig_v2_CreateCommandBuffer(node, macroCmdBuf);
}

bool CVisRegTest::LoadConfig_v2_InsertMacro( XmlNodeRef node, TCmdBuffer& cmdBuf )
{
	assert(node);
	if( !node )
		return false;

	if( !node->isTag( "InsertMacro" ) )
		return false;

	assert( node->haveAttr( "name" ) );
	if( !node->haveAttr( "name" ) )
		return false;

	string macroName( node->getAttr( "name" ) );

	TMacroBuffer::iterator itFound = m_macroBuf.find(macroName);
	if( itFound == m_macroBuf.end() )
	{
		VISREG_ERROR( "InsertMacro references to unknown macro '%s'", macroName.c_str() );
		return false;
	}

	VISREG_DEBUG( "inserting macro '%s'", macroName.c_str() );

	TCmdBuffer& macroCmdBuf		= itFound->second;
	TCmdBuffer::iterator itCmd	= macroCmdBuf.begin();
	TCmdBuffer::iterator itEnd	= macroCmdBuf.end();
	for(itCmd; itCmd != itEnd; ++itCmd)
	{
		SCmd cmd = *itCmd;
		ReplaceCounterVar(cmd.args, itCmd->args);
		cmdBuf.push_back(cmd);
	}
	return true;
}

//////////////////////////////////////////////////////////////////////////

bool CVisRegTest::LoadConfig_v2_Node_Tests( XmlNodeRef node )
{
	assert(node);
	if( !node )
		return false;

	if( !node->isTag( "Tests" ) )
		return false;

	for( int lvl2 = 0; lvl2 < node->getChildCount(); ++lvl2 )
	{
		if( !LoadConfig_v2_Node_Test(node->getChild( lvl2 )) )
			continue;
	}

	return true;
}

bool CVisRegTest::LoadConfig_v2_Node_Test( XmlNodeRef node )
{
	assert(node);
	if( !node )
		return false;

	if( !node->isTag( "Test" ) )
		return false;

	assert( node->haveAttr( "map" ) );	
	m_cmdBuf.push_back( SCmd( eCMDMap, node->getAttr( "map" ) ) );

	if( node->haveAttr( "auto" ) )
	{
		m_cmdBuf.push_back( SCmd( eCMDPostMap, node->getAttr( "auto" ) ) );
	}
	else
	{
		m_cmdBuf.push_back( SCmd( eCMDPostMap, "" ) );
	}
	

	return LoadConfig_v2_CreateCommandBuffer(node, m_cmdBuf);
}


bool CVisRegTest::LoadConfig_v2_CreateCommandBuffer( XmlNodeRef node, TCmdBuffer& cmdBuf )
{
	assert(node);
	if( !node )
		return false;

	for( int nodeId = 0; nodeId < node->getChildCount(); ++nodeId )
	{	
		SCmd cmd;
		if( LoadConfig_v2_CreateCommand(node->getChild( nodeId ), cmd) )
			cmdBuf.push_back( cmd );
		LoadConfig_v2_InsertMacro(node->getChild( nodeId ), cmdBuf);
	}
	return true;
}


bool CVisRegTest::LoadConfig_v2_CreateCommand( XmlNodeRef node, SCmd &cmd )
{
	assert(node);
	if(!node)
		return false;

	bool loadNodeOk = false;
	if( node->isTag( "WaitFrames" ) )
	{	
		loadNodeOk = LoadNodeWaitFrames(node, cmd);
	}
	else if( node->isTag( "ConsoleCmd" ) )
	{
		loadNodeOk = LoadNodeConsoleCmd(node, cmd);
	}
	else if( node->isTag( "Goto" ) )
	{
		loadNodeOk = LoadNodeGoto(node, cmd);
	}
	else if( node->isTag( "GotoEntity" ) )
	{
		loadNodeOk = LoadNodeGotoEntity(node, cmd);
	}
	else if( node->isTag( "Screenshot" ) )
	{
		loadNodeOk = LoadNodeScreenshot(node, cmd);
	}
	else if( node->isTag( "Log" ) )
	{
		loadNodeOk = LoadNodeLog(node, cmd);
	}
	else if( node->isTag( "CamGoto" ) )
	{
		loadNodeOk = LoadNodeCamGoto(node, cmd);
	}
	else if( node->isTag( "CamGotoEntity" ) )
	{
		loadNodeOk = LoadNodeCamGotoEntity(node, cmd);
	}
	else if( node->isTag( "CallMacro" ) )
	{
		loadNodeOk = LoadNodeCallMacro(node, cmd);
	}
	else if( node->isTag( "AutoVisReg" ) )
	{
		loadNodeOk = LoadNodeAutoVisReg(node, cmd);
	}
	else
	{
		loadNodeOk = false;
	}
	return loadNodeOk;
}


//////////////////////////////////////////////////////////////////////////


bool CVisRegTest::LoadNodeWaitFrames( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDWaitFrames;
	assert( node->haveAttr( "frames" ) );
	if( !node->haveAttr( "frames" ) )
		return false;
	
	cmd.args = node->getAttr( "frames" );
	return true;
}


bool CVisRegTest::LoadNodeConsoleCmd( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDConsoleCmd;
	assert( node->haveAttr( "cmd" ) );
	if( !node->haveAttr( "cmd" ) )
		return false;

	cmd.args = node->getAttr( "cmd" );
	return true;
}


bool CVisRegTest::LoadNodeGotoEntity( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDGotoEntity;
	return LoadGotoEntityData(node, cmd);
}

bool CVisRegTest::LoadNodeCamGotoEntity( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDCamGotoEntity;
	return LoadGotoEntityData(node, cmd);
}

bool CVisRegTest::LoadGotoEntityData( XmlNodeRef node, SCmd &cmd )
{
	assert( node->haveAttr( "name" ) );
	if( !node->haveAttr( "name" ) )
		return false;

	cmd.args = node->getAttr( "name" );
	return true;
}


bool CVisRegTest::LoadNodeGoto( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDGoto;
	return LoadGotoData(node, cmd);
}

bool CVisRegTest::LoadNodeCamGoto( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDCamGoto;
	return LoadGotoData(node, cmd);
}

bool CVisRegTest::LoadGotoData( XmlNodeRef node, SCmd &cmd )
{
	if( node->haveAttr( "location" ) )
	{
		cmd.args = node->getAttr( "location" );
		sscanf(cmd.args.c_str(), VISREG_READ_POS(cmd.pos, cmd.rot));
		VISREG_DEBUG( "goto data (sscanf): " VISREG_LOG_POS(cmd.pos, cmd.rot) ); //no comma is expected!
		return true;
	}
	else
	{
		assert( node->haveAttr( "px" ) );
		assert( node->haveAttr( "py" ) );
		assert( node->haveAttr( "pz" ) );		

		cmd.args = "";
		cmd.args.append( node->getAttr( "px" ) ); cmd.args.append( " " );
		cmd.args.append( node->getAttr( "py" ) ); cmd.args.append( " " );
		cmd.args.append( node->getAttr( "pz" ) );

		cmd.pos.x = (float)atof( node->getAttr( "px" ) );
		cmd.pos.y = (float)atof( node->getAttr( "py" ) );
		cmd.pos.z = (float)atof( node->getAttr( "pz" ) );

		if ( node->haveAttr( "rx" ) )
		{
			assert( node->haveAttr( "rx" ) );
			assert( node->haveAttr( "ry" ) );
			assert( node->haveAttr( "rz" ) );

			cmd.args.append( " " );
			cmd.args.append( node->getAttr( "rx" ) ); cmd.args.append( " " );
			cmd.args.append( node->getAttr( "ry" ) ); cmd.args.append( " " );
			cmd.args.append( node->getAttr( "rz" ) );

			cmd.rot.x = (float)atof( node->getAttr( "rx" ) );
			cmd.rot.y = (float)atof( node->getAttr( "ry" ) );
			cmd.rot.z = (float)atof( node->getAttr( "rz" ) );
		}
		VISREG_DEBUG( "goto data (single): " VISREG_LOG_POS(cmd.pos, cmd.rot) ); //no comma is expected!
		return true;
	}
	return false;
}


bool CVisRegTest::LoadNodeScreenshot( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDScreenshot;	//TODO: load @type for different eCMDScreenshot_@type

	assert( node->haveAttr( "name" ) );
	if( !node->haveAttr( "name" ) )
		return false;

	cmd.args = node->getAttr( "name" );
	return true;	
}


bool CVisRegTest::LoadNodeLog( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDLog;

	assert( node->haveAttr( "message" ) );
	if( !node->haveAttr( "message" ) )
		return false;

	cmd.args = node->getAttr( "message" );
	return true;
}


bool CVisRegTest::LoadNodeCallMacro( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDCallMacro;

	assert( node->haveAttr( "name" ) );
	if( !node->haveAttr( "name" ) )
		return false;

	cmd.args = node->getAttr( "name" );
	VISREG_DEBUG( "added call of macro: '%s'", cmd.args.c_str() );
	return true;
}


bool CVisRegTest::LoadNodeAutoVisReg( XmlNodeRef node, SCmd& cmd )
{
	cmd.cmd = eCMDAutoVisReg;

	assert( node->haveAttr( "macro" ) );
	if( !node->haveAttr( "macro" ) )
		return false;

	cmd.args = node->getAttr( "macro" );
	return true;
}

//////////////////////////////////////////////////////////////////////////
//command buffer execution

void CVisRegTest::ExecCommands()
{	
	static float labelColor[] = {0, 1, 0, 1};
	gEnv->pRenderer->Draw2dLabel( 10, 10, 2, labelColor, false, "Visual Regression Test" );

	// Forcibly disable user input	
	gEnv->pInput->EnableDevice( eDI_Keyboard, false );
	gEnv->pInput->EnableDevice( eDI_Mouse, false );
	gEnv->pInput->EnableDevice( eDI_XI, false );
	gEnv->pInput->EnableDevice( eDI_Unknown, false );

	if(m_waitFrames > 0)
	{
		--m_waitFrames;
		gEnv->pRenderer->Draw2dLabel( 10, 30, 2, labelColor, false, "(waiting %i frames)", m_waitFrames );
		return;
	}

	while( m_waitFrames <= 0 )
	{
		if(m_cmdBufCallStack.empty())
		{
			VISREG_ERROR( "callstack is empty" );
			gEnv->pConsole->ExecuteString("quit");
			break;
		}

		TCmdBuffer::iterator&	itCurCmd	= m_cmdBufCallStack.top().first;
		TCmdBuffer&				curCmdBuf	= *(m_cmdBufCallStack.top().second);
		
		if( itCurCmd != curCmdBuf.end() )		
		{	
			SCmd&	cmd	= *itCurCmd;
			++itCurCmd;

			switch( cmd.cmd )
			{
			case eCMDInit:
				OnCmdInit(cmd);
				break;

			case eCMDFinish:
				OnCmdFinish(cmd);
				break;

			case eCMDWaitFrames:
				OnCmdWaitFrames(cmd);
				break;

			case eCMDConsoleCmd:
				OnCmdConsoleCmd(cmd);
				break;

			case eCMDGoto:
				OnCmdGoto(cmd);
				break;

			case eCMDGotoEntity:
				OnCmdGotoEntity(cmd);
				break;

			case eCMDCamGoto:
				OnCmdCamGoto(cmd);
				break;

			case eCMDCamGotoEntity:
				OnCmdCamGotoEntity(cmd);
				break;

			case eCMDScreenshot:
				OnCmdScreenshot(cmd);
				break;

			case eCMDLog:
				OnCmdLog(cmd);
				break;

			case eCMDCallMacro:
				OnCmdCallMacro(cmd);	//this will m_cmdBufCallStack.push()
				break;

			case eCMDAutoVisReg:
				OnCmdAutoVisReg(cmd);	//this will create a new macro and m_cmdBufCallStack.push() it
				break;

			case eCMDMap:
				OnCmdMap(cmd);
				break;

			case eCMDPostMap:
				OnCmdPostMap(cmd);
				break;
			}
		}
		else
		{
			m_cmdBufCallStack.pop();
			VISREG_DEBUG( "popped macro from callstack" );
		}
		VISREG_DEBUG( "callstack depth %i", m_cmdBufCallStack.size() );		
	}	
}


//////////////////////////////////////////////////////////////////////////
//command reaction

void CVisRegTest::OnCmdInit( const SCmd& cmd )
{
	InitCSV();
}


void CVisRegTest::OnCmdScreenshot( const SCmd& cmd )
{
	string screenshotName("");	
	ReplaceCounterVar(screenshotName, cmd.args);
	stack_string fullSSPath(m_screenshotPath.c_str());
	fullSSPath.append(screenshotName.c_str());
	gEnv->pRenderer->ScreenShot( fullSSPath );
	
	VISREG_LOG( "saved screenshot: %s", fullSSPath.c_str() );
	LogStats(screenshotName.c_str());
}


void CVisRegTest::OnCmdLog( const SCmd& cmd )
{
	string logMsg("");
	ReplaceCounterVar(logMsg, cmd.args);
	LogStats(logMsg.c_str());	
}



void CVisRegTest::OnCmdWaitFrames( const SCmd& cmd )
{	
	m_waitFrames = (uint32)atoi( cmd.args.c_str() );
	VISREG_LOG( "waiting %i frames", m_waitFrames );
}


void CVisRegTest::OnCmdConsoleCmd( const SCmd& cmd )
{
	string cmdStr("");
	ReplaceCounterVar(cmdStr, cmd.args);
	VISREG_LOG( "executing ccmd: %s", cmdStr.c_str() );
	gEnv->pConsole->ExecuteString( cmdStr.c_str() );
}



void CVisRegTest::OnCmdGoto( const SCmd& cmd )
{
	VISREG_LOG( "PLAYER goto location: %s", cmd.args.c_str() );

	PlacePlayer(cmd.args.c_str());
	Log2CSV(cmd.pos, cmd.rot, cmd.args.c_str());
	m_waitFrames = 1;
}

void CVisRegTest::OnCmdGotoEntity( const SCmd& cmd )
{	
	VISREG_LOG( "PLAYER goto entity: %s", cmd.args.c_str() );

	Vec3 pos;
	Ang3 rot;
	if( GetEntityLocation(pos, rot, cmd.args.c_str()) )
	{
		char strbuf[256];
		sprintf( strbuf, VISREG_GOTO_POS(pos, rot) );

		PlacePlayer( strbuf );
		Log2CSV(pos, rot, cmd.args.c_str());
		m_waitFrames = 1;
	}
	else
	{
		VISREG_ERROR( "GotoEntity '%s' failed", cmd.args.c_str() );
	}	
}

void CVisRegTest::PlacePlayer( const char* location )
{
	stack_string tmp("goto ");
	tmp.append( location );	
	gEnv->pConsole->ExecuteString( tmp.c_str() );
	VISREG_DEBUG( "placing player at '%s'", location );
}



void CVisRegTest::OnCmdCamGoto( const SCmd& cmd )
{
	VISREG_LOG( "CAM goto location: %s", cmd.args.c_str() );

	PlaceCamera(cmd.pos, cmd.rot);
	Log2CSV(cmd.pos, RAD2DEG(cmd.rot), cmd.args.c_str());
	m_waitFrames = 1;
}

void CVisRegTest::OnCmdCamGotoEntity( const SCmd& cmd )
{
	VISREG_LOG( "CAM goto entity: %s", cmd.args.c_str() );

	Vec3 pos;
	Ang3 rot;
	if( GetEntityLocation(pos, rot, cmd.args.c_str()) )
	{
		PlaceCamera(pos, rot);
		Log2CSV(pos, RAD2DEG(rot), cmd.args.c_str());
		m_waitFrames = 1;
	}
	else
	{
		VISREG_ERROR( "CamGotoEntity '%s' failed", cmd.args.c_str() );
	}
}

void CVisRegTest::PlaceCamera( const Vec3& pos, const Ang3& rot )
{
	CCamera cam = gEnv->pSystem->GetViewCamera();	//gEnv->pRenderer->GetCamera();
	cam.SetPosition( pos );
	cam.SetAngles( rot );
	gEnv->pSystem->SetViewCamera( cam );	//gEnv->pRenderer->SetCamera( cam );
	
	//in case the gamedll/cryaction uses the ViewSystem, 
	//it will overwrite the camera values we set before
	//hence requiring the camgoto console command on the game side to force the
	//cam position to a certain location
	char location[256];
	sprintf( location, VISREG_GOTO_POS(pos, rot) );
 	stack_string tmp("camgoto ");
 	tmp.append( location );	
 	gEnv->pConsole->ExecuteString( tmp.c_str() );

	VISREG_DEBUG( "placing camera at " VISREG_LOG_POS(pos, rot) );
}


void CVisRegTest::OnCmdCallMacro( const SCmd& cmd )
{
	VISREG_LOG( "call macro: %s", cmd.args.c_str() );

	TMacroBuffer::iterator itFound = m_macroBuf.find(cmd.args);
	if( itFound == m_macroBuf.end() )
	{
		VISREG_ERROR( "CallMacro references to unknown macro '%s'", cmd.args.c_str() );
		return;
	}

	TCmdBuffer& macroCmdBuf		= itFound->second;
	m_cmdBufCallStack.push( std::make_pair(macroCmdBuf.begin(), &macroCmdBuf) );
	VISREG_DEBUG( "pushed macro to callstack '%s'", cmd.args.c_str() );
}


void CVisRegTest::OnCmdAutoVisReg( const SCmd& cmd )
{
	MacroVisReg( cmd.args.c_str() );
}


void CVisRegTest::OnCmdMap( const SCmd& cmd )
{
	VISREG_LOG( "loading map: %s", cmd.args.c_str() );

	stack_string tmp("map ");
	tmp.append( cmd.args.c_str() );
	gEnv->pConsole->ExecuteString( tmp.c_str() );
	m_waitFrames = 1;
}


void CVisRegTest::OnCmdPostMap( const SCmd& cmd )
{
	gEnv->pTimer->SetTimer( ITimer::ETIMER_GAME, 0 );
	gEnv->pTimer->SetTimer( ITimer::ETIMER_UI, 0 );

	char cvar_buf[256];
	sprintf(cvar_buf, "t_FixedStep %f", m_fixedTimeStep);	
	gEnv->pConsole->ExecuteString( cvar_buf );

	GetISystem()->GetISystemEventDispatcher()->OnSystemEvent( ESYSTEM_EVENT_RANDOM_SEED, 0, 0 );
	srand( 0 );

	VISREG_LOG( "Starting tests" );

	// Validate initial value of random number generator
	if( cry_rand() != 15196 )
		VISREG_ERROR( "Random number generator has unexpected value" );

	if(cmd.args != "")
		MacroVisReg( cmd.args.c_str() );
}


void CVisRegTest::OnCmdFinish( const SCmd& cmd )
{
	VISREG_LOG( "Finished tests" );

	gEnv->pInput->EnableDevice( eDI_Keyboard, true );
	gEnv->pInput->EnableDevice( eDI_Mouse, true );

	gEnv->pConsole->ExecuteString( "t_FixedStep 0" );
	gEnv->pTimer->SetTimeScale( 1 );
}

//////////////////////////////////////////////////////////////////////////


bool CVisRegTest::GetEntityLocation(Vec3& pos, Ang3& rot, const char* entityName)
{
	IEntity*	entity = gEnv->pEntitySystem->FindEntityByName( entityName );
	if( !entity )
	{
		VISREG_WARNING( "could not find entity '%s'", entityName );
		return false;
	}

	pos = entity->GetPos();
	rot = RAD2DEG(Ang3::GetAnglesXYZ( entity->GetRotation() ));
	VISREG_DEBUG( "found entity '%s'", entityName );
	VISREG_DEBUG( "at position " VISREG_LOG_POS(pos, rot) );
	return true;
}

//////////////////////////////////////////////////////////////////////////

void CVisRegTest::MacroVisReg( const char* visRegPointMacro )
{
	//create a new macro to hold cmds (goto + callmacro + ...) for all visregpoints
	//then push this macro on callstack

	string			vrpMacroName(visRegPointMacro);

	//generate unique name for macro	
	stack_string			cmdBufMacroProto(visRegPointMacro);
	cmdBufMacroProto.append( "_AVRP_${AVRP}" );

	stack_string			cmdBufMacroName("");
	ReplaceCounterVar(cmdBufMacroName, cmdBufMacroProto);
	
	TCmdBuffer&		visRegMacro		= m_macroBuf[cmdBufMacroName];
	assert( visRegMacro.empty() );
	if( !visRegMacro.empty() )
	{
		VISREG_WARNING( "AutoVisualRegression macro already exists '%s'", cmdBufMacroName.c_str() );
		//you should be worried if this arrives
	}

	//iterating over entities to auto-discover VisRegPoints
	unsigned long	entityCount		= 0;
	IEntityIt*		pIter			= gEnv->pEntitySystem->GetEntityIterator();
	pIter->MoveFirst();

	while( IEntity*	pEntity = pIter->Next() )
	{
		const char*		entityName		= pEntity->GetName();		
		const char*		className		= pEntity->GetClass()->GetName();

		++entityCount;
		VISREG_DEBUG( "counted %i entities so far", entityCount);

		//for each entity of class VISREG_ENTITY_CLASSNAME, push back cmds
		if(	className 
			&&	strcmp(className, VISREG_ENTITY_CLASSNAME) == 0)
		{
			VISREG_DEBUG( "discovered entity '%s'", entityName);

			visRegMacro.push_back( SCmd(eCMDConsoleCmd, m_detachCamCmdName) );
			visRegMacro.push_back( SCmd(eCMDCamGotoEntity, entityName) );
			visRegMacro.push_back( SCmd(eCMDCallMacro, vrpMacroName) );

			visRegMacro.push_back( SCmd(eCMDConsoleCmd, m_attachCamCmdName) );
			visRegMacro.push_back( SCmd(eCMDGotoEntity, entityName) );
			visRegMacro.push_back( SCmd(eCMDCallMacro, vrpMacroName) );
		}
	}

	//push macro to callstack
	m_cmdBufCallStack.push( std::make_pair(visRegMacro.begin(), &visRegMacro) );
	VISREG_DEBUG( "pushed macro to callstack '%s'", cmdBufMacroName.c_str() );
}

//////////////////////////////////////////////////////////////////////////

void CVisRegTest::LogStats(const char* msg)
{
	const CCamera &rCam = gEnv->pRenderer->GetCamera();			
	Ang3 aAng = RAD2DEG(Ang3::GetAnglesXYZ(rCam.GetMatrix())); // in degrees
	Vec3 vPos = rCam.GetPosition();

	VISREG_LOG( "%s", msg );
	VISREG_LOG( "current random number value: %i", cry_rand() );
	//VISREG_LOG( "current level: %s", gEnv->pSystem->GetIGame()->GetIGameFramework()->GetLevelName() );		//potential null pointer minefield
	VISREG_LOG( "current location: " VISREG_LOG_POS(vPos, aAng) );	//no comma is correct!
	VISREG_LOG( "current FPS: %f", gEnv->pTimer->GetFrameRate() );
	VISREG_LOG( "current (real) frametime (s): %f", gEnv->pTimer->GetRealFrameTime() );
	VISREG_LOG( "current (game) frametime (s): %f", gEnv->pTimer->GetFrameTime() );
	VISREG_LOG( "current drawcall count: %i", gEnv->pRenderer->GetCurrentNumberOfDrawCalls() );
	VISREG_LOG( "current memory use: %i", gEnv->pSystem->GetUsedMemory() );

	Log2CSV(vPos, aAng, msg);
}

//////////////////////////////////////////////////////////////////////////
//filesystem wrapping required for csv logging

//!	@struct FSWrapper
//!	\brief	simple wrapper class upon the filesystem
//!	required by CSV logging functionality
//!	was needed since writing to APP_HOME somehow failed on PS3 and just using defines was impossible
struct FSWrapper
{
#if defined(PS3)
	//trait PS3
	typedef int			FileDescriptor;
	typedef int			FileMode;
	enum			{	ModeWrite	= CELL_FS_O_CREAT | CELL_FS_O_EXCL | CELL_FS_O_WRONLY	};
	enum			{	ModeAppend	= CELL_FS_O_APPEND | CELL_FS_O_EXCL | CELL_FS_O_WRONLY	};	
	//trait PS3/
#else
	//trait PC/XBox
	typedef FILE*		FileDescriptor;
	typedef const char*	FileMode;
	static FileMode		ModeWrite;
	static FileMode		ModeAppend;
	//trait PC/XBox/
#endif //defined(PS3)

	static	FileDescriptor	Open(const char* filename, FileMode mode);
	static	void			Close(FileDescriptor fd);
	static	void			Write(FileDescriptor fd, byte* data, uint32 size);
};

#if defined(PS3)
extern const char* GetCellFsErrString(int cRetCode);

inline FSWrapper::FileDescriptor	FSWrapper::Open(const char* filename, FileMode mode)
{
	int	fd = 0;
	CellFsErrno	err = cellFsOpen(filename, mode, &fd, NULL, 0);
	if(err != CELL_FS_SUCCEEDED)
	{
		VISREG_ERROR( "cellFS failed to open %s: %i -> %s", filename, err, GetCellFsErrString(err));
		return 0;
	}
	return fd;
}

inline void	FSWrapper::Close(FileDescriptor fd)
{
	CellFsErrno	err = cellFsClose(fd);
	if(err != CELL_FS_SUCCEEDED)
	{
		VISREG_ERROR( "cellFS failed to close fd: %i -> %s", err, GetCellFsErrString(err));
	}
}

inline void	FSWrapper::Write(FileDescriptor fd, byte* data, uint32 size)
{
	CellFsErrno	err = cellFsWrite(fd, data, size,0);
	if(err != CELL_FS_SUCCEEDED)
	{
		VISREG_ERROR( "cellFS failed to write to fd: %i -> %s", err, GetCellFsErrString(err));
	}
}

#else
FSWrapper::FileMode	FSWrapper::ModeWrite	= "wt";
FSWrapper::FileMode	FSWrapper::ModeAppend	= "at";

inline FSWrapper::FileDescriptor	FSWrapper::Open(const char* filename, FileMode mode)
{
	return gEnv->pCryPak->FOpen(filename, mode);	//fxopen(filename, mode);
}

inline void	FSWrapper::Close(FileDescriptor fd)
{
	gEnv->pCryPak->FClose(fd);	//fclose(fd);
}

inline void	FSWrapper::Write(FileDescriptor fd, byte* data, uint32 size)
{
	gEnv->pCryPak->FWrite(data, size, fd);	//fwrite(fd, data, size);
}

#endif //defined(PS3)

//////////////////////////////////////////////////////////////////////////
//csv logging

void CVisRegTest::InitCSV()
{
	static stack_string	csvHeader("location;fps;frametime;drawcalls;memory;comment\n");
	VISREG_LOG( "CSV: %s", csvHeader.c_str() );

	VISREG_LOG( "logging performance stats to csv file: %s", m_csvName.c_str() );
	FSWrapper::FileDescriptor csvFd = FSWrapper::Open( m_csvName.c_str(), FSWrapper::ModeWrite );
	if(!csvFd)
	{
		VISREG_ERROR( "could not create csv file %s", m_csvName.c_str() );
		return;
	}	
	FSWrapper::Write(csvFd, (byte*)csvHeader.c_str(), csvHeader.size());
	FSWrapper::Close(csvFd);
}


void CVisRegTest::Log2CSV( const Vec3& vPos, const Ang3& aAng, const char* msg )
{	
	stack_string	csvLine("");
	static char		csvTemp[256];

	sprintf(csvTemp, VISREG_LOG_POS(vPos, aAng) );
	csvLine.append(csvTemp);

	sprintf(csvTemp, ";%f;%f;%i;%i;",		
		gEnv->pTimer->GetFrameRate(),
		gEnv->pTimer->GetRealFrameTime(),
		gEnv->pRenderer->GetCurrentNumberOfDrawCalls(),
		gEnv->pSystem->GetUsedMemory()
		);
	csvLine.append(csvTemp);

	csvLine.append(msg);
	csvLine.append("\n");
	VISREG_LOG( "CSV: %s", csvLine.c_str() );

	VISREG_LOG( "appending to csv file: %s", m_csvName.c_str() );	
	FSWrapper::FileDescriptor csvFd = FSWrapper::Open( m_csvName.c_str(), FSWrapper::ModeAppend );
	if(!csvFd)
	{
		VISREG_ERROR( "could not open csv file %s", m_csvName.c_str() );
		return;
	}	
	FSWrapper::Write(csvFd, (byte*)csvLine.c_str(), csvLine.size());
	FSWrapper::Close(csvFd);	
}

//////////////////////////////////////////////////////////////////////////
//<eof>
