////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2009.
// -------------------------------------------------------------------------
//  File name:   VisRegTest.h
//  Version:     v1.00
//  Created:     07/07/2009 by Nicolas Schulz.
//  Description: Visual Regression Test
// -------------------------------------------------------------------------
//  History:
//	07/07/2009		Created by Nicolas Schulz.
//	30/04/2010		Continued by Christian Helmich.
////////////////////////////////////////////////////////////////////////////

#ifndef __VisRegTest_h__
#define __VisRegTest_h__
#pragma once

class CVisRegTest
{
public:
	enum	ECmd
	{
		eCMDNone = -1,
		eCMDInit = 0,
		eCMDFinish,
		eCMDWaitFrames,
		eCMDConsoleCmd,
		eCMDGoto,
		eCMDGotoEntity,
		eCMDCamGoto,
		eCMDCamGotoEntity,
		eCMDScreenshot,	//write screen shot and logs position etc
		eCMDLog,		//logs position etc without screenshot
		eCMDMap,
		eCMDPostMap,
		eCMDCallMacro,
		eCMDAutoVisReg,

		eCMD_Count
	};

	struct	SCmd
	{
		ECmd    cmd;
		string  args;
		Vec3	pos;
		Ang3	rot;

		SCmd():
			cmd(eCMDNone), args(""), pos(0,0,0), rot(0,0,0) {}

		SCmd( ECmd cmd_, const string& args_ ):
			cmd(cmd_), args(args_), pos(0,0,0), rot(0,0,0)	{}

		SCmd( ECmd cmd_, const char* args_ ):
			cmd(cmd_), args(args_), pos(0,0,0), rot(0,0,0)	{}

		SCmd( ECmd cmd_, const Vec3& p, const Ang3& r ):
			cmd(cmd_), pos(p), rot(r), args("")				{}
	};
	
	typedef std::vector< SCmd >				TCmdBuffer;
	typedef std::map<string, TCmdBuffer>	TMacroBuffer;
	typedef std::map<string, uint32>		TRefCountBuffer;
	typedef std::pair<TCmdBuffer::iterator, TCmdBuffer*>			TCmdCallPair;
	typedef std::stack<TCmdCallPair, std::vector<TCmdCallPair> >	TCmdBufferCallStack;
	
protected:
	TCmdBufferCallStack	m_cmdBufCallStack;	
	
	TCmdBuffer			m_cmdBuf;	//the default command buffer
	TMacroBuffer		m_macroBuf;	//a named command buffer per macro
	
	TRefCountBuffer		m_refCountBuf;	//refcount for counter names (used in macros)
	
	uint32				m_waitFrames;
	string				m_fileName;
	string				m_csvName;
	string				m_screenshotPath;
	
	bool				m_deterministic;
	float				m_fixedTimeStep;
	float				m_timeScale;
	
	string				m_attachCamCmdName;
	string				m_detachCamCmdName;	

public:

	CVisRegTest(const char* filename = "visregtest.xml");
	
	void Init();
	void AfterRender();

protected:
	//load configurations
	bool LoadConfig();


	//old visreg xml format, kept for backwards compatibility
	bool LoadConfig_v1( XmlNodeRef rootNode );


	//new visreg xml format
	bool LoadConfig_v2( XmlNodeRef rootNode );

	bool LoadConfig_v2_Node_Settings( XmlNodeRef node );
	bool LoadConfig_v2_ApplySetting( XmlNodeRef node );

	bool LoadConfig_v2_Node_Macros( XmlNodeRef node );
	bool LoadConfig_v2_CreateMacro( XmlNodeRef node );
	bool LoadConfig_v2_InsertMacro( XmlNodeRef node, TCmdBuffer& cmdBuf );
	
	bool LoadConfig_v2_Node_Tests( XmlNodeRef node );
	bool LoadConfig_v2_Node_Test( XmlNodeRef node );
	bool LoadConfig_v2_CreateCommand( XmlNodeRef node, SCmd& cmd );
	bool LoadConfig_v2_CreateCommandBuffer( XmlNodeRef node, TCmdBuffer& cmdBuf );
	

	//xml node loading helpers
	bool LoadNodeGoto( XmlNodeRef node, SCmd& cmd );	
	bool LoadNodeGotoEntity( XmlNodeRef node, SCmd& cmd );

	bool LoadNodeCamGoto( XmlNodeRef node, SCmd& cmd );
	bool LoadNodeCamGotoEntity( XmlNodeRef node, SCmd& cmd );

	bool LoadGotoData( XmlNodeRef node, SCmd &cmd );
	bool LoadGotoEntityData( XmlNodeRef node, SCmd &cmd );

	bool LoadNodeScreenshot( XmlNodeRef node, SCmd& cmd );
	bool LoadNodeLog( XmlNodeRef node, SCmd& cmd );
	
	bool LoadNodeConsoleCmd( XmlNodeRef node, SCmd& cmd );
	bool LoadNodeWaitFrames( XmlNodeRef node, SCmd& cmd );

	bool LoadNodeCallMacro( XmlNodeRef node, SCmd& cmd );
	bool LoadNodeAutoVisReg( XmlNodeRef node, SCmd& cmd );


	//internal command buffer execution
	void ExecCommands();


	//command execution
	void OnCmdInit( const SCmd& cmd );
	void OnCmdMap( const SCmd& cmd );
	void OnCmdPostMap( const SCmd& cmd );
	
	void OnCmdGoto( const SCmd& cmd );
	void OnCmdGotoEntity( const SCmd& cmd );
	void PlacePlayer( const char* location );

	void OnCmdCamGoto( const SCmd& cmd );
	void OnCmdCamGotoEntity( const SCmd& cmd );
	void PlaceCamera( const Vec3& pos, const Ang3& rot );

	void OnCmdScreenshot( const SCmd& cmd );
	void OnCmdLog( const SCmd& cmd );

	void OnCmdCallMacro( const SCmd& cmd );
	void OnCmdAutoVisReg( const SCmd& cmd );
	
	void OnCmdConsoleCmd( const SCmd& cmd );
	void OnCmdWaitFrames( const SCmd& cmd );
	void OnCmdFinish( const SCmd& cmd );


	//utility
	bool GetEntityLocation(Vec3& pos, Ang3& rot, const char* entityName);
	void MacroVisReg( const char* visRegPointMacro );
	template<typename _STR>
	void ReplaceCounterVar(_STR& outStr, const _STR& inStr);


	//logging
	void LogStats(const char* msg);
	void InitCSV();
	void Log2CSV(const Vec3& vPos, const Ang3& aAng, const char* msg);
};


template<typename _STR>
void CVisRegTest::ReplaceCounterVar(_STR& outStr, const _STR& inStr)
{
	size_t countLoc	= inStr.find('$');
	size_t openLoc	= inStr.find('{');
	size_t closeLoc	= inStr.find('}');

	if(		countLoc	!= _STR::npos 
		&&	openLoc		!= _STR::npos
		&&	closeLoc	!= _STR::npos	)
	{		
		assert(countLoc+1 == openLoc);
		assert(openLoc < closeLoc);

		size_t countRefLen = closeLoc - openLoc - 1;
		_STR countRef( inStr.substr(openLoc+1, countRefLen) );

		++m_refCountBuf[countRef];

		//VISREG_DEBUG
		CryLogAlways( "countRef for '%s' -> %i", countRef.c_str(), m_refCountBuf[countRef]);

		char buf[32];
		sprintf(buf, "%d", m_refCountBuf[countRef]);

		outStr = inStr.substr(0, countLoc);
		outStr.append(buf);
		outStr.append( inStr.substr(closeLoc+1, _STR::npos) );

		//VISREG_DEBUG
		CryLogAlways( "replaced '%s' by '%s'", inStr.c_str(), outStr.c_str() );		
	}
	else
	{
		outStr = inStr;
	}
}


#endif // __VisRegTest_h__
