////////////////////////////////////////////////////////////////////////////
//
//	Crytek Engine Source File.
//	Copyright (C), Crytek Studios, 2008.
// -------------------------------------------------------------------------
//	File name:	 XenonDebugCallStack.cpp
//	Version:		 v1.00
//	Created:		 1/07/2008 by Alexey.
//	Compilers:	 Visual C++ 2005
//	Description: 
// -------------------------------------------------------------------------
//	History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"

#ifdef XENON

#include "XenonDebugCallStack.h"

#include "platform.h"
#include <ISystem.h>
#include <ILog.h>
#include <IConsole.h>

#include "System.h"
#include "Log.h"

template <typename T> void ByteReverse( T& data )
{
	BYTE* pData = ( BYTE* )&data;
	for( int i = 0; i < sizeof( data ) / 2; ++i )
	{
		std::swap( pData[i], pData[sizeof( data ) - 1 - i] );
	}
}

IDebugCallStack* IDebugCallStack::instance()
{
	static DebugCallStack sInstance;
	return &sInstance;
}


DebugCallStack::DebugCallStack(void)
{
	// make sure we don't have to allocate memory once we've crashed
	m_functions.reserve(MAX_DEBUG_STACK_ENTRIES);
	m_modules.reserve(MAX_MODULES_ENTRIES);

	m_rawFunctions.reserve(MAX_DEBUG_STACK_ENTRIES);
}

int	DebugCallStack::handleException( EXCEPTION_POINTERS *exception_pointer )
{
	((CLog*)(gEnv->pLog))->AllowDirectLoggingFromAnyThread(true);

	int ret = 0;
	static bool firstTime = true;

	/*if (g_cvars.sys_dump_aux_threads | g_cvars.sys_keyboard_break)
		for(int i=0;i<g_nDebugThreads;i++) if (g_idDebugThreads[i]!=GetCurrentThreadId())
			SuspendThread(OpenThread(THREAD_ALL_ACCESS,TRUE,g_idDebugThreads[i]));
*/

	if (!firstTime)
	{
		WriteLineToLog( "Critical Exception! Called Multiple Times!" );
		// Exception called more then once.
		return EXCEPTION_EXECUTE_HANDLER;
	}

	// Print exception info:
	{
		char excCode[80];
		char excAddr[80];
		WriteLineToLog( "<CRITICAL EXCEPTION>" );
		sprintf( excAddr,"0x%p",exception_pointer->ExceptionRecord->ExceptionAddress );
		sprintf( excCode,"0x%08X",exception_pointer->ExceptionRecord->ExceptionCode );
		WriteLineToLog( "Exception: %s, at Address: %s",excCode,excAddr );

		/*if(CSystem * pSystem = (CSystem*)DebugCallStack::instance()->GetSystem())
		{
			if(const char * pLoadingProfilerCallstack = pSystem->GetLoadingProfilerCallstack())
				if(pLoadingProfilerCallstack[0])
					WriteLineToLog( "<CrySystem> LoadingProfilerCallstack: %s", pLoadingProfilerCallstack );
		}*/
	}

	firstTime = false;

	CreateErrorLog(exception_pointer);
	SaveDumpMode();

	if(g_cvars.sys_no_crash_dialog != 0)
	{
		DebugBreak(); // halt execution so we can attach a debugger
	}
	else
	{
#ifdef _DEBUG
		DebugBreak(); // halt execution so we can attach a debugger
#endif
		DmCrashDump(false);
	}

	return EXCEPTION_EXECUTE_HANDLER;
}

void DebugCallStack::LogCallstack() 
{
	IDebugCallStack::LogCallstack();

	WriteLineToLog("Loaded Modules:");
	int len = (int)m_modules.size();
	for (int i = 0; i < len; i++)
	{
		const char* str = m_modules[i].c_str();
		WriteLineToLog( "%s",str );
	}
	WriteLineToLog( "=============================================================================" );

}

void DebugCallStack::CollectModules()
{
	m_modules.clear();

	char buffer[4096];
	DWORD writeCount = 0;
	// Walk the list of loaded code modules.
	HRESULT error;
	PDM_WALK_MODULES pWalkMod = NULL;
	DMN_MODLOAD modLoad;

	while( XBDM_NOERR == ( error = DmWalkLoadedModules( &pWalkMod, &modLoad ) ) )
	{
		// Find the signature that describes the PDB file of the current module.
		DM_PDB_SIGNATURE signature = {0};
		DmFindPdbSignature( modLoad.BaseAddress, &signature );

		ByteReverse( signature.Guid.Data1 );
		ByteReverse( signature.Guid.Data2 );
		ByteReverse( signature.Guid.Data3 );
		ByteReverse( signature.Age );
		
		//taken from the CallStackRecording example in Microsoft 360 SDK samples
		sprintf_s( buffer, "%s %s %p, %08X, %08X,{%08X-%04X-%04X-%02X %02X-%02X %02X %02X %02X %02X %02X}, %d", 
		modLoad.Name, signature.Path, 
		modLoad.BaseAddress, modLoad.Size,
		modLoad.TimeStamp,
		signature.Guid.Data1, signature.Guid.Data2, signature.Guid.Data3,
		signature.Guid.Data4[0], signature.Guid.Data4[1],
		signature.Guid.Data4[2], signature.Guid.Data4[3],
		signature.Guid.Data4[4], signature.Guid.Data4[5],
		signature.Guid.Data4[6], signature.Guid.Data4[7],
		signature.Age	);
		m_modules.push_back(buffer);
	}

	DmCloseLoadedModules( pWalkMod );
}

void DebugCallStack::FormatFunctions()
{
	m_functions.clear();

	char buffer[4096];

	const size_t numFunctions = m_rawFunctions.size();
	for (size_t i = 0; i < numFunctions; ++i)
	{
		sprintf_s( buffer, "\t%08X\r\n", (UINT_PTR)m_rawFunctions[ i ] );
		m_functions.push_back(buffer);
	}
}

void DebugCallStack::CollectCurrentCallStack(int maxStackEntries /*= MAX_DEBUG_STACK_ENTRIES*/)
{
	m_rawFunctions.resize(maxStackEntries); // we've already reserved this memory in the constructor

	// Capture a stack back trace.
	HRESULT hResult = DmCaptureStackBackTrace( m_rawFunctions.size(), &m_rawFunctions[0] );

	if( hResult == XBDM_NOERR )
	{
		if (m_rawFunctions[m_rawFunctions.size() - 1] != NULL)
		{
			WriteLineToLog("Warning: the crash handler collected a callstack that didn't end in null.\n"
				"This might mean that we have a partial callstack, and we need to collect more callstack entries.");
		}
		// remove trailing null entries
		while (!m_rawFunctions.empty() && m_rawFunctions[m_rawFunctions.size() - 1] == NULL)
		{
			m_rawFunctions.pop_back();
		}
	}
	else
	{
		WriteLineToLog("Error handling crash: couldn't capture a stack back trace (error code %d)", static_cast<int>(hResult));
		m_rawFunctions.clear();
	}

	FormatFunctions();
	CollectModules();
}

PREFAST_SUPPRESS_WARNING(6262) void DebugCallStack::CreateErrorLog(EXCEPTION_POINTERS* exception_pointer)
{
	static char errorString[32768] = "";

	// Time and Version.
	char versionbuf[1024];
	strcpy( versionbuf,"" );
	PutVersion( versionbuf );
	strcat( errorString,versionbuf );
	strcat( errorString,"\n" );

	//! Get call stack functions.
	CollectCurrentCallStack();		// is updating m_functions

	char excCode[80];
	char excAddr[80];
	sprintf( excAddr,"0x%p",exception_pointer->ExceptionRecord->ExceptionAddress );
	sprintf( excCode,"0x%08X",exception_pointer->ExceptionRecord->ExceptionCode );
	//string moduleName = getExceptionModule();
	const char *excModule = NULL;//moduleName.c_str();

	char desc[1024];
	const char *excName = TranslateExceptionCode(exception_pointer->ExceptionRecord->ExceptionCode);

	strcpy( desc,"" );

	if (exception_pointer->ExceptionRecord->ExceptionCode == EXCEPTION_ACCESS_VIOLATION)
	{
		if (exception_pointer->ExceptionRecord->NumberParameters > 1)
		{
			int iswrite = exception_pointer->ExceptionRecord->ExceptionInformation[0];
			DWORD64 accessAddr = exception_pointer->ExceptionRecord->ExceptionInformation[1];
			if (iswrite)
			{
				sprintf( desc,"Attempt to write data to address 0x%08p\r\nThe memory could not be \"written\"",(void*)accessAddr );
			}
			else
			{
				sprintf( desc,"Attempt to read from address 0x%08p\r\nThe memory could not be \"read\"",(void*)accessAddr );
			}
		}
	}

	char errs[32768];
	sprintf( errs,"Exception Code: %s\nException Addr: %s\nException Module: %s\nException Description: %s, %s\n",
		excCode,excAddr,excModule,excName,desc );

	strcat( errs,"\nCall Stack Trace:\n" );

	// Fill call stack.
	for (unsigned int i = 0; i < m_functions.size(); i++)
	{
		char temp[4096];
		sprintf( temp,"%2d) %s",m_functions.size()-i,(const char*)m_functions[i].c_str() );
		strcat( errs,temp );
	}

	strcat( errs, "\nLoaded Modules:\n");
	for (unsigned int i = 0; i < m_modules.size(); i++)
	{
		char temp[4096];
		sprintf( temp,"%s",(const char*)m_modules[i].c_str() );
		strcat( errs,temp );
		strcat( errs,"\n" );
	}

	strcat( errorString,errs );
	FILE *f = fxopen( "error.log","wt" );
	if (f)
	{
		fwrite( errorString,strlen(errorString),1,f );
/*		if (g_cvars.sys_dump_aux_threads | g_cvars.sys_keyboard_break)
		{
			funcs.clear();
			for(int i=0;i<g_nDebugThreads;i++) if (g_idDebugThreads[i]!=GetCurrentThreadId())
			{
				fprintf(f, "\n\nSuspended thread (%s):\n", g_nameDebugThreads[i]);
				HANDLE hThread = OpenThread(THREAD_ALL_ACCESS,TRUE,g_idDebugThreads[i]);
				GetThreadContext(hThread, &m_context);
				m_nSkipNumFunctions = 0;
				FillStackTrace(10, hThread);
				getCallStack( funcs );
				for (uint32 i=0; i<funcs.size(); i++)
					fprintf(f, "%2d) %s\n",funcs.size()-i,funcs[i].c_str() );
				ResumeThread(hThread);
			}
		}*/
		fclose(f);
	}

	WriteLineToLog( "Exception Code: %s",excCode );
	WriteLineToLog( "Exception Addr: %s",excAddr );
	WriteLineToLog( "Exception Module: %s",excModule );
	WriteLineToLog( "Exception Name  : %s",excName );
	WriteLineToLog( "Exception Description: %s",desc );
	LogCallstack();

	static int g_numScreenshots = 0;
	if (gEnv->pRenderer && !g_numScreenshots++)
	{
		gEnv->pRenderer->ScreenShot("error.tga");
	}
}

void DebugCallStack::SaveDumpMode()
{

	DM_DUMP_SETTINGS dumpSettings;
	
	switch(g_cvars.sys_dump_format)
	{
		case 0 :
			dumpSettings.ReportingFlags = DUMP_RF_FORMAT_NOHEAP;
			break;
		case 2 :
			dumpSettings.ReportingFlags = DUMP_RF_FORMAT_FULL;
			break;
		default:
			dumpSettings.ReportingFlags = DUMP_RF_FORMAT_PARTIAL;
	}

	DmSetDumpSettings(&dumpSettings);
}

#endif
