#ifdef WIN32

#include "CallStackWin32.h"

#pragma comment(lib, "dbghelp")

namespace CryUnit 
{
	const CallStackElement CallStackElementFactory::Create(HANDLE process, DWORD64 programCounter)
	{
		CallStackElement element;

		ExtractFunctionName(element, process, programCounter);
		ExtractModuleName(element, process, programCounter);
		ExtractFileNameAndLine(element, process, programCounter);

		return element;
	}

	void CallStackElementFactory::ExtractFunctionName(CallStackElement& element, HANDLE process, DWORD64 programCounter)
	{
		SYMBOL_INFO_PACKAGE symbolInfo;
		memset(&symbolInfo, 0, sizeof(symbolInfo));
		symbolInfo.si.MaxNameLen = MAX_SYM_NAME;
		if (SymFromAddr(process, programCounter, 0, &symbolInfo.si))
		{
			element.SetFunctionName(symbolInfo.si.Name);
		}
	}

	void CallStackElementFactory::ExtractModuleName(CallStackElement& element, HANDLE process, DWORD64 programCounter)
	{
		DWORD64 moduleBase = SymGetModuleBase64(process, programCounter);
		if (moduleBase) 
		{
			char moduleName[MAX_PATH];
			DWORD size = GetModuleFileNameA(reinterpret_cast<HMODULE>(moduleBase), moduleName, MAX_PATH);
			if (size > 0)
			{
				element.SetModuleName(moduleName);
			}
		}
	}

	void CallStackElementFactory::ExtractFileNameAndLine(CallStackElement& element, HANDLE process, DWORD64 programCounter)
	{
		IMAGEHLP_LINE64 imageHelp = { sizeof(IMAGEHLP_LINE64) };
		DWORD dummy = 0;
		if (SymGetLineFromAddr64(process, programCounter, &dummy, &imageHelp)) 
		{
			element.SetFileName(imageHelp.FileName);
			element.SetFileLineNumber(imageHelp.LineNumber);
		}
	}

	CallStack::CallStack()
		: m_process(GetCurrentProcess())
	{
		InitSymbols();
	}

	CallStack::~CallStack()
	{
		CleanupSymbols();
	}

	void CallStack::Retrieve(EXCEPTION_POINTERS* exceptionPointers)
	{
		STACKFRAME64 stackFrame;
		DWORD machineType;
		FillStackInfo(stackFrame, machineType, exceptionPointers->ContextRecord);
		NavigateStack(stackFrame, machineType, exceptionPointers->ContextRecord);
	}

	void CallStack::FillStackInfo(STACKFRAME64& stackFrame, DWORD& machineType, PCONTEXT context) const
	{
		memset(&stackFrame, 0, sizeof(stackFrame));

		stackFrame.AddrPC.Mode = AddrModeFlat;
		stackFrame.AddrStack.Mode = AddrModeFlat;
		stackFrame.AddrFrame.Mode = AddrModeFlat;

#if defined(_M_IX86)
		machineType = IMAGE_FILE_MACHINE_I386;
		stackFrame.AddrPC.Offset = context->Eip;
		stackFrame.AddrStack.Offset = context->Esp;
		stackFrame.AddrFrame.Offset = context->Ebp;
#elif defined(_M_X64)
		machineType = IMAGE_FILE_MACHINE_AMD64;
		stackFrame.AddrPC.Offset = context->Rip;
		stackFrame.AddrStack.Offset = context->Rsp;
		stackFrame.AddrFrame.Offset = context->Rdi;
#endif
	}

	void CallStack::NavigateStack(STACKFRAME64& stackFrame, DWORD machineType, PCONTEXT context)
	{
		HANDLE thread = GetCurrentThread();
		BOOL stackWalkRes = FALSE;
		do
		{
			stackWalkRes = StackWalk64(machineType, m_process, thread, &stackFrame, context, 0, &SymFunctionTableAccess64, &SymGetModuleBase64, 0);
			Add(CallStackElementFactory::Create(m_process, stackFrame.AddrPC.Offset));
		}
		while (stackWalkRes && stackFrame.AddrFrame.Offset);
	}

	void CallStack::InitSymbols()
	{
		DWORD options = SymGetOptions();
		options |= SYMOPT_LOAD_ANYTHING;
		SymSetOptions(options);
		SymInitialize(m_process, 0, TRUE);
	}

	void CallStack::CleanupSymbols()
	{
		SymCleanup(m_process);
	}
}

#endif
