#include "CryUnitAsserts.h"
#include "StringHelpers.h"
#include "TestFailedException.h"
#include "ISystemException.h"
#include <cmath>

#ifdef WIN32
#	include "SystemExceptionWin32.h"
#	include "CallStackWin32.h"
#endif

namespace CryUnit
{      
#ifdef WIN32
	const ICallStack& GetCallStackOnTestFailure()
	{
		const ICallStack* callStack = NULL;
		AttachSystemExceptionTranslator systemExceptionTranslator;
		try
		{
			systemExceptionTranslator.ThrowSystemException();
		}
		catch (const SehException& systemException)
		{
			callStack = &systemException.GetFilteredCallStack(5, 10);
		}
		return *callStack;
	}

	void Fail(const char* condition, const char* message, const char* file, int line)
	{
		throw TestFailedException(condition, message, file, line);
	}

	inline bool Equals(const char* expected, const char* actual)
	{
		return 0 == strcmp(expected, actual);
	}

	void AssertIsTrue(bool condition, const char* fileName , unsigned int line)
	{
		if (!condition)
		{
			Fail("", "AssertIsTrue failed.", fileName, line);
		}
	}

	void AssertIsFalse(bool condition, const char* fileName, unsigned int line)
	{
		if (condition)
		{
			Fail("", "AssertIsFalse failed.", fileName, line);
		}
	}

	void AssertAreEqual(const char* expected, const char* actual, const char* fileName, unsigned int line)
	{
		AssertAreEqual<const char*>(expected, actual, fileName, line);
	}

	void AssertIsNull(const void* ptr, const char* fileName, unsigned int line)
	{
		if (ptr != NULL)
		{
			const char* errorMessage = "AssertIsNull failed: Pointer is not NULL";
			Fail("", errorMessage, fileName, line);
		}
	}

	void AssertIsNotNull(const void* ptr, const char* fileName, unsigned int line)
	{
		if (ptr == NULL)
		{
			const char* errorMessage = "AssertIsNotNull failed: Pointer is NULL";
			Fail("", errorMessage, fileName, line);
		}
	}

	bool AreFloatEqual(float expected, float actual, float epsilon)
	{
		return fabs(expected - actual) <= epsilon;
	}

#else
	void Fail(const char* condition, const char* message, const char* file, int line) {}
	bool Equals(const char* expected, const char* actual) { return false; }
	void AssertIsTrue(bool condition, const char* fileName , unsigned int line) {}
	void AssertIsFalse(bool condition, const char* fileName, unsigned int line) {}
	void AssertAreEqual(const char* expected, const char* actual, const char* fileName, unsigned int line) {}
	void AssertIsNull(const void* ptr, const char* fileName, unsigned int line) {}
	void AssertIsNotNull(const void* ptr, const char* fileName, unsigned int line) {}
	bool AreFloatEqual(float expected, float actual, float epsilon) { return false; }
#endif
}
