#pragma once

#ifndef CRY_UNIT_ASSERT_H
#define CRY_UNIT_ASSERT_H

#include "StringStream.h"

namespace CryUnit
{
    void Fail(const char* condition, const char* message, const char* file, int line);
    bool Equals(const char* expected, const char* actual);

    template <typename T>
    bool Equals(const T& expected, const T& actual)
    {
        return expected == actual;
    }

    void AssertIsTrue(bool condition, const char* fileName , unsigned int line);
    void AssertIsFalse(bool condition, const char* fileName, unsigned int line);

    void AssertAreEqual(const char* expected, const char* actual, const char* fileName, unsigned int line);

    template <typename T>
    void AssertAreEqual(const T& expected, const T& actual, const char* fileName, unsigned int line)
    {
		if (!Equals(expected, actual))
        {
            StringStream errorStream;
            errorStream << "AssertAreEqual failed: Expected: '" << expected << "', got: '" << actual << "'";
            Fail("", errorStream.ToString(), fileName, line);
        }
    }

    template <typename T>
    void AssertAreNotEqual(const T& expected, const T& actual, const char* fileName, unsigned int line)
    {
        if (Equals(expected, actual))
        {
            StringStream errorStream;
            errorStream << "AssertAreNotEqual failed: Expected to not be: '" << expected << "'";
            Fail("", errorStream.ToString(), fileName, line);
        }
    }

    void AssertIsNull(const void* ptr, const char* fileName, unsigned int line);
    void AssertIsNotNull(const void* ptr, const char* fileName, unsigned int line);

    bool AreFloatEqual(float expected, float actual, float epsilon);

    template <typename T>
    void AssertFloatAreEqual(const T& expected, const T& actual, float epsilon, const char* fileName, unsigned int line)
    {
        if (!AreFloatEqual(expected, actual, epsilon))
        {
            StringStream errorStream;
            errorStream << "AssertFloatAreEqual failed: Expected: '" << expected << "', got: '" << actual << "'";
            Fail("", errorStream.ToString(), fileName, line);
        }
    }
}

#define CRY_UNIT_FAIL()										{ CryUnit::Fail("", "", __FILE__, __LINE__); }
#define CRY_UNIT_FAIL_MESSAGE(msg)							{ CryUnit::Fail("", msg, __FILE__, __LINE__); }
#define ASSERT_ARE_EQUAL(expected, actual)					{ CryUnit::AssertAreEqual(expected, actual, __FILE__, __LINE__); }
#define ASSERT_ARE_NOT_EQUAL(expected, actual)				{ CryUnit::AssertAreNotEqual(expected, actual, __FILE__, __LINE__); }
#define ASSERT_IS_TRUE(cond)								{ CryUnit::AssertIsTrue(cond, __FILE__, __LINE__); }
#define ASSERT_IS_FALSE(cond)								{ CryUnit::AssertIsFalse(cond, __FILE__, __LINE__); }
#define ASSERT_IS_NULL(ptr)									{ CryUnit::AssertIsNull(ptr, __FILE__, __LINE__); }
#define ASSERT_IS_NOT_NULL(ptr)								{ CryUnit::AssertIsNotNull(ptr, __FILE__, __LINE__); }
#define ASSERT_FLOAT_ARE_EQUAL(expected, actual, epsilon)	{ CryUnit::AssertFloatAreEqual(expected, actual, epsilon, __FILE__, __LINE__); }


#endif
