#include "StringHelpers.h"
#include <stdio.h>

namespace CryUnit
{
    namespace StringHelpers
    {
        const char* StringBuffer::ToString() const
        {
            return m_buffer.GetInternalData();
        }

#ifdef WIN32

        void StringBuffer::Append(int arg)
        {
            char buffer[256];
			sprintf_s(buffer, "%d", arg);
            m_buffer.Append(buffer);
        }

        void StringBuffer::Append(unsigned int arg)
        {
            char buffer[256];
            sprintf_s(buffer, "%u", arg);
            m_buffer.Append(buffer);
        }

        void StringBuffer::Append(float arg)
        {
            char buffer[256];
            sprintf_s(buffer, "%2.2f", arg);
            m_buffer.Append(buffer);
        }

        void StringBuffer::Append(const void* arg)
        {
            char buffer[256];
            sprintf_s(buffer, "%p", arg);
            m_buffer.Append(buffer);
        }

#else

		void StringBuffer::Append(int) {}
		void StringBuffer::Append(unsigned int) {}
		void StringBuffer::Append(float) {}
		void StringBuffer::Append(const void*) {}

#endif

		void StringBuffer::Append(const char* arg)
		{
			m_buffer.Append(arg);
		}

    }

	String::String()
		: m_capacity(m_defaultCapacity)
		, m_data(Alloc())
	{
		m_data[0] = '\0';
	}

	String::String(const char* data)
		: m_capacity(m_defaultCapacity)
		, m_data(Alloc())
	{
		Assign(data);
	}

	String::~String()
	{
		FreeData();
	}

	String::String(const String& str)
		: m_capacity(str.m_capacity)
		, m_data(Alloc())
	{
		Assign(str.m_data);
	}

	String& String::operator=(const String& str)
	{
		if (this != &str)
		{
			Assign(str.m_data);
		}

		return *this;
	}

	bool String::operator==(const String& str1) const
	{
		return Equal(m_data, str1.m_data);
	}

	const char* String::GetInternalData() const
	{
		return m_data;
	}

	void String::Append(const char* data)
	{
		size_t oldCapacity = m_capacity;
		ComputeCapacity(Length(data) + Length(m_data));

		if (m_capacity > oldCapacity)
		{
			Realloc();
		}

		Append(m_data, m_capacity, data);
	}

	void String::Assign(const char* data)
	{
		FreeData();

		ComputeCapacity(Length(data));
		m_data = Alloc();

		Copy(m_data, m_capacity, data);
	}

	char* String::Alloc()
	{
		return new char[m_capacity];
	}

	void String::FreeData()
	{
		delete [] m_data;
	}

	void String::Realloc()
	{
		char* data = Alloc();
		Copy(data, m_capacity, m_data);
		FreeData();
		m_data = data;
	}

	void String::ComputeCapacity(size_t len)
	{
		while (m_capacity < len + 1)
		{
			m_capacity *= 2;
		}
	}

	size_t String::Length(const char* str)
	{
		return strlen(str);
	}

#ifdef WIN32

	void String::Copy(char* dst, size_t capacity, const char* src)
	{
		strcpy_s(dst, capacity, src);
	}

	void String::Append(char* dst, size_t capacity, const char* src)
	{
		strcat_s(dst, capacity, src);
	}

#else

	void String::Copy(char* dst, size_t, const char* src)
	{
		strcpy(dst, src);
	}

	void String::Append(char* dst, size_t, const char* src)
	{
		strcat(dst, src);
	}

#endif

	bool String::Equal(const char* str1, const char* str2)
	{
		return 0 == strcmp(str1, str2);
	}

}
