#include "TestSuite.h"
#include "ITestListener.h"
#include "ITestCase.h"
#include "ICallStack.h"
#include "ITestFailedException.h"
#include "CallStackBase.h"
#include <memory>
#include <wchar.h>

namespace CryUnit
{
	// -----------------------------------------------------------------------------

	TestSuite::TestSuite(const char* name, Type type)
		: m_Name(name)
		, m_Type(type)
		, m_protectorEnabled(true)
	{

	}

	TestSuite::~TestSuite()
	{

	}

	void TestSuite::Release()
	{
		delete this;
	}

	void TestSuite::Run(ITestListener& listener) const
	{
		TestInfoList::const_iterator pos = m_ListOfTests.begin();
		TestInfoList::const_iterator end = m_ListOfTests.end();

		listener.TestSuiteRun(*this);

		for (; pos != end; ++pos)
		{
			RunTestCase(*pos, listener);
		}

		listener.TestSuiteEnd(*this);
	}

	void TestSuite::RegisterTestCase(const STestInfo& test_info)
	{
		m_ListOfTests.push_back(test_info);
	}

	const char* TestSuite::GetName() const
	{
		return m_Name;
	}

	TestSuite::Type TestSuite::GetType() const
	{
		return m_Type;
	}

	unsigned int TestSuite::GetNumberOfTests() const
	{
		return m_ListOfTests.size();
	}

	const STestInfo& TestSuite::GetTestInfoByIndex(unsigned int index) const
	{
		return m_ListOfTests.at(index);
	}

	const STestInfo* TestSuite::GetTestInfoByName(const char* name) const
	{
		for (TestInfoList::const_iterator it = m_ListOfTests.begin(); it != m_ListOfTests.end(); ++it)
		{
			if (0 == strcmp(it->Name, name))
			{
				return &(*it);
			}
		}
		return NULL;
	}

	void TestSuite::RunTestCase(const STestInfo& testInfo, ITestListener& listener) const
	{
		if (testInfo.Enabled)
		{
			CreateAndRunTestCase(*this, testInfo, listener, "CreateTest failed").Execute(m_protectorEnabled);
		}
		else
		{
			listener.TestSkipped(testInfo);
		}
	}

	void TestSuite::RunTestCase(ITestCase& testCase, ITestListener& listener) const
	{
		const STestInfo testInfo = testCase.GetTestInfo();
		listener.StartTestRun(testInfo);

		RunTestWithDefaultProtector(testCase, listener, "Unhandled exception").Execute(m_protectorEnabled);
		TearDownProtector(testCase, listener, "TearDown failed").Execute(m_protectorEnabled);

		listener.TestRun(testInfo);
	}

	void TestSuite::DisableProtector()
	{
		m_protectorEnabled = false;
	}

	// -----------------------------------------------------------------------------

	CreateAndRunTestCase::CreateAndRunTestCase(const ITestSuite& testSuite, const STestInfo& testInfo, ITestListener& listener, const char* message)
		: UnhandledExceptionProtector(testInfo, listener, message)
		, m_testSuite(testSuite) 
	{

	}

	void CreateAndRunTestCase::Protected()
	{
		std::auto_ptr<ITestCase> testCase(m_testInfo.TestFactory());
		m_testSuite.RunTestCase(*testCase, m_listener);
	}

	// -----------------------------------------------------------------------------

	RunTestWithDefaultProtector::RunTestWithDefaultProtector(ITestCase& testCase, ITestListener& listener, const char* message)
		: UnhandledExceptionProtector(testCase.GetTestInfo(), listener, message)
		, m_testCase(testCase) 
	{

	}

	void RunTestWithDefaultProtector::Protected()
	{
#ifdef WIN32
		try
		{
			m_testCase.SetUp();
			m_testCase.Run();
			m_listener.TestPassed(m_testInfo);
		}
		catch(const ITestFailedException& exception)
		{
			m_listener.TestFailed(m_testInfo, SFailureInfo(exception.GetCondition(), exception.GetMessage(), exception.GetFilePath(), exception.GetFileLine(), CallStackBase()));
		}
#endif
	}

	// -----------------------------------------------------------------------------

	TearDownProtector::TearDownProtector(ITestCase& testCase, ITestListener& listener, const char* message)
		: UnhandledExceptionProtector(testCase.GetTestInfo(), listener, message)
		, m_testCase(testCase) 
	{

	}

	void TearDownProtector::Protected()
	{
		m_testCase.TearDown();
	}

	// -----------------------------------------------------------------------------
}
