#include "UnhandledExceptionProtector.h"
#include "ICallStack.h"
#include "ISystemException.h"
#include "ITestListener.h"
#include "ITestCase.h"

namespace CryUnit
{

	UnhandledExceptionProtector::UnhandledExceptionProtector(const STestInfo& testInfo, ITestListener& listener, const char* message)
		: m_testInfo(testInfo)
		, m_listener(listener)
		, m_message(message)
	{

	}

	UnhandledExceptionProtector::~UnhandledExceptionProtector()
	{

	}

	void UnhandledExceptionProtector::Execute(bool runProtected/* = true*/)
	{
		if (runProtected)
		{
			RunCatchingAll();
		}
		else
		{
			Protected();
		}
	}

	void UnhandledExceptionProtector::RunCatchingAll()
	{
#ifdef WIN32
		AttachSystemExceptionTranslator systemExceptionTranslator;

		bool stackOverflowExceptionCatched = false;
		try
		{
			Protected();
		}
		catch (const IStackOverflowException&)
		{
			stackOverflowExceptionCatched = true;
		}
		catch (const ISystemException& systemException)
		{
			NotifyTestCrashed(systemException);
		}
		catch (...)
		{
			NotifyTestFailed("UnknownException");
		}

		if (stackOverflowExceptionCatched)
		{
			systemExceptionTranslator.ResetStack();
			NotifyTestFailed("StackOverflowException");
		}
#endif
	}

	void UnhandledExceptionProtector::NotifyTestFailed(const char* condition)
	{
		const ICallStack& callStack = ICallStack::CreateEmpty();
		m_listener.TestFailed(m_testInfo, SFailureInfo(condition, m_message, m_testInfo.FileName, m_testInfo.FileLineNumber, callStack));
		callStack.Release();
	}

	void UnhandledExceptionProtector::NotifyTestCrashed(const ISystemException &systemException)
	{
		const ICallStack& callStack = systemException.GetCallStack();
		m_listener.TestFailed(m_testInfo, SFailureInfo("SystemException", m_message, m_testInfo.FileName, m_testInfo.FileLineNumber, callStack));
		callStack.Release();
	}
}
