/*=============================================================================
	CommonRender.cpp: Crytek Common render helper functions and structures declarations.
	Copyright (c) 2001 Crytek Studios. All Rights Reserved.

	Revision history:
		* Created by Honich Andrey

=============================================================================*/

#include "StdAfx.h"

// Resource manager internal variables.
ResourceClassMap CBaseResource::m_sResources;
CryCriticalSection CBaseResource::s_cResLock;
CryCriticalSection CBaseResource::s_cResLockDel;
std::vector<CBaseResource *> CBaseResource::m_DeletedResources[2];

CBaseResource& CBaseResource::operator=(const CBaseResource& Src)
{
  return *this;
}

CBaseResource::~CBaseResource()
{
  UnRegister();
}

bool CBaseResource::IsValid()
{
  AUTO_LOCK(s_cResLock); // Not thread safe without this

  SResourceContainer *pContainer = GetResourcesForClass(m_ClassName);
  if (!pContainer)
    return false;

  ResourceClassMapItor itRM = m_sResources.find(m_ClassName);

  if (itRM == m_sResources.end())
    return false;
  if (itRM->second != pContainer)
    return false;
  ResourcesMapItor itRL = itRM->second->m_RMap.find(m_NameCRC);
  if (itRL == itRM->second->m_RMap.end())
    return false;
  if (itRL->second != this)
    return false;

  return true;
}

SResourceContainer *CBaseResource::GetResourcesForClass(const CCryNameTSCRC& className)
{
  ResourceClassMapItor itRM = m_sResources.find(className);
  if (itRM == m_sResources.end())
    return NULL;
  return itRM->second;
}

CBaseResource *CBaseResource::GetResource(const CCryNameTSCRC& className, int nID, bool bAddRef)
{
  AUTO_LOCK(s_cResLock); // Not thread safe without this

  SResourceContainer *pRL = GetResourcesForClass(className);
  if (!pRL)
    return NULL;

  //assert(pRL->m_RList.size() > nID);
  if (nID >= (int)pRL->m_RList.size() || nID < 0)
    return NULL;
  CBaseResource *pBR = pRL->m_RList[nID];
  if (pBR)
  {
    if (bAddRef)
      pBR->AddRef();
    return pBR;
  }
  return NULL;
}

CBaseResource *CBaseResource::GetResource(const CCryNameTSCRC& className, const CCryNameTSCRC& Name, bool bAddRef)
{
  AUTO_LOCK(s_cResLock); // Not thread safe without this

  SResourceContainer *pRL = GetResourcesForClass(className);
  if (!pRL)
    return NULL;

  CBaseResource *pBR = NULL;
  ResourcesMapItor itRL = pRL->m_RMap.find(Name);
  if (itRL != pRL->m_RMap.end())
  {
    pBR = itRL->second;
    if (bAddRef)
      pBR->AddRef();
    return pBR;
  }
  return NULL;
}

bool CBaseResource::Register(const CCryNameTSCRC& className, const CCryNameTSCRC& Name)
{
  AUTO_LOCK(s_cResLock); // Not thread safe without this

  SResourceContainer *pRL = GetResourcesForClass(className);
  if (!pRL)
  {
    pRL = new SResourceContainer;
    m_sResources.insert(ResourceClassMapItor::value_type(className, pRL));
  }

  assert(pRL);
  if (!pRL)
    return false;

  ResourcesMapItor itRL = pRL->m_RMap.find(Name);
  if (itRL != pRL->m_RMap.end())
    return false;
  pRL->m_RMap.insert(ResourcesMapItor::value_type(Name, this));
  int nIndex;
  if (pRL->m_AvailableIDs.size())
  {
    ResourceIds::iterator it = pRL->m_AvailableIDs.end()-1;
    nIndex = *it;
    pRL->m_AvailableIDs.erase(it);
    assert(nIndex < (int)pRL->m_RList.size());
    pRL->m_RList[nIndex] = this;
  }
  else
  {
    nIndex = pRL->m_RList.size();
    pRL->m_RList.push_back(this);
  }
  m_nID = nIndex;
  m_NameCRC = Name;
  m_ClassName = className;
  m_nRefCount = 1;

  return true;
}

bool CBaseResource::UnRegister()
{
  AUTO_LOCK(s_cResLock); // Not thread safe without this

  if (IsValid())
  {
    SResourceContainer *pContainer = GetResourcesForClass(m_ClassName);
    assert(pContainer);
    if (pContainer)
    {
      pContainer->m_RMap.erase(m_NameCRC);
      pContainer->m_RList[m_nID] = NULL;
      pContainer->m_AvailableIDs.push_back(m_nID);
    }
    return true;
  }
  return false;
}

int32 CBaseResource::Release()
{
  int32 nRef = CryInterlockedDecrement(&m_nRefCount);
  if (nRef <= 0)
  {
    if (gRenDev->m_pRT->IsRenderThread())
      delete this;
    else
    {
      UnRegister();
      int nFrame = gRenDev->m_nFrameSwapID & 1;
      {
        AUTO_LOCK(s_cResLockDel);
        m_DeletedResources[nFrame].push_back(this);
      }
    }
    return 0;
  }
  return nRef;
}

// Should be executed in render thread since shaders are using CryName's
void CBaseResource::Tick()
{
  int nFrame = 1 - (gRenDev->m_nFrameSwapID & 1);

  AUTO_LOCK(s_cResLockDel);

  uint32 i;
  for (i=0; i<m_DeletedResources[nFrame].size(); i++)
  {
    CBaseResource *pRes = m_DeletedResources[nFrame][i];
    SAFE_DELETE(pRes);
  }
  m_DeletedResources[nFrame].clear();
}
//=================================================================

SResourceContainer::~SResourceContainer()
{
  ResourcesMapItor it;

  // CBaseResource->Release() remove itself from the m_RMap, so we should to update "it" to begin() every iteration
  // or "it" will be invalid
  for (it=m_RMap.begin(); it!=m_RMap.end(); it=m_RMap.begin())
  {
    CBaseResource *pRes = it->second;
    if (pRes && CRenderer::CV_r_printmemoryleaks)
      iLog->Log("Warning: ~SResourceContainer: Resource %d was not deleted (%d)", pRes->GetID(), pRes->GetRefCounter());
    SAFE_RELEASE(pRes);
  }
  m_RMap.clear();
  m_RList.clear();
  m_AvailableIDs.clear();
}

void CBaseResource::ShutDown()
{
  if (CRenderer::CV_r_releaseallresourcesonexit)
  {
    ResourceClassMapItor it;
    for (it=m_sResources.begin(); it!=m_sResources.end(); it++)
    {
      SResourceContainer *pCN = it->second;
      SAFE_DELETE(pCN);
    }
    m_sResources.clear();
  }
}
