////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2001-2010.
// -------------------------------------------------------------------------
//  File name:   CryNameTS.h
//  Version:     v1.00
//  Created:     12/04/2010 by Andrey.
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef __CryNameTS_h__
#define __CryNameTS_h__
#pragma once

//////////////////////////////////////////////////////////////////////////
class CNameTableTS
{
public:
  // Name entry header, immediately after this header in memory starts actual string data.
  struct SNameEntryTS
  {
    // Reference count of this string.
    int nRefCount;
    // Current length of string.
    int nLength;
    // Size of memory allocated at the end of this class.
    int nAllocSize;
    // Here in memory starts character buffer of size nAllocSize.
    //char data[nAllocSize]

    const char* GetStr() { return (char*)(this+1); }
    void  AddRef() { CryInterlockedIncrement(&nRefCount);};
    int   Release() { return CryInterlockedDecrement(&nRefCount); };
    int   GetMemoryUsage() { return sizeof(SNameEntryTS)+strlen(GetStr()); }
    int		GetLength(){return nLength;}
  };

private:
	typedef stl::hash_map<const char*,SNameEntryTS*,stl::hash_stricmp<const char*> > NameMap;
	NameMap m_nameMap;

public:
	CNameTableTS() {}

	~CNameTableTS()
	{
		for( NameMap::iterator it = m_nameMap.begin() ; it != m_nameMap.end() ; ++it )
		{
			free(it->second);
		}				
	}

	// Only finds an existing name table entry, return 0 if not found.
	SNameEntryTS* FindEntry( const char *str )
	{
		SNameEntryTS *pEntry = stl::find_in_map(m_nameMap, str, 0);
		return pEntry;
	}

	// Finds an existing name table entry, or creates a new one if not found.
	SNameEntryTS* GetEntry( const char *str )
	{
		SNameEntryTS *pEntry = stl::find_in_map(m_nameMap, str, 0);
		if (!pEntry)
		{
			// Create a new entry.
			unsigned int nLen = strlen(str);
			unsigned int allocLen = sizeof(SNameEntryTS) + (nLen+1)*sizeof(char);
			pEntry = (SNameEntryTS*)malloc( allocLen );
			assert(pEntry!=NULL);
			pEntry->nRefCount = 0;
			pEntry->nLength = nLen;
			pEntry->nAllocSize = allocLen;
			// Copy string to the end of name entry.
			char *pEntryStr = const_cast<char*>(pEntry->GetStr());
			memcpy( pEntryStr,str,nLen+1 );
			// put in map.
			//m_nameMap.insert( NameMap::value_type(pEntry->GetStr(),pEntry) );
			m_nameMap[pEntry->GetStr()] = pEntry;
		}
		return pEntry;
	}

	// Release existing name table entry.
	void Release( SNameEntryTS *pEntry )
	{
		assert(pEntry);
		m_nameMap.erase( pEntry->GetStr() );
		free(pEntry);
	}
  int GetMemoryUsage()
  {
    int nSize = 0;
    NameMap::iterator it;
    int n = 0;
    for (it=m_nameMap.begin(); it!=m_nameMap.end(); it++)
    {
      nSize += strlen(it->first);
      nSize += it->second->GetMemoryUsage();
      n++;
    }
    nSize += n*8;

    return nSize;
  }
	void GetMemoryUsage( ICrySizer *pSizer ) const
	{
		pSizer->AddObject(this, sizeof(*this));
		pSizer->AddObject(m_nameMap);
	}
  int GetNumberOfEntries()
  {
    return m_nameMap.size();
  }

	// Log all names inside CryNameTS table.
	void LogNames()
	{
		NameMap::iterator it;
		for (it=m_nameMap.begin(); it!=m_nameMap.end(); ++it)
		{
			SNameEntryTS *pNameEntry = it->second;
			CryLog( "[%4d] %s",pNameEntry->nLength,pNameEntry->GetStr() );
		}
	}

};

///////////////////////////////////////////////////////////////////////////////
// Class CCryNameTS.
//////////////////////////////////////////////////////////////////////////
class	CCryNameTS
{
public:
	CCryNameTS();
	CCryNameTS( const CCryNameTS& n );
	CCryNameTS( const char *s );
	CCryNameTS( const char *s,bool bOnlyFind );
	~CCryNameTS();

	CCryNameTS& operator=( const CCryNameTS& n );
	CCryNameTS& operator=( const char *s );

  /*const char *c_str() {
#ifdef _DEBUG
#else
    return "";
#endif
  }*/

	bool	operator==( const CCryNameTS &n ) const;
	bool	operator!=( const CCryNameTS &n ) const;

	bool	operator==( const char *s ) const;
	bool	operator!=( const char *s ) const;

	bool	operator<( const CCryNameTS &n ) const;
	bool	operator>( const CCryNameTS &n ) const;

	bool	empty() const { return length() == 0; }
	void	reset()	{	_release(m_str);	m_str = 0; }
  void	addref()	{	_addref(m_str); }
	
	const	char*	c_str() const { 
#if defined(__SPU__)
		// to create an empty in in main memory, assume that the object itself lies there, than cast the 
		// 0 ptr to a char array containing '\0' (the empty string)
		return (m_str) ? m_str: SPU_MAIN_PTR( (char*)&(m_str) );
#else
		return (m_str) ? m_str: ""; 
#endif
	}
	int	length() const { return _length(); };

	static bool find( const char *str ) { return GetNameTable()->FindEntry(str) != 0; }
  static const char *create( const char *str )
  {
    CCryNameTS name = CCryNameTS(str);
    name._addref(name.c_str());
    return name.c_str();
  }
	void GetMemoryUsage( ICrySizer *pSizer ) const  
	{
		//pSizer->AddObject(m_str);
		pSizer->AddObject( GetNameTable() ); // cause for slowness?
	}
  static int GetMemoryUsage()
  {
    CNameTableTS *pTable = GetNameTable();
    return pTable->GetMemoryUsage();
  }
  static int GetNumberOfEntries()
  {
    CNameTableTS *pTable = GetNameTable();
    return pTable->GetNumberOfEntries();
  }

	// Compare functor for sorting CCryNames lexically.
	struct CmpLex
	{
		bool operator () (const CCryNameTS &n1, const CCryNameTS &n2) const
		{
			return strcmp(n1.c_str(), n2.c_str()) < 0;
		}
	};

private:
	typedef CNameTableTS::SNameEntryTS SNameEntry;

	static CNameTableTS* GetNameTable()
	{
    // Note: can not use a 'static CNameTable sTable' here, because that
    // implies a static destruction order dependency - the name table is
    // accessed from static destructor calls.
		static CNameTableTS *table = NULL;

    if (table == NULL)
      table = new CNameTableTS();
    return table;
	}
  static CryCriticalSection* GetCS()
  {
    // Note: can not use a 'static CNameTable sTable' here, because that
    // implies a static destruction order dependency - the name table is
    // accessed from static destructor calls.
    static CryCriticalSection *pCS = NULL;

    if (pCS == NULL)
      pCS = new CryCriticalSection();
    return pCS;
  }

	SNameEntry* _entry( const char *pBuffer ) const { assert(pBuffer); return ((SNameEntry*)pBuffer)-1; }
	void _release( const char *pBuffer )
  {
		if (pBuffer && _entry(pBuffer)->Release() <= 0)
			GetNameTable()->Release(_entry(pBuffer));
	}
	int  _length() const { return (m_str) ? _entry(m_str)->nLength : 0; };
	void _addref( const char *pBuffer ) { if (pBuffer) _entry(pBuffer)->AddRef(); }

	const char *m_str;
};


//////////////////////////////////////////////////////////////////////////
// CryName
//////////////////////////////////////////////////////////////////////////
inline CCryNameTS::CCryNameTS()
{
	m_str = 0;
}

//////////////////////////////////////////////////////////////////////////
inline CCryNameTS::CCryNameTS( const CCryNameTS& n )
{
	_addref( n.m_str );
	m_str = n.m_str;
}

//////////////////////////////////////////////////////////////////////////
inline CCryNameTS::CCryNameTS( const char *s )
{
	m_str = 0;
	*this = s;
}

//////////////////////////////////////////////////////////////////////////
inline CCryNameTS::CCryNameTS( const char *s,bool bOnlyFind )
{
	assert(s);
	m_str = 0;
	if (*s) // if not empty
	{
		SNameEntry *pNameEntry = GetNameTable()->FindEntry(s);
		if (pNameEntry)
		{
			m_str = pNameEntry->GetStr();
			_addref(m_str);
		}
	}
}

inline CCryNameTS::~CCryNameTS()
{
	_release(m_str);
}

//////////////////////////////////////////////////////////////////////////
inline CCryNameTS&	CCryNameTS::operator=( const CCryNameTS &n )
{
	if (m_str != n.m_str)
	{
		_release(m_str);
		m_str = n.m_str;
		_addref(m_str);
	}
	return *this;
}

//////////////////////////////////////////////////////////////////////////
inline CCryNameTS&	CCryNameTS::operator=( const char *s )
{
	assert(s);
	const char *pBuf = 0;
	if (s && *s) // if not empty
	{
		pBuf = GetNameTable()->GetEntry(s)->GetStr();
	}
	if (m_str != pBuf)
	{
		_release(m_str);
		m_str = pBuf;
		_addref(m_str);
	}
	return *this;
}


//////////////////////////////////////////////////////////////////////////
inline bool	CCryNameTS::operator==( const CCryNameTS &n ) const {
	return m_str == n.m_str;
}

inline bool	CCryNameTS::operator!=( const CCryNameTS &n ) const {
	return !(*this == n);
}

inline bool	CCryNameTS::operator==( const char* str ) const {
	return m_str && stricmp(m_str,str) == 0;
}

inline bool	CCryNameTS::operator!=( const char* str ) const {
	if (!m_str)
		return true;
	return stricmp(m_str,str) != 0;
}

inline bool	CCryNameTS::operator<( const CCryNameTS &n ) const {
	return m_str < n.m_str;
}

inline bool	CCryNameTS::operator>( const CCryNameTS &n ) const {
	return m_str > n.m_str;
}

inline bool	operator==( const string &s,const CCryNameTS &n ) {
	return n == s;
}
inline bool	operator!=( const string &s,const CCryNameTS &n ) {
	return n != s;
}

inline bool	operator==( const char *s,const CCryNameTS &n ) {
	return n == s;
}
inline bool	operator!=( const char *s,const CCryNameTS &n ) {
	return n != s;
}


///////////////////////////////////////////////////////////////////////////////
// Class CCryNameTSCRC.
//////////////////////////////////////////////////////////////////////////
class	CCryNameTSCRC
{
public:
  CCryNameTSCRC();
  CCryNameTSCRC( const CCryNameTSCRC& n );
  CCryNameTSCRC( const char *s );
  CCryNameTSCRC( const char *s,bool bOnlyFind );
  CCryNameTSCRC( uint32 n ) { m_nID = n; }
  ~CCryNameTSCRC();

  CCryNameTSCRC& operator=( const CCryNameTSCRC& n );
  CCryNameTSCRC& operator=( const char *s );

  bool	operator==( const CCryNameTSCRC &n ) const;
  bool	operator!=( const CCryNameTSCRC &n ) const;

  bool	operator==( const char *s ) const;
  bool	operator!=( const char *s ) const;

  bool	operator<( const CCryNameTSCRC &n ) const;
  bool	operator>( const CCryNameTSCRC &n ) const;

  bool	empty() const { return m_nID == 0; }
  void	reset()	{	m_nID = 0; }
  uint32 get()	{	return m_nID; }
  void  add(int nAdd) { m_nID += nAdd; }

  AUTO_STRUCT_INFO

  void GetMemoryUsage( ICrySizer *pSizer ) const {/*nothing*/}
private:

  uint32 m_nID;

};

//////////////////////////////////////////////////////////////////////////
// CCryNameTSCRC
//////////////////////////////////////////////////////////////////////////
#include <crc32.h>
extern Crc32Gen g_pRendCrc32Gen;

inline CCryNameTSCRC::CCryNameTSCRC()
{
  m_nID = 0;
}

//////////////////////////////////////////////////////////////////////////
inline CCryNameTSCRC::CCryNameTSCRC( const CCryNameTSCRC& n )
{
  m_nID = n.m_nID;
}

//////////////////////////////////////////////////////////////////////////
inline CCryNameTSCRC::CCryNameTSCRC( const char *s )
{
  m_nID = 0;
  *this = s;
}

inline CCryNameTSCRC::~CCryNameTSCRC()
{
  m_nID = 0;
}

//////////////////////////////////////////////////////////////////////////
inline CCryNameTSCRC&	CCryNameTSCRC::operator=( const CCryNameTSCRC &n )
{
  m_nID = n.m_nID;
  return *this;
}

//////////////////////////////////////////////////////////////////////////
inline CCryNameTSCRC&	CCryNameTSCRC::operator=( const char *s )
{
  assert(s);
  if (*s) // if not empty
  {
    g_pRendCrc32Gen.Init();
    m_nID = g_pRendCrc32Gen.GetCRC32Lowercase(s);
  }
  return *this;
}


//////////////////////////////////////////////////////////////////////////
inline bool	CCryNameTSCRC::operator==( const CCryNameTSCRC &n ) const {
  return m_nID == n.m_nID;
}

inline bool	CCryNameTSCRC::operator!=( const CCryNameTSCRC &n ) const {
  return !(*this == n);
}

inline bool	CCryNameTSCRC::operator==( const char* str ) const
{
  assert(str);
  if (*str) // if not empty
  {
    g_pRendCrc32Gen.Init();
    uint32 nID = g_pRendCrc32Gen.GetCRC32Lowercase(str);
    return m_nID == nID;
  }
  return m_nID == 0;
}

inline bool	CCryNameTSCRC::operator!=( const char* str ) const {
  if (!m_nID)
    return true;
  if (*str) // if not empty
  {
    g_pRendCrc32Gen.Init();
    uint32 nID = g_pRendCrc32Gen.GetCRC32Lowercase(str);
    return m_nID != nID;
  }
  return false;
}

inline bool	CCryNameTSCRC::operator<( const CCryNameTSCRC &n ) const {
  return m_nID < n.m_nID;
}

inline bool	CCryNameTSCRC::operator>( const CCryNameTSCRC &n ) const {
  return m_nID > n.m_nID;
}

inline bool	operator==( const string &s,const CCryNameTSCRC &n ) {
  return n == s;
}
inline bool	operator!=( const string &s,const CCryNameTSCRC &n ) {
  return n != s;
}

inline bool	operator==( const char *s,const CCryNameTSCRC &n ) {
  return n == s;
}
inline bool	operator!=( const char *s,const CCryNameTSCRC &n ) {
  return n != s;
}

#endif //__CryNameTS_h__
