/*=============================================================================
RemoteCompiler.h : socket wrapper for shader compile server connections
Copyright (c) 2008 Crytek Studios. All Rights Reserved.

Revision history:
* Created by Michael Kopietz

=============================================================================*/

#include "StdAfx.h"
#include "RemoteCompiler.h"

#if defined(PS3)
	#include <sys/types.h>
	#include <sys/time.h>
	#include <sys/socket.h>
	#include <sys/select.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#include <unistd.h>
	#include <netex/errno.h>
	#include <netdb.h>
#elif defined(XENON)
	#pragma comment(lib,"xnet.lib") 
#else
	#pragma comment(lib,"wsock32.lib") 
#endif

namespace NRemoteCompiler
{

uint32 CShaderSrv::m_LastWorkingServer=0;

CShaderSrv::CShaderSrv()
{
	Init();
}

void CShaderSrv::Init()
{
#ifdef _MSC_VER
	WSADATA Data;
	if(WSAStartup(MAKEWORD(2,0),&Data))
	{
		iLog->Log("ERROR: CShaderSrv::Init: Could not init root socket\n");
		return;
	}
#endif
}

CShaderSrv& CShaderSrv::Instance()
{
	static CShaderSrv g_ShaderSrv;
	return g_ShaderSrv;
}

string CShaderSrv::CreateXMLNode(const string& rTag,const string& rValue)	const
{
	string Tag=rTag;
	Tag+="=\"";
	Tag+=rValue;
	Tag+="\" ";
	return Tag;
}

/*string CShaderSrv::CreateXMLDataNode(const string& rTag,const string& rValue)	const
{
	string Tag="<";
	Tag+=rTag;
	Tag+="><![CDATA[";
	Tag+=rValue;
	Tag+="]]>";
	return Tag;
}*/

string CShaderSrv::TransformToXML(const string& rIn)	const
{
	string Out;
	for(size_t a=0,Size=rIn.size();a<Size;a++)
	{
		const char C=rIn.c_str()[a];
		if(C=='&')
			Out+="&amp;";
		else
		if(C=='<')
			Out+="&lt;";
		else
		if(C=='>')
			Out+="&gt;";
		else
		if(C=='\"')
			Out+="&quot;";
		else
		if(C=='\'')
			Out+="&apos;";
		else
			Out+=C;
	}
	return Out;
}

bool CShaderSrv::CreateRequest(	std::vector<uint8>&	rVec,
																std::vector<std::pair<string,string> >& rNodes)	const
{
	string Request="<?xml version=\"1.0\"?><Compile ";
	Request	+=	CreateXMLNode("Version",TransformToXML("2.0"));
	for(size_t a=0;a<rNodes.size();a++)
		Request	+=	CreateXMLNode(rNodes[a].first,TransformToXML(rNodes[a].second));

	Request	+=	" />";
	rVec	=	std::vector<uint8>(Request.c_str(),&Request.c_str()[Request.size()+1]);
	return true;
}

void CShaderSrv::RequestLine(const SCacheCombination& cmb,const string& rLine) const
{
  const char *szTarget;
  if (CParserBin::m_bXenon)
    szTarget = "x360";
  else
  if (CParserBin::m_bPS3)
    szTarget = "PS3";
  else
  if (CParserBin::m_bD3D11)
    szTarget = "D3D10";
  else
    szTarget = "D3D9";

	const string	List(string(szTarget)+"/"+cmb.Name.c_str()+"ShaderList.txt");
//	const string	Line(cmb.CacheName.c_str());
	RequestLine(List,rLine);
}

bool CShaderSrv::CommitPLCombinations(std::vector<SCacheCombination>&	rVec)
{
	const uint32 STEPSIZE=32;
	float T0	=	iTimer->GetAsyncCurTime();
  for(uint32 i=0; i<rVec.size(); i+=STEPSIZE)
	{
		string Line(rVec[i].CacheName.c_str());
	  for(uint32 j=1; j<STEPSIZE && i+j<rVec.size(); j++)
			Line+=string(";")+rVec[i+j].CacheName.c_str();
		RequestLine(rVec[i],Line);
	}
	float T1	=	iTimer->GetAsyncCurTime();
	iLog->Log("%3.3f to commit %d Combinations\n",T1-T0,rVec.size());


	return true;
}

EServerError CShaderSrv::Compile(	std::vector<uint8>&	rVec,
																							const char* 				pProfile,
																							const char* 				pProgram,
																							const char* 				pEntry,
																							const char* 				pCompileFlags)	const
{
	EServerError errCompile;

	std::vector<uint8>	CompileData;
	std::vector<std::pair<string,string> > Nodes;
	Nodes.resize(6);
	Nodes[0]	=	std::pair<string,string>(string("JobType"),string("Compile"));
	Nodes[1]	=	std::pair<string,string>(string("Profile"),string(pProfile));
	Nodes[2]	=	std::pair<string,string>(string("Program"),string(pProgram));
	Nodes[3]	=	std::pair<string,string>(string("Entry"),string(pEntry));
	Nodes[4]	=	std::pair<string,string>(string("CompileFlags"),string(pCompileFlags));
	Nodes[5]	=	std::pair<string,string>(string("HashStop"),string("1"));
	if(gRenDev->CV_r_ShaderCompilerDontCache)
	{
		Nodes.resize(Nodes.size()+1);
		Nodes[Nodes.size()-1]	=	std::pair<string,string>(string("Caching"),string("0"));
	}
//	Nodes[5]	=	std::pair<string,string>(string("ShaderRequest",string(pShaderRequestLine));
	int nRetries = 3;
	do 
	{
		if(!CreateRequest(CompileData,Nodes))
		{
			iLog->LogError("ERROR: CShaderSrv::Compile: failed composing Request XML\n");
			return ESFailed;
		}

		errCompile = Send(CompileData);
	} while (errCompile == ESRecvFailed && nRetries-- > 0);
	
	rVec	=	CompileData;

	if (errCompile != ESOK)
	{
		const char *why = (errCompile == ESNetworkError || errCompile == ESSendFailed || errCompile == ESRecvFailed) ? "Network Error" : "";
		iLog->LogError("ERROR: CShaderSrv::Compile: failed to compile %s (%s)",pEntry,why);
	}
	return errCompile;
}

void CShaderSrv::RequestLine(const string& rList,const string& rString)	const
{
	if(!gRenDev->CV_r_shaderssubmitrequestline)
		return;

	std::vector<uint8>	CompileData;
	std::vector<std::pair<string,string> > Nodes;
	Nodes.resize(3);
	Nodes[0]	=	std::pair<string,string>(string("JobType"),string("RequestLine"));
	Nodes[1]	=	std::pair<string,string>(string("Platform"),rList);
	Nodes[2]	=	std::pair<string,string>(string("ShaderRequest"),rString);
	if(!CreateRequest(CompileData,Nodes))
	{
		iLog->LogError("ERROR: CShaderSrv::RequestLine: failed composing Request XML\n");
		return;
	}
	Send(CompileData);
}

bool CShaderSrv::Send(SOCKET Socket, const char* pBuffer,uint32 Size)	const
{
	size_t w;
	size_t wTotal = 0;
	while(wTotal<Size)
	{
		w = send(Socket, pBuffer + wTotal, Size - wTotal, 0);
		if (w < 0)
		{
			iLog->Log("ERROR:CShaderSrv::Send failed (%d, %d)\n",	(int)w, WSAGetLastError());
			return false;
		}
		wTotal += (size_t)w;
	}
	return true;
}

bool CShaderSrv::Send(SOCKET Socket,std::vector<uint8>& rCompileData)	const
{
	const uint64 Size	=	static_cast<uint32>(rCompileData.size());
	return	Send(Socket,(const char*)&Size,8) &&
					Send(Socket,(const char*)&rCompileData[0],static_cast<uint32>(Size));
}

EServerError CShaderSrv::Recv(SOCKET Socket,std::vector<uint8>& rCompileData)	const
{
	const size_t Offset	=	5;	//version 2 has 4byte size and 1 byte state
//	const uint32 Size	=	static_cast<uint32>(rCompileData.size());
//	return	Send(Socket,(const char*)&Size,4) ||
//		Send(Socket,(const char*)&rCompileData[0],Size);


	//	delete[] optionsBuffer;
	uint32 nMsgLength = 0;
	uint32 nTotalRecived = 0;
	const size_t	BLOCKSIZE	=	16*1024;
	const size_t	SIZELIMIT	=	1024*1024;
	rCompileData.resize(0);
	rCompileData.reserve(64*1024);
	int CurrentPos	=	0;
	while(rCompileData.size()<SIZELIMIT)
	{
		rCompileData.resize(CurrentPos+BLOCKSIZE);
		int Recived = recv(Socket,reinterpret_cast<char*>(&rCompileData[CurrentPos]),BLOCKSIZE, 0);

		if (Recived >= 0)
			nTotalRecived += Recived;

		if (nTotalRecived >= 4)
			nMsgLength = *(uint32*)&rCompileData[0] + Offset;

		if(Recived == 0 || nTotalRecived == nMsgLength)
		{
			rCompileData.resize(nTotalRecived);
			break;
		}
		if(Recived < 0)
		{
			iLog->LogError("ERROR: CShaderSrv::Compile:  error in recv() from remote server at offset %lu: error %li, sys_net_errno=%i\n",(unsigned long)rCompileData.size(),(long)Recived,WSAGetLastError());
			return ESRecvFailed;
		}
		CurrentPos	+=	Recived;
	}
	if(rCompileData[4]!=1)	//1==ECSJS_DONE state on server, dont change!
			return ESFailed;
//	iLog->Log("Recv = %d",(unsigned long)rCompileData.size() );
	if (rCompileData.size() > Offset)
	{
		memmove( &rCompileData[0],&rCompileData[Offset],rCompileData.size()-Offset );
		rCompileData.resize(rCompileData.size()-Offset);
	}
	return	rCompileData.size()>=Offset && 
					rCompileData.size()<SIZELIMIT?ESOK:ESFailed;
}

void CShaderSrv::Tokenize(tdEntryVec& rRet,const string& Tokens,const string& Separator)	const
{
		rRet.clear();
		string::size_type Pt;
		string::size_type Start	= 0;
		string::size_type SSize	=	Separator.size();

		while((Pt = Tokens.find(Separator,Start)) != string::npos)
		{
			string  SubStr	=	Tokens.substr(Start,Pt-Start);
			rRet.push_back(SubStr);
			Start = Pt + SSize;
		}

		rRet.push_back(Tokens.substr(Start));
}

EServerError CShaderSrv::Send(std::vector<uint8>& rCompileData)	const
{
	SOCKET Socket	=	SOCKET_ERROR;
	int Err = SOCKET_ERROR;

	tdEntryVec ServerVec;
	if(gRenDev->CV_r_ShaderCompilerServer)
		Tokenize(ServerVec,gRenDev->CV_r_ShaderCompilerServer->GetString(),";");

	if(ServerVec.empty())
		ServerVec.push_back("localhost");
	
	//connect
	for(uint32 nRetries=m_LastWorkingServer;nRetries<m_LastWorkingServer+ServerVec.size()+6;nRetries++)
	{
		string Server	=	ServerVec[nRetries%ServerVec.size()];
		Socket = socket(AF_INET, SOCK_STREAM, 0);
		if(Socket == INVALID_SOCKET)
		{
			iLog->LogError("ERROR: CShaderSrv::Compile: can't create client socket: error %i\n",Socket);
			return ESNetworkError;
		}
		struct sockaddr_in addr;
		memset(&addr, 0, sizeof addr);
		addr.sin_family = AF_INET;
		addr.sin_port = htons(gRenDev->CV_r_ShaderCompilerPort);
		const char* pHostName	=	Server.c_str();
		bool IP=true;
		for(uint32 a=0,Size=strlen(pHostName);a<Size;a++)
			IP&=(pHostName[a]>='0' && pHostName[a]<='9') || pHostName[a]=='.' ;
		if(IP)
			addr.sin_addr.s_addr = inet_addr(pHostName);
		else
		{
#if defined(XENON)
			XNDNS* pHost=0;
			XNetDnsLookup(pHostName,0,&pHost);
			for(int a=0;pHost->iStatus == WSAEINPROGRESS;a++)
			{
				if(a>900)
				iLog->Log("Waiting for DNS resolve, attempt: %d...\n",a);
				Sleep(20);
				if(a==2000)
				{
					closesocket(Socket);
					return ESNetworkError;
				}
			}
			if(pHost->iStatus==WSAHOST_NOT_FOUND)
			{
				iLog->Log("ERROR: could not find %s...\n",pHostName);
				closesocket(Socket);
				return ESNetworkError;
			}
			else
			if(pHost->iStatus != 0)
			{
				iLog->Log("ERROR: resolving DNS ended with error %d...\n",pHost->iStatus);
				closesocket(Socket);
				return ESNetworkError;

				// An error occurred.  One of the following:
				// 	pxndns->iStatus == WSAHOST_NOT_FOUND - No such host
				//    pxndns->iStatus == WSAETIMEDOUT - No response from DNS server(s)
			}
			XNetDnsRelease(pHost);
			addr.sin_addr = pHost->aina[0];
#else
			hostent* pHost	= gethostbyname( pHostName );
			if (!pHost)
			{
				break;
			}
			addr.sin_addr.s_addr = ((struct in_addr *)(pHost->h_addr))->s_addr;
#endif
		}

		Err = connect(Socket, (struct sockaddr *)&addr, sizeof addr);
		if(Err>=0)
		{
			m_LastWorkingServer=nRetries%ServerVec.size();
			break;
		}
		if(Err<0)
		{
			iLog->LogError("ERROR: CShaderSrv::Compile: could not connect to %s\n", Server.c_str());
			//iLog->LogError("ERROR: CShaderSrv::Compile: can't connect to cgserver: error %i, sys_net_errno=%i, retrying %d\n", Err, WSAGetLastError(),nRetries);
			//socketclose(s);
			//return (size_t)-1;
			struct timeval tv;
			struct fd_set emptySet;
			FD_ZERO(&emptySet);
			tv.tv_sec = 1;
			tv.tv_usec = 0;
#if defined(PS3)
			socketselect(1, &emptySet, &emptySet, &emptySet, &tv);
#endif
			closesocket(Socket);
			Socket = INVALID_SOCKET;
			//return ESNetworkError;
		}
	}

	if (Socket == INVALID_SOCKET)
	{
		iLog->LogError("ERROR: CShaderSrv::Compile: can't connect to cgserver: error %i, sys_net_errno=%i\n", Err, WSAGetLastError() );
		return ESNetworkError;
	}

	if(!Send(Socket,rCompileData))
	{
		closesocket(Socket);
		return ESSendFailed;
	}

	EServerError	Error	=	Recv(Socket,rCompileData);
	closesocket(Socket);
	if(Error!=ESOK)
		return ESRecvFailed;

	if (rCompileData.size() < 4)
		return ESFailed;
	
	// Decompress incoming shader data
	std::vector<uint8> rCompressedData;
	rCompressedData.swap(rCompileData);

	uint32 nSrcUncompressedLen = *(uint32*)&rCompressedData[0];
	SwapEndian(nSrcUncompressedLen);

	size_t nUncompressedLen = (size_t)nSrcUncompressedLen;

	rCompileData.resize(nUncompressedLen);
	if (nUncompressedLen > 1000000)
	{
		// Shader too big, something is wrong.
		return ESFailed;
	}
	if (nUncompressedLen > 0)
	{
		if (!gEnv->pSystem->DecompressDataBlock( &rCompressedData[4],rCompressedData.size()-4,&rCompileData[0],nUncompressedLen ))
			return ESFailed;
	}

	//if (rCompileData.size() == 0 || strncmp( (char*)&rCompileData[0],"[ERROR]",MIN(rCompileData.size(),7)) == 0)
	//	return ESFailed;
	
	return ESOK;
}

}

