#include "StdAfx.h"
#include "D3DColorGradingController.h"
#include "DriverD3D.h"
#include "D3DPostProcess.h"
#include <StringUtils.h>


const int COLORCHART_SIZE = 16;
#ifdef XENON
const int COLORCHART_ALIGNED_SIZE = 32;
#else
const int COLORCHART_ALIGNED_SIZE = 16;
#endif
const int COLORCHART_WIDTH = COLORCHART_SIZE * COLORCHART_SIZE;
const int COLORCHART_RENDERTARGET_WIDTH = COLORCHART_SIZE * COLORCHART_ALIGNED_SIZE;
const int COLORCHART_HEIGHT = COLORCHART_SIZE;
const uint32 COLORCHART_TEXFLAGS = FT_NOMIPS |  FT_DONT_STREAM | FT_DONT_RESIZE | FT_STATE_CLAMP;
const char* COLORCHART_DEF_TEX = "textures/defaults/default_cch.tif";

#define COLORCHART_FORMAT eTF_A8R8G8B8

CColorGradingControllerD3D::CColorGradingControllerD3D(CD3D9Renderer* pRenderer)
: m_layers()
, m_pRenderer(pRenderer)
, m_pSlicesVB(NULL)
, m_pChartIdentity(NULL)
, m_pChartStatic(NULL)
, m_pChartToUse(NULL)
, m_p3DColorChart(NULL)
{
	assert(m_pRenderer);
	m_pMergeLayers[0] = NULL;
	m_pMergeLayers[1] = NULL;

#ifdef XENON	// work-around
	for ( UINT i = 0; i < COLORCHART_SIZE; ++i )
		m_p3DColorChartAux[i] = NULL;
#endif
}


CColorGradingControllerD3D::~CColorGradingControllerD3D()
{
	SAFE_RELEASE(m_pChartIdentity);
	SAFE_RELEASE(m_pChartStatic);
	SAFE_RELEASE(m_pMergeLayers[0]);
	SAFE_RELEASE(m_pMergeLayers[1]);
	SAFE_DELETE(m_pSlicesVB);
	if(m_p3DColorChart == m_pMergeLayers[0] || m_p3DColorChart == m_pMergeLayers[1])
		m_p3DColorChart = NULL;
	SAFE_RELEASE(m_p3DColorChart);
#ifdef XENON	// work-around
	for ( UINT i = 0; i < COLORCHART_SIZE; ++i )
	{
		if(m_p3DColorChartAux[i])
			XGOffsetBaseTextureAddress( m_p3DColorChartAux[i], NULL, NULL );	// don't release shared memory
		SAFE_DELETE(m_p3DColorChartAux[i]);
	}
#endif
}


bool CColorGradingControllerD3D::ValidateColorChart(const CTexture* pChart) const
{
  if (!CTexture::IsTextureExist(pChart))
		return false;

	if (pChart->IsNoTexture())
		return false;

	if (pChart->GetWidth() != COLORCHART_WIDTH || pChart->GetHeight() != COLORCHART_HEIGHT)
		return false;

	return true;
}


CTexture* CColorGradingControllerD3D::LoadColorChartInt(const char* pChartFilePath) const
{
	if (!pChartFilePath || !pChartFilePath[0])
		return 0;

	CTexture* pChart = (CTexture*) m_pRenderer->EF_LoadTexture(pChartFilePath, COLORCHART_TEXFLAGS, eTT_2D);
	if (!ValidateColorChart(pChart))
	{
		SAFE_RELEASE(pChart);
		return 0;
	}
	return pChart;
}


int CColorGradingControllerD3D::LoadColorChart(const char* pChartFilePath) const
{
	CTexture* pChart = LoadColorChartInt(pChartFilePath);
	return pChart ? pChart->GetID() : -1;
}


int CColorGradingControllerD3D::LoadDefaultColorChart() const
{
	CTexture* pChartIdentity = LoadColorChartInt(COLORCHART_DEF_TEX);
	return pChartIdentity ? pChartIdentity->GetID() : -1;
}


void CColorGradingControllerD3D::UnloadColorChart(int texID) const
{
	CTexture* pChart = CTexture::GetByID(texID);
	SAFE_RELEASE(pChart);
}


void CColorGradingControllerD3D::SetLayers(const SColorChartLayer* pLayers, uint32 numLayers)
{
	gRenDev->m_pRT->RC_CGCSetLayers(this, pLayers, numLayers);
}


void CColorGradingControllerD3D::RT_SetLayers(const SColorChartLayer* pLayerInfo, uint32 numLayers)
{
	m_layers.reserve(numLayers);
	m_layers.resize(0);

	if (numLayers)
	{
		float blendSum = 0;
		for (size_t i=0; i<numLayers; ++i)
		{
			const SColorChartLayer& l = pLayerInfo[i];
			if (l.m_texID > 0 && l.m_blendAmount > 0)
			{
				const CTexture* pChart = CTexture::GetByID(l.m_texID);
				if (ValidateColorChart(pChart))
				{
					m_layers.push_back(l);
					blendSum += l.m_blendAmount;
				}
			}
		}

		const size_t numActualLayers = m_layers.size();
		if (numActualLayers)
		{
			if (numActualLayers > 1)
			{
				float normalizeBlendAmount = (float) (1.0 / (double) blendSum);
				for (size_t i=0; i<numActualLayers; ++i)
					m_layers[i].m_blendAmount *= normalizeBlendAmount;
			}
			else
				m_layers[0].m_blendAmount = 1;
		}
	}
}


bool CColorGradingControllerD3D::InitResources()
{
	if (!m_pChartIdentity)
	{
		m_pChartIdentity = LoadColorChartInt(COLORCHART_DEF_TEX);
		if (!m_pChartIdentity)
		{
			if(m_pRenderer->GetFrameID() % 300 == 0)
				iLog->LogError("Failed to initialize Color Grading: Default color chart is missing");
			return false;
		}
	}

	if (!m_pMergeLayers[0])
	{
		// AntonK: it's not possible to render into RT with the pitch less than 1KB on PS3
		COMPILE_TIME_ASSERT(COLORCHART_RENDERTARGET_WIDTH * sizeof(uint32) >= 1024);
		m_pMergeLayers[0] = CTexture::CreateRenderTarget("ColorGradingMergeLayer0", COLORCHART_RENDERTARGET_WIDTH, COLORCHART_HEIGHT, eTT_2D, COLORCHART_TEXFLAGS, COLORCHART_FORMAT);
    if (!CTexture::IsTextureExist(m_pMergeLayers[0]))
			return false;
	}

	if (!m_pMergeLayers[1])
	{
		m_pMergeLayers[1] = CTexture::CreateRenderTarget("ColorGradingMergeLayer1", COLORCHART_RENDERTARGET_WIDTH, COLORCHART_HEIGHT, eTT_2D, COLORCHART_TEXFLAGS, COLORCHART_FORMAT);
    if (!CTexture::IsTextureExist(m_pMergeLayers[1]))
			return false;
	}

	Init3DTexture();

	if(!m_pSlicesVB)
	{
		assert(m_vecSlicesData.empty());
		m_vecSlicesData.resize(6 * COLORCHART_SIZE);

		const float fQuadSize = (float)COLORCHART_SIZE / COLORCHART_RENDERTARGET_WIDTH;
		const float fTexCoordSize = 1.f / COLORCHART_SIZE;
		for(uint32 iQuad = 0;iQuad < COLORCHART_SIZE;++iQuad)
		{
			const float fQuadShift = (float)iQuad / COLORCHART_SIZE;
			const float fBlue = (float)iQuad / (COLORCHART_SIZE - 1.f);

			m_vecSlicesData[iQuad*6+0].xyz =	Vec3( fQuadShift + fQuadSize,			1.f, 0.f);
			m_vecSlicesData[iQuad*6+0].st =		Vec2( fQuadShift + fTexCoordSize,	1.f);
			m_vecSlicesData[iQuad*6+0].color.dcolor = ColorF(1.f, 1.f, fBlue).pack_argb8888();
			m_vecSlicesData[iQuad*6+1].xyz =	Vec3( fQuadShift + fQuadSize,			0.f, 0.f);
			m_vecSlicesData[iQuad*6+1].st =		Vec2( fQuadShift + fTexCoordSize,	0.f);
			m_vecSlicesData[iQuad*6+1].color.dcolor = ColorF(1.f, 0.f, fBlue).pack_argb8888();
			m_vecSlicesData[iQuad*6+2].xyz =	Vec3( fQuadShift + 0.f,						1.f, 0.f);
			m_vecSlicesData[iQuad*6+2].st =		Vec2( fQuadShift + 0.f,						1.f);
			m_vecSlicesData[iQuad*6+2].color.dcolor = ColorF(0.f, 1.f, fBlue).pack_argb8888();

			m_vecSlicesData[iQuad*6+3].xyz =	Vec3( fQuadShift + 0.f,						1.f, 0.f);
			m_vecSlicesData[iQuad*6+3].st =		Vec2( fQuadShift + 0.f,						1.f);
			m_vecSlicesData[iQuad*6+3].color.dcolor = ColorF(0.f, 1.f, fBlue).pack_argb8888();
			m_vecSlicesData[iQuad*6+4].xyz =	Vec3( fQuadShift + fQuadSize,			0.f, 0.f);
			m_vecSlicesData[iQuad*6+4].st =		Vec2( fQuadShift + fTexCoordSize,	0.f);
			m_vecSlicesData[iQuad*6+4].color.dcolor = ColorF(1.f, 0.f, fBlue).pack_argb8888();
			m_vecSlicesData[iQuad*6+5].xyz =	Vec3( fQuadShift + 0.f,						0.f, 0.f);
			m_vecSlicesData[iQuad*6+5].st =		Vec2( fQuadShift + 0.f,						0.f);
			m_vecSlicesData[iQuad*6+5].color.dcolor = ColorF(0.f, 0.f, fBlue).pack_argb8888();
		}

		m_pSlicesVB = new CVertexBuffer(&m_vecSlicesData[0], eVF_P3F_C4B_T2F, 6 * COLORCHART_SIZE);
	}

	return true;
}

void CColorGradingControllerD3D::Init3DTexture()
{
	if(m_p3DColorChart && m_p3DColorChart->GetTexType() == eTT_3D)
		return;

	// creating 3D texture
#if defined(XENON) || defined(PS3)
	static const uint32 flags =  FT_DONT_STREAM | FT_USAGE_RENDERTARGET | FT_DONT_RESIZE | FT_NOMIPS;
	m_p3DColorChart = CTexture::Create3DTexture("$ColorChart3D", COLORCHART_SIZE, COLORCHART_SIZE, COLORCHART_SIZE, 
																								1, flags, NULL, COLORCHART_FORMAT, COLORCHART_FORMAT);
	assert(m_p3DColorChart);
	m_p3DColorChart->SetCustomID(TO_COLORCHART);
#endif

#ifdef XENON	// work-around
	CDeviceTexture* pVolTex = m_p3DColorChart->GetDevTexture();
	DWORD dwBaseAddress = pVolTex->GetVolumeTexture()->Format.BaseAddress << GPU_TEXTURE_ADDRESS_SHIFT;
	for ( UINT iSlice = 0; iSlice < COLORCHART_SIZE; ++iSlice )
	{
		m_p3DColorChartAux[iSlice] = new D3DVolumeTexture;
		XGSetVolumeTextureHeader( 32, 32, 32, 1, 0, 
															CTexture::DeviceFormatFromTexFormat(COLORCHART_FORMAT), D3DPOOL_DEFAULT, 0, 0, 
															m_p3DColorChartAux[iSlice], NULL, NULL );
		XGOffsetBaseTextureAddress( m_p3DColorChartAux[iSlice], ( VOID* )( dwBaseAddress - iSlice * GPU_TEXTURE_ALIGNMENT ), NULL );
	}
#elif defined(PS3)
	assert(m_pMergeLayers[0] && m_pMergeLayers[1]);
	// restructure data
	m_pMergeLayers[0]->GetDevTexture()->Get2DTexture()->DisableSwizzling();
	m_pMergeLayers[1]->GetDevTexture()->Get2DTexture()->DisableSwizzling();
	m_pMergeLayers[0]->GetDevTexture()->Get2DTexture()->Mode(ECDXPSTCF_FRAMEBUFFER);
	m_pMergeLayers[1]->GetDevTexture()->Get2DTexture()->Mode(ECDXPSTCF_FRAMEBUFFER);
	m_p3DColorChart->GetDevTexture()->GetVolumeTexture()->DisableSwizzling();
	m_p3DColorChart->GetDevTexture()->GetVolumeTexture()->ReleaseResources();
#endif
}


bool CColorGradingControllerD3D::Update(const SColorGradingMergeParams *pMergeParams)
{
	m_pChartToUse = 0;

	if (!m_pRenderer->CV_r_colorgrading_charts)
		return true;

	if (m_pChartStatic)
	{
		m_pChartToUse = m_pChartStatic;
		return true;
	}

	if (!InitResources())
	{
		m_pChartToUse = m_pChartIdentity;
		return m_pChartToUse != 0;
	}

	gRenDev->m_cEF.mfRefreshSystemShader("PostEffectsGame", CShaderMan::m_shPostEffectsGame);

	static const int texStatePntID = CTexture::GetTexState(STexState(FILTER_POINT, true));
	static const int texStateLinID = CTexture::GetTexState(STexState(FILTER_LINEAR, true));

	// merge layers
	const size_t numLayers = m_layers.size();
	if (!numLayers)
	{
		m_pChartToUse = m_pChartIdentity;
	}
	else
	{
		CTexture* pNewMergeResult = m_pMergeLayers[0];
		m_pRenderer->FX_PushRenderTarget(0, pNewMergeResult, 0);

		int numMergePasses = 0;
		for (size_t curLayer=0; curLayer<numLayers; )
		{
			size_t mergeLayerIdx[4] = {-1, -1, -1, -1};
			int numLayersPerPass = 0;

			for (; curLayer<numLayers && numLayersPerPass<4; ++curLayer)
			{
				if (m_layers[curLayer].m_blendAmount > 0.001f)
					mergeLayerIdx[numLayersPerPass++] = curLayer;
			}

			if (numLayersPerPass)
			{
				const uint64 nResetFlags = ~(g_HWSR_MaskBit[HWSR_SAMPLE0]|g_HWSR_MaskBit[HWSR_SAMPLE1]|g_HWSR_MaskBit[HWSR_SAMPLE2]);
				m_pRenderer->m_RP.m_FlagsShader_RT &= nResetFlags;
				if ((numLayersPerPass-1) & 1)
					gRenDev->m_RP.m_FlagsShader_RT |= g_HWSR_MaskBit[HWSR_SAMPLE0];
				if ((numLayersPerPass-1) & 2)
					gRenDev->m_RP.m_FlagsShader_RT |= g_HWSR_MaskBit[HWSR_SAMPLE1];
				// if it's the final pass and we have a 3D texture to render into
				if(!pMergeParams && m_p3DColorChart)
					gRenDev->m_RP.m_FlagsShader_RT |= g_HWSR_MaskBit[HWSR_SAMPLE2];

				CShader* pSh = CShaderMan::m_shPostEffectsGame;

				static CCryNameTSCRC techName("MergeColorCharts");
				SD3DPostEffectsUtils::ShBeginPass(pSh, techName, FEF_DONTSETTEXTURES | FEF_DONTSETSTATES);

				Vec4 layerBlendAmount(0, 0, 0, 0);
				for (int i=0; i<numLayersPerPass; ++i)
				{
					const SColorChartLayer& l = m_layers[mergeLayerIdx[i]];
					CTexture* pChart = CTexture::GetByID(l.m_texID);
					pChart->Apply(i, texStatePntID);
					layerBlendAmount[i] = l.m_blendAmount;
				}

				static CCryName semLayerBlendAmount("LayerBlendAmount");
				SD3DPostEffectsUtils::ShSetParamPS(semLayerBlendAmount, layerBlendAmount);

				m_pRenderer->EF_SetState(GS_NODEPTHTEST | (numMergePasses ? GS_BLSRC_ONE | GS_BLDST_ONE : 0));
				m_pRenderer->SetCullMode(R_CULL_NONE);
				gcpRendD3D->DrawPrimitives(m_pSlicesVB, COLORCHART_SIZE * 6, R_PRIMV_TRIANGLES);

				SD3DPostEffectsUtils::ShEndPass();
				++numMergePasses;

				m_pRenderer->m_RP.m_FlagsShader_RT &= nResetFlags;
			}

			
		}

		m_pChartToUse = numMergePasses ? pNewMergeResult : m_pChartIdentity;

		// do a final resolve to 3D texture
#ifdef PS3
		if(!pMergeParams)
#endif
			ResolveTo3DTexture(pNewMergeResult);

		m_pRenderer->FX_PopRenderTarget(0);
	}

	// combine merged layers with color grading stuff
	if (m_pChartToUse && pMergeParams)
	{
		uint64 nSaveFlagsShader_RT = gRenDev->m_RP.m_FlagsShader_RT;		
		gRenDev->m_RP.m_FlagsShader_RT = pMergeParams->nFlagsShaderRT;
		gRenDev->m_RP.m_FlagsShader_RT &= ~g_HWSR_MaskBit[HWSR_SAMPLE1];

		CTexture* pNewMergeResult = m_pMergeLayers[1];
		m_pRenderer->FX_PushRenderTarget(0, pNewMergeResult, 0); 
		CShader* pSh = CShaderMan::m_shPostEffectsGame;

		// if it's the final pass and we have a 3D texture to render into
		if(m_p3DColorChart && numLayers)
			gRenDev->m_RP.m_FlagsShader_RT |= g_HWSR_MaskBit[HWSR_SAMPLE1];

		static CCryNameTSCRC techName("CombineColorGradingWithColorChart");
		SD3DPostEffectsUtils::ShBeginPass(pSh, techName, FEF_DONTSETTEXTURES | FEF_DONTSETSTATES);

#ifndef PS3
		if(m_p3DColorChart && numLayers)
			m_p3DColorChart->Apply(0, texStateLinID);
		else
#endif
			m_pChartToUse->Apply(0, texStateLinID);

		static CCryName pParamName0("ColorGradingParams0");
		static CCryName pParamName1("ColorGradingParams1");
		static CCryName pParamName2("ColorGradingParams2");
		static CCryName pParamName3("ColorGradingParams3");
		static CCryName pParamName4("ColorGradingParams4");
		static CCryName pParamMatrix("mColorGradingMatrix");

		// clamp negative values as that causes blackness
		Vec4 cMatrix[3];
		cMatrix[0] = pMergeParams->pColorMatrix[0];
		cMatrix[1] = pMergeParams->pColorMatrix[1];
		cMatrix[2] = pMergeParams->pColorMatrix[2];
		cMatrix[0].w = clamp_tpl(cMatrix[0].w, 0.f, 1.f);
		cMatrix[1].w = clamp_tpl(cMatrix[1].w, 0.f, 1.f);
		cMatrix[2].w = clamp_tpl(cMatrix[2].w, 0.f, 1.f);

		pSh->FXSetPSFloat(pParamName0, &pMergeParams->pLevels[0], 1);
		pSh->FXSetPSFloat(pParamName1, &pMergeParams->pLevels[1], 1);
		pSh->FXSetPSFloat(pParamName2, &pMergeParams->pFilterColor, 1);
		pSh->FXSetPSFloat(pParamName3, &pMergeParams->pSelectiveColor[0], 1);
		pSh->FXSetPSFloat(pParamName4, &pMergeParams->pSelectiveColor[1], 1);
		pSh->FXSetPSFloat(pParamMatrix, &cMatrix[0], 3);

		m_pRenderer->EF_SetState(GS_NODEPTHTEST);
		m_pRenderer->SetCullMode(R_CULL_NONE);
		gcpRendD3D->DrawPrimitives(m_pSlicesVB, COLORCHART_SIZE * 6, R_PRIMV_TRIANGLES);

		SD3DPostEffectsUtils::ShEndPass();

		m_pChartToUse = pNewMergeResult;

		// do a final resolve to 3D texture
		ResolveTo3DTexture(pNewMergeResult);

		m_pRenderer->FX_PopRenderTarget(0);

		gRenDev->m_RP.m_FlagsShader_RT = nSaveFlagsShader_RT;
	}

	return m_pChartToUse != 0;
}

void CColorGradingControllerD3D::ResolveTo3DTexture(CTexture* pRT)
{
	if(!m_p3DColorChart || m_p3DColorChart->GetTexType() != eTT_3D)
		return;
	PROFILE_LABEL_PUSH( "IV_RESOLVE_3D" );
#	ifdef XENON
	for(int iSlice = 0;iSlice < COLORCHART_SIZE;++iSlice)
	{
		D3DRECT srcRect = { iSlice * COLORCHART_ALIGNED_SIZE, 0, iSlice * COLORCHART_ALIGNED_SIZE + COLORCHART_SIZE, COLORCHART_SIZE };
		gcpRendD3D->m_pd3dDevice->Resolve(D3DRESOLVE_RENDERTARGET0, &srcRect, m_p3DColorChartAux[iSlice], NULL, 0, iSlice, NULL, 0.f, 0x00, NULL);
	}
#elif defined(PS3)
	// bind volume texture to the same texture
	CDeviceTexture* pTex = pRT->GetDevTexture();
	CDeviceTexture* pVolTex = m_p3DColorChart->GetDevTexture();
	pVolTex->GetVolumeTexture()->MemItemID(pTex->Get2DTexture()->MemItemID());
	pVolTex->GetVolumeTexture()->Offset(pTex->Get2DTexture()->GcmTexture()->offset);
#	endif	// XENON || PS3

	// avoid redundant resolves
	pRT->SetResolved(true);

	PROFILE_LABEL_POP( "IV_RESOLVE_3D" );
}

CTexture* CColorGradingControllerD3D::GetColorChart() const
{
	if(m_p3DColorChart && (m_pChartToUse == m_pMergeLayers[0] || m_pChartToUse == m_pMergeLayers[1]))
		return m_p3DColorChart;
	return m_pChartToUse;
}

void CColorGradingControllerD3D::DrawLayer(float x, float y, float w, float h, CTexture* pChart, float blendAmount, const char* pLayerName) const
{
	CShader* pSh = CShaderMan::m_shPostEffectsGame;

	gRenDev->m_RP.m_FlagsShader_RT &= ~g_HWSR_MaskBit[HWSR_SAMPLE0];

	// If using merged color grading with color chart disable regular color transformations in display - only need to use color chart
	if( pChart && pChart->GetTexType() == eTT_3D)
		gRenDev->m_RP.m_FlagsShader_RT |= g_HWSR_MaskBit[HWSR_SAMPLE0];

	static CCryNameTSCRC techName("DisplayColorCharts");
	SD3DPostEffectsUtils::ShBeginPass(pSh, techName, FEF_DONTSETTEXTURES | FEF_DONTSETSTATES);

	static const int texStateID = CTexture::GetTexState(STexState(FILTER_POINT, true));
	if(pChart)
		pChart->Apply(0, texStateID);

	int offs;
	SVF_P3F_C4B_T2F* pVerts = (SVF_P3F_C4B_T2F*) m_pRenderer->GetVBPtr(4, offs, POOL_P3F_COL4UB_TEX2F);
#ifdef XENON
	SVF_P3F_C4B_T2F sysVerts[4];
	SVF_P3F_C4B_T2F* pDst = pVerts;
	pVerts = sysVerts;
#endif
	pVerts[0].xyz = Vec3(x, y, 0);
	pVerts[0].st = Vec2(0, 1);

	pVerts[1].xyz = Vec3(x+w, y, 0);
	pVerts[1].st = Vec2(1, 1);

	pVerts[2].xyz = Vec3(x, y+h, 0);
	pVerts[2].st = Vec2(0, 0);

	pVerts[3].xyz = Vec3(x+w, y+h, 0);
	pVerts[3].st = Vec2(1, 0);
#ifdef XENON
	memcpy(pDst, sysVerts, 4 * sizeof(SVF_P3F_C4B_T2F));
#endif
	m_pRenderer->UnlockVB(POOL_P3F_COL4UB_TEX2F);

	m_pRenderer->FX_Commit();
	m_pRenderer->EF_SetState(GS_NODEPTHTEST);
	m_pRenderer->SetCullMode(R_CULL_NONE);

	m_pRenderer->FX_SetVStream(0, m_pRenderer->m_pVB[POOL_P3F_COL4UB_TEX2F], 0, sizeof(SVF_P3F_C4B_T2F));
	if (!FAILED(m_pRenderer->FX_SetVertexDeclaration(0, eVF_P3F_C4B_T2F)))
	{
#if defined (DIRECT3D9) || defined (OPENGL)
		m_pRenderer->m_pd3dDevice->DrawPrimitive(D3DPT_TRIANGLESTRIP, offs, 2);
#elif defined (DIRECT3D10)
		m_pRenderer->SetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
		m_pRenderer->m_pd3dDeviceContext->Draw(4, offs);
#endif

		m_pRenderer->m_RP.m_PS[m_pRenderer->m_RP.m_nProcessThreadID].m_nPolygons[EFSLIST_GENERAL] += 2;
		m_pRenderer->m_RP.m_PS[m_pRenderer->m_RP.m_nProcessThreadID].m_nDIPs[EFSLIST_GENERAL]++;
	}

	SD3DPostEffectsUtils::ShEndPass();

	float color[4] = {1, 1, 1, 1};
	m_pRenderer->Draw2dLabel(x + w + 10.0f, y, 1.35f, color, false, "%2.1f%%", blendAmount * 100.0f);
	m_pRenderer->Draw2dLabel(x + w + 55.0f, y, 1.35f, color, false, "%s", pLayerName);
}


void CColorGradingControllerD3D::DrawDebugInfo() const
{
	if (m_pRenderer->CV_r_colorgrading_charts < 2)
		return;

	m_pRenderer->Set2DMode(true, m_pRenderer->GetWidth(), m_pRenderer->GetHeight());

	const float w = (float) COLORCHART_WIDTH;
	const float h = (float) COLORCHART_HEIGHT;

	float x = 16.0f;
	float y = 16.0f;

	if (!m_pChartStatic)
	{
		for (size_t i=0, numLayers=m_layers.size(); i<numLayers; ++i)
		{
			const SColorChartLayer& l = m_layers[i];
			CTexture* pChart = CTexture::GetByID(l.m_texID);
			DrawLayer(x, y, w, h, pChart, l.m_blendAmount, CryStringUtils::FindFileNameInPath(pChart->GetName()));
			y += h + 4;
		}
		if (GetColorChart())
			DrawLayer(x, y, w, h, GetColorChart(), 1, "FinalChart");
	}
	else
		DrawLayer(x, y, w, h, m_pChartStatic, 1, CryStringUtils::FindFileNameInPath(m_pChartStatic->GetName()));

	m_pRenderer->RT_FlushTextMessages();

	m_pRenderer->Set2DMode(false, 0, 0 );
}


bool CColorGradingControllerD3D::LoadStaticColorChart(const char* pChartFilePath)
{
	SAFE_RELEASE(m_pChartStatic);
	if (pChartFilePath && pChartFilePath[0] != '\0')
	{
		m_pChartStatic = LoadColorChartInt(pChartFilePath);
		return m_pChartStatic != 0;
	}
	return true;
}


const CTexture* CColorGradingControllerD3D::GetStaticColorChart() const
{
	return m_pChartStatic;
}
