/*=============================================================================
DeferredShading.h : Deferred shading pipeline
Copyright (c) 2001 Crytek Studios. All Rights Reserved.

=============================================================================*/

#ifndef _DEFERREDSHADING_H_
#define _DEFERREDSHADING_H_

struct IVisArea;

enum EDecalType
{
  DTYP_DARKEN,
  DTYP_BRIGHTEN,
  DTYP_ALPHABLEND,
  DTYP_ALPHABLEND_AND_BUMP,
  DTYP_DARKEN_LIGHTBUF,
  DTYP_NUM,
};

class CDeferredShading
{
public:

  static CDeferredShading &Instance()
  {
    return m_pInstance;
  }

  void Render();  
  void SetupPasses();
  void SetupGlobalConsts();

  bool AmbientPass(const CDLight *pGlobalCubemap);
  bool DrawAmbientIndexed();
  bool DeferredDecalPass( SDeferrredDecal& rDecal, EDecalType decalType);
  void DrawDeferredDecals(EDecalType decalType);
  void SunLightPasses(const CDLight& light, const int nLightID);
  bool ShadowLightPasses(const CDLight& light, const int nLightID);
  void DrawDecalVolume(const SDeferrredDecal& rDecal, Matrix44& mDecalLightProj);
  void DrawLightVolume(const CDLight *pDL);
  void LightStencilPrePass( const CDLight *pDL );
  void LightPass( const CDLight *pDL, bool bForceStencilDisable = false);
	void DeferredCubemaps( const TArray<CDLight>& rCubemaps, const uint32 nStartIndex = 0 );
	void DeferredCubemapPass( const CDLight *pDL );
	void DeferredLights( const TArray<CDLight>& rLights );
	void NegativeLightPass( const CDLight *pDL );
	void ComputeSSGI(CTexture* pSSGITarget, bool bMergeWithGI);

  void CreateDeferredMaps();
  void DestroyDeferredMaps();
  void Release();
  void Debug();

  uint32 AddLight( const CDLight &pDL, float fMult );
  inline uint32 AddVisArea( const IVisArea *pVisArea );
	inline void ResetLights();
	inline void ResetVisAreas();

	// called in between levels to free up memory
	void ReleaseData();

	uint32 GetVisAreaID( uint32 nThreadID, const IVisArea *pVisArea );

  TArray<CDLight>& GetLights(const int nThreadID, const int nCurRecLevel, const eDeferredLightType eType = eDLT_DeferredLight);
  uint32 GetLightsNum(const eDeferredLightType eType);

  inline uint32 GetLightsCount() const 
  {
    return m_nLightsProcessedCount;
  }

  inline Vec4 GetLightDepthBounds( const CDLight *pDL ) const
  {
    if( !CRenderer::CV_r_deferredshadingdepthboundstest ) 
      return Vec4(0.0f, 0.0f, 1.0f, 1.0f);

    float fMinZ = 0.0f, fMaxZ = 1.0f;
    float fMinW = 0.0f, fMaxW = 1.0f;

    Vec3 pBounds = m_pCamFront * pDL->m_fRadius; 
    Vec3 pMax = pDL->m_Origin - pBounds;
    Vec3 pMin = pDL->m_Origin + pBounds;

    fMinZ = m_mViewProj.m20 * pMin.x + m_mViewProj.m21 * pMin.y + m_mViewProj.m22 * pMin.z + m_mViewProj.m23;
    fMinW = m_mViewProj.m30 * pMin.x + m_mViewProj.m31 * pMin.y + m_mViewProj.m32 * pMin.z + m_mViewProj.m33;
    fMinZ = iszero(fMinW)? 1.0f: fMinZ / fMinW;
    if( fMinW < 0.0f)
      fMinZ = 0.0f; 

    fMinZ = max( fMinZ, 0.01f );

    fMaxZ = m_mViewProj.m20 * pMax.x + m_mViewProj.m21 * pMax.y + m_mViewProj.m22 * pMax.z + m_mViewProj.m23;
    fMaxW = m_mViewProj.m30 * pMax.x + m_mViewProj.m31 * pMax.y + m_mViewProj.m32 * pMax.z + m_mViewProj.m33; 
    fMaxZ = iszero(fMaxW)? 1.0f: fMaxZ / fMaxW;
    if( fMaxW < 0.0f )
      fMaxZ = 0.0f;

    return Vec4( fMinZ, max(fMinW /*/ m_fCamFar*/, 0.000001f ), fMaxZ, max(fMaxW/*/ m_fCamFar*/, 0.000001f ));
  }


	const Matrix44A& GetCameraProjMatrix() const { return m_mViewProj; }

private:

  CDeferredShading()
  {
    m_pShader = 0;
    m_pTechName = "DeferredLightPass";
		m_pAmbientTechName = "AmbientPass";
    m_pCubemapsTechName = "DeferredCubemapPass";
    m_pDebugTechName = "Debug";
		m_pDeferredDecalTechName = "DeferredDecal";
    m_pLightVolumeTechName = "DeferredLightVolume";
		m_pDeferredSceneTechName = "DeferredScenePass";
		
    m_pParamPalette = "g_palette";
    m_pParamLightPos = "g_LightPos";
    m_pParamCameraMatrix = "g_mCamera";
    m_pParamLightProjMatrix = "g_mLightProj";
    m_pParamLightProjParams = "g_LightProjParams";
    m_pGeneralParams = "g_GeneralParams";    
    m_pParamLightDiffuse = "g_LightDiffuse";        
    m_pParamInvViewProj = "g_mInvViewProj";

    m_pDiffuseAccRT = CTexture::s_ptexCurrentSceneDiffuseAccMap;
		m_pSpecularAccRT = CTexture::s_ptexSceneSpecularAccMap;
		m_pNormalsRT = CTexture::s_ptexSceneNormalsMap;
    m_pDepthRT = CTexture::s_ptexZTarget;
		m_pSceneTexturesRT = CTexture::s_ptexSceneTexturesMap;

    m_pDebugRT = 0;
    m_nLightsProcessedCount = 0;

		m_pSSAORT = NULL;

    m_nRenderState = GS_BLSRC_ONE|GS_BLDST_ONE;
		
		for(int i=0; i<RT_COMMAND_BUF_COUNT; ++i)
			for(int j=0; j<MAX_REND_RECURSION_LEVELS; ++j)
		    m_nVisAreasCount[i][j] = 0;
  }

  ~CDeferredShading()
  {
    Release();
  }

  // Allow disable mrt usage: for double zspeed and on other passes less fillrate hit
  void SpecularAccEnableMRT( bool bEnable );

private:

  // Vis areas for current view 
  typedef std::map< const IVisArea *, uint32 > VisAreaIDMap;    
  typedef VisAreaIDMap::iterator VisAreaIDMapItor;
  VisAreaIDMap m_pVisAreas[RT_COMMAND_BUF_COUNT][MAX_REND_RECURSION_LEVELS];
  uint32 m_nVisAreasCount[RT_COMMAND_BUF_COUNT][MAX_REND_RECURSION_LEVELS];

  // Deferred passes common 
	TArray<CDLight> m_pLights[eDLT_NumLightTypes][RT_COMMAND_BUF_COUNT][MAX_REND_RECURSION_LEVELS];

  Vec3 m_pCamPos;
  Vec3 m_pCamFront;
  float m_fCamFar;
  float m_fCamNear;

  float m_fRatioWidth;
  float m_fRatioHeight;

  CShader  *m_pShader; 
  CCryNameTSCRC m_pDeferredDecalTechName;
  CCryNameTSCRC m_pLightVolumeTechName;
  CCryNameTSCRC m_pTechName;
	CCryNameTSCRC m_pAmbientTechName;
	CCryNameTSCRC m_pCubemapsTechName;
  CCryNameTSCRC m_pDebugTechName;
	CCryNameTSCRC m_pDeferredSceneTechName;
  CCryName m_pParamLightPos;
  CCryName m_pParamPalette;
  CCryName m_pParamCameraMatrix;
  CCryName m_pParamLightDiffuse;        
  CCryName m_pParamInvViewProj;
  CCryName m_pParamLightProjMatrix;
  CCryName m_pParamLightProjParams;
  CCryName m_pGeneralParams;

  Matrix44A m_mViewProj;
  Matrix44A m_pViewProjI;
  Matrix44A m_pView;  

  Vec4 vWorldBasisX, vWorldBasisY, vWorldBasisZ;

  CTexture *m_pDiffuseAccRT;
	CTexture *m_pSpecularAccRT;
  CTexture *m_pNormalsRT;
  CTexture *m_pDepthRT;
	CTexture *m_pSceneTexturesRT;

  CTexture *m_pDebugRT;

  int m_nRenderState;
  uint32 m_nLightsProcessedCount;

  short m_nCurTargetWidth;
  short m_nCurTargetHeight;

  static CDeferredShading m_pInstance;
public:
	SDynTexture* m_pSSAORT;
};

#endif
