/*=============================================================================
Copyright (c) 2010 Crytek Studios. All Rights Reserved.

Revision history:
* Created by Nicolas Schulz

=============================================================================*/

#include "StdAfx.h"
#include "D3DStereo.h"
#include "DriverD3D.h"
#include "D3DPostProcess.h"

CD3DStereoRenderer::CD3DStereoRenderer(CD3D9Renderer& renderer)
  : m_renderer(renderer)
  , m_mode(STEREO_MODE_NO_STEREO)
  , m_output(STEREO_OUTPUT_NONE)
  , m_pHighResFrontBuffer(NULL)
{
}

CD3DStereoRenderer::~CD3DStereoRenderer()
{
	// temporary workaround for a crash
	// DestroyResources();
}

void CD3DStereoRenderer::PrepareStereo( StereoMode mode, StereoOutput output )
{
	if( m_mode != mode || m_output != output )
	{
		m_mode = mode;
		m_output = output;
		RecreateResources();
	}
}

bool CD3DStereoRenderer::NeedsToCreateResources() const
{
	if (CTexture::s_ptexStereoL == NULL)
		return true;

	if (CTexture::s_ptexStereoR == NULL)
		return true;

	int nW = m_renderer.GetWidth(); //m_d3dsdBackBuffem.Width;
	int nH = m_renderer.GetHeight(); //m_d3dsdBackBuffem.Height;

	if (CTexture::s_ptexStereoL->GetWidth() != nW || CTexture::s_ptexStereoL->GetHeight() != nH)
		return true;
	if (CTexture::s_ptexStereoR->GetWidth() != nW || CTexture::s_ptexStereoR->GetHeight() != nH)
		return true;

	return false;
}

void CD3DStereoRenderer::RecreateResources()
{
	if (!IsStereoEnabled())
		return;

	if (NeedsToCreateResources())
	{
		DestroyResources();
		CreateResources();
	}
}

void CD3DStereoRenderer::Update()
{
	if( CRenderer::CV_r_StereoMode == 0 && m_mode == 0 )  // Completely ignore stereo if it is and was disabled before
		return;
	
	m_renderer.m_pRT->RC_PrepareStereo(CRenderer::CV_r_StereoMode, CRenderer::CV_r_StereoOutput);
}

void CD3DStereoRenderer::ProcessScene(int sceneFlags)
{
	int nThreadID = m_renderer.m_RP.m_nFillThreadID;

	if (CRenderer::CV_r_StereoMode == STEREO_MODE_DUAL_RENDERING)
	{
		CCamera cam = m_renderer.m_RP.m_TI[nThreadID].m_cam;
		float camOffset = CRenderer::CV_r_StereoEyeDist * 0.5f;
		float frustumShift = camOffset * (cam.GetNearPlane() / CRenderer::CV_r_StereoScreenDist);

		// Left eye
		{
			m_renderer.PushProfileMarker("LEFT_EYE");
			CCamera camL = cam;
			Matrix34 stereoMat = Matrix34::CreateTranslationMat( Vec3( -camOffset, 0, 0 ) );
			camL.SetMatrix( camL.GetMatrix() * stereoMat );
			camL.SetAsymmetry( frustumShift, frustumShift, 0, 0 );
			m_renderer.SetCamera(camL);

			RenderScene(sceneFlags);
			CopyToStereoFromMainThread(0);
			m_renderer.PopProfileMarker("LEFT_EYE");
		}

		// Right eye
		{
			m_renderer.PushProfileMarker("RIGHT_EYE");
			CCamera camR = cam;
			Matrix34 stereoMat = Matrix34::CreateTranslationMat( Vec3( camOffset, 0, 0 ) );
			camR.SetMatrix( camR.GetMatrix() * stereoMat );
			camR.SetAsymmetry( -frustumShift, -frustumShift, 0, 0 );
			m_renderer.SetCamera(camR);

			RenderScene(sceneFlags);
			CopyToStereoFromMainThread(1);
			m_renderer.PopProfileMarker("RIGHT_EYE");
		}
	}
	else
	{
		RenderScene(sceneFlags);
	}
}

void CD3DStereoRenderer::CopyToStereo(int channel)
{
	assert(IsRenderThread());

	PROFILE_LABEL_PUSH("COPY_TO_STEREO");

	CTexture *pTex;

	if( channel == 0 )
		pTex = CTexture::s_ptexStereoL;
	else
		pTex = CTexture::s_ptexStereoR;

	if (pTex==NULL)
		return;

	GetUtils().SetSRGBWrite( false );
	GetUtils().CopyScreenToTexture( pTex );
	GetUtils().SetSRGBWrite( true );

	PROFILE_LABEL_POP("COPY_TO_STEREO");
}

void CD3DStereoRenderer::DisplayStereo()
{
	assert(IsRenderThread());
	  
	if (!IsStereoEnabled() || CTexture::s_ptexBackBuffer == NULL)
		return;

	ResolveStereoBuffers();

	gRenDev->m_cEF.mfRefreshSystemShader("Stereo", CShaderMan::m_ShaderStereo);

	PROFILE_LABEL_PUSH("DISPLAY_STEREO");

	GetUtils().SetSRGBWrite( false );

	if( m_pHighResFrontBuffer )
		m_renderer.FX_PushRenderTarget( 0, m_pHighResFrontBuffer, NULL, false );
	
	m_renderer.Set2DMode( true, 1, 1 );

	if (m_renderer.m_pSecondBackBuffer != 0)
	{
		m_renderer.FX_PushRenderTarget(1, m_renderer.m_pSecondBackBuffer, NULL);
		m_renderer.RT_SetViewport(0, 0, m_renderer.m_NewViewport.nWidth, m_renderer.m_NewViewport.nHeight);
	}

	CShader *pSH = m_renderer.m_cEF.m_ShaderStereo;
	SelectShaderTechnique();

	uint32 nPasses = 0;
	pSH->FXBegin(&nPasses, FEF_DONTSETSTATES);
	pSH->FXBeginPass(0);

	m_renderer.EF_SetState(GS_NODEPTHTEST);

  int width = CTexture::s_ptexBackBuffer->GetWidth();
  int height = CTexture::s_ptexBackBuffer->GetHeight();
  
  Vec4 pParams= Vec4((float) width, (float) height, 0, 0);
  static CCryName pParamName("SourceSize");
  CShaderMan::m_shPostEffects->FXSetPSFloat(pParamName, &pParams, 1);  

	GetUtils().SetTexture(!CRenderer::CV_r_StereoFlipEyes ? GetLeftEye() : GetRightEye(), 0, FILTER_LINEAR);
	GetUtils().SetTexture(!CRenderer::CV_r_StereoFlipEyes ? GetRightEye() : GetLeftEye(), 1, FILTER_LINEAR);
	
	GetUtils().DrawFullScreenQuad(CTexture::s_ptexBackBuffer->GetWidth(), CTexture::s_ptexBackBuffer->GetHeight());

	pSH->FXEndPass();
	pSH->FXEnd();

	m_renderer.Set2DMode( false, 1, 1 );
	
	if (m_renderer.m_pSecondBackBuffer != 0)
		m_renderer.FX_PopRenderTarget(1);
	if( m_pHighResFrontBuffer )
		m_renderer.FX_PopRenderTarget( 0 );

	GetUtils().SetSRGBWrite( true );

	PROFILE_LABEL_POP("DISPLAY_STEREO");
}

void CD3DStereoRenderer::BeginRenderingMRT()
{
	if (!IsStereoEnabled())
		return;

  PushRenderTargets();
}

void CD3DStereoRenderer::EndRenderingMRT(bool bResolve)
{
	if (!IsStereoEnabled())
		return;

  PopRenderTargets(bResolve);
}

void CD3DStereoRenderer::ResolveStereoBuffers()
{
	if (!IsStereoEnabled())
		return;

	PushRenderTargets();
	PopRenderTargets(true);
}

void CD3DStereoRenderer::BeginRenderingTo(Eye eye)
{
	if (eye ==  LEFT_EYE)
	{
		gRenDev->SetProfileMarker("LEFT_EYE", CRenderer::ESPM_PUSH);
		GetLeftEye()->SetRenderTargetTile(0);
		m_renderer.FX_PushRenderTarget(0, GetLeftEye(), &m_renderer.m_DepthBufferOrigFSAA);
	}
	else
	{
		gRenDev->SetProfileMarker("RIGHT_EYE", CRenderer::ESPM_PUSH);
		GetRightEye()->SetRenderTargetTile(1);
		m_renderer.FX_PushRenderTarget(0, GetRightEye(), &m_renderer.m_DepthBufferOrigFSAA);
	}

	m_renderer.FX_SetActiveRenderTargets();

  if (eye ==  LEFT_EYE)
    GetLeftEye()->SetResolved(true);
  else
    GetRightEye()->SetResolved(true);
}

void CD3DStereoRenderer::EndRenderingTo(Eye eye)
{
	if (eye ==  LEFT_EYE)
	{
		gRenDev->SetProfileMarker("LEFT_EYE", CRenderer::ESPM_POP);
		GetLeftEye()->SetResolved(true);
		m_renderer.FX_PopRenderTarget(0);
		GetLeftEye()->SetRenderTargetTile(0);
	}
	else
	{
		gRenDev->SetProfileMarker("RIGHT_EYE", CRenderer::ESPM_POP);
		GetRightEye()->SetResolved(true);
		m_renderer.FX_PopRenderTarget(0);
		GetRightEye()->SetRenderTargetTile(0);
	}
}

void CD3DStereoRenderer::CreateResources()
{
	if (GetStereoMode() == STEREO_MODE_NO_STEREO)
		return;

	assert(CTexture::s_ptexStereoL == NULL);
	assert(CTexture::s_ptexStereoR == NULL);

	uint32 nFlags = FT_DONT_STREAM | FT_USAGE_RENDERTARGET | FT_DONT_RELEASE | FT_USAGE_PREDICATED_TILING | FT_DONT_RESIZE;

	int nWidth = m_renderer.GetWidth();
	int nHeight = m_renderer.GetHeight();

	CTexture::s_ptexStereoL = CTexture::CreateRenderTarget( "$StereoL", nWidth, nHeight, eTT_2D, nFlags, eTF_A8R8G8B8 );
	CTexture::s_ptexStereoR = CTexture::CreateRenderTarget( "$StereoR", nWidth, nHeight, eTT_2D, nFlags, eTF_A8R8G8B8 );

#ifdef XENON	
	CTexture::s_ptexStereoL->SetClearOnResolve(false);
	CTexture::s_ptexStereoR->SetClearOnResolve(false);
	CreateHighResFrontBuffer();
#endif
}

void CD3DStereoRenderer::DestroyResources()
{
	SAFE_RELEASE(CTexture::s_ptexStereoL);
	SAFE_RELEASE(CTexture::s_ptexStereoR);

	if (m_pHighResFrontBuffer)
	{
		AddPhysicalD3DBlock(-m_pHighResFrontBuffer->GetActualSize());
		SAFE_RELEASE(m_pHighResFrontBuffer);
	}
}

void CD3DStereoRenderer::CreateHighResFrontBuffer()
{
#ifdef XENON
	if( m_output == STEREO_OUTPUT_INTERLACED )
	{
		XVIDEO_MODE videoMode;
		ZeroMemory( &videoMode, sizeof( videoMode ) );
		XGetVideoMode( &videoMode );

		if( videoMode.dwDisplayHeight > 720 )
		{	
			int width = 1280;
			int height = videoMode.dwDisplayHeight;
			D3DFORMAT nFormat = D3DFMT_LE_X8R8G8B8;
			D3DTexture *pD3DTex = NULL;

			MEMSTAT_CONTEXT(EMemStatContextTypes::MSC_Texture, 0, "D3D Back Buffer");
			m_pHighResFrontBuffer = CTexture::CreateTextureObject("$RT_HighResFrontBuffer", 0, 0, 1, eTT_2D, FT_DONT_RELEASE | FT_DONT_STREAM | FT_USAGE_RENDERTARGET | FT_DONT_RESIZE, eTF_Unknown);
			m_renderer.m_pd3dDevice->CreateTexture(width, height, 1, 0, nFormat, D3DPOOL_DEFAULT, &pD3DTex, NULL);
			m_pHighResFrontBuffer->SetDevTexture(new CDeviceTexture(pD3DTex));
			m_pHighResFrontBuffer->SetWidth(width);
			m_pHighResFrontBuffer->SetHeight(height);
			m_pHighResFrontBuffer->ClosestFormatSupported(eTF_A8R8G8B8);
			assert(m_pHighResFrontBuffer->GetPixelFormat());
			m_pHighResFrontBuffer->SetFlags(FT_USAGE_RENDERTARGET);
			m_pHighResFrontBuffer->SetActualSize(CTexture::TextureDataSize(width, height, 1, 1, CTexture::TexFormatFromDeviceFormat(nFormat)));
			AddPhysicalD3DBlock(m_pHighResFrontBuffer->GetActualSize());
		}
	}
#endif
}

void CD3DStereoRenderer::SelectShaderTechnique()
{
	gRenDev->m_cEF.mfRefreshSystemShader("Stereo", CShaderMan::m_ShaderStereo);

  CShader *pSH = m_renderer.m_cEF.m_ShaderStereo;

  if (CRenderer::CV_r_GetScreenShot)
  {
    pSH->FXSetTechnique("SideBySide");
    return;
  }

	switch(GetStereoOutput())
	{
	case STEREO_OUTPUT_GENERIC_DUAL_HEAD:
		pSH->FXSetTechnique("DualHead");
		break;
	case STEREO_OUTPUT_IZ3D:
		pSH->FXSetTechnique("IZ3D");
		break;
	case STEREO_OUTPUT_SIDE_BY_SIDE:
		pSH->FXSetTechnique("SideBySide");
		break;
	case STEREO_OUTPUT_INTERLACED:
		pSH->FXSetTechnique("Interlaced");
		break;
	case STEREO_OUTPUT_ANAGLYPH:
		pSH->FXSetTechnique("Anaglyph");
		break;
	default:
		pSH->FXSetTechnique("DualHead");
	}
}

void CD3DStereoRenderer::RenderScene(int sceneFlags)
{
	m_renderer.EF_Scene3D(m_renderer.m_MainRTViewport, sceneFlags);
}

bool CD3DStereoRenderer::IsRenderThread() const
{
	return m_renderer.m_pRT->IsRenderThread();
}

void CD3DStereoRenderer::CopyToStereoFromMainThread(int channel)
{
	m_renderer.m_pRT->RC_CopyToStereoTex(channel);
}

void CD3DStereoRenderer::PushRenderTargets()
{
  GetLeftEye()->SetRenderTargetTile(0);
  GetRightEye()->SetRenderTargetTile(0);

  m_renderer.FX_PushRenderTarget(0, CTexture::s_ptexStereoL,  &gcpRendD3D->m_DepthBufferOrigFSAA); 
  m_renderer.FX_PushRenderTarget(1, CTexture::s_ptexStereoR, NULL); 

  m_renderer.RT_SetViewport(0, 0, CTexture::s_ptexStereoL->GetWidth(), CTexture::s_ptexStereoL->GetHeight());

  m_renderer.FX_SetActiveRenderTargets();
}

void CD3DStereoRenderer::PopRenderTargets(bool bResolve)
{
  GetLeftEye()->SetResolved(!bResolve);
  GetRightEye()->SetResolved(!bResolve);

  m_renderer.FX_PopRenderTarget(0); 
  m_renderer.FX_PopRenderTarget(1);
}
