/*=============================================================================
PostProcessHud3D : 3D hud post processing
Copyright (c) 2001 Crytek Studios. All Rights Reserved.

Revision history:
* 23/02/2005: Re-factored/Converted to CryEngine 2.0 by Tiago Sousa
* Created by Tiago Sousa

=============================================================================*/

#include "StdAfx.h"
#include "DriverD3D.h"
#include "I3DEngine.h"
#include "D3DPostProcess.h"
#include "D3DStereo.h"
#include <IFlashPlayer.h>

#pragma warning(disable: 4244)

int16 C3DHud::SHudData::s_nFlashWidthMax = 0;
int16 C3DHud::SHudData::s_nFlashHeightMax = 0;

void C3DHud::SHudData::Init()
{
	nSortVal = nFlashWidth = nFlashHeight = 0;
	pFlashPlayer = NULL;

	pDiffuse = pShaderResources->GetTexture( EFTT_DIFFUSE );	
	assert( pDiffuse );

	IDynTextureSource::EDynTextureSource type(IDynTextureSource::DTS_I_FLASHPLAYER);				
	if( pDiffuse->m_Sampler.m_pDynTexSource )
	{
		pDiffuse->m_Sampler.m_pDynTexSource->GetDynTextureSource((void*&)pFlashPlayer, type);
    if (pFlashPlayer) 
    {
      nFlashWidth = pFlashPlayer->GetWidth();
      nFlashHeight = pFlashPlayer->GetHeight();
		
      s_nFlashWidthMax = max(nFlashWidth, s_nFlashWidthMax);
      s_nFlashHeightMax = max(nFlashHeight, s_nFlashHeightMax);
    }
	}

	if( type != IDynTextureSource::DTS_I_FLASHPLAYER )
		pFlashPlayer = NULL;

	nSortVal = nFlashWidth * nFlashHeight; // we sort by size
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

void C3DHud::Reset()
{
	for(uint32 i=0; i<RT_COMMAND_BUF_COUNT; ++i)
		m_pRenderData[i].resize(0);
	m_pOpacityMul->ResetParam(1.0f);
	m_pHudColor->ResetParamVec4(Vec4(1.0f, 1.0f, 1.0f, 1.0f));
	m_pGlowMul->ResetParam(1.0f);
	m_pChromaShift->ResetParam(0.0f);
	m_pDofMultiplier->ResetParam(1.0f);
	m_fProjOffsetX = 0.0f;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

void C3DHud::AddRE( const CRendElementBase *re, const SShaderItem *pShaderItem, const CRenderObject *pObj)
{
	// Main thread
	const uint32 nThreadID = gcpRendD3D->m_RP.m_nFillThreadID;		
	// Only add valid render elements
	if( (SRendItem::m_RecurseLevel[nThreadID] - 1) == 0 && re && pObj && pShaderItem && pShaderItem->m_pShaderResources)         
	{
		SHudData pHudData(re, (const SRenderShaderResources*)pShaderItem->m_pShaderResources, pObj);
		m_pRenderData[nThreadID].push_back( pHudData );
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

void C3DHud::SetShaderParams( SHudData &pData )
{
	SRenderShaderResources *pShaderResources = (SRenderShaderResources *)pData.pShaderResources;	
	float fOpacity = pShaderResources->Opacity() * m_pOpacityMul->GetParam();

	const CRenderObject *pRO = pData.pRO;	
	Matrix44A mObjCurr, mViewProj;
	mObjCurr.Transpose(pRO->m_II.m_Matrix);   			

	Matrix44A mProj = gRenDev->m_CameraProjMatrix;
	mProj.m20 += m_fProjOffsetX;
	
	mViewProj.Multiply(mObjCurr, mProj); 
	mViewProj.Transpose();

	CShaderMan::m_sh3DHUD->FXSetVSFloat(m_pMatViewProjParamName, (Vec4 *) mViewProj.GetData(), 4);		
	
	// Since we can have multiple flash files with different resolutions we need to update resolution for correct mapping	
	Vec4 vHudTexCoordParams = Vec4((float) pData.nFlashWidth  / (float) m_pHUD_RT->GetWidth(), 
																 (float) pData.nFlashHeight / (float) m_pHUD_RT->GetHeight(),
																 (float) m_pHUD_RT->GetWidth() / (float) pData.s_nFlashWidthMax, 
																 (float) m_pHUD_RT->GetHeight() / (float) pData.s_nFlashHeightMax);

	CShaderMan::m_sh3DHUD->FXSetVSFloat(m_pHudTexCoordParamName, &vHudTexCoordParams, 1);
	CShaderMan::m_sh3DHUD->FXSetPSFloat(m_pHudTexCoordParamName, &vHudTexCoordParams, 1);

	ColorF cDiffuse = pShaderResources->GetDiffuseColor();
	if (pShaderResources->m_ResFlags & MTL_FLAG_ADDITIVE)
		cDiffuse *= fOpacity;

	Vec4 vHudParams = Vec4(cDiffuse.r, cDiffuse.g, cDiffuse.b, fOpacity) * m_pHudColor->GetParamVec4();  
	CShaderMan::m_sh3DHUD->FXSetPSFloat(m_pHudParamName, &vHudParams, 1); 
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

void C3DHud::SetTextures( SHudData &pData )
{
	SEfResTexture *pDiffuse = pData.pDiffuse;
	if( pDiffuse && pDiffuse->m_Sampler.m_pTex )
		GetUtils().SetTexture(pDiffuse->m_Sampler.m_pTex, 0, FILTER_LINEAR);  
	else
		GetUtils().SetTexture(m_pHUD_RT,  0, FILTER_LINEAR);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

void C3DHud::RenderMesh( const CRendElementBase *pRE, SShaderPass *pPass )
{
	CD3D9Renderer *const __restrict rd = gcpRendD3D;
	CREMesh *pRenderMesh = (CREMesh*) pRE;
	
	// Create/Update vertex buffer stream
	if( pRenderMesh->m_pRenderMesh )
		pRenderMesh->m_pRenderMesh->CheckUpdate( pRenderMesh->m_pRenderMesh->_GetVertexFormat(), 0);

	rd->m_RP.m_pRE = const_cast<CRendElementBase*>( pRE );
	if( rd->FX_CommitStreams(pPass, true) )
	{
		rd->m_RP.m_FirstVertex = pRenderMesh->m_nFirstVertId;
		rd->m_RP.m_FirstIndex = pRenderMesh->m_nFirstIndexId;
		rd->m_RP.m_RendNumIndices = pRenderMesh->m_nNumIndices;
		rd->m_RP.m_RendNumVerts = pRenderMesh->m_nNumVerts;
		rd->m_RP.m_pRE->mfDraw(CShaderMan::m_sh3DHUD, pPass);
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

void C3DHud::DownsampleHud2x2( CTexture *pDstRT )
{
	CD3D9Renderer *const __restrict rd = gcpRendD3D;

	PROFILE_LABEL_PUSH( "3D HUD DOWNSAMPLE 2X2" );

	rd->Set2DMode(true, 1, 1);	

	// Enable rendering outside backbuffer edram range
	pDstRT->SetRenderTargetTile(1); 
	
	rd->FX_PushRenderTarget(0,  pDstRT,  &rd->m_DepthBufferOrigFSAA); 
	rd->RT_SetViewport(0, 0, pDstRT->GetWidth(), pDstRT->GetHeight());        

	GetUtils().ShBeginPass(CShaderMan::m_sh3DHUD, m_pDownsampleTechName, FEF_DONTSETTEXTURES | FEF_DONTSETSTATES);          

	// We might have multiple different flash files - render with additive mode to accumulate results
	rd->EF_SetState( GS_NODEPTHTEST );   
	rd->SetCullMode( R_CULL_BACK);

	GetUtils().SetTexture(m_pHUD_RT, 0, FILTER_LINEAR);  

	// Rescale uv's to match viewport
	Vec4 vTCScale = Vec4(	1, 1, (float) SHudData::s_nFlashWidthMax/(float) m_pHUD_RT->GetWidth(), (float) SHudData::s_nFlashHeightMax/(float) m_pHUD_RT->GetHeight());
	CShaderMan::m_sh3DHUD->FXSetVSFloat(m_pHudTexCoordParamName, &vTCScale, 1);
	CShaderMan::m_sh3DHUD->FXSetPSFloat(m_pHudTexCoordParamName, &vTCScale, 1);

	GetUtils().DrawFullScreenQuad(pDstRT->GetWidth(),pDstRT->GetHeight());    
		  
	GetUtils().ShEndPass();	
	rd->FX_PopRenderTarget(0);
		
	GetUtils().TexBlurIterative(pDstRT, 1);
	rd->Set2DMode(false, 1, 1);

	pDstRT->SetRenderTargetTile(0);

	PROFILE_LABEL_POP( "3D HUD DOWNSAMPLE 2X2" );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

// Reminder: for efficient multiple flash files to work correctly - uv's must not override
void C3DHud::FlashUpdateRT(void)
{
	CD3D9Renderer *const __restrict rd = gcpRendD3D;		
	const uint32 nThreadID = rd->m_RP.m_nProcessThreadID;
	uint32 nRECount = m_pRenderData[nThreadID].size();
	
	if( nRECount )//&& m_nFlashUpdateFrameID != rd->GetFrameID(false) )
	{
		// Share hud render target with scene normals
		m_pHUD_RT = CTexture::s_ptexSceneNormalsMap;	
		m_pHUDScaled_RT = CTexture::s_ptexBackBufferScaled[1];	

		rd->Set2DMode(false, 1, 1);
		m_nFlashUpdateFrameID = rd->GetFrameID(false);

		PROFILE_LABEL_PUSH( "3D HUD FLASHPLAYER UPDATES" );		
				
		SEfResTexture *pDiffuse = NULL;
		SEfResTexture *pPrevDiffuse = NULL;
		
		// Enable rendering outside backbuffer edram range
		m_pHUD_RT->SetRenderTargetTile(1);	
		rd->FX_PushRenderTarget(0,  m_pHUD_RT,  &rd->m_DepthBufferOrigFSAA);  
		
		int32 nFlashWidth = SHudData::s_nFlashWidthMax + (8 - SHudData::s_nFlashWidthMax%8);		
		int32 nFlashHeight = SHudData::s_nFlashHeightMax + (8 - SHudData::s_nFlashHeightMax%8);
		rd->RT_SetViewport(0, 0, nFlashWidth, nFlashHeight);     

		ColorF clearCol(0, 0, 0, 0);		
		int32 clearFlags = FRT_CLEAR_COLOR|FRT_CLEAR_IMMEDIATE;
		gcpRendD3D->EF_ClearBuffers(clearFlags, &clearCol);

		for(int r = 0; r < nRECount; ++r) 
		{
			SHudData &pData = m_pRenderData[nThreadID][r];

			pDiffuse = pData.pDiffuse;
			if( !pDiffuse || !pDiffuse->m_Sampler.m_pDynTexSource )
				continue;
			
			if(pData.pFlashPlayer && pPrevDiffuse != pDiffuse)
			{					
				PROFILE_LABEL_PUSH( "3D HUD FLASHFILE" );

				pDiffuse->m_Sampler.m_pDynTexSource->Update(m_pHUD_RT, false); 
				pPrevDiffuse = pDiffuse;
				
				PROFILE_LABEL_POP( "3D HUD FLASHFILE" );

				if( CRenderer::CV_r_PostProcessHUD3D == 1)
					break;
			}
		}

#if defined(XENON)
		// Don't resolve entire target, only resolve/copy from updated source rect
		rd->RT_SetViewport(0, 0, nFlashWidth, nFlashHeight);     
		m_pHUD_RT->Resolve(0, true);
#endif

		rd->FX_PopRenderTarget(0);
		m_pHUD_RT->SetRenderTargetTile(0);

		// Downsample hud into half res target _1 time only_ - we'll use this for Bloom/Dof
		DownsampleHud2x2( m_pHUDScaled_RT );

		PROFILE_LABEL_POP( "3D HUD FLASHPLAYER UPDATES" );

		rd->Set2DMode(true, 1, 1);
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

void C3DHud::FinalPass()
{
	CD3D9Renderer *const __restrict rd = gcpRendD3D;

	PROFILE_LABEL_PUSH( "3D HUD FINAL PASS" );	
	rd->Set2DMode(false, 1, 1);

	const uint32 nThreadID = rd->m_RP.m_nProcessThreadID;
	SShaderTechnique *pShaderTech = CShaderMan::m_sh3DHUD->mfFindTechnique(m_pGeneralTechName);
	
	SEfResTexture *pDiffuse = NULL;
	SEfResTexture *pPrevDiffuse = NULL;

	// Hud simple 2D dof blend
	CPostEffect *pDofPostEffect = PostEffectMgr()->GetEffect(ePFX_eDepthOfField);
	bool bGameDof = pDofPostEffect->IsActive();

	static float fDofBlend = 0.0f;
	fDofBlend += ((bGameDof?1.0f:0.0f) - fDofBlend) * gEnv->pTimer->GetFrameTime() * 10.0f; 
	float fCurrentDofBlend = fDofBlend * clamp_tpl<float>(m_pDofMultiplier->GetParam(), 0.0f, 1.0f);

	uint32 nRECount = m_pRenderData[nThreadID].size();
	for(int r = 0; r < nRECount; ++r) 
	{
		SHudData &pData = m_pRenderData[nThreadID][r];
		SRenderShaderResources *pShaderResources = (SRenderShaderResources*)pData.pShaderResources;

		pDiffuse = pData.pDiffuse;
		if( !pShaderResources || !pDiffuse || !pDiffuse->m_Sampler.m_pDynTexSource )
			continue;

		int32 nRenderState = GS_NODEPTHTEST; 
		if ((pShaderResources->Opacity()) != 1.0f)
		{
			if (pShaderResources->m_ResFlags & MTL_FLAG_ADDITIVE)
				nRenderState |= GS_BLSRC_ONE | GS_BLDST_ONE;
			else
				nRenderState |= GS_BLSRC_SRCALPHA | GS_BLDST_ONEMINUSSRCALPHA;
		}

		GetUtils().ShBeginPass(CShaderMan::m_sh3DHUD, m_pGeneralTechName, FEF_DONTSETTEXTURES | FEF_DONTSETSTATES);                                   

		rd->EF_SetState( nRenderState );   
		rd->SetCullMode( R_CULL_BACK );

		SetShaderParams( pData );

		// Set additional parameters
		Vec4 vHudEffectParams = Vec4(fCurrentDofBlend, pShaderResources->Glow() * m_pGlowMul->GetParam(), 0.0f, m_pChromaShift->GetParam());  
		CShaderMan::m_sh3DHUD->FXSetPSFloat(m_pHudEffectsParamName, &vHudEffectParams, 1); 

		SetTextures( pData );		
		GetUtils().SetTexture(m_pHUDScaled_RT, 1, FILTER_LINEAR);    

		RenderMesh( pData.pRE, &pShaderTech->m_Passes[0] );

		GetUtils().ShEndPass(); 

		pPrevDiffuse = pDiffuse;
	}

	rd->Set2DMode(true, 1, 1);

	PROFILE_LABEL_POP( "3D HUD FINAL PASS" );
};

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

void C3DHud::Render()
{
	PROFILE_SHADER_START
	PROFILE_LABEL_PUSH( "3D HUD" );

	gcpRendD3D->m_RP.m_FlagsShader_RT &= ~(g_HWSR_MaskBit[HWSR_SAMPLE0]|g_HWSR_MaskBit[HWSR_SAMPLE1]|g_HWSR_MaskBit[HWSR_SAMPLE2]);

	CD3DStereoRenderer &pS3DRend = gcpRendD3D->GetS3DRend();
	bool bPostProcStereo = pS3DRend.GetStereoMode() == STEREO_MODE_POST_STEREO;

	// If post-stereo not enabled, update flash player
	if( !bPostProcStereo )
		FlashUpdateRT(); 
					
	// Render hud with projection offset or with same projection offset in MRT mode (a bit faster)
	if( bPostProcStereo )
	{		
		if( CRenderer::CV_r_StereoHudScreenDist )
		{
			float fRecipWidth = 1.0f / gRenDev->GetWidth();

			// Render left eye
			pS3DRend.BeginRenderingTo(LEFT_EYE);		
			m_fProjOffsetX = CRenderer::CV_r_StereoHudScreenDist * fRecipWidth;
			FinalPass();    
			pS3DRend.EndRenderingTo(LEFT_EYE);
					
			// Render right eye
			pS3DRend.BeginRenderingTo(RIGHT_EYE);			
			m_fProjOffsetX = -CRenderer::CV_r_StereoHudScreenDist * fRecipWidth;
			FinalPass();
			pS3DRend.EndRenderingTo(RIGHT_EYE);
			
			m_fProjOffsetX = 0;
		}
		else
		{
			pS3DRend.BeginRenderingMRT();
			gcpRendD3D->m_RP.m_FlagsShader_RT |= g_HWSR_MaskBit[HWSR_SAMPLE0];
			FinalPass();
			pS3DRend.EndRenderingMRT();
		}
	}
	else
	{
		FinalPass();
	}
	
	PROFILE_LABEL_POP( "3D HUD" );

  gcpRendD3D->FX_Flush(); 
  PROFILE_SHADER_END  
}
