#include "StdAfx.h"
#include <LoadSeq.h>

#if defined(USE_LOADSEQ)

#include <pthread.h>
#include <cell/cell_fs.h>

// Define this to enable file name tracking.
#define LOADSEQ_ENABLE_NAME_TRACKING 1
//#undef LOADSEQ_ENABLE_NAME_TRACKING

extern "C"
{
  int __REAL__open(const char *, int, mode_t);
  ssize_t __REAL__read(int, void *, size_t);
  off_t __REAL__lseek(int, off_t, int);
  off64_t __REAL__lseek64(int, off64_t, int);
  int __REAL__stat(const char *, struct stat *);
  int __REAL__fstat(int, struct stat *);
  int __REAL__close(int);

  CellFsErrno __REAL__cellFsOpen(const char *, int, int *,
      const void *, uint64_t);
  CellFsErrno __REAL__cellFsRead(int, void *, uint64_t, uint64_t *);
  CellFsErrno __REAL__cellFsLseek(int, int64_t, int, uint64_t *);
  CellFsErrno __REAL__cellFsStat(const char *, CellFsStat *);
  CellFsErrno __REAL__cellFsFstat(int, CellFsStat *);
  CellFsErrno __REAL__cellFsClose(int);
}

extern const char *GetCellFsErrString(int);

namespace LoadSeq
{
  // File format version.
  static const uint32_t FILEFORMAT_VERSION = 1;

  // File format magic string.
  static const uint8_t FILEFORMAT_MAGIC[12] = "LOADSEQ:REC";

  // State variable indicating the operation mode.
  IOMode g_IOMode = PASSTHROUGH;

  // Flag indicating if a playback should automatically fall through to a
  // recording if a playback file is not found.
  bool g_bAutoRecord = true;

  // Flag indicating if setting the mode is OK.  This flag is cleared by the
  // first call to LoadSeq::Begin().
  bool g_SetModeOK = true;

  // Global lock for Begin()/End()/Cancel().
  pthread_mutex_t g_Lock = PTHREAD_MUTEX_INITIALIZER;

  // Check if a write open is OK for the specified path.
  bool IsWriteOk(const char *pathLower);

  // Check if dropping the write bit from a read-write open to the file is OK.
  bool IsReadWriteOk(const char *pathLower);

  // Check if a path is a pass-through path.
  bool IsPassThrough(const char *pathLower);

  // Check if a path refers to a persistent file.  Persitent files may be kept
  // open accross multiple recorded sequences.
  bool IsPersistent(const char *pathLower);

  // Get the file name of a recording for the specified sequence name.  The
  // function returns 0 on success and -1 in case of an error.
  int RecordFilename(const char *name, char *buffer, size_t bufferSize);

  // Log a message.  This function will _not_ recurse into the load sequencer
  // (the messages are either dumped to the console or written to the game log
  // file directly - the engine's logging facility is _not_ used!)
  void Log(const char *, ...) PRINTF_PARAMS(1, 2);
  void LogV(const char *, va_list) PRINTF_PARAMS(1, 0);

  // Enumeration of substituted functions.
  enum Function
  {
    FN_NONE = 0,
    FN_open = 0x11110001,
    FN_read,
    FN_lseek,
    FN_lseek64,
    FN_stat,
    FN_fstat,
    FN_close,
    FN_cellFsOpen = 0x22220001,
    FN_cellFsRead,
    FN_cellFsLseek,
    FN_cellFsStat,
    FN_cellFsFstat,
    FN_cellFsClose,
    FN_END = 0x44444444
  };

  inline const char *FunctionName(Function fn)
  {
    const char *fnName = "<unknown>";

    switch (fn)
    {
    case FN_NONE: fnName = "<none>"; break;
    case FN_open: fnName = "open"; break;
    case FN_read: fnName = "read"; break;
    case FN_lseek: fnName = "lseek"; break;
    case FN_lseek64: fnName = "lseek64"; break;
    case FN_stat: fnName = "stat"; break;
    case FN_fstat: fnName = "fstat"; break;
    case FN_close: fnName = "close"; break;
    case FN_cellFsOpen: fnName = "cellFsOpen"; break;
    case FN_cellFsRead: fnName = "cellFsRead"; break;
    case FN_cellFsLseek: fnName = "cellFsLseek"; break;
    case FN_cellFsStat: fnName = "cellFsStat"; break;
    case FN_cellFsFstat: fnName = "cellFsFstat"; break;
    case FN_cellFsClose: fnName = "cellFsClose"; break;
    case FN_END: fnName = "END"; break;
    }
    return fnName;
  }

  // File descriptor bit indicating a recorded file descriptor.  We're using
  // bit 14, because the standard IO library wants to store file descriptors
  // in a 'signed short int'.
  //static const int FD_RECORDED = 1 << 14;
  static const int FD_RECORDED = 1 << 7;

  struct FdInfo
  {
    int m_Fd;
    int m_RealFd;
    uint64_t m_Offset;
    bool m_bPersistent;

#ifdef LOADSEQ_ENABLE_NAME_TRACKING
    string m_Path;
#endif

    FdInfo()
      : m_Fd(-1), m_RealFd(-1), m_Offset((uint64_t)-1), m_bPersistent(false)
    { }

    FdInfo(const FdInfo &other)
      : m_Fd(other.m_Fd), m_RealFd(other.m_RealFd),
        m_Offset(other.m_Offset),
        m_bPersistent(other.m_bPersistent)
#ifdef LOADSEQ_ENABLE_NAME_TRACKING
        , m_Path(other.m_Path)
#endif
    { }
  };

  class FdTracker
  {
    FdInfo m_Map[FD_RECORDED];

  public:
    void Open(int fd, const char *path, bool bPlayback = false)
    {
      if (fd < 0) return;
      if (fd >= FD_RECORDED)
      {
        Log("FdTracker: invalid fd %i in Open()", fd);
        abort();
      }
      FdInfo &info = m_Map[fd];
      if (info.m_Fd != -1)
      {
        Log("FdTracker: duplicate fd %i in Open()", fd);
        abort();
      }
      info.m_Fd = fd;
      info.m_Offset = 0;
#ifdef LOADSEQ_ENABLE_NAME_TRACKING
      info.m_Path = path;
#endif
      if (LoadSeq::IsPersistent(path))
      {
        info.m_bPersistent = true;
        if (bPlayback)
        {
          int realFd = __REAL__open(path, O_RDONLY, 0);
          if (realFd == -1)
          {
            Log("FdTracker: can not access original file '%s' for fd %d: %s",
                path, fd, strerror(errno));
            abort();
          }
          info.m_RealFd = realFd;
        }
        else
          info.m_RealFd = fd;
      }
      else
        info.m_RealFd = -1;
    }

    void Close(int fd)
    {
      if (fd < 0) return;
      if (fd >= FD_RECORDED)
      {
        Log("FdTracker: invalid fd %i in Close()", fd);
        abort();
      }
      FdInfo &info = m_Map[fd];
      if (info.m_Fd == -1) return;
      if (info.m_bPersistent) close(info.m_RealFd);
      info.m_Fd = -1;
      info.m_RealFd = -1;
      info.m_Offset = (uint64_t)-1;
#ifdef LOADSEQ_ENABLE_NAME_TRACKING
      info.m_Path.clear();
#endif
    }

    void Seek(int fd, int64_t offset)
    {
      if (fd < 0) return;
      if (fd >= FD_RECORDED)
      {
        Log("FdTracker: invalid fd %i in Seek()", fd);
        abort();
      }
      FdInfo &info = m_Map[fd];
      if (info.m_Fd == -1)
      {
#ifdef _DEBUG
        Log("FdTracker: adopting fd %u at offset %llu",
            info.m_Fd, (unsigned long long)offset);
#endif
        info.m_Fd = fd;
      }
      info.m_Offset = static_cast<uint64_t>(offset);
    }

    void Read(int fd, uint64_t count)
    {
      if (fd < 0) return;
      if (fd >= FD_RECORDED)
      {
        Log("FdTracker: invalid fd %i in Read()", fd);
        abort();
      }
      FdInfo &info = m_Map[fd];
      if (info.m_Fd == -1)
      {
        // The file descriptor is not known to the tracker, but it might be a
        // tracked file descriptor from a previous sequence.
        uint64_t position = 0;
        CellFsErrno err = __REAL__cellFsLseek(
            info.m_Fd, 0, SEEK_CUR, &position);
        if (err != CELL_OK)
        {
          Log("FdTracker: error adopting tracked fd %d: err=0x%x (%s)",
              info.m_Fd, static_cast<unsigned>(err), GetCellFsErrString(err));
          abort();
        }
#ifdef _DEBUG
        Log("FdTracker: adopting fd %u at offset %llu",
            info.m_Fd, (unsigned long long)position);
#endif
        info.m_Offset = position;
        info.m_Fd = fd;
      }
      info.m_Offset += count;
    }

    void End() const
    {
      bool allClosed = true;
      const FdInfo *map = m_Map;

      for (int fd = 0; fd < FD_RECORDED; ++fd)
      {
        const FdInfo &info = map[fd];
        if (info.m_Fd != -1)
        {
          if (!info.m_bPersistent)
          {
#ifdef LOADSEQ_ENABLE_NAME_TRACKING
            Log("FdTracker: fd %i still open ('%s')",
                fd, info.m_Path.c_str());
#else
            Log("FdTracker: fd %i still open", fd);
#endif
            allClosed = false;
          }
          else
          {
            uint64_t position = 0;
            assert(info.m_RealFd != -1);
            CellFsErrno err = __REAL__cellFsLseek(
                info.m_RealFd, info.m_Offset, SEEK_SET, &position);
            if (err != CELL_OK)
            {
              Log("FdTracker: failed to reset persistent fd %d",
                  info.m_RealFd);
              abort();
            }
          }
        }
      }
      if (!allClosed) abort();
    }

    int RealFd(int fd) const
    {
      if (fd < 0 || fd >= FD_RECORDED) return -1;
      const FdInfo &info = m_Map[fd];
      return info.m_RealFd;
    }

    bool IsPersistent(int fd) const
    {
      if (fd < 0 || fd >= FD_RECORDED) return false;
      const FdInfo &info = m_Map[fd];
      return info.m_bPersistent;
    }
  };

  class Sequencer
  {
  protected:
    // The name of the recording.
    string m_Name;

    // The file name of the recording file.
    string m_RecordFile;

    // Record file descriptor.  This is set to -1 if the recording has been
    // cancelled or closed.
    int m_RecordFd;

    // Flag indicating a read or write error on the recorded file.
    bool m_RecordError;

    // The file descriptor tracker.
    FdTracker m_Tracker;

#ifndef _DEBUG
    void Tick() { }
#else
    unsigned m_TickCounter;

    void Tick();
#endif

    void Init(const char *filename, const char *name)
    {
      m_RecordError = false;
      m_RecordFd = -1;
      m_RecordFile = filename;
      m_Name = name;
#ifdef _DEBUG
      m_TickCounter = 0;
#endif
    }

    int RealFd(int fd) const { return m_Tracker.RealFd(fd); }
    bool IsPersistent(int fd) const { return m_Tracker.IsPersistent(fd); }

    Sequencer()
      : m_RecordError(false),
        m_RecordFd(-1)
#ifdef _DEBUG
        , m_TickCounter(0)
#endif
    { }

  public:
    // Check if the recorder or playback is active.  Returns false if the
    // recorder or playback has been cancelled or closed.
    bool IsActive() const { return m_RecordFd != -1; }

    // Get the name of the sequencer.
    const char *Name() const { return m_Name.c_str(); }
  };

#ifdef _DEBUG
  void Sequencer::Tick()
  {
    ++m_TickCounter;
  }
#endif

  class Recorder : public Sequencer
  {
    // Write to the recording file.
    void Write(void *buffer, size_t size);

  public:
    // Open a recording file.  The method returns 0 on success and -1 in case
    // of an error.
    int Open(const char *filename, const char *name);

    // Cancel the recording and delete the recorded file.
    void Cancel();

    // Close the recording.  The method returns 0 on success and -1 in case of
    // an error.
    int Close();

    int LoadSeq_open(const char *path, int flags, mode_t mode);
    ssize_t LoadSeq_read(int fd, void *buffer, size_t size);
    off_t LoadSeq_lseek(int fd, off_t offset, int whence);
    off64_t LoadSeq_lseek64(int fd, off64_t offset, int whence);
    int LoadSeq_stat(const char *path, struct stat *buf);
    int LoadSeq_fstat(int fd, struct stat *buf);
    int LoadSeq_close(int fd);

    CellFsErrno LoadSeq_cellFsOpen(const char *path, int flags, int *fd,
        const void *arg, uint64_t size);
    CellFsErrno LoadSeq_cellFsRead(int fd, void *buffer,
        uint64_t nbytes, uint64_t *nread);
    CellFsErrno LoadSeq_cellFsLseek(int fd, int64_t offset, int whence,
        uint64_t *pos);
    CellFsErrno LoadSeq_cellFsStat(const char *path, CellFsStat *sb);
    CellFsErrno LoadSeq_cellFsFstat(int fd, CellFsStat *sb);
    CellFsErrno LoadSeq_cellFsClose(int fd);
  };

  class Playback : public Sequencer
  {
    // The size of the ring buffer.
    static const uint64_t RINGBUF_SIZE;

    // The size of a single transfer unit block.
    static const uint64_t RINGBUF_BLOCKSIZE;

    // Read the specified amount of data into the specified buffer.  Return 0
    // on success and -1 in case of an error.
    int Read(void *buffer, size_t size);

    // Peek the next 32bit word from the input.  Return 0 on success and -1 in
    // case of an error.
    int Peek(uint32_t &value);

    // Indicate a playback mismatch.  This method logs a message and cancels
    // the playback.
    void Mismatch(const char *fn, const uint32_t *hdr = NULL,
        size_t hdrLength = 0, const char *message = NULL, ...)
      PRINTF_PARAMS(5, 6);

    // Flag indicating if the playback should fall through to a recorder.
    bool m_bIsRecording;

    void Init(const char *filename, const char *name)
    {
      m_bIsRecording = false;
      Sequencer::Init(filename, name);
    }

  public:
    // Open a recording file for playback.  The method returns 0 on success
    // and -1 in case of an error.
    int Open(const char *filename, const char *name);

    // Cancel the playback.
    void Cancel();

    // Close the recording file.  The method returns 0.
    int Close();

    // Check if the playback falls through to a recording.
    bool IsRecording() const { return m_bIsRecording; }

    int LoadSeq_open(const char *path, int flags, mode_t mode);
    ssize_t LoadSeq_read(int fd, void *buffer, size_t size);
    off_t LoadSeq_lseek(int fd, off_t offset, int whence);
    off64_t LoadSeq_lseek64(int fd, off64_t offset, int whence);
    int LoadSeq_stat(const char *path, struct stat *buf);
    int LoadSeq_fstat(int fd, struct stat *buf);
    int LoadSeq_close(int fd);

    CellFsErrno LoadSeq_cellFsOpen(const char *path, int flags, int *fd,
        const void *arg, uint64_t size);
    CellFsErrno LoadSeq_cellFsRead(int fd, void *buffer,
        uint64_t nbytes, uint64_t *nread);
    CellFsErrno LoadSeq_cellFsLseek(int fd, int64_t offset, int whence,
        uint64_t *pos);
    CellFsErrno LoadSeq_cellFsStat(const char *path, CellFsStat *sb);
    CellFsErrno LoadSeq_cellFsFstat(int fd, CellFsStat *sb);
    CellFsErrno LoadSeq_cellFsClose(int fd);
  };

  //const uint64_t Playback::RINGBUF_SIZE = 128 * 1024;
  //const uint64_t Playback::RINGBUF_BLOCKSIZE = 8 * 1024;
  const uint64_t Playback::RINGBUF_SIZE = 1 * 1024 * 1024;
  const uint64_t Playback::RINGBUF_BLOCKSIZE = 64 * 1024;

  static sys_ppu_thread_t g_MainThread;
  static bool g_MainThreadInitialized = false;
  static Recorder g_Recorder;
  static Playback g_Playback;
};

bool LoadSeq::IsWriteOk(const char *pathLower)
{
  size_t len = strlen(pathLower);
  if (len > 4 && !strcmp(pathLower + len - 4, ".log"))
    return true;
  if (len > 14 && !strcmp(pathLower + len - 14, "/aisignals.csv"))
    return true;
  return false;
}

bool LoadSeq::IsReadWriteOk(const char *pathLower)
{
  if (strstr(pathLower, "/shaders/cache/")) return true;
  return false;
}

bool LoadSeq::IsPassThrough(const char *pathLower)
{
  if (!strncmp(pathLower, SYS_APP_HOME, strlen(SYS_APP_HOME)))
    return true;
  size_t len = strlen(pathLower);
  if (len > 15 && !strcmp(pathLower + len - 15, "/shaderlist.txt"))
    return true;
  if (len > 11 && !strcmp(pathLower + len - 11, ".crysisjmsf"))
    return true;
  if (len > 5 && (!strcmp(pathLower + len - 5, ".fxca")
        || !strcmp(pathLower + len - 5, ".fxcb")))
    return true;
  return false;
}

bool LoadSeq::IsPersistent(const char *pathLower)
{
  size_t len = strlen(pathLower);
  if (len > 4 && !strcmp(pathLower + len - 4, ".pak")) return true;
  if (len > 5 && !strcmp(pathLower + len - 5, ".fxcb")) return true;
  return false;
}

int LoadSeq::RecordFilename(
    const char *name,
    char *buffer,
    size_t bufferSize
    )
{
  static bool firstCall = true;
  const char *pathSep = "/";
#if 1
  const char *baseDirectory = gPS3Env->pCurDirHDD0;
#else
  // For debugging only: dump to SYS_APP_HOME.
  const char *baseDirectory = SYS_APP_HOME;
#endif

  if (baseDirectory[0] && baseDirectory[strlen(baseDirectory) - 1] == '/')
    pathSep = "";
  if (firstCall && g_IOMode != PASSTHROUGH)
  {
    firstCall = false;
    // Make sure the sequencer directory exists.
    snprintf(buffer, bufferSize, "%s%sloadseq", baseDirectory, pathSep);
    buffer[bufferSize - 1] = 0;
    CellFsErrno err = cellFsMkdir(buffer, CELL_FS_DEFAULT_CREATE_MODE_1);
    if (err != CELL_OK && err != CELL_FS_EEXIST)
    {
      Log("can not create recording directory '%s': err=0x%x (%s)",
          buffer, static_cast<unsigned>(err), GetCellFsErrString(err));
      g_IOMode = PASSTHROUGH;
      return -1;
    }
  }

  snprintf(buffer, bufferSize, "%s%sloadseq/%s.rec",
      baseDirectory, pathSep, name);
  buffer[bufferSize - 1] = 0;
  return 0;
}

void LoadSeq::Log(const char *format, ...)
{
  va_list ap;

  va_start(ap, format);
  LoadSeq::LogV(format, ap);
  va_end(ap);
}

void LoadSeq::LogV(const char *format, va_list ap)
{
  fputs("LoadSeq: ", stderr);
  vfprintf(stderr, format, ap);
  if (format[0] && format[strlen(format) - 1] != '\n') fputc('\n', stderr);
}

void LoadSeq::Recorder::Write(void *buffer, size_t size)
{
  if (m_RecordError) return;

  uint64_t w = 0;
  CellFsErrno err = cellFsWrite(m_RecordFd, buffer, size, &w);
  if (err != CELL_OK)
  {
    Log("I/O error writing to recording '%s': err=0x%x (%s)",
        m_RecordFile.c_str(), static_cast<unsigned>(err),
        GetCellFsErrString(err));
    m_RecordError = true;
  }
  else if (w < size)
  {
    Log("short write (%u of %u bytes) in recording '%s'",
        static_cast<unsigned>(w), static_cast<unsigned>(size),
        m_RecordFile.c_str());
    m_RecordError = true;
  }
}

int LoadSeq::Recorder::Open(const char *filename, const char *name)
{
  Init(filename, name);
  cellFsUnlink(filename);
  int fd = -1;
  CellFsErrno err = __REAL__cellFsOpen(
      filename,
      CELL_FS_O_CREAT | CELL_FS_O_EXCL | CELL_FS_O_WRONLY,
      &fd,
      NULL,
      0);
  if (err != CELL_OK)
  {
    Log("error opening recording '%s' for writing: err=0x%x (%s)",
        filename, static_cast<unsigned>(err), GetCellFsErrString(err));
    m_RecordError = true;
    return -1;
  }
  uint32_t hdr[5];
  memcpy(hdr, FILEFORMAT_MAGIC, 12);
  hdr[3] = FILEFORMAT_VERSION;
  hdr[4] = 0; // Number of additional header fields.
  m_RecordFd = fd;
  Write(hdr, sizeof hdr);
  Log("recording sequence '%s' to '%s'", name, filename);
  return 0;
}

void LoadSeq::Recorder::Cancel()
{
  if (IsActive())
  {
    __REAL__cellFsClose(m_RecordFd);
    m_RecordFd = -1;
    cellFsUnlink(m_RecordFile.c_str());
  }
}

int LoadSeq::Recorder::Close()
{
  if (!IsActive()) return 0;

  uint32_t hdr[16];
  memset(hdr, 0, sizeof hdr);
  hdr[0] = FN_END;
  Write(hdr, sizeof hdr);

  CellFsErrno err = __REAL__cellFsClose(m_RecordFd);
  if (err != CELL_OK)
  {
    Log("error closing recording '%s': err=0x%x (%s)",
        m_RecordFile.c_str(), static_cast<unsigned>(err),
        GetCellFsErrString(err));
    m_RecordError = true;
  }
  m_RecordFd = -1;

  if (m_RecordError)
  {
    Log("removing failed recording '%s'",
        m_RecordFile.c_str());
    cellFsUnlink(m_RecordFile.c_str());
    return -1;
  }

  m_Tracker.End();
  Log("sequence '%s': recording closed", Name());
  return 0;
}

int LoadSeq::Recorder::LoadSeq_open(const char *path, int flags, mode_t mode)
{
  if (!IsActive()) return __REAL__open(path, flags, mode);

  size_t pathLen = strlen(path);
  char pathLower[pathLen + 1];
  strcpy(pathLower, path);
  strlwr(pathLower);

  if (IsPassThrough(pathLower))
    return __REAL__open(path, flags, mode);
  const int rwMask = O_RDWR | O_RDONLY | O_WRONLY;
  int rwFlags = flags & rwMask;
  if (rwFlags != O_RDONLY)
  {
    if (rwFlags == O_RDWR && IsReadWriteOk(pathLower))
    {
      // This file is accepted and the open is treated as a read-only open
      // operation.
      flags &= ~rwMask;
      flags |= O_RDONLY;
    }
    else if (!IsWriteOk(pathLower) || rwFlags == O_RDWR)
    {
      Log("recording canceled for sequence '%s': "
          "writing to file '%s' (%s)",
          Name(), path,
          rwFlags == CELL_FS_O_RDWR ? "read-write" : "write-only");
      Cancel();
      int fd = __REAL__open(path, flags, mode);
      if (fd >= FD_RECORDED)
      {
        Log("file descriptor overrun in sequence '%s'", Name());
        abort();
      }
      return fd;
    }
    else
    {
      // File is opened for writing - pass through.
      int fd = __REAL__open(path, flags, mode);
      if (fd >= FD_RECORDED)
      {
        Log("file descriptor overrun in sequence '%s'", Name());
        abort();
      }
      return fd;
    }
  }
  Tick();
  uint32_t hdr[1024];
  memset(hdr, 0, sizeof hdr);
  if (pathLen > sizeof hdr - 64) abort();
  int fd = __REAL__open(path, flags, mode);
  if (fd >= FD_RECORDED)
  {
    Log("file descriptor overrun in sequence '%s'", Name());
    abort();
  }
  int openErrno = fd == -1 ? errno : 0;
  hdr[0] = FN_open;
  hdr[1] = pathLen;
  hdr[2] = flags;
  hdr[3] = fd;
  hdr[4] = openErrno;
  memcpy(hdr + 5, path, pathLen);
  Write(hdr, (6 + pathLen / 4) * 4);
  if (openErrno) errno = openErrno;
  m_Tracker.Open(fd, path, true);
  return fd | FD_RECORDED;
}

ssize_t LoadSeq::Recorder::LoadSeq_read(int fd, void *buffer, size_t size)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    ssize_t result;
    if (IsActive())
    {
      result = __REAL__read(fd, buffer, size);
      uint32_t hdr[4];
      hdr[0] = FN_read;
      hdr[1] = fd;
      hdr[2] = size;
      hdr[3] = result;
      Write(hdr, sizeof hdr);
      if (result > 0) Write(buffer, ((result - 1) / 4 + 1) * 4);
    }
    else
      result = __REAL__read(RealFd(fd), buffer, size);
    if (result > 0) m_Tracker.Read(fd, result);
    return result;
  }
  else
    return __REAL__read(fd, buffer, size);
}

off_t LoadSeq::Recorder::LoadSeq_lseek(int fd, off_t offset, int whence)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    off_t result;
    if (IsActive())
    {
      result = __REAL__lseek(fd, offset, whence);
      uint32_t hdr[5];
      hdr[0] = FN_lseek;
      hdr[1] = fd;
      hdr[2] = offset;
      hdr[3] = whence;
      hdr[4] = result;
      Write(hdr, sizeof hdr);
    }
    else
      result = __REAL__lseek(RealFd(fd), offset, whence);
    if (result >= 0) m_Tracker.Seek(fd, result);
    return result;
  }
  else
    return __REAL__lseek(fd, offset, whence);
}

off64_t LoadSeq::Recorder::LoadSeq_lseek64(int fd, off64_t offset, int whence)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    off64_t result;
    if (IsActive())
    {
      result = __REAL__lseek64(fd, offset, whence);
      uint32_t hdr[7];
      hdr[0] = FN_lseek64;
      hdr[1] = fd;
      hdr[2] = (uint32_t)(offset >> 32);
      hdr[3] = (uint32_t)offset;
      hdr[4] = whence;
      hdr[5] = (uint32_t)(result >> 32);
      hdr[6] = (uint32_t)result;
      Write(hdr, sizeof hdr);
    }
    else
      result = __REAL__lseek64(RealFd(fd), offset, whence);
    if (result >= 0) m_Tracker.Seek(fd, result);
    return result;
  }
  else
    return __REAL__lseek64(fd, offset, whence);
}

namespace
{
  // The size of a recorded stat buffer.
  static const unsigned LS_STAT_HDR_SIZE = 17;

  static inline void LS_stat2hdr(uint32_t *hdr, const struct stat *buf)
  {
    hdr[0] = buf->st_dev;
    hdr[1] = buf->st_ino;
    hdr[2] = buf->st_mode;
    hdr[3] = buf->st_nlink;
    hdr[4] = buf->st_uid;
    hdr[5] = buf->st_gid;
    hdr[6] = buf->st_rdev;
    hdr[7] = (uint32_t)(buf->st_size >> 32);
    hdr[8] = (uint32_t)(buf->st_size);
    hdr[9] = (uint32_t)(buf->st_atime >> 32);
    hdr[10] = (uint32_t)(buf->st_atime);
    hdr[11] = (uint32_t)(buf->st_mtime >> 32);
    hdr[12] = (uint32_t)(buf->st_mtime);
    hdr[13] = (uint32_t)(buf->st_ctime >> 32);
    hdr[14] = (uint32_t)(buf->st_ctime);
    hdr[15] = buf->st_blksize;
    hdr[16] = buf->st_blocks;
  }

  static inline void LS_hdr2stat(struct stat *buf, const uint32_t *hdr)
  {
    memset(buf, 0, sizeof(struct stat));
    buf->st_dev = hdr[0];
    buf->st_ino = hdr[1];
    buf->st_mode = hdr[2];
    buf->st_nlink = hdr[3];
    buf->st_uid = hdr[4];
    buf->st_gid = hdr[5];
    buf->st_rdev = hdr[6];
    buf->st_size = ((uint64_t)hdr[7] << 32) | hdr[8];
    buf->st_atime = ((uint64_t)hdr[9] << 32) | hdr[10];
    buf->st_mtime = ((uint64_t)hdr[11] << 32) | hdr[12];
    buf->st_ctime = ((uint64_t)hdr[13] << 32) | hdr[14];
    buf->st_blksize = hdr[15];
    buf->st_blocks = hdr[16];
  }
}

int LoadSeq::Recorder::LoadSeq_stat(const char *path, struct stat *buf)
{
  if (!IsActive()) return __REAL__stat(path, buf);

  size_t pathLen = strlen(path);
  char pathLower[pathLen + 1];
  strcpy(pathLower, path);
  strlwr(pathLower);

  if (IsPassThrough(pathLower)) return __REAL__stat(path, buf);
  Tick();
  uint32_t hdr[1024];
  memset(hdr, 0, sizeof hdr);
  if (pathLen > sizeof hdr - 128) abort();
  hdr[0] = FN_stat;
  hdr[1] = pathLen;
  struct stat statBuf;
  memset(&statBuf, 0, sizeof statBuf);
  int result = __REAL__stat(path, &statBuf);
  int statErrno = result == -1 ? errno : 0;
  LS_stat2hdr(hdr + 2, &statBuf);
  hdr[LS_STAT_HDR_SIZE + 2] = result;
  hdr[LS_STAT_HDR_SIZE + 3] = statErrno;
  memcpy(hdr + LS_STAT_HDR_SIZE + 4, path, pathLen);
  Write(hdr, (LS_STAT_HDR_SIZE + 5 + pathLen / 4) * 4);
  if (statErrno) errno = statErrno;
  if (buf != NULL) *buf = statBuf;
  return result;
}

int LoadSeq::Recorder::LoadSeq_fstat(int fd, struct stat *buf)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    if (IsActive())
    {
      uint32_t hdr[LS_STAT_HDR_SIZE + 4];
      hdr[0] = FN_fstat;
      hdr[1] = fd;
      struct stat statBuf;
      memset(&statBuf, 0, sizeof statBuf);
      int result = __REAL__fstat(fd, &statBuf);
      int fstatErrno = errno;
      LS_stat2hdr(hdr + 2, &statBuf);
      hdr[LS_STAT_HDR_SIZE + 2] = result;
      hdr[LS_STAT_HDR_SIZE + 3] = fstatErrno;
      Write(hdr, sizeof hdr);
      if (buf != NULL) *buf = statBuf;
      errno = fstatErrno;
      return result;
    }
    else
      return __REAL__fstat(RealFd(fd), buf);
  }
  else
    return __REAL__fstat(fd, buf);
}

int LoadSeq::Recorder::LoadSeq_close(int fd)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    int result;
    if (IsActive())
    {
      result = __REAL__close(fd);
      uint32_t hdr[3];
      hdr[0] = FN_close;
      hdr[1] = fd;
      hdr[2] = result;
      Write(hdr, sizeof hdr);
    }
    m_Tracker.Close(fd); // Will close the real fd if present.
    return result;
  }
  else
    return __REAL__close(fd);
}

CellFsErrno LoadSeq::Recorder::LoadSeq_cellFsOpen(
    const char *path,
    int flags,
    int *fd,
    const void *arg,
    uint64_t size
    )
{
  if (!IsActive()) return __REAL__cellFsOpen(path, flags, fd, arg, size);

  size_t pathLen = strlen(path);
  char pathLower[pathLen + 1];
  strcpy(pathLower, path);
  strlwr(pathLower);

  if (IsPassThrough(pathLower))
    return __REAL__cellFsOpen(path, flags, fd, arg, size);
  const int rwMask = CELL_FS_O_RDWR | CELL_FS_O_RDONLY | CELL_FS_O_WRONLY;
  int rwFlags = flags & rwMask;
  if (rwFlags != CELL_FS_O_RDONLY)
  {
    if (rwFlags == CELL_FS_O_RDWR && IsReadWriteOk(pathLower))
    {
      // This file is accepted and the open is treated as a read-only open
      // operation.
      flags &= ~rwMask;
      flags |= CELL_FS_O_RDONLY;
    }
    else if (!IsWriteOk(pathLower) || rwFlags == CELL_FS_O_RDWR)
    {
      Log("recording canceled for sequence '%s': "
          "writing to file '%s' (%s)",
          Name(), path,
          rwFlags == CELL_FS_O_RDWR ? "read-write" : "write-only");
      Cancel();
      CellFsErrno err = __REAL__cellFsOpen(path, flags, fd, arg, size);
      if (err == CELL_OK && fd != NULL && *fd >= FD_RECORDED)
      {
        Log("file descriptor overrun in sequence '%s'", Name());
        abort();
      }
      return err;
    }
    else
    {
      // File is opened for writing - pass through.
      CellFsErrno err = __REAL__cellFsOpen(path, flags, fd, arg, size);
      if (err == CELL_OK && fd != NULL && *fd >= FD_RECORDED)
      {
        Log("file descriptor overrun in sequence '%s'", Name());
        abort();
      }
      return err;
    }
  }
  Tick();
  uint32_t hdr[1024];
  memset(hdr, 0, sizeof hdr);
  if (pathLen > sizeof hdr - 64) abort();
  int fdBuf = -1;
  CellFsErrno err = __REAL__cellFsOpen(path, flags, &fdBuf, arg, size);
  if (err == CELL_OK && fdBuf >= FD_RECORDED)
  {
    Log("file descriptor overrun in sequence '%s'", Name());
    abort();
  }
  hdr[0] = FN_cellFsOpen;
  hdr[1] = pathLen;
  hdr[2] = flags;
  hdr[3] = fdBuf;
  hdr[4] = err;
  memcpy(hdr + 5, path, pathLen);
  Write(hdr, (6 + pathLen / 4) * 4);
  if (fd != NULL) *fd = fdBuf | FD_RECORDED;
  m_Tracker.Open(fdBuf, path);
  return err;
}

CellFsErrno LoadSeq::Recorder::LoadSeq_cellFsRead(
    int fd,
    void *buffer,
    uint64_t nbytes,
    uint64_t *nread
    )
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    uint64_t nreadBuf = 0;
    CellFsErrno err;
    if (IsActive())
    {
      err = __REAL__cellFsRead(fd, buffer, nbytes, &nreadBuf);
      uint32_t hdr[7];
      hdr[0] = FN_cellFsRead;
      hdr[1] = fd;
      hdr[2] = (uint32_t)(nbytes >> 32);
      hdr[3] = (uint32_t)nbytes;
      hdr[4] = (uint32_t)(nreadBuf >> 32);
      hdr[5] = (uint32_t)nreadBuf;
      hdr[6] = err;
      Write(hdr, sizeof hdr);
      if (nreadBuf > 0) Write(buffer, ((nreadBuf - 1) / 4 + 1) * 4);
    }
    else
      err = __REAL__cellFsRead(RealFd(fd), buffer, nbytes, &nreadBuf);
    if (nread != NULL) *nread = nreadBuf;
    if (nreadBuf > 0) m_Tracker.Read(fd, nreadBuf);
    return err;
  }
  else
    return __REAL__cellFsRead(fd, buffer, nbytes, nread);
}

CellFsErrno LoadSeq::Recorder::LoadSeq_cellFsLseek(
    int fd,
    int64_t offset,
    int whence,
    uint64_t *pos
    )
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    uint64_t posBuf = 0;
    CellFsErrno err;
    if (IsActive())
    {
      err = __REAL__cellFsLseek(fd, offset, whence, &posBuf);
      uint32_t hdr[8];
      hdr[0] = FN_cellFsLseek;
      hdr[1] = fd;
      hdr[2] = (uint32_t)(offset >> 32);
      hdr[3] = (uint32_t)offset;
      hdr[4] = whence;
      hdr[5] = (uint32_t)(posBuf >> 32);
      hdr[6] = (uint32_t)posBuf;
      hdr[7] = err;
      Write(hdr, sizeof hdr);
    }
    else
      err = __REAL__cellFsLseek(RealFd(fd), offset, whence, &posBuf);
    if (pos != NULL)
      *pos = posBuf;
    if (err == CELL_OK) m_Tracker.Seek(fd, posBuf);
    return err;
  }
  else
    return __REAL__cellFsLseek(fd, offset, whence, pos);
}

namespace
{
  static const unsigned LS_CELLFSSTAT_HDR_SIZE = 13;

  static inline void LS_CellFsStat2hdr(uint32_t *hdr, const CellFsStat *sb)
  {
    hdr[0] = sb->st_mode;
    hdr[1] = sb->st_uid;
    hdr[2] = sb->st_gid;
    hdr[3] = (uint32_t)(sb->st_atime >> 32);
    hdr[4] = (uint32_t)(sb->st_atime);
    hdr[5] = (uint32_t)(sb->st_mtime >> 32);
    hdr[6] = (uint32_t)(sb->st_mtime);
    hdr[7] = (uint32_t)(sb->st_ctime >> 32);
    hdr[8] = (uint32_t)(sb->st_ctime);
    hdr[9] = (uint32_t)(sb->st_size >> 32);
    hdr[10] = (uint32_t)(sb->st_size);
    hdr[11] = (uint32_t)(sb->st_blksize >> 32);
    hdr[12] = (uint32_t)(sb->st_blksize);
  }

  static inline void LS_hdr2CellFsStat(CellFsStat *sb, const uint32_t *hdr)
  {
    sb->st_mode = hdr[0];
    sb->st_uid = hdr[1];
    sb->st_gid = hdr[2];
    sb->st_atime = ((uint64_t)hdr[3] << 32) | hdr[4];
    sb->st_mtime = ((uint64_t)hdr[5] << 32) | hdr[6];
    sb->st_ctime = ((uint64_t)hdr[7] << 32) | hdr[8];
    sb->st_size = ((uint64_t)hdr[9] << 32) | hdr[10];
    sb->st_blksize = ((uint64_t)hdr[11] << 32) | hdr[12];
  }
}

CellFsErrno LoadSeq::Recorder::LoadSeq_cellFsStat(
    const char *path,
    CellFsStat *sb
    )
{
  if (!IsActive()) return __REAL__cellFsStat(path, sb);

  size_t pathLen = strlen(path);
  char pathLower[pathLen + 1];
  strcpy(pathLower, path);
  strlwr(pathLower);

  if (IsPassThrough(pathLower))
    return __REAL__cellFsStat(path, sb);
  Tick();
  uint32_t hdr[1024];
  memset(hdr, 0, sizeof hdr);
  if (pathLen > sizeof hdr - 128) abort();
  hdr[0] = FN_cellFsStat;
  hdr[1] = pathLen;
  CellFsStat statBuf;
  memset(&statBuf, 0, sizeof statBuf);
  CellFsErrno err = __REAL__cellFsStat(path, &statBuf);
  LS_CellFsStat2hdr(hdr + 2, &statBuf);
  hdr[LS_CELLFSSTAT_HDR_SIZE + 2] = err;
  memcpy(hdr + LS_CELLFSSTAT_HDR_SIZE + 3, path, pathLen);
  Write(hdr, (LS_CELLFSSTAT_HDR_SIZE + 4 + pathLen / 4) * 4);
  if (sb != NULL) *sb = statBuf;
  return err;
}

CellFsErrno LoadSeq::Recorder::LoadSeq_cellFsFstat(int fd, CellFsStat *sb)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    CellFsErrno err;
    if (IsActive())
    {
      uint32_t hdr[LS_CELLFSSTAT_HDR_SIZE + 3];
      hdr[0] = FN_cellFsFstat;
      hdr[1] = fd;
      CellFsStat statBuf;
      memset(&statBuf, 0, sizeof statBuf);
      err = __REAL__cellFsFstat(fd, &statBuf);
      LS_CellFsStat2hdr(hdr + 2, &statBuf);
      hdr[LS_CELLFSSTAT_HDR_SIZE + 2] = err;
      Write(hdr, sizeof hdr);
      if (sb != NULL) *sb = statBuf;
    }
    else
      err = __REAL__cellFsFstat(RealFd(fd), sb);
    return err;
  }
  else
    return __REAL__cellFsFstat(fd, sb);
}

CellFsErrno LoadSeq::Recorder::LoadSeq_cellFsClose(int fd)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= FD_RECORDED;
    CellFsErrno err;
    if (IsActive())
    {
      err = __REAL__cellFsClose(fd);
      uint32_t hdr[3];
      hdr[0] = FN_cellFsClose;
      hdr[1] = fd;
      hdr[2] = err;
      Write(hdr, sizeof hdr);
    }
    m_Tracker.Close(fd);
    return err;
  }
  else
    return __REAL__cellFsClose(fd);
}

int LoadSeq::Playback::Read(void *buffer, size_t size)
{
  uint8_t *bufferP = static_cast<uint8_t *>(buffer);
  uint8_t *bufferEnd = bufferP + size;
  int fd = m_RecordFd;

  while (bufferP < bufferEnd)
  {
    uint64_t waitSize = std::min(
        static_cast<uint64_t>(bufferEnd - bufferP),
        RINGBUF_BLOCKSIZE);
    CellFsErrno err = cellFsStReadWait(fd, waitSize);
    if (err != CELL_OK)
    {
      Log("error waiting for ring buffer on sequence '%s': err=0x%x (%s)",
          Name(), static_cast<unsigned>(err), GetCellFsErrString(err));
      m_RecordError = true; return -1;
    }
    char *addr = NULL;
    uint64_t readSize = 0;
    err = cellFsStReadGetCurrentAddr(fd, &addr, &readSize);
    if (err != CELL_OK)
    {
      Log("cellFsStReadGetCurrentAddr() err=0x%x (%s)",
          static_cast<unsigned>(err), GetCellFsErrString(err));
      m_RecordError = true;
      return -1;
    }
    readSize = std::min(readSize, waitSize);
    memcpy(bufferP, addr, readSize);
    bufferP += readSize;
    err = cellFsStReadPutCurrentAddr(fd, addr, readSize);
    if (err != CELL_OK)
    {
      Log("cellFsStReadPutCurrentAddr() err=0x%x (%s)",
          static_cast<unsigned>(err), GetCellFsErrString(err));
      m_RecordError = true;
      return -1;
    }
  }
  return 0;
}

int LoadSeq::Playback::Peek(uint32_t &value)
{
  int fd = m_RecordFd;
  CellFsErrno err = cellFsStReadWait(fd, 4);
  if (err != CELL_OK)
  {
    Log("error waiting for ring buffer on sequence '%s': err=0x%x (%s)",
        Name(), static_cast<unsigned>(err), GetCellFsErrString(err));
    m_RecordError = true;
    return -1;
  }
  char *addr = NULL;
  uint64_t readSize = 0;
  err = cellFsStReadGetCurrentAddr(fd, &addr, &readSize);
  if (err != CELL_OK)
  {
    Log("cellFsStReadGetCurrentAddr() err=0x%x (%s)",
        static_cast<unsigned>(err), GetCellFsErrString(err));
    m_RecordError = true;
    return -1;
  }
  readSize = std::min(readSize, static_cast<uint64_t>(4));
  if (readSize != 4)
  {
    Log("unexpected unaligned read from sequence '%s'",
        Name());
    m_RecordError = true;
    return -1;
  }
  // Assume data is alignmed.
  value = *reinterpret_cast<uint32_t *>(addr);
  return 0;
}

namespace
{
  static void HdrToString(
      const uint32_t *hdr,
      size_t hdrLength,
      char *buffer,
      size_t bufferSize
      )
  {
    char *p = buffer, *pEnd = buffer + bufferSize;

    if (hdrLength == 0)
    {
      strncpy(p, "?()", pEnd - p);
      pEnd[-1] = 0;
      return;
    }
    LoadSeq::Function fn = static_cast<LoadSeq::Function>(hdr[0]);
    const char *fnName = FunctionName(fn);
    snprintf(p, pEnd - p, "%s(", fnName);
    pEnd[-1] = 0;
    p += strlen(p);

    switch (fn)
    {
    case LoadSeq::FN_open:
      if (hdrLength > 4)
      {
        size_t pathLen = static_cast<size_t>(hdr[1]);
        int flags = static_cast<int>(hdr[2]);
        int fd = static_cast<int>(hdr[3]);
        int openErrno = static_cast<int>(hdr[4]);
        char pathBuffer[MAX_PATH];
        const char *path = NULL;
        if (hdrLength > 5 + pathLen / 4)
        {
          memcpy(pathBuffer, hdr + 5, std::max(pathLen, sizeof pathBuffer));
          pathBuffer[sizeof pathBuffer - 1] = 0;
          path = pathBuffer;
        }
        if (path != NULL)
          snprintf(p, pEnd - p, "\"%s\", ", path);
        else
          strncpy(p, "?, ", pEnd - p);
        pEnd[-1] = 0;
        p += strlen(p);
        snprintf(p, pEnd - p, "0x%x) = %i [errno=%i %s]",
            static_cast<unsigned>(flags),
            fd,
            openErrno,
            strerror(openErrno));
        pEnd[-1] = 0;
        p += strlen(p);
      }
      else
      {
        if (p < pEnd - 1) { *p++ = ')'; *p++ = 0; }
      }
      break;

    case LoadSeq::FN_lseek:
      if (hdrLength > 1)
      {
        int fd = static_cast<int>(hdr[1]);
        snprintf(p, pEnd - p, "%i, ", fd);
        pEnd[-1] = 0;
        p += strlen(p);
      }
      if (hdrLength > 2)
      {
        off_t offset = static_cast<off_t>(hdr[2]);
        snprintf(p, pEnd - p, "%lu, ", static_cast<unsigned long>(offset));
        pEnd[-1] = 0;
        p += strlen(p);
      }
      if (hdrLength > 3)
      {
        int whence = static_cast<int>(hdr[3]);
        snprintf(p, pEnd - p, "%i)", whence);
      }
      else
        strncpy(p, "?)", pEnd - p);
      pEnd[-1] = 0;
      p += strlen(p);
      break;

    case LoadSeq::FN_cellFsOpen:
      if (hdrLength > 4)
      {
        size_t pathLen = static_cast<size_t>(hdr[1]);
        int flags = static_cast<int>(hdr[2]);
        int fd = static_cast<int>(hdr[3]);
        CellFsErrno err = static_cast<CellFsErrno>(hdr[4]);
        char pathBuffer[MAX_PATH];
        const char *path = NULL;
        if (hdrLength > 5 + pathLen / 4)
        {
          memcpy(pathBuffer, hdr + 5, std::max(pathLen, sizeof pathBuffer));
          pathBuffer[sizeof pathBuffer - 1] = 0;
          path = pathBuffer;
        }
        if (path != NULL)
          snprintf(p, pEnd - p, "\"%s\", ", path);
        else
          strncpy(p, "?, ", pEnd - p);
        pEnd[-1] = 0;
        p += strlen(p);
        snprintf(p, pEnd - p, "0x%x, *fd = %i, NULL, 0) = %i [%s]",
            static_cast<unsigned>(flags),
            fd,
            static_cast<int>(err),
            GetCellFsErrString(err));
        pEnd[-1] = 0;
        p += strlen(p);
      }
      else
      {
        if (p < pEnd - 1) { *p++ = ')'; *p++ = 0; }
      }
      break;

    default:
      // FIXME implement:
      if (p < pEnd - 1) { *p++ = ')'; *p++ = 0; }
      break;
    }
  }
}

__attribute__ ((noinline))
void LoadSeq::Playback::Mismatch(
    const char *fn,
    const uint32_t *hdr,
    size_t hdrLength,
    const char *format,
    ...
    )
{
  char buffer[1024];

  Log("playback mismatch in sequence '%s': "
      "unexpected %s() call, playback cancelled",
      Name(), fn);
  if (hdr != NULL)
  {
    HdrToString(hdr, hdrLength, buffer, sizeof buffer);
    buffer[sizeof buffer - 1] = 0;
    Log("  expected call: %s\n", buffer);
  }
  if (format != NULL)
  {
    va_list ap;
    va_start(ap, format);
    vsnprintf(buffer, sizeof buffer, format, ap);
    buffer[sizeof buffer - 1] = 0;
    va_end(ap);
    Log("  received call: %s\n", buffer);
  }
  Cancel();
}

int LoadSeq::Playback::Open(const char *filename, const char *name)
{
  Init(filename, name);
  int fd = -1;
  CellFsErrno err = __REAL__cellFsOpen(
      filename, CELL_FS_O_RDONLY, &fd, NULL, 0);
  if (err != CELL_OK)
  {
    if (err == CELL_FS_ENOENT)
    {
      Log("no recording file for sequence '%s'", name);
      if (g_bAutoRecord)
      {
        m_bIsRecording = true;
        return 0;
      }
      else
        return -1;
    }
    else
      Log("error opening recording file '%s': err=0x%x (%s)",
          filename, static_cast<unsigned>(err), GetCellFsErrString(err));
    return -1;
  }
  CellFsStat statBuf;
  err = cellFsFstat(fd, &statBuf);
  if (err != CELL_OK)
  {
    Log("can't stat recording file '%s': err=0x%x (%s)",
        filename, static_cast<unsigned>(err), GetCellFsErrString(err));
    cellFsClose(fd);
    m_RecordError = true;
    return -1;
  }
  CellFsRingBuffer ringBuf;
  memset(&ringBuf, 0, sizeof ringBuf);
  ringBuf.ringbuf_size = RINGBUF_SIZE;
  ringBuf.block_size = RINGBUF_BLOCKSIZE;
  ringBuf.copy = CELL_FS_ST_COPYLESS;
  err = cellFsStReadInit(fd, &ringBuf);
  if (err != CELL_OK)
  {
    Log("error initializing ring buffer for sequence '%s': err=0x%x (%s)",
        name, static_cast<unsigned>(err), GetCellFsErrString(err));
    cellFsClose(fd);
    m_RecordError = true;
    return -1;
  }
  err = cellFsStReadStart(fd, 0, statBuf.st_size);
  if (err != CELL_OK)
  {
    Log("error starting ring buffer for sequence '%s': err=0x%x (%s)",
        name, static_cast<unsigned>(err), GetCellFsErrString(err));
    cellFsClose(fd);
    m_RecordError = true;
    return -1;
  }
  m_RecordFd = fd;
  uint32_t hdr[5];
  if (Read(hdr, sizeof hdr) == -1) return -1;
  if (memcmp(hdr, FILEFORMAT_MAGIC, 12))
  {
    Log("disk file '%s' is not a valid recoding file", filename);
    m_RecordError = true;
    return -1;
  }
  if (hdr[3] != FILEFORMAT_VERSION)
  {
    Log("recoding file '%s' version mismatch: expected v%u, file is v%u",
        filename, FILEFORMAT_VERSION, hdr[3]);
    m_RecordError = true;
    return -1;
  }
  unsigned nExtraHdr = hdr[4];
  if (nExtraHdr > 0x10000)
  {
    Log("recoding file '%s' corrupted", filename);
    m_RecordError = true;
    return -1;
  }
  if (nExtraHdr > 0)
  {
    uint32_t extraHdr[nExtraHdr];
    if (Read(extraHdr, nExtraHdr * 4) == -1) return -1;
  }
  Log("starting playback for sequence '%s' from '%s'",
      Name(), m_RecordFile.c_str());
  return 0;
}

void LoadSeq::Playback::Cancel()
{
  Close();
}

int LoadSeq::Playback::Close()
{
  int fd = m_RecordFd;
  bool error = false;

  if (fd == -1) return 0;
  CellFsErrno err = cellFsStReadStop(fd);
  if (err != CELL_OK)
  {
    Log("error stopping ring buffer for sequence '%s': err=0x%x (%s)",
        Name(), static_cast<unsigned>(err), GetCellFsErrString(err));
    error = true;
  }
  err = cellFsStReadFinish(fd);
  if (err != CELL_OK)
  {
    Log("error closing ring buffer for sequence '%s': err=0x%x (%s)",
        Name(), static_cast<unsigned>(err), GetCellFsErrString(err));
    error = true;
  }
  err = __REAL__cellFsClose(fd);
  m_RecordFd = -1;
  if (err != CELL_OK)
  {
    Log("unexpected error closing playback sequence '%s': err=0x%x (%s)",
        Name(), static_cast<unsigned>(err), GetCellFsErrString(err));
    error = true;
  }
  m_Tracker.End();
  if (!error && !m_RecordError)
    Log("playback for sequence '%s' finished", Name());
  return error ? -1 : 0;
}

int LoadSeq::Playback::LoadSeq_open(const char *path, int flags, mode_t mode)
{
  if (!IsActive()) return __REAL__open(path, flags, mode);

  size_t len = strlen(path);
  char pathLower[len + 1];
  strcpy(pathLower, path);
  strlwr(pathLower);

  const int rwMask = O_RDWR | O_RDONLY | O_WRONLY;
  int rwFlags = flags & rwMask;
  if (rwFlags == O_RDWR && IsReadWriteOk(pathLower))
  {
    flags &= ~rwMask;
    flags |= O_RDONLY;
    rwFlags = O_RDONLY;
  }
  if (rwFlags != O_RDONLY || IsPassThrough(pathLower))
    return __REAL__open(path, flags, mode);
  Tick();
  uint32_t hdr[1024];
  if (Peek(hdr[0]) == -1) return -1;
  if (hdr[0] != FN_open)
  {
    Mismatch("open", hdr, 1,
        "open(\"%s\", 0x%x)", path, static_cast<unsigned>(flags));
    return __REAL__open(path, flags, mode);
  }
  if (Read(hdr, 5 * 4) == -1) return -1;
  size_t pathLen = hdr[1];
  if (pathLen > sizeof hdr - 64)
  {
    Log("corrupted sequence file '%s', bad FN_open record",
        m_RecordFile.c_str());
    m_RecordError = true;
    return -1;
  }
  if (Read(hdr + 5, (pathLen / 4 + 1) * 4) == -1) return -1;
  if (pathLen != len
      || strcmp(path, reinterpret_cast<const char *>(hdr + 5))
      || flags != hdr[2])
  {
    Mismatch("open", hdr, sizeof hdr / sizeof hdr[0],
        "open(\"%s\", 0x%x)", path, static_cast<unsigned>(flags));
    return __REAL__open(path, flags, mode);
  }
  if (hdr[4] != 0) errno = hdr[4];
  m_Tracker.Open(hdr[3], path, true);
  return hdr[3] | FD_RECORDED;
}

ssize_t LoadSeq::Playback::LoadSeq_read(int fd, void *buffer, size_t size)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    ssize_t result;
    if (IsActive())
    {
      uint32_t hdr[4];
      if (Read(hdr, sizeof hdr) == -1) return -1;
      if (hdr[0] != FN_read || fd != hdr[1] || size != hdr[2])
      {
        Mismatch("read", hdr, sizeof hdr / sizeof hdr[0],
            "read(%i, buffer, %lu)",
            fd, static_cast<unsigned long>(size));
        if (!IsPersistent(fd)) return -1;
        result = __REAL__read(RealFd(fd), buffer, size);
      }
      else
      {
        result = static_cast<ssize_t>(hdr[3]);
        if (Read(buffer, result) == -1) return -1;
        if (result % 4 && Read(hdr, 4 - result % 4) == -1) return -1;
      }
    }
    else
      result = __REAL__read(RealFd(fd), buffer, size);
    if (result > 0)
      m_Tracker.Read(fd, result);
    return result;
  }
  else
    return __REAL__read(fd, buffer, size);
}

off_t LoadSeq::Playback::LoadSeq_lseek(int fd, off_t offset, int whence)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    off_t result;
    if (IsActive())
    {
      uint32_t hdr[5];
      if (Read(hdr, sizeof hdr) == -1) return -1;
      if (hdr[0] != FN_lseek
          || hdr[1] != fd
          || hdr[2] != offset
          || hdr[3] != whence)
      {
        Mismatch("lseek", hdr, sizeof hdr / sizeof hdr[0],
            "lseek(%i, %lu, %i)",
            fd, static_cast<unsigned long>(offset), whence);
        if (!IsPersistent(fd)) return -1;
        result = __REAL__lseek(RealFd(fd), offset, whence);
      }
      else
        result = static_cast<off_t>(hdr[4]);
    }
    else
      result = __REAL__lseek(RealFd(fd), offset, whence);
    if (result >= 0) m_Tracker.Seek(fd, result);
    return result;
  }
  else
    return __REAL__lseek(fd, offset, whence);
}

off64_t LoadSeq::Playback::LoadSeq_lseek64(int fd, off64_t offset, int whence)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    off64_t result;
    if (IsActive())
    {
      uint32_t hdr[7];
      if (Read(hdr, sizeof hdr) == -1) return -1;
      if (hdr[0] != FN_lseek64
          || hdr[1] != fd
          || hdr[2] != (uint32_t)(offset >> 32)
          || hdr[3] != (uint32_t)offset
          || hdr[4] != whence)
      {
        Mismatch("lseek64", hdr, sizeof hdr / sizeof hdr[0],
            "lseek64(%i, %llu, %i)",
            fd, static_cast<unsigned long long>(offset), whence);
        if (!IsPersistent(fd)) return -1;
        result = __REAL__lseek64(RealFd(fd), offset, whence);
      }
      else
        result = (static_cast<off64_t>(hdr[5]) << 32) | hdr[6];
    }
    else
      result = __REAL__lseek64(RealFd(fd), offset, whence);
    if (result >= 0) m_Tracker.Seek(fd, result);
    return result;
  }
  else
    return __REAL__lseek64(fd, offset, whence);
}

int LoadSeq::Playback::LoadSeq_stat(const char *path, struct stat *buf)
{
  if (!IsActive()) return __REAL__stat(path, buf);

  size_t len = strlen(path);
  char pathLower[len + 1];
  strcpy(pathLower, path);
  strlwr(pathLower);

  if (IsPassThrough(pathLower))
    return __REAL__stat(path, buf);
  Tick();
  uint32_t hdr[1024];
  if (Peek(hdr[0]) == -1) return -1;
  if (hdr[0] != FN_stat)
  {
    Mismatch("stat", hdr, 1, "stat(\"%s\", buf)", path);
    return __REAL__stat(path, buf);
  }
  if (Read(hdr, (LS_STAT_HDR_SIZE + 4) * 4) == -1) return -1;
  size_t pathLen = hdr[1];
  if (pathLen > sizeof hdr - 128)
  {
    Log("corrupted sequence file '%s', bad FN_stat record",
        m_RecordFile.c_str());
    m_RecordError = true;
    return -1;
  }
  if (Read(hdr + LS_STAT_HDR_SIZE + 4, (pathLen / 4 + 1) * 4) == -1)
    return -1;
  if (pathLen != len
      || strcmp(path, reinterpret_cast<const char *>(
          hdr + LS_STAT_HDR_SIZE + 4)))
  {
    Mismatch("stat", hdr, sizeof hdr / sizeof hdr[0],
        "stat(\"%s\", buf)", path);
    return __REAL__stat(path, buf);
  }
  if (buf != NULL) LS_hdr2stat(buf, hdr + 2);
  if (hdr[LS_STAT_HDR_SIZE + 3]) errno = hdr[LS_STAT_HDR_SIZE + 4];
  return static_cast<int>(hdr[LS_STAT_HDR_SIZE + 3]);
}

int LoadSeq::Playback::LoadSeq_fstat(int fd, struct stat *buf)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    int result;
    if (IsActive())
    {
      uint32_t hdr[LS_STAT_HDR_SIZE + 4];
      if (Read(hdr, sizeof hdr) == -1) return -1;
      if (hdr[0] != FN_fstat || hdr[1] != fd)
      {
        Mismatch("fstat", hdr, sizeof hdr / sizeof hdr[0],
            "fstat(%i, buf)", fd);
        if (!IsPersistent(fd)) return -1;
        result = __REAL__fstat(RealFd(fd), buf);
      }
      else
      {
        if (buf != NULL) LS_hdr2stat(buf, hdr + 2);
        if (hdr[LS_STAT_HDR_SIZE + 4]) errno = hdr[LS_STAT_HDR_SIZE + 4];
        result = static_cast<int>(hdr[LS_STAT_HDR_SIZE + 3]);
      }
    }
    else
      result = __REAL__fstat(RealFd(fd), buf);
    return result;
  }
  else
    return __REAL__fstat(fd, buf);
}

int LoadSeq::Playback::LoadSeq_close(int fd)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    int result;
    if (IsActive())
    {
      uint32_t hdr[3];
      if (Read(hdr, sizeof hdr) == -1) return -1;
      if (hdr[0] != FN_close || hdr[1] != fd)
      {
        Mismatch("close", hdr, sizeof hdr / sizeof hdr[0], "close(%i)", fd);
        if (!IsPersistent(fd)) return -1;
        result = 0;
      }
      else
        result = static_cast<int>(hdr[2]);
    }
    m_Tracker.Close(fd);
    return result;
  }
  else
    return __REAL__close(fd);
}

CellFsErrno LoadSeq::Playback::LoadSeq_cellFsOpen(
    const char *path,
    int flags,
    int *fd,
    const void *arg,
    uint64_t size
    )
{
  if (!IsActive()) return __REAL__cellFsOpen(path, flags, fd, arg, size);

  size_t len = strlen(path);
  char pathLower[len + 1];
  strcpy(pathLower, path);
  strlwr(pathLower);

  const int rwMask = CELL_FS_O_RDWR | CELL_FS_O_RDONLY | CELL_FS_O_WRONLY;
  int rwFlags = flags & rwMask;
  if (rwFlags == CELL_FS_O_RDWR && IsReadWriteOk(pathLower))
  {
    flags &= ~rwMask;
    flags |= CELL_FS_O_RDONLY;
    rwFlags = CELL_FS_O_RDONLY;
  }
  if (rwFlags != CELL_FS_O_RDONLY || IsPassThrough(path))
    return __REAL__cellFsOpen(path, flags, fd, arg, size);
  uint32_t hdr[1024];
  Tick();
  if (Peek(hdr[0]) == -1) return CELL_FS_EIO;
  if (hdr[0] != FN_cellFsOpen)
  {
    Mismatch("cellFsOpen", hdr, 1,
        "cellFsOpen(\"%s\", 0x%x, *fd, NULL, 0)",
        path, static_cast<unsigned>(flags));
    return __REAL__cellFsOpen(path, flags, fd, arg, size);
  }
  if (Read(hdr, 5 * 4) == -1) return CELL_FS_EIO;
  size_t pathLen = hdr[1];
  if (pathLen > sizeof hdr - 64)
  {
    Log("corrupted sequence file '%s', bad FN_cellFsOpen record",
        m_RecordFile.c_str());
    m_RecordError = true;
    return CELL_FS_EIO;
  }
  if (Read(hdr + 5, (pathLen / 4 + 1) * 4) == -1) return CELL_FS_EIO;
  if (pathLen != strlen(path)
      || strcmp(path, reinterpret_cast<const char *>(hdr + 5))
      || flags != hdr[2])
  {
    Mismatch("cellFsOpen", hdr, sizeof hdr / sizeof hdr[0],
        "cellFsOpen(\"%s\", 0x%x, *fd, NULL, 0)",
        path, static_cast<unsigned>(flags));
    return __REAL__cellFsOpen(path, flags, fd, arg, size);
  }
  if (fd != NULL)
  {
    *fd = static_cast<int>(hdr[3]);
    if (*fd != -1) *fd |= FD_RECORDED;
  }
  m_Tracker.Open(hdr[3], path, true);
  return static_cast<CellFsErrno>(hdr[4]);
}

CellFsErrno LoadSeq::Playback::LoadSeq_cellFsRead(
    int fd,
    void *buffer,
    uint64_t nbytes,
    uint64_t *nread
    )
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    uint64_t nreadBuf = 0;
    CellFsErrno err;
    if (IsActive())
    {
      uint32_t hdr[7];
      if (Read(hdr, sizeof hdr) == -1) return CELL_FS_EIO;
      if (hdr[0] != FN_cellFsRead
          || hdr[1] != fd
          || hdr[2] != (uint32_t)(nbytes >> 32)
          || hdr[3] != (uint32_t)nbytes)
      {
        Mismatch("cellFsRead", hdr, sizeof hdr / sizeof hdr[0],
            "cellFsRead(%i, buffer, %llu, nread)",
            fd, static_cast<unsigned long long>(nbytes));
        if (!IsPersistent(fd)) return CELL_FS_EIO;
        err = __REAL__cellFsRead(RealFd(fd), buffer, nbytes, &nreadBuf);
      }
      else
      {
        nreadBuf = ((uint64_t)hdr[4] << 32) | hdr[5];
        if (Read(buffer, static_cast<size_t>(nreadBuf)) == -1)
          return CELL_FS_EIO;
        if (nreadBuf % 4 && Read(hdr, 4 - nreadBuf % 4) == -1)
          return CELL_FS_EIO;
        err = static_cast<CellFsErrno>(hdr[6]);
      }
    }
    else
      err = __REAL__cellFsRead(RealFd(fd), buffer, nbytes, &nreadBuf);
    if (nread != NULL) *nread = nreadBuf;
    if (err == CELL_OK) m_Tracker.Read(fd, nreadBuf);
    return err;
  }
  else
    return __REAL__cellFsRead(fd, buffer, nbytes, nread);
}

CellFsErrno LoadSeq::Playback::LoadSeq_cellFsLseek(
    int fd,
    int64_t offset,
    int whence,
    uint64_t *pos
    )
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    uint64_t posBuf = 0;
    CellFsErrno err;
    if (IsActive())
    {
      uint32_t hdr[8];
      if (Read(hdr, sizeof hdr) == -1) return CELL_FS_EIO;
      if (hdr[0] != FN_cellFsLseek
          || hdr[1] != fd
          || hdr[2] != (uint32_t)(offset >> 32)
          || hdr[3] != (uint32_t)offset
          || hdr[4] != whence)
      {
        Mismatch("cellFsLseek", hdr, sizeof hdr / sizeof hdr[0],
            "cellFsLseek(%i, %llu, %i, pos)",
            fd, static_cast<unsigned long long>(offset), whence);
        if (!IsPersistent(fd)) return CELL_FS_EIO;
        err = __REAL__cellFsLseek(RealFd(fd), offset, whence, &posBuf);
      }
      else
      {
        posBuf = ((uint64_t)hdr[5] << 32) | hdr[6];
        err = static_cast<CellFsErrno>(hdr[7]);
      }
    }
    else
      err = __REAL__cellFsLseek(RealFd(fd), offset, whence, &posBuf);
    if (pos != NULL) *pos = posBuf;
    if (err == CELL_OK) m_Tracker.Seek(fd, posBuf);
    return err;
  }
  else
    return __REAL__cellFsLseek(fd, offset, whence, pos);
}

CellFsErrno LoadSeq::Playback::LoadSeq_cellFsStat(
    const char *path,
    CellFsStat *sb
    )
{
  if (!IsActive()) return __REAL__cellFsStat(path, sb);

  size_t len = strlen(path);
  char pathLower[len + 1];
  strcpy(pathLower, path);
  strlwr(pathLower);

  if (IsPassThrough(pathLower)) return __REAL__cellFsStat(path, sb);
  Tick();
  uint32_t hdr[1024];
  if (Peek(hdr[0]) == -1) return CELL_FS_EIO;
  if (hdr[0] != FN_cellFsStat)
  {
    Mismatch("cellFsStat", hdr, 1, "cellFsStat(\"%s\", sb)", path);
    return __REAL__cellFsStat(path, sb);
  }
  if (Read(hdr, (LS_CELLFSSTAT_HDR_SIZE + 3) * 4) == -1) return CELL_FS_EIO;
  size_t pathLen = hdr[1];
  if (pathLen > sizeof hdr - 128)
  {
    Log("corrupted sequence file '%s', bad FN_cellFsStat record",
        m_RecordFile.c_str());
    m_RecordError = true;
    return CELL_FS_EIO;
  }
  if (Read(hdr + LS_CELLFSSTAT_HDR_SIZE + 3, (pathLen / 4 + 1) * 4) == -1)
    return CELL_FS_EIO;
  if (pathLen != strlen(path)
      || strcmp(path, reinterpret_cast<const char *>(
          hdr + LS_CELLFSSTAT_HDR_SIZE + 3)))
  {
    Mismatch("cellFsStat", hdr, sizeof hdr / sizeof hdr[0],
        "cellFsStat(\"%s\", sb)", path);
    return __REAL__cellFsStat(path, sb);
  }
  if (sb != NULL) LS_hdr2CellFsStat(sb, hdr + 2);
  return static_cast<CellFsErrno>(hdr[LS_CELLFSSTAT_HDR_SIZE + 2]);
}

CellFsErrno LoadSeq::Playback::LoadSeq_cellFsFstat(int fd, CellFsStat *sb)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    CellFsErrno err;
    if (IsActive())
    {
      uint32_t hdr[LS_CELLFSSTAT_HDR_SIZE + 3];
      if (Read(hdr, sizeof hdr) == -1) return CELL_FS_EIO;
      if (hdr[0] != FN_cellFsFstat || hdr[1] != fd)
      {
        Mismatch("cellFsFstat", hdr, sizeof hdr / sizeof hdr[0],
            "cellFsFstat(%i, fd)", fd);
        if (!IsPersistent(fd)) return CELL_FS_EIO;
        err = __REAL__cellFsFstat(RealFd(fd), sb);
      }
      else
      {
        if (sb != NULL) LS_hdr2CellFsStat(sb, hdr + 2);
        err = static_cast<CellFsErrno>(hdr[LS_CELLFSSTAT_HDR_SIZE + 3]);
      }
    }
    else
      err == __REAL__cellFsFstat(RealFd(fd), sb);
    return err;
  }
  else
    return __REAL__cellFsFstat(fd, sb);
}

CellFsErrno LoadSeq::Playback::LoadSeq_cellFsClose(int fd)
{
  if (fd & FD_RECORDED)
  {
    Tick();
    fd &= ~FD_RECORDED;
    CellFsErrno err = CELL_OK;
    if (IsActive())
    {
      uint32_t hdr[3];
      if (Read(hdr, sizeof hdr) == -1) return CELL_FS_EIO;
      if (hdr[0] != FN_cellFsClose || hdr[1] != fd)
      {
        Mismatch("cellFsClose", hdr, sizeof hdr / sizeof hdr[0],
            "cellFsClose(%i)", fd);
        if (!IsPersistent(fd)) return CELL_FS_EIO;
      }
      else
        err = static_cast<CellFsErrno>(hdr[2]);
    }
    m_Tracker.Close(fd);
    return err;
  }
  else
    return __REAL__cellFsClose(fd);
}

#define LOADSEQ_DISPATCH(FUNC, ...) \
  if (LoadSeq::g_MainThreadInitialized) \
  { \
    sys_ppu_thread_t currentThread; \
    sys_ppu_thread_get_id(&currentThread); \
    if (currentThread == LoadSeq::g_MainThread) \
    { \
      switch (LoadSeq::g_IOMode) \
      { \
      case LoadSeq::PASSTHROUGH: \
      case LoadSeq::DELETE: \
        break; \
      case LoadSeq::PLAYBACK: \
        if (!LoadSeq::g_Playback.IsRecording()) \
          return LoadSeq::g_Playback.LoadSeq_ ## FUNC(__VA_ARGS__); \
        /* Fall through. */ \
      case LoadSeq::RECORD: \
        return LoadSeq::g_Recorder.LoadSeq_ ## FUNC(__VA_ARGS__); \
      } \
    } \
  } \
  return __REAL__ ## FUNC(__VA_ARGS__);

extern "C" int open(const char *path, int flags, mode_t mode)
{
  LOADSEQ_DISPATCH(open, path, flags, mode);
}

extern "C" ssize_t read(int fd, void *buffer, size_t size)
{
  LOADSEQ_DISPATCH(read, fd, buffer, size);
}

extern "C" off_t lseek(int fd, off_t offset, int whence)
{
  LOADSEQ_DISPATCH(lseek, fd, offset, whence);
}

extern "C" off64_t lseek64(int fd, off64_t offset, int whence)
{
  LOADSEQ_DISPATCH(lseek64, fd, offset, whence);
}

extern "C" int stat(const char *path, struct stat *buf)
{
  LOADSEQ_DISPATCH(stat, path, buf);
}

extern "C" int fstat(int fd, struct stat *buf)
{
  LOADSEQ_DISPATCH(fstat, fd, buf);
}

extern "C" int close(int fd)
{
  LOADSEQ_DISPATCH(close, fd);
}

extern "C" CellFsErrno cellFsOpen(
    const char *path,
    int flags,
    int *fd,
    const void *arg,
    uint64_t size
    )
{
  LOADSEQ_DISPATCH(cellFsOpen, path, flags, fd, arg, size);
}

extern "C" CellFsErrno cellFsRead(
    int fd,
    void *buffer,
    uint64_t nbytes,
    uint64_t *nread
    )
{
  LOADSEQ_DISPATCH(cellFsRead, fd, buffer, nbytes, nread);
}

extern "C" CellFsErrno cellFsLseek(
    int fd,
    int64_t offset,
    int whence,
    uint64_t *pos
    )
{
  LOADSEQ_DISPATCH(cellFsLseek, fd, offset, whence, pos);
}

extern "C" CellFsErrno cellFsStat(const char *path, CellFsStat *sb)
{
  LOADSEQ_DISPATCH(cellFsStat, path, sb);
}

extern "C" CellFsErrno cellFsFstat(int fd, CellFsStat *sb)
{
  LOADSEQ_DISPATCH(cellFsFstat, fd, sb);
}

extern "C" CellFsErrno cellFsClose(int fd)
{
  LOADSEQ_DISPATCH(cellFsClose, fd);
}

// Set the operation mode of the load sequencer.
//
// This function must be called before 'LoadSeq::Begin()' is called to
// initialize a record or playback operation.
//
// The parameter 'mode' is one of the following:
// - LoadSeq::PASSTHROUGH: The load sequencer is disabled.  All calls are
//   passed through to the system unchanged.  This is the default mode.
// - LoadSeq::PLAYBACK: Sequenced operations should be played back from a
//   recording file (if available).  If no recording file is available (or if
//   the operations are out of sync), then calls are passed through to the
//   system.
// - LoadSeq::RECORD: Sequenced operations are recorded to a recording file.
// - LoadSeq::DELETE: Like PASSTHROUGH, but deletes any recording file
//   referenced in a LoadSeq::Begin() call.
//
// This function may be called more than once before the first call to
// 'LoadSeq::Begin()'.
void LoadSeq::Mode(LoadSeq::IOMode mode)
{
  pthread_mutex_lock(&g_Lock);
  assert(g_SetModeOK);
  g_IOMode = mode;
  pthread_mutex_unlock(&g_Lock);
}

// Set or clear the auto recording flag.
//
// If the auto recording flag is set, then an attempt to playback a recording
// file that does not exist causes that recording file to be created.
void LoadSeq::SetAutoRecord(bool enable) { g_bAutoRecord = enable; }

// Cancel the operation of the load sequencer.
//
// All active playbacks or recordings are cancelled and the incomplete
// recorded files are deleted.
void LoadSeq::Cancel()
{
  pthread_mutex_lock(&g_Lock);
  switch (g_IOMode)
  {
  case PASSTHROUGH:
  case DELETE:
    // Nothing to do.
    break;
  case PLAYBACK:
    if (g_Playback.IsRecording()) g_Recorder.Cancel();
    g_Playback.Cancel();
    break;
  case RECORD:
    g_Recorder.Cancel();
    break;
  }
  // Note: We don't set the load sequencer to passthrough mode, because
  // persistent file descriptors still need to be tracked correctly (which is
  // done by the cancelled playback/recorder).
  //g_IOMode = PASSTHROUGH;
  pthread_mutex_unlock(&g_Lock);
}

// Begin a sequence.
//
// The parameter specifies a symbolic name for the recording - the recording
// file name is derived from the specified name.
//
// Depening on the operation mode, the call starts a playback or a recording
// or does nothing (passthrough).
void LoadSeq::Begin(const char *name)
{
  pthread_mutex_lock(&g_Lock);

  if (!g_MainThreadInitialized)
  {
    sys_ppu_thread_get_id(&g_MainThread);
    g_MainThreadInitialized = true;
  }

  if (g_IOMode == PASSTHROUGH)
  {
    pthread_mutex_unlock(&g_Lock);
    return;
  }

  char nameBuffer[MAX_PATH];
  if (RecordFilename(name, nameBuffer, sizeof nameBuffer) == -1)
  {
    pthread_mutex_unlock(&g_Lock);
    return;
  }

  if (g_IOMode == DELETE)
  {
    cellFsUnlink(nameBuffer);
    Log("deleting recording for sequence '%s': %s\n",
        name, nameBuffer);
    pthread_mutex_unlock(&g_Lock);
    return;
  }

  bool isRecording = false;
  if (g_IOMode == PLAYBACK)
  {
    Playback &playback = g_Playback;
    if (playback.IsActive())
    {
      Log("can not start playback '%s' while playback '%s' is running",
          name, playback.Name());
      abort();
    }
    playback.Open(nameBuffer, name);
    // The playback will set the IsRecording flag if no recording file is
    // found.  In that case we'll also initialize the recorder.
    if (playback.IsRecording()) isRecording = true;
  }
  else
  {
    assert(g_IOMode == RECORD);
    isRecording = true;
  }
  if (isRecording)
  {
    Recorder &recorder = g_Recorder;
    if (recorder.IsActive())
    {
      Log("can not start sequence '%s' while sequence '%s' is running",
          name, recorder.Name());
      abort();
    }
    recorder.Open(nameBuffer, name);
  }

  pthread_mutex_unlock(&g_Lock);
}

// End a sequence.
//
// The specified name must match the recording name of the preceeding Begin()
// call within the same thread.
void LoadSeq::End(const char *name)
{
  pthread_mutex_lock(&g_Lock);

  bool found = false;

  switch (g_IOMode)
  {
  case PASSTHROUGH:
  case DELETE:
    found = true;
    break;
  case PLAYBACK:
    {
      Playback &playback = g_Playback;
      bool isRecording = playback.IsRecording();
      if (!strcmp(name, playback.Name()))
      {
        found = true;
        playback.Close();
        if (isRecording) g_Recorder.Close();
      }
    }
    break;
  case RECORD:
    {
      Recorder &recorder = g_Recorder;
      if (!strcmp(name, recorder.Name()))
      {
        found = true;
        recorder.Close();
      }
    }
    break;
  }

  if (!found)
  {
    Log("call to End(name = \"%s\") with no matching begin call", name);
    abort();
  }

  pthread_mutex_unlock(&g_Lock);
}

#endif // defined(USE_LOADSEQ)

// vim:sw=2:ts=2:expandtab

