#ifndef _MTraceDump_h_
#define _MTraceDump_h_ 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <errno.h>

#include <set>
#include <vector>
#include <string>

class Profile;
class ProfileType;

class CAddr
{
	uint32_t m_Addr;
	mutable char *m_Desc;

	// List of all operations linked called from the address.
	mutable std::vector<unsigned> m_Ops;

	// List of all nodes linked to the address.
	mutable std::vector<unsigned> m_Nodes;

public:
	CAddr() : m_Addr(0), m_Desc(NULL) { }

	CAddr(uint32_t addr) : m_Addr(addr), m_Desc(NULL) { }

	CAddr(const CAddr &addr) : m_Addr(addr.m_Addr), m_Desc(NULL)
	{
		SetDesc(addr.m_Desc);
	}

	~CAddr() { delete[] m_Desc; }

	CAddr &operator= (const CAddr &addr)
	{
		m_Addr = addr.m_Addr;
		SetDesc(addr.m_Desc);
		return *this;
	}

	bool operator< (const uint32_t addr) const { return m_Addr < addr; }
	bool operator<= (const uint32_t addr) const { return m_Addr <= addr; }
	bool operator> (const uint32_t addr) const { return m_Addr > addr; }
	bool operator>= (const uint32_t addr) const { return m_Addr >= addr; }
	bool operator!= (const uint32_t addr) const { return m_Addr != addr; }
	bool operator== (const uint32_t addr) const { return m_Addr == addr; }

	bool operator< (const CAddr &addr) const { return *this < addr.m_Addr; }
	bool operator<= (const CAddr &addr) const { return *this <= addr.m_Addr; }
	bool operator> (const CAddr &addr) const { return *this > addr.m_Addr; }
	bool operator>= (const CAddr &addr) const { return *this >= addr.m_Addr; }
	bool operator!= (const CAddr &addr) const { return *this != addr.m_Addr; }
	bool operator== (const CAddr &addr) const { return *this == addr.m_Addr; }

	operator uint32_t () const { return m_Addr; }

	const char *Desc() const { return m_Desc; }

	const CAddr &SetDesc(const char *desc) const
	{
		delete[] m_Desc;
		if (desc != NULL)
		{
			size_t descLen = strlen(desc);
			m_Desc = new char[descLen + 1];
			strcpy(m_Desc, desc);
		}
		else
			m_Desc = NULL;
		return *this;
	}

	uint32_t Addr() const { return m_Addr; }

	void AddOp(unsigned opId) const { m_Ops.push_back(opId); }
	const std::vector<unsigned> &Ops() const { return m_Ops; }

	void AddNode(unsigned nodeId) const { m_Nodes.push_back(nodeId); }
	const std::vector<unsigned> &Nodes() const { return m_Nodes; }
};

class CAddrMap
{
	std::set<CAddr> m_Map;

	CAddrMap(const CAddrMap &);
	CAddrMap &operator= (const CAddrMap &);

public:
	CAddrMap() { }
	~CAddrMap() { }

	const CAddr &operator[] (uint32_t addr) const
	{
		const static CAddr nullAddr;
		std::set<CAddr>::const_iterator it = m_Map.find(addr);
		if (it == m_Map.end()) return nullAddr;
		return *it;
	}

	const CAddr &operator[] (uint32_t addr)
	{
		const CAddr addrKey = addr;
		std::set<CAddr>::iterator it = m_Map.lower_bound(addrKey);
		if (it == m_Map.end() || *it != addr)
			return *m_Map.insert(addrKey).first;
		else
			return *it;
	}
};

enum MemOpFn
{
	FN_calloc = 0x63616c6c,
	FN_malloc = 0x6d616c6c,
	FN_free = 0x66726565,
	FN_realloc = 0x7265616c,
	FN_valloc = 0x76616c6c,
	FN_memalign = 0x6d656d61
};

inline int NumArgs(MemOpFn fn)
{
	switch (fn)
	{
	case FN_calloc: return 3;
	case FN_malloc: return 2;
	case FN_free: return 1;
	case FN_realloc: return 3;
	case FN_valloc: return 2;
	case FN_memalign: return 3;
	}
	return -1;
}

const char *MemOpFnName(MemOpFn fn);

class MemOp
{
	MemOpFn m_Fn;
	uint32_t m_Addr;
	uint32_t m_Args[3];
	unsigned m_Parent;
	unsigned m_AllocId;
	unsigned m_FreeId;

public:
	MemOp(
			MemOpFn fn,
			uint32_t addr,
			const uint32_t *args
			)
		: m_Fn(fn),
			m_Addr(addr),
			m_Parent(~0U),
			m_AllocId(~0U),
			m_FreeId(~0U)
	{
		memcpy(m_Args, args, sizeof m_Args);
	}

	MemOpFn Fn() const { return m_Fn; }

	uint32_t Addr() const { return m_Addr; }

	uint32_t Arg(unsigned index) const { return m_Args[index]; }

	unsigned Parent() const { return m_Parent; }
	void SetParent(unsigned nodeId) { m_Parent = nodeId; }

	unsigned AllocId() const { return m_AllocId; }
	void SetAllocId(unsigned id) { m_AllocId = id; }

	unsigned FreeId() const { return m_FreeId; }
	void SetFreeId(unsigned id) { m_FreeId = id; }

	bool IsAlloc() const
	{
		return m_Fn == FN_calloc
			|| m_Fn == FN_malloc
			|| m_Fn == FN_valloc
			|| m_Fn == FN_memalign
			|| (m_Fn == FN_realloc && m_Args[0] == 0);
	}
	bool IsRealloc() const
	{
		return m_Fn == FN_realloc && m_Args[0] != 0;
	}
	bool IsFree() const { return m_Fn == FN_free; }

	uint32_t Size() const
	{
		switch (m_Fn)
		{
		case FN_calloc: return m_Args[0] * m_Args[1];
		case FN_malloc: return m_Args[0];
		case FN_realloc: return m_Args[1];
		case FN_valloc: return m_Args[0];
		case FN_memalign: return m_Args[1];
		default: break;
		}
		return 0;
	}

	uint32_t Pointer() const
	{
		switch (m_Fn)
		{
		case FN_calloc: return m_Args[2];
		case FN_malloc: return m_Args[1];
		case FN_realloc: return m_Args[2];
		case FN_valloc: return m_Args[1];
		case FN_memalign: return m_Args[2];
		case FN_free: return m_Args[0];
		}
		return 0;
	}

	uint32_t FreedSize(const Profile &) const;

	bool IsDescendantOf(const Profile &, unsigned nodeId) const;

	template <typename IterT>
	bool IsDescendantOf(
			const Profile &profile,
			IterT itBegin,
			IterT itEnd
			) const
	{
		for (IterT it = itBegin; it != itEnd; ++it)
			if (IsDescendantOf(profile, *it))
				return true;
		return false;
	}

	bool IsDescendantOf(const Profile &,
			const std::vector<bool> &nodeIds) const;
	
	void Dump(const Profile &, FILE *out, unsigned depth = 0) const;
	void DumpStack(const Profile &, FILE *out, unsigned indent = 0) const;
	void GetStack(const Profile &, std::vector<std::string> &) const;
};

class Node
{
	uint32_t m_Addr;

	unsigned m_AllocCount;
	unsigned m_ReallocCount;
	unsigned m_FreeCount;
	unsigned m_Allocated;
	unsigned m_AllocatedBlocks;
	unsigned m_TotalAllocated;

	unsigned m_nChildren;
	unsigned *m_Children;
	unsigned m_Parent;
	std::vector<unsigned> m_Ops;

public:
	Node()
		: m_Addr(0),
			m_AllocCount(0),
			m_ReallocCount(0),
			m_FreeCount(0),
			m_Allocated(0),
			m_AllocatedBlocks(0),
			m_TotalAllocated(0),
			m_nChildren(0),
			m_Children(NULL),
			m_Parent(~0U)
	{ }

	Node(const Node &node)
		: m_nChildren(0),
			m_Children(NULL)
	{
		*this = node;
	}

	Node(uint32_t addr, unsigned parent)
		: m_Addr(addr),
			m_AllocCount(0),
			m_ReallocCount(0),
			m_FreeCount(0),
			m_Allocated(0),
			m_AllocatedBlocks(0),
			m_TotalAllocated(0),
			m_nChildren(0),
			m_Children(NULL),
			m_Parent(parent)
	{ }

	~Node()
	{
		if (m_Children != NULL) free(m_Children);
	}

	Node &operator= (const Node &node)
	{
		m_Addr = node.m_Addr;
		m_AllocCount = node.m_AllocCount;
		m_ReallocCount = node.m_ReallocCount;
		m_FreeCount = node.m_FreeCount;
		m_Allocated = node.m_Allocated;
		m_AllocatedBlocks = node.m_AllocatedBlocks;
		m_TotalAllocated = node.m_TotalAllocated;
		m_nChildren = node.m_nChildren;
		if (node.m_Children != NULL)
		{
			assert(m_nChildren != 0);
			if (m_Children != NULL)
				m_Children = (unsigned *)realloc(
						m_Children,
						m_nChildren * sizeof m_Children[0]);
			else
				m_Children = (unsigned *)malloc(
						m_nChildren * sizeof m_Children[0]);
			memcpy(m_Children, node.m_Children, m_nChildren * sizeof m_Children[0]);
		}
		else
		{
			assert(m_nChildren == 0);
			if (m_Children != NULL)
			{
				free(m_Children);
				m_Children = NULL;
			}
		}
		m_Parent = node.m_Parent;
		m_Ops = node.m_Ops;
		return *this;
	}

	uint32_t Addr() const { return m_Addr; }

	const unsigned *Children(unsigned &nChildren) const
	{
		nChildren = m_nChildren;
		return m_Children;
	}

	unsigned Parent() const { return m_Parent; }

	void AddChild(unsigned childId)
	{
		if (m_Children == NULL)
		{
			assert(m_nChildren == 0);
			m_Children = (unsigned *)malloc(sizeof m_Children[0]);
			m_nChildren = 1;
		}
		else
		{
			++m_nChildren;
			m_Children = (unsigned *)realloc(
					m_Children, m_nChildren * sizeof m_Children[0]);
		}
		m_Children[m_nChildren - 1] = childId;
	}

	void AddOp(unsigned opId)
	{
		m_Ops.push_back(opId);
	}

	unsigned AllocCount() const { return m_AllocCount; }
	unsigned ReallocCount() const { return m_ReallocCount; }
	unsigned FreeCount() const { return m_FreeCount; }
	unsigned Allocated() const { return m_Allocated; }
	unsigned AllocatedBlocks() const { return m_AllocatedBlocks; }
	unsigned TotalAllocated() const { return m_TotalAllocated; }

	void Alloc(uint32_t size)
	{
		m_Allocated += size;
		++m_AllocatedBlocks;
		m_TotalAllocated += size;
	}

	void Free(uint32_t size)
	{
		assert(size <= m_Allocated);
		m_Allocated -= size;
		assert(m_AllocatedBlocks > 0);
		--m_AllocatedBlocks;
	}

	void CountAlloc() { ++m_AllocCount; }
	void CountRealloc() { ++m_ReallocCount; }
	void CountFree() { ++m_FreeCount; }

	bool IsDescendantOf(const Profile &, unsigned nodeId) const;

	bool IsDescendantOf(const Profile &,
			const std::vector<bool> &nodeIds) const;

	struct Threshold
	{
		unsigned m_MaxDepth;
		unsigned m_MinOpCount;
		unsigned m_MinAllocated;
		unsigned m_DepthBelow;
	};

	void Dump(const Profile &, FILE *out, unsigned depth, Threshold) const;
	void Dump(const Profile &, FILE *out, unsigned depth, const ProfileType &,
			uint64_t threshold) const;
	void DumpSelf(const Profile &, FILE *out, bool numeric = false) const;
	void DumpStack(const Profile &, FILE *out, unsigned frame,
			unsigned indent = 0) const;
	void DumpOps(const Profile &, FILE *out, unsigned depth) const;
	void GetStack(const Profile &, std::vector<std::string> &) const;
};

class Thread
{
	uint32_t m_ThreadId;
	std::vector<unsigned> m_Roots;

public:
	Thread(uint32_t threadId) : m_ThreadId(threadId) { }

	uint32_t ThreadId() const { return m_ThreadId; }
	const std::vector<unsigned> &Roots() const { return m_Roots; }
	void AddRoot(unsigned rootId) { m_Roots.push_back(rootId); }

	unsigned AllocCount(const Profile &) const;
	unsigned ReallocCount(const Profile &) const;
	unsigned FreeCount(const Profile &) const;
	unsigned Allocated(const Profile &) const;
	unsigned AllocatedBlocks(const Profile &) const;
	unsigned TotalAllocated(const Profile &) const;

	void Dump(const Profile &, FILE *out, const Node::Threshold &) const;
	void DumpSummary(const Profile &, FILE *out) const;
};

// A profile type generates a profiling value for a given code address.
class ProfileType
{
protected:
	const Profile &m_Profile;

	// Collect all non-recursive nodes and operations referring to the specified
	// code address.
	void Collect(
			uint32_t addr,
			std::vector<unsigned> &referringNodes,
			std::vector<unsigned> &referringOps) const;

public:
	ProfileType(const Profile &profile) : m_Profile(profile) { }
	virtual ~ProfileType() { }

	// Return the name of the profile type.
	virtual const char *Name() const = 0;

	// Return a description of the value generated by the profile type.
	virtual const char *Desc() const = 0;

	// Get the profiling value for the specified code address.
	uint64_t Value(uint32_t addr) const;

	// Get the profiling value for the specified node.
	virtual uint64_t NodeValue(unsigned nodeId) const = 0;

	// Get the profiling value for the specified memory operation.
	virtual uint64_t OpValue(unsigned opId) const = 0;

	// Get the value cut-off limit.
	virtual uint64_t Limit() const = 0;

	// An address/value pair suitable for sorting.
	struct AddrValuePair
	{
		uint32_t m_Addr;
		uint64_t m_Value;

		AddrValuePair(uint32_t addr) : m_Addr(addr), m_Value(0) { }

		bool operator< (const AddrValuePair &other) const
		{
			return m_Value > other.m_Value;
		}
	};
};

class CompareNodesByProfileValue
{
	const ProfileType &m_Type;

public:
	CompareNodesByProfileValue(const ProfileType &type) : m_Type(type) { }

	bool operator() (unsigned nodeId1, unsigned nodeId2) const
	{
		return m_Type.NodeValue(nodeId1) > m_Type.NodeValue(nodeId2);
	}
};

class CompareOpsByProfileValue
{
	const ProfileType &m_Type;

public:
	CompareOpsByProfileValue(const ProfileType &type) : m_Type(type) { }

	bool operator() (unsigned opId1, unsigned opId2) const
	{
		return m_Type.OpValue(opId1) > m_Type.OpValue(opId2);
	}
};

class ProfileAllocated : public ProfileType
{
public:
	ProfileAllocated(const Profile &profile) : ProfileType(profile) { }

	virtual const char *Name() const;
	virtual const char *Desc() const;
	virtual uint64_t NodeValue(unsigned nodeId) const;
	virtual uint64_t OpValue(unsigned opId) const;
	virtual uint64_t Limit() const;
};

class ProfileOpCount : public ProfileType
{
public:
	ProfileOpCount(const Profile &profile) : ProfileType(profile) { }

	virtual const char *Name() const;
	virtual const char *Desc() const;
	virtual uint64_t NodeValue(unsigned nodeId) const;
	virtual uint64_t OpValue(unsigned opId) const;
	virtual uint64_t Limit() const;
};

class Profile
{
	std::vector<MemOp> m_Ops;
	std::vector<Thread> m_Threads;
	std::vector<Node> m_Nodes;
	std::set<CAddr> m_AddrMap;

	std::string m_ElfFile;
	std::string m_AddrMapFile;

public:
	Profile() { }
	~Profile() { }

	const MemOp &GetOp(unsigned opId) const { return m_Ops[opId]; }

	const Node &GetNode(unsigned nodeId) const { return m_Nodes[nodeId]; }

	unsigned GetNodeId(const Node &node) const
	{
		return static_cast<unsigned>(&node - &m_Nodes.front());
	}

	const Thread &GetThread(unsigned threadId) const
	{
		return m_Threads[threadId];
	}

	const CAddr &GetCAddr(uint32_t addr) const;

	void SetElfFile(const char *file) { m_ElfFile = file; }
	void SetAddrMapFile(const char *file) { m_AddrMapFile = file; }

	int ReadOps(const char *filename);
	void ProcessOp(unsigned opId, uint32_t threadId,
			uint32_t stackDepth, const uint32_t *stack);

	void UpdateTree();
	void UpdateAlloc(uint32_t opId);
	void UpdateFree(uint32_t opId);
	void UpdateOpCount(uint32_t opId);

	void UpdateCAddrMap();

	void DumpProfile(FILE *out, const ProfileType &,
			const std::vector<std::string> *traceList = NULL) const;

	void DumpSummary(FILE *out) const;
	static void DumpSummaryCounters(FILE *out, unsigned, unsigned, unsigned,
			unsigned, unsigned, unsigned);

	void Dump(FILE *out, const Node::Threshold &) const;
	void DumpAddr(FILE *out, uint32_t addr) const;
	void DumpFrame(FILE *out, uint32_t addr, unsigned frame) const;
};

std::string ByteCountToString(const uint64_t count);
void DumpMemInfo(const char *message);

void HLine(FILE *out, char c, unsigned indent = 0);

#endif

