#include "StdAfx.h"
#include "HUD_Tagnames.h"

#include "HUD/HUDDefines.h"
#include "HUD/HUDCVars.h"
#include "HUD/HUD_Crosshair.h"
#include "HUD/ScreenLayoutManager.h"

#include "Flash/Flash.h"

#include "Player.h"

#include "Utility/CryWatch.h"

#include <IWorldQuery.h>
#include <IRenderer.h>

//////////////////////////////////////////////////////////////////////////

#define HUDTN_ORIGWIDTH  1280.0f
#define HUDTN_ORIGHEIGHT 720.0f

static const ColorF COLOR_DEBUG(1.0f,0.8f,0.0f);

//////////////////////////////////////////////////////////////////////////

#undef DEBUG_TAGNAMES
#if ENABLE_HUD_EXTRA_DEBUG
#define DEBUG_TAGNAMES(...) if(g_pGame->GetHUD()->GetCVars()->hud_mpTagNames_debug > 0) { CryWatch("<NewTagNames> " __VA_ARGS__); }
#else
#define DEBUG_TAGNAMES(...) (0)
#endif

//////////////////////////////////////////////////////////////////////////

CHUD_Tagnames::STagInfo::SFlashTagname::SFlashTagname( void )
: vo_name(NULL)
, vo_healthbar(NULL)
{
	// ...
}

void CHUD_Tagnames::STagInfo::SFlashTagname::SetName( const char* name )
{
	vo_name->SetText(name);
}

void CHUD_Tagnames::STagInfo::SFlashTagname::SetHealth( const int healthPercent )
{
	vo_healthbar->Invoke1( "setHealth", healthPercent );
}

void CHUD_Tagnames::STagInfo::SFlashTagname::SetHealthVisible( const bool show )
{
	vo_healthbar->SetVisible( show );
}

//////////////////////////////////////////////////////////////////////////

CHUD_Tagnames::STagInfo::STagInfo()
: m_curFrame(eTIF_Friendly)
, vo_tagname(NULL)
, m_flags(eTIUF_None)
, m_curDepth(-1)
, m_lastDepth(0)
, m_entityId(0)
//, m_size(0)
, m_timeUntilIgnoreEnds(-1.0f)
, m_distance(0)
, m_health(0)
, m_healthVisible(false)
, m_friendly(false)
, m_dead(false)
, m_ignore(false)
, m_visible(false)
, m_pos(0.0f,0.0f,0.0f)
, m_locked(false)
, m_timeUntilLock(0.0f)
, m_timeUntilLockLost(0.0f)
, m_canSee(false)
, m_drawStatus(false)
{
	di_tagname.Clear();
}

CHUD_Tagnames::STagInfo::~STagInfo()
{
	HUD_FLASHOBJ_SAFERELEASE(vo_tagname);
	for(int i=0; i<k_numberOfTagnameFrames; i++)
	{
		HUD_FLASHOBJ_SAFERELEASE( vos_tagnames[i].vo_name );
		HUD_FLASHOBJ_SAFERELEASE( vos_tagnames[i].vo_healthbar );
	}
}

/*const CHUD_Tagnames::STagInfo& CHUD_Tagnames::STagInfo::operator=( const STagInfo& _info )
{
	m_entityId = _info.m_entityId;
	m_friendly = _info.m_friendly;
	m_size     = _info.m_size;
	m_distance = _info.m_distance;
	di_tagname = _info.di_tagname;
	return *this;
}*/

void CHUD_Tagnames::STagInfo::Init(IHUDAsset* pAsset, const int tagidx )
{	
	CryFixedStringT<35> path;

	path.Format( "_root.HUD_Tagnames.tagname%d", tagidx );
	HUD_FLASVAROBJ_REG( pAsset, path.c_str(), vo_tagname );
	vo_tagname->GetDisplayInfo(di_tagname);

	path.Format( "_root.HUD_Tagnames.tagname%d.name", tagidx );
	CryFixedStringT<35> path_healthbar;
	path_healthbar.Format( "_root.HUD_Tagnames.tagname%d.healthbar", tagidx );
	for(int i=0; i<k_numberOfTagnameFrames; i++)
	{
		vo_tagname->GotoAndStop(i);
		HUD_FLASVAROBJ_REG( pAsset, path.c_str(), vos_tagnames[i].vo_name );
		HUD_FLASVAROBJ_REG( pAsset, path_healthbar.c_str(), vos_tagnames[i].vo_healthbar );
	}

	m_lastDepth = m_curDepth;
	m_curDepth = tagidx;

	Clear();
	Update(0,0);
}

void CHUD_Tagnames::STagInfo::Clear( void )
{
	Set( 0, true, false, Vec3(0.0f,0.0f,0.0f), 0, 0, false, 0, false );
}

void CHUD_Tagnames::STagInfo::Set( const EntityId entityId, 
                                   const bool     dead, 
                                   const bool     friendly, 
                                   const Vec3&    pos, 
                                   const float    size, 
                                   const float    distance, 
                                   const bool     visible, 
                                   const int      healthPercent,
                                   const bool     healthVisible )
{
	m_entityId = entityId;
	m_health   = healthPercent;
	m_healthVisible = healthVisible;
	m_friendly = friendly;
	m_dead     = dead;
	m_pos      = pos;
	m_distance = distance;
	m_visible  = visible;

	vo_tagname->SetVisible(visible);
	
	m_flags = eTIUF_Frame | eTIUF_Health | eTIUF_Name | eTIUF_HealthVisiblity;
}

bool CHUD_Tagnames::STagInfo::CanSee( void )
{
	const CHUDCVars* pCvars = g_pGame->GetHUD()->GetCVars();

	if(pCvars->hud_mpTagNames_ForceDraw)
		return true;

	if( m_ignore )
		return false;

	if( m_friendly )
		return (pCvars->hud_mpTagNames_ThroughGeom_friendies>0);

	if( m_dead )
		return (pCvars->hud_mpTagNames_ThroughGeom_dead>0) && g_pGame->GetPlayerVisTable()->CanLocalPlayerSee(m_entityId, 5);

	// Dealing with  enemy
	if( m_locked )
	{
		if( pCvars->hud_mpTagNames_ThroughGeom_enemies )
			return true;

		if( g_pGame->GetPlayerVisTable()->CanLocalPlayerSee(m_entityId, 5) )
			return true;
	}

	return false;
}

void CHUD_Tagnames::STagInfo::UpdateLocks( const float frameDelatTime, const EntityId lookingAt )
{
	if( m_ignore )
	{
		if( m_timeUntilIgnoreEnds >= 0 )
		{
			m_timeUntilIgnoreEnds -= frameDelatTime;
			if( m_timeUntilIgnoreEnds <= 0 )
			{
				m_ignore = false;
				m_timeUntilIgnoreEnds = -1.0f;
			}
		}
	}
	else if( !m_friendly && !m_dead )
	{// Enemy, only draws if locked

		if( lookingAt == m_entityId )
		{//set or reset locks

			/* Looking at range...
			if( ! pPlayer->GetActorSuitGameParameters().IsCloakEnabled() )
			{
				const Vec3 dirToEnemyUnit = (pEntity->GetPos() - pClientActor->GetEntity()->GetPos()).GetNormalized();
				const Vec3 & lookDirUnit = pClientActor->GetGameObject()->GetWorldQuery()->GetDir();
				assert (lookDirUnit.IsUnit());
				addIt = (lookDirUnit.dot(dirToEnemyUnit) > pCvars->hud_tagNames_showOverEnemiesWhenLookDirIsThisAccurate);
				if(!addIt)
				{
					const EntityId lookAt = pClientActor->GetGameObject()->GetWorldQuery()->GetLookAtEntityId();
					addIt = (lookAt == pEntity->GetId());
				}
			}
			*/

			if( m_locked )
			{// locked, reset lock timeout.
				const CHUDCVars* pCvars = g_pGame->GetHUD()->GetCVars();
				m_timeUntilLockLost = pCvars->hud_mpTagNames_Duration;
			}
			else
			{// not locked, start or complete lock

				IActor *pActor = gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(m_entityId);
				assert(pActor);
				CPlayer* pPlayer = static_cast<CPlayer*>(pActor);

				const CHUDCVars* pCvars = g_pGame->GetHUD()->GetCVars();
				const float asyncTime = gEnv->pTimer->GetCurrTime(ITimer::ETIMER_GAME);
				if( pPlayer->GetLastDamageSeconds() + pCvars->hud_mpTagNames_Duration > asyncTime
				    || ( m_timeUntilLock <= 0 ) )
				{
					m_locked = true;
					m_timeUntilLock = pCvars->hud_mpTagNames_LockDuration;
				}
				else
				{
					m_timeUntilLock -= frameDelatTime;
				}
			}
		}
		else
		{// not looking at entity, start or increment lock count down.
			if( m_locked )
			{
				if( m_timeUntilLockLost<=0.0f )
				{// lock lost
					m_locked=false;
				}
				else
				{// decrement lock lost timer
					m_timeUntilLockLost-= frameDelatTime;
				}
			}
		}
	}
}

void CHUD_Tagnames::STagInfo::Update( const float    frameDeltaTime, 
                                      const EntityId lookingAt )
{
	bool bValidActor = ( m_entityId != 0 );

	IActor *pActor = gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(m_entityId);
	CActor* const pClientActor = static_cast<CActor*>(gEnv->pGame->GetIGameFramework()->GetClientActor());	
	if( bValidActor )
	{
		CRY_ASSERT_MESSAGE( pActor && pClientActor, "HUD: Tagnames: Can't find actor/entity although they're being tracked!");
		if( !(pActor 
		      && pClientActor 
					&& pActor->IsPlayer() 
					&& pActor->GetEntity() 
					&& CheckActorIsInValidStateToShowTagName(pActor) 
					&& pActor->GetEntity()->IsActive()) )
		{
			bValidActor = false;
		}
	}

	Vec3 drawPos(0.0f, 0.0f, 0.0f);
	float size=0.0f;
	if( bValidActor && !GetTagNamePosition(pActor->GetEntity(), &drawPos, &size, &m_distance) )
	{
		bValidActor = false;
	}

	if( bValidActor )
	{
		// Dead?
		const bool isDead = pActor->IsDead();
		if( m_dead != isDead )
		{
			m_dead = isDead;
			m_flags |= eTIUF_Frame | eTIUF_Health | eTIUF_Name;
		}

		// Friendly?
		const bool isFriendly = (pClientActor->IsFriendlyEntity(m_entityId) != 0);
		if( m_friendly != isFriendly )
		{
			m_friendly = isFriendly;
			m_flags |= eTIUF_Frame | eTIUF_Health | eTIUF_Name;
		}

		// Health?
		CPlayer* pClientPlayer = static_cast<CPlayer*>(pClientActor);	
		bool drawStatus = pClientPlayer->GetActorSuitGameParameters().GetMode() == eNanoSuitMode_Tactical;
		if( drawStatus && !m_friendly )
		{
			drawStatus = drawStatus && pClientPlayer->IsPerkActive(ePerk_EnemyStatus);
		}

		if( drawStatus )
		{
			const int newHealth = int_round((float)pActor->GetHealth() / (float)pActor->GetMaxHealth() *100.0f);// Percentage for flash...
			if( m_health != newHealth )
			{
				m_health = newHealth;
				m_flags |= eTIUF_Health;
			}

			if( !m_healthVisible )
			{
				m_healthVisible = true;
				m_flags |= eTIUF_HealthVisiblity;
			}
		}
		else
		{
			if( m_healthVisible )
			{
				m_healthVisible = false;
				m_flags |= eTIUF_HealthVisiblity;
			}
		}

		// Visible?
		UpdateLocks( frameDeltaTime, lookingAt );
		m_canSee = CanSee();
		SetVisible( m_canSee );

		// Position
		if( m_canSee )
		{
			GetTagNamePositionScreen( pActor->GetEntity(), &m_pos, &m_distance );
			ConvertScreenToFlashSpace( m_pos );
		}

#if ENABLE_HUD_EXTRA_DEBUG
		if( g_pGame->GetHUD()->GetCVars()->hud_mpTagNames_debug )
		{
			Vec3 posInScreenSpace(-1.0f,-1.0f,-1.0f);
			GetTagNamePositionScreen( pActor->GetEntity(), &posInScreenSpace, &m_distance );

			if( m_canSee )
			{
				DEBUG_TAGNAMES("\t (%s) Rendering %s at s(%d,%d) fs(%d,%d) with %d%% health: Z%d", (m_friendly ? "F" : "E"), pActor->GetEntity()->GetName(), int_round(posInScreenSpace.x), int_round(posInScreenSpace.y), int_round(m_pos.x), int_round(m_pos.y), m_health, m_curDepth );
				// Draw a sphere at the reg point for the tag name.
				Vec3 worldPos(0.0f,0.0f,0.0f);
				float unused=0.0f;
				GetTagNamePositionWorld( pActor->GetEntity(), &worldPos, &unused );
				gEnv->pRenderer->GetIRenderAuxGeom()->DrawSphere( worldPos, 0.03f, COLOR_DEBUG, true );
			}
			else
			{
				DEBUG_TAGNAMES("\t (%s) Hiding    %s at s(%d,%d) fs(%d,%d) with %d%% health: Z%d", (m_friendly ? "F" : "E"), pActor->GetEntity()->GetName(), int_round(posInScreenSpace.x), int_round(posInScreenSpace.y), int_round(m_pos.x), int_round(m_pos.y), m_health, m_curDepth );
			}
		}
#endif // ENABLE_HUD_EXTRA_DEBUG
	}
	else
	{
		SetVisible( false );
	}
}

void CHUD_Tagnames::STagInfo::ZSwapWith( const int newDepth, STagInfo& inout_ti )
{
	assert( newDepth == inout_ti.m_curDepth );
	if( newDepth == m_curDepth )
	{
		return;
	}

	// Readdress tagnames flash representations
	std::swap( vo_tagname, inout_ti.vo_tagname );
	vo_tagname->SetDisplayInfo( di_tagname );
	for(int i=0; i<k_numberOfTagnameFrames; ++i)
	{
		vos_tagnames[i].ZSwapWith( inout_ti.vos_tagnames[i] );
	}

	// set last depths
	m_lastDepth = m_curDepth;
	inout_ti.m_lastDepth = inout_ti.m_curDepth;
	// set new depths
	m_curDepth = inout_ti.m_lastDepth;
	inout_ti.m_curDepth = m_lastDepth;
}

void CHUD_Tagnames::STagInfo::PostZSortUpdate( void )
{
	bool doUpdate = m_canSee;

	if( m_curDepth != m_lastDepth )
	{
		// update everything
		m_flags |= eTIUF_Frame | eTIUF_Health | eTIUF_Name | eTIUF_HealthVisiblity;
		doUpdate = true;
	}

	if( doUpdate )
	{
		// update colours.
		if( m_flags & eTIUF_Frame )
		{
			if( m_dead )
			{
				m_curFrame = eTIF_Dead;
			}
			else if( m_friendly )
			{
				m_curFrame = eTIF_Friendly;
			}
			else //if( !m_friendly )
			{
				m_curFrame = eTIF_Enemy;
			}
			vo_tagname->GotoAndStop(m_curFrame);
		}

		// update health
		if( m_flags & eTIUF_Health )
		{
			vos_tagnames[m_curFrame-1].SetHealth( m_health );
		}

		if( m_flags & eTIUF_HealthVisiblity )
		{
			vos_tagnames[m_curFrame-1].SetHealthVisible(m_healthVisible);
		}

		// update names
		if( m_flags & eTIUF_Name )
		{
			IEntity *pEntity = gEnv->pEntitySystem->GetEntity(m_entityId);
			if( pEntity )
			{
				vos_tagnames[m_curFrame-1].SetName( pEntity->GetName() );
			}
		}

		// update position/display info
		di_tagname.SetPosition(m_pos.x, m_pos.y);
		vo_tagname->SetDisplayInfo(di_tagname);


		m_flags = eTIUF_None;
	}
}

void CHUD_Tagnames::STagInfo::SetVisible( const bool visible )
{
	//if( m_visible != visible )
	{
		vo_tagname->SetVisible(visible);
		di_tagname.SetVisible(visible);
		m_visible = visible;
	}
}

void CHUD_Tagnames::STagInfo::ConvertScreenToFlashSpace( Vec3& inout_pos ) const
{
	// Flash space in this case ranges from (0,0) -> (HUDTN_ORIGWIDTH,HUDTN_ORIGHEIGHT)
	// So normalise first (x,y) in render space first.
	// The multiply by (HUDTN_ORIGWIDTH,HUDTN_ORIGHEIGHT)

	ScreenLayoutManager* pLayoutManager = g_pGame->GetHUD()->GetLayoutManager(); // should _never_ be null when retieved from HUD elements.
	pLayoutManager->ConvertFromVirtualToRenderScreenSpace( &inout_pos.x, &inout_pos.y );
	const Vec3 nrs( inout_pos.x/gEnv->pRenderer->GetWidth(), inout_pos.y/gEnv->pRenderer->GetHeight(), 0.0f );
	//IHUDAsset* pAsset = GetAsset("Tagnames");
	//const SHUDSizeAndPositionInfo* info = pAsset->GetInfo();
	inout_pos.x = nrs.x * HUDTN_ORIGWIDTH;//info->w;//
	inout_pos.y = nrs.y * HUDTN_ORIGHEIGHT;//info->h;//
}

//////////////////////////////////////////////////////////////////////////

CHUD_Tagnames::CHUD_Tagnames()
:m_tagsCount(0)
,m_lookAt(0) // needed?
,m_enemyToLockOnTo(0)
,m_timeLockedOnEnemy(0.0f)
,m_scaleFactor(0.0f)
{
	memset( m_tagInfo,             0, HUD_MAX_TAGNAMES*sizeof(m_tagInfo[HUD_MAX_TAGNAMES]));
	memset( m_zsortedTagInfosPrev, 0, HUD_MAX_TAGNAMES*sizeof(m_zsortedTagInfosPrev[HUD_MAX_TAGNAMES]));
	memset( m_ignoreEnts,          0, HUD_MAX_TAGNAMES*sizeof(m_ignoreEnts[HUD_MAX_TAGNAMES]));

	for(int i=0; i<HUD_MAX_TAGNAMES; ++i)
	{
		m_zsortedTagInfos[i] = i; // 0 near, HUD_MAX_TAGNAMES far
	}
}

CHUD_Tagnames::~CHUD_Tagnames()
{
}

void CHUD_Tagnames::Init( void )
{
	IHUDAsset* pAsset = GetAsset( "Tagnames" );
	for( int i=0; i<HUD_MAX_TAGNAMES; ++i )
	{
		m_tagInfo[i].Init( pAsset, i+1 );
	}
}

void CHUD_Tagnames::Update( float frameTime )
{
	CActor* pClientActor = static_cast<CActor*>( gEnv->pGame->GetIGameFramework()->GetClientActor() );

	if (!pClientActor)
	{
		DEBUG_TAGNAMES("Not Entering update (%s)", pClientActor == NULL ? "Null Client Actor" : "Valid Client Actor");
		return;
	}

	CHUD* pHud = g_pGame->GetHUD();
	const CHUDCVars* pCvars = pHud->GetCVars();

	// first update data : visibility, distance, screen space pos, etc.
	for( int idx=0; idx<HUD_MAX_TAGNAMES; idx++ )
	{
		STagInfo& sti = m_tagInfo[idx];
		sti.Update(frameTime, m_lookAt);
	}

	// ZSort based on distance.
	ZSort();

	// Update date after ZSort (variable object pointers may be obsolete).
	for( int idx=0; idx<HUD_MAX_TAGNAMES; idx++ )
	{
		STagInfo& sti = m_tagInfo[idx];
		sti.PostZSortUpdate( );
	}
}

void CHUD_Tagnames::Draw( )
{
	GetAsset("Tagnames")->Draw();

	if( g_pGame->GetHUD()->GetCVars()->hud_mpTagNames_debug )
	{
		string trackedEnts("");
		int numberOfTrackedEnts = 0;
		for( int i=0; i<HUD_MAX_TAGNAMES; i++ )
		{
			if( m_tagInfo[i].m_entityId )
			{
				numberOfTrackedEnts++;
				IActor *pActor = gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(m_tagInfo[i].m_entityId);
				trackedEnts.Format( "%s, %s", trackedEnts.c_str(), (pActor ? pActor->GetEntity()->GetName() : string().Format( "Unkown Actor #%d", m_tagInfo[i].m_entityId ).c_str() ) );
			}
		}
		DEBUG_TAGNAMES( "Tracking %d entities: %s", numberOfTrackedEnts, trackedEnts.c_str() );
	}
}

void CHUD_Tagnames::OnHUDEvent(const SHUDEvent& event)
{
	switch( event.eventType )
	{
	case eHUDEvent_OnLookAtChanged :
		m_lookAt = (EntityId)event.GetData(0).GetInt();
		switch( CHUD_Crosshair::GetLookAtType(m_lookAt) )
		{
		case CHUD_Crosshair::eTT_Enemy :
			if( m_enemyToLockOnTo != m_lookAt )
			{
				// looking at some-one/thing else
				m_enemyToLockOnTo = m_lookAt;
				m_timeLockedOnEnemy = g_pGame->GetHUD()->GetCVars()->hud_mpTagNames_LockDuration;
			}
			break;
		default :
			m_timeLockedOnEnemy = -1.0f;
			m_enemyToLockOnTo = 0;
			// not looking at an enemy allow timer reduction.
			break;
		}
		break;
	case eHUDEvent_OnIgnoreEntity:
		{
			const EntityId entityId = static_cast<EntityId>(event.GetData(0).GetInt());
			float timeToIgnore = -1.0f;
			if( event.GetDataSize() > 1 )
			{
				timeToIgnore = event.GetData(1).GetFloat();
			}
			IgnoreEntity(entityId, true, timeToIgnore);
		}
		break;
	case eHUDEvent_OnStopIgnoringEntity:
		{
			const EntityId entityId = static_cast<EntityId>(event.GetData(0).GetInt());
			IgnoreEntity(entityId, false, -1.0f);
		}
		break;
	case eHUDEvent_OnEnterGame_RemotePlayer :
		{
			const EntityId remotePlayerId = static_cast<EntityId>( event.GetData(0).GetInt() );
			TrackEntity( remotePlayerId );
		}
		break;
	case eHUDEvent_OnInitPlayer :  // Fall through to eHUDEvent_OnLeaveGame_RemotePlayer desired
	case eHUDEvent_OnLeaveGame_RemotePlayer :
		{
			const EntityId remotePlayerId = static_cast<EntityId>( event.GetData(0).GetInt() );
			UntrackEntity( remotePlayerId );
		}
		break;
	case  eHUDEvent_OnHUDUnload:
		{
			IActorIteratorPtr pIter = gEnv->pGame->GetIGameFramework()->GetIActorSystem()->CreateActorIterator();
			IActor *pActor = NULL;
			while( (pActor = pIter->Next()) )
			{
				UntrackEntity(pActor->GetEntityId());
			}
		}
		break;
	case eHUDEvent_OnHUDReload :
		{
#if IS_HUD_DEV
			IActorIteratorPtr pIter = gEnv->pGame->GetIGameFramework()->GetIActorSystem()->CreateActorIterator();
			IActor *pActor = NULL;
			while( (pActor = pIter->Next()) )
			{
				if( pActor->IsPlayer() && !pActor->IsClient() )
				{
					TrackEntity(pActor->GetEntityId());
				}
			}
#endif // IS_HUD_DEV
		}
		break;
	case eHUDEvent_OnEndBlind : // Fall through intentional
	case eHUDEvent_OnSpawn :
		m_tagnameRootDI.SetAlpha(100.0f);
		break;
	default :
		assert( event.eventType == eHUDEvent_OnBlind );
		//don't want to alpha out the map screen
		m_tagnameRootDI.SetAlpha(100.0f*event.GetData(0).GetFloat());
		break;
	}
}


void CHUD_Tagnames::ZSort( void )
{
	// z sort
	memcpy(&m_zsortedTagInfosPrev, &m_zsortedTagInfos, HUD_MAX_TAGNAMES*sizeof(m_zsortedTagInfos[0]) );

	for(int i = 0; i < HUD_MAX_TAGNAMES; i++)
	{
		//m_tagInfo[i].Set(0, string().Format("%d, %d", 100*i, 100*i).c_str(), false, true, Vec3(i*100.0f,i*100.0f, 0.0f), 100.0f, 1.0f, true );
		//continue;
		if( m_tagInfo[i].m_entityId > 0 ) // have a tracked entity
		{
			const float di = m_tagInfo[i].m_distance;
			m_zsortedTagInfos[i] = i;

			for(int j = 0; j < i; j++)
			{
				const float dj = m_tagInfo[j].m_distance;
				if( di > dj && m_tagInfo[j].m_entityId > 0 ) // di is further away and have a tracked entity)
				{
					// shift j down 1
					memmove((void*)&m_zsortedTagInfos[j+1],(const void*)&m_zsortedTagInfos[j], (i-j)*sizeof(m_zsortedTagInfos[0]) );

					// insert i at j
					m_zsortedTagInfos[j] = i;
					break;
				}
			}
		}
	}

#if ENABLE_HUD_EXTRA_DEBUG
	if( g_pGame->GetHUD()->GetCVars()->hud_mpTagNames_debug )
	{
		string curZ;
		string prevZ;
		for(int i = 0; i < HUD_MAX_TAGNAMES; i++)
		{
			curZ.Format( "%s, %d", curZ.c_str(), m_zsortedTagInfos[i] );
			prevZ.Format( "%s, %d", prevZ.c_str(), m_zsortedTagInfosPrev[i] );
		}
		DEBUG_TAGNAMES( "%s", curZ.c_str() );
		DEBUG_TAGNAMES( "%s", prevZ.c_str() );
	}
#endif // ENABLE_HUD_EXTRA_DEBUG

	// update flash depths
	for(int i = 0; i < HUD_MAX_TAGNAMES; i++)
	{
		const int newTagRepresentationDepth  = m_zsortedTagInfos[i];
		int prevTagRepresentationDepth = m_zsortedTagInfosPrev[i];
		if( newTagRepresentationDepth != prevTagRepresentationDepth )
		{
			const int flashDepthId = (i+1);
			if( m_tagInfo[newTagRepresentationDepth].m_curDepth != flashDepthId )
			{
				if( flashDepthId != m_tagInfo[prevTagRepresentationDepth].m_curDepth )
				{// need to find the correct element to swap with
					for(int k = 0; k < HUD_MAX_TAGNAMES; k++)
					{
						if( m_tagInfo[k].m_curDepth == flashDepthId )
						{
							//if( i!=k )
							{
								prevTagRepresentationDepth = k;
								break;
							}
						}
					}
				}

				// this call simply readdressed the flash rep used for this flag.
				m_tagInfo[newTagRepresentationDepth].ZSwapWith( flashDepthId, m_tagInfo[prevTagRepresentationDepth] );
			}
		}
	}
}

void CHUD_Tagnames::RecalcScaleFactor( void )
{
	// Assuming that the asset was designed at a specific resolution
	// set the scale factor for the Tagnames contained within so that 
	// the tagnames appear a consistent size across resolutions.
	m_scaleFactor = HUDTN_ORIGHEIGHT / gEnv->pRenderer->GetHeight();
}

const int CHUD_Tagnames::GetTagIdxForEntity( const EntityId entityId ) 
{
	int firstFree = -1;
	for(int i=0; i<HUD_MAX_TAGNAMES; ++i)
	{
		if( m_tagInfo[i].m_entityId == entityId )
		{
			return i;
		}
		else if( firstFree==-1 && m_tagInfo[i].m_entityId == 0 )
		{
			firstFree = i;
		}
	}

	//m_tagInfo[firstFree].m_entityId = entityId;
	return firstFree;//m_tagInfo[firstFree];
}

void CHUD_Tagnames::TrackEntity( const EntityId remotePlayerId )
{
	const int playerIdx = GetTagIdxForEntity( remotePlayerId );
	m_tagInfo[playerIdx].m_entityId = remotePlayerId;
}


void CHUD_Tagnames::UntrackEntity( const EntityId remotePlayerId )
{
	const int playerIdx = GetTagIdxForEntity( remotePlayerId );
	
	if( playerIdx >= 0)
	{
		m_tagInfo[playerIdx].Clear();
	}
}

void CHUD_Tagnames::IgnoreEntity( const EntityId entityId, const bool ignore, const float time )
{
	const int playerIdx = GetTagIdxForEntity(entityId);
	if( playerIdx >= 0)
	{
		m_tagInfo[playerIdx].m_ignore = ignore;
		m_tagInfo[playerIdx].m_timeUntilIgnoreEnds = time;
	}
}